/***************************************************************************
                          html.cpp  -  description
                             -------------------
    begin                : Sat Mar 16 2002
    copyright            : (C) 2001 by Will DeRousse
    email                : badhack@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// for holding html functions outside of dbmusic.cpp.

#include "dbmusic.h"
/* Put here so it's not automatically included by dbinit.h */
#include "dbmusicvalidator.h"
#include <qstring.h>
#include <qdatetime.h>
#include <klocale.h>
#include <qfont.h>

/**
 * Generates <DOCTYPE> and charset information
 */
void HTMLDocType(QString &file, const QString &lang)
{
	if (lang.isEmpty())
		return;
	file="<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//";
	file.append(lang);
	file.append("\">\n");
}
/**
 * Generates <HEAD> and <META> information for reports
 */
void HTMLHeader(QString &file, const QString &title, const QString &metaname, const QString &version, const reportInfo *const i, const colorInfo *const c) 
{
	if (title.isEmpty() || metaname.isEmpty() || version.isEmpty() || version.isEmpty())
		return;
	file.append("\n<HEAD>\n<TITLE>");
	file.append(title);
	switch(i->report_type)
	{
		case REPORT_ALL:
			file.append(i18n(": All Titles (KmusicdB)"));
			break;
		case REPORT_YES:
			file.append(i18n(": Collected Titles (KmusicdB)"));
			break;
		default:
			file.append(i18n(": Non Collected Titles (KmusicdB)"));
	};
	file.append("</TITLE>\n<META http-equiv=\"Content-Type\" content=\"text/html; charset=" + i->charset + "\">\n");
	file.append("<META name=\"" + i18n("description") + "\" content=\"" + metaname + "\">\n");
	file.append("<META name=\"" + i18n("keywords") + "\" content=\"KmusicdB\">\n");
	file.append("<META name=\"" + i18n("version") + "\" content=\"" + version + "\">\n");

	HTMLCSS(file,i,c);

	file.append("</HEAD>\n");
}
/**
 * Inserts CSS information into the document. MUST be used or document formatting will not occur.
 */
void HTMLCSS(QString &file, const reportInfo *const info, const colorInfo *const color)
{
	QString trash;
	trash.setNum(info->rfont.pointSize());
	file.append("<STYLE TYPE=\"text/css\">\n");
	file.append("BODY\t{ background-color:");
	file.append(color->background);
	file.append("; font-family:\"");
	file.append(info->rfont.family());
	file.append("\", serif; font-size:");
	file.append(trash);
	file.append("pt; ");
	if (info->rfont.italic())
		file.append("font-style:italic;");
	if (info->rfont.bold())
		file.append("font-weight:bold;");
	file.append(" color:");
	file.append(color->text);
	file.append("}\nTABLE\t{ font-size:");
	file.append(trash);
	file.append("pt; }\nTD.misc\t{ font-weight:bold; text-align:center; background-color:");
	file.append(color->misc);
	file.append("; }\nTD.totals\t{ font-weight:bold; background-color:");
	file.append(color->totals);
	file.append("; }\nTD.artist\t{ background-color:");
	file.append(color->artist);
	file.append("; }\nTD.track\t{ background-color:");
	file.append(color->track);
	file.append("; }\nTD.title\t{ background-color:");
	file.append(color->title);
	file.append("; }\nP\t{ font-style:italic }\n");
	file.append("</STYLE>\n");
}
/**
 * Genereates <BODY> tag.
 * I combined them both because the reports will always contain them, less overhead.
 */
void HTMLBody(QString &file)
{
	file.append("<BODY>\n");
}
/**
 * Starts a new table
 */
void HTMLTable(QString &file)
{
	file.append("<TABLE WIDTH=\"100%\" BORDER=0>\n");
}
/**
 * Generates Table Header
 */
void HTMLSimpleTableHeader(QString &file, const QString &numcols, const QString &tablename)
{
	if (numcols.isEmpty() || tablename.isEmpty())
		return;
	file.append("<TR><TD CLASS=misc COLSPAN=");
	file.append(numcols);
	file.append(">");
	file.append(tablename);
	file.append("</TD></TR>\n");
}
/**
 * Generates Report Header
 */
void HTMLTableHeader(QString &file, const QString &numcols, const QString &reportname, const reportInfo *const info)
{
	if (numcols.isEmpty() || reportname.isEmpty())
		return;
	file.append("<TR><TD CLASS=misc COLSPAN=");
	file.append(numcols);
	file.append("><H2>");
	file.append(reportname);

	switch(info->report_type)
	{
		case REPORT_ALL:
			file.append(i18n("<BR>All Titles"));
			break;
		case REPORT_YES:
			file.append(i18n("<BR>Collected"));
			break;
		default:
			file.append(i18n("<BR>Not Collected"));
	};

	if (info->date_range_enabled)
	{
		file.append(i18n(" From "));
		file.append(info->start);
		file.append(i18n(" To "));
		file.append(info->end);
	}
	file.append("</H2></TD></TR>\n");

}
/**
 * Generates column headers
 */
void HTMLColumnHeaders(QString &file, const QStringList *const l)
{
	file.append("<TR>\n");
	QStringList::ConstIterator it;
	for (it=l->begin();it!=l->end();++it)
	{
		file.append("<TD CLASS=misc>");
		file.append(*it);
		file.append("</TD>\n");
	}
	file.append("</TR>\n");
}
/**
 * Fills in a sum header for the end of a report
 */
void HTMLSumHeader(QString &file, const QString &col, const QString &name, const QString &value)
{
	if (col.isEmpty() || name.isEmpty() || value.isEmpty())
		return;
	file.append("<TR><TD CLASS=totals COLSPAN=");
	file.append(col);
	file.append(">");
	file.append(name);
	file.append(": ");
	file.append(value);
	file.append("</TD></TR>\n");
}
/**
 * Populates the QString with table row data from the server.
 */
void HTMLPopulateData(dbMusic *db, QString &file, const QStringList *const l, const int &split)
{
	QStringList::ConstIterator it;
	int j=0,i=0;
	file.append("<TR>");
	for (i=0;i<db->Tuples();i++)
	{
		for (it=l->begin();it!=l->end();it++)
		{
			file.append("<TD CLASS=");
			file.append(*it);
			file.append(">");
			file.append(db->GetValue(i,j));
			file.append("</TD>");
			j++;
		}
		j=0;

		if (i%75==0)
			kapp->processEvents();
		if (split && (i%2==0))
			file.append("<TD><pre>  </pre></TD>");
		else
			file.append("</TR>\n<TR>");
	}
	if (split && (i%2==1))
		file.append("</TR>\n");
	else
		file.remove(file.length()-4,4);
}
/**
 * A function for statistics to reduce code base.
 * Populates with category information.
 */
void HTMLPopulateData2(dbMusic *db, QString &file, const float &denom, const QStringList *const c, const reportInfo *const info)
{
	QStringList::ConstIterator it;
	float numerator;
	QString perc;

	file.append("<TR>");
	for (int i=0;i<db->Tuples();i++)
	{
		it=c->begin();
		for (int j=0;j<2;j++)
		{
			file.append("<TD CLASS=");
			file.append(*it);
			file.append(">");
			file.append(db->GetValue(i,j));
			file.append("</TD>");
			it++;
		}
		file.append("<TD CLASS=");
		file.append(*it);
		file.append(">");
		perc=db->GetValue(i,1);
		numerator=perc.toFloat();
		if (denom==0)
			perc="0.0";
		else
			perc.setNum(100*numerator/denom);
		file.append(perc);
		file.append("</TD></TR>\n<TR>");

		if (i%75==0)
			kapp->processEvents();
	}
	file.remove(file.length()-4,4);
}
/**
 * Appends the date, user, dbname, server address information
 */
void HTMLSysInfo(QString &file, const QString &dbname)
{
	if (dbname.isEmpty())
		return;
	file.append(i18n("\n<P>Date: "));
	file.append(QDateTime::currentDateTime().toString());
	file.append(i18n("<BR>Database: "));
	file.append(dbname);
	file.append(i18n("<BR>Local User: "));
	file.append(getenv("USER"));
	file.append("</P>\n");
}
/**
 * Adds the date range syntax to a query
 */
void queryRange(QString &query, const QString &begin, const QString &end, const reportInfo *const i)
{
	if (!i->date_range_enabled)
		return;
	if (query.find("WHERE") > 0)
		query.append(" AND ");
	else
		query.append(" WHERE ");
	query.append(validateQString(begin));
	query.append(" >= ");
	query.append(validateNumQString(i->start));
	query.append(" AND ");
	query.append(validateQString(end));
	query.append(" <= ");
	query.append(validateNumQString(i->end));
}
/**
 * Adds the collect/follow syntax to a query
 */
void queryBoolStatus(QString &query, const QString &name, const reportInfo *const i)
{
	if (name.isEmpty())
		return;
	if (i->report_type==REPORT_ALL)
		return;

	if (query.find("WHERE") > 0)
		query.append(" AND ");
	else
		query.append(" WHERE ");
	query.append(validateQString(name));

	if (i->report_type==REPORT_YES)
		query.append("=true");
	else
		query.append("=false");
}
/**
 * Adds matching for two fields (a.name=t.name)
 */
void queryMatch(QString &query, const QString &name1, const QString &name2)
{
	if (name1.isEmpty() || name2.isEmpty())
		return;
	if (query.find("WHERE") > 0)
		query.append(" AND ");
	else
		query.append(" WHERE ");
	query.append(validateQString(name1));
	query.append("=");
	query.append(validateQString(name2));
}
/**
 * Adds ORDER BY syntax to a query
 */
void queryOrderBy(QString &query, const QStringList *const l, const reportInfo *const info)
{
	if (query.find("ORDER BY") < 0)
		query.append(" ORDER BY ");
	else
		query.append(", ");
	QStringList::ConstIterator it;
	for (it=l->begin();it!=l->end();it++)
	{
		query.append(validateQString((*it)));
		if (info->sort_order==ASC)
			query.append(" ASC, ");
		else
			query.append(" DESC, ");
	}
	query.remove(query.length()-2,2);
}
