/* 
 * $Id: print.c,v 1.29 2000/08/29 22:24:17 cbond Exp $
 *
 * libarr - a screen management toolkit
 *
 * Copyright (C) 2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Author: Chris Bond
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <sys/types.h>

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>

#include "arr.h"
#include "term_iface.h"
#include "types.h"
#include "misc.h"
#include "tinfo.h"
#include "cap_offset.h"

/* arr_norm_conv and arr_ext_conv are lists of good and bad characters.  The
 * bad ones, however, are replaced with spaces, so we won't be caught printing
 * them to the screen.
 */
u_char arr_norm_conv[256], arr_ext_conv[256];

/* `attribs' is a set of attributes.  It will actually always reflect the state
 * of the screen, so it's safe to rely on it.
 */
static struct __c_attr_t {
	byte_t ext;		/* extended charset mode */
	byte_t inverse;		/* standout mode */
	byte_t bold;		/* bold mode */
} attribs = { 0, 0, 0 };

ansi_color_t *
build_color(mask)
	arr_color_t mask;
{
	static ansi_color_t b_mask;

	if ((mask & FGA_BOLD)) {
		b_mask.attrib = 1;
		mask &= ~FGA_BOLD;
	}
	else
		b_mask.attrib = 0;

	b_mask.fg = ((mask >> 4) & 0xF);
	b_mask.bg = (mask & 0x7);

	return (ansi_color_t *)&b_mask;
}

void
arr_nprint_col(chr, color, n)
	u_char chr;
	arr_color_t color;
	int n;
{
	tsize_t t_seek;

	t_seek = TERM_SEEK(arr_scr);

	while (n-- > 0) {
		if (t_seek > arr_scr->seek.max)
			return;

		arr_scr->main[t_seek].chr = chr;
		arr_scr->main[t_seek].color = color;
		arr_scr->main[t_seek].ext = 0;

		TERM_INC_XY(arr_scr);
		++t_seek;
	}
}

void
arr_nprint_xy_col(chr, x, y, color, n)
	u_char chr;
	tsize_t x, y;
	arr_color_t color;
	int n;
{
	tsize_t t_seek;

	t_seek = ((arr_scr->term.co * y) + x);

	while (n-- > 0) {
		if (t_seek > arr_scr->seek.max)
			return;

		arr_scr->main[t_seek].chr = chr;
		arr_scr->main[t_seek].color = color;
		arr_scr->main[t_seek].ext = 0;
		++t_seek;
	}
}

void
arr_nprint_acs_col(chr, color, n)
	int chr;
	arr_color_t color;
	int n;
{
	tsize_t t_seek;

	t_seek = TERM_SEEK(arr_scr);

	while (n-- > 0) {
		if (t_seek > arr_scr->seek.max)
			return;

		arr_scr->main[t_seek].color = color;
		
		if (chr < 256)
		{
			arr_scr->main[t_seek].chr = chr;
			arr_scr->main[t_seek].ext = 1;
		}
		else
		{
			arr_scr->main[t_seek].chr =
				arr_scr->term.acs_map[chr - 256].chr;
			arr_scr->main[t_seek].ext =
				arr_scr->term.acs_map[chr - 256].ext;
		}

		TERM_INC_XY(arr_scr);
		++t_seek;
	}
}

void
arr_nprint_acs_xy_col(chr, x, y, color, n)
	int chr;
	tsize_t x, y;
	arr_color_t color;
	int n;
{
	tsize_t t_seek;

	t_seek = ((arr_scr->term.co * y) + x);

	while (n-- > 0) {
		if (t_seek > arr_scr->seek.max)
			return;

		arr_scr->main[t_seek].color = color;
		
		if (chr < 256)
		{
			arr_scr->main[t_seek].chr = chr;
			arr_scr->main[t_seek].ext = 1;
		}
		else
		{
			arr_scr->main[t_seek].chr =
				arr_scr->term.acs_map[chr - 256].chr;
			arr_scr->main[t_seek].ext =
				arr_scr->term.acs_map[chr - 256].ext;
		}

		++t_seek;
	}
}

void
arr_strprint_col(str, color)
	char *str;
	arr_color_t color;
{
	tsize_t t_seek;

	t_seek = TERM_SEEK(arr_scr);

	for (; (*str); str++) {
		if (t_seek > arr_scr->seek.max)
			return;

		arr_scr->main[t_seek].chr = *str;
		arr_scr->main[t_seek].color = color;
		arr_scr->main[t_seek].ext = 0;

		TERM_INC_XY(arr_scr);
		++t_seek;
	}
}

void
arr_strprint_xy_col(str, x, y, color)
	char *str;
	tsize_t x, y;
	arr_color_t color;
{
	tsize_t t_seek;

	t_seek = ((arr_scr->term.co * y) + x);

	for (; *str; str++) {
		if (t_seek > arr_scr->seek.max)
			return;

		arr_scr->main[t_seek].chr = *str;
		arr_scr->main[t_seek].color = color;
		arr_scr->main[t_seek].ext = 0;
		++t_seek;
	}
}

void
arr_nstrprint(str, len)
	char *str;
	int len;
{
	tsize_t t_seek;

	t_seek = TERM_SEEK(arr_scr);

	for (; (len-- > 0) && (*str); str++) {
		if (t_seek > arr_scr->seek.max)
			return;

		arr_scr->main[t_seek].chr = *str;
		arr_scr->main[t_seek].color = arr_scr->pen_color;
		arr_scr->main[t_seek].ext = 0;

		TERM_INC_XY(arr_scr);
		++t_seek;
	}
}

void
arr_hide_cursor(void)
{
	arr_scr->cursor.visible = 0;
}

void
arr_show_cursor(void)
{
	arr_scr->cursor.visible = 1;
}

void
arr_set_cursor_pos(x, y)
	tsize_t x, y;
{
	arr_scr->cursor.x = x;
	arr_scr->cursor.y = y;
}

void
arr_beep(void)
{
	if (attribs.ext) {
		TERM_NOTHROTTLE_WRITE(tinfo_getstr(tc_entry, OFFSET_AE));
		attribs.ext = 0;
	}

	TERM_NOTHROTTLE_WRITE(tinfo_getstr(tc_entry, OFFSET_BL));
}

void
arr_clear(void)
{
	if (attribs.ext) {
		TERM_NOTHROTTLE_WRITE(tinfo_getstr(tc_entry, OFFSET_AE));
		attribs.ext = 0;
	}

	TERM_NOTHROTTLE_WRITE(tinfo_getstr(tc_entry, OFFSET_CL));
}

void
term_buffer(ptr)
	char *ptr;
{
	register char *save;
	register int ln;

	for (; *ptr; ptr++) {
		if ((*ptr == '$') && (*(ptr + 1) == '<')) {
			save = ptr;
			ptr += 2;

			while ((*ptr) && (isdigit(*ptr)))
				ptr++;

			if (*ptr != '>') {
				/*
				 * Oh shit!  This *wasn't* a delay!  Abort!
				 */
				TERM_CBUFFER('$');
				ptr = ++save;
				continue;
			}
			else {
				/*
				 * atoi() is smart and will break off when the
				 * number ends, so we don't have to terminate
				 * this string.
				 */
				for (ln = atoi((save + 2)); (ln >= 0); ln--)
					TERM_CBUFFER(padding_chr);
			}
		}
		else
			TERM_CBUFFER(*ptr);
	}
}

void
set_color(cl)
	arr_color_t cl;
{
	ansi_color_t *ptr;

	ptr = build_color(cl);

	if (*tinfo_getstr(tc_entry, OFFSET_AF) == NUL) {
		if (bold_color[((cl & FG_MASK) >> 4)]) {
			if (!attribs.bold) {
				term_buffer(tinfo_getstr(tc_entry, OFFSET_MD));
				attribs.bold = 1;
			}
		}
		else {
			if (attribs.bold) {
				term_buffer(tinfo_getstr(tc_entry, OFFSET_ME));
				attribs.bold = 0;
				attribs.ext = 0;
			}
		}
	}
	else {
		if (ptr->attrib) {
			if (!attribs.bold) {
				term_buffer(tinfo_getstr(tc_entry, OFFSET_MD));
				attribs.bold = !0;
			}
		}
		else {
			if (attribs.bold) {
				term_buffer(tinfo_getstr(tc_entry, OFFSET_ME));
				attribs.bold = 0;
				attribs.ext = 0;
			}
		}

		term_buffer(ARR_TPARM(tinfo_getstr(tc_entry, OFFSET_AF),
				      ptr->fg, 0));
	}

	if ((*tinfo_getstr(tc_entry, OFFSET_AB)) == NUL) {
		if (inverse_color[(cl & BG_MASK)]) {
			if (!attribs.inverse) {
				term_buffer(tinfo_getstr(tc_entry, OFFSET_SO));
				attribs.inverse = !0;
			}
		}
		else {
			if (attribs.inverse) {
				term_buffer(tinfo_getstr(tc_entry, OFFSET_SE));
				attribs.inverse = 0;

				if (attribs.bold)
					term_buffer(tinfo_getstr(tc_entry,
								OFFSET_MD));
			}
		}
	}
	else
		term_buffer(ARR_TPARM(tinfo_getstr(tc_entry, OFFSET_AB),
				      ptr->bg, 0));

}

int
arr_redraw(void)
{
	memset(arr_scr->back, CL_INVALID, TERM_SIZE(&arr_scr->term) *
					  sizeof(struct bchr));
	return arr_dump_screen();
}

static void
multibyte_prepass(void)
{
	int x, y, bp;
	int mb_start;

	for (y = 0; y < arr_scr->term.li; y++) {
		mb_start = -1;	/* No multi-byte characters yet. */

		for (x = 0; x < arr_scr->term.co; x++) {
			bp = ((y * arr_scr->term.co) + x);

			if ((arr_scr->main[bp].chr >= term_mbyte_pairs->r1) &&
			    (arr_scr->main[bp].chr <= term_mbyte_pairs->r2) &&
			    (arr_scr->main[bp].ext == 0)) {
				if (mb_start == -1)
					mb_start = bp;

				if (!memcmp(&arr_scr->main[bp], 
				            &arr_scr->back[bp],
				            sizeof(struct bchr)))
					continue;

				if ((bp - mb_start) & 1) {	/* odd */
					arr_scr->back[(bp - 1)].color =
								CL_INVALID;
					arr_scr->main[(bp - 1)].color =
								arr_scr->
								main[bp].color;
				}
				else {				/* even */
					arr_scr->back[(bp + 1)].color =
								CL_INVALID;
					arr_scr->main[(bp + 1)].color =
								arr_scr->
								main[bp].color;
				}
			}
			else
				mb_start = -1;
		}
	}
}

int
arr_dump_screen(void)
{
	int pen_color, x, y, bp;
	int jump;

	if ((*tinfo_getstr(tc_entry, OFFSET_CM)) == NUL)
		memset(arr_scr->back, 0, TERM_SIZE(&arr_scr->term) *
					 sizeof(struct bchr));
	else if (term_mbyte_pairs != NULL)
		multibyte_prepass();

	jump = !0;			/* Yes, we need to jump initially. */
	pen_color = CL_INVALID;		/* Undefined colour (no colors yet) */

	term_buffer(tinfo_getstr(tc_entry, OFFSET_VI));

	for (y = 0; y < arr_scr->term.li; ++y) {
		for (x = 0; x < arr_scr->term.co; ++x) {
			bp = ((y * arr_scr->term.co) + x);

			if (memcmp(&arr_scr->main[bp], &arr_scr->back[bp],
				   sizeof(struct bchr)) != 0) {
				if (jump) {
					term_buffer(ARR_TPARM(tinfo_getstr(
						tc_entry, OFFSET_CM), y, x));
					jump = 0;
				}

				if (pen_color != arr_scr->main[bp].color) {
					set_color(arr_scr->main[bp].color);
					pen_color = arr_scr->main[bp].color;
				}

				if (arr_scr->main[bp].ext != 0) {
				    	if (attribs.ext == 0) {
				    		term_buffer(tinfo_getstr(
							tc_entry, OFFSET_AS));
				    		attribs.ext = 1;
				    	}
				    	
					TERM_CBUFFER(arr_ext_conv[
							arr_scr->main[bp].chr]);
				}
				else {
					if (attribs.ext != 0) {
						term_buffer(tinfo_getstr(
							tc_entry, OFFSET_AE));
						attribs.ext = 0;
					}

					TERM_CBUFFER(arr_norm_conv[arr_scr->main
							      [bp].chr]);
				}
			}
			else
				jump = 1;
		}

		if (*tinfo_getstr(tc_entry, OFFSET_CM) != NUL)
			jump = 1;
		else {
			if (attribs.ext) {
				term_buffer(tinfo_getstr(tc_entry, OFFSET_AE));
				attribs.ext = 0;
			}

			term_buffer("\r\n");
		}
	}

	if (arr_scr->cursor.visible != 0) {
		term_buffer(tinfo_getstr(tc_entry, OFFSET_VE));
		term_buffer(ARR_TPARM(tinfo_getstr(tc_entry, OFFSET_CM),
					arr_scr->cursor.y, arr_scr->cursor.x));
	}

	TERM_FLUSH();

	/* Keep the front and back buffers in synch:
	 */
	memcpy(arr_scr->back, arr_scr->main, TERM_SIZE(&arr_scr->term) *
					     sizeof(struct bchr));

	return 0;
}
