/* 
**  mod_repository.c -- Apache repository module
*/ 

#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "ap_config.h"
#include "http_log.h"

module MODULE_VAR_EXPORT repository_module;

#define WATCHPOINT printf("WATCHPOINT %s %d\n", __FILE__, __LINE__);

typedef struct {
	char *repository_root_directory;
	char *trigger;
	int max_size;
	int levels;
	int hide_post_page;
	table *types;
}
repository_conf;

static void *create_mconfig(pool *p, char *dir) {
	repository_conf *cfg = ap_pcalloc(p, sizeof(repository_conf));
	cfg->max_size = 12582912;
	cfg->levels = 6;
	cfg->trigger = NULL;
	cfg->types = ap_make_table(p, 4);
	cfg->hide_post_page = 0;

	return cfg;
}
static int include_virtual(request_rec *r, char *uri, char *key) {
	int status = OK;
	request_rec *subr;

	ap_table_set(r->headers_in, "Content-Length", "0");
	subr = (request_rec *) ap_sub_req_method_uri("GET", uri, r);
	ap_table_set(subr->subprocess_env, "REPOSITORY_KEY", key);
	status = ap_run_sub_req(subr);
	ap_destroy_sub_req(subr);

	return status;
}


static int read_content(request_rec *r, char *data, long length) {
	int rc;
	char argsbuffer[HUGE_STRING_LEN];
	int rsize, rpos=0;
	long len_read = 0;

	if ((rc = ap_setup_client_block(r, REQUEST_CHUNKED_ERROR)) != OK){
		return rc;
	}

	if (ap_should_client_block(r)) {

		ap_hard_timeout("repository_read", r);
		while((len_read = ap_get_client_block(r, argsbuffer, HUGE_STRING_LEN)) > 0 ){
			ap_reset_timeout(r);
			if ((rpos + (int)len_read) > length) {
				rsize = length -rpos;
			} else {
				rsize = (int)len_read;
			}
			memcpy(data + rpos , argsbuffer, rsize);
			rpos += rsize;
		}

		ap_kill_timeout(r);
	}

	return rc;
}

char *location(request_rec *r) {
	repository_conf *cfg = ap_get_module_config(r->per_dir_config, &repository_module);
	int x;
	const char *short_filename;
	char *filename;

	short_filename = &r->uri[ap_rind(r->uri, '/')];
	filename = ap_psprintf(r->pool,"%s/",cfg->repository_root_directory);
	for(x=1; x < cfg->levels; x++,x++) {
		filename = ap_psprintf(r->pool,"%s/%c%c",filename, short_filename[x],short_filename[x+1]);
	}
	ap_getparents(filename);

	return ap_psprintf(r->pool,"%s/%s",filename, short_filename);
}

char *mklocation(char *key, request_rec *r) {
	repository_conf *cfg = ap_get_module_config(r->per_dir_config, &repository_module);
	char *loc;
	int x;

	loc = ap_psprintf(r->pool,"%s/",cfg->repository_root_directory);

	for(x=0; x < cfg->levels; x++,x++) {
		loc = ap_psprintf(r->pool,"%s/%c%c",loc, key[x],key[x+1]);
		if(!(ap_is_directory(loc))){
			ap_getparents(loc);
			(void)mkdir(loc ,0755);
		}
	}

	return ap_psprintf(r->pool,"%s/%s", loc, key);
}

int get_request(request_rec *r){
	FILE *file;
	char *filename;
	char *info_filename;
	char buffer[512];
	repository_conf *cfg = ap_get_module_config(r->per_dir_config, &repository_module);
	int x;

	filename = location(r);
	info_filename = ap_psprintf(r->pool,"%s.info",filename);

	if (!(file = ap_pfopen(r->pool, info_filename, "r"))) { 
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r, "File not found: %s", info_filename);
		return HTTP_NOT_FOUND;
	}
	fgets(buffer, 512, file);
	r->content_type = ap_pstrdup(r->pool, buffer);
	ap_pfclose(r->pool, file);

	ap_send_http_header(r);
	if (!(file = ap_pfopen(r->pool, filename, "r"))) { 
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r, "File not found: %s", filename);
		return HTTP_NOT_FOUND;
	}
	while((x = fgetc(file)) != EOF){
		ap_rputc(x,r);
	}
	ap_pfclose(r->pool, file);

	return OK;

}

int post_request(request_rec *r){
	unsigned char *data;
	char *content_length;
	char *key;
	char *filename;
	FILE *file;
	int x;
	long length;
	request_rec *subr;
	const char *type;
	int rc;
	repository_conf *cfg = ap_get_module_config(r->per_dir_config, &repository_module);

	r->content_type = ap_pstrdup(r->pool, "text/plain");      
	ap_send_http_header(r);

	content_length = (char *)ap_table_get(r->headers_in, "Content-Length");
	
	length = (content_length ? atoi(content_length) : 0);
	if (length > cfg->max_size) {
		return HTTP_REQUEST_ENTITY_TOO_LARGE;
	}

	data = ap_palloc(r->pool, length);
	if(rc = read_content(r, data, length) != OK) {
		return rc;
	}
	
	key = (char *)ap_md5_binary(r->pool, data, length);

	filename = mklocation(key,r);

	if (!(file = ap_pfopen(r->pool, filename, "w"))) {
		return HTTP_FORBIDDEN;
	}

	subr = (request_rec *) ap_sub_req_lookup_file(r->uri, r);
	type = subr->content_type;
	ap_destroy_sub_req(subr);

	fwrite(data, length, 1, file);

	ap_pfclose(r->pool, file);

	/* Now tackle the info file */
	filename = ap_psprintf(r->pool,"%s.info",filename);
	if (!(file = ap_pfopen(r->pool, filename, "w"))) {
		return HTTP_FORBIDDEN;
	}
	if(type) {
		(void)fputs(type, file);
	} else {
		type = ap_pstrdup(r->pool, "unknown");      
	}
	ap_pfclose(r->pool, file);

	if (cfg->trigger && ap_table_get(cfg->types, type)) {
		/* Now lets see if the triger works */
		if((rc = include_virtual(r,cfg->trigger,key)) != OK) {
			ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r, "The following error occured while processing trigger %s : %d", cfg->trigger, rc);
		}
		/* Since this has a trigger we can see if we need to not send output */
		if(cfg->hide_post_page) {
			return OK;
		} 
	}

	ap_rprintf(r, "\n%s\n", key);

	return OK;
}
			
int delete_request(request_rec *r){
	const char *filename;

	r->content_type = "text/plain";      
	ap_send_http_header(r);
	ap_rputs("Completed\n", r);
	filename = location(r);
	if(unlink(filename)) {
		return HTTP_NOT_FOUND;
	}
	filename = ap_psprintf(r->pool, "%s.info", filename);
	if(unlink(filename)) {
		return HTTP_NOT_FOUND;
	}

	return OK;
}
	

static int repository_handler(request_rec *r) {
	/* This is for the paranoid AKA just to make sure
	 that we don't end up in a loop do to calling
	 anothor URI during the process.
	*/
	if (r->main) {
		return DECLINED;
	}

	if (r->header_only) {
		r->content_type = "text/plain";      
		ap_send_http_header(r);

		return OK;
	}
	switch (r->method_number) {
		case M_GET:
			return get_request(r);
			break;

		case M_POST:
		case M_PUT:
			return post_request(r);
			break;

		case M_DELETE:
			return delete_request(r);
			break;

		default :
			r->content_type = "text/plain";      
			ap_send_http_header(r);
			return HTTP_NOT_IMPLEMENTED;
			break;
	}

	return OK;
}

/* Dispatch list of content handlers */
static const handler_rec repository_handlers[] = { 
    { "repository", repository_handler }, 
    { NULL, NULL }
};

static const char *
add_repos (cmd_parms *cmd, void *mconfig, char *directory)
{
	repository_conf *cfg = (repository_conf *) mconfig;

	cfg->repository_root_directory = ap_pstrdup(cmd->pool, directory);

	return NULL;
}

static const char *
add_file_size (cmd_parms *cmd, void *mconfig, char *filesize)
{
	repository_conf *cfg = (repository_conf *) mconfig;

	cfg->max_size = (filesize ? atoi(filesize) : 0);

	return NULL;
}

static const char *
add_file_levels (cmd_parms *cmd, void *mconfig, char *value)
{
	repository_conf *cfg = (repository_conf *) mconfig;

	cfg->levels = (value ? atoi(value) * 2 : 0);
	if (cfg->levels > 32) {
		cfg->levels = 32;
	}

	return NULL;
}

static const char *
add_trigger (cmd_parms *cmd, void *mconfig, char *trigger)
{
	repository_conf *cfg = (repository_conf *) mconfig;

	cfg->trigger = ap_pstrdup(cmd->pool, trigger);

	return NULL;
}

static const char *
add_type (cmd_parms *cmd, void *mconfig, char *type) {
	repository_conf *cfg = (repository_conf *) mconfig;

	ap_table_addn(cfg->types, type, "enabled");

	return NULL;
}

static const char *
add_hide_key (cmd_parms *cmd, void *mconfig, int flag) {
	repository_conf *cfg = (repository_conf *) mconfig;

	cfg->hide_post_page = flag;

	return NULL;
}

static const command_rec repository_cmds[] =
{
	  { "RepositoryRoot", add_repos, NULL, OR_ALL, TAKE1, "The document root of the repository." },
	  { "RepositoryFileSize", add_file_size, NULL, OR_ALL, TAKE1, "Maximum size of a file that can be inserted into the repository." },
	  { "RepositoryFileLevels", add_file_levels, NULL, OR_ALL, TAKE1, "Maximum size of a file that can be inserted into the repository." },
	  { "RepositoryTrigger", add_trigger, NULL, OR_ALL, TAKE1, "This is a CGI/servlet that will be executed whenever a file is inserted." },
	  { "RepositoryType", add_type, NULL, OR_ALL, TAKE1, "This is a CGI/servlet that will be executed whenever a file is inserted." },
	  { "RepositoryHideKey", add_hide_key, NULL, OR_ALL, FLAG, "This tells a post request to not print the key on its own." },
		(NULL)
};

/* Dispatch list for API hooks */
module MODULE_VAR_EXPORT repository_module = {
    STANDARD_MODULE_STUFF, 
    NULL,                  /* module initializer                  */
    create_mconfig,        /* create per-dir    config structures */
    NULL,                  /* merge  per-dir    config structures */
    NULL,                  /* create per-server config structures */
    NULL,                  /* merge  per-server config structures */
    repository_cmds,        /* table of config file commands       */
    repository_handlers,   /* [#8] MIME-typed-dispatched handlers */
    NULL,                  /* [#1] URI to filename translation    */
    NULL,                  /* [#4] validate user id from request  */
    NULL,                  /* [#5] check if the user is ok _here_ */
    NULL,                  /* [#3] check access by host address   */
    NULL,                  /* [#6] determine MIME type            */
    NULL,                  /* [#7] pre-run fixups                 */
    NULL,                  /* [#9] log a transaction              */
    NULL,                  /* [#2] header parser                  */
    NULL,                  /* child_init                          */
    NULL,                  /* child_exit                          */
    NULL                   /* [#0] post read-request              */
#ifdef EAPI
   ,NULL,                  /* EAPI: add_module                    */
    NULL,                  /* EAPI: remove_module                 */
    NULL,                  /* EAPI: rewrite_command               */
    NULL                   /* EAPI: new_connection                */
#endif
};

