/*
 *  config.c
 *  mod_musicindex
 *
 *  $Id: config.c 899 2009-12-04 13:55:29Z varenet $
 *
 *  Created by Thibaut VARENE on Thu Mar 20 2003.
 *  Copyright (c) 2003-2006 Regis BOUDIN
 *  Copyright (c) 2003-2005,2007,2009 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * Configuration handling and default settings.
 *
 * @author Regis Boudin
 * @author Thibaut Varene
 * @version $Revision: 899 $
 * @date 2003-2007
 * @date 2009
 *
 * That file takes care of the module configuration, either via setting some
 * known default values, and/or by getting them from environmental configuration
 * variables passed to apache in its config file.
 *
 * http://www.apachetutor.org/dev/config
 */

#include "config.h"
#include "sort.h"

#ifdef	ENABLE_CACHE_FILE
#include "cache-file.h"
#endif
#ifdef	ENABLE_CACHE_MYSQL
#include "cache-mysql.h"
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>	/* atoi */
#endif

typedef int (*cache_backend_setup)(cmd_parms *cmd, const char *setup_string, mu_config *const conf);

/** list of cache setup functions, to be handled manually */
static const cache_backend_setup const cache_setups[] = {
#ifdef	ENABLE_CACHE_FILE
	cache_file_setup,
#endif
#ifdef	ENABLE_CACHE_MYSQL
	cache_mysql_setup,
#endif
	NULL
};

/** Default sort order */
static const unsigned char const default_order[] = {
	SB_ALBUM,
	SB_POSN,
	SB_TRACK,
	SB_ARTIST,
	SB_TITLE,
	SB_LENGTH,
	SB_BITRATE,
	SB_FREQ,
	SB_FILETYPE,
	SB_FILENAME,
	SB_URI,
};

/** Default fields displayed. */
static const unsigned char const default_fields[] = {
	SB_TITLE,
	SB_ARTIST,
	SB_ALBUM,
	SB_LENGTH,
	SB_BITRATE,
};

static const char const default_directory[] = "/musicindex";	/**< Default musicindex directory */
static const char const default_rootname[] = "Music";		/**< Default root directory name */
static const char const default_css[] = "musicindex.css";	/**< Default CSS file name */

/**
 * Generates a per-directory configuration.
 *
 * This function creates a new configuration structure and fills it with
 * default values.
 *
 * @param p Apache pool for memory allocation.
 * @param dummy String given by apache we don't use.
 *
 * @return The newly created configuration structure.
 */
void *create_musicindex_config(apr_pool_t *p, char *dummy)
{
	mu_config *const new = (mu_config *)apr_pcalloc(p, sizeof(mu_config));

	memset(new->order, 0, sizeof(new->order));
	memset(new->fields, 0, sizeof(new->fields));

	memcpy(new->fields, default_fields, sizeof(default_fields));
	memcpy(new->order, default_order, sizeof(default_order));
	
	new->title = default_rootname;

	new->directory = default_directory;
	new->css = default_css;
	new->search = NULL;
	new->iceserver = NULL;

	new->cache = NULL;
	new->cache_setup = NULL;

	new->cookie_life = CONF_COOKIE_LIFE;
	new->rss_items = 0;

	new->options = 0;
	new->options_not = 0;

	new->dir_per_line = CONF_DIRPERLINE;

	return (void *)new;
}

/**
 * Merge per-directory configurations.
 *
 * This function is supposed to merge two per-directory configurations. We
 * apply a simple inheritance pattern from the parent directory(ies).
 *
 * @note In the absence of a .htaccess in the hierarchy of folders being accessed,
 * @a basev contains a freshly created config structure (through
 * create_musicindex_config()) and @a addv contains another config struct created
 * through create_musicindex_config(), with the configuration parameters modified
 * by <Directory> directive (if any) set appropriately.
 *
 * @note In the presence of a .htaccess file in the directory being accessed,
 * @a basev contains the result of the merging of all "parent" configuration directives
 * (<Directory> being the top ancestor, and any subsquent .htaccess being merged
 * using merge_musicindex_configs()) up to the one in the current directory being
 * accessed, which config options are set in @a addv.
 *
 * @note In the presence of a .htaccess anywhere in the parent hierarchy of the
 * directory being accessed @b BUT in said directory, @a basev contains a pristine
 * config structure (as per create_musicindex_config()) and @a addv contains the
 * result of merging all parent configuration directives (<Directory> being the
 * top ancestor).
 *
 * Hence the logic of this function: if @a addv contains init data (data setup in
 * create_musicindex_config()), use @a basev element. Otherwise, @a addv contains
 * modified config directive, use it instead.
 *
 * @param p Apache pool for memory allocation.
 * @param basev Pointer to main configuration structure.
 * @param addv Pointer to parent dir configuration structure,
 *
 * @return The newly created configuration structure.
 */
void *merge_musicindex_configs(apr_pool_t *p, void *basev, void *addv)
{
	mu_config *const restrict new = (mu_config *)apr_pcalloc(p, sizeof(mu_config));
	mu_config *const restrict base = (mu_config *)basev;
	mu_config *const restrict add = (mu_config *)addv;

	if (!memcmp(add->order, default_order, sizeof(default_order)))
		memcpy(new->order, base->order, sizeof(base->order));
	else
		memcpy(new->order, add->order, sizeof(add->order));

	if (!memcmp(add->fields, default_fields, sizeof(default_fields)))
		memcpy(new->fields, base->fields, sizeof(base->fields));
	else
		memcpy(new->fields, add->fields, sizeof(add->fields));

	if (add->title == default_rootname)
		new->title = base->title;
	else
		new->title = add->title;

	/* directory is currently non-modifiable */
	new->directory = default_directory;

	if (add->css == default_css)
		new->css = base->css;
	else
		new->css = add->css;

	if (add->cache == NULL) {
		new->cache = base->cache;
		new->cache_setup = base->cache_setup;
	} else {
		new->cache = add->cache;
		new->cache_setup = add->cache_setup;
	}

	if (add->iceserver == NULL)
		new->iceserver = base->iceserver;
	else
		new->iceserver = add->iceserver;

	new->options = base->options | add->options;
	new->options &= ~(base->options_not | add->options_not);
	
	new->options_not = base->options_not | add->options_not;
	new->options_not &= ~(base->options | add->options);

	if (add->cookie_life == CONF_COOKIE_LIFE)
		new->cookie_life = base->cookie_life;
	else
		new->cookie_life = add->cookie_life;

	if (add->rss_items == 0)
		new->rss_items = base->rss_items;
	else
		new->rss_items = add->rss_items;

	if (add->dir_per_line == CONF_DIRPERLINE)
		new->dir_per_line = base->dir_per_line;
	else
		new->dir_per_line = add->dir_per_line;

	return new;
}

static const struct {
	const char *const string;
	const char value;
} const options[] = {
	{"track",	SB_TRACK},
	{"disc",	SB_POSN},
	{"length",	SB_LENGTH},
	{"bitrate",	SB_BITRATE},
	{"freq",	SB_FREQ},
	{"artist",	SB_ARTIST},
	{"album",	SB_ALBUM},
	{"title",	SB_TITLE},
	{"filename",	SB_FILENAME},
	{"date",	SB_DATE},
	{"filetype",	SB_FILETYPE},
	{"genre",	SB_GENRE},
	{"uri",		SB_URI},
	{"size",	SB_SIZE},
	{NULL, 		0}
};

static unsigned short sort_or_fields(cmd_parms *cmd, unsigned char * restrict list, const char *optstr)
{
	const char *r;
	register unsigned short i = 0, j;

	while (optstr[0] && (i < SB_MAX)) {
		r = ap_getword_conf(cmd->temp_pool, &optstr);
		for (j=0; options[j].string; j++) {
			if (!strcasecmp(r, options[j].string)) {
				list[i++] = options[j].value;
				break;
			}
		}
	}

	return i;
}

static const char *sort_order(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *const d_cfg = (mu_config *)d;
	unsigned short i = sort_or_fields(cmd, d_cfg->order, optstr);
	d_cfg->order[i] = SB_URI;	/* fallback on URI sort */
	return NULL;
}

static const char *set_fields(cmd_parms *cmd, void *d, const char *optstr)
{
	sort_or_fields(cmd, ((mu_config *)d)->fields, optstr);
	return NULL;
}

/**
 * Enables the module options.
 *
 * Read the different options given as parameters, and set the various flags.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *basic_config(cmd_parms *cmd, void *d, const char *optstr)
{
	const char *r;
	mu_config *const cfg = (mu_config *)d;
	
	while (optstr[0] != '\0') {
		int enable = 1;
		unsigned short flag = 0;
		r = ap_getword_conf(cmd->temp_pool, &optstr);

		/* Should we enable or disable ? Is there a +/- prefix ? */
		if ( *r == '-' ) {
			enable = 0;
			r++;
		} else if (*r == '+') {
			r++;
		}

		/* Which option are we talking about ? */
		if (strcmp(r, "On") == 0) {
			enable = 1;
			flag = MI_ACTIVE;
		} else if (strcmp(r, "Off") == 0) {
			enable = 0;
			flag = MI_ACTIVE;
		} else if (strcmp(r, "Stream") == 0) {
			flag = MI_ALLOWSTREAM;
		} else if (strcmp(r, "Download") == 0) {
			flag = MI_ALLOWDWNLD;
		} else if (strcmp(r, "Search") == 0) {
			flag = MI_ALLOWSEARCH;
#ifdef ENABLE_OUTPUT_ARCHIVE
		} else if (strcmp(r, "Tarball") == 0) {
			flag = MI_ALLOWTARBALL;
#endif
		} else if (strcmp(r, "Rss") == 0) {
			if (enable == 1)
				cfg->rss_items = CONF_RSS_ITEMS;
			else {
				cfg->rss_items = -1;
			}
		}

		/* Have this done in one place instead of five */
		if (flag != 0) {
			if (enable == 1) {
				cfg->options |= flag;
				cfg->options_not &= ~flag;
			} else {
				cfg->options &= ~flag;
				cfg->options_not |= flag;
			}
		}
	}

	return NULL;
}

/**
 * Configures the cache location in the filesystem.
 *
 * This function sets the configuration string for the cache subsystem.
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_cache_uri(cmd_parms *cmd, void *d, const char *optstr)
{
	int i, ret = 1;

	for (i = 0; (ret != 0) && (cache_setups[i] != NULL); i++)
		ret = cache_setups[i](cmd, optstr, (mu_config *)d);

	return NULL;
}

/**
 * Sets the title of the page.
 *
 * This function simply chages the name of the root.
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_page_title(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *const cfg = (mu_config *)d;
	if ((optstr == NULL) || (optstr[0] == '\0'))
		cfg->title = NULL;
	else
		cfg->title = apr_pstrdup(cmd->pool, optstr);
	return NULL;
}

/**
 * Sets the icecast server address.
 * This function sets an icecast server for streaming.
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_ice_server(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *const cfg = (mu_config *)d;
	cfg->iceserver = apr_pstrdup(cmd->pool, optstr);
	return NULL;
}

/**
 * Sets the default CSS file.
 * This function sets which CSS file will be used by default
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_css_default(cmd_parms *cmd, void *d, const char *optstr)
{
	((mu_config *)d)->css = apr_pstrdup(cmd->pool, optstr);
	return NULL;
}

/**
 * Sets the cookie lifetime.
 * This function sets the lifetime in seconds of the cookie used for custom
 * playlist constructs.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_cookie_life(cmd_parms *cmd, void *d, const char *optstr)
{
	((mu_config *)d)->cookie_life = atoi(optstr);
	return NULL;
}

/**
 * Sets the number of directories to display per line.
 * Negative values will disable the "pretty folders" system.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_dirperline(cmd_parms *cmd, void *d, const char *optstr)
{
	((mu_config *)d)->dir_per_line = (atoi(optstr) ? atoi(optstr) : CONF_DIRPERLINE);
	return NULL;
}

/**
 * Sets the default display.
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_display(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *const cfg = (mu_config *)d;
	if (strcmp(optstr, "RSS") == 0) {
		cfg->options |= MI_RSS;
		cfg->order[0] = SB_MTIME;
		cfg->order[1] = SB_URI;
		cfg->options &= ~MI_RECURSIVE;
		cfg->rss_items = CONF_RSS_ITEMS;
	}
	else if (strcmp(optstr, "HTML") == 0) {
		cfg->options &= ~MI_RSS;
		cfg->options_not |= MI_RSS;
		memcpy(cfg->order, default_order, sizeof(default_order));
	}
	return NULL;
}

/* XXX if somebody puts a ACCESS_CONF setting in .htaccess, Apache triggers a 500 */
const command_rec musicindex_cmds[] = {
	AP_INIT_RAW_ARGS("MusicIndex", basic_config, NULL, OR_INDEXES,
		"can be : On/Off +/-Stream +/-Download +/-Search +/-Rss"),
	AP_INIT_RAW_ARGS("MusicSortOrder", sort_order, NULL, OR_INDEXES,
		"can be : title album artist track disc length bitrate filetype genre filename date uri"),
	AP_INIT_RAW_ARGS("MusicFields", set_fields, NULL, OR_INDEXES,
		"can be : title album artist track disc length bitrate filetype genre filename date"),
	AP_INIT_RAW_ARGS("MusicIndexCache", set_cache_uri, NULL, ACCESS_CONF,
		"Set the cache configuration string"),
	AP_INIT_RAW_ARGS("MusicPageTitle", set_page_title, NULL, OR_INDEXES,
		"Set the root title of the page."),
	AP_INIT_TAKE1("MusicIceServer", set_ice_server, NULL, ACCESS_CONF,
		"Set the icecast server address : [server.domain.org]:8000"),
	AP_INIT_TAKE1("MusicDefaultCss", set_css_default, NULL, OR_INDEXES,
		"Set the default CSS file"),
	AP_INIT_TAKE1("MusicCookieLife", set_cookie_life, NULL, ACCESS_CONF,
		"Set the lifetime (in seconds) of the cookie sent for custom playlists"),
	AP_INIT_TAKE1("MusicDefaultDisplay", set_display, NULL, ACCESS_CONF,
		"Set the default display returned by the module (HTML or RSS)"),
	AP_INIT_TAKE1("MusicDirPerLine", set_dirperline, NULL, OR_INDEXES,
		"Set the number of directories per line in the directory listing"),
	{NULL}
};
