/* ======================================================================
 * Copyright (c) 1998-1999 The Johns Hopkins University.
 * All rights reserved.
 * The following code was written by Theo Schlossnagle for use in the
 * Backhand project at The Center for Networking and Distributed Systems
 * at The Johns Hopkins University.
 * Please refer to the LICENSE file before using this software.
 * ======================================================================
*/

#include <stdlib.h>

#include "httpd.h"

#include "mod_backhand.h"
#include "builtins.h"
#include "math.h"

int find_highest_arriba(serverstat *);

int backhand_off(request_rec *r, ServerSlot *Servers, int *n, char *arg) {
  *n=0;
  return 0;
}

int removeSelf(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  int i, j;
  for (i=0,j=0; i<*n; i++) 
    if(servers[i].id != 0)
      servers[j++] = servers[i];
  *n = j;
  return *n;
}

int addSelf(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  int i;
  for(i=0;i<*n;i++)
    if(servers[i].id == 0) /* We are already here */
      return *n;
  servers[*n].id = 0;
  (*n)++;
  return *n;
}

int addPrediction(request_rec *e, ServerSlot *servers, int *n, char *arg) {
  int addload;
  if(!*n) return *n;
  addload = serverstats[0].tatime/6000+1;
  if(servers[0].id == 0) addload>>=1;
  serverstats[servers[0].id].load+=addload;
  return *n;
}

int byAge(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  /* This function returns x of n servers where
     Age(i) <= o, where o is defined in header unless
     supplied as an argument.  Always include yourself */

  int mycount, i, myage;
  time_t now;
  if(!arg || ((myage=atoi(arg))<1))
    myage=SERVER_TIMEOUT;
  now = time(NULL);
  mycount=0;
  for(i=0;i<*n;i++)
    if((servers[i].id==0) || is_alive(servers[i].id,myage,now))
      servers[mycount++]=servers[i];
  *n = mycount;
  return mycount;
}
int byCost(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  /* This function returns x of n servers where
     Cost(i) <= Cost(j) for all j in n and i in x */
 
  float lowcost = HUGE;
  int mycount, i;
  long hwm = serverstats[0].load_hwm;
  static time_t lastcalc = 0;
  static int max_arriba = 0;
  if(lastcalc != serverstats[0].mtime) {
    /* only once per self update */
    max_arriba = find_highest_arriba(serverstats);  
    lastcalc = serverstats[0].mtime;
  }
  mycount = 0;
  for (i = 0; i < *n; i++) {
    float cpupower =
      (float)serverstats[servers[i].id].load/1000.0;
    float cpucost, mempower, memcost;
    cpupower *=
      (float)max_arriba/
      (float)serverstats[servers[i].id].arriba;
    cpupower /= (float)hwm/1000.0;
    cpucost = pow((float)*n,cpupower);
    mempower = 1.0 -
      ((float)(serverstats[servers[i].id].amem-15000000)/
       (float)serverstats[servers[i].id].tmem);
    memcost = pow((float)*n,mempower);
    if ((cpucost + memcost) <= lowcost) {
      if ((cpucost + memcost) < lowcost) {
        lowcost = cpucost + memcost;
        mycount = 0; /* Set it to begining */
      }
      servers[mycount++] = servers[i];
    }
/*    ap_log_error(APLOG_MARK, APLOG_NOTICE|APLOG_NOERRNO, NULL,
	"byCost [%s %d::%d] : %3.2f + %3.2f = %3.2f",
	serverstats[servers[i]].hostname, i, mycount-1,
	cpucost, memcost, lowcost); */
  }
  *n = mycount;
  return mycount;
}

static float personalpreference = 0.0;
/* This is a hack to allow use of qsort with a parameter.
   No, this is not thread safe, but this is Apache 1.3, so it
   doesn't matter */
static int _load_compare(const void *A, const void * B) {
  register float aload, bload;
  register short a, b;
  register float adj;
  a = ((ServerSlot *)A)->id;
  b = ((ServerSlot *)B)->id;
  adj = (float)MAX(serverstats[a].arriba,
		   serverstats[b].arriba);
  aload = (float)serverstats[a].load;
  aload += (a)?personalpreference:0;
  aload *= adj/(float)serverstats[a].arriba;
  bload = (float)serverstats[b].load;
  bload += (b)?personalpreference:0;
  bload *= adj/(float)serverstats[b].arriba;
  return (aload==bload)?0:((aload<bload)?-1:1);
}
int byLoad(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  /* This function returns x of n servers where
     Load(i) <= Load(j) for all j in n and i in x */
  if(arg) personalpreference = 1000.0*atof(arg);
  else personalpreference = 0.0;
  qsort(servers, *n, sizeof(ServerSlot), _load_compare);
  return *n;
}

static int bbcpersonalpreference = 0;
/* This is a hack to allow use of qsort with a parameter.
   No, this is not thread safe, but this is Apache 1.3, so it
   doesn't matter */
static int _busychildren_compare(const void *A, const void * B) {
  register int aload, bload;
  register short a, b;
  a = ((ServerSlot *)A)->id;
  b = ((ServerSlot *)B)->id;
  aload = serverstats[a].nservers-serverstats[a].aservers;
  aload += (a)?bbcpersonalpreference:0;
  bload = serverstats[b].nservers-serverstats[b].aservers;
  bload += (b)?bbcpersonalpreference:0;
  return (aload==bload)?0:((aload<bload)?-1:1);
}

int byBusyChildren(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  /* This function returns x of n servers where
     BusyChildren(i) <= BusyChildren(j) for all j in n and i in x */
  if(arg) bbcpersonalpreference = atoi(arg);
  else bbcpersonalpreference = 0;
  qsort(servers, *n, sizeof(ServerSlot), _busychildren_compare);
  return *n;
}

int byCPU(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  /* This function returns x of n servers where
     Idle(i) >= Idle(j) for all j in n and i in x */

  int highidle, mycount, i;
  for(i=0;i<*n;i++)
    if(serverstats[servers[i].id].cpu > highidle)
      highidle=serverstats[servers[i].id].cpu;
  mycount=0;
  for(i=0;i<*n;i++)
    if(serverstats[servers[i].id].cpu == highidle)
      servers[mycount++]=servers[i];
  *n=mycount;
  return mycount;
}

int byLogWindow(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  /*  Returns log2(n) of n servers */

  int mycount=0, oldsize=*n;
  while(oldsize >>= 1) mycount++;
  /* new size is now floor(log2(*n)) */
  *n = mycount;
  return mycount;
}

int byRandom(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  /*  Returns n or n server as a random permutation of the input
      vector */

  int newcard, mycount, i;
  ServerSlot swap;
  static int rs=-1;
  newcard = *n;
  if(rs==-1) srand(time(NULL));
  mycount=0;
  for(i=0;i<newcard;i++) {
    rs = rand()%(*n-i);
    swap = servers[i];
    servers[mycount++] = servers[i+rs];
    servers[i+rs] = swap;
  }
  *n = mycount;
  return mycount;
}

static char *nameConstructor(char *redirhost, char *arg,
			     const char *hostname,
			     const request_rec *r) {
  char scratch[MAXHOSTNAMELEN];
  const char *host;
  char *scp = arg;
  char *dcp = redirhost;
  
  host = ap_table_get(r->headers_in, "Host");
  if(!host) host=hostname;
  
  while(*scp) {
    char *tcp=scp;
    int dir=1;
    int crop=0;
    if(*tcp == '%' && tcp++) {
      if(*tcp == '-' && tcp++)
	dir=-1;
      while(isdigit(*tcp)) {
	crop=(crop*10)+(*tcp - '0');
	tcp++;
      }
      switch(*tcp) {
      case 'S':
	/* -n means chop off n parts from the right side 
	   n means to save n parts from the left side */
	if(dir==-1) {
	  dir = strlen(hostname);
	  while((crop>0) && (dir>0)) {
	    if(hostname[dir] == '.') crop--;
	    dir--;
	  }
	  if(hostname[dir+1] == '.') dir++;
	  strncpy(dcp, hostname, MIN(dir,MAXHOSTNAMELEN-(dcp-redirhost)));
	  dcp+=MIN(dir,MAXHOSTNAMELEN-(dcp-redirhost));
	} else {
	  const char *ncp = hostname;
	  int tocopy;
	  dir=0;
	  while(*ncp!='\0' && crop>0) {
	    if(*ncp == '.') crop--;
	    ncp++;
	  }
	  if(ncp != hostname &&
	     ncp[-1] == '.') ncp--;
	  tocopy = MIN(ncp-hostname,
		       MAXHOSTNAMELEN-(dcp-redirhost));
	  strncpy(dcp, hostname, tocopy);
	  dcp+=tocopy;
	}
	break;
      case 'H':
	/* Here -n means save n part on the right side */
	/* n means chop off n part from the left side */
	if(dir==-1) {
	  dir = strlen(host);
	  while((crop>0) && (dir>0)) {
	    if(host[dir] == '.') crop--;
	    dir--;
	  }
	  if(host[dir+1]=='.') dir+=2;
	  strncpy(dcp, host+dir, MAXHOSTNAMELEN-(dcp-redirhost));
	  dcp+=MIN(strlen(host)-dir, MAXHOSTNAMELEN-(dcp-redirhost));
	} else {
	  const char *ncp = host;
	  dir=0;
	  while(*ncp!='\0' && crop>0) {
	    if(*ncp == '.') crop--;
	      ncp++;
	  }
	  strncpy(dcp, ncp, MAXHOSTNAMELEN-(dcp-redirhost));
	  dcp+=MIN(strlen(host)-(ncp-host), MAXHOSTNAMELEN-(dcp-redirhost));
	}
	break;	  
      default:
	tcp = scp;
	break;
      }
    }
    if(tcp==scp) {
      *dcp = *scp;
      dcp++;
    } else {
      scp = tcp;
    }
    scp++;
  }
  *dcp = '\0';
  return redirhost;
}

int HTTPRedirectToName(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  int i;
  char redirhost[MAXHOSTNAMELEN];
  const char *host = NULL;
  for(i=0;i<*n;i++) {
    servers[i].redirect = MB_HTTP_REDIRECT;
    servers[i].hosttype = MB_HOSTTYPE_NAME;
  }
  if(arg) {
    host = ap_table_get(r->headers_in, "Host");
    if(!host) host=serverstats[0].hostname;
    nameConstructor(redirhost, arg, serverstats[0].hostname, r);
    if(!strcmp(redirhost, host)) {
      /* They are the same, we do'nt redirect! */
      *n = 0;
      return *n;
    }
    if(servers[0].id != 0) {
      /* The last nameConstructor was called with me and I am not the choice */
      nameConstructor(redirhost, arg, serverstats[servers[0].id].hostname, r);
    }
    ap_table_set(r->notes, "Backhand-Redirect-Host", redirhost);
  } else {
    host = ap_table_get(r->headers_in, "Host");
    if(host!=NULL && !strcmp(serverstats[0].hostname, host)) {
      /* They came to me with my ServerName... must be fate */
      *n = 0;
      return *n;
    }
  }
  return *n;
}

int HTTPRedirectToIP(request_rec *r, ServerSlot *servers, int *n, char *arg) {
  int i;
  for(i=0;i<*n;i++) {
    servers[i].redirect = MB_HTTP_REDIRECT;
    servers[i].hosttype = MB_HOSTTYPE_IP;
  }
  return *n;
}

/* See README.bySession */
static const char *DEFAULT_PHPSESSID = "PHPSESSID=";
int bySession(request_rec *r, ServerSlot *servers, int *n, char *identifier) {
  const char *session_id=NULL, *cookie;
  int i, j, c, dec[4];
  struct in_addr res;
 
  if(!identifier)
    identifier = (char *)DEFAULT_PHPSESSID;
  
  /* Note: this won't work if the client sends more than one cookie.
   * I've tested mozilla M18, sends one cookie header containing all
   * variables that have been set via cookies. Hope this is standard
   * and all the other clients act the same way.
   */
  if(cookie = ap_table_get(r->headers_in, "Cookie")) {
    if(session_id = strstr(cookie, identifier))
      /* Cookie containing the SESSIONID found */
      session_id += strlen(identifier);
  }
  if(!session_id && r->args) {
    /* No cookie set. perhaps we have a phpsessid in the URI? */
    if(session_id = strstr(r->args, identifier))
      session_id += strlen(identifier);
  }

  /* Fall through if no session_id found */
  /* 8 bytes to hex encode a IPv4 address */
  if(!session_id || (strlen(session_id)<8)) {
    return(*n);
  }

  /* hex decode it... */
  /* scribbled that. might be done shorter */
  for(j=0; j<4; j++) {
    dec[j] = 0;
    for(i=0; i<2; i++) {
      c = toupper(*(session_id + i + j*2));
      if(isdigit(c))
        dec[j] = dec[j]*16 + c - '0';
      else
        if(((c - 'F') <= 0) && ((c - 'A') >= 0))
          dec[j] = dec[j]*16 + c - 'A' + 10;
    }
  }
  res.s_addr = htonl((dec[0] << 24) | (dec[1] << 16) | (dec[2] << 8) | dec[3]);

  /* ...and find the server */
  for(i=0; i<*n; i++) {
    if(serverstats[servers[i].id].contact.sin_addr.s_addr == res.s_addr) {
      /* Bingo, we got it */
      servers[0].id = servers[i].id;
      *n=1;
      return 1;
    }
  }
  /* Invalid session.  Pick a good machine instead. */
  return *n;
}
