/* ======================================================================
 * Copyright (c) 1998-1999 The Johns Hopkins University.
 * All rights reserved.
 * The following code was written by Theo Schlossnagle for use in the
 * Backhand project at The Center for Networking and Distributed Systems
 * at The Johns Hopkins University.
 * Please refer to the LICENSE file before using this software.
 * ======================================================================
*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>

#include "httpd.h"
#include "http_config.h"
#include "http_conf_globals.h"
#include "http_log.h"
#include "http_protocol.h"
#include "scoreboard.h"

#include "mod_backhand.h"
#include "back_util.h"
#include "arriba.h"
#include "apue.h"

extern listen_rec *ap_listeners;

static server_rec *s=NULL;
static int loglevel=0;
static void fillstat(serverstat *, server_rec *, struct in_addr, unsigned int);
static void initstat(void);

/* We INCLUDE this so as to make the fuctions static... for simplicity */
/* Very unconventional... but then again, I am lazy */
#include "platform.c"

void backhand_initstat(void) {
  /* Run our benchmark here... Whatever it may be... It should
     be multithreaded or multiprocessed to account for SMP machines */
  mod_backhand_personal_arriba = backhand_bench();
}

typedef struct {
  int fd;
  int pid;
} aConn;

static aConn connection_pool[MAXSERVERS][MAXSESSIONSPERSERVER];
static aConn children[HARD_SERVER_LIMIT];

int birth_control(void) {
  int i;
  for(i=0;i<HARD_SERVER_LIMIT;i++)
    if(children[i].fd < 0)
      return i;
  return -1;
}
void add_child_fd(int fd, int pid) {
  int i;
  for(i=0;i<HARD_SERVER_LIMIT;i++)
    if(children[i].fd < 0) {
      children[i].fd = fd;
      children[i].pid = pid;
      return;
    }
}
int find_server(struct in_addr *sin) {
  register int i;
  for(i=0;i<MAXSERVERS;i++)
    if(memcmp(sin, &serverstats[i].contact.sin_addr,
	      sizeof(struct in_addr))==0)
      return i;
  return -1;
}
int new_session(struct sockaddr_in *sin) {
  int nsd;
  struct sockaddr_in sain;
  nsd = socket(AF_INET, SOCK_STREAM, 0);
  if(nsd==-1) {
    if(loglevel & MBLL_NET3) {
      ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
		   "mod_backhand: Error in new_session:socket()");
    }
    return nsd;
  }
#if defined(TCP_NODELAY) && !defined(MPE)
  i=1;
  if(setsockopt(nsd, IPPROTO_TCP, TCP_NODELAY, (char *) &i,
		sizeof(int)) < 0) {
    if(loglevel & MBLL_NET4) {
      ap_log_error(APLOG_MARK, APLOG_WARNING, NULL,
		   "setsockopt: (TCP_NODELAY)");
    }
  }
#endif
  sain = *sin;
  sain.sin_family = AF_INET;
  if(connect(nsd, (struct sockaddr *)&sain, sizeof(struct sockaddr_in))<0) {
    if(loglevel & MBLL_NET3) {
      ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
		   "mod_backhand: Error in new_session:connect()");    
    }
    close(nsd);
    nsd=-1;
  }
  return nsd;
}
static int reserve_session(struct in_addr *request, pid_t pid) {
  int i,j,rfd=-1;
  if((i=find_server(request))==-1) return -1;
  for(j=0;j<MAXSESSIONSPERSERVER && rfd<0;j++)
    if(connection_pool[i][j].fd>=0) {
      rfd=connection_pool[i][j].fd;
      /* do this because keeping track is REALLY complicated */
      connection_pool[i][j].fd=-1;
      connection_pool[i][j].pid=-1;
    }      
  for(j=0;j<MAXSESSIONSPERSERVER && rfd<0;j++)
    if(connection_pool[i][j].fd<0) {
      connection_pool[i][j].fd = new_session(&serverstats[i].contact);
      connection_pool[i][j].pid = pid;
      rfd=connection_pool[i][j].fd;
      /* do this because keeping track is REALLY complicated */
      connection_pool[i][j].fd=-1;
      connection_pool[i][j].pid=-1;
    }
  return rfd;
}
void replace_session(struct in_addr *submission, int fd) {
  int i,j;
  if((i=find_server(submission))==-1) {
    if(loglevel & MBLL_NET3) {
      ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, NULL,
		   "mod_backhand: Sent session to unknown server: close(%d)",
		   fd);
    }
    close(fd);
    return;
  }
  for(j=0;j<MAXSESSIONSPERSERVER;j++)
    if(connection_pool[i][j].fd<0) {
      connection_pool[i][j].fd=fd;
      connection_pool[i][j].pid=1;
      return;
    }
  if(loglevel & MBLL_NET3) {
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, NULL,
		 "mod_backhand: Received session > max sessions per server:\
close(%d)",
		 fd);
  }
  close(fd);
}

void html_print_serverstats_table(request_rec *r) {
  if(serverstats) {
    int i;
    int mypid;
    char hostaddress[22]; /* aaa.bbb.ccc.ddd:12345\0 = 22 */
    unsigned short tempport;
    struct in_addr tempaddr;
    time_t now = time(NULL);
    mypid = getpid();
    ap_rputs("<TABLE CELLSPACING=0 CELLPADDING=4 BORDER=0 BGCOLOR=#ffffff>\
<TR bgcolor=#9999ee><TD><B align=center>Entry</B></TD><TD><B>Hostname</B></TD>\
<TD align=right><B>Age</B></TD><TD align=center><B>Address</B></TD>\
<TD align=right><B>Total Mem</B></TD><TD align=left><B>Avail Mem</B></TD>\
<TD align=center><B># ready servers/<BR># total servers</B></TD>\
<TD align=center><B>~ms/req [#req]</B></TD>\
<TD align=right><B>Arriba</B></TD><TD align=center><B># CPUs</B></TD>\
<TD align=center><B>Load/HWM</B></TD><TD align=right><B>CPU Idle</B></TD></TR>\n",
	     r);
    for(i=0;i<MAXSERVERS;i++) {
      tempport = ntohs(serverstats[i].contact.sin_port);
      tempaddr.s_addr = ntohl(serverstats[i].contact.sin_addr.s_addr);
      ap_snprintf(hostaddress, 21, "%s:%d",
		  inet_ntoa(serverstats[i].contact.sin_addr),
		  tempport);
      hostaddress[21] = '\0';
      if(serverstats[i].contact.sin_addr.s_addr!=0)
	ap_rprintf(r, "<TR bgcolor=%s><TD align=center>%d</TD><TD>%s</TD>\
<TD align=right>%d</TD><TD align=center>%s</TD><TD align=right>%u&nbsp;MB</TD>\
<TD align=left>%u&nbsp;MB</TD><TD align=center>%d/%d</TD><TD align=center>%d [%d]</TD>\
<TD align=right>%d</TD>\
<TD align=center>%d</TD><TD align=center>%0.3f/%d</TD><TD align=right>%0.3f</TD></TR>\n",
		   (!is_alive(i,SERVER_TIMEOUT,now))?"#ff4444":
		   (i%2)?"#aaaaaa":"#cccccc",
		   i,                          /* Entry */
		   serverstats[i].hostname,    /* hostname */
		   (int)(time(NULL) - serverstats[i].mtime), /* Age */
		   hostaddress,                /* aaa.bbb.ccc.ddd:port */
		   /* Total physical memory in megabytes */
		   ((unsigned int)serverstats[i].tmem)/(1024*1024),
		   /* Avilable memory in megabytes */
		   ((unsigned int)serverstats[i].amem)/(1024*1024),
		   serverstats[i].aservers,    /* # ready Apache procs */
		   serverstats[i].nservers,    /* # of Apache procs */
		   serverstats[i].tatime,      /* Ave ms / request */
		   serverstats[i].numbacked,   /* # reqs in ave above */
		   serverstats[i].arriba,      /* How fast? */
		   serverstats[i].ncpu,        /* Number of CPUs */
		   serverstats[i].load/1000.0, /* load on machine */
		   serverstats[i].load_hwm/1000, /* load_hwm for cluster */
		   serverstats[i].cpu/1000.0); /* CPU utilization on machine*/
    }
    ap_rputs("</TABLE>\n", r);
  } else {
    ap_rprintf(r, "<B>An error occured: serverstats=%p (shmxxx failed?)</B>\n",
	       serverstats);
  }
}

#define tvdiff(a,b,c,d,e,f) {(e=(b>=d)); \
                             f=e?(b-d):(1000000L+b-d); \
                             e=e?(a-c):(a-c-1L);}
static void calc_time(struct timeval *diff, struct timeval *until) {
  struct timeval now;
  gettimeofday(&now,NULL);
  if(now.tv_sec > until->tv_sec ||
     (now.tv_sec == until->tv_sec && now.tv_usec > until->tv_usec)) {
    /* now is past until */
    tvdiff(now.tv_sec, now.tv_usec,
	   until->tv_sec, until->tv_usec,
	   diff->tv_sec, diff->tv_usec);
    diff->tv_sec *= -1L;
    diff->tv_usec *= -1L;
  } else if (now.tv_sec == until->tv_sec && now.tv_sec == until->tv_sec) {
    diff->tv_sec = diff->tv_usec = 0L;
  } else {
    /* until is yet to come */
    tvdiff(until->tv_sec, until->tv_usec,
	   now.tv_sec, now.tv_usec,
	   diff->tv_sec, diff->tv_usec);  
  }
}

static void setnowplusquantum(struct timeval *next) {
  gettimeofday(next, NULL);
  next->tv_usec += QUANTUM%1000000;
  next->tv_sec += QUANTUM/1000000;
  if(next->tv_usec >= 1000000) {
    /* next->tv_sec += next.tv_usec/1000000;  This will always be > += 1. */
    next->tv_sec += 1;
    next->tv_usec = next->tv_usec%1000000;
  }
}

ACL *matchACL(ACL *acl, struct in_addr *in) {
  ACL *iter;
  struct in_addr testaddr;
  memcpy(&testaddr, in, sizeof(struct in_addr));
  for(iter=acl;iter!=NULL;iter=iter->next) {
    if((testaddr.s_addr & iter->mask.s_addr) ==
       (iter->ip.s_addr & iter->mask.s_addr))
      return iter;
  }
  return NULL; /* failed */
}

int alreadybound(int slot, StatsSI *slots) {
  int i;
  for(i=0;i<slot;i++)
    if(slots[i].sin.sin_port == slots[slot].sin.sin_port)
      return i;
  return -1;
}

void sendtomany(int *ssd, serverstat *buf, StatsSI *sins, int nsins) {
  int i;
  serverstat lc;
  struct sockaddr_in tempsin;
  memcpy(&lc, buf, sizeof(serverstat));
  for(i=0;i<nsins;i++) {
    memcpy(&lc.contact.sin_addr, &sins[i].from, sizeof(struct in_addr));
    memcpy(&tempsin, &sins[i].sin, sizeof(struct sockaddr_in));
    tempsin.sin_family = AF_INET;
    sendto(ssd[i], (char *)&lc, sizeof(serverstat), 0,
	   (struct sockaddr *)&tempsin, sizeof(struct sockaddr_in));
  }
}

int broadcast_my_stats(void *data, child_info *ci) {
  int ufd, *rsd, *ssd, selectret, i, j, nsins;
  struct sockaddr_in ears;
  StatsSI *sins;
  mbcfg *cfg;
  struct timeval next, towait;
  serverstat mystat, astat;
  fd_set mask, read_mask, dummy_mask, except_mask;
  long on;
  int sillycompiler = 1; /* For while loop */
  char ud_fn[MAXPATHLEN];
  daemon_params *dparam = (daemon_params *)data;

  pool *my_pool;
  ACL *acl;
  char *UD_DN;
  
  my_pool = dparam->backhand_pool;
  s = dparam->server;
  acl = dparam->UDPacl;
  UD_DN = dparam->UD_DN;

  cfg = our_sconfig(s);

  if(loglevel & MBLL_NET2) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ALERT, NULL,
		 "setuid: trying to set uid/gid to %d/%d",
		 ap_user_id, ap_group_id);
  }
  if(!geteuid()) {
    if(setgid(ap_group_id) == -1) {
      if(loglevel & MBLL_NET4) {
        ap_log_error(APLOG_MARK, APLOG_ALERT, NULL,
		     "setuid: unable to change gid");
      }
      exit(-1);
    }
    if(setuid(ap_user_id) == -1) {
      if(loglevel & MBLL_NET4) {
        ap_log_error(APLOG_MARK, APLOG_ALERT, NULL,
		     "setuid: unable to change uid");
      }
      exit(-1);
    }
  }

  /* Create a Unix Domain socket for passing TCP/IP sessions */
  ap_snprintf(ud_fn, MAXPATHLEN, "%s/bparent", UD_DN);
  ufd = serv_listen(ud_fn);
  if(ufd<0) {
    if(loglevel & MBLL_NET4) {
      ap_log_error(APLOG_MARK, APLOG_ERR, NULL, 
		   "serv_listen: listen on unix domain socket");
      ap_log_error(APLOG_MARK, APLOG_ERR, NULL, 
		   "mod_backhand: permissions problem on umbilical: \"%s\"?",
		   ud_fn);
    }
    exit(-1);
  }

  nsins = dparam->nsins;
  sins = dparam->sins;
  rsd = (int *)malloc(sizeof(int)*nsins);
  ssd = (int *)malloc(sizeof(int)*nsins);

  /* -1 out the connection_pool */
  for(j=0;j<HARD_SERVER_LIMIT;j++) {
    children[j].fd=-1;
    children[j].pid=-1;
  }
  for(j=0;j<MAXSESSIONSPERSERVER;j++) {
    connection_pool[0][j].fd=-1;
    connection_pool[0][j].pid=-1;
  }
  for(i=1;i<MAXSERVERS;i++)
    memcpy(connection_pool[i], connection_pool[0],
	   sizeof(aConn)*MAXSESSIONSPERSERVER);
  
  FD_ZERO(&mask);
  FD_SET(ufd, &mask);
  
  for(i=0;i<nsins;i++) {
    int afd;
    /* Build out all the IPs on which we listen for statistics */
    /* Create a Datagram socket */
    if(isbroadcast(sins[i].sin)) {
      if((ssd[i] = ap_psocket(my_pool, AF_INET, SOCK_DGRAM, 0)) < 0) {
	if(loglevel & MBLL_NET4) {
	  ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
		       "mod_backhand: cannot allocate socket for \
server statistic binding.");
	}
	exit(0);
      }
      on=1;
      if(setsockopt(ssd[i], SOL_SOCKET, SO_BROADCAST,
		    (char *)&on, sizeof(on))<0) {
	if(loglevel & MBLL_NET4) {
	  ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
		       "mod_backhand: cannot setsockopt to \
SO_BROADCAST for server statistic broadcasting.");
	}
	exit(0);
      }
      if((afd=alreadybound(i,sins))<0) {
	rsd[i]=ssd[i];
	ears.sin_family = AF_INET;
	ears.sin_addr.s_addr = htonl(INADDR_ANY);
	ears.sin_port = sins[i].sin.sin_port;
        on=1;
        if(setsockopt(ssd[i], SOL_SOCKET, SO_REUSEADDR,
		      (char *)&on, sizeof(on))<0) {
	  if(loglevel & MBLL_NET4) {
	    ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			 "mod_backhand: cannot setsockopt to \
SO_REUSERADDR for server statistic binding.");
	  }
	  exit(0);
        }
	if(bind(rsd[i], (struct sockaddr *)&ears,
		sizeof(struct sockaddr_in))<0) {
	  if(loglevel & MBLL_NET4) {
	    ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			 "mod_backhand: (broadcast) cannot bind to port %d.",
			 ntohs(ears.sin_port));
	  }
	  exit(0);
	}
      } else {
	ssd[i]=ssd[afd];
	close(rsd[i]);
	rsd[i]=-1;
      }
    } else if(ismulticast(sins[i].sin)) {
      struct ip_mreq mreq;
      unsigned char ttl_val;
      if((ssd[i] = ap_psocket(my_pool, AF_INET, SOCK_DGRAM, 0)) < 0) {
	if(loglevel & MBLL_NET4) {
	  ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
		       "mod_backhand: cannot allocate socket for \
server statistic binding.");
	}
	exit(0);
      }
      if((afd=alreadybound(i,sins))<0) {
	rsd[i]=ssd[i];
	ears.sin_family = AF_INET;
	ears.sin_addr.s_addr = htonl(INADDR_ANY);
	ears.sin_port = sins[i].sin.sin_port;
        on=1;
        if(setsockopt(ssd[i], SOL_SOCKET, SO_REUSEADDR,
		      (char *)&on, sizeof(on))<0) {
	  if(loglevel & MBLL_NET3) {
	    ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			 "mod_backhand: cannot setsockopt to \
SO_REUSERADDR for server statistic binding.");
	  }
	  exit(0);
        }
	if(bind(rsd[i], (struct sockaddr *)&ears,
		sizeof(struct sockaddr_in))<0) {
	  if(loglevel & MBLL_NET4) {
	    ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			 "mod_backhand: (multicast) cannot bind to port %d.",
			 ntohs(ears.sin_port));
	  }
	  exit(0);
	}
      } else {
	/* we have already bound to that port */
	ssd[i]=ssd[afd];
	close(rsd[i]);
	rsd[i]=-1;
      }
      ttl_val=sins[i].sin.sin_family;
      sins[i].sin.sin_family = AF_INET; /* the only thing it COULD be */
      mreq.imr_multiaddr.s_addr = sins[i].sin.sin_addr.s_addr;
      mreq.imr_interface.s_addr = htonl(INADDR_ANY);
      if(setsockopt(ssd[i], IPPROTO_IP, IP_ADD_MEMBERSHIP,
		    (char *)&mreq, sizeof(struct ip_mreq))<0) {
	if(loglevel & MBLL_NET4) {
	  ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
		       "mod_backhand: cannot setsockopt \
IP_ADD_MEMBERSHIP for server statistic multicasting.");
	}
	exit(0);
      }
      if(setsockopt(ssd[i], IPPROTO_IP, IP_MULTICAST_TTL,
		    (char *)&ttl_val, sizeof(unsigned char))<0) {
	if(loglevel & MBLL_NET4) {
	  ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
		       "mod_backhand: cannot setsockopt \
IP_MULTICAST_TTL for server statistic multicasting.");
	}
	exit(0);
      }

    }
    if(rsd[i]>0)
      FD_SET(rsd[i], &mask);
  }
  /* Start it off with a transmission */
  initstat();
  memset(&mystat, 0, sizeof(serverstat));
  fillstat(&mystat, s, sins[0].from, sins[0].webport);
  /* Convert it to network endian */
  memcpy(serverstats, &mystat, sizeof(serverstat));
  serverstats[0].numbacked = serverstats[0].tatime = 0;
  htonss(mystat);
  sendtomany(ssd, &mystat, (StatsSI *)sins, nsins);
  setnowplusquantum(&next);
  /* send it away */
  while(sillycompiler) { /* is *always* 1. We know this. compiler doesn't */
    selectret=0;
    read_mask = mask;
    except_mask = mask;
    FD_ZERO(&dummy_mask);
    calc_time(&towait, &next);
    if(((towait.tv_sec + towait.tv_usec) > 0) &&
       (selectret=select(FD_SETSIZE,
			 &read_mask, &dummy_mask, &dummy_mask,
			 &towait)>0)) {
      for(i=0;i<nsins;i++) {
	if(rsd[i]>0 && FD_ISSET(rsd[i], &read_mask)) {
	  struct sockaddr_in sfrom;
	  unsigned int sasize;
	  /* receive a transmission and file it away */
	  sasize = sizeof(struct sockaddr_in);
	  recvfrom(rsd[i], &astat, sizeof(serverstat), 0,
		   (struct sockaddr *)&sfrom, &sasize);
	  /* convert all data to host endian */
	  ntohss(astat);
	  /* Fill out stuff that we determine */
	  if(astat.contact.sin_addr.s_addr == 0)
	    memcpy(&astat.contact.sin_addr, &sfrom.sin_addr,
		   sizeof(struct in_addr));
	  astat.mtime = time(NULL);
	  /* Copy it into the shared memory... if it is already there */
	  for(j=-1, i=0; i<MAXSERVERS; i++) { /* as if has been inserted */
	    if((memcmp(&(serverstats[i].contact.sin_addr),
		      &astat.contact.sin_addr,
		      sizeof(struct in_addr))==0) &&
	       (serverstats[i].contact.sin_port == astat.contact.sin_port)) {
	      memcpy(&serverstats[i], &astat, sizeof(serverstat));
	      j=-1; /* has been inserted */
	      break;
	    }
	    if(j==-1 && serverstats[i].contact.sin_addr.s_addr==0)
	      j=i; /* lowest available insertion point */
	  }
	  if(j>=0) {
	    if(matchACL(acl, &sfrom.sin_addr)!=NULL) {
	      /* NOTE: ACL against the sfrom NOT astat.contact
		 there is free space AND it fits the ACL */
	      memcpy(&serverstats[j], &astat, sizeof(serverstat));
	    } else {
	      if(loglevel & MBLL_NET2) {
		ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, NULL,
			     "mod_backhand: cannot accept stats from %s.",
			     inet_ntoa(astat.contact.sin_addr));
	      }
	    }
	  }
	}
      }
      if(FD_ISSET(ufd, &read_mask)) {
	/* an APACHE child wants to talk */
	int newfd;
	unsigned int newpid=0;
	if((newfd = serv_accept(ufd, &newpid)) < 0) {
	  if(loglevel & MBLL_NET3) {
	    ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			 "mod_backhand: cannot accept on %s.",
			 ud_fn);
	  }
	} else {
	  /* successful accept */
	  if(birth_control()>=0) {
	    add_child_fd(newfd, newpid);
	    FD_SET(newfd, &mask); /* Another whiner to listen to */
	    if(loglevel & MBLL_NET1) {
	      ap_log_error(APLOG_MARK, APLOG_NOTICE|APLOG_NOERRNO, NULL,
			   "mod_backhand: child at %d:%d.",
			   newfd, newpid);
	    }
	  } else {
	    /* This really shouldn't happen as there can only be
	       HARD_SERVER_LIMIT children at one time, did we not
	       clean up properly? */
	    if(loglevel & MBLL_NET3) {
	      ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			   "mod_backhand: afterbirth problem.(%d)",
			   newfd);
	    }
	    close(newfd);
	  }
	}
      }
      /* Not a gossip.... not a birth... AH HA! Child care? */
      for(i=0; i<HARD_SERVER_LIMIT; i++) {
	/* Check on each child */
	if(children[i].fd < 0) continue;
	if(FD_ISSET(children[i].fd, &read_mask)) {
	  char chreq[512]; /* Child request */
	  int nread;
	  struct in_addr servreq;
	  if((nread = read(children[i].fd, chreq,
			   MBCSP_REQ_SIZE))<0) {
	    if(loglevel & MBLL_NET2) {
	      ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			   "mod_backhand: Read error from child %d.",
			   children[i].pid);
	    }
	    close(children[i].fd);
	    children[i].fd = children[i].pid = -1;
	  } else if(nread == 0) {
	    if(loglevel & MBLL_NET1) {
	      ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			   "mod_backhand: Child %d disconnected.",
			   children[i].pid);
	    }
	    FD_CLR(children[i].fd, &mask);
	    close(children[i].fd);
	    children[i].fd = children[i].pid = -1;
	  } else if(nread != MBCSP_REQ_SIZE) {
	    if(loglevel & MBLL_NET2) {
	      ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			   "mod_backhand: Protocol deviation from child %d.",
			   children[i].pid);
	    }
	    FD_CLR(children[i].fd, &mask);
	    close(children[i].fd);
	    children[i].fd = children[i].pid = -1;
	  } else {
	    /* Child is giving us or asking for a session */
	    memcpy(&servreq, chreq+1, sizeof(struct in_addr));
	    if(*chreq == MBCSP_GET) {
	      /* We need to send them a session */
	      int reqfd;
	      reqfd = reserve_session(&servreq, children[i].pid);
	      send_fd(children[i].fd, reqfd);
              close(reqfd);
	      if(loglevel & MBLL_NET1) {
		ap_log_error(APLOG_MARK, APLOG_NOTICE|APLOG_NOERRNO, NULL,
			     "mod_backhand: Sending fd(%d).",
			     reqfd);
	      }
	    } else if(*chreq == MBCSP_PUT) {
	      /* We are getting a session form them */
	      int recvdfd;
	      if(loglevel & MBLL_NET1) {
		ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, NULL,
			     "mod_backhand: Receiving back fd(??).");
	      }
	      if((recvdfd = recv_fd(children[i].fd))>0)
		replace_session(&servreq, recvdfd);
	      if(loglevel & MBLL_NET1) {
		ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, NULL,
			     "mod_backhand: Received back fd(%d).",
			     recvdfd);
	      }
	    } else if(*chreq == MBCSP_CLOSE) {
	      if(loglevel & MBLL_NET2) {
		ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			     "mod_backhand: child %d requests disconnect.",
			     children[i].pid);
	      }
	      FD_CLR(children[i].fd, &mask);
	      close(children[i].fd);
	      children[i].fd = children[i].pid = -1;
	    } else {
	      if(loglevel & MBLL_NET2) {
		ap_log_error(APLOG_MARK, APLOG_ERR, NULL,
			     "mod_backhand: Protocol deviation from child %d.",
			     children[i].pid);
	      }
	      FD_CLR(children[i].fd, &mask);
	      close(children[i].fd);
	      children[i].fd = children[i].pid = -1;
	    }		
	  } 
	}
      }
    } else if(selectret==0) {
      /* timeout -- send our transmission */
      memset(&mystat, 0, sizeof(serverstat));
      fillstat(&mystat, s, sins[0].from, sins[0].webport);
      /* Convert it to network endian */
      mystat.numbacked = serverstats[0].numbacked;
      mystat.tatime = serverstats[0].tatime;
      htonss(mystat);
      sendtomany(ssd, &mystat, (StatsSI *)sins, nsins);
      setnowplusquantum(&next);
    }
  }
  return 0; /* Make some compilers happy... sillycompilers */
}
