-------------------------------------------------------------------------------
--                                                                           --
--  Ada Interface to the X Window System and Motif(tm)/Lesstif               --
--  Copyright (c) 1996-2000 Hans-Frieder Vogt                                --
--                                                                           --
--  This program is free software; you can redistribute it and/or modify     --
--  it under the terms of the GNU General Public License as published by     --
--  the Free Software Foundation; either version 2 of the License, or        --
--  (at your option) any later version.                                      --
--                                                                           --
--  This program is distributed in the hope that it will be useful,          --
--  but WITHOUT ANY WARRANTY; without even the implied warranty of           --
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     --
--  See the GNU General Public License for more details.                     --
--                                                                           --
--  You should have received a copy of the GNU General Public License        --
--  along with this program; if not, write to the                            --
--  Free Software Foundation, Inc.,                                          --
--  59 Temple Place - Suite 330,                                             --
--  Boston, MA 02111-1307, USA.                                              --
--                                                                           --
--                                                                           --
--  X Window System is copyrighted by the X Consortium                       --
--  Motif(tm)       is copyrighted by the Open Software Foundation, Inc.     --
--                                                                           --
--                                                                           --
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
--
-- HISTORY:
-- 11.01.97 H.-F. Vogt: name the newly created Ada-File after the xpm-file
-- 25.04.98 H.-F. Vogt: renamed program to xpm2axpm, adapted it to the changed
--                      xpm_lib of adabindx version 0.6
--                      => Version 1.2
-- 06.05.98 H.-F. Vogt: corrected wrongly typed Pixmap_Data_Type
--                      => Version 1.2.1
--
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
--
-- take a xpm-file and produce an output includeable (with-able) in Ada-source
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1,
     Ada.Command_Line,
     Ada.Text_Io,
     Interfaces.C.Strings,
     String_List,
     Xpm_Lib;
use  Ada.Text_Io,
     String_List;
procedure Xpm2AXpm is


   Version  : constant String := "1.2.1";
   Progname : constant String := Ada.Command_Line.Command_Name;

   Data : String_List.Element_Access_List;

   procedure Print_Info is
   begin
      Set_Output (Standard_Error);
      New_Line;
      Put_Line (
"Xpm2Ada-Xpm V" & Version & " -- (c)1996-1998 Hans-Frieder Vogt");
      Put_Line (
"converts XPM pixmap files into a form 'with'-able in Ada-programs");
      New_Line;
      Set_Output (Standard_Output);
   end Print_Info;

   procedure Print_Usage is
   begin
      Set_Output (Standard_Error);
      New_Line;
      Put_Line (
"   Usage:");
      Put_Line (
"          " & Progname & " Xpm-File > Xpm-In-Ada-File");
      New_Line;
      Set_Output (Standard_Output);
   end Print_Usage;


   function Find_Pixmap_Name (Arg : in String) return String is
      Last_Point_Pos : Natural := Arg'Last+1;
      Last_Slash_Pos : Natural := Arg'First-1;
   begin
      for I in reverse Arg'Range loop
         if Arg (I) = '.' and then Last_Point_Pos = Arg'Last+1 then
            Last_Point_Pos := I;
         elsif Arg (I) = '/' then
            Last_Slash_Pos := I;
            exit;
         end if;
      end loop;
      return Arg (Last_Slash_Pos+1 .. Last_Point_Pos-1) & "_Xpm";
   end Find_Pixmap_Name;


   function Remove_Tabulators (Str : in String) return String is
      Return_Str : String (Str'Range) := Str;
   begin
      for I in Return_Str'Range loop
         if Return_Str (I) = Ada.Characters.Latin_1.HT then
            Return_Str (I) := ' ';
         end if;
      end loop;
      return Return_Str;
   end Remove_Tabulators;


begin
   Print_Info;
   if Ada.Command_Line.Argument_Count < 1 then
      Put_Line (Standard_Error, Progname & ": ERROR: not enough parameters");
      Print_Usage;
      Ada.Command_Line.Set_Exit_Status (Ada.Command_Line.Failure);
      return;
   else
      begin
         Xpm_Lib.Xpm_Read_File_To_Data (Ada.Command_Line.Argument (1), Data);
      exception
         when Xpm_Lib.Xpm_Error_Open_Failed =>
            Put_Line (Standard_Error, Progname & ": ERROR: Couldn't open file " & Ada.Command_Line.Argument (1));
            Ada.Command_Line.Set_Exit_Status (Ada.Command_Line.Failure);
            return;
         when Xpm_Lib.Xpm_Error_File_Invalid =>
            Put_Line (Standard_Error, Progname & ": ERROR: Are you sure the file is an XPM-File?");
            Ada.Command_Line.Set_Exit_Status (Ada.Command_Line.Failure);
            return;
         when Xpm_Lib.Xpm_Error_No_Memory =>
            Put_Line (Standard_Error, Progname & ": ERROR: Not enough memory");
            Ada.Command_Line.Set_Exit_Status (Ada.Command_Line.Failure);
            return;
         when others =>
            raise;
      end;
   end if;

   declare
      Pix_Name : constant String := Find_Pixmap_Name (Ada.Command_Line.Argument (1));
   begin

      -- first the header
      Put_Line ("with String_List,");
      Put_Line ("     Xpm_Lib;");
      Put_Line ("use  String_List;");
      Put_Line ("package " & Pix_Name & " is");
      New_Line;
      Put_Line ("   Pix : Xpm_Lib.Pixmap_Data_Type :="); 
      Put_Line ("         Xpm_Lib.Null_Pixmap_Data &");

      -- now the data lines
      for I in 1 .. Length (Data)-1 loop
         Put_Line ("""" & Remove_Tabulators (Element (Data, I)) & """ &");
      end loop;
      Put_Line ("""" & Remove_Tabulators (Element (Data, Length (Data))) & """;");

      -- Trailer
      New_Line;
      Put_Line ("end " & Pix_Name & ";");

   end;

   Ada.Command_Line.Set_Exit_Status (Ada.Command_Line.Success);

end Xpm2AXpm;
