/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2003 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 */

#include "gtk-all.h"
#include <gdk/gdkkeysyms.h>
#include <gdk/gdktypes.h>
#include <stdio.h>
#include "util.h"
#include "conf.h"
#include "jamdoc.h"

typedef struct {
	GtkWindow *win;
	JamDoc *doc;
	gint sel_type, clip_type;
	gchar *sel_input, *clip_input;
} LinkRequest;

#define JAM_MAGIC_LINK_WINDOW 50 /* ms */

/* selection types */
#define JAM_SELECTION_HAS_TEXT (1 << 0)
#define JAM_SELECTION_HAS_URL  (1 << 1)

static void
select_toggled(GtkToggleButton *tb, GtkWidget *focus) {
	if (gtk_toggle_button_get_active(tb))
		gtk_widget_grab_focus(focus);
}

static void
entry_focus_cb(GtkWidget *w, GdkEventFocus *e, GtkToggleButton *radio) {
	gtk_toggle_button_set_active(radio, TRUE);
}

static GtkWidget*
radio_option(GSList *g, GtkWidget **radio, GtkWidget **entry,
		const char *caption, const char *label, const char *initial) {
	GtkWidget *vbox;

	vbox = gtk_vbox_new(FALSE, 0);
	*radio = gtk_radio_button_new_with_mnemonic(g, caption);
	gtk_box_pack_start(GTK_BOX(vbox), *radio, FALSE, FALSE, 0);

	*entry = gtk_entry_new();
	gtk_entry_set_activates_default(GTK_ENTRY(*entry), TRUE);
	g_signal_connect(G_OBJECT(*entry), "focus-in-event",
			G_CALLBACK(entry_focus_cb), *radio);
	if (initial)
		gtk_entry_set_text(GTK_ENTRY(*entry), initial);
	gtk_widget_set_size_request(*entry, 75, -1);
	g_signal_connect(G_OBJECT(*radio), "toggled",
			G_CALLBACK(select_toggled), *entry);
	g_signal_connect(G_OBJECT(*radio), "activate",
			G_CALLBACK(select_toggled), *entry);

	if (label) {
		GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(hbox), 
				gtk_label_new(label),
				FALSE, FALSE, 0);
		gtk_box_pack_start(GTK_BOX(hbox), *entry, TRUE, TRUE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	} else {
		gtk_box_pack_start(GTK_BOX(vbox), *entry, TRUE, TRUE, 0);
	}

	return vbox;
}

#if !GLIB_CHECK_VERSION(2,2,0)
static gboolean
g_str_has_prefix(const gchar *str, const gchar *prefix) {
	return (strncmp(str, prefix, strlen(prefix)) == 0);
}
#endif

gchar*
link_magic(LinkRequest *lr) {
	gchar *a, *b;
	if (! ((lr->clip_type | lr->sel_type) & JAM_SELECTION_HAS_URL))
		return NULL;
	
	xml_escape(&lr->clip_input);

	/* boo for no list primitives in c */
	if (lr->clip_type & JAM_SELECTION_HAS_URL) {
		a = lr->clip_input; b = lr->sel_input;
	} else {
		b = lr->clip_input; a = lr->sel_input;
	}

	return g_strdup_printf("<a href=\"%s\">%s</a>", a, b);
}

gchar*
link_interactive(LinkRequest *lr) {
	GtkWidget *dlg;
	GtkWidget *vbox;

	GtkWidget *etext, *ruser, *euser, *rurl, *eurl;
	GtkWidget *subbox;
	GSList *rgroup;
	char *url, *text, *user, *link = NULL;

	JamAccount *acc = jam_doc_get_account(lr->doc);
	
	dlg = gtk_dialog_new_with_buttons(_("Make Link"),
			lr->win, GTK_DIALOG_MODAL,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OK, GTK_RESPONSE_OK,
			NULL);
	gtk_dialog_set_default_response(GTK_DIALOG(dlg), GTK_RESPONSE_OK);

	vbox = gtk_vbox_new(FALSE, 10);

	etext = gtk_entry_new();
	gtk_entry_set_activates_default(GTK_ENTRY(etext), TRUE);
	gtk_entry_set_text(GTK_ENTRY(etext),
			((lr->sel_type & JAM_SELECTION_HAS_TEXT) & ~JAM_SELECTION_HAS_URL)
			? lr->sel_input : "");
	subbox = labelled_box_new(_("Link _Text:"), etext);
	gtk_box_pack_start(GTK_BOX(vbox), subbox, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), 
			radio_option(NULL, &rurl, &eurl, _("_URL:"), NULL,
				(lr->sel_type & JAM_SELECTION_HAS_URL) ? lr->sel_input : ""),
			FALSE, FALSE, 0);

	rgroup = gtk_radio_button_get_group(GTK_RADIO_BUTTON(rurl));

	if (JAM_ACCOUNT_IS_LJ(acc)) {
		gtk_box_pack_start(GTK_BOX(vbox), 
				radio_option(rgroup, &ruser, &euser, 
					_("_LiveJournal User:"), ".../users/", ""),
				FALSE, FALSE, 0);
	}

	jam_dialog_set_contents(GTK_DIALOG(dlg), vbox);
	if (gtk_dialog_run(GTK_DIALOG(dlg)) != GTK_RESPONSE_OK) {
		gtk_widget_destroy(dlg);
		return NULL;
	}

	url  = gtk_editable_get_chars(GTK_EDITABLE(eurl),  0, -1);
	xml_escape(&url);
	user = gtk_editable_get_chars(GTK_EDITABLE(euser), 0, -1);
	xml_escape(&user);
	text = gtk_editable_get_chars(GTK_EDITABLE(etext), 0, -1);
	xml_escape(&text);

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(rurl))) {
		/* build a "url" link. */
		link = g_strdup_printf("<a href=\"%s\">%s</a>", url, text);
	} else if (ruser && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ruser))) {
		/* build a "friends" link. */
		g_free(url);
		url = g_strdup(jam_account_lj_get_server(JAM_ACCOUNT_LJ(acc))->url);
		xml_escape(&url);
		link = g_strdup_printf("<a href=\"%s/users/%s\">%s</a>",
				url, user, text);
	} 
	g_free(url);
	g_free(user);
	g_free(text);
	
	gtk_widget_destroy(dlg);
	return link;
}
static gint
get_type(const gchar *str) {
	gint type = 0;
	if (str == NULL)
		return 0;
	type |= JAM_SELECTION_HAS_TEXT;
	if (g_str_has_prefix(str, "http://"))
		type |= JAM_SELECTION_HAS_URL;

	return type;
}

/* The dialog actually gets suppressed in the "magic" case where
 * information from the X clipboard and the main entry selection
 * can be used together to infer a URL. In that case, the link is
 * injected directly into the entry without any special GUI. */
void
link_dialog_run(GtkWindow *win, JamDoc *doc) {
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gchar *link = NULL;

	LinkRequest lr = {0};
	lr.win = win;
	lr.doc = doc;
	lr.sel_input = g_strdup("");

	buffer = jam_doc_get_text_buffer(doc);
	if (gtk_text_buffer_get_selection_bounds(buffer, &start, &end)) {
		string_replace(&lr.sel_input,
				gtk_text_buffer_get_text(buffer, &start, &end, FALSE));
		lr.sel_type = get_type(lr.sel_input);
	}
	
	lr.clip_input = jam_clipboard_wait_for_text_timeout(
		gtk_clipboard_get(GDK_SELECTION_CLIPBOARD),
		JAM_MAGIC_LINK_WINDOW);
	lr.clip_type  = get_type(lr.clip_input);

	if ((link = link_magic(&lr)) || (link = link_interactive(&lr))) {
		gtk_text_buffer_begin_user_action(buffer);
		gtk_text_buffer_delete(buffer, &start, &end);
		gtk_text_buffer_insert(buffer, &start, link, -1);
		gtk_text_buffer_end_user_action(buffer);
	}
	g_free(lr.sel_input);
	g_free(lr.clip_input);
	g_free(link);
}

static GtkWidget*
add_menu_item(GtkMenuShell *ms, const gchar *id, const gchar *text) {
	GtkWidget *hbox;
	GtkWidget *item;

	hbox = gtk_hbox_new(FALSE, 3);
	gtk_box_pack_start(GTK_BOX(hbox), 
			gtk_image_new_from_stock(id, GTK_ICON_SIZE_MENU), 
			FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), 
			gtk_label_new(text), FALSE, FALSE, 0);

	item = gtk_menu_item_new();
	gtk_container_add(GTK_CONTAINER(item), hbox);
	gtk_menu_shell_append(ms, item);
	return item;
}

static GtkWidget*
make_usertype_omenu() {
	GtkWidget *omenu, *menu;

	menu = gtk_menu_new();
	add_menu_item(GTK_MENU_SHELL(menu), "logjam-ljuser", _("User"));
	add_menu_item(GTK_MENU_SHELL(menu), "logjam-ljcomm", _("Community"));

	omenu = gtk_option_menu_new();
	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	return omenu;
}

void
link_jam_dialog_run(GtkWindow *win, JamDoc *doc) {
	GtkWidget *dlg;
	GtkWidget *vbox, *hbox, *entry, *omenu;
	GtkSizeGroup *sizegroup;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	char *username = NULL;
	int usertype;
	gboolean selection = FALSE;

	buffer = jam_doc_get_text_buffer(doc);
	if (gtk_text_buffer_get_selection_bounds(buffer, &start, &end)) {
		username = gtk_text_buffer_get_text(buffer, &start, &end, FALSE);
		selection = TRUE;
	}

	dlg = gtk_dialog_new_with_buttons(_("Insert lj user / lj community Tag"),
			win, GTK_DIALOG_MODAL,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OK, GTK_RESPONSE_OK,
			NULL);
	gtk_dialog_set_default_response(GTK_DIALOG(dlg), GTK_RESPONSE_OK);
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

	sizegroup = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

	entry = gtk_entry_new();
	gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);
	if (username) {
		gtk_entry_set_text(GTK_ENTRY(entry), username);
		g_free(username);
	}
	hbox = labelled_box_new_sg(_("_Username:"), entry, sizegroup);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	omenu = make_usertype_omenu();
	hbox = labelled_box_new_sg(_("User _Type:"), omenu, sizegroup);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	jam_dialog_set_contents(GTK_DIALOG(dlg), vbox);

	if (gtk_dialog_run(GTK_DIALOG(dlg)) != GTK_RESPONSE_OK) {
		gtk_widget_destroy(dlg);
		return;
	}
	username = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);
	usertype = gtk_option_menu_get_history(GTK_OPTION_MENU(omenu));
	gtk_widget_destroy(dlg);
	if (username[0] == 0) {
		g_free(username);
		return;
	}

	if (selection)
		gtk_text_buffer_delete(buffer, &start, &end);
	else
		gtk_text_buffer_get_iter_at_mark(buffer, &start,
				gtk_text_buffer_get_insert(buffer));

	gtk_text_buffer_insert(buffer, &start, "<lj ", -1);
	if (usertype == 0)
		gtk_text_buffer_insert(buffer, &start, "user=\"", -1);
	else
		gtk_text_buffer_insert(buffer, &start, "comm=\"", -1);
	xml_escape(&username);
	gtk_text_buffer_insert(buffer, &start, username, -1);
	g_free(username);
	gtk_text_buffer_insert(buffer, &start, "\" />", -1);
}

