#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/types.h>
#include <string.h>
#include <time.h>
#include <sys/wait.h>
#include <signal.h>

#include "config.h"

float loadAvg();

pid_t           child;
pid_t           children[100];
int             copies = 1;

void sighandler()
{
    if (child)
    {
        kill(child, SIGTERM);
        sleep(2);
        exit(0);
    }
    else
    {
        int i;
        
        for (i = 0; i < copies; i++)
        {
            kill(children[i], SIGTERM);
        }
    }
}

void
usage()
{
    fprintf(stderr, "loadwatch [-d <int>] [-h <float>] [-l float] -- command\n");
    fprintf(stderr, "\t-d <int>\tload sampling interval (10 seconds)\n");
    fprintf(stderr, "\t-h <float>\thigh load mark (1.25)\n");
    fprintf(stderr, "\t-l <float>\tlow load mark (0.25)\n");
}

main(int argc, char **argv)
{
    int         delay = 10;
    float       highmark = 1.25;
    float       lowmark = 0.25;
    int         c;
    pid_t       childgroup;
    struct sigaction   handler;
    
    while ((c = getopt(argc, argv, "d:l:h:c:")) != -1)
    {
        switch (c)
        {
        case 'c':
            copies = atoi(optarg);
            break;
            
        case 'd':
            delay = atoi(optarg);
            break;

        case 'h':
            highmark = atof(optarg);
            break;

        case 'l':
            lowmark = atof(optarg);
            break;
            
        default:
            usage();
            exit(1);
        }
    }

    handler.sa_handler = &sighandler;
    sigaction(SIGTERM, &handler, 0);
    sigaction(SIGINT, &handler, 0);
    
    if (copies < 1)
    {
        fprintf(stderr, "must run 1 or more copies\n");
        exit(1);
    }

    if (optind == argc)
    {
        usage();
        printf("current load: %g\n", loadAvg());
        exit(1);
    }
    
    if ((child = fork()) == 0)
    {
        int i;

        setpgrp();

        for (i = 0; i < copies; i++)
        {
            if ((children[i] = fork()) == 0)
                execv(argv[optind], argv + optind);
        }
        
        while (1)
        {
            pid_t       status;
            time_t      now;
            char        *outline;
            
            now = time(0);
            outline = ctime(&now);
            *(strchr(outline, '\n')) = 0;
            
            switch(status = waitpid(0, 0, 0))
            {
            case -1:
            case 0:
                break;

            default:
                fprintf(stderr, "%s (%d): child %d finished.\n", outline,
                        getpid(), status);
                if (--copies == 0)
                    exit(0);
            }
        }
    }
    else
    {
        int     state = 0;
        char    line[80];
        char    *outline;
        pid_t   status;
        float   avg;
        float   loadAvg();
        
        sleep(1);
        
        childgroup = getpgid(child);
        
        while (1)
        {
            time_t  now = time(0);
            outline = ctime(&now);
            *(strchr(outline, '\n')) = 0;

            switch (status = waitpid(child, 0, WNOHANG))
            {
            case -1:
                perror("problems waiting for child");
                kill(-childgroup, SIGTERM);
                exit(1);
                break;

            case 0:
                break;

            default:
                fprintf(stderr, "%s: no child process, exiting.\n", outline);
                exit(0);
                break;
            }

            avg = loadAvg();
            //fprintf(stderr, "load == %g\n", avg);
            if (state == 0 && avg >= highmark)
            {
                fprintf(stderr, "%s: load too high, stopping.\n", outline);
                state = 1;
                kill(-childgroup, SIGSTOP);
            }
            else if (state == 1 && avg <= lowmark)
            {
                fprintf(stderr, "%s: load low, continuing.\n", outline);
                state = 0;
                kill(-childgroup, SIGCONT);
            }
            
            sleep(delay);
        }
    }
}

#ifdef linux
float
loadAvg()
{
    float       avg = 0.0;
    FILE        *file;
    char        line[81];
    
    file = fopen("/proc/loadavg", "r");
    if (fgets(line, 80, file))
    {
        avg = atof(line);
    }
    fclose(file);
    return avg;
}
#elif defined(HAVE_KSTAT)
#include <kstat.h>
#include <sys/param.h>

static kstat_ctl_t *kc;

float loadAvg()
{
    kstat_t       *ks;
    kstat_named_t *kn;

    if (!kc)
    {
        if (!(kc = kstat_open()))
        {
            perror("opening kstat");
            return 0.0;
        }
    }
    
    ks = kstat_lookup(kc, "unix", 0, "system_misc");
    if (kstat_read(kc, ks, 0) == -1)
    {
        perror("kstat_read");
        return 0.0;
    }

    kn = kstat_data_lookup(ks, "avenrun_1min");
    if (kn)
        return (float)kn->value.ul/FSCALE;
    else
    {
        perror("grabbing avenrun_1min");
        return 0.0;
    }
}

#else
float
loadAvg()
{
    float       avg = 0.0;
    FILE        *file;
    char        line[81], *p;
    
    file = popen("uptime", "r");
    if (fgets(line, 80, file) && (p = strstr(line, "average:")))
    {
        avg = atof(p + 9);
    }
    pclose(file);
    return avg;
}

#endif
