/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */



/*!
    \page lcmaps_verify_proxy.mod Verification plugin plugin
 
    \section posixsynopsis SYNOPSIS
    \b lcmaps_verify_proxy.mod
 
    \section posixdesc DESCRIPTION
 
The Verify Proxy module will verify the validity of the proxy (with its chain) or any other X.509 certificate chain that has been used in the LCMAPS framework.
The module will report to the LCMAPS framework the validity of the chain.

 
*/


/*!
    \file   lcmaps_verify_proxy.c
    \brief  Plugin to verify the proxy (and its chain) on its validity.
    \author Oscar Koeroo for the EGEE.
*/





/*****************************************************************************
                            Include header files
******************************************************************************/

#include "proxylifetime.h"


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <pwd.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/x509_vfy.h>
#include <openssl/asn1.h>
#include <openssl/bio.h>
#include <time.h>




/******************************************************************************
                                Definitions
******************************************************************************/

 
/******************************************************************************
                          Module specific prototypes
******************************************************************************/


/******************************************************************************
                       Define module specific variables
******************************************************************************/


/*****************************************************************************/




/******************************************************************************
Function:   verifyLifeTimeProxy
Description:
    Trys to verify the LifeTimes within the proxyies to let this comply with policy
Parameters:
    STACK_OF(X509) * myChain
Returns:
    1   : succes: verify ok
    0   : verify failed
******************************************************************************/
int verifyProxyLifeTime (struct TProxyLevelTTL * plt, STACK_OF(X509) *chain, int depth)
{
    int         i = 0;
/*    int         depth = sk_X509_num (chain); */
    int         amount_of_CAs = 0;
    time_t      delta_time = 0;
   /*
    int         days  = 0;
    int         hours = 0;
    int         minutes = 0;
    */
    X509 *      cert = NULL;
    char *      dn_buffer = NULL;
    int         proxyLevel = 0;
    int         proxyType = 0;
    time_t      levelTime = 0;
    time_t      notAfter  = 0;
    time_t      notBefore = 0;



    /* How many CA certs are there in the chain? */
    for (i = 0; i < depth; i++)
    {
        if (grid_x509IsCA(sk_X509_value(chain, i)))
            amount_of_CAs++;
    }

    /* Gimme some scratch bat */
    dn_buffer = malloc (sizeof(char) * 256);

    /* 
     * Changed this value to start checking the proxy and such
     * ->  This means I start working after the CA(s) and user_cert
     *
     * Start at proxyLevel 0
     *
     */

    for (i = depth - (amount_of_CAs + 2); i >= 0; i--)
    {
        lcmaps_log_debug (1, "%s: checking proxy level: %d of depth %d\n", __func__, i, depth);

        /* Init cycle */
        proxyType = UNDEFINED_PROXY;

        /* Check for X509 certificate and point to it with 'cert' */
        if ((cert = sk_X509_value(chain, i)) != NULL)
        {
            /* The chain descend result in first being confronted with a MYPROXY_PROXY (not necessarily a proxy created with myproxy) */
            if (proxyLevel == 0) 
                proxyType = MYPROXY_PROXY; 
            else 
                proxyType = INNER_PROXY;

            /* Overruling the proxyType when it is a LEAF_PROXY (ending the chain) */
            if (i == 0)
                proxyType = LEAF_PROXY;

            X509_NAME_oneline(X509_get_subject_name(cert), dn_buffer, 256);
            lcmaps_log_debug (2, "%s: Current cert: %s\n", __func__, dn_buffer);
            
            notAfter  = asn1TimeToTimeT ((char *) ASN1_STRING_data (X509_get_notAfter(cert)));
            notBefore = asn1TimeToTimeT ((char *) ASN1_STRING_data (X509_get_notBefore(cert)));


            /* Delta Time is the amount of seconds between the two fixed time points notAfter and notBefore */
            delta_time = notAfter - notBefore;

            lcmaps_log_debug (2, "%s: Valid time period (Proxy LifeTime): %d hours, %d minutes en %d seconds\n", 
                                __func__,
                                delta_time / 3600, 
                                (delta_time % 3600) / 60, 
                                (delta_time % 3600) % 60);

           
 
            /* 
             * Retrieve the maximum proxy life time at the current level from the internal data structure (created a plugin init stage)
             * 
             * var i is counting backwards to zero.
             * if i == 0 then we are at the top of the chain or actually the lower ends of it, which is *always* indicated as the LEAF proxy
             * If i == 0 then we can safely retrieve the LEAF_PROXY proxyLevel info
             *
             * NOTE: The proxylevel which is being evaluated WILL BE overriden by the set LEAF_PROXY policy except when there is no LEAF_PROXY policy set.
             */
            if (i == 0) 
            {
                if ((levelTime = Search_TTL_By_Level(plt, LEAF_PROXY))) 
                {
                    lcmaps_log_debug (1, "%s: Overruling specific Proxy Level maximum TTL with leaf proxy policy. Leaf proxy found at Proxy Level %d\n", __func__, proxyLevel);
                }
                else
                {
                    lcmaps_log_debug (2, "%s: No policy for LEAF Proxy at Proxy Level %d. Trying policy for the current Proxy Level\n", __func__, proxyLevel);
                    
                    /* switching back from LEAF Proxy policy mode to current proxyLevel */
                    if ((levelTime = Search_TTL_By_Level(plt, proxyLevel)))
                    {
                        lcmaps_log_debug (5, "%s: Succesfully found config for Proxy level %d\n", __func__, proxyLevel);
                    }
                    else
                    {
                        lcmaps_log_debug (5, "%s:     No policy for Proxy level %d\n", __func__, proxyLevel);
                    }
                }
            }
            else
            {
                /* Search for a registered policy at proxyLevel */
                if ((levelTime = Search_TTL_By_Level(plt, proxyLevel)))
                {
                    lcmaps_log_debug (2, "%s: Succesfully found config for Proxy level %d\n", __func__, proxyLevel);
                }
                else
                {
                    lcmaps_log_debug (2, "%s: No policy for Proxy level %d\n", __func__, proxyLevel);
                }
            }


            /* Proxy LifeTime Level Check,
             * is zero, then no levelTime has been set in the initialization phase
             */
            if (levelTime != 0)
            {
                lcmaps_log_debug (2, "%s: Max Leveltime @ proxyLevel %d and proxytype %s: %d hours, %d minutes and %d seconds\n", 
                                     __func__,
                                     proxyLevel,
                                     proxyType == LEAF_PROXY ? "LEAF" : proxyType == INNER_PROXY ? "INNER" : proxyType == MYPROXY_PROXY ? "MYPROXY/FIRST" : "unknown type",
                                     levelTime / 3600, 
                                     (levelTime % 3600) / 60, 
                                     (levelTime % 3600) % 60);


                /* The level time is the _max_ time at that proxy level */
                if (delta_time <= levelTime)
                {
                    lcmaps_log_debug (1, "%s:     Proxy Life Time policy check approaved at Proxy Level %d.\n", __func__, proxyLevel);
                }
                else
                {
                    unsigned int dt, days, hours, min, sec;
                    unsigned int diff_dt_lt, diff_days, diff_hours, diff_min, diff_sec;

                    dt = delta_time;
                    days = dt / (3600 * 24);
                    dt = dt % (3600 * 24);
                    hours = dt / 3600;
                    dt = dt % 3600;
                    min = dt / 60;
                    dt = dt % 60;
                    sec = dt;

                    diff_dt_lt = delta_time - levelTime;
                    diff_days = diff_dt_lt / (3600 * 24);
                    diff_dt_lt = diff_dt_lt % (3600 * 24);
                    diff_hours = diff_dt_lt / 3600;
                    diff_dt_lt = diff_dt_lt % 3600;
                    diff_min = diff_dt_lt / 60;
                    diff_dt_lt = diff_dt_lt % 60;
                    diff_sec = diff_dt_lt;

                    /* Access/Policy Violation! */
                    lcmaps_log(LOG_ERR, "%s: Error: Proxy Life Time Violation. Proxy at level %d has a life time of '%d day(s), %d hour(s), %d min(s), %d sec(s)' which exceed the policy by '%d day(s), %d hour(s), %d min(s), %d sec(s)'.\n", 
                                    __func__,
                                    proxyLevel,
                                    days,
                                    hours,
                                    min,
                                    sec,
                                    diff_days,
                                    diff_hours,
                                    diff_min,
                                    diff_sec);
                    goto failure;
                }
            }
            else
            {
                /* No policy set */
                lcmaps_log_debug (5, "%s: No Proxy LifeTime check performed on this proxy level.\n", __func__);
            }

            /* kick proxy certificate level up by one and clean up */
            proxyLevel++;
            levelTime = 0;
        }
    }   


/* Success! */
    free(dn_buffer);
    return 1;


failure:
    free(dn_buffer);
    return 0;
}


/******************************************************************************
Function:   verifyVOMSLifeTime
Description:
            Verification of the VOMS Attributes' Life Time
Parameters:
            lcmaps_vomsdata_t * vomsdata
Returns:
    0    : succes
    !0   : failure
******************************************************************************/
int verifyVOMSLifeTime (struct TProxyLevelTTL * plt, lcmaps_vomsdata_t * lcmaps_vomsdata)
{
    char *      logstr = "verifyVOMSLifeTime()";
    int         i = 0;
    time_t      levelTime = 0;

    time_t      start;
    time_t      end;
    time_t      delta_time;

    time_t      now = time((time_t *)NULL);


    /* And there was (current) time... */
    time(&now);


    /* Do we have Any VOMS Attribs? */
    if (lcmaps_vomsdata)
    {
        for (i = 0; i < lcmaps_vomsdata->nvoms; i++)
        {
            start = asn1TimeToTimeT (lcmaps_vomsdata->voms[i].date1);
            end   = asn1TimeToTimeT (lcmaps_vomsdata->voms[i].date2);


            /* First sanity checks to see if the VOMS Attributes are valid at this moment in time */
            if (timeIsInBetween (now, start, end))
            {
                /* Performing VOMS LifeTime (Policy) check -- level '0' means the one and only VOMS Life Time Policy*/
                if ((levelTime = Search_TTL_By_Level(plt, 0)))
                {
                    /* lcmaps_log_debug (5, "\t%s: VOMS Attributes for the VO '%s' will now be checked against the VOMS LifeTime policy.\n", logstr, lcmaps_vomsdata->voms[i].voname); */
                    
                 
                    /* Delta Time is the amount of seconds between the two fixed time points notAfter and notBefore */
                    delta_time = end - start;


                    /* Check if the lifetime of the VOMS Attribs is bigger then the set policy (a bad thing) */
                    if (delta_time > levelTime)
                    {
                        lcmaps_log (LOG_NOTICE, "%s: Error: Proxy Life Time Violation. The VOMS Attributes for the VO '%s' exceed the set VOMS LifeTime policy of '%d hours, %d minutes en %d seconds' by '%d hours, %d minutes en %d seconds'\n",
                                        __func__,
                                        lcmaps_vomsdata->voms[i].voname,
                                        levelTime / 3600,
                                        (levelTime % 3600) / 60,
                                        (levelTime % 3600) % 60,
                                        (delta_time - levelTime) / 3600,
                                        ((delta_time -levelTime) % 3600) / 60,
                                        ((delta_time -levelTime) % 3600) % 60
                                   );
                        lcmaps_log_debug (5, "    %s: Lifetime of the VOMS Attributes for the VO '%s': %d hours, %d minutes en %d seconds\n",
                                          logstr,
                                          lcmaps_vomsdata->voms[i].voname,
                                          delta_time / 3600,
                                         (delta_time % 3600) / 60,
                                         (delta_time % 3600) % 60);

                        goto failure;
                    }
                    else
                    {
                        lcmaps_log_debug (3, "    %s: Ok: Lifetime of the VOMS Attributes for the VO '%s': '%d hours, %d minutes en %d seconds'. The set policy for the VOMS LifeTime: '%d hours, %d minutes en %d seconds.'\n",
                                          logstr,
                                          lcmaps_vomsdata->voms[i].voname,
                                          delta_time / 3600,
                                         (delta_time % 3600) / 60,
                                         (delta_time % 3600) % 60,
                                          levelTime / 3600,
                                         (levelTime % 3600) / 60,
                                         (levelTime % 3600) % 60);
   
                    }
                }
                else
                {
                    lcmaps_log_debug (1, "    %s: No VOMS Attribute Lifetime policy set to enforce, skipping VOMS Lifetime check.\n", logstr);
                }
            }
            else
            {
                if (now < asn1TimeToTimeT (lcmaps_vomsdata->voms[i].date1))
                    lcmaps_log (LOG_ERR, "    %s: VOMS Attributes for the VO '%s' are not valid yet!\n", logstr, lcmaps_vomsdata->voms[i].voname);

                if (now > asn1TimeToTimeT (lcmaps_vomsdata->voms[i].date2))
                    lcmaps_log (LOG_ERR, "    %s: VOMS Attributes for the VO '%s' are not valid anymore!\n", logstr, lcmaps_vomsdata->voms[i].voname);
 
                goto failure;
            }
        }
    }
    else
    {
        lcmaps_log_debug (3, "%s: No LCMAPS VOMS Data found, VOMS checks do not apply.\n", logstr);
    }


    return 1;

failure:
    return 0;

}


int timeIsInBetween (time_t this, time_t t1, time_t t2)
{
    time_t low, high;

    if (t1 > t2)
    {
        high = t1;
        low  = t2;
    }
    else
    {
        high = t2;
        low  = t1;
    }


    if (this < high && this > low)
        return 1;
    else
        return 2;
    
}


/******************************************************************************
Function:    print_lcmaps_vomsdata
Description:
             print the lcmaps_vomsdata structure
Parameters:
    lcmaps_vomsdata_t * lcmaps_vomsdata
Returns:
    void
******************************************************************************/
void print_lcmaps_vomsdata_local(lcmaps_vomsdata_t * lcmaps_vomsdata)
{
    char * logstr = "    print_lcmaps_vomsdata()";
    int    i      = 0;
    int    j      = 0;


    if (lcmaps_vomsdata)
    {
        for (i = 0; i < lcmaps_vomsdata->nvoms; i++)
        {
            lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[ %d / %d ]\n",                 logstr, i + 1, lcmaps_vomsdata->nvoms);
            lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].user_dn           : %s\n", logstr, i + 1, lcmaps_vomsdata->voms[i].user_dn);
            lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].user_ca           : %s\n", logstr, i + 1, lcmaps_vomsdata->voms[i].user_ca);
            lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].voms_issuer_dn    : %s\n", logstr, i + 1, lcmaps_vomsdata->voms[i].voms_issuer_dn);
            lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].voms_issuer_ca    : %s\n", logstr, i + 1, lcmaps_vomsdata->voms[i].voms_issuer_ca);
            lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].uri               : %s\n", logstr, i + 1, lcmaps_vomsdata->voms[i].uri);
            lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].date1             : %s\n", logstr, i + 1, lcmaps_vomsdata->voms[i].date1);
            lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].date2             : %s\n", logstr, i + 1, lcmaps_vomsdata->voms[i].date2);
            lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].voname            : %s\n", logstr, i + 1, lcmaps_vomsdata->voms[i].voname);

            for (j = 0; j < lcmaps_vomsdata->voms[i].nfqan; j++)
            {
                 lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].fqan_unix[ %d / %d ]\n",    logstr, i + 1, j + 1, lcmaps_vomsdata->voms[i].nfqan);
                 lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].fqan_unix[%d].fqan : %s\n", logstr, i + 1, j + 1, lcmaps_vomsdata->voms[i].fqan_unix[j].fqan);
                 lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].fqan_unix[%d].uid  : %d\n", logstr, i + 1, j + 1, lcmaps_vomsdata->voms[i].fqan_unix[j].uid);
                 lcmaps_log_debug (3, "%s: lcmaps_vomsdata->voms[%d].fqan_unix[%d].gid  : %d\n", logstr, i + 1, j + 1, lcmaps_vomsdata->voms[i].fqan_unix[j].gid);
            }
        }

        lcmaps_log_debug (3, "%s: lcmaps_vomsdata->workvo                    : %s\n", logstr, lcmaps_vomsdata->workvo);
        lcmaps_log_debug (3, "%s: lcmaps_vomsdata->extra_data                : %s\n", logstr, lcmaps_vomsdata->extra_data);
    }
    else
    {
        lcmaps_log_debug (3, "%s: No LCMAPS vomsdata found\n");
    }
}


/******************************************************************************
Function:   createTime
Description:
            Will construct a time structure from the time_t t;

NOTE -->    I know this is not good, but what the heck, 
            you shouldn't create a proxy of a life long span anyway, 
            mail me otherwise to convince me 
            -- Oscar Koeroo : 02-02-2006

NOTE 2 -->  The precision is capped on 'days'.
          
Parameters:
            time_t t
Returns:
    !NULL   : succes
    NULL    : failure
******************************************************************************/
/*
struct tm {
    int tm_sec;        seconds after the minute (from 0)
    int tm_min;        minutes after the hour (from 0)
    int tm_hour;       hour of the day (from 0)
    int tm_mday;       day of the month (from 1)
    int tm_mon;        month of the year (from 0)
    int tm_year;       years since 1900 (from 0)
    int tm_wday;       days since Sunday (from 0)
    int tm_yday;       day of the year (from 0)
    int tm_isdst;      Daylight Saving Time flag
} 
*/
struct tm * createTime (const time_t t)
{
    time_t timeleft = t;
    /* time_t item = 0; */
    struct tm * time_s = NULL;

    time_s = (struct tm *) malloc (sizeof (struct tm));

    /* Seconds */
    time_s->tm_sec = timeleft % 60;
    timeleft -= timeleft % 60; 

    if (time_s->tm_sec >= 60)
    {
        free (time_s);
        return NULL;
    }

    /* Minutes */
    time_s->tm_min = timeleft % 3600;
    timeleft -= timeleft % 3600;

    if (time_s->tm_min >= 60)
    {
        free (time_s);
        return NULL;
    }

    /* Hours */
    time_s->tm_hour = timeleft % (24 * 3600);
    timeleft -= timeleft % (24 * 3600);

    if (time_s->tm_hour >= 24)
    {
        free (time_s);
        return NULL;
    }

    /* Days */
    time_s->tm_mday = timeleft / (24 * 3600);

    return time_s;
}


/******************************************************************************
Function:   ASN1_GENERALIZEDTIME_2ilb
Description:
        converts a ASN1_TIME into a time_t
        (It's a little hack, but should work nice and neat).

Parameters:
        a ASN1_TIME, 

Returns:
        a time_t

******************************************************************************/
#if 0
static struct tm * ASN1_TIME_2_time_t(ASN1_TIME *asn1_time)
{
        //time_t MyTime = (time_t) 0;  
        int i;
        static struct tm newTimeSpace;

        if (asn1_time->length != 13) 
            return NULL;
            //return (time_t) -1;

        for (i = 0; i < asn1_time->length / 2; i++)
        {
            char tmp[3];
            
            tmp[0] = asn1_time->data[0 + (i*2)];
            tmp[1] = asn1_time->data[1 + (i*2)];
            tmp[2] = '\0';
            
            switch(i)
            {
                case 0 : if (atoi(tmp) > 69)
                              newTimeSpace.tm_year = atoi(tmp) + 1900;
                         else
                              newTimeSpace.tm_year = atoi(tmp) + 100;
                         break;
                case 1 : newTimeSpace.tm_mon  = atoi(tmp) - 1; break;
                case 2 : newTimeSpace.tm_mday = atoi(tmp); break;
                case 3 : newTimeSpace.tm_hour = atoi(tmp); break;
                case 4 : newTimeSpace.tm_min  = atoi(tmp); break;
                case 5 : newTimeSpace.tm_sec  = atoi(tmp); break;
            }
        }
        //MyTime = mktime(&newTimeSpace);
        //return MyTime;

        return (&newTimeSpace);
}
#endif

/******************************************************************************
Function:   sttm_2_char
Description:
        converts a struct tm into a humanly readable char * of the time it's
        presenting.
        The representation will be in a GMT time, not Local.
 
Parameters:
        time
 
Returns:
        The time in a humanly readle format (and compatible to MySQL) as known
        in the variations of GMT (GMT conversion of the time parameter seen as
        a localtime), LOCAL (plain old local time) or LEAVE_TIME_ITS_OK (this
        means you should not convert/change the time but just leave it, in
        reality this means give me the time as a localtime thus without
        conversion).
 
******************************************************************************/
char * sttm_2_char (const struct tm * MyDateTime)
{
    char  * datetime = malloc(sizeof(char) * 21);

    snprintf(datetime, 21, "%04d-%02d-%02d %02d:%02d:%02d",
             MyDateTime->tm_year + 1900, MyDateTime->tm_mon + 1, MyDateTime->tm_mday,
             MyDateTime->tm_hour, MyDateTime->tm_min, MyDateTime->tm_sec);
 
    return datetime;
}



/******************************************************************************
Function:    strnclean
Description: cleans a dirty string, forces '\0' on the bufsize of **s
Parameters:
     input  : char **s
     output : char **s (cleaned string and NULLified)
Returns:
    SUCCESS
    FAILURE
 
******************************************************************************/
int strnclean (char **s, int bufsize)
{
    int i = 0;
 
    if (s == NULL)
        return -1;
 
    if (*s == NULL)
        return -1; 
 
    for (i = 0; i < bufsize; i++)
    {
        (*s)[i] = '\0';
    }
 
    return 0;
}
 
 
/******************************************************************************
Function:    strclean
Description: cleans a dirty string, forces '\0' on the strlen() of **s
Parameters:
     input  : char **s
     output : char **s (cleaned string and NULLified)
Returns:
    SUCCESS
    FAILURE
 
******************************************************************************/
int strclean (char **s)
{
    return strnclean(s, strlen(*s));
}



/******************************************************************************
Function:    asn1TimeToTimeT
Description: converts an asn1Time to a time_t
Parameters:
     char * asn1time, len
Returns:
     time_t
******************************************************************************/
time_t asn1TimeToTimeT(char *asn1time)
{
   char   zone;
   struct tm time_tm;
   size_t len = 0;

   /* if (len == 0) len = strlen(asn1time); */
   len = strlen(asn1time);

   if ((len != 13) && (len != 15)) return 0; /* dont understand */

   if ((len == 13) &&
       ((sscanf(asn1time, "%02d%02d%02d%02d%02d%02d%c",
         &(time_tm.tm_year),
         &(time_tm.tm_mon),
         &(time_tm.tm_mday),
         &(time_tm.tm_hour),
         &(time_tm.tm_min),
         &(time_tm.tm_sec),
         &zone) != 7) || (zone != 'Z'))) return 0; /* dont understand */

   if ((len == 15) &&
       ((sscanf(asn1time, "20%02d%02d%02d%02d%02d%02d%c",
         &(time_tm.tm_year),
         &(time_tm.tm_mon),
         &(time_tm.tm_mday),
         &(time_tm.tm_hour),
         &(time_tm.tm_min),
         &(time_tm.tm_sec),
         &zone) != 7) || (zone != 'Z'))) return 0; /* dont understand */

   /* time format fixups */

   if (time_tm.tm_year < 90) time_tm.tm_year += 100;
   --(time_tm.tm_mon);

   
   return my_timegm(&time_tm);
}


/*****************************************************************************/


/******************************************************************************
Function:   ttl_char2time_t
Description:
    converts a string of a special time format to a time_t
Parameters:
    a datetime string in the format:  2d-13:37
    which means a time period of 2 days, 13 hours and 37 minutes
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
time_t ttl_char2time_t (char * datetime)
{
    /*
     * Allowed notation:  [2m-][2d-]13:37
     */


     /* 
      * I know it's totally not clean, but it will be in the near future 
      */


    int    i = 0;
    int    do_days = 0;
    char * c = NULL;
    char * emitetad = NULL;
    int    minutes = 0;
    int    hours   = 0;
    int    days    = 0;
    /*int    months  = 0; */
    int    tot_sec = 0;
/*    int    years   = 0;  No you shouldn't want to wish to use this... */


    lcmaps_log_debug (2, "Proxy Time To Live parsing: %s\n", datetime);

    /*
     * extract time - which must be in a 24h notation!
     */
    if (strlen(datetime) < 4)
    {
        lcmaps_log(LOG_ERR, "Error: Parse error in datetime, implicit full 24h notation expected: range from 00:00 to 24:59, found: %\n", datetime);
        return -1;
    }

    /*
     * Flip datetime
     */

    /* Create a buffer */
    c = (char *) calloc (2, sizeof (char));

    emitetad = (char *) calloc (strlen(datetime) + 2, sizeof (char));
    for (i = 0; i < strlen(datetime); i++)
    {
        emitetad[i] = datetime[strlen(datetime) - 1 - i];
    }
    datetime[strlen(datetime)] = '\0';



    /*
     * Time extraction
     */

    for (i = 0; i < strlen(emitetad); i++)
    {
        c[0] = emitetad[i];

        // Minutes
        switch (i)
        {
            case 0:
            {
                minutes = atoi (c);
                break;
            }
            case 1:
            {
                minutes = minutes + 10 * atoi (c);
                break;
            }

            case 2: 
            {
                if (emitetad[i] == ':')
                    continue;
                else
                    return -1;
            }
            case 3:
            {
                hours = atoi (c);
                break;
            }
            case 4:
            {
                hours = hours + 10 * atoi (c);
                break;
            }
 
            case 5:
            {
                if (emitetad[i] == '-')
                    continue;
                else
                    return -1;
            }

            case 6:
            {
                if ( (emitetad[i] == 'd') || (emitetad[i] == 'D') )
                {
                    do_days = 1;
                    continue;
                }
                else
                    return -1;  /* This is not always true, also a month should be selectable, without a day specification */
            }
           
            case 7:
            {
                /* I know it's totally not clean, but it will be in the near future */
                if (!do_days)
                    return -1;
                else
                {
                    days = atoi(c);
                }
                break;
            }
            case 8:
            {
                /* I know it's totally not clean, but it will be in the near future */
                if (!do_days)
                    return -1;
                else
                {
                    days = days + 10 * atoi(c);
                }
                break;
            }

            default:
                return -1;
        }
    }

    /* Free buffers */
    free(emitetad);
    free(c);

    /* Calc seconds: */
    tot_sec = minutes * 60 + hours * 3600 + days * 3600 * 24;

 
    lcmaps_log_debug (2, "Succesfully finished Proxy Time To Live parsing: %d days, %d hours, %d minutes makes %d seconds.\n", days, hours, minutes, tot_sec);

    return tot_sec;
}


/******************************************************************************
Function:   Push_New_Entry
Description:
            Pushing New Entry to the list
Parameters:
            
            int    level
            time_t ttl
Returns:
            void
******************************************************************************/
void Push_New_Entry (struct TProxyLevelTTL ** p, int level, time_t ttl)
{
    struct TProxyLevelTTL * lp = *p;
    struct TProxyLevelTTL *new = NULL, *hand = NULL;


    /* if first entry */
    if (lp == NULL)
    {
        /* Clean alloc and fill with char '\0' */
        lp = (struct TProxyLevelTTL *) malloc (sizeof (struct TProxyLevelTTL));
        memset (lp, 0, sizeof (struct TProxyLevelTTL));     

        lp->level = level;
        lp->ttl   = ttl;
        lp->next  = NULL;
    }
    else
    {
        
        new = (struct TProxyLevelTTL *) malloc (sizeof (struct TProxyLevelTTL));
        memset (new, 0, sizeof (struct TProxyLevelTTL));

        new->level = level;
        new->ttl   = ttl;
        new->next  = NULL;

        hand = lp;
        lp = new;
        new->next = hand;
    }

    *p = lp;
}


/******************************************************************************
Function:   Search_TTL_By_Level
Description:
            Search TTL By Level
Parameters:
            struct TProxyLevelTTL * p
            int    level
            time_t ttl
Returns:
            time_t
******************************************************************************/
time_t Search_TTL_By_Level (struct TProxyLevelTTL * p, int level)
{
    struct TProxyLevelTTL * curr = NULL;

    curr = p;

    while (curr != NULL)
    {
        if (curr->level == level)
        {
            return curr->ttl;   
        } 

        curr = curr->next;
    }

    return 0;
}

/******************************************************************************
Function:   Print_TTL_By_Level
Description:
            Print TTL By Level
Parameters:
Returns:
******************************************************************************/
void Print_TTL_By_Level (struct TProxyLevelTTL * p)
{
    struct TProxyLevelTTL * curr = NULL;

    curr = p;

    while (curr != NULL)
    {
        lcmaps_log_debug(5, "Print_TTL_By_Level  Max TTL @ Level:  %d seconds @ %d\n", curr->ttl, curr->level);
        curr = curr->next;
    }
}



/******************************************************************************
Function:   Free Proxy Level TTL
description:
            Free's the linked list nicely
Parameters:
Returns:
            void
******************************************************************************/
void FreeProxyLevelTTL (struct TProxyLevelTTL ** p)
{
    struct TProxyLevelTTL * curr = NULL;
    struct TProxyLevelTTL * hulp = NULL;

    curr = *p;
    *p  = NULL;

    while (curr)
    {
        hulp = curr;
        curr = curr->next;
        free (hulp);
    }
}

