// =============================================================================
//
//      --- kvi_stat_controller.cpp ---
//
//   This is a plugin for KVIrc 2. It allows you to see
//   statistics of your IRC sessions - total words said etc.
//   More features are still to come.
//
//   (c) Krzysztof Godlewski <kristoff@poczta.wprost.pl>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviStatController"

#define __KVIRC_PLUGIN__

#include "kvi_locale.h"
#include "kvi_messagebox.h"
#include "kvi_stat_chan.h"
#include "kvi_stat_controller.h"
#include "kvi_stat_options.h"
#include "kvi_stat_systray.h"
#include "kvi_stat_window.h"
#include "kvirc_plugin.h"

       KviStatWindow  *g_pStatPluginStatWindow = 0;
extern KviStatOptions *g_pStatPluginOptionsWindow;

KviStatController::KviStatController()
	: QObject()
{
	m_pSysTrayList = new QPtrList<KviStatSysTray>;
	m_pSysTrayList->setAutoDelete(false);

	m_pStatChanList = new QPtrList<KviStatChan>;
	m_pStatChanList->setAutoDelete(true);

	m_bShowConsoleStats = true;

	loadConfig();
	loadStats();
}

KviStatController::~KviStatController()
{
	saveStats();
	saveConfig();

	while( KviStatSysTray *w = m_pSysTrayList->first() ) {
		kvirc_plugin_remove_systray_widget(w->frame(), w, true);
	}

	delete m_pSysTrayList; m_pSysTrayList = 0;
	delete m_pStatChanList;      m_pStatChanList      = 0;
}

void KviStatController::loadStats()
{
	KviStr tmp;
	kvirc_plugin_get_config_file_path(tmp, "stat.db");
	KviConfig cfg(tmp.ptr());

	cfg.setGroup("Main");

	m_stats.queriesWords    = cfg.readUIntEntry("QueryWords",     0);
	m_stats.queriesLetters  = cfg.readUIntEntry("QueryLetters",   0);
	m_stats.dccsWords       = cfg.readUIntEntry("DccWords",       0);
	m_stats.dccsLetters     = cfg.readUIntEntry("DccLetters",     0);
	m_stats.chansWords      = cfg.readUIntEntry("ChannelWords",   0);
	m_stats.chansLetters    = cfg.readUIntEntry("ChannelLetters", 0);
	m_stats.totalWords      = cfg.readUIntEntry("TotalWords",     0);
	m_stats.totalLetters    = cfg.readUIntEntry("TotalLetters",   0);
	m_stats.kicks           = cfg.readUIntEntry("Kicks",          0);
	m_stats.bans            = cfg.readUIntEntry("Bans",           0);
	m_stats.onIrc           = cfg.readUIntEntry("IrcSessions",    0);
	m_stats.joins           = cfg.readUIntEntry("Joins",          0);
	m_stats.topics          = cfg.readUIntEntry("Topics",         0);
	m_stats.sessionWords    = cfg.readUIntEntry("SessionWords",   0);
	m_stats.wordsRecord     = cfg.readUIntEntry("WordsRecord",    0);
	KviStr date = QDateTime::currentDateTime().toString();
	m_stats.startDate       = cfg.readEntry("StartDate", date.ptr());

	int idx = cfg.readIntEntry("NumChannels", 0);
	for( int i = 0; i < idx; i++ ) {
		KviStr tmp(KviStr::Format, "Chan%d", i);
		cfg.setGroup(tmp.ptr());

		KviStr chName = cfg.readEntry("Name", "");
		if( chName.hasData() ) {
			unsigned int jo = cfg.readUIntEntry("Joins",   0);
			unsigned int wo = cfg.readUIntEntry("Words",   0);
			unsigned int ki = cfg.readUIntEntry("Kicks",   0);
			unsigned int ba = cfg.readUIntEntry("Bans",    0);
			unsigned int to = cfg.readUIntEntry("Topics",  0);
			unsigned int ac = cfg.readUIntEntry("Actions", 0);

			KviStatChan *c = new KviStatChan(chName.ptr(), jo, wo, ki, ba, to, ac);
			m_pStatChanList->append(c);
		}
	}
}

void KviStatController::saveStats() const
{
	KviStr tmp;
	kvirc_plugin_get_config_file_path(tmp, "stat.db");
	KviConfig cfg(tmp.ptr());

	cfg.setGroup("Main");

	cfg.writeEntry("QueryWords",     m_stats.queriesWords);
	cfg.writeEntry("QueryLetters",   m_stats.queriesLetters);
	cfg.writeEntry("DccWords",       m_stats.dccsWords);
	cfg.writeEntry("DccLetters",     m_stats.dccsLetters);
	cfg.writeEntry("ChannelWords",   m_stats.chansWords);
	cfg.writeEntry("ChannelLetters", m_stats.chansLetters);
	cfg.writeEntry("TotalWords",     m_stats.totalWords);
	cfg.writeEntry("TotalLetters",   m_stats.totalLetters);
	cfg.writeEntry("Kicks",          m_stats.kicks);
	cfg.writeEntry("Bans",           m_stats.bans);
	cfg.writeEntry("IrcSessions",    m_stats.onIrc);
	cfg.writeEntry("Joins",          m_stats.joins);
	cfg.writeEntry("Topics",         m_stats.topics);
	cfg.writeEntry("SessionWords",   m_stats.sessionWords);
	cfg.writeEntry("WordsRecord",    m_stats.wordsRecord);
	cfg.writeEntry("StartDate",      m_stats.startDate.ptr());

	int idx = 0;

	for( KviStatChan *c = m_pStatChanList->first(); c; c = m_pStatChanList->next() ) {
		KviStr tmp(KviStr::Format, "Chan%d", idx);
		idx++;
		cfg.setGroup(tmp.ptr());
		cfg.writeEntry("Name",    c->name());
		cfg.writeEntry("Joins",   c->joins());
		cfg.writeEntry("Words",   c->words());
		cfg.writeEntry("Kicks",   c->kicks());
		cfg.writeEntry("Bans",    c->bans());
		cfg.writeEntry("Topics",  c->topics());
		cfg.writeEntry("Actions", c->actions());
	}

	cfg.setGroup("Main");
	cfg.writeEntry("NumChannels", idx);
}

void KviStatController::loadConfig()
{
	KviStr tmp;
	kvirc_plugin_get_config_file_path(tmp, "stat.cfg");
	KviConfig cfg(tmp.ptr());

	m_config.joinStatType      = cfg.readIntEntry("JoinStatType", ShowFullStatsOnJoin);
	// On channel join show:
	m_config.showJoins         = cfg.readBoolEntry("ShowJoins",  true);
	m_config.showWords         = cfg.readBoolEntry("ShowWords",  true);
	m_config.showKicks         = cfg.readBoolEntry("ShowKicks",  false);
	m_config.showBans          = cfg.readBoolEntry("ShowBans",   false);
	m_config.showTopics        = cfg.readBoolEntry("ShowTopics", false);
	// Enable systray on startup:
	m_config.sysTrayOnStartup  = cfg.readBoolEntry("STWOnStartup", true);
	// In systray widget show:
	m_config.STWshowJoins      = cfg.readBoolEntry("STWShowJoins",  true);
	m_config.STWshowWords      = cfg.readBoolEntry("STWShowWords",  true);
	m_config.STWshowKicks      = cfg.readBoolEntry("STWShowKicks",  true);
	m_config.STWshowBans       = cfg.readBoolEntry("STWShowBans",   true);
	m_config.STWshowTopics     = cfg.readBoolEntry("STWShowTopics", true);
	// Systray widget scrolling options:
	m_config.ScrollingWhen     = cfg.readIntEntry("ScrollingWhen",  ScrollingWhenNeeded);
	m_config.ScrollingDelay    = cfg.readIntEntry("ScrollingDelay", 200);
	m_config.ScrollingTextJump = cfg.readIntEntry("ScrollingStep",  3);
	m_config.ScrollingDirLeft  = cfg.readBoolEntry("ScrollingLeft", true);
}

void KviStatController::saveConfig() const
{
	KviStr tmp;
	kvirc_plugin_get_config_file_path(tmp, "stat.cfg");
	KviConfig cfg(tmp.ptr());

	cfg.writeEntry("JoinStatType",   m_config.joinStatType);
	// On channel join show:
	cfg.writeEntry("ShowJoins",      m_config.showJoins);
	cfg.writeEntry("ShowWords",      m_config.showWords);
	cfg.writeEntry("ShowKicks",      m_config.showKicks);
	cfg.writeEntry("ShowBans",       m_config.showBans);
	cfg.writeEntry("ShowTopics",     m_config.showTopics);
	// Enable systray on startup:
	cfg.writeEntry("STWOnStartup",   m_config.sysTrayOnStartup );
	// In systray widget show:
	cfg.writeEntry("STWShowJoins",   m_config.STWshowJoins);
	cfg.writeEntry("STWShowWords",   m_config.STWshowWords);
	cfg.writeEntry("STWShowKicks",   m_config.STWshowKicks);
	cfg.writeEntry("STWShowBans",    m_config.STWshowBans);
	cfg.writeEntry("STWShowTopics",  m_config.STWshowTopics);
	// Systray widget scrolling options:
	cfg.writeEntry("ScrollingWhen",  m_config.ScrollingWhen);
	cfg.writeEntry("ScrollingDelay", m_config.ScrollingDelay);
	cfg.writeEntry("ScrollingStep",  m_config.ScrollingTextJump);
	cfg.writeEntry("ScrollingLeft",  m_config.ScrollingDirLeft);
}

KviStatChan *const KviStatController::findStatChan(const char *channame) const
{
	for( KviStatChan *c = m_pStatChanList->first(); c; c = m_pStatChanList->next() ) {
		if( kvi_strEqualCI(channame, c->name()) )
			return c;
	}
	return 0;
}

void KviStatController::setCurrentChan(KviStatChan *ptr)
{
	m_pStatChanList->setAutoDelete(false);

	if( m_pStatChanList->removeRef(ptr) )
		m_pStatChanList->insert(0, ptr); // List head!

	m_pStatChanList->setAutoDelete(true);
}

void KviStatController::slotShowConfig()
{
	if( g_pStatPluginOptionsWindow ) {
		if( !g_pStatPluginOptionsWindow->isVisible() )
			g_pStatPluginOptionsWindow->show();
		return;
	}
	g_pStatPluginOptionsWindow = new KviStatOptions();

	g_pStatPluginOptionsWindow->setOptions(
		m_config.joinStatType, m_config.showJoins,
		m_config.showWords,    m_config.showKicks,
		m_config.showBans,     m_config.showTopics
	);

	g_pStatPluginOptionsWindow->setSysTrayOptions(
		m_config.sysTrayOnStartup, m_config.STWshowWords,
		m_config.STWshowJoins,     m_config.STWshowKicks,
		m_config.STWshowBans,      m_config.STWshowTopics
	);

	g_pStatPluginOptionsWindow->setScrollingOptions(
		m_config.ScrollingWhen,     m_config.ScrollingDelay,
		m_config.ScrollingTextJump, m_config.ScrollingDirLeft
	);

	connect(g_pStatPluginOptionsWindow, SIGNAL(applyButtonPressed()),   SLOT(slotApplyOptions()));
	connect(g_pStatPluginOptionsWindow, SIGNAL(defaultButtonPressed()), SLOT(slotApplyOptions()));
	connect(g_pStatPluginOptionsWindow, SIGNAL(cancelButtonPressed()),  SLOT(slotKillOptionsWindow()));

	g_pStatPluginOptionsWindow->show();
}

void KviStatController::slotShowStats()
{
	if( g_pStatPluginStatWindow ) return; // We want it to exist only when needed
	g_pStatPluginStatWindow = new KviStatWindow();
	connect(g_pStatPluginStatWindow, SIGNAL(applyButtonPressed()), SLOT(slotKillStatWindow()));

	g_pStatPluginStatWindow->updateStats();
	g_pStatPluginStatWindow->show();
}

void KviStatController::slotReset()
{
	doReset();
}

void KviStatController::slotSaveStats()
{
	saveStats();
}

void KviStatController::slotRemoveAllChans()
{
	if( m_pStatChanList->isEmpty() ) return;

	if( KviMessageBox::warningYesNo(
		_CHAR_2_QSTRING(_i18n_("Are you sure you want to remove all channels from your statistics?")),
		_CHAR_2_QSTRING(_i18n_("Remove all channels"))) == KviMessageBox::Yes
	) {
		m_pStatChanList->clear();
		g_pStatPluginStatWindow->updateStats();
	}
}

const KviStatChan *const KviStatController::currentChan() const  { return m_pStatChanList->first(); }

void KviStatController::addChan   (KviStatChan *ptr)             { m_pStatChanList->append(ptr); }
void KviStatController::removeChan(KviStatChan *ptr)             { m_pStatChanList->removeRef(ptr); }

const KviStatData &KviStatController::statData() const           { return m_stats; }
QPtrList<KviStatChan> *const KviStatController::chanList() const { return m_pStatChanList;}

bool KviStatController::showConsoleStats() const                 { return m_bShowConsoleStats; }
bool KviStatController::sysTrayOnStartup() const                 { return m_config.sysTrayOnStartup; }

void KviStatController::addTotalKick()                           { ++m_stats.kicks; };
void KviStatController::addTotalBan()                            { ++m_stats.bans; };
void KviStatController::addOnIrc()                               { ++m_stats.onIrc; };
void KviStatController::addTotalJoin()                           { ++m_stats.joins; };
void KviStatController::addTotalTopic()                          { ++m_stats.topics; };

void KviStatController::addChansLetters(unsigned int num)        { m_stats.chansLetters += num; }
void KviStatController::addQueriesLetters(unsigned int num)      { m_stats.queriesLetters += num;}
void KviStatController::addDccsLetters(unsigned int num)         { m_stats.dccsLetters += num; }
void KviStatController::addTotalLetters(unsigned int num)        { m_stats.totalLetters += num; }

void KviStatController::addChansWords(unsigned int num)          { m_stats.chansWords += num; }
void KviStatController::addQueriesWords(unsigned int num)        { m_stats.queriesWords += num; }
void KviStatController::addDccsWords(unsigned int num)           { m_stats.dccsWords += num; }
void KviStatController::addTotalWords(unsigned int num)          { m_stats.totalWords += num; }

unsigned int KviStatController::totalWords()   const             { return m_stats.totalWords; }
unsigned int KviStatController::chansWords()   const             { return m_stats.chansWords; }
unsigned int KviStatController::queriesWords() const             { return m_stats.queriesWords; }
unsigned int KviStatController::dccsWords()    const             { return m_stats.dccsWords; }

unsigned int KviStatController::totalLetters()   const           { return m_stats.totalLetters; }
unsigned int KviStatController::chansLetters()   const           { return m_stats.chansLetters; }
unsigned int KviStatController::queriesLetters() const           { return m_stats.queriesLetters; }
unsigned int KviStatController::dccsLetters()    const           { return m_stats.dccsLetters; }

unsigned int KviStatController::numOnIrc()  const                { return m_stats.onIrc; }
unsigned int KviStatController::numKicks()  const                { return m_stats.kicks; }
unsigned int KviStatController::numBans()   const                { return m_stats.bans; }
unsigned int KviStatController::numTopics() const                { return m_stats.topics; }
unsigned int KviStatController::numJoins()  const                { return m_stats.joins; }

const char *KviStatController::startDate() const                 { return m_stats.startDate.ptr(); }
void KviStatController::addSessionWords(int num)                 { m_stats.sessionWords += num; }
unsigned int KviStatController::sessionWords() const             { return m_stats.sessionWords; }
void KviStatController::resetSessionWords()                      { m_stats.sessionWords = 0; }
unsigned int KviStatController::wordsRecord() const              { return m_stats.wordsRecord; }
void KviStatController::setWordsRecord(unsigned int n)           { m_stats.wordsRecord = n; }

unsigned int KviStatController::joinStatType() const             { return m_config.joinStatType; }
bool KviStatController::showJoins()  const                       { return m_config.showJoins; }
bool KviStatController::showWords()  const                       { return m_config.showWords; }
bool KviStatController::showKicks()  const                       { return m_config.showKicks; }
bool KviStatController::showBans()   const                       { return m_config.showBans; }
bool KviStatController::showTopics() const                       { return m_config.showTopics; }

bool KviStatController::doReset()
{
	if( KviMessageBox::warningYesNo(
		_CHAR_2_QSTRING(_i18n_("Are you sure you want to reset your statistics?")),
		_CHAR_2_QSTRING(_i18n_("Reset stats"))) == KviMessageBox::No
	) {
		return false;
	}
	m_stats.startDate = QDateTime::currentDateTime().toString();
	m_stats.chansWords     = 0;
	m_stats.chansLetters   = 0;
	m_stats.queriesWords   = 0;
	m_stats.queriesLetters = 0;
	m_stats.dccsWords      = 0;
	m_stats.dccsLetters    = 0;
	m_stats.totalWords     = 0;
	m_stats.totalLetters   = 0;
	m_stats.onIrc          = 0;
	m_stats.bans           = 0;
	m_stats.kicks          = 0;
	m_stats.joins          = 0;
	m_stats.topics         = 0;
	m_stats.wordsRecord    = 0;

	m_pStatChanList->clear();

	QPtrListIterator<KviStatSysTray> it(*m_pSysTrayList);
	for( ; it.current(); ++it )
		it.current()->setStartDate(m_stats.startDate.ptr());

	saveStats();
	return true;
}

void KviStatController::slotKillOptionsWindow()
{
	if( !g_pStatPluginOptionsWindow ) return;
	delete g_pStatPluginOptionsWindow;
	g_pStatPluginOptionsWindow = 0;
}

void KviStatController::slotKillStatWindow()
{
	if( !g_pStatPluginStatWindow ) return;
	delete g_pStatPluginStatWindow;
	g_pStatPluginStatWindow = 0;
}

void KviStatController::slotApplyOptions()
{
	if( !g_pStatPluginOptionsWindow ) return;

	g_pStatPluginOptionsWindow->getOptions(
		m_config.joinStatType, m_config.showJoins,
		m_config.showWords,    m_config.showKicks,
		m_config.showBans,     m_config.showTopics
	);

	g_pStatPluginOptionsWindow->getSysTrayOptions(
		m_config.sysTrayOnStartup, m_config.STWshowWords,
		m_config.STWshowJoins,     m_config.STWshowKicks,
		m_config.STWshowBans,      m_config.STWshowTopics
	);

	g_pStatPluginOptionsWindow->getScrollingOptions(
		m_config.ScrollingWhen,     m_config.ScrollingDelay,
		m_config.ScrollingTextJump, m_config.ScrollingDirLeft
	);

	QPtrListIterator<KviStatSysTray> it(*m_pSysTrayList);

	for( ; it.current(); ++it ) {
		it.current()->setOptions(
			m_config.STWshowWords, m_config.STWshowJoins,
			m_config.STWshowKicks, m_config.STWshowBans, m_config.STWshowTopics
		);

		it.current()->setScrollingOptions(
			m_config.ScrollingWhen,     m_config.ScrollingDelay,
			m_config.ScrollingTextJump, m_config.ScrollingDirLeft
		);
	}

	slotKillOptionsWindow();
}

bool KviStatController::eventFilter(QObject *obj, QEvent *e)
{
	KviWindow *win = (KviWindow*) obj->parent();

	if( e->type() == QEvent::FocusIn ) {
		if( win ) {
			if( kvi_strEqualCI(win->className(), "KviChannel") ) {
				KviStatChan *c = findStatChan(win->caption());
				if( c ) {
					setCurrentChan(c);
					m_bShowConsoleStats = false;
				} else
					m_bShowConsoleStats = true;
			}
		}
	}

	return false;
}

void KviStatController::setSysTrayOptions()
{
	QPtrListIterator<KviStatSysTray> it(*m_pSysTrayList);
	for( ; it.current(); ++it ) {
		it.current()->setOptions(
			m_config.STWshowJoins, m_config.STWshowWords,
			m_config.STWshowKicks, m_config.STWshowBans, m_config.STWshowTopics
		);
		it.current()->setStartDate(m_stats.startDate.ptr());
		it.current()->setScrollingOptions(
			m_config.ScrollingWhen,     m_config.ScrollingDelay,
			m_config.ScrollingTextJump, m_config.ScrollingDirLeft
		);
	}
}

void KviStatController::registerStatTray(KviStatSysTray *ptr)
{
	m_pSysTrayList->append(ptr);
}

void KviStatController::unregisterStatTray(KviStatSysTray *ptr)
{
	m_pSysTrayList->removeRef(ptr);
}

#include "m_kvi_stat_controller.moc"
