// =============================================================================
//
//      --- kvi_usertoolbareditor.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviUserToolBarEditor"

#include <qcursor.h>
#include <qhbox.h>
#include <qheader.h>
#include <qlayout.h>
#include <qsplitter.h>
#include <qtoolbutton.h>

#include "kvi_app.h"
#include "kvi_defines.h"
#include "kvi_filedialog.h"
#include "kvi_label.h"
#include "kvi_lineedit.h"
#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_messagebox.h"
#include "kvi_popupmenu.h"
#include "kvi_pushbutton.h"
#include "kvi_script_editor.h"
#include "kvi_toolbarlistview.h"
#include "kvi_transparentwidget.h"
#include "kvi_userpopupmenu.h"
#include "kvi_usertoolbar.h"
#include "kvi_usertoolbareditor.h"
#include "kvi_usertoolbartemplate.h"

// TODO: Totally missing quick help!

extern KviUserToolBarTemplate *g_pUserToolBarTemplate;

extern QPixmap *g_pixViewOut[KVI_OUT_NUM_IMAGES];

/*
	@quickhelp: QWidget_ToolBarTab
	@widget: Toolbar editor
		Here you can edit your toolbar.<br>
*/

/**
 *
 * TOOLBAR EDITOR
 *
 * This widget allows editing of the user toolbar
 *
 */
KviUserToolBarEditor::KviUserToolBarEditor(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	m_pEditedToolButton = 0;
	m_pEditedToolBar    = 0;

	QGridLayout *g = new QGridLayout(this, 5, 3, 4, 2);

	m_pToolBar = new KviUserToolBar((KviFrame *) (g_pApp->m_pFrameList->first()), this, Qt::Unmanaged);
	m_pToolBar->copyFromTemplate(g_pUserToolBarTemplate, false);

	m_pToolBar->setStretchableWidget(new KviTransparentWidget(m_pToolBar));
	g->addMultiCellWidget(m_pToolBar, 0, 0, 0, 1);

	KviPushButton *b = new KviPushButton(_CHAR_2_QSTRING(_i18n_("&Clear Toolbar")), this);
	g->addWidget(b, 0, 2);
	connect(b, SIGNAL(clicked()), this, SLOT(clearToolBar()));

	QSplitter *splitter = new QSplitter(Qt::Vertical, this);
	splitter->setOpaqueResize();

	m_pToolBarView = new KviListView(splitter);
	m_pToolBarView->addColumn(_CHAR_2_QSTRING(_i18n_("Item")));
	m_pToolBarView->addColumn(_CHAR_2_QSTRING(_i18n_("Type")));
	m_pToolBarView->setRootIsDecorated(true);
	m_pToolBarView->setMultiSelection(false);
	m_pToolBarView->header()->setClickEnabled(false);
	m_pToolBarView->viewport()->installEventFilter(this); // Need to adjust the behaviour of mousePressEvent

	m_pToolBarPopup = new KviPopupMenu(this);

	connect(m_pToolBarView, SIGNAL(selectionChanged(KviListViewItem *)), this, SLOT(toolButtonSelected(KviListViewItem *)));
	connect(m_pToolBarView,
		SIGNAL(rightButtonPressed(KviListViewItem *, const QPoint &, int)),
		this,
		SLOT(toolBarViewRightButtonPressed(KviListViewItem *, const QPoint &, int))
	);

	m_pToolBarEditor = new KviScriptEditor(splitter);
	g->addMultiCellWidget(splitter, 1, 1, 0, 2);

	m_pIconPopup = new KviPopupMenu(this);
	KviPopupMenu *ppp = m_pIconPopup;
	connect(ppp, SIGNAL(activated(int)), this, SLOT(iconPopupActivated(int)));
	for( int i = 0; i < KVI_OUT_NUM_IMAGES; i++ ) {
		ppp->insertItem(*(g_pixViewOut[i]), i);
		if( ((i % 24) == 0) && (i != 0) ) {
			KviPopupMenu *sub = new KviPopupMenu(ppp);
			ppp->insertItem(_i18n_("More..."), sub);
			ppp = sub;
			connect(ppp, SIGNAL(activated(int)), this, SLOT(iconPopupActivated(int)));
		}
	}

	KviLabel *l = new KviLabel(_CHAR_2_QSTRING("Icon:"), this);
	g->addWidget(l, 2, 0);

	QHBox *box = new QHBox(this);
	g->addMultiCellWidget(box, 2, 2, 1, 2);

	m_pToolBarIconEditor = new KviLineEdit(box);
	m_pToolBarIconEditor->installEventFilter(this);
	connect(m_pToolBarIconEditor, SIGNAL(lostFocus()), this, SLOT(iconLostFocus()));

	m_pIconButton = new QToolButton(DownArrow, box);
	m_pIconButton->setFocusPolicy(NoFocus);
	connect(m_pIconButton, SIGNAL(clicked()), this, SLOT(popupIconMenu()));

	m_pBrowseButton = new KviPushButton(_CHAR_2_QSTRING(_i18n_("&Browse...")), box);
	m_pBrowseButton->setFocusPolicy(NoFocus);
	connect(m_pBrowseButton, SIGNAL(clicked()), this, SLOT(pickIconFilename()));

	l = new KviLabel(_CHAR_2_QSTRING("Name:"), this);
	g->addWidget(l, 3, 0);

	m_pToolBarNameEditor = new KviLineEdit(this);
	g->addMultiCellWidget(m_pToolBarNameEditor, 3, 3, 1, 2);
	connect(m_pToolBarNameEditor , SIGNAL(textChanged(const QString &)), this, SLOT(nameChanged(const QString &)));

	l = new KviLabel(_CHAR_2_QSTRING("Statusbar text:"), this);
	g->addWidget(l, 4, 0);

	m_pToolBarStatusBarEditor = new KviLineEdit(this);
	g->addMultiCellWidget(m_pToolBarStatusBarEditor, 4, 4, 1, 2);

	g->setColStretch(1, 10);
	startEditToolBar(m_pToolBar);
}

KviUserToolBarEditor::~KviUserToolBarEditor()
{
	if( m_pToolBar ) {
		delete m_pToolBar;
		m_pToolBar = 0;
	}
}

void KviUserToolBarEditor::iconLostFocus()
{
	bool bOk;
	QString text = m_pToolBarIconEditor->text();
	if( m_iconString == text )
		return;
	m_iconString = text;
	int i = text.toInt(&bOk);
	if( bOk )
		iconPopupActivated(i);
	else
		toolButtonSelected(m_pEditedToolButton);
}

void KviUserToolBarEditor::nameChanged(const QString &text)
{
	if( m_pEditedToolButton )
		m_pEditedToolButton->setText(0, text);
}

void KviUserToolBarEditor::pickIconFilename()
{
	QString s = KviFileDialog::getOpenFileName(m_pToolBarIconEditor->text(), QString::null, this);
	if( !s.isNull() ) {
		m_pToolBarIconEditor->setText(s);
		toolButtonSelected(m_pEditedToolButton);
	}
}

void KviUserToolBarEditor::commit()
{
	if( m_pEditedToolBar )
		saveToolBar(m_pEditedToolBar);
	g_pUserToolBarTemplate->copyFromToolBar(m_pEditedToolBar);
}

void KviUserToolBarEditor::popupIconMenu()
{
	m_pIconPopup->popup(QCursor::pos());
}

bool KviUserToolBarEditor::eventFilter(QObject *o, QEvent *e)
{
	if( o == m_pToolBarIconEditor ) {
		if( (e->type() == QEvent::MouseButtonPress) && ((QMouseEvent *) e)->button() & RightButton ) {
			popupIconMenu();
			return true;
		}
		return false;
	}

	if( e->type() == QEvent::MouseButtonPress ) {
		if( o == m_pToolBarView->viewport() ) {
			if( !(m_pToolBarView->firstChild()) ) {
				toolBarViewRightButtonPressed(0, m_pToolBarView->mapToGlobal(((QMouseEvent *) e)->pos()), 0);
				return true;
			}
		}
	}
	return QWidget::eventFilter(o, e);
}

void KviUserToolBarEditor::iconPopupActivated(int id)
{
	if( (id >= 0) && (id <= KVI_OUT_NUM_IMAGES) && m_pToolBarIconEditor->isEnabled() ) {
		QString tmp;
		tmp.setNum(id);
		m_pToolBarIconEditor->setText(tmp);
		if( m_pEditedToolButton )
			saveToolButton(m_pEditedToolButton);
	}
}

/**
 * ==================== startEditToolBar =================
 */
void KviUserToolBarEditor::startEditToolBar(KviUserToolBar *p)
{
	if( m_pEditedToolBar )
		saveToolBar(m_pEditedToolBar);
	m_pEditedToolBar = p;
	m_pToolBarView->clear();
	m_pToolBarView->setSorting(-1);
	appendToolButton(p);
	m_pEditedToolButton = 0;
	if( !(m_pToolBarView->firstChild()) )
		disableToolBarEditors();
	else
		m_pToolBarView->setSelected(m_pToolBarView->firstChild(), true);
}

/**
 * ==================== clearToolBar =================
 */
void KviUserToolBarEditor::clearToolBar()
{
	if( KviMessageBox::warningYesNo(
		_i18n_("Are you sure you want to clear the user toolbar?"),
		_i18n_("Clear User Toolbar"),
		this) == KviMessageBox::Yes
	) {
		m_pToolBarView->clear();
		m_pToolBarView->setSorting(-1);
	}
	m_pEditedToolButton = 0;
	disableToolBarEditors();
	updateToolBar(m_pEditedToolBar, 0);
}

/**
 * ==================== selectToolButtonById =================
 */
bool KviUserToolBarEditor::selectToolButtonById(KviToolBarListViewItem *par, int id)
{
	KviToolBarListViewItem *it = (KviToolBarListViewItem *) (par ? par->firstChild() : m_pToolBarView->firstChild());
	while( it ) {
		if( it->m_id == id ) {
			// Open all parents
			par = (KviToolBarListViewItem *) it->parent();
			while( par ) {
				par->setOpen(true);
				par = (KviToolBarListViewItem *) par->parent();
			}
			// Select it
			m_pToolBarView->setSelected(it, true);
			// And ensure it is visible
			m_pToolBarView->ensureItemVisible(it);
			return true;
		}
		it = (KviToolBarListViewItem *) it->nextSibling();
	}
	return false;
}

/**
 * ==================== saveToolBar =================
 */
void KviUserToolBarEditor::saveToolBar(KviUserToolBar *p)
{
	if( m_pEditedToolButton )
		saveToolButton(m_pEditedToolButton); // Do not forget to save the last one
}

/**
 * ==================== saveToolButton =================
 */
void KviUserToolBarEditor::saveToolButton(KviToolBarListViewItem *p)
{
	if( m_pEditedToolButton->m_type == KviToolBarListViewItem::Button ||
	    m_pEditedToolButton->m_type == KviToolBarListViewItem::Menu   ||
	    m_pEditedToolButton->m_type == KviToolBarListViewItem::Item
	) {
		m_pEditedToolButton->m_buffer = m_pToolBarEditor->text();
		m_pEditedToolButton->m_icon   = m_pToolBarIconEditor->text();
		m_pEditedToolButton->m_sbar   = m_pToolBarStatusBarEditor->text();
		QString s = m_pToolBarNameEditor->text();
		s.stripWhiteSpace();
		if( s.isEmpty() )
			s = _CHAR_2_QSTRING(_i18n_("Unnamed"));
		m_pEditedToolButton->setText(0, s);
		m_pEditedToolButton->tryLoadImage();
	} else if( m_pEditedToolButton->m_type == KviToolBarListViewItem::Submenu ) {
		m_pEditedToolButton->m_icon = m_pToolBarIconEditor->text();
		QString s = m_pToolBarNameEditor->text();
		s.stripWhiteSpace();
		if( s.isEmpty() )
			s = _CHAR_2_QSTRING(_i18n_("Unnamed"));
		m_pEditedToolButton->setText(0, s);
		m_pEditedToolButton->tryLoadImage();
	}
	updateToolBar(m_pEditedToolBar, 0);
}

void KviUserToolBarEditor::saveToolBarMenu(KviUserPopupMenu *p, KviToolBarListViewItem *par)
{
	KviToolBarListViewItem *it = (KviToolBarListViewItem *) (par ? par->firstChild() : m_pToolBarView->firstChild());
	while( it ) {
		if( it->m_type == KviToolBarListViewItem::Separator ) {
			p->insertSeparatorItem();
			it->m_id = 0;
		} else {
			if( it->m_type == KviToolBarListViewItem::Item ) {
				KviStr tmp = it->text(0);
				tmp.stripWhiteSpace();
				if( tmp.isEmpty() )
					tmp = _i18n_("Unnamed");
				it->m_id = p->insertNormalItem(tmp.ptr(), it->m_icon.ptr(), it->m_buffer.ptr());
			} else if( it->m_type == KviToolBarListViewItem::Menu ) {
				KviStr tmp = it->text(0);
				tmp.stripWhiteSpace();
				if( tmp.isEmpty() )
					tmp = _i18n_("Unnamed");
				KviStr yourName(KviStr::Format, "usertoolbar_script_popup_%s", tmp.ptr());

				KviUserPopupMenu *sub = new KviUserPopupMenu(0, yourName.ptr());
				saveToolBarMenu(sub, it);

				p->insertPopupItem(tmp.ptr(), it->m_icon.ptr(), sub);
				it->m_id = 0;
			}
		}
		it = (KviToolBarListViewItem *) it->nextSibling();
	}
}

/**
 * ==================== updateToolBar =================
 */
void KviUserToolBarEditor::updateToolBar(KviUserToolBar *p, KviToolBarListViewItem *par)
{
	p->setUpdatesEnabled(false);
	p->clearAll();
	KviToolBarListViewItem *it = (KviToolBarListViewItem *) (par ? par->firstChild() : m_pToolBarView->firstChild());
	while( it ) {
		if( it->m_type == KviToolBarListViewItem::Separator ) {
			p->insertSeparatorItem();
			it->m_id = 0;
		} else
		if( it->m_type == KviToolBarListViewItem::Button ) {
			KviStr tmp = it->text(0);
			tmp.stripWhiteSpace();
			if( tmp.isEmpty() )
				tmp = _i18n_("Unnamed");
			p->insertNormalItem(tmp.ptr(), it->m_sbar.ptr(), it->m_icon.ptr(), it->m_buffer.ptr());
		} else
		if( it->m_type == KviToolBarListViewItem::Menu ) {
			KviStr tmp = it->text(0);
			tmp.stripWhiteSpace();
			if( tmp.isEmpty() )
				tmp = _i18n_("Unnamed");
			KviStr yourName(KviStr::Format, "usertoolbar_script_popup_%s", tmp.ptr());
			KviUserPopupMenu *m = new KviUserPopupMenu(0, yourName.ptr());
			saveToolBarMenu(m, it);

			p->insertNormalItem(tmp.ptr(), it->m_sbar.ptr(), it->m_icon.ptr(), it->m_buffer.ptr(), m);
		}
		it = (KviToolBarListViewItem *) it->nextSibling();
	}
	QWidget *w = new QWidget(p, "bar");
	p->setStretchableWidget(w);
	p->setUpdatesEnabled(true);
	p->update();
}

/**
 * ==================== disableToolBarEditors =================
 */
void KviUserToolBarEditor::disableToolBarEditors()
{
	m_pToolBarEditor->setText(_CHAR_2_QSTRING(""));
	m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(""));
	m_pToolBarNameEditor->setText(_CHAR_2_QSTRING(""));
	m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(""));
	m_pToolBarEditor->setEnabled(false);
	m_pToolBarIconEditor->setEnabled(false);
	m_pToolBarNameEditor->setEnabled(false);
	m_pToolBarStatusBarEditor->setEnabled(false);
	m_pBrowseButton->setEnabled(false);
	m_pIconButton->setEnabled(false);
}

/**
 * ==================== appendToolButton ========================
 */
void KviUserToolBarEditor::appendToolButton(KviUserToolBar *p)
{
	KviToolBarListViewItem *it = 0;
	for( KviUserToolBarData *d = p->m_pDataList->first(); d; d = p->m_pDataList->next() ) {
		if( it ) {
			switch( d->type ) {
				case KviUserToolBar::Separator:
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,
						m_pToolBarView, it,
						"----", "separator", 0, 0
					);
					break;
				case KviUserToolBar::Button:
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,
						m_pToolBarView, it,
						d->text.ptr(), "button", d->buffer.ptr(), d->image.ptr(), d->sbar.ptr()
					);
					break;
				case KviUserToolBar::Menu:
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,
						m_pToolBarView, it,
						d->text.ptr(), "menu", d->buffer.ptr(), d->image.ptr(), d->sbar.ptr()
					);
					appendToolBarMenu(it, d->menu);
					break;
			}
		} else {
			switch( d->type ) {
				case KviUserToolBar::Separator: {
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator, m_pToolBarView,
						"----", "separator", 0, 0
					);
					break;
				}
				case KviUserToolBar::Button: {
					KviStr tmp(d->text);
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Button, m_pToolBarView,
						tmp.ptr(), "button", d->buffer.ptr(), d->image.ptr(), d->sbar.ptr()
					);
					break;
				}
				case KviUserToolBar::Menu: {
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Menu, m_pToolBarView,
						d->text.ptr(), "menu", d->buffer.ptr(), d->image.ptr(), d->sbar.ptr()
					);
					appendToolBarMenu(it, d->menu);
					break;
				}
			}
		}
	}
}

void KviUserToolBarEditor::appendToolBarMenu(KviToolBarListViewItem *parent, KviUserPopupMenu *p)
{
	KviToolBarListViewItem *it = 0;
	for( KviUserPopupMenuData *d = p->m_pDataList.first(); d; d = p->m_pDataList.next() ) {
		if( d->menu ) {
			KviStr tmp(p->text(d->id));
			if( it ) { // Insert after (Qt strangely sometimes inverts the insertion order)
				it = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,
					parent, it, tmp.ptr(), "submenu", 0, d->image.ptr()
				);
			} else {
				it = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,
					parent, tmp.ptr(), "submenu", 0, d->image.ptr()
				);
			}
			appendToolBarMenu(it, d->menu);
		} else {
			if( (d->id == 0) && kvi_strEqualCS(d->buffer.ptr(), " ") ) {
				if( it ) {
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator, parent, it,
						"----", "separator", 0, 0
					);
				} else {
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator, parent,
						"----", "separator", 0, 0
					);
				}
			} else {
				KviStr tmp(p->text(d->id));
				if( it ) {
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Item, parent, it,
						tmp.ptr(), "item", d->buffer.ptr(), d->image.ptr()
					);
				} else {
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Item, parent,
						tmp.ptr(), "item", d->buffer.ptr(), d->image.ptr()
					);
				}
			}
		}
	}
}

/**
 * ==================== toolButtonSelected ========================
 */
void KviUserToolBarEditor::toolButtonSelected(KviListViewItem *i)
{
	if( m_pEditedToolButton )
		saveToolButton(m_pEditedToolButton);

	KviToolBarListViewItem *it = (KviToolBarListViewItem *) i;
	m_pEditedToolButton = it;
	if( !it ) {
		disableToolBarEditors();
		return;
	}

	switch( it->m_type ) {
		case KviToolBarListViewItem::Separator:
			m_pToolBarEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarEditor->setEnabled(false);
			m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarIconEditor->setEnabled(false);
			m_pToolBarNameEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarNameEditor->setEnabled(false);
			m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarStatusBarEditor->setEnabled(false);
			break;
		case KviToolBarListViewItem::Menu:
			m_pToolBarEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarEditor->setEnabled(false);
			m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
			m_pToolBarIconEditor->setEnabled(true);
			m_pBrowseButton->setEnabled(true);
			m_pIconButton->setEnabled(true);
			m_pToolBarNameEditor->setText(it->text(0));
			m_pToolBarNameEditor->setEnabled(true);
			m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(it->m_sbar.ptr()));
			m_pToolBarStatusBarEditor->setEnabled(true);
			break;
		case KviToolBarListViewItem::Item:
			m_pToolBarEditor->setText(_CHAR_2_QSTRING(it->m_buffer.ptr()));
			m_pToolBarEditor->setEnabled(true);
			m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
			m_pToolBarIconEditor->setEnabled(true);
			m_pBrowseButton->setEnabled(true);
			m_pIconButton->setEnabled(true);
			m_pToolBarNameEditor->setText(it->text(0));
			m_pToolBarNameEditor->setEnabled(true);
			m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarStatusBarEditor->setEnabled(false);
			break;
		default:
			m_pToolBarEditor->setText(_CHAR_2_QSTRING(it->m_buffer.ptr()));
			m_pToolBarEditor->setEnabled(true);
			m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
			m_pToolBarIconEditor->setEnabled(true);
			m_pBrowseButton->setEnabled(true);
			m_pIconButton->setEnabled(true);
			m_pToolBarNameEditor->setText(it->text(0));
			m_pToolBarNameEditor->setEnabled(true);
			m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(it->m_sbar.ptr()));
			m_pToolBarStatusBarEditor->setEnabled(true);
			break;
	}
}

/**
 * ==================== toolBarViewRightButtonPressed ========================
 */
void KviUserToolBarEditor::toolBarViewRightButtonPressed(KviListViewItem *i, const QPoint &p, int)
{
	if( i )
		m_pToolBarView->setSelected(i, true);
	else {
		m_pToolBarView->clearSelection();
		if( m_pEditedToolButton )
			saveToolButton(m_pEditedToolButton);
	}
	m_pEditedToolButton = (KviToolBarListViewItem *) i;

	if( m_pEditedToolButton ) {
		m_pToolBarPopup->clear();
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("&Remove Item")), this, SLOT(removeEditedToolButton()));
		m_pToolBarPopup->insertSeparator();
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("&Add Item Below")), this, SLOT(addEmptyToolButton()));
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("Add &Submenu Below")), this, SLOT(addToolBarMenu()));
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("Add S&eparator Below")), this, SLOT(addToolBarSeparatorItem()));
		if( m_pEditedToolButton->parent() == m_pEditedToolButton->itemAbove() ) {
			m_pToolBarPopup->insertSeparator();
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("A&dd Item Above")), this, SLOT(addEmptyToolButtonAbove()));
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("Add S&ubmenu Above")), this, SLOT(addToolBarMenuAbove()));
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("Add Separat&or Above")), this, SLOT(addToolBarSeparatorItemAbove()));
		}
		if( m_pEditedToolButton->m_type == KviToolBarListViewItem::Button ||
		    m_pEditedToolButton->m_type == KviToolBarListViewItem::Item
		) {
			m_pToolBarPopup->insertSeparator();
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("&Copy Item")), this, SLOT(copyToolButton()));
		}
		if( m_szClipboardItemName.hasData() ) {
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("&Paste Item Below")), this, SLOT(pasteToolButtonBelow()));
			if( m_pEditedToolButton->parent() == m_pEditedToolButton->itemAbove() ) {
				m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("Paste &Item Above")), this, SLOT(pasteToolButtonAbove()));
			}
		}
	} else {
		// At the end?
		disableToolBarEditors();
		m_pToolBarPopup->clear();
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("&Add Item")), this, SLOT(addEmptyToolButton()));
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("Add &Submenu")), this, SLOT(addToolBarMenu()));
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(_i18n_("Add S&eparator")), this, SLOT(addToolBarSeparatorItem()));
	}
	m_pToolBarPopup->popup(p);
}

/**
 * ==================== removeEditedToolButton ========================
 */
void KviUserToolBarEditor::removeEditedToolButton()
{
	if( !m_pEditedToolButton ) return;

	KviListViewItem *p = m_pEditedToolButton->parent();
	KviToolBarListViewItem *tmp = m_pEditedToolButton;
	// Needs to be 0 before the delete (the KviListViewItem destructor
	// triggers an update in the KviListView that calls selectionChanged that
	// attempts to save the popup while it is being deleted!
	m_pEditedToolButton = 0;
	delete tmp;
	if( p )
		m_pToolBarView->setSelected(p, true);
	else {
		if( m_pToolBarView->firstChild() )
			m_pToolBarView->setSelected(m_pToolBarView->firstChild(), true);
		else
			disableToolBarEditors();
	}
	updateToolBar(m_pEditedToolBar, 0);
}

/**
 * ==================== copyToolButton ========================
 */
void KviUserToolBarEditor::copyToolButton()
{
	if( !m_pEditedToolButton ) return;

	if( m_pEditedToolButton->m_type != KviToolBarListViewItem::Button &&
	    m_pEditedToolButton->m_type != KviToolBarListViewItem::Item
	) {
		return;
	}
	m_szClipboardItemBuffer = m_pEditedToolButton->m_buffer;
	m_szClipboardItemIcon   = m_pEditedToolButton->m_icon;
	m_szClipboardItemName   = m_pEditedToolButton->text(0);
	m_szClipboardItemSbar	= m_pEditedToolButton->m_sbar;
}

/**
 * ==================== addEmptyToolButton ========================
 */
void KviUserToolBarEditor::addEmptyToolButton()
{
	KviToolBarListViewItem *it;
	if( m_pEditedToolButton ) {
		if( m_pEditedToolButton->parent() ) {
			it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,
				(KviToolBarListViewItem *) m_pEditedToolButton->parent(),
				m_pEditedToolButton, "unnamed", "item",
				_i18n_("# Enter the script for the new item here"), ""
			);
		} else {
			it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,
				m_pToolBarView,
				m_pEditedToolButton, "unnamed", "button",
				_i18n_("# Enter the script for the new item here"), ""
			);
		}
	} else {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,
			m_pToolBarView, "unnamed", "button",
			_i18n_("# Enter the script for the new item here"), ""
		);
	}
	m_pToolBarView->setSelected(it, true);
	updateToolBar(m_pEditedToolBar, 0);
}

void KviUserToolBarEditor::addToolBarMenu()
{
	KviToolBarListViewItem *sb;
	if( m_pEditedToolButton ) {
		if( m_pEditedToolButton->parent() ) {
			sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,
				(KviToolBarListViewItem *) m_pEditedToolButton->parent(),
				m_pEditedToolButton, "unnamed", "submenu", "", ""
			);
		} else {
			sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,
				m_pToolBarView,
				m_pEditedToolButton, "unnamed", "menu", "", ""
			);
		}
	} else {
		sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,
			m_pToolBarView, "unnamed", "menu", "", ""
		);
	}

	KviToolBarListViewItem *it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,
		sb, "unnamed", "item", _i18n_("# Enter the script for the new item here"), ""
	);
	sb->setOpen(true);

	m_pToolBarView->setSelected(it, true);
	updateToolBar(m_pEditedToolBar, 0);
}

void KviUserToolBarEditor::addToolBarMenuAbove()
{
	if( !m_pEditedToolButton ) return;

	if( m_pEditedToolButton->parent() != m_pEditedToolButton->itemAbove() )
		return; // Ensure first item

	KviToolBarListViewItem *sb;
	if( m_pEditedToolButton->parent() ) {
		sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,
			(KviToolBarListViewItem *) m_pEditedToolButton->parent(),
			"unnamed", "submenu", "", ""
		);
	} else {
		sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,
			m_pToolBarView, "unnamed", "menu", "", ""
		);
	}

	KviToolBarListViewItem *it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,
		sb, "unnamed", "item", _i18n_("# Enter the script for the new item here"), ""
	);
	sb->setOpen(true);

	m_pToolBarView->setSelected(it, true);
	updateToolBar(m_pEditedToolBar, 0);
}

/**
 * ==================== pasteToolButtonBelow ========================
 */
void KviUserToolBarEditor::pasteToolButtonBelow()
{
	if( m_szClipboardItemName.isEmpty() ) return;

	KviToolBarListViewItem *it;
	if( m_pEditedToolButton ) {
		if( m_pEditedToolButton->parent() ) {
			it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,
				(KviToolBarListViewItem *) m_pEditedToolButton->parent(),
				m_pEditedToolButton, m_szClipboardItemName.ptr(), "item",
				m_szClipboardItemBuffer.ptr(), m_szClipboardItemIcon.ptr()
			);
		} else {
			it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,
				m_pToolBarView, m_pEditedToolButton, m_szClipboardItemName.ptr(), "button",
				m_szClipboardItemBuffer.ptr(), m_szClipboardItemIcon.ptr(), m_szClipboardItemSbar.ptr()
			);
		}
	} else {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,
			m_pToolBarView, m_szClipboardItemName.ptr(), "button",
			m_szClipboardItemBuffer.ptr(), m_szClipboardItemIcon.ptr(), m_szClipboardItemSbar.ptr()
		);
	}
	m_pToolBarView->setSelected(it, true);
}

/**
 * ==================== pasteToolButtonAbove ========================
 */
void KviUserToolBarEditor::pasteToolButtonAbove()
{
	if( !m_pEditedToolButton ) return;

	if( m_pEditedToolButton->parent() != m_pEditedToolButton->itemAbove() )
		return; // Ensure first item

	if( m_szClipboardItemName.isEmpty() )
		return;

	KviToolBarListViewItem *it;
	if( m_pEditedToolButton->parent() ) {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,
			(KviToolBarListViewItem *) m_pEditedToolButton->parent(),
			m_szClipboardItemName.ptr(), "item",
			m_szClipboardItemBuffer.ptr(), m_szClipboardItemIcon.ptr(), ""
		);
	} else {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,
			m_pToolBarView, m_szClipboardItemName.ptr(), "button",
			m_szClipboardItemBuffer.ptr(), m_szClipboardItemIcon.ptr(), m_szClipboardItemSbar.ptr()
		);
	}

	m_pToolBarView->setSelected(it, true);
}

/**
 * ==================== addEmptyToolButtonAbove ========================
 */
void KviUserToolBarEditor::addEmptyToolButtonAbove()
{
	if( !m_pEditedToolButton ) return;

	if( m_pEditedToolButton->parent() != m_pEditedToolButton->itemAbove() )
		return; // Ensure first item

	KviToolBarListViewItem *it;
	if( m_pEditedToolButton->parent() ) {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,
			(KviToolBarListViewItem *) m_pEditedToolButton->parent(),
			"unnamed", "item", _i18n_("# Enter the script for the new item here"), "", ""
		);
	} else {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,
			m_pToolBarView,
			"unnamed", "button", _i18n_("# Enter the script for the new item here"), "", ""
		);
	}
	m_pToolBarView->setSelected(it, true);
}

/**
 * ==================== addToolBarSeparatorItemAbove ========================
 */
void KviUserToolBarEditor::addToolBarSeparatorItemAbove()
{
	if( !m_pEditedToolButton ) return;

	if( m_pEditedToolButton->parent() != m_pEditedToolButton->itemAbove() )
		return; // Ensure first item

	KviToolBarListViewItem *it;
	if( m_pEditedToolButton->parent() ) {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,
			(KviToolBarListViewItem *) m_pEditedToolButton->parent(),
			"----", "separator", "", "", ""
		);
	} else {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,
			m_pToolBarView,
			"----", "separator", "", "", ""
		);
	}
	m_pToolBarView->setSelected(it, true);
}

/**
 * ==================== addToolBarSeparatorItem ========================
 */
void KviUserToolBarEditor::addToolBarSeparatorItem()
{
	KviToolBarListViewItem *it;
	if( m_pEditedToolButton ) {
		if( m_pEditedToolButton->parent() ) {
			it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,
				(KviToolBarListViewItem *) m_pEditedToolButton->parent(),
				m_pEditedToolButton, "----", "separator", _i18n_("# Enter the script for the new item here"), ""
			);
		} else {
			it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,
				m_pToolBarView, m_pEditedToolButton, "----", "separator", "", ""
			);
		}
	} else {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,
			m_pToolBarView, "----", "separator", "", ""
		);
	}
	m_pToolBarView->setSelected(it, true);
}

#include "m_kvi_usertoolbareditor.moc"
