// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptDns"

#include "kvi_debug.h"
#include "kvi_error.h"
#include "kvi_script_dns.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_settings.h"

/*
	@class: dns
	@short:
		A class for async dns lookups
	@inherits:
		[class]object[/class]<br>

	@functions:
		!fn: $resolve(&lt;hostname&gt;)
		This function will start the lookup for the provided &lt;hostname&gt;.<br>
		If the class is in the IPv6 mode, the lookup will be attempted by using the IPv6
		protocol, othervise the standard IPv4 mode will be used.<br>
		This function returns '1' if successful, or '0' in case of a failure.<br>
		This function can fail if the hostname is not valid or if there is already
		another running lookup.<br>

		!fn: $setMode(&lt;mode&gt;)
		Sets the IP protocol version type.<br>
		&lt;mode&gt; can be "ipv6" or "ipv4".<br>
		The default mode of this class is IPv4, and that mode is always working.<br>
		If KVIrc has been compiled with the IPv6 support, the IPv6 mode will be also
		working.<br>
		This function returns '1' if successful, '0' otherwise.<br>

		!fn: $isRunning()
		This function returns '1' if a hostname lookup is currently in progress, '0' otherwise.<br>

		!fn: $mode()
		Returns the string "ipv6" if the object is in the IPv6 mode, the string "ipv4" otherwise.<br>

		!fn: $abort()
		Aborts a running dns lookup.<br>
		Returns 1 if the object was in "lookup" state, '0' otherwise.<br>

		!fn: $query()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the string containing the hostname that was passed to the [classfnc:dns]$resolve[/classfnc]() function

		!fn: $hostname()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the hostname of the resolved host.<br>
		Please note that this may be not the same as returned by [classfnc:dns]$query[/classfnc]().<br>

		!fn: $ip()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the IP address of the resolved host.<br>

		!fn: $alias1()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the alias hostname of the resolved host (if available).<br>

		!fn: $alias2()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the second alias hostname of the resolved host (if available).<br>

		!fn: $success()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		Returns '1' if the last lookup was successful, '0' otherwise (and in that
		case the last error can be retrieved by using [classfnc:dns]$lastError[/classfnc].<br>

		!fn: $lastError()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		Returns the error string in case of an unsuccessful lookup.<br>

	@events:
		!ev: OnDnsDone()
		Triggered when a host has been resolved.<br>
		<b>This event has a default event handler that emits the "done" signal.</b><br>
		If you provide a new handler for this event, and still want the signal to be emitted
		you have to emit the signal by yourself.<br>
		<example>
		[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc](done)
		</example>

	@signals:
		!sg: done()
		Emitted from the default OnDnsDone event handler.<br>

	@description:
		A class that implements asynchronous dns lookups.<br>
		It can lookup the IPv4 hostnames and the IPv6 ones if KVIrc has been
		compiled with such support.<br>
		Basically, you create a dns object instance and either override the [classevent:dns]OnDnsDone[/classevent]
		event or connect the [classsignal:dns]done[/classsignal] signal:
		then call the [classfnc:dns]$resolve[/classfnc]() function.<br>
		In the event handler or signal slot you retrieve the information
		by calling the [classfnc:dns]$ip[/classfnc](), [classfnc:dns]$hostname[/classfnc](),
		[classfnc:dns]$query[/classfnc](), [classfnc:dns]$success[/classfnc](),
		[classfnc:dns]$lastError[/classfnc](), [classfnc:dns]$alias1[/classfnc]() and
		[classfnc:dns]$alias2[/classfnc]() functions.<br>
		The object can resolve only one hostname at once, but can be used
		multiple times.<br>
		A lookup can be aborted by using the [classfnc:dns]$abort[/classfnc]() function.<br>

	@examples:
		Simple example:<br>
		A standalone resolver object: we define a subclass of dns that is useful for experiments.<br>
		<example>
		[cmd]class[/cmd](mydns, dns)
		{
		&nbsp;	[cmd:class]function[/cmd](constructor)
		&nbsp;	{
		&nbsp;		[cmd]connect[/cmd] [fnc]$this[/fnc] done [fnc]$this[/fnc] hostResolved
		&nbsp;	}
		&nbsp;	[cmd:class]function[/cmd](hostResolved)
		&nbsp;	{
		&nbsp;		[cmd]echo[/cmd] Success ? : [fnc]$this[/fnc]->[classfnc:dns]$success[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Last error : [fnc]$this[/fnc]->[classfnc:dns]$lastError[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Query : [fnc]$this[/fnc]->[classfnc:dns]$query[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Hostname : [fnc]$this[/fnc]->[classfnc:dns]$hostname[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] IP : [fnc]$this[/fnc]->[classfnc:dns]$ip[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Alias1 : [fnc]$this[/fnc]->[classfnc:dns]$alias1[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Alias2 : [fnc]$this[/fnc]->[classfnc:dns]$alias2[/classfnc]()
		&nbsp;	}
		}
		</example>
		Copy the example above to the code tester in the script dialog and execute it once.<br>
		You will have the "mydns" class defined.<br>
		Now you can create an instance of this class by typing in the console:<br>
		<example>
		%X = [fnc]$new[/fnc](mydns, [fnc]$root[/fnc], mytestdns)
		</example>
		The object id in the variable %X now points to the instance.<br>
		Now you are able to resolve hosts by typing:<br>
		<example>
		%X->[classfnc:dns]$resolve[/classfnc](&lt;hostname_here&gt;)
		</example>
		The result will be "echoed" to the console window.<br>
		Try it with valid and invalid hostnames, and also with IP addresses.<br>
		You can also test the $abort function and the return values from each call.<br>
	@seealso:
		class [class]object[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * DNS class
 */
void KviScriptDns::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("resolve",   (scriptObjectFunction) &KviScriptDns::builtinFunction_RESOLVE);
	d->addBuiltinFunction("isRunning", (scriptObjectFunction) &KviScriptDns::builtinFunction_ISRUNNING);
	d->addBuiltinFunction("setMode",   (scriptObjectFunction) &KviScriptDns::builtinFunction_SETMODE);
	d->addBuiltinFunction("abort",     (scriptObjectFunction) &KviScriptDns::builtinFunction_ABORT);
	d->addBuiltinFunction("query",     (scriptObjectFunction) &KviScriptDns::builtinFunction_QUERY);
	d->addBuiltinFunction("hostname",  (scriptObjectFunction) &KviScriptDns::builtinFunction_HOSTNAME);
	d->addBuiltinFunction("ip",        (scriptObjectFunction) &KviScriptDns::builtinFunction_IP);
	d->addBuiltinFunction("alias1",    (scriptObjectFunction) &KviScriptDns::builtinFunction_ALIAS1);
	d->addBuiltinFunction("alias2",    (scriptObjectFunction) &KviScriptDns::builtinFunction_ALIAS2);
	d->addBuiltinFunction("success",   (scriptObjectFunction) &KviScriptDns::builtinFunction_SUCCESS);
	d->addBuiltinFunction("lastError", (scriptObjectFunction) &KviScriptDns::builtinFunction_LASTERROR);
	d->addBuiltinFunction("mode",      (scriptObjectFunction) &KviScriptDns::builtinFunction_MODE);

	KviScriptEventStruct *s = new KviScriptEventStruct();
	s->szName   = "OnDnsDone";
	s->szBuffer = "$this->$emit(done)";
	d->addDefaultEvent(s);
}

KviScriptDns::KviScriptDns(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptObject(cntrl, p, name, pDef)
{
	m_pDns = new KviDns();
	connect(m_pDns, SIGNAL(finished(KviDnsData *)), this, SLOT(finished(KviDnsData *)));
	m_bIPv6    = false;
	m_pDnsData = 0;
}

KviScriptDns::~KviScriptDns()
{
	delete m_pDns;
	m_pDns = 0;
	if( m_pDnsData ) {
		delete m_pDnsData;
		m_pDnsData = 0;
	}
}

void KviScriptDns::finished(KviDnsData *dns)
{
	__range_valid(m_pDnsData == 0);
	m_pDnsData = m_pDns->releaseData();
	KviStr params;
	triggerEvent("OnDnsDone", params);
}

int KviScriptDns::builtinFunction_RESOLVE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( m_pDns->isWorking() ) {
		buffer.append('0');
		return KVI_ERROR_Success;
	}
	if( m_pDnsData ) {
		delete m_pDnsData;
		m_pDnsData = 0;
	}
	if( !params )
		return KVI_ERROR_MissingParameter;
	KviStr *pStr = params->first();
	if( !pStr )
		return KVI_ERROR_MissingParameter;
	if( !m_pDns->resolve(pStr->ptr(), m_bIPv6) )
		buffer.append('0');
	else
		buffer.append('1');
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_ISRUNNING(QPtrList<KviStr> *params, KviStr &buffer)
{
	buffer.append(m_pDns->isWorking() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_MODE(QPtrList<KviStr> *params, KviStr &buffer)
{
	buffer.append(m_bIPv6 ? "ipv6" : "ipv4");
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_QUERY(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( m_pDnsData )
		buffer.append(m_pDnsData->hostname);
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_HOSTNAME(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( m_pDnsData )
		buffer.append(m_pDnsData->hostname);
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_IP(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( m_pDnsData && !m_pDnsData->addresses.empty() ) {
		QHostAddress addr = m_pDnsData->addresses.first();
		if( !addr.isNull() )
			buffer.append(addr.toString());
	}
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_ALIAS1(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( m_pDnsData && (m_pDnsData->addresses.count() > 1) ) {
		QHostAddress addr = m_pDnsData->addresses[1];
		if( !addr.isNull() )
			buffer.append(addr.toString());
	}
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_ALIAS2(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( m_pDnsData && (m_pDnsData->addresses.count() > 2) ) {
		QHostAddress addr = m_pDnsData->addresses[2];
		if( !addr.isNull() )
			buffer.append(addr.toString());
	}
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_SUCCESS(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( m_pDnsData )
		buffer.append((m_pDnsData->iError == KVI_ERROR_Success) ? '1' : '0');
	else
		buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_LASTERROR(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( m_pDnsData )
		buffer.append(kvi_getErrorString(m_pDnsData->iError));
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_ABORT(QPtrList<KviStr> *params, KviStr &buffer)
{
	buffer.append(m_pDns->abort() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_SETMODE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( !params ) return KVI_ERROR_MissingParameter;

	KviStr *pStr = params->first();
	if( !pStr )
		return KVI_ERROR_MissingParameter;
	if( kvi_strEqualCI(pStr->ptr(), "ipv6") ) {
#ifdef COMPILE_NEED_IPV6
		m_bIPv6 = true;
#else
		buffer.append('0'); // No support
		return KVI_ERROR_Success;
#endif
	} else if( kvi_strEqualCI(pStr->ptr(), "ipv4") )
		m_bIPv6 = false;
	else
		return KVI_ERROR_InvalidParameter;
	buffer.append('1');
	return KVI_ERROR_Success;
}

#include "m_kvi_script_dns.moc"
