// =============================================================================
//
//      --- kvi_options_logging.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviOptionsLogging"

#include <qheader.h>
#include <qlayout.h>

#include "kvi_boolselector.h"
#include "kvi_defines.h"
#include "kvi_label.h"
#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_messagetypes.h"
#include "kvi_options.h"
#include "kvi_options_logging.h"
#include "kvi_stringselector.h"

// Declared in kvi_app.cpp and managed by KviApp class
extern QPixmap *g_pixViewOut[KVI_OUT_NUM_IMAGES];

/*
	@quickhelp: KviOptionsLogging
	@widget: Logging options
		Here you can setup logging of KVIrc windows.<br>
		Each KVIrc window view has the logging feature built-in.<br>
		You can enable the logging by clicking with the middle button
		in the window view and selecting "start logging".<br>
		If you do not want to repeat this procedure for each window
		that you want to log, you can set the "auto-enable logging" option
		for the specified window type.<br>
		For example, if you auto-enable logging in query windows, logging
		will be automatically started in all the newly opened query windows.<br>
		Please note that this will NOT enable the logging for the currently open windows,
		you must do it "by hand".<br>
		The log filename will be automatically generated from the window caption, time and date.<br>
		If you use to stay on multiple servers at the same time it may be useful
		to add the server name to the filename of the log.<br>
		This will prevent log filenames collision when you are on channels with the same name
		but on different server.<br>
		Finally, you can enable or disable logging for specific message types.<br>
		For example you might want to avoid to log the 'internally started processes output'.<br>
*/
KviOptionsLogging::KviOptionsLogging(QWidget *parent)
	: KviOptionsWidget(parent, KVI_GENERAL_OPTIONS_WIDGET_ID_LOGGING)
{
	QGridLayout *g = new QGridLayout(
		this, 11, 1, KVI_GENERAL_OPTIONS_GRIDLAYOUT_BORDER, KVI_GENERAL_OPTIONS_GRIDLAYOUT_SPACE
	);

	KviBoolSelector *b = new KviBoolSelector(this,
		_i18n_("Automatically enable logging in console window"), &(g_pOptions->m_bAutoLogConsole)
	);
	g->addWidget(b, 0, 0);
	b = new KviBoolSelector(this,
		_i18n_("Automatically enable logging in channel windows"), &(g_pOptions->m_bAutoLogChannels)
	);
	g->addWidget(b, 1, 0);
	b = new KviBoolSelector(this,
		_i18n_("Automatically enable logging in query windows"), &(g_pOptions->m_bAutoLogQueries)
	);
	g->addWidget(b, 2, 0);
	b = new KviBoolSelector(this,
		_i18n_("Automatically enable logging in DCC chat windows"), &(g_pOptions->m_bAutoLogChats)
	);
	g->addWidget(b, 3, 0);
	b = new KviBoolSelector(this,
		_i18n_("Automatically enable logging in DCC send windows"), &(g_pOptions->m_bAutoLogSends)
	);
	g->addWidget(b, 4, 0);

	QFrame *f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addWidget(f, 5, 0);

	b = new KviBoolSelector(this,
		_i18n_("Append server name to log filename (if possible)"), &(g_pOptions->m_bAppendServerNameToLogName)
	);
	g->addWidget(b, 6, 0);

	b = new KviBoolSelector(this, _i18n_("Append date to log filename"), &(g_pOptions->m_bAppendDateToLogName));
	g->addWidget(b, 7, 0);

	f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addWidget(f, 8, 0);

	KviLabel *l = new KviLabel(_CHAR_2_QSTRING(_i18n_("Log message types")), this);
	g->addWidget(l, 9, 0);

	m_pLogMsgTypesView = new KviListView(this);
	m_pLogMsgTypesView->addColumn(_CHAR_2_QSTRING(_i18n_("Message Type")));
	m_pLogMsgTypesView->addColumn(_CHAR_2_QSTRING(_i18n_("Enable Logging")));
	m_pLogMsgTypesView->setSorting(-1);
	m_pLogMsgTypesView->header()->setClickEnabled(false);
	m_pLogMsgTypesView->setMultiSelection(false);

	KviListViewItem *it = 0;
	// Do not translate yes and no here!!!
	for( int i = 0; i < KVI_OUT_NUM_IMAGES; i++ ) {
		it = new KviListViewItem(m_pLogMsgTypesView,
			_CHAR_2_QSTRING(_i18n_(kvi_messagetypes[i])),
			(g_pOptions->m_bLogMsgType[i] ? _CHAR_2_QSTRING("yes") : _CHAR_2_QSTRING("no"))
		);
		it->setPixmap(0, (*g_pixViewOut[i]));
	}

	connect(m_pLogMsgTypesView,
		SIGNAL(doubleClicked(KviListViewItem *)),
		this,
		SLOT(logMsgTypeViewItemDoubleClicked(KviListViewItem *))
	);

	g->addWidget(m_pLogMsgTypesView, 10, 0);
	g->setRowStretch(10, 2);
}

KviOptionsLogging::~KviOptionsLogging()
{
	// Nothing here
}

void KviOptionsLogging::logMsgTypeViewItemDoubleClicked(KviListViewItem *it)
{
	if( !it ) return;

	KviStr state = it->text(1);
	if( kvi_strEqualCIN("yes", state.ptr(), 3) )
		it->setText(1, _CHAR_2_QSTRING("no"));
	else
		it->setText(1, _CHAR_2_QSTRING("yes"));
}

void KviOptionsLogging::commitChanges()
{
	KviBoolSelector::commitAll(this);
	KviListViewItem *it = m_pLogMsgTypesView->firstChild();
	while( it ) {
		// Cannot rely on sorting order of the KviListView :(
		KviStr tmp = it->text(0);
		for( int i = 0; tmp.hasData() && (i < KVI_OUT_NUM_IMAGES); i++ ) {
			if( kvi_strEqualCI(_i18n_(kvi_messagetypes[i]), tmp.ptr()) ) {
				tmp = it->text(1);
				g_pOptions->m_bLogMsgType[i] = kvi_strEqualCIN("yes", tmp.ptr(), 3);
				tmp = "";
			}
		}
		if( tmp.hasData() )
			debug(_i18n_("Warning: cannot update logging state for message type %s"), tmp.ptr());
		it = it->nextSibling();
	}
}

#include "m_kvi_options_logging.moc"
