// =============================================================================
//
//      --- kvi_app.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 19f99-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviApp"

#define _KVI_APP_CPP_

#include "kvi_settings.h"

#include <qclipboard.h>
#include <qdir.h>
#include <qsplitter.h>
#include <qstylefactory.h>
#include <qtimer.h>
#ifdef COMPILE_WITH_KDE
	#include <kconfig.h>
	#include <kimageio.h>
	#include <kstyle.h>
#endif

#include "kvi_alias.h"
#include "kvi_app.h"
#include "kvi_colorwindow.h"
#include "kvi_config.h"
#include "kvi_console.h"
#include "kvi_debug.h"
#include "kvi_dirbrowser.h"
#include "kvi_dirbrowser_widget.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_helpqml.h"
#include "kvi_iconloader.h"
#include "kvi_imagelibrary.h"
#include "kvi_irc_server.h"
#include "kvi_irc_view.h"
#include "kvi_jpegio.h"
#include "kvi_loadscript.h"
#include "kvi_locale.h"
#include "kvi_mainmenu.h"
#include "kvi_malloc.h"
#include "kvi_mimemanager.h"
#include "kvi_messagebox.h"
#include "kvi_options.h"
#include "kvi_optionsdialog.h"
#include "kvi_rawevent.h"
#include "kvi_regusers.h"
#include "kvi_script_center.h"
#include "kvi_script_wizard.h"
#include "kvi_serverdialog.h"
#include "kvi_splashscreen.h"
#include "kvi_tipoftheday.h"
#include "kvi_userdialog.h"
#include "kvi_userparser.h"
#include "kvi_userpopupmenu.h"
#include "kvi_usertoolbartemplate.h"
#include "kvi_variablecache.h"
#include "kvi_wmdock.h"
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	#include "kvi_charset_translator.h"
	KviCharsetTranslator *g_pCharsetTranslator;
#endif

/**
 * Global application pointer.
 * Auto declared as extern in kvi_app.h
 */
KviApp *g_pApp;

// Other global stuff.
// Need to explicitly declare it as extern in other files
KviColorWindow *g_pColorWindow;
KviPopupMenu   *g_pInputPopup;
KviPopupMenu   *g_pTaskBarPopup;
KviPopupMenu   *g_pViewSelectPopup;
KviPopupMenu   *g_pWindowPopup;

QPixmap *g_pListBoxOpPixmap;
QPixmap *g_pListBoxVoicePixmap;
QPixmap *g_pListBoxHelpOpPixmap;
QPixmap *g_pListBoxUserOpPixmap;
QPixmap *g_pListBoxOwnerPixmap;

// Used by KviIrcView
QPixmap *g_pixViewOut[KVI_OUT_NUM_IMAGES];

// Used here and in KviOptions
KviSplashScreen *g_pSplashScreen;

// Created and destroyed here, loaded and saved by KviOptions
KviAliasManager    *g_pAliasManager;
KviEventManager    *g_pEventManager;
KviVariableCache   *g_pVarCache;
KviRawEventManager *g_pRawEventManager;

KviUserPopupMenu *g_pChannelPopup;
KviUserPopupMenu *g_pUserlistPopup;
KviUserPopupMenu *g_pConsolePopup;
KviUserPopupMenu *g_pNotifylistPopup;
KviUserPopupMenu *g_pQueryPopup;
KviUserPopupMenu *g_pDccChatPopup;

KviUserToolBarTemplate *g_pUserToolBarTemplate;

KviConfig *g_pUserConfig;

#ifdef COMPILE_PLUGIN_SUPPORT
	KviPluginManager *g_pPluginManager;
#endif

// IRC VIEW GLOBAL
QPixmap *g_pIrcViewMemBuffer;

Qt::HANDLE            g_hIrcViewMemBuffer;
QPtrList<KviIrcView> *g_pIrcViewWidgetList;

void g_setSplashScreenText(const char *text)
{
	if( g_pSplashScreen )
		g_pSplashScreen->setText(text);
}

void g_setSplashScreenProgress(int progress)
{
	if( g_pSplashScreen )
		g_pSplashScreen->setProgress(progress);
}

QStyle *g_getMotifStyle()      { return QStyleFactory::create("motif"); }
QStyle *g_getCDEStyle()        { return QStyleFactory::create("cde"); }
QStyle *g_getCompactStyle()    { return QStyleFactory::create("compact"); }
QStyle *g_getMotifPlusStyle()  { return QStyleFactory::create("motifplus"); }
QStyle *g_getSGIStyle()        { return QStyleFactory::create("sgi"); }
QStyle *g_getWindowsStyle()    { return QStyleFactory::create("windows"); }
QStyle *g_getPlatinumStyle()   { return QStyleFactory::create("platinum"); }

// =============================== KviApp ===================================
/**
 * Application constructor
 */
KviApp::KviApp(int argc, char **argv)
	: KviApplication(argc, argv, "kvirc")
{
	m_startupTime = QDateTime::currentDateTime();

#ifdef COMPILE_JPEG_SUPPORT
	kvi_initJpegIo();
#endif

#ifdef COMPILE_WITH_KDE
	KImageIO::registerFormats(); // Support all the formats supported by KDE
#endif
	// Initialize the main thread (setup signal handlers etc.)
	// Create our lists
	m_pFrameList   = new QPtrList<KviFrame>;
	m_pFrameList->setAutoDelete(false);
	m_pDestroyList = new QPtrList<QObject>;
	m_pDestroyList->setAutoDelete(true);
	m_pStyleList   = new QPtrList<KviStyleEntry>;
	m_pStyleList->setAutoDelete(true);
	m_pIconCache   = new QAsciiDict<QPixmap>;
	m_pIconCache->setAutoDelete(true);
	// Config stuff
	m_pConfigFile   = 0;
	m_pCreateConfig = false;
	// Cleanup all our pointers to avoid problems.
	// Do not call anything else that could use the g_pApp pointer.
	// It is not set yet here!
	m_pUserOptionsDialog     = 0;
	m_pServerOptionsDialog   = 0;
	m_pScriptCenter          = 0;
	m_pRegUsersDialog        = 0;
	m_pTipOfTheDayDialog     = 0;
	m_pOptionsDialog         = 0;
	m_pScriptWizard          = 0;
	m_pWMDockWidget          = 0;
	g_pSplashScreen          = 0;
	m_bGuiUpdatePending      = false;
	m_bGlobalRepaintPending  = false;
	m_savedAppPalette        = palette();
	m_szLastGUIStyle         = "default";
	g_pUserToolBarTemplate   = 0;
}

// ============================= setup ===============================
/**
 * Here we initialize the whole engine
 */
bool KviApp::setup()
{
	// Find the standard directories
	if( !loadDirectories() )
		return false;
	// Now we can load the splash image... create the splash screen
	createSplashScreen();
	g_setSplashScreenProgress(0);
	g_setSplashScreenText(_i18n_("Loading icons..."));
	// Now we have a first widget out on the world.
	g_pIrcViewWidgetList = new QPtrList<KviIrcView>;
	g_pIrcViewWidgetList->setAutoDelete(false);
	g_pIrcViewMemBuffer = new QPixmap(16, 16); // Just dummy 16x16 pixmap by now
	g_hIrcViewMemBuffer = g_pIrcViewMemBuffer->handle();
	// Load all our images
	loadPixmaps(); // 0-13%
	// Create globals
	g_setSplashScreenText(_i18n_("Preparing to parse config files..."));
	// The KviPopups MUST be created before KviOptions (and destroyed after)
	g_pChannelPopup        = new KviUserPopupMenu(0, "channel_script_popup");
	g_pConsolePopup        = new KviUserPopupMenu(0, "console_script_popup");
	g_pUserlistPopup       = new KviUserPopupMenu(0, "userlist_script_popup");
	g_pNotifylistPopup     = new KviUserPopupMenu(0, "notifylist_script_popup");
	g_pQueryPopup          = new KviUserPopupMenu(0, "query_script_popup");
	g_pDccChatPopup        = new KviUserPopupMenu(0, "dccchat_script_popup");

	g_pUserToolBarTemplate = new KviUserToolBarTemplate();

	g_pVarCache            = new KviVariableCache();
	openUserConfig();
	g_pAliasManager        = new KviAliasManager();  // This MUST be created BEFORE the g_pOptions
	g_pEventManager        = new KviEventManager();  // Same as the above
	g_pRawEventManager     = new KviRawEventManager();
#ifdef COMPILE_PLUGIN_SUPPORT
	g_pPluginManager       = new KviPluginManager(); // This MUST be created AFTER g_pEventManager
#endif

#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	g_pCharsetTranslator   = new KviCharsetTranslator();
#endif

	g_pOptions = new KviOptions(); // 14-83%  (memory allocation and reading on disks eats really much time)
	// Set the fonts for the popups created before g_pOptions
	g_pChannelPopup->setFont(g_pOptions->m_fntApplication);
	g_pConsolePopup->setFont(g_pOptions->m_fntApplication);
	g_pUserlistPopup->setFont(g_pOptions->m_fntApplication);
	g_pNotifylistPopup->setFont(g_pOptions->m_fntApplication);
	g_pQueryPopup->setFont(g_pOptions->m_fntApplication);
	g_pDccChatPopup->setFont(g_pOptions->m_fntApplication);
	// Set the fonts for the rest of the widgets...
	setFont(g_pOptions->m_fntApplication); // Do not repaint widgets

	if( g_pOptions->m_bSetGlobalPaletteAtStartup ) {
		QColorGroup cg(
			g_pOptions->m_clrNormalGlobalForeground,
			g_pOptions->m_clrNormalGlobalButton,
			g_pOptions->m_clrNormalGlobalLight,
			g_pOptions->m_clrNormalGlobalDark,
			g_pOptions->m_clrNormalGlobalMid,
			g_pOptions->m_clrNormalGlobalText,
			g_pOptions->m_clrNormalGlobalText.light(150),
			g_pOptions->m_clrNormalGlobalBase,
			g_pOptions->m_clrNormalGlobalButton
		);
		QColorGroup cgdis(
			g_pOptions->m_clrDisabledGlobalForeground,
			g_pOptions->m_clrNormalGlobalButton,
			g_pOptions->m_clrNormalGlobalLight,
			g_pOptions->m_clrNormalGlobalDark,
			g_pOptions->m_clrNormalGlobalMid,
			g_pOptions->m_clrDisabledGlobalText,
			g_pOptions->m_clrNormalGlobalText.light(150),
			g_pOptions->m_clrDisabledGlobalBase,
			g_pOptions->m_clrNormalGlobalButton
		);
		QPalette pal(cg, cgdis, cg);
		g_pChannelPopup->setPaletteRecursive(pal);
		g_pConsolePopup->setPaletteRecursive(pal);
		g_pUserlistPopup->setPaletteRecursive(pal);
		g_pNotifylistPopup->setPaletteRecursive(pal);
		g_pQueryPopup->setPaletteRecursive(pal);
		g_pDccChatPopup->setPaletteRecursive(pal);
		setPalette(pal); // Do not repaint widgets
	}

	g_pColorWindow     = new KviColorWindow(); // 84-88%
	g_pInputPopup      = new KviPopupMenu();
	g_pTaskBarPopup    = new KviPopupMenu();
	g_pViewSelectPopup = new KviPopupMenu();
	g_pWindowPopup     = new KviPopupMenu();

	g_setSplashScreenProgress(88);
	g_setSplashScreenText(_i18n_("Booting the nuclear reactor..."));
	createQMLProvider();
	m_szLastHelpDocument = KVI_HELP_QML_MAIN_TOPIC_FILE;

	g_setSplashScreenProgress(90);

	KviFrame *frm = new KviFrame(); // 89-100 (do not show yet)

	g_setSplashScreenProgress(100);
	g_setSplashScreenText(_i18n_("Go!"));

	destroySplashScreen();

	registerStyle("CDE",         0, g_getCDEStyle);
	registerStyle("Compact",     0, g_getCompactStyle);
	registerStyle("Motif",       0, g_getMotifStyle);
	registerStyle("Motif+",      0, g_getMotifPlusStyle);
	registerStyle("Platinum",    0, g_getPlatinumStyle);
	registerStyle("SGI",         0, g_getSGIStyle);
	registerStyle("Windows",     0, g_getWindowsStyle);

	setApplicationStyle();
	// Call it this way to ensure that the frame is inserted correctly
	__range_valid(frm == m_pFrameList->first());
	frm->show();

	if( g_pOptions->m_bEnableDockWidget         ) slot_createWMDockWidget();
	if( g_pOptions->m_bShowTipOfTheDayAtStartup ) slot_doTipOfTheDay();
	return true;
}

// ============ ~KviApp ============
KviApp::~KviApp()
{
	destroy();
}

void KviApp::destroy()
{
	saveKVIrcDirectory();
	// Remove the dialogs if needed
	if( m_pUserOptionsDialog    )  { delete m_pUserOptionsDialog;     m_pUserOptionsDialog     = 0; }
	if( m_pServerOptionsDialog  )  { delete m_pServerOptionsDialog;   m_pServerOptionsDialog   = 0; }
	if( m_pScriptCenter         )  { delete m_pScriptCenter;          m_pScriptCenter          = 0; }
	if( m_pRegUsersDialog       )  { delete m_pRegUsersDialog;        m_pRegUsersDialog        = 0; }
	if( m_pTipOfTheDayDialog    )  { delete m_pTipOfTheDayDialog;     m_pTipOfTheDayDialog     = 0; }
	if( m_pScriptWizard         )  { delete m_pScriptWizard;          m_pScriptWizard          = 0; }
	if( m_pWMDockWidget         )  { delete m_pWMDockWidget;          m_pWMDockWidget          = 0; }
	if( m_pOptionsDialog        )  { delete m_pOptionsDialog;         m_pOptionsDialog         = 0; }
	// There should be no remaining windows now in the list...
	__range_valid(m_pFrameList->isEmpty());
	if( m_pFrameList )             { delete m_pFrameList;             m_pFrameList             = 0; }
	m_pFrameList = 0;
	// No more frames = no more widgets = no more paint requests
	__range_valid(g_pIrcViewWidgetList->isEmpty());
	if( g_pIrcViewWidgetList )     { delete g_pIrcViewWidgetList;     g_pIrcViewWidgetList     = 0; }

	emptyDestroyList();
	if( m_pDestroyList )           { delete m_pDestroyList;           m_pDestroyList           = 0; }

	// Remove globals
	if( g_pOptions )               { delete g_pOptions;               g_pOptions               = 0; }
	if( g_pUserConfig   )          { delete g_pUserConfig;            g_pUserConfig            = 0; }
	if( g_pVarCache     )          { delete g_pVarCache;              g_pVarCache              = 0; }
	// This must be removed AFTER g_pOptions:
	if( g_pAliasManager )          { delete g_pAliasManager;          g_pAliasManager          = 0; }
#ifdef COMPILE_PLUGIN_SUPPORT
	// This must be removed BEFORE g_pEventManager (will unload all the plugins and unregister handles)
	// Must be also removed AFTER g_pOptions.
	if( g_pPluginManager )         { delete g_pPluginManager;         g_pPluginManager         = 0; }
#endif
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	if( g_pCharsetTranslator )     { delete g_pCharsetTranslator;     g_pCharsetTranslator     = 0; }
#endif
	// These must be removed AFTER g_pOptions:
	if( g_pEventManager       )    { delete g_pEventManager;          g_pEventManager          = 0; }
	if( g_pRawEventManager    )    { delete g_pRawEventManager;       g_pRawEventManager       = 0; }
	if( g_pColorWindow        )    { delete g_pColorWindow;           g_pColorWindow           = 0; }
	if( g_pInputPopup         )    { delete g_pInputPopup;            g_pInputPopup            = 0; }
	if( g_pTaskBarPopup       )    { delete g_pTaskBarPopup;          g_pTaskBarPopup          = 0; }
	if( g_pViewSelectPopup    )    { delete g_pViewSelectPopup;       g_pViewSelectPopup       = 0; }
	if( g_pChannelPopup       )    { delete g_pChannelPopup;          g_pChannelPopup          = 0; }
	if( g_pConsolePopup       )    { delete g_pConsolePopup;          g_pConsolePopup          = 0; }
	if( g_pUserlistPopup      )    { delete g_pUserlistPopup;         g_pUserlistPopup         = 0; }
	if( g_pNotifylistPopup    )    { delete g_pNotifylistPopup;       g_pNotifylistPopup       = 0; }
	if( g_pQueryPopup         )    { delete g_pQueryPopup;            g_pQueryPopup            = 0; }
	if( g_pDccChatPopup       )    { delete g_pDccChatPopup;          g_pDccChatPopup          = 0; }
	if( g_pWindowPopup        )    { delete g_pWindowPopup;           g_pWindowPopup           = 0; }
	if( g_pUserToolBarTemplate)    { delete g_pUserToolBarTemplate;   g_pUserToolBarTemplate   = 0; }
	destroyPixmaps();
	destroySplashScreen(); // Make sure that it is destroyed
	// Delete this as late as possible...
	kvi_destroyLocale(this);
	unregisterStylesFor(0); // Crash test :)
	if( m_pStyleList )             { delete m_pStyleList;             m_pStyleList             = 0; }
	if( m_pIconCache )             { delete m_pIconCache;             m_pIconCache             = 0; }
}

bool KviApp::scriptCenterDialogOpen()
{
	return m_pScriptCenter != 0;
}

KviUserParser *KviApp::anyUserParser()
{
	KviFrame *f = m_pFrameList->first();
	return f ? f->m_pUserParser : 0;
}

bool KviApp::cacheIcon(const char *filename)
{
	if( m_pIconCache->find(filename) )
		return true;
	QPixmap *pix = new QPixmap(KviIconLoader::loadMimeIcon(filename));
	__range_valid(pix);
	if( pix->isNull() ) {
		delete pix;
		debug("Could not load icon file %s", filename);
		return false;
	} else m_pIconCache->insert(filename, pix);
	return true;
}

void KviApp::refreshMimeTypeIcons()
{
	m_pIconCache->clear();
	KviMimeType *m;
	for( m = g_pOptions->m_pMimeManager->m_pMimeList->first(); m; m = g_pOptions->m_pMimeManager->m_pMimeList->next() ) {
		m->iconPath.stripWhiteSpace();
		if( m->iconPath.hasData() ) {
			cacheIcon(m->iconPath.ptr());
		}
	}
}

QPixmap *KviApp::getFileIcon(const char *filename)
{
	QPixmap *pix = m_pIconCache->find(filename);
	if( !pix ) {
		if( cacheIcon(filename) )
			return m_pIconCache->find(filename);
	}
	return pix;
}

void KviApp::registerStyle(const char *name, void *plugin_handle, QStyle *(*style_provider_routine)())
{
	KviStyleEntry *e = new KviStyleEntry();
	e->name             = name;
	e->plugin_handle    = plugin_handle;
	e->get_style_object = style_provider_routine;
	m_pStyleList->append(e);
}

void KviApp::unregisterStyle(const char *name, void *plugin_handle)
{
	KviStyleEntry *e = 0;
	for( e = m_pStyleList->first(); e; e = m_pStyleList->next() ) {
		if( kvi_strEqualCI(e->name.ptr(), name) ) {
			if( plugin_handle == e->plugin_handle ) {
				if( g_pOptions ) { // Options may be zero when closing down the app
					// Not closing down :)
					if( kvi_strEqualCI(e->name.ptr(), g_pOptions->m_szApplicationGUIStyle.ptr()) ) {
						// This is the current style. Hmmm...
						// The object code may be not available anymore
						g_pOptions->m_szApplicationGUIStyle = "default";
						setApplicationStyle();
					}
				} // else: g_pOptions already deleted :)
				m_pStyleList->removeRef(e);
				return;
			}
		}
	}
}

void KviApp::unregisterStylesFor(void *plugin_handle)
{
	QPtrList<KviStyleEntry> l;
	l.setAutoDelete(false);
	KviStyleEntry *e = 0;
	for( e = m_pStyleList->first(); e; e = m_pStyleList->next() ) {
		if( plugin_handle == e->plugin_handle )
			l.append(e);
	}
	while( l.first() ) {
		unregisterStyle(l.first()->name.ptr(), plugin_handle);
		l.removeFirst();
	}
}

/**
 * Do a safe quit...
 * Each frame is self-deleted in the close event
 * and removes itself from m_pFrameList in the destructor.
 */
void KviApp::slot_safeQuit()
{
	while( !m_pFrameList->isEmpty() )
		m_pFrameList->first()->close();
	// When the last frame is closed, the app should exit automatically...
}

void KviApp::destroyLater(QObject *o)
{
	m_pDestroyList->append(o);
	QTimer::singleShot(100, this, SLOT(emptyDestroyList()));
}

void KviApp::emptyDestroyList()
{
	while( !m_pDestroyList->isEmpty() )
		m_pDestroyList->removeFirst();
}

void KviApp::connectAfterStartup(const char *server, const char *port)
{
	__range_valid(m_pFrameList->first()); // We must be called after setup()
	g_pOptions->m_pServerManager->iWantThisServerToBeCurrent(server ? server : "irc.openprojects.net", port ? port : "6667");
	// Jump out of this handler, call it from the main event loop...
	m_pFrameList->first()->doAsyncServerCommandCall(); // Will run in exec() loop
}

void KviApp::setApplicationStyle()
{
	if( kvi_strEqualCI(m_szLastGUIStyle.ptr(), g_pOptions->m_szApplicationGUIStyle.ptr()) )
		return;

	QStyle *s = 0;
	if( kvi_strEqualCI(g_pOptions->m_szApplicationGUIStyle.ptr(), "default") ) {
		// Default (NONE) style
#ifdef COMPILE_WITH_KDE
		s = 0;
#else
		s = g_getMotifStyle(); // Qt default
#endif
	} else {
		// Lookup in the registered styles table
		for( KviStyleEntry *e = m_pStyleList->first(); e; e = m_pStyleList->next() ) {
			if( kvi_strEqualCI(g_pOptions->m_szApplicationGUIStyle.ptr(), e->name.ptr()) ) {
				// Got the style
				s = e->get_style_object();
			}
		}
	}

	if( !s ) {
		// For some reason we got a null style...
		// Fall back to the default
		g_pOptions->m_szApplicationGUIStyle = "default";
#ifdef COMPILE_WITH_KDE
		disableStyles();
		enableStyles();
#else
		s = g_getMotifStyle();
#endif
	}

	m_szLastGUIStyle = g_pOptions->m_szApplicationGUIStyle;

	if( s ) setStyle(s);
}

// =========== createSplashScreen ============
void KviApp::createSplashScreen()
{
	KviStr szImage;
	g_pSplashScreen = 0;
	if( findImage(szImage, KVI_SPLASH_IMAGE_NAME) ) {
		QPixmap pix(szImage.ptr());
		if( !pix.isNull() ) {
			g_pSplashScreen = new KviSplashScreen(pix);
		}
	}
	if( !g_pSplashScreen ) {
		debug("Could not load the splash screen image %s", szImage.ptr());
		g_pSplashScreen = new KviSplashScreen(); // No splash image?
	}
	g_pSplashScreen->show();
}

// =========== destroySplashScreen ============
void KviApp::destroySplashScreen()
{
	if( g_pSplashScreen ) {
		delete g_pSplashScreen;
		g_pSplashScreen = 0;
	}
}

QString KviApp::getClipboardText()
{
	QString buffer;
	buffer = clipboard()->text(QClipboard::Selection);
	if( buffer.isEmpty() ) // Look up the global clipboard
		buffer = clipboard()->text(QClipboard::Clipboard);
	return buffer;
}

// =============================================================================
//
// ### Load settings and create globals ###
//
// =============================================================================

void KviApp::createQMLProvider()
{
	// Set the default help files search path
	QStringList list;
	KviStr tmp;
	getLocalKVIrcDirectory(tmp, Help);
	list.append(tmp.ptr());
	getGlobalKVIrcDirectory(tmp, Help);
	list.append(tmp.ptr());
	getGlobalKVIrcDirectory(tmp, HelpEN);
	list.append(tmp.ptr());
	QMimeSourceFactory::defaultFactory()->setFilePath(list);
	QMimeSourceFactory::defaultFactory()->setExtensionType("kvihelp", "text/html");

	KviHelpQML *s = new KviHelpQML(this);
	QStyleSheet::setDefaultSheet(s);
}

void KviApp::loadPixmaps()
{
	// This function should set progress from 0 to 21 %
	KviStr szP1;
	KviStr szP2;
	g_setSplashScreenProgress(1);

	// List box: two images
	getLocalKVIrcDirectory (szP1, Pics, KVI_LISTBOX_IMAGE_LIBRARY_NAME);
	getGlobalKVIrcDirectory(szP2, Pics, KVI_LISTBOX_IMAGE_LIBRARY_NAME);
	KviImageLibrary lib_listbox(szP1.ptr(), szP2.ptr(), 16, 16);
	g_setSplashScreenProgress(5);
	if( !lib_listbox.libraryLoaded() )
		warningBox(_i18n_("The image library %s cannot be found."), KVI_LISTBOX_IMAGE_LIBRARY_NAME);
	g_pListBoxOpPixmap     = new QPixmap(lib_listbox.getImage(0));
	g_pListBoxVoicePixmap  = new QPixmap(lib_listbox.getImage(1));
	g_pListBoxHelpOpPixmap = new QPixmap(lib_listbox.getImage(2));
	g_pListBoxUserOpPixmap = new QPixmap(lib_listbox.getImage(3));
	g_pListBoxOwnerPixmap  = new QPixmap(lib_listbox.getImage(4));

	// Small image (view icons) library...16x16
	getLocalKVIrcDirectory (szP1, Pics, KVI_OUT_ICONS_IMAGE_LIBRARY_NAME);
	getGlobalKVIrcDirectory(szP2, Pics, KVI_OUT_ICONS_IMAGE_LIBRARY_NAME);
	KviImageLibrary lib_view(szP1.ptr(), szP2.ptr(), 16, 16);
	g_setSplashScreenProgress(9);
	if( !lib_view.libraryLoaded() )
		warningBox(_i18n_("The image library %s cannot be found."), KVI_OUT_ICONS_IMAGE_LIBRARY_NAME);
	for( int i = 0; i < KVI_OUT_NUM_IMAGES; i++ )
		g_pixViewOut[i] = new QPixmap(lib_view.getImage(i));
	g_setSplashScreenProgress(13);
}

// =============================================================================
//
// ### Save settings ###
//
// =============================================================================

void KviApp::saveKVIrcDirectory()
{
	// Here we save the local directory path
#ifdef COMPILE_WITH_KDE
	// In KDE we use the application config file
	if( !m_pConfigFile ) { // Not if user supplied a config file
		KConfig *cfg = config();
		if( cfg ) {
			if( cfg->getConfigState() == KConfig::ReadWrite ) {
				cfg->setGroup("LocalKVIrcDirectory");
				cfg->writeEntry("LocalKVIrcDirectory", m_szLocalKVIrcDir.ptr());
				cfg->sync();
			}
		}
		// Write both kdehome and standard rc
	}
#endif
	// In NON-KDE we use $HOME/.kvirc-VERSION.rc
	KviStr szF(QDir::homeDirPath());
	if( m_pConfigFile ) {
		if( *m_pConfigFile != '/' ) { // Assume $HOME/ if not absolute
			szF.append(KviStr::Format, "/%s", m_pConfigFile);
		} else
			szF = m_pConfigFile;
	} else
		szF.append(KviStr::Format, "/%s", KVI_HOME_CONFIG_FILE_NAME);

	QFile f(szF.ptr());
	if( f.open(IO_WriteOnly | IO_Truncate) ) {
		QTextStream t(&f);
		t << m_szLocalKVIrcDir.ptr();
		f.close();
		__debug_1arg("KVIrc local directory saved to %s", szF.ptr());
		removeOldConfig();
	} else {
		warningBox(_i18n_("Failed writing the KVIrc home directory path to %s.\n"
			"You will need to do it manually, otherwise KVIrc will restart with\n"
			"the setup!"), szF.ptr()
		);
	}
}

void KviApp::removeOldConfig(void)
{
	KviStr szF(QDir::homeDirPath());
	szF.append(KviStr::Format, "/%s", KVI_OLD_HOME_CONFIG_FILE_NAME);
	QFile cFile(szF.ptr());
	if( cFile.exists() ) {
		if( cFile.remove() )
			__debug_1arg("Removed old config file %s", szF.ptr());
	}
}

// =============================================================================
//
// ### Main frames handling ###
//
// =============================================================================

/**
 * Creates a new toplevel window.
 */
void KviApp::createNewFrame()
{
	KviFrame *pFrm = new KviFrame();
	pFrm->show();
}

void KviApp::insertNewFrame(KviFrame *frm)
{
	// KviFrame calls this in the constructor.
	m_pFrameList->append(frm);
}

void KviApp::removeFrame(KviFrame *frm)
{
	// KviFrame calls this in the destructor.
	m_pFrameList->removeRef(frm);
	// If the last window has been closed, die :)
	if( m_pFrameList->isEmpty() )
		quit();
}

// =============================================================================
//
// ### Application exits : destroy global stuff ###
//
// =============================================================================

void KviApp::destroyPixmaps()
{
	for( int i = 0; i < KVI_OUT_NUM_IMAGES; i++ ) {
		delete g_pixViewOut[i];
		g_pixViewOut[i] = 0;
	}

	if( g_pListBoxOpPixmap      ) { delete g_pListBoxOpPixmap;      g_pListBoxOpPixmap      = 0; }
	if( g_pListBoxVoicePixmap   ) { delete g_pListBoxVoicePixmap;   g_pListBoxVoicePixmap   = 0; }
	if( g_pListBoxHelpOpPixmap  ) { delete g_pListBoxHelpOpPixmap;  g_pListBoxHelpOpPixmap  = 0; }
	if( g_pListBoxUserOpPixmap  ) { delete g_pListBoxUserOpPixmap;  g_pListBoxUserOpPixmap  = 0; }
	if( g_pListBoxOwnerPixmap   ) { delete g_pListBoxOwnerPixmap;   g_pListBoxOwnerPixmap   = 0; }
}

// =============================================================================
//
// ### Global utilities ###
//
// =============================================================================

// =============== warningBox ===============
void KviApp::warningBox(const char *fmt, ...)
{
	// Realloc 256 bytes
	char *buf = (char *) kvi_malloc(256);
	va_list list;
	va_start(list, fmt);
	// Print... with max 256 chars
	int len = kvi_vsnprintf(buf, 256, fmt, list);
	va_end(list);
	// Check if we failed
	if( len < 0 ) {
		// Yes, failed
		int ilen = 256;
		do { // We failed, so retry with 256 more chars
			ilen += 256;
			// Realloc
			buf = (char *) kvi_realloc(buf, ilen);
			// Print...
			va_start(list, fmt);
			len = kvi_vsnprintf(buf, ilen, fmt, list);
			va_end(list);
		} while( len < 0 );
	}
	// Done...
	// Now m_len is the length of the written string not including the terminator...
	// Perfect! :)
	buf = (char *) kvi_realloc(buf, len+1);
	KviMessageBox::error(buf, _CHAR_2_QSTRING(_i18n_("KVIrc: WARNING")));
	kvi_free(buf);
}

void KviApp::openUserConfig()
{
	KviStr tmp;
	getLocalKVIrcDirectory(tmp, Config, "kvi.userconfig.conf");
	g_pUserConfig = new KviConfig(tmp.ptr());
}

#ifdef COMPILE_PLUGIN_SUPPORT
void KviApp::getPluginConfigFilePath(KviStr &buffer, const char *plugin_name)
{
	KviStr fName(KviStr::Format, "kvi.%s.conf", plugin_name);
	getLocalKVIrcDirectory(buffer, ConfigPlugins, fName.ptr());
}

bool KviApp::getReadOnlyPluginConfigFilePath(KviStr &buffer, const char *plugin_name)
{
	KviStr fName(KviStr::Format, "kvi.%s.conf", plugin_name);
	// Take a look in the local directory
	getLocalKVIrcDirectory(buffer, ConfigPlugins, fName.ptr());
	if( !kvi_fileExists(buffer.ptr()) ) {
		// No saved config yet... check for defaults
		KviStr tmp;
		getGlobalKVIrcDirectory(tmp, ConfigPlugins, fName.ptr());
		if( !kvi_fileExists(tmp.ptr()) ) {
			// No defaults... no such config file at all.
			return false;
		} else
			buffer = tmp; // Get the defaults this time
	} // else: file exists...
	return true;
}
#endif

void KviApp::raiseTopLevelWidget(QWidget *ptr)
{
	if( ptr->hasFocus() ) return;
	if( ptr->isActiveWindow() ) return;
	if( g_pOptions->m_bDialogShowHackForBrainDamagedWM )
		ptr->hide(); // Enlightenment works fine with raise, but does not like hide()
	processEvents();
	ptr->show();
	processEvents();
	ptr->raise();
	processEvents();
	ptr->setActiveWindow();
	processEvents();
	ptr->setFocus();
}

void KviApp::executeFileWithCommand(const char *commandline, const char *filePath, KviFrame *callerFrame, bool bRemoteExec)
{
	if( !callerFrame ) callerFrame = m_pFrameList->first();
	__range_valid(callerFrame);
	KviStr cmd = commandline;
	cmd.stripWhiteSpace();
	if( cmd.isEmpty() ) return;
	callerFrame->m_pUserParser->m_szLastExecFilename = filePath;
	callerFrame->m_pUserParser->m_bLastIsRemoteExec  = bRemoteExec;
	callerFrame->m_pUserParser->parseCommand(cmd.ptr(), callerFrame->m_pConsole);
}

// =============================================================================
//
// ### Option dialogs ###
//
// =============================================================================

void KviApp::slot_doUserOptionsDialog()
{
	if( m_pUserOptionsDialog )
		raiseTopLevelWidget(m_pUserOptionsDialog);
	else {
		m_pUserOptionsDialog = new KviUserDialog(0);
		connect(m_pUserOptionsDialog, SIGNAL(finished(bool)), this, SLOT(slot_userOptionsDialogFinished(bool)));
		m_pUserOptionsDialog->show();
	}
}

void KviApp::slot_userOptionsDialogFinished(bool bAccepted)
{
	__range_valid(m_pUserOptionsDialog);
	if( m_pUserOptionsDialog ) {
		delete m_pUserOptionsDialog;
		m_pUserOptionsDialog = 0;
	}
}

void KviApp::slot_doServerOptionsDialog()
{
	if( m_pServerOptionsDialog )
		raiseTopLevelWidget(m_pServerOptionsDialog);
	else {
		m_pServerOptionsDialog = new KviServerDialog(0);
		connect(m_pServerOptionsDialog, SIGNAL(finished(bool)), this, SLOT(slot_serverOptionsDialogFinished(bool)));
		m_pServerOptionsDialog->show();
	}
}

void KviApp::slot_serverOptionsDialogFinished(bool bAccepted)
{
	__range_valid(m_pServerOptionsDialog);
	if( m_pServerOptionsDialog ) {
		delete m_pServerOptionsDialog;
		m_pServerOptionsDialog = 0;
	}
}

void KviApp::showScriptCenter(KviFrame *par)
{
	if( g_pOptions->m_bScriptCenterShowAsDialog ) {
		if( m_pScriptCenter ) {
			if( m_pScriptCenter->parent() )
				m_pScriptCenter->reparent(0, QPoint(0, 0), true);
		} else {
			m_pScriptCenter = new KviScriptCenter(0);
			connect(m_pScriptCenter, SIGNAL(closed()), this, SLOT(slot_scriptDialogFinished()));
			m_pScriptCenter->show();
		}
	} else {
		if( !par ) return;
		if( m_pScriptCenter ) {
			if( m_pScriptCenter->parent() != par->m_pSplitter )
				m_pScriptCenter->reparent(par->m_pSplitter, QPoint(0, 0), true);
		} else {
			m_pScriptCenter = new KviScriptCenter(par->m_pSplitter);
			connect(m_pScriptCenter, SIGNAL(closed()), this, SLOT(slot_scriptDialogFinished()));
			m_pScriptCenter->show();
		}
	}
}

void KviApp::slot_scriptDialogFinished()
{
	m_pScriptCenter = 0;
}

void KviApp::requestHelpOn(const char *topic)
{
	KviFrame *frm = m_pFrameList->first();
	if( frm ) frm->requestHelpOn(topic);
}

void KviApp::slot_whatIsThisRequest()
{
	KviFrame *frm = m_pFrameList->first();
	if( frm ) frm->slot_whatIsThisRequest();
}

void KviApp::slot_generalOptionsDialog()
{
	if( m_pOptionsDialog )
		raiseTopLevelWidget(m_pOptionsDialog);
	else {
		m_pOptionsDialog = new KviOptionsDialog(0);
		connect(m_pOptionsDialog, SIGNAL(finished()), this, SLOT(slot_generalOptionsDialogFinished()));
		m_pOptionsDialog->show();
	}
}

void KviApp::slot_generalOptionsDialogFinished()
{
	if( m_pOptionsDialog ) {
		m_pOptionsDialog->delayedDestruct();
		m_pOptionsDialog = 0;
	}
}

void KviApp::updateAllFrameCaptions()
{
	for( KviFrame *f = m_pFrameList->first(); f; f = m_pFrameList->next() )
		f->updateCaption();
}

void KviApp::updateAllDirectoryBrowsers()
{
	for( KviFrame *frm = m_pFrameList->first(); frm; frm = m_pFrameList->next() ) {
		if( frm->m_pDirBrowser ) {
			if( frm->m_pDirBrowser->isVisible() )
				frm->m_pDirBrowser->applyOptions();
		}
	}
}

void KviApp::triggerGlobalRepaint()
{
	if( !m_bGlobalRepaintPending ) {
		m_bGlobalRepaintPending = true;
		QTimer::singleShot(50, this, SLOT(globalRepaint()));
	}
}

void KviApp::globalRepaint()
{
	m_bGlobalRepaintPending = false;
	for( KviFrame *f = m_pFrameList->first(); f; f = m_pFrameList->next() ) {
		f->applyOptions();
		f->applyToolbarOptions();
	}
}

void KviApp::slot_doRegUsersDialog()
{
	if( m_pRegUsersDialog )
		raiseTopLevelWidget(m_pRegUsersDialog);
	else {
		m_pRegUsersDialog = new KviRegisteredUsers(0);
		connect(m_pRegUsersDialog, SIGNAL(finished(bool)), this, SLOT(slot_regUsersDialogFinished(bool)));
		m_pRegUsersDialog->show();
	}
}

void KviApp::slot_regUsersDialogFinished(bool bAccepted)
{
	__range_valid(m_pRegUsersDialog);
	if( m_pRegUsersDialog ) {
		delete m_pRegUsersDialog;
		m_pRegUsersDialog = 0;
	}
	restartNotifyLists();
}

void KviApp::slot_doScriptWizard()
{
	if( m_pScriptWizard )
		raiseTopLevelWidget(m_pScriptWizard);
	else {
		m_pScriptWizard = new KviScriptWizard();
		connect(m_pScriptWizard, SIGNAL(finished(bool)), this, SLOT(slot_scriptWizardFinished(bool)));
		m_pScriptWizard->show();
	}
}

void KviApp::slot_scriptWizardFinished(bool bAccepted)
{
	__range_valid(m_pScriptWizard);
	KviSaveScriptStruct s;
	s.pWnd = m_pFrameList->first()->activeWindow();
	if( bAccepted ) m_pScriptWizard->fillSaveScriptStruct(&s);
	if( m_pScriptWizard ) {
		delete m_pScriptWizard;
		m_pScriptWizard = 0;
	}
	if( bAccepted ) {
		if( saveScript(&s) )
			s.pWnd->output(KVI_OUT_INTERNAL, _i18n_("Script successfully saved"));
		else
			s.pWnd->output(KVI_OUT_ERROR, _i18n_("Failed to save the script"));
		if( s.pExtraFilesList ) delete s.pExtraFilesList;
	}
}

void KviApp::restartNotifyLists()
{
	for( KviFrame *f = m_pFrameList->first(); f; f = m_pFrameList->next() ) {
		f->updateNotifyOrWatchList();
	}
}

void KviApp::slot_doTipOfTheDay()
{
	if( !m_pTipOfTheDayDialog ) {
		m_pTipOfTheDayDialog = new KviTipOfTheDay();
		m_pTipOfTheDayDialog->show();
		connect(m_pTipOfTheDayDialog, SIGNAL(closed()), this, SLOT(slot_TipOfTheDayClosed()));
	} else {
		m_pTipOfTheDayDialog->raise();
		m_pTipOfTheDayDialog->setFocus();
		m_pTipOfTheDayDialog->hide();
		m_pTipOfTheDayDialog->show();
	}
}

void KviApp::slot_TipOfTheDayClosed()
{
	__range_valid(m_pTipOfTheDayDialog);
	if( m_pTipOfTheDayDialog ) {
		delete m_pTipOfTheDayDialog;
		m_pTipOfTheDayDialog = 0;
	}
}

void KviApp::slot_doAboutDialog()
{
	KviFrame *frm = m_pFrameList->first();
	frm->m_pUserParser->parseCommand("if( !$pluginLoaded(about) ) plugin -q -u load about; aboutkvirc", frm->m_pConsole);
}

void KviApp::slot_doLoadScriptDialog()
{
	if( !m_pLoadScriptDialog ) {
		m_pLoadScriptDialog = new KviLoadScript(0);
		m_pLoadScriptDialog->show();
		connect(m_pLoadScriptDialog, SIGNAL(finished(bool)), this, SLOT(slot_loadScriptDialogFinished(bool)));
	} else {
		raiseTopLevelWidget(m_pLoadScriptDialog);
	}
}

void KviApp::slot_loadScriptDialogFinished(bool bAccepted)
{
	__range_valid(m_pLoadScriptDialog);
	if( bAccepted ) {
		KviFrame *f = m_pFrameList->first();
		KviStr scriptName;
		bool bRemoveAliases;
		bool bRemovePopups;
		bool bRemoveEvents;
		m_pLoadScriptDialog->getSelectedScriptFileName(scriptName, bRemoveAliases, bRemovePopups, bRemoveEvents);
		if( scriptName.hasData() ) {
			if( bRemoveAliases ) g_pAliasManager->clearAll();
			if( bRemovePopups ) {
				g_pChannelPopup->clearAll();
				g_pConsolePopup->clearAll();
				g_pUserlistPopup->clearAll();
				g_pNotifylistPopup->clearAll();
				g_pQueryPopup->clearAll();
				g_pDccChatPopup->clearAll();
			}
			if( bRemoveEvents ) {
				g_pEventManager->cleanup();
				g_pRawEventManager->cleanup();
			}

			KviStr tmp(KviStr::Format, "PARSE \"%s\"", scriptName.ptr());
			f->m_pUserParser->parseCommand(tmp.ptr(), f->m_pConsole);
			g_pOptions->save(); // Save options
		}
	}
	if( m_pLoadScriptDialog ) {
		delete m_pLoadScriptDialog;
		m_pLoadScriptDialog = 0;
	}
}

// /////////////////////////////////////////////////////

void KviApp::slot_saveSettings()
{
	g_pOptions->save();
}

// /////////////////////////////////////////////////////

void KviApp::adjustPopupMenusForWMDockWidget()
{
	for( KviFrame *f = m_pFrameList->first(); f; f = m_pFrameList->next() ) {
		f->m_pMainMenu->wmDockWidgetStateChanged(m_pWMDockWidget != 0);
	}
}

void KviApp::wmDockWidgetDestroyed()
{
	if( !m_pWMDockWidget ) return;

	m_pWMDockWidget = 0;
	showAllFrames();
	adjustPopupMenusForWMDockWidget();
}

void KviApp::slot_createWMDockWidget()
{
	if( !m_pWMDockWidget )
		m_pWMDockWidget = new KviWMDockWidget();
	adjustPopupMenusForWMDockWidget();
}

void KviApp::slot_destroyWMDockWidget()
{
	if( m_pWMDockWidget ) {
		delete m_pWMDockWidget;
		m_pWMDockWidget = 0;
	}
	adjustPopupMenusForWMDockWidget();
}

void KviApp::globalHighlight(bool bAltColor)
{
#ifdef COMPILE_NEED_KDE
	if( !m_pWMDockWidget ) return;
	if( !m_pWMDockWidget->framesVisible() ) {
		m_pWMDockWidget->highlight(bAltColor);
	}
#endif
}

void KviApp::hideAllFrames()
{
	for(KviFrame *f = m_pFrameList->first(); f; f = m_pFrameList->next() ) {
		f->hide();
	}
#ifdef COMPILE_NEED_KDE
	if( m_pWMDockWidget )
		m_pWMDockWidget->setFramesVisible(false);
#endif
}

void KviApp::showAllFrames()
{
	// This does not work with Enlightenment :/
	for( KviFrame *f = m_pFrameList->first(); f; f = m_pFrameList->next() ) {
		if( g_pOptions->m_bDialogShowHackForBrainDamagedWM ) {
			f->hide(); // Enlightenment works fine with raise, but does not like hide()
			processEvents();
		}
		f->showNormal();
		f->show();
		f->raise();
	}
#ifdef COMPILE_NEED_KDE
	if( m_pWMDockWidget )
		m_pWMDockWidget->setFramesVisible(true);
#endif
}

void KviApp::toggleFrame(int id)
{
	if( id < 0 ) return;
	__range_valid(((uint) id) < m_pFrameList->count());
	KviFrame *f = m_pFrameList->at(id);
	if( !f ) return;
	if( f->isHidden() ) {
		if( g_pOptions->m_bDialogShowHackForBrainDamagedWM ) {
			f->hide(); // Blah Enlightenment
			processEvents();
		}
		f->showNormal();
		f->show();
		f->raise();
	} else {
		f->hide();
	}
}

#include "m_kvi_app.moc"
