// =============================================================================
//
//      --- kvi_dcc_thread.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDccThread"

#define _KVI_DCC_CORE_CPP_

#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>

#include "kvi_application.h"
#include "kvi_dcc_event.h"
#include "kvi_dcc_thread.h"
#include "kvi_error.h"
#include "kvi_netutils.h"

#define KVI_DCC_IDLE_TIME_USECS 200

KviDccThread::KviDccThread()
{
	m_socket = 0;
}

KviDccThread::~KviDccThread()
{
	if( m_socket )
		close(m_socket);
}

bool KviDccThread::selectForRead(int sock)
{
	m_socket = sock;

	fd_set in_fd_set;
	struct timeval tv;

	FD_ZERO(&in_fd_set);
	FD_SET(m_socket, &in_fd_set);

	tv.tv_sec  = 0;
	tv.tv_usec = KVI_DCC_IDLE_TIME_USECS;

	int retval = false;
	if( !m_bStopRequested )
		retval = select(m_socket + 1, &in_fd_set, 0, 0, &tv);
	// TODO: check for select() errors here if retval == -1
	return (retval == 1);
}

int KviDccThread::connect(struct sockaddr *hostSockAddr, int addrlen)
{
	for( ;; ) {
		// EINTR loop... threads use signals to communicate...

		if( m_bStopRequested )
			return KVI_ERROR_DccThreadAborted;

		if(::connect(m_socket, hostSockAddr, addrlen) < 0 ) {
			if( errno != EINTR ) {
				switch( errno ) {
					case EINPROGRESS:  return KVI_ERROR_Success;               break;
					case EBADF:        return KVI_ERROR_BadFileDescriptor;     break;
					case EFAULT:       return KVI_ERROR_OutOfAddressSpace;     break;
					case ECONNREFUSED: return KVI_ERROR_ConnectionRefused;     break;
					case ENOTSOCK:     return KVI_ERROR_KernelNetworkingPanic; break;
					case ETIMEDOUT:    return KVI_ERROR_ConnectionTimedOut;    break;
					case ENETUNREACH:  return KVI_ERROR_NetworkUnreachable;    break;
					case EPIPE:        return KVI_ERROR_BrokenPipe;            break;
					// Unhandled error; pass errno to the strerror function
					default:           return -errno;                          break;
				}
			} // else: try again
		} else return KVI_ERROR_Success;
	}
}

void KviDccThread::abort(const char *errorMsg, QObject *target)
{
	KviDccEvent *e = new KviDccEvent(KVI_DCC_EVENT_ERROR, errorMsg);
	KviApplication::postEvent(target, e);
	stop();
}

int KviDccThread::waitForIncomingConnection(unsigned long *addr, unsigned short *port, KviStr *ret_address)
{
	// Now loop.
	// select() the socket for reading...
	// Never time out: timeout is nearly useless.
	// The user will just close the window if
	// there is no connection in several minutes...

	struct sockaddr_in connectedAddr;
	int newsock = -1;

	for( ;; ) {
		if( m_bStopRequested ) return 0;

		if( KviDccThread::selectForRead(m_socket) ) {
			// Can try to accept
			if( m_bStopRequested ) return 0;

			socklen_t iSize = sizeof(connectedAddr);
			newsock = accept(m_socket, (struct sockaddr *) &connectedAddr, &iSize);
			if( newsock == -1 )
				debug("warning: accept() returns -1... waiting for the next accept() call");
			else {
				// Connected
				*addr = connectedAddr.sin_addr.s_addr;
				*port = ntohs(connectedAddr.sin_port);
				kvi_binaryIpToString(connectedAddr.sin_addr, *ret_address);
				return newsock;
			}
		} else {
			// Give time to other threads...
			if( m_bStopRequested ) return 0;
			usleep(KVI_DCC_IDLE_TIME_USECS);
		}
	}
}

int KviDccThread::waitForOutgoingConnection()
{
	fd_set fset;
	struct timeval tv;

	// select() the socket for writing...
	for( ;; ) {
		FD_ZERO(&fset);
		FD_SET(m_socket, &fset);
		tv.tv_sec  = 0;
		tv.tv_usec = KVI_DCC_IDLE_TIME_USECS;

		if( m_bStopRequested )
			return KVI_ERROR_DccThreadAborted;
		int retval = select(m_socket + 1, 0, &fset, 0, &tv);
		if( m_bStopRequested )
			return KVI_ERROR_DccThreadAborted;

		if( retval == 1 ) {
			// Connected?
			// Check for errors
			int nSockErr;
			socklen_t iSize = sizeof(int);
			if( getsockopt(m_socket, SOL_SOCKET, SO_ERROR, (void *) &nSockErr, &iSize) == -1 )
				return KVI_ERROR_BadFileDescriptor;
			if( nSockErr != 0 ) {
				// Oops... error
				if( (nSockErr != EINTR) && (nSockErr != EINPROGRESS) ) {
					switch( nSockErr ) {
						case EBADF:        return KVI_ERROR_BadFileDescriptor;     break;
						case EFAULT:       return KVI_ERROR_OutOfAddressSpace;     break;
						case ECONNREFUSED: return KVI_ERROR_ConnectionRefused;     break;
						case ENOTSOCK:     return KVI_ERROR_KernelNetworkingPanic; break;
						case ETIMEDOUT:    return KVI_ERROR_ConnectionTimedOut;    break;
						case ENETUNREACH:  return KVI_ERROR_NetworkUnreachable;    break;
						case EPIPE:        return KVI_ERROR_BrokenPipe;            break;
						// Unhandled error; pass errno to the strerror function
						default:			return -nSockErr;                       break;
					}
				} // else: EINTR or EINPROGRESS
			} else return KVI_ERROR_Success;
		} else {
			// Retval != 1... continue loop
			// TODO: Test for select() errors here if retval == -1
			if( retval < 0 )
				debug("select() error");
			if( m_bStopRequested )
				return KVI_ERROR_DccThreadAborted;
			// Give time to other threads...
			usleep(KVI_DCC_IDLE_TIME_USECS);
		}
	}
}
