//
//   This file is part of the KVIrc irc client distribution
//   Copyright (C) 1999-2000 Andrea Parrella (anandrea@iname.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#ifdef QT_NO_ASCII_CAST
#undef QT_NO_ASCII_CAST
#endif

#define __KVIRC_PLUGIN__

#include "kvirc_plugin.h"
#include "libkviurl.h"
#include "kvi_process.h"
#include "kvi_app.h"
#include "kvi_mdi.h"
#include "icons.h"
#include <qlist.h>
#include <qfiledialog.h>

typedef struct UrlDlgList
{
	KviFrame *frm;
	UrlDialog *dlg;
	UrlToolBar *toolbar;
};

const char *g_pUrlListFilename = "/list.kviurl";
const char *g_pBanListFilename = "/list.kviban";

QList<KviUrl> *g_pList;
QList<UrlDlgList> *g_pUrlDlgList;
QList<KviStr> *g_pBanList;
ConfigDialog *g_pConfigDialog;

KviStr configpath;

void saveUrlList();
void loadUrlList();
void saveBanList();
void loadBanList();
UrlDlgList *findFrame(KviFrame *frame);
bool urllist(KviFrame *frame);
void url_plugin_help();

// ---------------------------- CLASS URLDIALOG ------------------------begin //

UrlDialog::UrlDialog(QList<KviUrl> *g_pList, KviFrame *lpFrm)
:KviWindow("!URL List",KVI_WND_TYPE_PLUGIN,lpFrm)
{
	m_pMenuBar = new QMenuBar(this);
	m_pUrlList = new QListView(this,"list");
	KviConfig cfg(configpath.ptr());

	QPopupMenu *pop;

	pop = new QPopupMenu(this);
	pop->insertItem(__tr("&Configure"),this,SLOT(config()));
	pop->insertItem(__tr("&Help"),this,SLOT(help()));
	pop->insertItem(__tr("Close &Window"),this,SLOT(close_slot()));
	m_pMenuBar->insertItem(__tr("&Plugin"),pop);

	pop = new QPopupMenu(this);
	pop->insertItem(__tr("&Load"),this,SLOT(loadList()));
	pop->insertItem(__tr("&Save"),this,SLOT(saveList()));
	pop->insertItem(__tr("&Clear"),this,SLOT(clear()));
	m_pMenuBar->insertItem(__tr("&List"),pop);

	m_pUrlList->setShowSortIndicator(true);
	m_pUrlList->addColumn(QString(__tr("URL")));
	m_pUrlList->addColumn(QString(__tr("Window")));
	m_pUrlList->addColumn(QString(__tr("Count")));
	m_pUrlList->addColumn(QString(__tr("Timestamp")));

	cfg.setGroup("colsWidth");
	m_pUrlList->setColumnWidth(0,cfg.readIntEntry("Url",170));
	m_pUrlList->setColumnWidth(1,cfg.readIntEntry("Window",130));
	m_pUrlList->setColumnWidth(2,cfg.readIntEntry("Count",70));
	m_pUrlList->setColumnWidth(3,cfg.readIntEntry("Timestamp",70));

	connect(m_pUrlList,SIGNAL(doubleClicked(QListViewItem *)),SLOT(dblclk_url(QListViewItem *)));
	connect(m_pUrlList,SIGNAL(rightButtonPressed(QListViewItem *, const QPoint &, int)),SLOT(popup(QListViewItem *, const QPoint &, int)));

	setFocusHandlerNoChildren(m_pUrlList);
	m_pUrlList->setFocus();
}

void UrlDialog::config()
{
	if (!g_pConfigDialog) g_pConfigDialog = new ConfigDialog();
}

void UrlDialog::help()
{
	m_pFrm->requestHelpOn("doc_plugin_url.kvihelp");
}

void UrlDialog::saveList()
{
	saveUrlList();
}

void UrlDialog::loadList()
{
	loadUrlList();
}

void UrlDialog::clear()
{
	g_pList->clear();
	for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (tmpitem->dlg) tmpitem->dlg->m_pUrlList->clear();
	}
}

void UrlDialog::saveProperties()
{
	KviWindowProperty p;
	p.rect = externalGeometry();
	p.isDocked = isAttached();
	p.splitWidth1 = 0;
	p.splitWidth2 = 0;
	p.timestamp = 0;
	p.imagesVisible = 0;
	KviWindow * w = m_pFrm->activeWindow();
	p.isMaximized = isAttached() && w ? w->isMaximized() : isMaximized();
	p.topSplitWidth1 = 0;
	p.topSplitWidth2 = 0;
	p.topSplitWidth3 = 0;
	g_pOptions->m_pWinPropertiesList->setProperty(caption(),&p);
}

void UrlDialog::close_slot()
{
	close();
}

void UrlDialog::remove()
{
	if (!m_pUrlList->currentItem()) {
		kvirc_plugin_warning_box(__tr("Select an URL"));
		return;
	}
	for(KviUrl *tmp=g_pList->first();tmp;tmp=g_pList->next())
	{
		if (tmp->url == KviStr(m_pUrlList->currentItem()->text(0))) {
			g_pList->find(tmp);
			g_pList->remove();
			m_pUrlList->takeItem(m_pUrlList->currentItem());
			return;
		}
		
	}
}

void UrlDialog::findtext()
{
	if (!m_pUrlList->currentItem()) {
		kvirc_plugin_warning_box(__tr("Select an URL"));
		return;
	}
	for(KviUrl *tmp=g_pList->first();tmp;tmp=g_pList->next())
	{
		if (tmp->url == KviStr(m_pUrlList->currentItem()->text(0))) {
			g_pList->find(tmp);
			KviStr ft="findtext %";
			ft.replaceAll('%',tmp->url.ptr());
			KviWindow *wnd = m_pFrm->findWindow(tmp->window.ptr());
			if (wnd) {
				if (kvirc_plugin_execute_command(wnd,ft.ptr())) {
					if (wnd->mdiParent()) m_pFrm->m_pMdi->setTopChild(wnd->mdiParent(),true);
				}
			} else kvirc_plugin_warning_box(__tr("Window not found"));
		}
		
	}
}

void UrlDialog::dblclk_url(QListViewItem *item)
{
	KviStr tmp = g_pOptions->m_szBrowserCommandline;
	tmp.stripWhiteSpace();
	if(tmp.hasData()){
		tmp.replaceAll('%',item->text(0));
		tmp.stripWhiteSpace();
		if(tmp.hasData()){
			KviProcess proc;
			if(!proc.run(tmp.ptr())){
				g_pApp->warningBox(__tr("Could not execute %s"),tmp.ptr());
			}
		}
	}
	return;
}

void UrlDialog::popup(QListViewItem *item, const QPoint &p, int col)
{
	if (col == 0) {
		m_szUrl = item->text(0);
		QPopupMenu p;
		p.insertItem(__tr("&Remove"),this,SLOT(remove()));
		p.insertItem(__tr("&Find Text"),this,SLOT(findtext()));
		p.insertSeparator();
		m_pListPopup = new QPopupMenu();
		int i=0;
		for(KviWindow *w=m_pFrm->m_pWinList->first();w;w=m_pFrm->m_pWinList->next()){
			if ((w->type() <= 3) || (w->type() == 8)) {	// values defined in kvi_define.h (console,channel,query,chat,uwindow)
				m_pListPopup->insertItem(w->caption(),i);
				m_pListPopup->connectItem(i,this,SLOT(sayToWin(int)));
				i++;
			}
		}
		p.insertItem(__tr("&Say to Window"),m_pListPopup);
		p.exec(QCursor::pos());
	}
}

void UrlDialog::sayToWin(int itemID)
{
	KviStr say="say %1 %2";
	say.replaceAll("%1",m_pListPopup->text(itemID).latin1());
	say.replaceAll("%2",m_szUrl.ptr());
	KviWindow *wnd = m_pFrm->findWindow(m_pListPopup->text(itemID).latin1());
	if (wnd) {
		if (!kvirc_plugin_execute_command(wnd,say.ptr())) {
			wnd->raise();
			wnd->setActiveWindow();
			wnd->setFocus();
		}
	} else kvirc_plugin_warning_box(__tr("Window not found"));
}

QPixmap *UrlDialog::myIconPtr()
{
	QPixmap *icon = new QPixmap(url_icon_xpm);
	return icon;
}

void UrlDialog::addUrl(QString url, QString window, QString count, QString timestamp)
{
	QListViewItem *UrlItem = new QListViewItem(m_pUrlList);

	UrlItem->setText(0, url);
	UrlItem->setText(1, window);
	UrlItem->setText(2, count);
	UrlItem->setText(3, timestamp);
}

void UrlDialog::resizeEvent(QResizeEvent *)
{
	int hght = m_pMenuBar->heightForWidth(width());
	m_pMenuBar->setGeometry(0,0,width(),hght);
	m_pUrlList->setGeometry(0,hght,width(),height() - hght);
}

UrlDialog::~UrlDialog()
{
	KviConfig cfg(configpath.ptr());
	cfg.setGroup("ConfigDialog");
	if (cfg.readBoolEntry("SaveColumnWidthOnClose",false)) {
		cfg.setGroup("ColsWidth");
		cfg.writeEntry("Url",m_pUrlList->columnWidth(0));
		cfg.writeEntry("Window",m_pUrlList->columnWidth(1));
		cfg.writeEntry("Count",m_pUrlList->columnWidth(2));
		cfg.writeEntry("Timestamp",m_pUrlList->columnWidth(3));
	}

	delete m_pUrlList;
	UrlDlgList *tmpitem = findFrame(m_pFrm);
	tmpitem->dlg = 0;
}

// ----------------------------- CLASS URLDIALOG -------------------------end //

// --------------------------- CLASS CONFIGDIALOG ----------------------begin //

ConfigDialog::ConfigDialog()
:QDialog()
{
	setCaption(__tr("URL Plugin Configuration"));

	QGridLayout *g = new QGridLayout(this,4,2,10,10);

	KviConfig *cfg = new KviConfig(configpath.ptr());
	cfg->setGroup("ConfigDialog");

	cb[0] = new QCheckBox(__tr("Save URL list on plugin unload"),this);
	cb[0]->setChecked(cfg->readBoolEntry("SaveUrlListOnUnload",false));
	g->addMultiCellWidget(cb[0],0,0,0,1);

	cb[1] = new QCheckBox(__tr("Save columns width on URL list close"),this);
	cb[1]->setChecked(cfg->readBoolEntry("SaveColumnWidthOnClose",false));
	g->addMultiCellWidget(cb[1],1,1,0,1);

	cb[2] = new QCheckBox(__tr("Load URL toolbar on startup"),this);
	cb[2]->setChecked(cfg->readBoolEntry("LoadUrlToolbarOnStartup",false));
	g->addMultiCellWidget(cb[2],2,2,0,1);

	bool tmp = cfg->readBoolEntry("BanEnabled",false);
	delete cfg;

	m_pBanFrame = new BanFrame(this,"banlist",tmp);
	g->addMultiCellWidget(m_pBanFrame,3,3,0,1);

	QPushButton *b;
	// configure buttons
        b = new QPushButton(__tr("Discard changes"),this,"discard");
	connect(b,SIGNAL(clicked()),this,SLOT(discardbtn()));
	g->addWidget(b,4,0);

	b = new QPushButton(__tr("Accept changes"),this,"accept");
	connect(b,SIGNAL(clicked()),this,SLOT(acceptbtn()));
	g->addWidget(b,4,1);

	show();
}

void ConfigDialog::discardbtn()
{
	delete this;
}

void ConfigDialog::acceptbtn()
{
	if (m_pBanFrame) m_pBanFrame->saveBans();

	KviConfig *cfg = new KviConfig(configpath.ptr());
	cfg->setGroup("ConfigDialog");
	cfg->writeEntry("SaveUrlListOnUnload",cb[0]->isChecked());
	cfg->writeEntry("SaveColumnWidthOnClose",cb[1]->isChecked());
	cfg->writeEntry("LoadUrlToolbarOnStartup",cb[2]->isChecked());
	delete cfg;

	delete this;
}

void ConfigDialog::closeEvent(QCloseEvent *)
{
	delete this;
}

ConfigDialog::~ConfigDialog()
{
	for(int i=0;i<cbnum;i++) delete cb[i];
	g_pConfigDialog = 0;
}

// --------------------------- CLASS CONFIGDIALOG ------------------------end //

// ---------------------------- CLASS BANFRAME ------------------------begin //

BanFrame::BanFrame(QWidget *parent, const char *name, bool banEnabled)
:QFrame(parent,name)
{
	setFrameStyle(QFrame::Panel | QFrame::Raised);

	QGridLayout *g = new QGridLayout(this,2,2,10,10);

	m_pEnable = new QCheckBox(__tr("Enable URL ban list"),this);
	connect(m_pEnable,SIGNAL(clicked()),this,SLOT(enableClicked()));
	m_pEnable->setChecked(banEnabled);
	g->addMultiCellWidget(m_pEnable,0,0,0,1);

	m_pBanList = new QListBox(this,"listbox");
	m_pBanList->setMinimumHeight(100);
	loadBanList();
	for(KviStr *tmp=g_pBanList->first();tmp;tmp=g_pBanList->next()) m_pBanList->insertItem(tmp->ptr()); // load ban list into listbox
	m_pBanList->setEnabled(m_pEnable->isChecked());
	g->addMultiCellWidget(m_pBanList,1,1,0,1);

	m_pAddBtn = new QPushButton(__tr("Add ban"),this,"add");
	connect(m_pAddBtn,SIGNAL(clicked()),this,SLOT(addBan()));
	m_pAddBtn->setEnabled(m_pEnable->isChecked());
	g->addWidget(m_pAddBtn,2,0);

	m_pRemoveBtn = new QPushButton(__tr("Remove selected"),this,"remove");
	connect(m_pRemoveBtn,SIGNAL(clicked()),this,SLOT(removeBan()));
	m_pRemoveBtn->setEnabled(m_pEnable->isChecked());
	g->addWidget(m_pRemoveBtn,2,1);
}

void BanFrame::enableClicked()
{
	m_pBanList->setEnabled(m_pEnable->isChecked());
	m_pAddBtn->setEnabled(m_pEnable->isChecked());
	m_pRemoveBtn->setEnabled(m_pEnable->isChecked());
}

void BanFrame::addBan()
{
	bool ok = false;
	KviStr *text = new KviStr(QInputDialog::getText(__tr("URL ban list"),__tr("Add"),QString::null,&ok,this));
	if (ok && !text->isEmpty()) {
		g_pBanList->append(text);
		m_pBanList->insertItem(text->ptr());
	}
}

void BanFrame::removeBan()
{
	uint i = 0;
	while ((!m_pBanList->isSelected(i)) && (i < m_pBanList->count())) i++;
	if (!m_pBanList->isSelected(i)) {
		kvirc_plugin_warning_box(__tr("Select a ban"));
		return;
	}
	KviStr item(m_pBanList->text(i).latin1());
	for(KviStr *tmp=g_pBanList->first();tmp;tmp=g_pBanList->next())
	{
		if (*tmp == item) g_pBanList->remove();
	}

	m_pBanList->removeItem(i);

}

void BanFrame::saveBans()
{
	if (m_pEnable->isChecked()) saveBanList();
	KviConfig *cfg = new KviConfig(configpath.ptr());
	cfg->setGroup("ConfigDialog");
	cfg->writeEntry("BanEnabled",m_pEnable->isChecked());
	delete cfg;
}

BanFrame::~BanFrame()
{

}

// ---------------------------- CLASS BANFRAME ---------------------------end //

// ---------------------------- CLASS URLTOOLBAR -----------------------begin //

UrlToolBar::UrlToolBar(KviFrame *parent,QMainWindow::ToolBarDock dock)
:QToolBar("UrlToolBar",parent,dock,false,"url_tool_bar")
{
	m_pParent = parent;
	m_pUrlBtn = new KviToolBarButton(url_toolbar_xpm,__tr("URL List Window"),
			__tr("Load URL list window"),this,SLOT(loadUrlListWindow()),this,"url_list_btn");
	m_pConfBtn = new KviToolBarButton(urlconf_toolbar_xpm,__tr("Configuration Dialog"),
			__tr("Load configuration dialog"),this,SLOT(loadConfigure()),this,"configure_btn");

	m_pUrlBtn->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
	m_pConfBtn->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
	setHorizontalStretchable(false);
	setVerticalStretchable(false);
	setBackgroundMode(QWidget::PaletteBackground);
}

void UrlToolBar::loadUrlListWindow()
{
	UrlDlgList *tmpitem = findFrame(m_pParent);
	if (tmpitem->toolbar) tmpitem->toolbar->m_pUrlBtn->setPixmap(url_toolbar_xpm);
	if (tmpitem->dlg) {
		tmpitem->dlg->close();
	} else {
		urllist(m_pParent);
	}
}

void UrlToolBar::loadConfigure()
{
	if (!g_pConfigDialog) g_pConfigDialog = new ConfigDialog();
}

UrlToolBar::~UrlToolBar()
{
	delete m_pUrlBtn;
	delete m_pConfBtn;
	UrlDlgList *tmpitem = findFrame(m_pParent);
	tmpitem->toolbar = 0;
}

// ---------------------------- CLASS URLTOOLBAR -------------------------end //

void saveUrlList()
{
	KviStr urllist;
	g_pApp->getLocalKVircDirectory(urllist,KviApp::ConfigPlugins);
	urllist += g_pUrlListFilename;
	QFile file;
	file.setName(urllist.ptr());
	file.open(IO_WriteOnly);

	QTextStream stream(&file);
	
	stream << g_pList->count() << endl;

	for(KviUrl *tmp=g_pList->first();tmp;tmp=g_pList->next())
	{
		stream << tmp->url.ptr() << endl;
		stream << tmp->window.ptr() << endl;
		stream << tmp->count << endl;
		stream << tmp->timestamp.ptr() << endl;
	}
	file.flush();
	file.close();
}

void loadUrlList()
{
	KviStr urllist;
	g_pApp->getLocalKVircDirectory(urllist,KviApp::ConfigPlugins);
	urllist += g_pUrlListFilename;
	QFile file;
	file.setName(urllist.ptr());
	if(!file.exists())
	{
		KviStr s;
		s.sprintf( "touch %s", urllist.ptr() );
		KviProcess proc;
		if(!proc.run(s.ptr())) g_pApp->warningBox(__tr("Could not execute %s"),s.ptr());
	}
	if (!file.open(IO_ReadOnly))return;
	QTextStream stream(&file);

	g_pList->clear();

	for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (tmpitem->dlg) tmpitem->dlg->m_pUrlList->clear();
	}
	KviUrl *tmp;
	int i=0;
	int num = stream.readLine().toInt();
	while ((!stream.eof()) && (i<num)){
		tmp = new KviUrl();
		tmp->url = stream.readLine();
		tmp->window = stream.readLine();
		tmp->count = stream.readLine().toInt();
		tmp->timestamp = stream.readLine();

		g_pList->append(tmp);

		for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
			if (tmpitem->dlg) {
				QString tmpCount;
				tmpCount.setNum(tmp->count);
				tmpitem->dlg->addUrl(QString(tmp->url.ptr()), QString(tmp->window.ptr()), tmpCount, QString(tmp->timestamp.ptr()));
			}
		}
		i++;
	}
	file.close();
}

void saveBanList()
{
	KviStr banlist;
	g_pApp->getLocalKVircDirectory(banlist,KviApp::ConfigPlugins);
	banlist += g_pBanListFilename;
	QFile file;
	file.setName(banlist.ptr());
	file.open(IO_WriteOnly);

	QTextStream stream(&file);
	
	stream << g_pBanList->count() << endl;
	for(KviStr *tmp=g_pBanList->first();tmp;tmp=g_pBanList->next())
	{
		stream << tmp->ptr() << endl;
	}
	file.flush();
	file.close();
}

void loadBanList()
{
	KviStr banlist;
	g_pApp->getLocalKVircDirectory(banlist,KviApp::ConfigPlugins);
	banlist += g_pBanListFilename;
	QFile file;
	file.setName(banlist.ptr());
	if(!file.exists())
	{
		KviStr s;
		s.sprintf( "touch %s", banlist.ptr() );
		KviProcess proc;
		if(!proc.run(s.ptr())) g_pApp->warningBox(__tr("Could not execute %s"),s.ptr());
	}
	if (!file.open(IO_ReadOnly))return;
	QTextStream stream(&file);

	g_pBanList->clear();

	int i=0;
	int num = stream.readLine().toInt();
	while ((!stream.eof()) && (i<num)){
		KviStr *tmp = new KviStr(stream.readLine());
		g_pBanList->append(tmp);
		i++;
	}
	file.close();
}

bool urllist_command(KviPluginCommandStruct *cmd)
{
	urllist(cmd->frame);
	return true;
}

UrlDlgList *findFrame(KviFrame *frame)
{
	UrlDlgList *tmpitem;
	for (tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (frame == tmpitem->frm) break;
	}
	if (!tmpitem) {
		UrlDlgList *udl = new UrlDlgList();
		udl->frm = frame;
		udl->dlg = 0;
		udl->toolbar = 0;
		g_pUrlDlgList->append(udl);
		tmpitem = g_pUrlDlgList->current();
	}
	return tmpitem;
}

bool urllist(KviFrame *frame)
{
	UrlDlgList *tmpitem = findFrame(frame);
	if (tmpitem->dlg) return false;

	tmpitem->dlg = new UrlDialog(g_pList,tmpitem->frm);
	tmpitem->frm->addWindow(tmpitem->dlg);

	for(KviUrl *tmp=g_pList->first();tmp;tmp=g_pList->next())
	{
		QString tmpCount;
		tmpCount.setNum(tmp->count);
		tmpitem->dlg->addUrl(QString(tmp->url.ptr()), QString(tmp->window.ptr()), tmpCount, QString(tmp->timestamp.ptr()));
	}
	return true;
}

bool urllistconfig(KviPluginCommandStruct *cmd)
{
	if (!g_pConfigDialog) g_pConfigDialog = new ConfigDialog();
	return true;
}

int check_url(KviPluginCommandStruct *cmd)		// return 0 if no occurence of the url were found
{
	int tmp = 0;

	for(KviStr *tmpitem=g_pBanList->first();tmpitem;tmpitem=g_pBanList->next())
	{
		if (cmd->params->at(1)->contains(tmpitem->ptr())) tmp++;
	}
	if (tmp > 0) return tmp;

	for(KviUrl *u = g_pList->first();u;u=g_pList->next())
	{
		if (u->url == cmd->params->at(1)->ptr()) {
			u->window = cmd->window->caption();
			u->count++;
			tmp++;
		}
	}

	for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (tmpitem->dlg) {
			QListViewItemIterator lvi(tmpitem->dlg->m_pUrlList);
			for(;lvi.current();++lvi)
			{
				if (lvi.current()->text(0) == cmd->params->at(1)->ptr()) {
					int tmpCount = lvi.current()->text(2).toInt();
					tmpCount++;
					QString tmpStr;
					tmpStr.setNum(tmpCount);
					lvi.current()->setText(2,tmpStr);
					lvi.current()->setText(1,cmd->window->caption());
				}
			}
		}
	}	
	return tmp;
}

bool urllist_plugin_hook_onUrl(KviPluginCommandStruct *cmd)
{
	if (check_url(cmd) == 0) {
	
		KviUrl *tmp = new KviUrl;
		KviStr tmpTimestamp;
		QDate d = QDate::currentDate();
		KviStr date(KviStr::Format,"%d-%d%d-%d%d",d.year(),d.month() / 10,d.month() % 10,d.day() / 10,d.day() % 10);
		tmpTimestamp = "["+date+"]"+" [";
		tmpTimestamp += QTime::currentTime().toString()+"]";

		tmp->url = kvirc_plugin_param(cmd,1);
		tmp->window = cmd->window->caption();
		tmp->count = 1;
		tmp->timestamp = tmpTimestamp;

		g_pList->append(tmp);

		for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
			if (tmpitem->dlg) {
				QString tmpCount;
				tmpCount.setNum(tmp->count);
				tmpitem->dlg->addUrl(QString(tmp->url.ptr()), QString(tmp->window.ptr()), tmpCount, QString(tmp->timestamp.ptr()));
				tmpitem->dlg->highlight();
			} else {
				if (tmpitem->toolbar) tmpitem->toolbar->m_pUrlBtn->setPixmap(urlhigh_toolbar_xpm);
			}
		}
	}
	return true;
}

bool urllist_plugin_hook_onStartup(KviPluginCommandStruct *cmd)
{
	KviConfig cfg(configpath.ptr());
	cfg.setGroup("ConfigDialog");
	if (cfg.readBoolEntry("LoadUrlToolbarOnStartup",false)) {
		UrlDlgList *tmpitem = findFrame(cmd->frame);
		tmpitem->toolbar = new UrlToolBar(cmd->frame,QMainWindow::Top);
	}
	return false;
}

bool url_plugin_init(KviPluginCommandStruct *cmd)
{

	g_pList = new QList<KviUrl>;
	g_pList->setAutoDelete(true);

	g_pUrlDlgList = new QList<UrlDlgList>;
	g_pUrlDlgList->setAutoDelete(true);

	g_pBanList = new QList<KviStr>;
	g_pBanList->setAutoDelete(true);

	kvirc_plugin_register_command(cmd->handle,"urllist",urllist_command);
	kvirc_plugin_register_command(cmd->handle,"urllistconfig",urllistconfig);
	kvirc_plugin_add_hook(cmd->handle,KviEvent_OnUrl,urllist_plugin_hook_onUrl);
	kvirc_plugin_add_hook(cmd->handle,KviEvent_OnStartup,urllist_plugin_hook_onStartup);

	g_pApp->getLocalKVircDirectory(configpath,KviApp::ConfigPlugins);
	configpath += "/kvi.url.conf";

	loadUrlList();
	loadBanList();

	return true;
}

void url_plugin_cleanup()
{
	KviConfig cfg(configpath.ptr());
	cfg.setGroup("ConfigDialog");
	if (cfg.readBoolEntry("SaveUrlListOnUnload",false) == true) saveUrlList();
	for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (tmpitem->toolbar) delete tmpitem->toolbar;
		if (tmpitem->dlg) tmpitem->dlg->close();
        }

	delete g_pList;
	delete g_pBanList;

	kvirc_plugin_unregister_meta_object("UrlDialog");
	kvirc_plugin_unregister_meta_object("BanFrame");
	kvirc_plugin_unregister_meta_object("ConfigDialog");
	kvirc_plugin_unregister_meta_object("UrlToolBar");
}

void url_plugin_config()
{
	if (!g_pConfigDialog) g_pConfigDialog = new ConfigDialog();
}

/*
	@document: doc_plugin_url.kvihelp
	@title: The Url list plugin
		This plugin keeps track of all urls shown in kvirc windows.
		<H3>Exported commands:</H3>
		<B>/urllist</B> : this command opens a window containing the urls' list.
		In the list there is other information:<BR>
		<U>Window</U> : window where the url has been shown last<BR>
		<U>Count</U> : number of urls shown<BR>
		<U>Timestamp</U> : date/time when the url has been shown first<BR>
		Clicking right on the url column of the list a menu will popup, through it
		you can remove the selected item, find the url in the window it appeared last, and
		say it to: <I>@Console, Channels, Querys, DCC Chats</I> and <I>User windows</I>.<BR>
		The list is saved to file when you click on the menu item or when you unload the plugin
		on condition that you have checked the relative checkbox in configuration dialog.<BR>
		You can also open the url in your web browser double clicking on it in the url list window.<BR><BR>

		<B>/urllistconfig</B> : this command opens a configuration window where it is possible
		to setup plugin's parameters. You can also open this window by
		using popup menu in the url list window or by clicking on the "configure plugin" button
		in plugins options.<BR><BR>
		<H3>Configure dialog options:</H3>
		There is an option which loads a small <U>toolbar</U> when the kvirc onStartup event is fired.<BR>
		In the toolbar you'll find 2 icons:<BR>
		<UL><LI>the first toggles the url list window and when the onUrl event is fired the button is highlighted;</LI>
		<LI>the second opens the configure dialog.</LI></UL>
		There is also a ban list widget, which allows to have a list of words that plugin mustn't catch.<BR><BR>
		<I>E.g.<BR>
		<blockquote>if the word "ftp" is inserted in the ban list and if in a window there is an output like "ftp.kvirc.net",
		the url will not be catched.</blockquote></I>
		<HR>
		Mail me if you have any suggestion or you want to notice a bug.<BR>
		<B>Andrea 'YaP' Parrella</B> &lt;anandrea@iname.com&gt;<BR><BR>
		Thanks to:<BR>
		<B>Szymon Stefanek</B> and <B>Till Bush</B> for their help.<BR>
		<B>Ted Podgurski</B> for toolbar's icons.
*/

void url_plugin_help()
{
	kvirc_plugin_show_help_page("doc_plugin_url.kvihelp");
}

KviPlugin kvirc_plugin =
{
	"url" ,
	"URL list plugin" ,
	"0.1" ,
	"Andrea Parrella <anandrea@iname.com>" ,
	"This plugin implements a list of URLs\n"
	"shown in KVIrc windows.\n"
	"Commands it exports are:\n"
	"  /URLLIST\n"
	"  /URLLISTCONFIG",
	url_plugin_init ,
	url_plugin_cleanup ,
	url_plugin_config ,
	url_plugin_help
};


#include "m_libkviurl.moc"
