/**vim: ts=4, wrap, tw=80
  *
  *		--- libkvibiffsocket.cpp ---
  *
  * Authors:
  *		Szymon Stefanek <stefanek@tin.it>
  * 	Krzysztof Godlewski <kristoff@poczta.wprost.pl>
  * 
  * This program is FREE software. You can redistribute it and/or
  * modify it under the terms of the GNU General Public License
  * as published by the Free Software Foundation; either version 2
  * of the License, or (at your opinion) any later version.
  *
  * This program is distributed in the HOPE that it will be USEFUL,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.

  * You should have received a copy of the GNU General Public License
  * along with this program. If not, write to the Free Software Foundation,
  * Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  **/
 

#define __KVIRC_PLUGIN__

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviBiffSocket"
#include "kvi_debug.h"

#include "kvirc_plugin.h"

#include "libkvibiff.h"
#include "libkvibiffsystray.h"

#include "libkvibiffsocket.h"

#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_netutils.h"

#include <errno.h>

#include <fcntl.h>

extern KviBiff * g_pBiff;

KviBiffSocket::KviBiffSocket()
: QObject(0)
{
	m_fd          = -1;
	m_pNotifier   = 0;
	m_szHostIp    = "127.0.0.1";
	m_uPort       = 110;
	m_pDns        = 0;
	m_lastCommand = "[!- NULL -!]";
}

KviBiffSocket::~KviBiffSocket()
{
	if(m_pDns)delete m_pDns;
	if(m_pNotifier)delete m_pNotifier;
	if(m_fd != -1)close(m_fd);
}

bool KviBiffSocket::run(KviBiffMailbox * box)
{
	if(m_pDns || (m_fd != -1))
		return false; //already running

	m_pMailbox = box; // DO NOT USE THIS POINTER!...it may be deleted from outside
	m_uPort  = box->port();
	m_szHostname = box->hostname();
	m_szUsername = box->username();
	m_szPassword = box->password();

	m_pDns = new KviAsyncDns();
	connect(m_pDns,SIGNAL(dnsFinished(KviDnsStruct *)),this,
		SLOT(dnsDone(KviDnsStruct *)));
	emit resolving();
	m_pDns->resolve(box->hostname());
	
	return true;
}

void KviBiffSocket::dnsDone(KviDnsStruct *dns)
{
	__debug("Dns done");
	if(dns->iError != KVI_ERROR_Success)
	{
		__debug("Dns Failed");
		KviStr tmp(KviStr::Format,__tr("Dns failure: %s"),
			kvi_getErrorString(dns->iError));
		delete m_pDns;
		m_pDns = 0;
		emit error(tmp.ptr());
		return;
	}
	// hostname resolved
	m_szHostIp = dns->szIp;
	KviStr s;
	s.sprintf("Host resolved to %s",dns->szIp.ptr());
	g_pBiff->systrayMsg(s.ptr());
	delete m_pDns;
	m_pDns = 0;

	struct sockaddr_in serv_sin;
	serv_sin.sin_family = AF_INET;
	serv_sin.sin_port   = htons(m_uPort);

	if(!kvi_stringIpToBinaryIp(m_szHostIp.ptr(),&(serv_sin.sin_addr)))
	{
		emit error(__tr("Internal error"));
		return;
	}

	if((m_fd = socket(PF_INET,SOCK_STREAM,IPPROTO_TCP)) < 0)
	{
		emit error(__tr("Socket creation failure"));
		return;
	}

	if(fcntl(m_fd,F_SETFL,O_NONBLOCK) < 0)
	{
		close(m_fd);
		m_fd = -1;
		emit error(__tr("Internal error : fcntl()"));
		return;
	}

	if(::connect(m_fd,(struct sockaddr *)&serv_sin,sizeof(serv_sin)) < 0)
	{
		if(errno != EINPROGRESS)
		{
			close(m_fd);
			m_fd = -1;
			/* FIXME : Maybe a more verbose error code ? */
			emit error(__tr("Connect failed"));
			return;
		}
	}

	m_pNotifier = new QSocketNotifier(m_fd,QSocketNotifier::Write);
	QObject::connect(m_pNotifier,SIGNAL(activated(int)), this,
		SLOT(writeNotifierFired(int)));
	m_pNotifier->setEnabled(true);

	KviStr m;
	m.sprintf( "Connecting to %s", m_pMailbox->hostname());
	g_pBiff->systrayMsg(m.ptr());
	// FIXME : add a timer with a timeout here ?
	// FIXME : emit a message saying "connecting..."
}

void KviBiffSocket::writeNotifierFired(int)
{
	// step3 : connected...kill the write notifier
	//         setup the read one and go on
	//         with the pop3 protocol....
	//         read data in readNotifierFired() and write
	//         to the socket accordingly

	delete m_pNotifier;
	m_pNotifier = 0;
	int sockError;
	_this_should_be_socklen_t iSize=sizeof(int);
	if(getsockopt(m_fd,SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize)<0)
		sockError = -1;
	if(sockError != 0)
	{
		close(m_fd);
		m_fd = -1;
		emit error(strerror(sockError));
		return;
	}
//		if(sockError > 0)setErrorFromSystemError(sockError);
//		else m_error = KVI_ERROR_UnknownError; //Error 0 ?
//		m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);

	m_pNotifier = new QSocketNotifier(m_fd,QSocketNotifier::Read);
// FIXME:
//	QObject::connect(m_pNotifier,SIGNAL(activated(int)),this,
//		SLOT(readNotifierFired(int)));
//	m_pNotifier->setEnabled(true);

	emit connected();
	/* FIXME : Probably will need to keep track of the state here */
	
	char buffer[1024];
	int readLength = 0;

// I'm lazy :) A macro to make my life easier
// Do some reading / writing and some checks
// It's argument MUST be KviStr, the second one is the output buffer.
// There is some delay between writing to the socket and reading from it.. it
// is just right for connections to the local server, so in the Net it can be
// smaller (lagzz). On the other hand, when the communication is too fast the
// plugin will segfault ! I don't know how to prevent this... I can only hope
// that when readNotifierFired() starts to work fine everything will be ok
//
// The "for" loop is here to make this whole thing not block the GUI
#define BIFF_SEND_CMD(x, y) \
write(m_fd, x.ptr(), x.len()); \
for( int i = 0; i < 20; i ++ ) \
{ \
	usleep( 15000 ); \
	qApp->processOneEvent(); \
} \
memset((y), '\0', sizeof((y))); \
readLength = read(m_fd, (y), sizeof((y))); \
if(readLength <= 0) \
{ \
	__debug( "readLength < 0" ); \
	close(m_fd); \
	m_fd = -1; \
	emit error( __tr("Disconnected")); \
	return; \
} \
if( y[0] == '-' ) \
{ \
	__debug( "-ERRor" ); \
	KviStr s; \
	KviStr _s = KviStr((y)).cutToFirst(' '); \
	int _idx = _s.findFirstIdx('\r'); \
	if( _idx != -1 ) _s.cut(_idx, 1); \
	s.sprintf( __tr("Error in command %s, server replied: %s" ), \
		m_lastCommand.ptr(), _s.ptr() ); \
	write(m_fd,"QUIT\r\n", 6); \
	emit error(s.ptr()); \
	return; \
}


//
// USER
	KviStr tmp(KviStr::Format,"USER %s\r\n",m_szUsername.ptr());
	m_lastCommand = "USER";
	BIFF_SEND_CMD(tmp, buffer)
//	__debug( "buffer: %s", buffer );

//
// PASS
	tmp.sprintf("PASS %s\r\n",m_szPassword.ptr());
	m_lastCommand = "PASS";
	BIFF_SEND_CMD(tmp, buffer)
//	__debug( "buffer: %s", buffer );

	emit loggedIn();
	
//
// STAT
	tmp.sprintf( "STAT\r\n" );
	m_lastCommand = "STAT";
	BIFF_SEND_CMD(tmp, buffer)
//	__debug( "buffer: %s", buffer );
	KviStr numMsg(buffer);
	tmp = numMsg.middle( strlen("+OK "), 3 );
	uint num_msg = tmp.cutFromFirst(' ').toUInt();

//
// UIDL
	tmp.sprintf( "UIDL\r\n");
	BIFF_SEND_CMD(tmp, buffer );
	tmp = buffer;
	KviStr uids(tmp.cutToFirst('1', false));

//#warning "TODO: Need a good way to find new messages number"
// TODO: Find a decent way to know new messages number. It cannot be done simply
// by sth like newMsgCount = num_msg - m_pMailbox->messageCount(), because
// usualy after you check your account with biff you download your mail, and it
// is deleted from server, so m_pMailbox->messageList()->clear() would have to
// be called at every mail check. But this doesn't solve the problem because not
// all mail has to be retrieved from the server, and for example there can be:
// * on first check 10 msgs
// * on second one 11 msgs - one new ? This might be not true, because user
// could've gotten 11 mails between two checks, retrieving the 10 mentioned
// earlier !
// I thought about keeping 2 QList<KviBiffMessage> objects. One would be
// currentMail list, and the other one lastMailList. Then we could compare the
// messages uid's and I think this would work. But maybe there is another, less
// complicated way ? Maybe sth with the pop3 protocol (message status?). Any
// ideas ?

//	uint newMessagesCount = num_msg - m_pMailbox->messageCount();
	for( uint i = 0; i < num_msg; i++ )
	{
// Find the message's unique id	
		uids.getLine(tmp);					// get the first line of the uids
		KviStr uid = tmp.cutToFirst(' ');	// kill the message number
		int idx = uid.findFirstIdx('\r');
		if( idx != -1 )
			uid.cut(idx, 1);
//		__debug ("uid is: %s", uid.ptr());
		
		tmp.sprintf("TOP %u 0\r\n", i + 1);
		BIFF_SEND_CMD(tmp, buffer)
//		__debug("TOP: %s", buffer );
		tmp = buffer;
		KviStr from(tmp.middle(tmp.findFirstIdx("From: "), 100));
		from.getLine(tmp);
		from = tmp.middle( strlen("From: "), 100);
		from.findFirstIdx('\r');
		if( idx != -1 )
			from.cut(idx,1);
		tmp = buffer;
		KviStr subject( tmp.middle(tmp.findFirstIdx("Subject: "), 100));
		subject.getLine(tmp);
		subject = tmp.middle( strlen( "Subject: "), 100 );
		idx = subject.findFirstIdx('\r');
		if( idx != -1 )
			subject.cut(idx,1);

		if( ! m_pMailbox->findMessageByUid(uid.ptr()) )
		{	// The message was deleted from the account
			KviBiffMessage *m = new KviBiffMessage( from.ptr(), subject.ptr(),
			uid.ptr());
			m_pMailbox->messageList()->append(m);
		}
	}

//	m_pMailbox->setNewMessagesCount( newMessagesCount );

	tmp.sprintf( "QUIT\r\n" );
	m_lastCommand = "QUIT";
	BIFF_SEND_CMD(tmp, buffer)
//	__debug( "buffer: %s", buffer );

	emit jobDone();
}

void KviBiffSocket::readNotifierFired(int fd)
{
//#warning "Still can't get the readNotifier to work !"
//
// Sometimes all I get is "Server ready message", sometimes the output reaches
// LIST command...strange.. I've only tested in on my machine (popd), don't know
// how it will act in the Net...
// Maybe the commands are being sent to fast and the server doesn't reply
// because of the flood ? Can it be possible ?
//
//
// Ok, so I've added usleep() and it helped...the problem is that read notifier
// is getting fired not after every command sent to server, but after some time
// This is strange, but I have no idea what the problem is, since I don't know
// too much about sockets and stuff :)) Guess I'll just have to keep on
// experimenting with this..
//
// Right.... For now I will do it without the read notifier...maybe Pragma will
// be able to fix this.... I'll do it the simplest way it can be done :)))
//


	// all the communication job is done here
	debug("[biff] : read notifier fired !");
	char buffer[1025];
	int readLength = read(m_fd,buffer,sizeof(buffer));
	if(readLength <= 0)
	{
		close(m_fd);
		m_fd = -1;
		emit error(__tr("Disconnected"));
		return;
	}
	
	buffer[readLength - 1] = '\0';	// readLength - 1 is a '\n' char - kill it
	debug("[biff] : received data (%s)",buffer);
	
	/* FIXME : Need to keep track of the state and chat with the server
	   conseguently */
	// debug( "buffer[0]: %s", buffer[0] );
	if( buffer[0] == '-' )		// "-ERROR" from server
	{
		KviStr s;
		s.sprintf( __tr("Error in command %s (server replied: %s" ),
			m_lastCommand.ptr(), buffer );
		emit error(s.ptr());
		return;
	}
	
	emit jobDone(); //FIXME : Just for testing....DO NOT REFERENCE this AFTER
					// THIS SIGNAL HAS BEEN EMITTED!
}

#include "libkvibiffsocket.moc"
