/**vim: ts=4, wrap, tw=80
  *
  *		--- libkvibiff.h ---
  *
  * Authors:
  * 	Krzysztof Godlewski <kristoff@poczta.wprost.pl>
  * 	Szymon Stefanek <stefanek@tin.it>
  * 
  * This program is FREE software. You can redistribute it and/or
  * modify it under the terms of the GNU General Public License
  * as published by the Free Software Foundation; either version 2
  * of the License, or (at your opinion) any later version.
  *
  * This program is distributed in the HOPE that it will be USEFUL,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.

  * You should have received a copy of the GNU General Public License
  * along with this program. If not, write to the Free Software Foundation,
  * Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  **/

#ifndef LIBKVIBIFF_H
#define LIBKVIBIFF_H

#include <qfile.h>
#include <qstring.h>
#include <qlist.h>
#include <qtimer.h>

#include <stdlib.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include <netdb.h>

#include "kvi_string.h"

#define LIBKVIBIFF_VERSION	"0.1.0"

class KviWindow;

class KviBiffMessage
{
public:
	KviBiffMessage( const char *_from, const char *_subject, const char * _uid )
		{ m_from = _from; m_subject = _subject; m_uid = _uid; }
	~KviBiffMessage() {}

	const char * const from() const			{ return m_from.ptr(); }
	const char * const subject() const		{ return m_subject.ptr(); }
	const char * const uid() const			{ return m_uid.ptr(); }

private:
	KviStr m_from;
	KviStr m_subject;
	KviStr m_uid;		// unique message number - given by the server
};


class KviBiffMailbox
{
public:
	KviBiffMailbox(const char *host, const char *user, const char *pass,
		unsigned int port = 110, bool auto_check = true );
	~KviBiffMailbox();

	char * hostname()   const				{ return m_hostname.ptr(); }
	char * username()   const				{ return m_username.ptr(); }
	char * password()   const				{ return m_password.ptr(); }
	unsigned int port() const				{ return m_port; }

//	void setNewMessagesCount( uint num )	{ m_newMessagesCount = num; }
//	const uint newMessagesCount() const		{ return m_newMessagesCount; }

	unsigned int messageCount()				{ return m_pMessageList->count(); }
	QList<KviBiffMessage> * messageList()	{ return m_pMessageList; }

	KviBiffMessage * const findMessageByUid(const char *msg) const;

	const bool autoCheck() const			{ return m_autoCheck; }

private:
	KviStr m_hostname;
	KviStr m_username;
	KviStr m_password;
	int    m_port;

//	uint m_newMessagesCount;
	bool m_autoCheck;

	QList<KviBiffMessage> * m_pMessageList; // list of messages in the mailbox
};

class KviBiffWidget;
class KviBiffSocket;

typedef struct _KviBiffConfig
{
	bool				autoCheckAll;
	unsigned int		autoCheckInterval;

	bool				systrayOnStartup;

	bool				beVerbose;
	unsigned int		timeout;
} KviBiffConfig;

class KviBiff : public QObject
{
	Q_OBJECT

public:
	KviBiff();
	~KviBiff();
	
protected:
	QList<KviBiffMailbox> * m_pMailboxList;     // the main list of mailboxes
	QPixmap				  * m_pCurrentIcon;
	QPixmap               * m_pNormalIcon;
	QPixmap				  * m_pCheckingIcon;
	QPixmap				  * m_pGotNewMailIcon;
	QList<KviBiffWidget>  * m_pBiffWidgetList; // the list of the docked widgets
	KviBiffSocket         * m_pSocket;
	
	QTimer                * m_pTimer;
	KviBiffMailbox        * m_pCurMailbox;
	KviWindow             * m_window;

	KviBiffConfig           m_config;

	bool 					m_checkingAll;
	
public:
	QList<KviBiffMailbox> * const mailboxList() const { return m_pMailboxList; }
	const KviBiffConfig & configStruct() const { return m_config; }

	const KviBiffMailbox * const findMailbox( const char *user,
		const char *hostname ) const;

	QPixmap * currentWidgetIcon();
	void registerSysTrayWidget(KviBiffWidget * w);
	void unregisterSysTrayWidget(KviBiffWidget * w);
	void start();
	void stop();

	bool isRunning(){ return (m_pTimer != 0); }; // the check timer is running ?
//	bool isChecking(){ return (m_pSocket != 0); }; // are we checking a mailbox now ?
// show message on systrays for some time:
	void systrayMsg(const char * t= 0, uint time = 70000, bool permt = false);

public slots:
	void slotConfig();
	void slotConfigureMailboxes();
	void slotCheckMailIn(int);		// activated by signal from popup menu
	void slotCheckAll();
	void stopCheckingMail();

private:
	void readConfig();
	void saveConfig();

	const KviStr encryptString( const KviStr & ) const;
	const KviStr decryptString( const KviStr & ) const;

	bool checkMailbox( KviBiffMailbox * );
	
protected slots:
	void checkAllMailboxes();
	void socketResolving();
	void socketConnected();
	void socketLoggedIn();
	void socketError(const char * err);
	void socketJobDone();

	void slotApplyOptions();
	void slotKillConfigDialog();

signals:
	void connectedToServer();
};

#endif	// LIBKVIBIFF_H
