/**vim: ts=4, wrap, tw=80
  *
  *		--- libkvibiff.cpp ---
  *
  * Authors:
  * 	Krzysztof Godlewski	<kristoff@poczta.wprost.pl>
  * 	Szymon Stefanek		<stefanek@tin.it>
  * 
  * This program is FREE software. You can redistribute it and/or
  * modify it under the terms of the GNU General Public License
  * as published by the Free Software Foundation; either version 2
  * of the License, or (at your opinion) any later version.
  *
  * This program is distributed in the HOPE that it will be USEFUL,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.

  * You should have received a copy of the GNU General Public License
  * along with this program. If not, write to the Free Software Foundation,
  * Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  **/
 
//#define _XOPEN_SOURCE		// that's what "man crypt" says
//#include <unistd.h>
//
//#include <crypt.h>

#define __KVIRC_PLUGIN__

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviBiff"
#include "kvi_debug.h"

#include "libkvibiff.h"

#include "kvirc_plugin.h"

#include <qtextstream.h>
#include <qdialog.h>
#include <qlineedit.h>

#include <errno.h>

#include "libkvibiffsystray.h"
#include "libkvibiffsocket.h"
#include "libkvibiffconfig.h"

#include "kvi_config.h"
#include "kvi_locale.h"

KviBiff * g_pBiff = 0;					// THE biff (the controller class)
KviBiffConfigDlg * g_pConfigDlg = 0;	// config dialog

#include "libkvibifficons.h"

//#warning "Most recent TODO list: "
//#warning "* Interval auto-check for all mailboxes (the ones user wants to be
//auto-checked)"
//#warning "* Config dialog - external mail client etc."


KviBiffMailbox::KviBiffMailbox( const char *host, const char *user,
	const char *pass, unsigned int port, bool auto_check )
{
	m_hostname = host;
	m_port = port;
	m_username = user;
	m_password = pass;
	m_autoCheck = auto_check;
	m_pMessageList = new QList<KviBiffMessage>;
	m_pMessageList->setAutoDelete(true);
//	m_newMessagesCount = 0;
}

// - - -

KviBiffMailbox::~KviBiffMailbox()
{
	delete m_pMessageList;
}

// - - -

KviBiffMessage * const KviBiffMailbox::findMessageByUid( const char *uid ) const
{
	for( KviBiffMessage *m = m_pMessageList->first(); m; m =
		m_pMessageList->next())
	{
		if( kvi_strEqualCI(m->uid(), uid))
			return m;
	}
	return 0;
}

//
// This is the main plugin object , it is unique in the plugin
// It controls all the actions :
//     Keeps a list of the KviBiffMailbox widgets and signals them the changes
//     Keeps a list of KviBiffMailbox objects that contain the state of
//       the mailbox (list of messages and so on)
//     Keeps the config stuff
//     Controls the socket operations
// 
//     The KviBiffSocket is triggered once in a while (timeout)
//     to check a mailbox state and to update a specified KviBiffMailbox object
//     
//     The KviBiffWidget objects will be just "looping" thru these structures
//
//     A "difficult" part is handling the m_pMailboxList list:
//     It is shared between asynchronous objects
//     - The sockets sould never directly access the objects (always pass thru
//         the KviBiff class)
//         the existence of the entry to update should be always checked with
//         findRef()
//         READ-ONLY ACCESS! (Write access tru KviBiff)
//
//     - The widgets may get the data from the list but should not mess with it
//         and should not rely on the order or existence of the entries
//         the existence of a specified entry should be always checked with
//         findRef()
//         After a succesfull findRef() the widget may rely on the existence of
//         an entry
//         and the order of the list only for the duration of one function...
//         READ-ONLY ACCESS at all
//
//     - The biff class itself should not rely on the existence and order of
//         the entries
//         the existence of the current entry (m_pCurMailbox pointer) should be
//         always checked with a findRef()
//         READ-WRITE access
//

KviBiff::KviBiff()
{
	m_pBiffWidgetList = new QList<KviBiffWidget>;
	m_pBiffWidgetList->setAutoDelete(false);
	m_pMailboxList = new QList<KviBiffMailbox>;
	m_pMailboxList->setAutoDelete(true);
	
	m_pNormalIcon = new QPixmap( biff_icon_normal_xpm );
	m_pCheckingIcon = new QPixmap( biff_icon_checking_xpm );
	m_pGotNewMailIcon = new QPixmap( biff_icon_got_new_mail_xpm );

	m_pCurrentIcon = m_pNormalIcon;
	
	m_window = 0;
	m_pSocket = 0;
	m_pTimer = 0;
	m_pCurMailbox = 0;

	readConfig();

	m_checkingAll = false;
	
	if(m_config.timeout > 1000)
		start(); // one sec interval is impossible
}

KviBiff::~KviBiff()
{
	saveConfig();

	if(m_pSocket) delete m_pSocket;
	if(m_pTimer) delete m_pTimer;

	// A trick that will kill all the systray widgets...

	m_pBiffWidgetList->setAutoDelete(false);

	QList<KviBiffWidget> l;
	l.setAutoDelete(false);

	for(KviBiffWidget * w = m_pBiffWidgetList->first();w;
		w = m_pBiffWidgetList->next())
	{
		l.append(w);
	}

	for(KviBiffWidget * w = l.first();w;w = l.next())
	{
		kvirc_plugin_remove_systray_widget(w->frame(),w,true);
	}

	// Now m_pBiffWidgetList should be empty!
	__range_invalid(m_pBiffWidgetList->first());
	delete m_pBiffWidgetList;

	delete m_pMailboxList;
	delete m_pNormalIcon;
}

void KviBiff::start()
{
	if(m_pTimer)stop();
	m_pTimer = new QTimer(this);
	// WARNING : DO NOT START IMMEDIATELY...just after a timeout (even 0)
	// This function may be called at kvirc startup time
	// while many internal objects are not yet in a consistent state
	// (g_pOptions for example)
	//QObject::connect(m_pTimer,SIGNAL(timeout()),this,SLOT(checkAllMailboxes()));
	m_pTimer->start(m_config.timeout);
}

void KviBiff::stop()
{
	if(m_pTimer)
		delete m_pTimer;
	m_pTimer = 0;
}

//
// TODO: Take care of this one - a config dialog for auto-check
void KviBiff::checkAllMailboxes()
{
	KviFrame * frm = g_pApp->m_pFrameList->first();
	if(frm) m_window = frm->m_pConsole;	// FIXME : Need a decent way to get the
										//current window!

	__debug("checkAllMailboxes()");
	m_checkingAll = true;

	if(m_pSocket)
	{
		systrayMsg( "Already checking for mail !" );
		return;
	}

	if( ! m_pCurMailbox )
		m_pCurMailbox = m_pMailboxList->first();

	checkMailbox( m_pCurMailbox );
	m_pCurMailbox = m_pMailboxList->next();
	if( ! m_pCurMailbox )
		m_checkingAll = false;
}

// - - -

//
// I've spiltted checkMailboxes() into two functions, for convenience
bool KviBiff::checkMailbox(KviBiffMailbox *mbox)
{
	if( !mbox )
		return false;
	
	m_pCurrentIcon = m_pCheckingIcon;
	m_pCurMailbox = mbox;
		
// TODO : mbox and imap3 support ?....we should be choosing the correct
// handler object here...by now only pop3 : so KviBiffSocket
// mbox : oehansen suggestions:
// [17:41:04] <oehansen> -rw-------   1 oehansen staff       16216 Apr 24 06:09 /var/spool/mail/oehansen
// [17:41:48] <oehansen> it's what rfc822 describes
	m_pSocket = new KviBiffSocket();
	QObject::connect(m_pSocket, SIGNAL(resolving()), this,
		SLOT(socketResolving()));
	QObject::connect(m_pSocket,SIGNAL(connected()), this,
		SLOT(socketConnected()));
	QObject::connect(m_pSocket,SIGNAL(loggedIn()), this,
		SLOT(socketLoggedIn()));
	QObject::connect(m_pSocket,SIGNAL(error(const char *)), this,
		SLOT(socketError(const char *)));
	QObject::connect(m_pSocket,SIGNAL(jobDone()), this,
		SLOT(socketJobDone()));

	if(!m_pSocket->run(mbox))
	{
		// failure !
		systrayMsg(__tr("Failed to startup mail checking !"));
		delete m_pSocket;
		m_pSocket = 0;
		return false;
	}

// everything went fine
	return true;
}

// - - -

void KviBiff::socketResolving()
{
	if( ! m_pCurMailbox )
		return;

	KviStr s;
	s.sprintf( "Resolving host %s", m_pCurMailbox->hostname());
	systrayMsg(s.ptr());
}

// - - -

void KviBiff::socketConnected()
{
	for( KviBiffWidget *w = m_pBiffWidgetList->first(); w;
		w = m_pBiffWidgetList->next() )
	{
		KviStr s;
		s.sprintf( __tr("Connected to %s (%s) on port %d, logging in."),
			m_pSocket->hostname(),	m_pSocket->hostIp(), m_pSocket->port() );
		systrayMsg( s.ptr() );
	}
}

// - - -

void KviBiff::socketLoggedIn()
{
	if( ! m_pCurMailbox )
		return;

	KviStr s;
	s.sprintf( __tr("Logged in to %s as %s, checking for new mail..."),
		m_pCurMailbox->hostname(), m_pCurMailbox->username());
	systrayMsg( s.ptr(), 1, true );
}

// - - -

void KviBiff::socketError(const char * err)
{
	delete m_pSocket;
	m_pSocket = 0;
	m_pCurrentIcon = m_pNormalIcon;
	
	if( ! m_pCurMailbox )
		return;
		
	KviStr s;
	s.sprintf(__tr("Error while checking mail at %s@%s:\n%s"),
		m_pCurMailbox->username(),m_pCurMailbox->hostname(), err);
	systrayMsg(s.ptr());
	
	m_pCurMailbox = 0;
}

// - - -

void KviBiff::socketJobDone()
{
	if (! m_pCurMailbox )
	{
		__debug( "--- ERROR !!! ---" );
		systrayMsg( "Critical error...\nPlease submit a bug report." );
		return;
	}
	
	uint msg_count = m_pCurMailbox->messageCount();
	KviStr msg;
	if( msg_count == 0 )
	{
		msg.sprintf( __tr("There are no messages in %s@%s."),
			m_pCurMailbox->username(), m_pCurMailbox->hostname());
		m_pCurrentIcon = m_pNormalIcon;
	}
	else
	{
		if( msg_count == 1 )
			msg.sprintf( __tr("There is 1 message in %s@%s." /*, %u new." */),
				m_pCurMailbox->username(), m_pCurMailbox->hostname()/*,
				 m_pCurMailbox->newMessagesCount() */);
		else
			msg.sprintf( __tr("There are %u messages in %s@%s." /*, %u new."*/),
				msg_count,m_pCurMailbox->username(),m_pCurMailbox->hostname()/*,
				m_pCurMailbox->newMessagesCount() */);
//		if( m_pCurMailbox->newMessagesCount() )
		if( m_pCurMailbox->messageCount() )
			m_pCurrentIcon = m_pGotNewMailIcon;
		else
			m_pCurrentIcon = m_pNormalIcon;
	}

	systrayMsg(msg.ptr(), 1, true);
	
	delete m_pSocket;
	m_pSocket = 0;
	if( m_checkingAll )
		checkAllMailboxes();
	else
		m_pCurMailbox = 0;
}

// - - -

QPixmap * KviBiff::currentWidgetIcon()
{
	return m_pCurrentIcon;
}

// - - -

void KviBiff::registerSysTrayWidget(KviBiffWidget * w)
{
	m_pBiffWidgetList->append(w);
}

// - - -

void KviBiff::unregisterSysTrayWidget(KviBiffWidget * w)
{
	if(m_pBiffWidgetList->autoDelete())return; //we're deleting the widgets!
	m_pBiffWidgetList->removeRef(w);
}

// - - -

void KviBiff::readConfig()
{
	KviStr s;
	kvirc_plugin_get_config_file_path( s, "libkvibiff" );

// ensure that the mailbox list is empty
	if(m_pMailboxList->count())
		while(m_pMailboxList->first())
			m_pMailboxList->removeFirst();

	KviConfig cfg(s.ptr());
	m_config.beVerbose			= cfg.readBoolEntry("BeVerbose",true);
	m_config.timeout			= cfg.readUIntEntry("Timeout",180000);
	m_config.autoCheckAll		= cfg.readBoolEntry("AutoCheckAll", true );
	m_config.autoCheckInterval	= cfg.readUIntEntry("AutoCheckInterval",180000);
	m_config.systrayOnStartup	= cfg.readBoolEntry("SysTrayOnStartup", true);
	
	int entries  = cfg.readIntEntry("MailboxEntries",0);
	for(int i=0;i<entries;i++)
	{
		KviStr entry(KviStr::Format,"Mailbox%dHost",i);
		KviStr hostname = cfg.readEntry(entry.ptr(),"localhost");
		entry.sprintf("Mailbox%dUser",i);
		KviStr username = cfg.readEntry(entry.ptr(),"root");
		entry.sprintf( "Mailbox%dPass", i);
		KviStr enc_pass = cfg.readEntry(entry.ptr(),"");
		KviStr password = decryptString( enc_pass );
		entry.sprintf("Mailbox%dPort",i);
		unsigned int port = cfg.readUIntEntry(entry.ptr(),110);
		entry.sprintf("Mailbox%dAutoCheck", i);
		bool auto_check = cfg.readBoolEntry(entry.ptr(), true);

		KviBiffMailbox * m = new KviBiffMailbox(hostname.ptr(),username.ptr(),
			password.ptr(),port, auto_check);
		m_pMailboxList->append(m);
	}
}

// - - -

void KviBiff::saveConfig()
{
	KviStr s;
	kvirc_plugin_get_config_file_path( s, "libkvibiff" );

	KviConfig cfg(s.ptr());

	cfg.writeEntry("BeVerbose",m_config.beVerbose);
	cfg.writeEntry("Timeout",m_config.timeout);
	cfg.writeEntry("AutoCheckAll", m_config.autoCheckAll);
	cfg.writeEntry("AutoCheckInterval", m_config.autoCheckInterval);
	cfg.writeEntry("SysTrayOnStartup", m_config.systrayOnStartup);
	
	cfg.writeEntry("MailboxEntries",m_pMailboxList->count());
	int i = 0;
	for(KviBiffMailbox * m = m_pMailboxList->first();m;m=m_pMailboxList->next())
	{
		KviStr entry(KviStr::Format,"Mailbox%dHost",i);
		cfg.writeEntry(entry.ptr(),m->hostname());
		entry.sprintf("Mailbox%dUser",i);
		cfg.writeEntry(entry.ptr(),m->username());
		entry.sprintf("Mailbox%dPass",i);
		cfg.writeEntry(entry.ptr(), encryptString(m->password()).ptr() );
		entry.sprintf("Mailbox%dPort",i);
		cfg.writeEntry(entry.ptr(),m->port());
		entry.sprintf("Mailbox%dAutoCheck", i);
		cfg.writeEntry(entry.ptr(),m->autoCheck());
		i++;
	}
}

// - - -

const KviStr KviBiff::encryptString( const KviStr &s ) const
{
	const unsigned int len = s.len();

	//
	// Step 1: Reverse the string
#define FIXED_ARRAY_SIZE_TO_SATISFY_DAMN_ANSI_CPP_STANDARDS		256
	char tmp[FIXED_ARRAY_SIZE_TO_SATISFY_DAMN_ANSI_CPP_STANDARDS];
	int j = len;
	for( unsigned int i = 0; i < len; i++ )
	{
		tmp[i] = s.at(j - 1);
		j--;
		if( j == 0 )
			tmp[len] = '\0';
	}

	KviStr es( tmp );
	memset( tmp, '\0', FIXED_ARRAY_SIZE_TO_SATISFY_DAMN_ANSI_CPP_STANDARDS);

	//
	// Step 2: Encrypt the string. The method is kinda lame, but the encrypted
	// strings look hideous, and some of the chars are invisible :)
	const int mod = len % 2;
	for( unsigned int i = 0; i < len; i++ )
	{
		char c = es.at(i);
		mod ? mod % 2 ? c += len / mod : c -= len / mod : c -= len;
		tmp[i] = - c;
	}

	es = tmp;

	return es;

//	return KviStr(::crypt(s.ptr(), "kv"));
}

// - - -

const KviStr KviBiff::decryptString( const KviStr &s ) const
{
	return encryptString( s );
/*
	char *tmp = new char;
	*tmp = *s.ptr();
	::encrypt( tmp, 0 );
	KviStr ns = tmp;
	delete tmp;
	return ns;
*/
}

// - - -

void KviBiff::slotConfig()
{
	if( g_pConfigDlg )
	{
		__debug( "Config dialog already exists" );
		if( ! g_pConfigDlg->isVisibleToTLW() )
			g_pConfigDlg->raise();
		return;
	}

	g_pConfigDlg = new KviBiffConfigDlg();
	if( ! g_pConfigDlg )
	{
		systrayMsg( "Couldn't create config dialog ! ( no free mem?)" );
		debug( "[biff]: Couldn't create config dialog ! (no free mem?)" );
		return;
	}
	g_pConfigDlg->show();
	g_pConfigDlg->setOptions();
	connect( g_pConfigDlg, SIGNAL(defaultButtonPressed()), g_pBiff,
		SLOT(slotApplyOptions()));
	connect( g_pConfigDlg, SIGNAL(applyButtonPressed()), g_pBiff,
		SLOT(slotApplyOptions()));
	connect( g_pConfigDlg, SIGNAL(cancelButtonPressed()), g_pBiff,
		SLOT(slotKillConfigDialog()));
}

// - - -

void KviBiff::slotApplyOptions()
{
	g_pConfigDlg->getOptions();
	slotKillConfigDialog();
//	saveConfig();
}

// - - -

void KviBiff::slotKillConfigDialog()
{
	if( ! g_pConfigDlg )
		return;

	delete g_pConfigDlg;
	g_pConfigDlg = 0;
}

// - - -

void KviBiff::slotConfigureMailboxes()
{
	slotConfig();
	g_pConfigDlg->activateMailboxesTab();
}

// - - -

const KviBiffMailbox * const KviBiff::findMailbox( const char *username,
	const char *hostname ) const
{
	for( KviBiffMailbox *mb = m_pMailboxList->first(); mb;
		mb = m_pMailboxList->next())
	{
		if( kvi_strEqualCI(mb->username(), username ) &&
			kvi_strEqualCI(mb->hostname(), hostname) )
			return mb;
	}

	return 0;
}

// - - -

void KviBiff::slotCheckMailIn(int idx)
{
	if( m_pSocket )
	{
		systrayMsg( __tr("Already checking for mail !") );
		return;
	}

	m_checkingAll = false;
	checkMailbox( m_pMailboxList->at(idx));
}

// - - -

void KviBiff::slotCheckAll()
{
	m_checkingAll = true;
	checkAllMailboxes();
}

// - - -

void KviBiff::stopCheckingMail()
{
	if( ! m_pSocket )
	{
		systrayMsg( __tr("Not checking mail right now !"));
		return;
	}

	delete m_pSocket;
	m_pSocket = 0;
	systrayMsg( __tr( "Stopped checking mail." ));
	m_pCurrentIcon = m_pNormalIcon;
}

// - - -

void KviBiff::systrayMsg(const char *txt, uint time, bool p )
{
	if(!txt)
		return;

	for( KviBiffWidget *w=m_pBiffWidgetList->first(); w;
		w = m_pBiffWidgetList->next())
		w->showText(txt, time, p);
}


//
// -----
// GLOBAL STUFF
// -----
//


bool biff_plugin_command_biff(KviPluginCommandStruct *cmd)
{
	// syntax : biff [dock]
	//          biff undock

	KviStr param = kvirc_plugin_param(cmd,1);
	if(param.hasData())
	{
		if(kvi_strEqualCI(param.ptr(),"undock"))
		{
			KviSysTrayWidget * w = kvirc_plugin_lookup_systray_widget(
				cmd->frame,"KviBiffWidget");
			if(!w)
			{
				cmd->error = KVI_ERROR_InvalidParameter;
				cmd->errorstr = __tr("No biff widget to undock");
				return false;
			}
			kvirc_plugin_remove_systray_widget(cmd->frame,w,true);
			return true;
		}
	}
	// else it is dock (default)
	KviSysTrayWidget * w = kvirc_plugin_lookup_systray_widget(cmd->frame,
		"KviBiffWidget");
	if(w)
	{
		cmd->error = KVI_ERROR_InvalidParameter;
		cmd->errorstr = __tr("Biff widget already docked in this frame");
		return false;
	}

	KviBiffWidget * biff = new KviBiffWidget(kvirc_plugin_get_systray(
		cmd->frame ),cmd->frame,__tr("Biff"));
	kvirc_plugin_add_systray_widget(cmd->handle,cmd->frame,biff,true);

	return true;
}

bool biff_plugin_init(KviPluginCommandStruct *cmd)
{
	g_pBiff = new KviBiff();
	if(!g_pBiff)return false;

	kvirc_plugin_register_command( cmd->handle, "BIFF", 
		biff_plugin_command_biff );
	return true;
}

void biff_plugin_cleanup()
{
	delete g_pBiff;
	kvirc_plugin_unregister_meta_object("KviBiff");
	kvirc_plugin_unregister_meta_object("KviBiffConfigDlg");
	kvirc_plugin_unregister_meta_object("KviBiffSocket");
	kvirc_plugin_unregister_meta_object("KviBiffWidget");
}

void biff_plugin_config()
{
	g_pBiff->slotConfig();
}


/*

	@document: doc_plugin_biff.kvihelp
	@title: The KVirc Biff plugin
		<docsubtitle>KviBiff plugin</docsubtitle>
		Ok... The plugin now works... Not many features that are in plans have
		been implemented, but be patient - they will be :) As for now you can
		check how many messages are in you mailbox and how many <b>new</b>
		messages are there. 
		<ul>The plans for future are:
			<li>A lot of things to configure: Auto-check-all-mailboxes interval,
			auto-check interval for specific mailboxes, you'll be able to decide
			which mailboxes are auto-checked in "global" check</li>
			<li>Possibility to fire-up you favourite mail client to get the
			messages - the plugin will only display sender and subject. I still
			don't know how - will it be all in systray, or maybe we'll make a
			separate window (like in Yap's url plugin) ? Time will tell ;-)</li>
			<li>You will be able to delete chosen messages not even having to
			read them - if you see this is a spam message, or any other unwanted
			e-mail you can delete it from server, so you don't have to download
			it with your mail-client</li>
			<li>Maybe there will be some kind of sound-alarm when new mail
			arrives. This shouldn't be too much of a problem since KVIrc already
			has multimedia support (Yes !! Next plugin I plan to write is
			digital-cam-support-plugin also know as 
			"see-the-other-guy-in-the-systray" ;-} )</li>
		</ul>
		<br>
		That would be it... Ideas are welcome of course.<br>
		<hr>
		The <b>Biff</b> plugin was (is beeing actually) written by:<br>
		Krzysztof <i>"Kristoff"</i> Godlewski &lt;kristoff@poczta.wprost.pl&gt;
		<br>
		Szymon <i>"Pragma"</i> Stefanek &lt;stefanek@tin.it&gt;
	</docbody>
		
*/

void biff_plugin_help()
{
	kvirc_plugin_show_help_page( "doc_plugin_biff.kvihelp" );
}

KviPlugin kvirc_plugin =
{
	"biff",
	"Mail checking plugin",
	LIBKVIBIFF_VERSION,
	"Krzysztof Godlewski <kristoff@poczta.wprost.pl>\n" \
	"Szymon Stefanek <stefanek@tin.it>",
	"<html>This plugin allows you to check your mail accounts for new " \
	"messages without having to fire up any external mail clients.<br>" \
	"Exports <b>/BIFF</b> command.<br>See help page for more details.<br><br>" \
	"Please note that this plugin is still <u>under development</u>, so " \
	"all bug-reports are welcome.",
	biff_plugin_init,
	biff_plugin_cleanup,
	biff_plugin_config,
	biff_plugin_help
};


#include "libkvibiff.moc"
