//
//   File : kvi_uparser.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_uparser.cpp)
//   Last major modification : Fri Feb 12 1999 04:26:04 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_
//#define _KVI_DEBUG_CLASS_NAME_ "KviUserParser"
#include "kvi_debug.h"

#include "kvi_uparser.h"
#include "kvi_app.h"
#include "kvi_ircsocket.h"
#include "kvi_console.h"
#include "kvi_channel.h"
#include "kvi_defines.h"
#include "kvi_frame.h"
#include "kvi_options.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_asyncdialogs.h"
#include "kvi_alias.h"
#include "kvi_event.h"
#include "kvi_dcc_chat.h"
#include "kvi_slaveio.h"
#include "kvi_asyncwhois.h"
#include "kvi_malloc.h"
#include "kvi_scriptobject.h"

#ifdef COMPILE_PLUGIN_SUPPORT
	#include "kvi_plugin.h"
	extern KviPluginManager * g_pPluginManager;
#endif

//Created in KviApp constructor
extern KviAliasManager *g_pAliasManager;
extern KviEventManager *g_pEventManager;
extern KviVariableCache *g_pVarCache;

//===========================================================================//
// Class KviCommand
//
// Wrapper for the command string buffer.
// Contains a cache for local variables, parameters, and error codes
//

KviCommand::KviCommand(const char *pBuf,KviWindow *pWnd)
{
	m_err            = KVI_ERROR_NoError;
	m_ptr            = pBuf;
	m_wnd            = pWnd;
	m_pLocalVarCache = new KviVariableCache();
	m_pParamList     = 0;
	m_switchList     = new QList<KviStr>;
	m_switchList->setAutoDelete(true);
	m_pScopeObject   = 0;
	m_flags          = KVI_COMMAND_FLAG_RECOGNIZE_OBJECT_SCOPE_OPERATOR;
}
KviCommand::~KviCommand()
{
	if(m_pParamList)delete m_pParamList;
	delete m_pLocalVarCache;
	clearSwitchList();
	delete m_switchList;
}

bool KviCommand::hasSwitch(char letter)
{
	for(KviStr * s=m_switchList->first();s;s=m_switchList->next()){
		if(tolower(*(s->ptr())) == tolower(letter))return true;
	}
	return false;
}

bool KviCommand::getSwitchValue(char letter,KviStr &buffer)
{
	for(KviStr * s=m_switchList->first();s;s=m_switchList->next()){
		if(tolower(*(s->ptr())) == tolower(letter)){
			if(s->len() < 2)return false;
			buffer = s->ptr();
			buffer.cutLeft(2);
			return true;
		}
	}
	return false;
}

bool KviCommand::hasError(){ return (m_err != KVI_ERROR_NoError); }

void KviCommand::getParamCount(KviStr &buffer)
{
	if(!hasParams())buffer.append("0");
	else {
		KviStr num(KviStr::Format,"%d",m_pParamList->count());
		buffer.append(num);
//		KviStr num(KviStr::Format,"%d",m_pParams->numParams);
//		buffer.append(num);
	}
}
void KviCommand::setReturnValue(const char *retBuffer)
{
	m_retBuffer = retBuffer;
}

// $1, $1- $0- $0-4 $21-24 $12-

void KviCommand::getSingleParam(int param,KviStr &buffer)
{
	if(!hasParams())return;
	if(m_pParamList->count() > ((unsigned int)param))
	{
		KviStr * par = m_pParamList->at(param);
		if(par)buffer.append(*par);
	}
}

void KviCommand::getParam(int fromParam,int toParam,KviStr &buffer)
{
	if(!hasParams())return;
	if(((unsigned int)fromParam) >= m_pParamList->count())return;

	KviStr * par = m_pParamList->at(fromParam);
	if(par)
	{
		buffer.append(*par);
		if(toParam < fromParam)
		{
			// up to the end
			for(par = m_pParamList->next();par;par = m_pParamList->next())
			{
				buffer.append(' ');
				buffer.append(*par);
			}
		} else {
			if(((unsigned int)toParam) >= m_pParamList->count())toParam = m_pParamList->count();
			fromParam++;
			par = m_pParamList->next();
			while(par && (fromParam <= toParam))
			{
				buffer.append(' ');
				buffer.append(*par);
				par = m_pParamList->next();
				fromParam++;
			}
		}
	}
}

void KviCommand::setParams(const KviStr &param0,QList<KviStr> * pList)
{
	__range_invalid(hasParams());
	if(!pList)pList = new QList<KviStr>;
	pList->setAutoDelete(true);
	m_pParamList = pList;
	m_pParamList->insert(0,new KviStr(param0));
}

void KviCommand::setParams(const KviStr &param0,const char * paramBuffer)
{
	__range_valid(!hasParams());
	m_pParamList = new QList<KviStr>;
	m_pParamList->setAutoDelete(true);

	m_pParamList->append(new KviStr(param0));
	const char *aux;
	for(;;){
		// skip initial spaces
		while((*paramBuffer == ' ') || (*paramBuffer == '\t'))paramBuffer++;
		if(!*paramBuffer)return;
		aux = paramBuffer;
		while(*paramBuffer && (*paramBuffer != ' ') && (*paramBuffer != '\t'))paramBuffer++;
		m_pParamList->append(new KviStr(aux,paramBuffer - aux));

	}
}

bool KviCommand::setParamsFromCommand(const KviStr &param0,KviCommand *c,KviUserParser * u)
{
	__range_invalid(hasParams());

	m_pParamList = new QList<KviStr>;
	m_pParamList->setAutoDelete(true);
	m_pParamList->append(new KviStr(param0));
	for(;;){
		c->skipSpace();
		if(((*c->m_ptr) == ';') || ((*c->m_ptr) == '\0') || ((*c->m_ptr) == '\n'))break;
		c->clearBuffer();
		if(!u->processCmdSingleToken(c))return false;
		if(c->m_buffer.hasData())m_pParamList->append(new KviStr(c->m_buffer));
		else return u->processCmdFinalPart(c);
	}

	return true;
}

// This one returns ALWAYS false
bool KviCommand::setError(int iError,const char *szLocation,const char *szErrToken)
{
	m_err             = iError;
	m_szErrorLocation = szLocation;
	if(szErrToken){
		m_szErrorToken.setStr(szErrToken,40);
		if(m_szErrorToken.len() == 40)m_szErrorToken+="...";
	}
	return false;
}

void KviCommand::skipWhitespace()
{
	while(*m_ptr && isspace(*m_ptr))m_ptr++;
	if(*m_ptr == '\\'){
		//escape char
		m_ptr++;
		if(*m_ptr && isspace(*m_ptr)){
			//escaped whitespace
			m_ptr++;
			skipWhitespace();
		} else m_ptr--;
	}
}

void KviCommand::skipSpace()
{
	while((*m_ptr == ' ')||(*m_ptr == '\t'))m_ptr++;
	if(*m_ptr == '\\'){
		//escape char
		m_ptr++;
		if((*m_ptr == '\n')||(*m_ptr == ' ')||(*m_ptr == '\t')){
			//escaped newline, space or tab
			m_ptr++;
			skipSpace();
		} else m_ptr--;
	}
}

// This one ALWAYS returns true
bool KviCommand::warning(const char *format,...)
{
	__range_valid(!g_pOptions->m_bPedanticParser);
	if(g_pOptions->m_bAvoidParserWarnings)return true;
	char txt_ptr[512]; //It should be enough for all outputs...
	va_list list;
	va_start(list,format);
	if(kvi_vsnprintf(txt_ptr,512,format,list) < 0){
		//Just in case...
		va_end(list);
		int len = 512;
		char *long_txt_ptr = 0;
		int result;
		do{
			len += 512;
			//first time long_txt_ptr == 0 so it is equivalent to malloc
			//At least the man page says that...
			long_txt_ptr = (char *)kvi_realloc((void *)long_txt_ptr,len);
			va_start(list,format);
			result = kvi_vsnprintf(long_txt_ptr,len,format,list);
			va_end(list);
		} while(result < 0);
		m_wnd->output(KVI_OUT_INTERNAL,"[Parser]: %s",long_txt_ptr);
		kvi_free((void *)long_txt_ptr);
	} else {
		//Succesful vsnprintf
		va_end(list);
		m_wnd->output(KVI_OUT_INTERNAL,"[Parser]: %s",txt_ptr);
	}
	return true;
}


//=== KviUserParser class ===================================================//

KviUserParser::KviUserParser(KviFrame *pFrm)
:QObject()
{
	m_pFrm = pFrm;
	m_pAsyncDialogList = new QList<KviAsyncDialogCore>;
	m_pAsyncDialogList->setAutoDelete(false);
	m_pAsyncDnsList = new QList<KviAsyncDns>;
	m_pAsyncDnsList->setAutoDelete(true);
	m_pTimerList     = new QList<KviWindowTimer>;
	m_pTimerList->setAutoDelete(true);
	m_pSlaveIOController = new KviSlaveIO(pFrm,this);
	m_pAsyncWhoisController = new KviAsyncWhois(pFrm, this);
	m_szLastTryResult = "1";
	m_pScriptObjectController = new KviScriptObjectController(this);
}
KviUserParser::~KviUserParser()
{
	delete m_pScriptObjectController; // the first to die...remove all the objects (that might call something in the OnDestroy events)
	m_pAsyncDialogList->setAutoDelete(true); //Kill all the remaining dialogs
	delete m_pAsyncDialogList;
	delete m_pAsyncDnsList;
	while(!m_pTimerList->isEmpty()){
		KviWindowTimer * timer = m_pTimerList->first();
		__range_valid(timer);
		killTimer(timer->id);
		m_pTimerList->removeFirst();
	}
	delete m_pSlaveIOController;
	delete m_pAsyncWhoisController;
}
void KviUserParser::slaveAsyncProcessExited(KviStr &job,KviStr &commandline,KviStr &stdoutBuf,KviStr &stderrBuf,KviStr &magic,KviStr &window,unsigned int uPid,int iExitCode)
{
	m_szLastProcCommandline  = commandline;
	m_szLastProcMagic        = magic;
	m_szLastProcStdout       = stdoutBuf;
	m_szLastProcStderr       = stderrBuf;
	m_szLastProcPid.setNum(uPid);
	m_szLastProcExitCode.setNum(iExitCode);
	m_szLastProcWindow       = window;

	KviWindow * wnd = m_pFrm->findWindow(window.ptr());

	parseCommand(job.ptr(),wnd ? wnd : m_pConsole);

	m_szLastProcCommandline  = "";
	m_szLastProcMagic        = "";
	m_szLastProcStdout       = "";
	m_szLastProcStderr       = "";
	m_szLastProcPid          = "";
	m_szLastProcWindow       = "";
	m_szLastProcExitCode     = "";
}

void KviUserParser::asyncWhoisEnd(
	KviStr &asyncCmd,KviStr &magic,KviStr &window,KviStr &nick,KviStr &user,
	KviStr &host,KviStr &realName,KviStr &server,KviStr &idle,KviStr &status,
	bool bIrcOp, KviStr &chans,bool bSuccess)
{

	m_szLastAsyncWhoisNick		= nick;
	m_szLastAsyncWhoisUsername	= user;
	m_szLastAsyncWhoisHost		= host;
	m_szLastAsyncWhoisRealName	= realName;
	m_szLastAsyncWhoisServer	= server;
	m_szLastAsyncWhoisIdleTime	= idle;
	m_szLastAsyncWhoisStatus	= status;
	m_szLastAsyncWhoisIrcOp		= (bIrcOp ? "1" : "0");
	m_szLastAsyncWhoisChannels	= chans;
	m_szLastAsyncWhoisWindow	= window;
	m_szLastAsyncWhoisMagic		= magic;
	m_szLastAsyncWhoisSuccess   = (bSuccess ? "1" : "0");

	KviWindow * wnd = m_pFrm->findWindow(window.ptr());

	parseCommand(asyncCmd.ptr(),wnd ? wnd : m_pConsole);

	m_szLastAsyncWhoisNick		= "";
	m_szLastAsyncWhoisUsername	= "";
	m_szLastAsyncWhoisHost		= "";
	m_szLastAsyncWhoisRealName	= "";
	m_szLastAsyncWhoisServer	= "";
	m_szLastAsyncWhoisIdleTime	= "";
	m_szLastAsyncWhoisStatus	= "";
	m_szLastAsyncWhoisIrcOp		= "";
	m_szLastAsyncWhoisChannels	= "";
	m_szLastAsyncWhoisWindow	= "";
	m_szLastAsyncWhoisMagic		= "";
	m_szLastAsyncWhoisSuccess   = "";
}

int KviUserParser::completeCommand(KviStr &word,KviStr &match,KviStr &multiple)
{
	// completes the command in "word"
	// returns the number of matches
	//	1 if there is an exact match (and it is in match)
	//  0 if there are no matches
	//  n is there are n matches : multiple contains a list of the matches, match contains the common part
	KviCommandEntry * cmdTable;
	if(word.isEmpty())return 0;
	char init = tolower(*(word.ptr()));

	if(init < 'o')cmdTable = ((init < 'h') ? cmdTable_A_G : cmdTable_H_N);
	else cmdTable = ((init < 't') ? cmdTable_O_S : cmdTable_T_Z);

	int iMatches = 0;

	for(int i = 0;cmdTable[i].cmdName;i++){
		if(kvi_strEqualNoLocaleCIN(word.ptr(),cmdTable[i].cmdName,word.len())){
			if(match.hasData())
			{
				const char * c1 = match.ptr();
				const char * c2 = cmdTable[i].cmdName;
				while(*c1 && (tolower(*c1) == tolower(*c2)))
				{
					c1++;
					c2++;
				}
				match.cutRight(match.len() - (c1 - match.ptr()));
			} else match = cmdTable[i].cmdName;
			if(multiple.hasData())multiple.append(", ");
			multiple.append(cmdTable[i].cmdName);
			iMatches++;
		}
	}

	if(iMatches == 1)match.append(' ');
	return iMatches;
}

int KviUserParser::completeFunctionOrIdentifier(KviStr &word,KviStr &match,KviStr &multiple)
{
	// completes the function or identifer in "word"
	// returns the number of matches
	//	1 if there is an exact match (and it is in match)
	//  0 if there are no matches
	//  n is there are n matches : multiple contains a list of the matches, match contains the common part
	if(word.isEmpty())return 0;
	char init = tolower(*(word.ptr()));
	int iMatches = 0;

	KviFunctionEntry * fncTable;

	if(init < 'o')fncTable = ((init < 'h') ? fncTable_A_G : fncTable_H_N);
	else fncTable = ((init < 't') ? fncTable_O_S : fncTable_T_Z);


	for(int i = 0;fncTable[i].fncName;i++){
		if(kvi_strEqualNoLocaleCIN(word.ptr(),fncTable[i].fncName,word.len())){
			if(match.hasData())
			{
				const char * c1 = match.ptr();
				const char * c2 = fncTable[i].fncName;
				while(*c1 && (tolower(*c1) == tolower(*c2)))
				{
					c1++;
					c2++;
				}
				match.cutRight(match.len() - (c1 - match.ptr()));
			} else match.sprintf("%s(",fncTable[i].fncName);
			if(multiple.hasData())multiple.append(", ");
			multiple.append('$');
			multiple.append(fncTable[i].fncName);
			multiple.append("()");
			iMatches++;
		}
	}

	KviIdentifierEntry * idnTable;

	if(init < 'o')idnTable = ((init < 'h') ? idnTable_A_G : idnTable_H_N);
	else idnTable = ((init < 't') ? idnTable_O_S : idnTable_T_Z);


	for(int i = 0;idnTable[i].idnName;i++){
		if(kvi_strEqualNoLocaleCIN(word.ptr(),idnTable[i].idnName,word.len())){
			if(match.hasData())
			{
				const char * c1 = match.ptr();
				const char * c2 = idnTable[i].idnName;
				while(*c1 && (tolower(*c1) == tolower(*c2)))
				{
					c1++;
					c2++;
				}
				match.cutRight(match.len() - (c1 - match.ptr()));
			} else match = idnTable[i].idnName;
			if(multiple.hasData())multiple.append(", ");
			multiple.append('$');
			multiple.append(idnTable[i].idnName);
			iMatches++;
		}
	}
	if(iMatches == 1)
	{
		if(!match.lastCharIs('('))match.append(' ');
	}
	return iMatches;
}

void KviUserParser::timerEvent(QTimerEvent *e)
{
	KviWindowTimer * t = findTimerById(e->timerId());
	__range_valid(t);
	if(m_pFrm->windowExists(t->window)){
		m_szLastTimerMagic = t->magic;
		m_szLastTimerName  = t->name;
		if(!parseCommand(t->job.ptr(),t->window)){
			// Heed to kill the timer...there was an error
			t->singleShot = true;
		}
	} else t->singleShot = true; //kill it...no window
	if(t->singleShot){
		killTimer(t->id);
		m_pTimerList->removeRef(t);
	}
}

void KviUserParser::unregisterTimersFor(KviWindow * wnd)
{
	QList<KviWindowTimer> l;
	l.setAutoDelete(false);
	for(KviWindowTimer * t = m_pTimerList->first();t;t = m_pTimerList->next()){
		if(t->window == wnd)l.append(t);
	}
	for(KviWindowTimer * tim = l.first();tim;tim = l.next()){
		killTimer(tim->id);
		m_pTimerList->removeRef(tim);
	}
}

KviWindowTimer * KviUserParser::findTimerById(int id)
{
	for(KviWindowTimer * t = m_pTimerList->first();t;t = m_pTimerList->next()){
		if(t->id == id)return t;
	}
	return 0;
}

KviWindowTimer * KviUserParser::findTimerByName(const char *name)
{
	for(KviWindowTimer * t = m_pTimerList->first();t;t = m_pTimerList->next()){
		if(kvi_strEqualCI(name,t->name.ptr()))return t;
	}
	return 0;
}

void KviUserParser::addTimer(KviWindowTimer * t)
{
	KviWindowTimer * old = findTimerByName(t->name.ptr());
	if(!m_pFrm->windowExists(t->window)){
		__range_valid(m_pFrm->windowExists(t->window));
		return; //Make sure that exists
	}
	if(old){
		killTimer(old->id);
		m_pTimerList->removeRef(old);
	}
	m_pTimerList->append(t);
	t->id = startTimer(t->timeout);
}

bool KviUserParser::removeTimer(const char *name)
{
	KviWindowTimer * t = findTimerByName(name);
	if(!t)return false;
	killTimer(t->id);
	m_pTimerList->removeRef(t);
	return true;
}

void KviUserParser::setup(KviIrcSocket *pSock,KviConsole *pConsole)
{
	m_pConsole = pConsole;
	m_pSocket  = pSock;
}

void KviUserParser::addAsyncDialog(KviAsyncDialogCore *dlg)
{
	m_pAsyncDialogList->append(dlg);
}

void KviUserParser::removeAsyncDialog(KviAsyncDialogCore *dlg)
{
	__debug(">>> KviUserParser::removeAsycDialog");
	if(m_pAsyncDialogList->autoDelete())return; //KviUserParser destructor deleted the dialog and removed it from the list
	m_pAsyncDialogList->removeRef(dlg);
	__debug("<<< KviUserParser::removeAsyncDialog");
}

//=== parseUserCommand ======================================================//
//
// This is called directly from the KviInput class
//

void KviUserParser::parseUserCommand(KviStr &str,KviWindow *pWnd)
{
	KviChannel * chan = m_pFrm->findChannel(pWnd->caption());
	KviStr pme = m_pFrm->m_global.szCurrentNick;
	KviStr pmask = m_pFrm->m_global.szCurrentMaskFromServer;
	// This one parses a command buffer that comes from the input line
	// If the buffer contains an initial slash it is interpreted
	// as a buffer containing commands to be parsed
	// otherwise it is simply send to the appropriate target
	char *ptr = str.ptr();
	while(*ptr && (((unsigned char)*ptr) < 33))++ptr;               // skip initial whitespace...
	if(!*ptr)return;                                                // empty data buffer...
	if((*ptr=='/')||(*ptr==g_pOptions->m_cPersonalCommandPrefix)){  // a command
		ptr++;                                                      // skip the slash
		if(!*ptr)return;                                            // empty command
		parseCommand(ptr,pWnd);
	} else {                                                        // non command
		// check for a backslash at the beginning
		// and remove it (escaped /)...is it the righ way ? :)
		if(*ptr == '\\'){
			ptr++;
			if(*ptr)str.cut((ptr - str.ptr()) - 1,1);

		}
		switch(pWnd->type()){					// BuTi --- 1999-07-01 added events
			case KVI_WND_TYPE_CONSOLE:
				if(g_pEventManager->eventEnabled(KviEvent_OnConsoleInput)){
					KviStr eventparms(KviStr::Format,"%s",str.ptr());
					if(callEvent(KviEvent_OnConsoleInput,m_pConsole,eventparms)) return;
				}
				if(m_pSocket->sendData(str.ptr(),str.len()))
					pWnd->output(KVI_OUT_RAW,__tr("Raw to server: %s"),str.ptr());
				else pWnd->outputNoFmt(KVI_OUT_ERROR,__tr("Cannot send data: Not connected to server"));
			break;
			case KVI_WND_TYPE_CHANNEL:
				if(g_pEventManager->eventEnabled(KviEvent_OnChannelInput)){
					KviStr eventparms(KviStr::Format,"%s %s",pWnd->caption(),str.ptr());
					if(callEvent(KviEvent_OnChannelInput,pWnd,eventparms)) return;
				}
				if (((KviChannel *)pWnd)->m_bOnChannel){
					if (m_pSocket->sendFmtData("PRIVMSG %s :%s",pWnd->caption(),str.ptr())){
						if(g_pOptions->m_bEnableNickModePrefix){
							if(chan->isMeOwner()){
								pme.prepend('*');
								pmask.prepend('*');
							} else if(chan->isMeOp()){
								pme.prepend('@');
								pmask.prepend('@');
							} else if (chan->isMeHalfOp()){
								pme.prepend('%');
								pmask.prepend('%');
							} else if (chan->isMeVoice()){
								pme.prepend('+');
								pmask.prepend('+');
							} else if (chan->isMeUserOp()){
								pme.prepend('-');
								pmask.prepend('-');
							}
						}
						m_pFrm->outputPrivmsg(pWnd,KVI_OUT_OWN,pme.ptr(),
							pmask.ptr(),str.ptr());
					} else pWnd->outputNoFmt(KVI_OUT_ERROR,__tr("Cannot send data: Not connected to server"));
				} else pWnd->outputNoFmt(KVI_OUT_ERROR,__tr("Cannot send data: Not on channel"));
			break;
			case KVI_WND_TYPE_QUERY:
				if(g_pEventManager->eventEnabled(KviEvent_OnQueryInput)){
					KviStr eventparms(KviStr::Format,"%s %s",pWnd->caption(),str.ptr());
					if(callEvent(KviEvent_OnQueryInput,pWnd,eventparms)) return;
				}
				if (m_pSocket->sendFmtData("PRIVMSG %s :%s",pWnd->caption(),str.ptr())){
					m_pFrm->outputPrivmsg(pWnd,KVI_OUT_OWN,m_pFrm->m_global.szCurrentNick.ptr(),
						m_pFrm->m_global.szCurrentMaskFromServer.ptr(),str.ptr());
				} else pWnd->outputNoFmt(KVI_OUT_ERROR,__tr("Cannot send data: Not connected to server"));
			break;
			case KVI_WND_TYPE_CHAT:
				if(g_pEventManager->eventEnabled(KviEvent_OnDCCInput)){
					KviStr eventparms(KviStr::Format,"%s %s",pWnd->caption(),str.ptr());
					if(callEvent(KviEvent_OnDCCInput,pWnd,eventparms)) return;
				}
				if(((KviDccChat *)pWnd)->sendData(str.ptr())){
					m_pFrm->outputPrivmsg(pWnd,KVI_OUT_OWN,((KviDccChat *)pWnd)->m_szLocalNick.ptr(),
						((KviDccChat *)pWnd)->m_szLocalMask.ptr(),str.ptr());
				} else pWnd->output(KVI_OUT_ERROR,__tr("Cannot send data: No connection"));
			break;
			case KVI_WND_TYPE_UWINDOW:
				if(g_pEventManager->eventEnabled(KviEvent_OnUserWindowInput)){
					KviStr eventparms(KviStr::Format,"%s %s",pWnd->caption(),str.ptr());
					if(callEvent(KviEvent_OnUserWindowInput,pWnd,eventparms)) return;
				}
				pWnd->output(KVI_OUT_NONE,str.ptr());
			break;
			default:
				//do nothing
			break;
		}
	}
}

bool KviUserParser::parseCommand(const char *ptr,KviWindow *pWnd,const char * params)
{
	KviCommand cmd(ptr,pWnd);
	if(params)cmd.setParams(KviStr("commandline"),params);
	if(!execCommandBuffer(&cmd)){
		// stop the execution
		// if there is no error then we simply halt
		// if there is an error so spit it out
		if(cmd.hasError()){
			printError(&cmd,ptr);
			return false;
		}
	}
	return true;
}

//=== parseAsyncDialog callback =============================================//
//
// This is called directly from the KviAsyncDialogCore class
//

void KviUserParser::parseAsyncDialogCallback(KviStr &str,KviWindow *pWnd)
{
	// This one parses a command buffer that comes from the async dialog
	// Really similar to
	// First problem...the window pWnd may be not existing!
	if(!m_pFrm->windowExists(pWnd))pWnd = m_pConsole; //use the console instead
	char *ptr = str.ptr();
	while(*ptr && (*ptr < 33))++ptr;                                // skip initial whitespace...
	if(!*ptr)return;                                                // empty data buffer...
	KviCommand cmd(ptr,pWnd);
	if(!execCommandBuffer(&cmd)){
		if(cmd.hasError())printError(&cmd,ptr);
	}
}

void KviUserParser::getErrorInfo(KviCommand *c,const char * bufferBegin,KviCommandErrorInfo * inf)
{
	inf->iLineNum = 1;
	const char *ptr = bufferBegin;
	const char *lastLine = ptr;
	while(*ptr){
		if(ptr == c->m_ptr)break;
		if(*ptr++ == '\n')++(inf->iLineNum), lastLine = ptr;
	}
	// The current line data
	KviStr tmp;
	tmp.setStr(lastLine,35);
	tmp.getLine(inf->szLineStr);
	inf->szLineStr+="...";
	inf->iCharNum = c->m_ptr - lastLine;
}

void KviUserParser::printError(KviCommand *c,const char *bufferBegin)
{
	// The current line number
	KviCommandErrorInfo inf;
	getErrorInfo(c,bufferBegin,&inf);

	c->m_wnd->output(KVI_OUT_ERROR,"===================================->>>");
	c->m_wnd->output(KVI_OUT_ERROR,__tr("Error in command   : %s"),kvi_getErrorString(c->m_err));
	c->m_wnd->output(KVI_OUT_ERROR,"===================================->>>");
	c->m_wnd->output(KVI_OUT_ERROR,__tr("Parsing stopped at :"));
	c->m_wnd->output(KVI_OUT_ERROR,__tr("Line %d : %s"),inf.iLineNum,inf.szLineStr.ptr());
	c->m_wnd->output(KVI_OUT_ERROR,__tr("Character %d"),inf.iCharNum);
	c->m_wnd->output(KVI_OUT_ERROR,__tr("Logical location : %s"),c->m_szErrorLocation.ptr());
	if(c->m_szErrorToken.hasData()){
		int idx = c->m_szErrorToken.findFirstIdx('\n');
		KviStr tok((idx == -1) ? c->m_szErrorToken : c->m_szErrorToken.left(idx));
		if(tok.hasData())c->m_wnd->output(KVI_OUT_ERROR,__tr("Error detail : %s"),tok.ptr());
	}
}

//=== THE REAL STUFF ========================================================//


bool KviUserParser::execCommandBuffer(KviCommand *c)
{
	// This one parses a whole command buffer executing it
	// Returns false in case of execution stopped
	// If cmd->err is non-zero in that moment, it
	// means that an error occured, otherwise the execution was simply halted
	c->skipWhitespace();
	while(*(c->m_ptr)){
		if(*(c->m_ptr) == '{'){
			if(!execCommandBlock(c))return false;
			// Here c->m_ptr points to the char immediately after the closing brace
		} else {
			if(!execSingleCommand(c))return false;
			// Here c->m_ptr points to the char immediately after the separator...
		}
		c->skipWhitespace();
	}
	return true;	
}

bool KviUserParser::notConnectedToServer(KviCommand *c,const char *inCommand)
{
	if(g_pOptions->m_bPedanticParser)return c->setError(KVI_ERROR_NotConnectedToServer,inCommand);
	else return c->warning(__tr("%s: Not connected to server : Ignoring"),inCommand);
}

bool KviUserParser::recoverableError(KviCommand *c,int error,const char *cmdName,const char *additionalData)
{
	if(g_pOptions->m_bPedanticParser)return c->setError(error,cmdName,additionalData);
	else {
		if(*additionalData)return c->warning(__tr("%s: %s (%s) : Ignoring"),cmdName,kvi_getErrorString(error),additionalData);
		else return c->warning(__tr("%s: %s : Ignoring"),cmdName,kvi_getErrorString(error));
	}
}

bool KviUserParser::execCommandBlock(KviCommand *c)
{
	// This one parses a block of commands enclosed in braces
	// returns false in case of an error somewhere in the command calls
	// or if there are not matching braces...
	// moves cmd->ptr up to the char after the matching closing brace.
	__range_valid(*(c->m_ptr)=='{');
	++(c->m_ptr);
	c->skipWhitespace();
	while(*(c->m_ptr)){
		switch(*(c->m_ptr)){
			case '}': ++(c->m_ptr);            return true;  break;
			case '{': if(!execCommandBlock(c)) return false; break;
			default : if(!execSingleCommand(c))return false; break;
		}
		c->skipWhitespace();
	}
	// We should newer reach this point
	c->setError(KVI_ERROR_MissingClosingBrace,_i18n_("COMMAND-BLOCK"));
	return false;
}



bool KviUserParser::execSingleCommand(KviCommand *c)
{
	// This one is called when c->m_ptr points to the beginning
	// of a command. (ex : echo -w $window some funny text;...)
	// If the first char is a KVI_GLOBAL_VAR_CHAR, we assume it to be
	// an assignment (or another unary operator)....
	// otherwise we look for the correct command proc to call

	c->clearBuffer();

	if(*(c->m_ptr) == ';'){
		++(c->m_ptr); //empty command;
		return true;
	}

	if((*(c->m_ptr) == KVI_GLOBAL_VAR_CHAR) || (*(c->m_ptr) == '$'))return parseCmd_LVALUE(c);

	// find the correct command table
	KviCommandEntry * cmdTable;
	char init = tolower(*(c->m_ptr));

	if(init < 'o')cmdTable = ((init < 'h') ? cmdTable_A_G : cmdTable_H_N);
	else cmdTable = ((init < 't') ? cmdTable_O_S : cmdTable_T_Z);

	for(int i = 0;cmdTable[i].cmdName;i++){
		if(kvi_strEqualNoLocaleCIN(c->m_ptr,cmdTable[i].cmdName,cmdTable[i].cmdLen)){
			// check if the command is followed by a space or terminator
			// or something that is not a letter or digit
			const char *aux = (c->m_ptr + cmdTable[i].cmdLen);
			if((!isalnum(*aux)) && (*aux !='_') && (*aux != '.')){ //yes...got it...
				c->m_ptr      = aux;
				return (this->*(cmdTable[i].cmdProc))(c);
			}
		}
	}

	if(*(c->m_ptr) == KVI_COMMENT_BEGIN_CHAR)return skipComment(c);

	const char *aux = c->m_ptr;
	while(*(c->m_ptr) && (isalnum(*(c->m_ptr)) || (*(c->m_ptr)== '_') || (*(c->m_ptr)=='.')))++(c->m_ptr);
	KviStr tmp(aux,c->m_ptr - aux);

#ifdef COMPILE_PLUGIN_SUPPORT
	// Plugin command ?
	KviPluginCommandHandler * h=g_pPluginManager->findCommandHandler(tmp.ptr());
	if(h)return parsePluginCommand(h,c);
#endif

	//Alias ?
	KviAlias *a=g_pAliasManager->findAlias(tmp.ptr());
	if(a)return parseAlias(c,a);

	//nope.
	// send command...
	if(g_pOptions->m_bSendUnknownCommandsToServer)return parseUnknownCommand(tmp,c);
	// or return error?
	c->setError(KVI_ERROR_UnknownCommand,"COMMAND",c->m_ptr);
	return false;
}

bool KviUserParser::parseAlias(KviCommand *c,KviAlias *a)
{
	KviCommand cmd(a->szBuffer.ptr(),c->m_wnd);
	c->skipSpace();
	if(!cmd.setParamsFromCommand(a->szName,c,this))return false;
	a->bLocked = true; //Protect against self modification
	if(!execCommandBuffer(&cmd)){
		a->bLocked = false; //Unlock
		if(cmd.hasError()){
			printError(&cmd,a->szBuffer.ptr());
			return false; //and halt
		}
	}
	c->setReturnValue(cmd.m_retBuffer.ptr());
	a->bLocked = false; //Unlock
	return true;
}

bool KviUserParser::parsePluginCommand(KviPluginCommandHandler * h,KviCommand *c)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviPluginCommandStruct cmdstr;
	cmdstr.handle  = h->plugin_handle;
	cmdstr.app     = g_pApp;
	cmdstr.params  = new QList<KviStr>;
	cmdstr.params->setAutoDelete(true);
	cmdstr.sock    = m_pSocket;
	cmdstr.console = m_pConsole;
	c->skipSpace();

	cmdstr.params->append(new KviStr(h->cmd_name.ptr()));

	while((*(c->m_ptr) != ';') && (*(c->m_ptr) != '\n') && (*(c->m_ptr))){
		if(!processCmdSingleToken(c)){
			while(cmdstr.params->first())cmdstr.params->removeFirst();
			delete cmdstr.params;
			return false;
		}
		cmdstr.params->append(new KviStr(c->m_buffer));
		c->m_buffer = "";
		c->skipSpace();
	}
	
	if((*(c->m_ptr)))c->m_ptr++;

	cmdstr.frame  = m_pFrm;
	cmdstr.window = c->m_wnd;
	cmdstr.error  = KVI_ERROR_NoError;

	bool bResult = h->handler_routine(&cmdstr);

	delete cmdstr.params;

	// error in plugin
	if(!bResult)c->setError(cmdstr.error,h->cmd_name.ptr(),cmdstr.errorstr.ptr());

	return bResult;
#else //!COMPILE_PLUGIN_SUPPORT
	return false;
#endif //!COMPILE_PLUGIN_SUPPORT
}

bool KviUserParser::parsePluginFunction(KviPluginFunctionHandler *h,KviCommand *c,KviStr &buffer)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviPluginCommandStruct cmdstr;
	cmdstr.handle  = h->plugin_handle;
	cmdstr.app     = g_pApp;
	cmdstr.params  = new QList<KviStr>;
	cmdstr.params->setAutoDelete(true);
	cmdstr.sock    = m_pSocket;
	cmdstr.console = m_pConsole;
	c->skipSpace();

	cmdstr.params->append(new KviStr(h->fnc_name.ptr()));

	KviStr tmp;

	while((*(c->m_ptr) != ')') && (*(c->m_ptr))){
		if(!processFncSingleParam(c,tmp)){
			while(cmdstr.params->first())cmdstr.params->removeFirst();
			delete cmdstr.params;
			return false;
		}
		cmdstr.params->append(new KviStr(tmp));
		tmp = "";
		c->skipSpace();
	}
	
	if((*(c->m_ptr)))c->m_ptr++;

	cmdstr.frame  = m_pFrm;
	cmdstr.window = c->m_wnd;
	cmdstr.error  = KVI_ERROR_NoError;

	bool bResult = h->handler_routine(&cmdstr,&tmp);

	buffer.append(tmp);

	delete cmdstr.params;

		// error in plugin
	if(!bResult)c->setError(cmdstr.error,h->fnc_name.ptr(),cmdstr.errorstr.ptr());

	return bResult;
#else //!COMPILE_PLUGIN_SUPPORT
	return false;
#endif //!COMPILE_PLUGIN_SUPPORT
}

bool KviUserParser::parsePluginIdentifier(KviPluginFunctionHandler *h,KviCommand *c,KviStr &buffer)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviPluginCommandStruct cmdstr;
	cmdstr.handle  = h->plugin_handle;
	cmdstr.app     = g_pApp;
	cmdstr.params  = 0;
	cmdstr.sock    = m_pSocket;
	cmdstr.console = m_pConsole;
	c->skipSpace();
	cmdstr.frame  = m_pFrm;
	cmdstr.window = c->m_wnd;
	cmdstr.error  = KVI_ERROR_NoError;

	KviStr tmp;
	bool bResult = h->handler_routine(&cmdstr,&tmp);
	buffer.append(tmp);

	if(!bResult){
		// error in plugin
		c->setError(cmdstr.error,h->fnc_name.ptr(),cmdstr.errorstr.ptr());
	}

	return bResult;
#else //!COMPILE_PLUGIN_SUPPORT
	return false;
#endif //!COMPILE_PLUGIN_SUPPORT
}

bool KviUserParser::callEvent(int index,KviWindow *pWnd,const KviStr &params)
{
	// returns TRUE if the execution was HALTED!
	// otherwise FALSE!
#ifdef COMPILE_PLUGIN_SUPPORT
	__range_valid(g_pEventManager->m_bEventEnabled[index] || g_pEventManager->m_pEventHandlerList[index]);
#else
	__range_valid(g_pEventManager->m_bEventEnabled[index]);
#endif
//	if(!g_pEventManager->m_bEventEnabled[index])debug("Event not enabled %d",index);

	bool bHalt = false;

#ifdef COMPILE_PLUGIN_SUPPORT
	if(g_pEventManager->m_pEventHandlerList[index]){
		KviPluginCommandStruct cmd;
		cmd.app     = g_pApp;
		cmd.params  = new QList<KviStr>;
		cmd.params->setAutoDelete(true);
		cmd.params->append(new KviStr(g_pEventManager->m_eventDescriptor[index].eventName));
		cmd.sock    = m_pSocket;
		cmd.console = m_pConsole;
		const char *aux = params.ptr();
		KviStr token;
		while(*aux){
			aux = kvi_extractToken(token,aux,' ');
			token.stripSpace();
			if(token.hasData())cmd.params->append(new KviStr(token));
		}
		cmd.window = pWnd;
		cmd.frame  = m_pFrm;
		cmd.error  = KVI_ERROR_NoError;

		for(KviPluginEventHandlerStruct *s=g_pEventManager->m_pEventHandlerList[index]->first();s;s=g_pEventManager->m_pEventHandlerList[index]->next()){
			cmd.handle = s->plugin_handle;
			bHalt = bHalt || (s->handler_routine)(&cmd);
		}
		delete cmd.params;
	}

	if(g_pEventManager->m_bEventEnabled[index]){
#endif
		KviCommand cmd(g_pEventManager->m_szEventBuffer[index].ptr(),pWnd);
		cmd.setParams(KviStr(g_pEventManager->m_eventDescriptor[index].eventName),params.ptr());
		if(!execCommandBuffer(&cmd)){
			if(cmd.hasError()){
				printError(&cmd,g_pEventManager->m_szEventBuffer[index].ptr());
				pWnd->output(KVI_OUT_ERROR,__tr("Caller event     : %s"),g_pEventManager->m_eventDescriptor[index].eventName);
				if(g_pOptions->m_bAutoDisableBrokenEvents){
					pWnd->output(KVI_OUT_ERROR,__tr("Event handler '%s' is broken : Auto-disabling"),g_pEventManager->m_eventDescriptor[index].eventName);
					g_pEventManager->m_bEventEnabled[index] = false;
				}
				if(index != KviEvent_OnEventError)
				{
					if(g_pEventManager->eventEnabled(KviEvent_OnEventError)){

						KviCommandErrorInfo inf;
						getErrorInfo(&cmd,g_pEventManager->m_szEventBuffer[index].ptr(),&inf);

						KviStr eventparms(KviStr::Format,"Event: %s Location: %s Line: %d Char: %d Error: %s Window: %s LastLine: %s",g_pEventManager->m_eventDescriptor[index].eventName,
							cmd.m_szErrorLocation.ptr(),inf.iLineNum,inf.iCharNum,kvi_getErrorString(cmd.m_err),pWnd->caption(),inf.szLineStr.ptr());

						callEvent(KviEvent_OnEventError,m_pConsole,eventparms);
					}
				}

				return bHalt;
			} else return true; //halted!
		}
#ifdef COMPILE_PLUGIN_SUPPORT
	}
#endif
	return bHalt;
}

bool KviUserParser::parseUnknownCommand(KviStr &cmd,KviCommand *c)
{
	if(!processCmdFinalPart(c))return false;
	if(!m_pSocket->sendFmtData("%s :%s",cmd.ptr(),c->m_buffer.ptr()))return notConnectedToServer(c,cmd.ptr());
	return true;
}

bool KviUserParser::callRawEvent(const KviStr &szCommand,const KviStr &szPrefix,const KviStr &szBuffer,const KviStr &szParams,const KviStr &szSourceMask,KviStr &retValue)
{
	KviCommand cmd(szBuffer.ptr(),m_pConsole);
	KviStr tmp(KviStr::Format,"%s %s",szPrefix.ptr(),szParams.ptr());
	cmd.setParams(szCommand,tmp.ptr());
	if(!execCommandBuffer(&cmd)){
		retValue = cmd.m_retBuffer;
		if(cmd.hasError()){
			printError(&cmd,szBuffer.ptr());
			m_pConsole->output(KVI_OUT_ERROR,__tr("Caller event     : Raw %s (source mask: %s)"),szCommand.ptr(),szSourceMask.ptr());

			if(g_pEventManager->eventEnabled(KviEvent_OnEventError)){

				KviCommandErrorInfo inf;
				getErrorInfo(&cmd,szBuffer.ptr(),&inf);

				KviStr eventparms(KviStr::Format,"Event: Raw %s Location: %s Line: %d Char: %d Error: %s SourceMask: %s LastLine: %s",szCommand.ptr(),
					cmd.m_szErrorLocation.ptr(),inf.iLineNum,inf.iCharNum,kvi_getErrorString(cmd.m_err),szSourceMask.ptr(),inf.szLineStr.ptr());

				callEvent(KviEvent_OnEventError,m_pConsole,eventparms);
			}

			return false;
		} else return true; //halted!
	} else retValue = cmd.m_retBuffer;
	return false;
}

bool KviUserParser::triggerObjectEvent(KviScriptObject *o,const char *eventName,const char *eventBuf,const KviStr &params)
{
	KviStr tmp = eventBuf;
	KviCommand cmd(tmp.ptr(),m_pConsole);
	cmd.setParams(KviStr(eventName),params.ptr());
	cmd.setThisId(o->id());
	if(!execCommandBuffer(&cmd)){
		if(cmd.hasError()){
			printError(&cmd,tmp.ptr());
			m_pConsole->output(KVI_OUT_ERROR,__tr("Caller event     : %s"),eventName);
			m_pConsole->output(KVI_OUT_ERROR,__tr("Caller object    : type(%s) name(%s) id(%s)"),o->getClass(),o->getName(),o->id());
			if(g_pOptions->m_bAutoDisableBrokenEvents){
				m_pConsole->output(KVI_OUT_ERROR,__tr("Event handler '%s' for this object is broken: auto-removing"),eventName);
				o->removeEventHandler(eventName);
			}

			if(g_pEventManager->eventEnabled(KviEvent_OnEventError)){

				KviCommandErrorInfo inf;
				getErrorInfo(&cmd,tmp.ptr(),&inf);

				KviStr eventparms(KviStr::Format,"Event: Obj-%s Location: %s Line: %d Char: %d Error: %s CallerObjectClass: %s CallerObjectName: %s LastLine: %s",eventName,
					cmd.m_szErrorLocation.ptr(),inf.iLineNum,inf.iCharNum,kvi_getErrorString(cmd.m_err),o->getClass(),o->getName(),inf.szLineStr.ptr());

				callEvent(KviEvent_OnEventError,m_pConsole,eventparms);
			}

		} else return true; //halted!
	}
	return false; // Don't stop
}

bool KviUserParser::callObjectFunction(KviScriptObject *o,const char *fncName,const char *fncBuf,QList<KviStr> * params,KviStr &buffer)
{
	// If the params list is non null, it will be deleted!
	KviStr tmp = fncBuf;
	KviCommand cmd(tmp.ptr(),m_pConsole);

	cmd.setParams(KviStr(fncName),params);
	cmd.setThisId(o->id());

	bool bRet = execCommandBuffer(&cmd);
	buffer.append(cmd.m_retBuffer);
	if(!bRet)
	{
		if(cmd.hasError())
		{
			printError(&cmd,tmp.ptr());
			m_pConsole->output(KVI_OUT_ERROR,__tr("Caller function     : %s"),fncName);
			m_pConsole->output(KVI_OUT_ERROR,__tr("Caller object    : type(%s) name(%s) id(%s)"),o->getClass(),o->getName(),o->id());
			return false;
		}
	}
	return true; // everything OK
}

bool KviUserParser::extractSwitches(KviCommand *c)
{
	// This one 
	c->clearSwitchList();
	c->skipSpace();
	while(*(c->m_ptr) == '-'){
		if(!extractSingleSwitch(c))return false;
		c->skipSpace();
	}
	return true;
}

bool KviUserParser::extractSingleSwitch(KviCommand *c)
{
	__range_valid(*(c->m_ptr) == '-');

	++(c->m_ptr); // skip the - char

	// now a LETTER is required
	if( (!(*(c->m_ptr))) || (!isalpha(*(c->m_ptr))) ){
		c->setError(KVI_ERROR_InvalidOption,__tr("SWITCH PARSING AFTER '-' DELIMITER"),c->m_ptr);
		return false;
	}

	KviStr * sw = new KviStr();
	sw->append(*(c->m_ptr));

	++(c->m_ptr); // skip the switch letter

	c->skipSpace();

	// now if we have a '=', a token follows the switch

//	debug("switch %s",sw->ptr());

	if(*(c->m_ptr) == '='){
		// a token follows the switch
//		debug("A token follows it!");
		++(c->m_ptr);
		c->skipSpace();
		sw->append('=');

		const char *aux_ptr = c->m_ptr;

		bool bInString = false;

		for(;;){
			// Now skip all the non interesting chars
			while(*aux_ptr && (*aux_ptr!='\\') && (*aux_ptr!='\n') &&
					(*aux_ptr!=' ') && (*aux_ptr!='\t') && (*aux_ptr!=';') &&
					(*aux_ptr!='$') && (*aux_ptr!='\"') && (*aux_ptr!=KVI_GLOBAL_VAR_CHAR))aux_ptr++;
			// Interesting char
			switch(*aux_ptr){
				case ' ' :
				case '\t':
				case ';' :
					if(!bInString){ // End of the token...append the last block to the buffer and return
						sw->append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = aux_ptr;
//						debug("Complete switch %s",sw->ptr());
						c->m_switchList->append(sw);
						return true;
					} else while((*aux_ptr == ' ')||(*aux_ptr == '\t')||
								(*aux_ptr == ';'))aux_ptr++; //In string...must skip it
				break;
				case '\n': // end of command
				case '\0': // end of command buffer...append the last block to the buffer and return
					if(bInString){
						// unsecaped newline or end of data
						c->setError(KVI_ERROR_UnexpectedEndInString,__tr("SWITCH PARSING: PARAMETER EXTRACTION"));
						delete sw;
						return false;
					} // else it is the end of the command
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					c->m_switchList->append(sw);
					return true;
				break;
				case '\"': // beginning or end of a string
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					bInString = !bInString;
					c->m_ptr = ++aux_ptr;
				break;
				case KVI_GLOBAL_VAR_CHAR: //variable: append the last block to the buffer and process the var
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!processVariable(c,*sw)){
						delete sw;
						return false;
					}
					aux_ptr = c->m_ptr;
				break;
				case '$': //system identifier: append the last block to the buffer and process the ident
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!processIdentifier(c,*sw)){
						delete sw;
						return false;
					}
					aux_ptr = c->m_ptr;
				break;
				case '\\': //escape character: append the last block to the processed buffer...
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr;
					switch(*aux_ptr){
						case '\0':
							if(bInString){
								c->setError(KVI_ERROR_UnexpectedEndInString,__tr("SWITCH PARSING: PARAMETER EXTRACTION"));
								delete sw;
								return false;
							}
							c->m_switchList->append(sw);
							return true;
						break;
						case '\n':  //escaped newline
							c->m_ptr = ++aux_ptr; //skip it
							c->skipSpace();       //skip leading spaces
							aux_ptr = c->m_ptr;   //continue
						break;
						default: // Must be copied to the buffer without modification
							sw->append(*aux_ptr);
							c->m_ptr = ++aux_ptr;
						break;
					}
				break;
			}
		}
	}

	// not '='
	c->m_switchList->append(sw);

	return true;
}

bool KviUserParser::processObjectScopeRValue(KviScriptObject *o,KviCommand *c,KviStr &buffer)
{
	// This one must be called when c->m_ptr points to
	// a '%' or '$'  char just after
	// the '->' object scope operator
	KviScriptObject * aux = c->scopeObject(); // save the current one
	c->setScopeObject(o);
	switch(*(c->m_ptr))
	{
		case '%':
			// Object variable
			if(!processVariable(c,buffer))return false;
		break;
		case '$':
			// Object function or identifier call
			if(!processIdentifier(c,buffer))return false;
		break;
		default:
			// Invalid !
			debug("Invalid char in object scope %s, check the code man!",c->m_ptr);
			return false;
		break;
	}
	c->setScopeObject(aux);
	return true;
}

bool KviUserParser::processVariable(KviCommand *c,KviStr &buffer)
{
	// This one parses the variable pointed by ptr
	// adds its value to c->m_buffer
	// (or an empty string if the var is not defined)
	// then moves c->m_ptr to the characted immediately 
	// after the variable and returns

	// <KVI_GLOBAL_VAR_CHAR>Variable = global variables
	// <KVI_GLOBAL_VAR_CHAR>variable = local variables
	// <KVI_GLOBAL_VAR_CHAR>Variable[index] = global dict
	// <KVI_GLOBAL_VAR_CHAR>lariable[index] = local dict

	__range_valid(*c->m_ptr == KVI_GLOBAL_VAR_CHAR);
	const char *aux = ++(c->m_ptr);

	KviVariableCache *cache;

	if(c->scopeObject())
	{
		cache = c->scopeObject()->varCache();
	} else {
		if(islower(*aux))cache = c->m_pLocalVarCache;
		else cache = g_pVarCache;
	}

	while(*aux && (isalnum(*aux)||(*aux == '.')||(*aux == '_')))aux++;
	KviStr szVariable(c->m_ptr,aux - c->m_ptr);
	c->m_ptr = aux;

	const char * varVal = 0;

	if(szVariable.isEmpty())buffer.append(KVI_GLOBAL_VAR_CHAR);
	else {
//		debug("Variable = %s Object scope = %d, cache = %d, local cache = %d, global = %d",szVariable.ptr(),c->scopeObject(),(int)cache,(int)(c->m_pLocalVarCache),(int)g_pVarCache);
		if(*(c->m_ptr) == '['){
			c->m_ptr++;
			KviStr szKey;
			// No scope object for variable keys
			KviScriptObject * scObj = c->scopeObject();
			c->setScopeObject(0);
			if(!processVariableKey(c,szKey))return false;
			c->setScopeObject(scObj);
			// Ok...got the key
			if(szKey.isEmpty()){
				return c->setError(KVI_ERROR_MissingDictionaryKey,__tr("VARIABLE EVALUATION"),szVariable.ptr());
//				else return c->warning(__tr("VARIABLE EVALUATION : Missing dictionary key for variable %s[] : ignoring"),szVariable.ptr());
			}
			// KviStr szDictionary = szVariable :)
			varVal = cache->findDictVariable(szVariable.ptr(),szKey.ptr());
		} else {
			varVal = cache->find(szVariable.ptr());
		}
		// Object scope ?
		if(*(c->m_ptr) == '-')
		{
			if(*(c->m_ptr + 1) == '>')
			{
				c->m_ptr += 2;

				if((*(c->m_ptr) == '%') || (*(c->m_ptr) == '$'))
				{
					if(varVal)
					{
						KviScriptObject * o = m_pScriptObjectController->findObjectById(varVal);
						if(o)return processObjectScopeRValue(o,c,buffer);
						else return c->setError(KVI_ERROR_ObjectNotFound,__tr("OBJECT OPERATOR -> (RVALUE)"),varVal);
					} else return c->setError(KVI_ERROR_ObjectNotFound,__tr("OBJECT_OPERATOR -> (RVALUE)"),__tr("The variable contained no object id"));
				} else return c->setError(KVI_ERROR_VariableOrIdentifierExpected,__tr("OBJECT OPERATOR -> (RVALUE)"),c->m_ptr);
			}
		}

		if(varVal)buffer.append(varVal);
	}
	return true;
}

bool KviUserParser::processVariableKey(KviCommand *c,KviStr &buffer)
{
	// This one must be called when c->m_ptr points
	// somewhere in a variable parameters block.
	// The parsing is stopped by a '\n', ']' or '\0'
	// on '\n' and '\0' an error is returned
	// The parsing is NOT stopped if ']'
	// is in a string (delimitated by two '"' chars)
	// Skips the ending ']' character
	// This function EATS leading and trailing spaces and tabs
	// and reduces the middle ones to one per group.
	// A key like this:
	//          string    "full  of  words"  and  spaces      ,
	// Becomes like this:
	// string full  of  words and spaces,

	// Skip leading spaces and tabs (the escaped ones too)
	c->skipSpace();
	register const char *aux_ptr = c->m_ptr;

	bool inString      = false;

	for(;;){
		// First skip all the non interesting chars
		while(	*aux_ptr         && (*aux_ptr!=KVI_GLOBAL_VAR_CHAR) &&
				(*aux_ptr!='\n') && (*aux_ptr!=']')  && (*aux_ptr!='$')  && 
				(*aux_ptr!=' ')  && (*aux_ptr!='"')  && (*aux_ptr!='\\') &&
				(*aux_ptr!='\t'))aux_ptr++;
		// Interesting char
		switch(*aux_ptr){
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				if(inString)c->setError(KVI_ERROR_UnexpectedEndInString,__tr("DICTIONARY KEY"));
				else c->setError(KVI_ERROR_UnexpectedEndOfCommand,__tr("DICTIONARY KEY"));
				return false;
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!inString){
					// append the last part
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != ']'))buffer.append(' ');
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case '"': // Not escaped '"'...enter or exit the string (skip the '"')
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				inString = !inString;
			break;
			case ']': // If not in string, exit
				if(!inString){
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr;
					return true;
				}
				++aux_ptr;
			break;
			case KVI_GLOBAL_VAR_CHAR: //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(inString)c->setError(KVI_ERROR_UnexpectedEndInString,__tr("DICTIONARY KEY"));
						else c->setError(KVI_ERROR_UnexpectedEndOfCommand,__tr("DICTIONARY KEY"));
						return false;
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return true;
}

bool KviUserParser::processParameterIdentifier(KviCommand *c,KviStr &buffer)
{
	// This one expects c->m_ptr to point to a digit or '#'
	__range_valid(isdigit(*(c->m_ptr)) || ((*(c->m_ptr))=='#'));
	if(*(c->m_ptr) == '#'){
		c->getParamCount(buffer);
		++(c->m_ptr);
		return true;
	}
	int param = (*(c->m_ptr) - '0');
	++(c->m_ptr);
	while(isdigit(*(c->m_ptr)))
	{
		param = param * 10;
		param += (*(c->m_ptr) - '0');
		++(c->m_ptr);
	}
	if(*(c->m_ptr) == '-'){
		++(c->m_ptr);
		if(isdigit(*(c->m_ptr)))
		{
			int parEnd = (*(c->m_ptr) - '0');
			++(c->m_ptr);
			while(isdigit(*(c->m_ptr)))
			{
				parEnd = parEnd * 10;
				parEnd += (*(c->m_ptr) - '0');
				++(c->m_ptr);
			}
			c->getParam(param,parEnd,buffer);
		} else {
			if(*(c->m_ptr) == '>')
			{
				// ops!...was an object scope operator!
				--(c->m_ptr);
				c->getSingleParam(param,buffer);
			} else c->getParam(param,-1,buffer); //up to the end
		}
	} else c->getSingleParam(param,buffer);
	return true;
}

bool KviUserParser::processIdentifier(KviCommand *c,KviStr &buffer)
{
	// This one parses the identifier pointed by c->m_ptr
	// adds its value to buffer (or stops if the identifier is not valid)
	// then moves c->m_ptr to the characted immediately 
	// after the variable and returns it
	__range_valid(*c->m_ptr == '$');

//	debug("Processing identifier %s",c->m_ptr);

	const char *aux = ++(c->m_ptr);              // skip the $

	KviStr tmpBuffer;

	if(isdigit(*aux)||(*aux == '#'))
	{
		// $0,$1,$2...
		// $0 name of the command
		// $# number of params
		if(!processParameterIdentifier(c,tmpBuffer))return false;
	} else {
		if(*aux == '{')
		{
			// A command call
			c->m_ptr = aux;
			if(!skipCommand(c))return false;
			KviStr tmp(aux,c->m_ptr);
			KviCommand cmd(tmp.ptr(),c->m_wnd);
			if(!execCommandBuffer(&cmd)){
				if(cmd.hasError()){
					printError(&cmd,tmp.ptr());
					return false; // We could do better ?
				}
			}
			tmpBuffer.append(cmd.m_retBuffer);
		} else {
			// Normal identifier

			if(c->scopeObject())while(*aux && (isalnum(*aux)||(*aux == '.')||(*aux == '_')||(*aux == ':')))aux++; // run up to the end (first non alphanumeric)
			else while(*aux && (isalnum(*aux)||(*aux == '.')||(*aux == '_')))aux++;
			KviStr ident(c->m_ptr,aux - c->m_ptr);       // and get the ident name
			c->m_ptr = aux;

			bool bDone = false;

//			debug("Processing function call %s",ident.ptr());
//			if(c->scopeObject())debug("Object scope : %s",c->scopeObject()->getClass());
		
			if(ident.isEmpty()){
				tmpBuffer.append('$');  // single $...nope
//				while(iSkipClasses--)tmpBuffer.append(':');
			} else {                                       // have an identifier
				if(c->scopeObject())
				{
//					debug("Scope object for function call %s",ident.ptr());

					QList<KviStr> * pList = 0;
					if(*(c->m_ptr) == '(')
					{
//						debug("'('");
						// Have parameters
						pList = new QList<KviStr>;
						pList->setAutoDelete(true);
						KviStr tmp;
						++(c->m_ptr);
						// No scope object for function parameters
						KviScriptObject * scObj = c->scopeObject();
						c->setScopeObject(0);
						bool bRecognizedObjScopeOperator = c->recognizeObjectScopeOperator();
						c->setRecognizeObjectScopeOperator(true);
//						debug("Processing parameters of ident %s, resetting object scope to zero",ident.ptr());
						do
						{
							tmp = "";
							if(!processFncSingleParam(c,tmp))
							{
								delete pList;
								return false;
							}
							pList->append(new KviStr(tmp));
						} while(tmp.hasData());
						if(!processFncFinalPart(c,tmp))
						{
							delete pList;
							return false;
						}
						c->setScopeObject(scObj);
//						debug("Processed parameters of ident %s, reputting object scope to %s",ident.ptr(),scObj->getClass());
						c->setRecognizeObjectScopeOperator(bRecognizedObjScopeOperator);
					}
					int retVal;
					int idx = ident.findFirstIdx(':');
					KviStr classOverride;
//					debug("Calling %s with scope object %s,%s",ident.ptr(),c->scopeObject()->getName(),c->scopeObject()->id());
					if(idx != -1){
						// class override
						classOverride = ident.left(idx);
						ident.cutLeft(idx + 1);
						retVal = c->scopeObject()->callFunction(ident.ptr(),pList,tmpBuffer,classOverride.ptr());
					} else retVal = c->scopeObject()->callFunction(ident.ptr(),pList,tmpBuffer,0);
//					if(pList)delete pList; //<--- the list is owned by the KviCommand class
//					debug("callFunction %s returned %d %s",ident.ptr(),retVal,tmpBuffer.ptr());
					if(retVal != KVI_ERROR_Success)
					{
						KviStr er;
						if(classOverride.hasData())er.sprintf(__tr("function name: %s class override: %s"),ident.ptr(),classOverride.ptr());
						else er.sprintf(__tr("function name: %s"),ident.ptr());
						return c->setError(retVal,__tr("OBJECT FUNCTION CALL"),er.ptr());
					}
				} else {
					// not scope object
//					if(iSkipClasses)return c->setError(KVI_ERROR_NoAncestorClassToSkipTo,__tr("FUNCTION CALL (GLOBAL NAMESPACE)"),ident.ptr());

					if(*(c->m_ptr) == '('){
						++(c->m_ptr);
						// the ident is followed by parameters...it is a function
						ident.toUpper();
			
						KviFunctionEntry * fncTable;
						char init = *(ident.ptr());
			
						if(init < 'O')fncTable = ((init < 'H') ? fncTable_A_G : fncTable_H_N);
						else fncTable = ((init < 'T') ? fncTable_O_S : fncTable_T_Z);
			
						// evaluate it...
						for(int i = 0;fncTable[i].fncName;i++){
							if(kvi_strEqualCS(ident.ptr(),fncTable[i].fncName)){
								if(!(this->*(fncTable[i].fncProc))(c,tmpBuffer))return false;
								bDone = true;
								break;
							}
						}
#ifdef COMPILE_PLUGIN_SUPPORT
						if(!bDone){
							// Plugin function ?
							KviPluginFunctionHandler * h=g_pPluginManager->findFunctionHandler(ident.ptr());
							if(h)
							{
								if(!parsePluginFunction(h,c,tmpBuffer))return false;
								bDone = true;
							}
						}
#endif
			
						// Can't recover without parsing the function paramenters...
						if(!bDone)return c->setError(KVI_ERROR_UnknownFunction,__tr("FUNCTION"),ident.ptr());
					} else {
						ident.toUpper();
						// evaluate it...
						KviIdentifierEntry * idnTable;
						char init = *(ident.ptr());
			
						if(init < 'O')idnTable = ((init < 'H') ? idnTable_A_G : idnTable_H_N);
						else idnTable = ((init < 'T') ? idnTable_O_S : idnTable_T_Z);
			
						// evaluate it...
						for(int i = 0;idnTable[i].idnName;i++){
							if(kvi_strEqualCS(ident.ptr(),idnTable[i].idnName)){
								// will set the cmd->err on errors!
								if(!(this->*(idnTable[i].idnProc))(c,tmpBuffer))return false;
								bDone = true;
								break;
							}
						}
#ifdef COMPILE_PLUGIN_SUPPORT
						if(!bDone)
						{
							// Plugin function ?
							KviPluginFunctionHandler * h=g_pPluginManager->findFunctionHandler(ident.ptr());
							if(h)
							{
								if(!parsePluginIdentifier(h,c,tmpBuffer))return false;
								bDone = true;
							}
						}
#endif
						// Can recover from this
						if(!bDone)
						{
							if(g_pOptions->m_bPedanticParser)return c->setError(KVI_ERROR_UnknownIdentifier,__tr("IDENTIFIER"),ident.ptr());
							else return c->warning(__tr("IDENTIFIER : Unknown identifier (%s) : ignoring"),ident.ptr());
						}
					}
				}
			}
		}
	}

//	debug("No object function processed (%s), currently at %s",tmpBuffer.ptr(),c->m_ptr);

	if(c->recognizeObjectScopeOperator())
	{
		// Object scope ?
		if(*(c->m_ptr) == '-')
		{
			if(*(c->m_ptr + 1) == '>')
			{
				c->m_ptr += 2;
				if((*(c->m_ptr) == '%') || (*(c->m_ptr) == '$'))
				{
					if(tmpBuffer.hasData())
					{
						KviScriptObject * o = m_pScriptObjectController->findObjectById(tmpBuffer.ptr());
						if(o)return processObjectScopeRValue(o,c,buffer);
						else return c->setError(KVI_ERROR_ObjectNotFound,__tr("OBJECT OPERATOR -> (RVALUE)"),tmpBuffer.ptr());
					} else return c->setError(KVI_ERROR_ObjectNotFound,__tr("OBJECT_OPERATOR -> (RVALUE)"),__tr("The function/identifier call returned no object id"));
				} else return c->setError(KVI_ERROR_VariableOrIdentifierExpected,__tr("OBJECT OPERATOR -> (RVALUE)"),c->m_ptr);
			}
		}
	}

	buffer.append(tmpBuffer);

	return true;
}
	
bool KviUserParser::processCmdSingleToken(KviCommand *c)
{
	// This one must be called when c->m_ptr points
	// somewhere in a command buffer.
	// It extracts the first token encountered and
	// preprocesses it (subst variables, etc...)
	// then it appends everything to the c->m_buffer.
	// The parsing is stopped by a '\n',' ',';' or '\0'
	// Note that this function does nothing if c->m_ptr
	// points to the end of a command...

	// Skip leading spaces...even the escaped ones
	c->skipSpace();
	const char *aux_ptr = c->m_ptr;

	bool bInString = false;

	for(;;){
		// Now skip all the non interesting chars
		while(*aux_ptr && (*aux_ptr!='\\') && (*aux_ptr!='\n') &&
				(*aux_ptr!=' ') && (*aux_ptr!='\t') && (*aux_ptr!=';') &&
				(*aux_ptr!='$') && (*aux_ptr!='\"') && (*aux_ptr!=KVI_GLOBAL_VAR_CHAR))aux_ptr++;
		// Interesting char
		switch(*aux_ptr){
			case ' ' :
			case '\t':
			case ';' :
				if(!bInString){ // End of the token...append the last block to the buffer and return
					c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					return true;
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t')||
							(*aux_ptr == ';'))aux_ptr++; //In string...must skip it
			break;
			case '\n': // end of command
			case '\0': // end of command buffer...append the last block to the buffer and return
				if(bInString){
					// unsecaped newline or end of data
					c->setError(KVI_ERROR_UnexpectedEndInString,__tr("COMMAND-PARAMS"));
					return false;
				} // else it is the end of the command
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				return true;
			break;
			case '\"': // beginning or end of a string
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				bInString = !bInString;
				c->m_ptr = ++aux_ptr;
			break;
			case KVI_GLOBAL_VAR_CHAR: //variable: append the last block to the buffer and process the var
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processVariable(c,c->m_buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processIdentifier(c,c->m_buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(bInString){
							c->setError(KVI_ERROR_UnexpectedEndInString,__tr("COMMAND-PARAMS"));
							return false;
						}
						return true;
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					default: // Must be copied to the buffer without modification
						c->m_buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return true;
}

bool KviUserParser::processCmdFinalPart(KviCommand *c)
{
	// Preprocesses the final part of a standard command...
	// It is stopped by a newline, a ';' or the end of the buffer.
	// This one extracts the command parameters, evaluates it,
	// and appends everything to cmd->buffer.
	// skip spaces...but keep the terminators!!!

	c->skipSpace();
	const char *aux_ptr = c->m_ptr;

	bool bInString = false;

	for(;;){
		// Now skip all the non interesting chars
		while(*aux_ptr && (*aux_ptr != '\\') && (*aux_ptr != '\n') && (*aux_ptr != '\t') &&
				(*aux_ptr != ';') && (*aux_ptr != '$') && (*aux_ptr != ' ') &&
				(*aux_ptr != KVI_GLOBAL_VAR_CHAR) && (*aux_ptr != '\"'))++aux_ptr;
		// Interesting char
		switch(*aux_ptr){
			case '\0': // end of command buffer...append the last block to the buffer and return
			case '\n':
				if(bInString){
					// ops....end of buffer while looking for matching '"'
					c->setError(KVI_ERROR_UnexpectedEndInString,__tr("COMMAND-PARAMS"));
					return false;
				}
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				if(*aux_ptr == '\n')aux_ptr++; // point after the semicolon
				c->m_ptr = aux_ptr;
				return true;
			break;
			case '\"': // beginning or end of a string
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				bInString = !bInString;
				c->m_ptr = ++aux_ptr; //don't forget to point to next char
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!bInString){
					// append the last part
					c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != '\0') && (*aux_ptr != '\n') && (*aux_ptr != ';'))c->m_buffer.append(' ');
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case ';' :
				if(!bInString){ // end of command...append the last block to the buffer and return
					c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr; //don't forget to point to next char
					return true;
				} else ++aux_ptr; // just skip it
			break;
			case KVI_GLOBAL_VAR_CHAR: //variable: append the last block to the buffer and process the var
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processVariable(c,c->m_buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processIdentifier(c,c->m_buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(bInString){
							c->setError(KVI_ERROR_UnexpectedEndInString,__tr("COMMAND-PARAMS"));
							return false;
						} else return true;
					break; //escaped nothing...
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					default: // Must be copied to the buffer without modification
						c->m_buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	return true;
}


bool KviUserParser::processCmdFinalPartNoIdent(KviCommand *c)
{
	// Preprocesses the final part of a standard command...
	// It is stopped by a newline, a ';' or the end of the buffer.
	// This one extracts the command parameters, does not evaluate it,
	// appends everything to cmd->buffer.

	c->skipSpace();
	const char *aux_ptr = c->m_ptr;

	bool bInString = false;

	for(;;){
		// Now skip all the non interesting chars
		while(*aux_ptr && (*aux_ptr != '\\') && (*aux_ptr != '\n') && (*aux_ptr != '\t') &&
				(*aux_ptr != ';') && (*aux_ptr != ' ') && (*aux_ptr != '\"'))++aux_ptr;
		// Interesting char
		switch(*aux_ptr){
			case '\0': // end of command buffer...append the last block to the buffer and return
			case '\n':
				if(bInString){
					// ops....end of buffer while looking for matching '"'
					c->setError(KVI_ERROR_UnexpectedEndInString,__tr("COMMAND-PARAMS"));
					return false;
				}
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				if(*aux_ptr == '\n')aux_ptr++; //point after the semicolon
				c->m_ptr = aux_ptr;
				return true;
			break;
			case '\"': // beginning or end of a string
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				bInString = !bInString;
				c->m_ptr = ++aux_ptr; //don't forget to point to next char
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!bInString){
					// append the last part
					c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != '\0') && (*aux_ptr != '\n') && (*aux_ptr != ';'))c->m_buffer.append(' ');
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case ';' :
				if(!bInString){ // end of command...append the last block to the buffer and return
					c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr; //don't forget to point to next char
					return true;
				} else ++aux_ptr; // just skip it
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				c->m_buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(bInString){
							c->setError(KVI_ERROR_UnexpectedEndInString,__tr("COMMAND-PARAMS"));
							return false;
						} else return true;
					break; //escaped nothing...
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					default: // Must be copied to the buffer without modification
						c->m_buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	return true;
}

bool KviUserParser::processString(const char *ptr,KviWindow *pWnd,KviStr &buffer)
{
	KviCommand *cmd = new KviCommand(ptr,pWnd);
	if(!processCmdFinalPart(cmd))return false;
	buffer = cmd->m_buffer.ptr();
	return true;
}

//===========================================================================//
//
// Process function parameters
// The following three functions process parameters of functions:
// blocks of data enclosed in () parentheses.
// All expect c->m_ptr to point somewhere inside that block, so after the first
// parenthesis and before (or directly to) the ending one.
//

bool KviUserParser::processFncSingleParam(KviCommand *c,KviStr &buffer)
{
	// This one will extract the first parameter of the function
	// and if it is followed by ',' it will skip it.
	// It will not skip the ending ')' so the caller will be able
	// to know if we reached the end...
	// This is different in processFncFinalPart
	//that will skip the ending ')' too

	if(!processFncSingleParamInternal(c,buffer))return false;
	__range_valid((*(c->m_ptr)==',')||(*(c->m_ptr)==')'));
	if(*(c->m_ptr)==',')++(c->m_ptr);
	return true;
}

bool KviUserParser::processFncSingleParamInternal(KviCommand *c,KviStr &buffer)
{
	// This one must be called when c->m_ptr points
	// somewhere in a function parameters block.
	// It extracts the first parameter encountered and
	// preprocesses it (subst variables, etc...)
	// then it appends everything to the buffer
	// The parsing is stopped by a '\n', ',', ')' or '\0'
	// on '\n' and '\0' an error is returned
	// The parsing is NOT stopped if one of ',' or ')'
	// is in a string (delimitated by two '"' chars)
	// Note that this function does nothing if c->m_ptr
	// points to the end of the param buffer...
	// (eg. c->m_ptr comes from previous calls to processFncSingleParam())
	// It will also skip matching parenthesis :
	// so () is a valid parameter
	// It will return false in case of an error ;
	// If it returns true, c->m_ptr surely points to a ',' or a ')'
	// This function EATS leading and trailing spaces and tabs
	// and reduces the middle ones to one per group.
	// A param like this:
	//          string    "full  of  words"  and  spaces      ,
	// Becomes like this:
	// string full  of  words and spaces,

	// Skip leading spaces and tabs (the escaped ones too)
	c->skipSpace();
	register const char *aux_ptr = c->m_ptr;

	int parLevel       = 0;
	bool inString      = false;

	for(;;){
		// First skip all the non interesting chars
		while(	*aux_ptr         && (*aux_ptr!=KVI_GLOBAL_VAR_CHAR) &&
				(*aux_ptr!='\n') && (*aux_ptr!=')')  && 
				(*aux_ptr!=',')  && (*aux_ptr!='$')  && 
				(*aux_ptr!=' ')  && (*aux_ptr!='(')  &&
				(*aux_ptr!='"')  && (*aux_ptr!='\\') &&
				(*aux_ptr!='\t'))aux_ptr++;
		// Interesting char
		switch(*aux_ptr){
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				if(inString)c->setError(KVI_ERROR_UnexpectedEndInString,__tr("FUNCTION-PARAMS"));
				else c->setError(KVI_ERROR_UnexpectedEndOfCommand,__tr("FUNCTION-PARAMS"));
				return false;
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!inString){
					// append the last part
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != ')') && (*aux_ptr != ','))buffer.append(' ');
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case '"': // Not escaped '"'...enter or exit the string (skip the '"')
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				inString = !inString;
			break;
			case '(': // Keep track of matching parenthesis (if not in string)
				++aux_ptr;
				if(!inString)++parLevel;
			break;
			case ')': // If not in string, exit if parenthesis level is 0, otherwise decrease it
				if(!inString){
					if(parLevel == 0){
						buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = aux_ptr;
						return true;
					} else --parLevel;
				}
				++aux_ptr;
			break;
			case ',': // end of command buffer...append the last block to the buffer and return
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				if(!inString){
					c->m_ptr = aux_ptr;
					return true;
				}
				c->m_ptr = aux_ptr++; //in string...',' must be included
			break;
			case KVI_GLOBAL_VAR_CHAR: //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(inString)c->setError(KVI_ERROR_UnexpectedEndInString,__tr("FUNCTION-PARAMS"));
						else c->setError(KVI_ERROR_UnexpectedEndOfCommand,__tr("FUNCTION-PARAMS"));
						return false;
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return true;
}

/*
bool KviUserParser::processFncSingleParamInternal(KviCommand *c,KviStr &buffer)
{
	// This one must be called when c->m_ptr points
	// somewhere in a function parameters block.
	// It extracts the first parameter encountered and
	// preprocesses it (subst variables, etc...)
	// then it appends everything to the buffer
	// The parsing is stopped by a '\n', ',' , ')' or '\0'
	// on '\n' and '\0' an error is returned
	// The parsing is NOT stopped if one of ',' or ')'
	// is in a string (delimitated by two '"' chars)
	// Note that this function does nothing if c->m_ptr
	// points to the end of the param buffer...
	// (eg. c->m_ptr comes from previous calls to processFncSingleParam())
	// It will also skip matching parenthesis :
	// so () is a valid parameter
	// It will return false in case of an error ;
	// If it returns true, c->m_ptr surely points to a ',' or a ')'

	// Skip leading spaces and tabs (no newline allowed)
	while(*(c->m_ptr) && ((*(c->m_ptr) == ' ')||(*(c->m_ptr) == '\t')))++(c->m_ptr);
	const char *aux_ptr = c->m_ptr;

	int parLevel = 0;
	bool inString = false;

	for(;;){
		// Now skip all the non interesting chars
		while(	*aux_ptr         && (*aux_ptr!=KVI_GLOBAL_VAR_CHAR) &&
				(*aux_ptr!='\n') && (*aux_ptr!=')') && 
				(*aux_ptr!=',')  && (*aux_ptr!='$') &&
				(*aux_ptr!='(')  && (*aux_ptr!='"') &&
				(*aux_ptr!='\\'))aux_ptr++;
		// Interesting char
		switch(*aux_ptr){
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				if(inString)c->setError(KVI_ERROR_UnexpectedEndInString,__tr("FUNCTION-PARAMS"));
				else c->setError(KVI_ERROR_UnexpectedEndOfCommand,__tr("FUNCTION-PARAMS"));
				return false;
			break;
			case '"': // Not escaped '"'...enter or exit the string (skip the '"')
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				inString = !inString;
			break;
			case '(': // Keep track of matching parenthesis (if not in string)
				++aux_ptr;
				if(!inString)++parLevel;
			break;
			case ')': // If not in string, exit if parenthesis level is 0, otherwise decrease it
				if(!inString){
					if(parLevel == 0){
						buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = aux_ptr;
						return true;
					} else --parLevel;
				}
				++aux_ptr;
			break;
			case ',': // end of command buffer...append the last block to the buffer and return
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				if(!inString){
					c->m_ptr = aux_ptr;
					return true;
				}
				c->m_ptr = aux_ptr++; //in string...',' must be included
			break;
			case KVI_GLOBAL_VAR_CHAR: //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(inString)c->setError(KVI_ERROR_UnexpectedEndInString,_i18n_("FUNCTION-PARAMS"));
						else c->setError(KVI_ERROR_UnexpectedEndOfCommand,_i18n_("FUNCTION-PARAMS"));
						return false;
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return true;
}
*/

bool KviUserParser::processFncFinalPart(KviCommand *c,KviStr &buffer)
{
	// Preprocesses the final part of a function parameters block...
	// It is stopped by a newline, a ')' or the end of the buffer.
	// Extracts the command parameters, evaluates it, 
	// and appends everything to the buffer.
	// If the return value is true, c->m_ptr points to the character
	// after the closing ')'.

	for(;;){
		if(!processFncSingleParamInternal(c,buffer))return false;
		__range_valid((*(c->m_ptr)==',')||(*(c->m_ptr)==')'));
		if(*(c->m_ptr) == ','){
			++(c->m_ptr);
			buffer.append(',');
		} else {
			++(c->m_ptr);
			return true;
		}
	}
	__range_invalid(true);
	return false; //newer here
}

//===========================================================================//
//
// Skip parts of command buffer.
// Used in 'if', 'while' and 'do' parsing...
//

bool KviUserParser::skipCommand(KviCommand *c)
{
	if(*(c->m_ptr) == '{'){
		if(!skipCommandBlock(c))return false;
		// Here c->m_ptr points to the char immediately after the closing brace
	} else {
		if(!skipSingleCommand(c))return false;
		// Here c->m_ptr points to the char immediately after the separator...
	}
	return true;
}
bool KviUserParser::skipCommandBlock(KviCommand *c)
{
	__range_valid(*(c->m_ptr)=='{');
	++(c->m_ptr);
	c->skipWhitespace();
	while(*(c->m_ptr)){
		switch(*(c->m_ptr)){
			case '}': ++(c->m_ptr);            return true;  break;
			case '{': if(!skipCommandBlock(c)) return false; break;
			default : if(!skipSingleCommand(c))return false; break;
		}
		c->skipWhitespace();
	}
	// We should newer reach this point
	c->setError(KVI_ERROR_MissingClosingBrace,_i18n_("COMMAND-BLOCK"));
	return false;
}

bool KviUserParser::skipSingleCommand(KviCommand *c)
{
	while(*(c->m_ptr) && ((*(c->m_ptr) == ' ')||(*(c->m_ptr) == '\t')))++(c->m_ptr);
	// an empty command ?
	if(*(c->m_ptr) == ';'){
		++(c->m_ptr);
		return true;
	}
	if((*(c->m_ptr) == KVI_GLOBAL_VAR_CHAR) || (*(c->m_ptr) == '$'))return skipLValue(c);
	// a comment ?
	if(*(c->m_ptr) == KVI_COMMENT_BEGIN_CHAR)return skipComment(c);
	// extract the command name
	const char *aux = c->m_ptr;
	// command names contain only ascii chars, digits, _ and .
	while(*(c->m_ptr) && (isalnum(*(c->m_ptr)) || (*(c->m_ptr)== '_') || (*(c->m_ptr)=='.')))++(c->m_ptr);
	KviStr tmp(aux,c->m_ptr);

	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"IF"))return skipIf(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"WHILE"))return skipWhile(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"DO"))return skipDo(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"ALIAS"))return skipAlias(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"SWITCH"))return skipSwitch(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"DIALOG"))return skipDialog(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"POPUP"))return skipPopup(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"FOREACH"))return skipForeach(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"TIMER"))return skipTimer(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"EXECV"))return skipExecv(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"AWHOIS"))return skipAwhois(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"UTOOLBAR"))return skipUtoolbar(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"TRY"))return skipTry(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"OBJ_SETEVENT"))return skipObj_setEvent(c);
	if(kvi_strEqualNoLocaleCI(tmp.ptr(),"CLASS"))return skipClass(c);
	
	// skip normal command
	if(!skipSwitches(c))return false;

	for(;;){
		// Now skip all the non interesting chars
		while(*(c->m_ptr) && (*(c->m_ptr) != '\\') && (*(c->m_ptr) != '\n') && (*(c->m_ptr) != KVI_GLOBAL_VAR_CHAR) &&
				(*(c->m_ptr) != ';') && (*(c->m_ptr) != '$'))++(c->m_ptr);
		// Interesting char
		switch(*(c->m_ptr)){
			case '\0': return true; break;
			case '\n': // end of command
			case ';' : ++(c->m_ptr); return true; break;
			case '$' : if(!skipIdentifier(c))return false; break;
			case KVI_GLOBAL_VAR_CHAR : if(!skipVariable(c))return false; break;
			case '\\': //escape character: append the last block to the processed buffer...
				++(c->m_ptr);
				switch(*(c->m_ptr)){
					case '\0': return true; break; //escaped nothing...
					default: ++(c->m_ptr); break;
				}
			break;
		}
	}
	return true;
}

bool KviUserParser::skipExpressionBody(KviCommand *c)
{
	// skips a thing like '( <expression like for calc> )'
	__range_valid(*(c->m_ptr) == '(');

	++(c->m_ptr);

	int parLevel = 0;
	bool inString =false;

	for(;;){
		// Now skip all the non interesting chars
		while(*(c->m_ptr) && (*(c->m_ptr) != '\\') && (*(c->m_ptr) != KVI_GLOBAL_VAR_CHAR) &&
			 (*(c->m_ptr) != '\n') && (*(c->m_ptr) != '(') &&
			 (*(c->m_ptr) != ')') && (*(c->m_ptr) != '$') &&
			 (*(c->m_ptr) != '"'))(c->m_ptr)++;
		// Interesting char
		switch(*(c->m_ptr)){
			case '\0':
			case '\n': // error...
				c->setError(KVI_ERROR_UnexpectedEndOfCommand,_i18n_("EXPRESSION-BODY"));
				return false;
			break;
			case '"': ++(c->m_ptr); inString = !inString; break;
			case '(': ++(c->m_ptr); if(!inString)++parLevel; break;
			case ')' :
				if(!inString){
					if(parLevel == 0){ ++(c->m_ptr); return true; }
					else{ ++(c->m_ptr); --parLevel; }
				} else ++(c->m_ptr);
			break;
			case KVI_GLOBAL_VAR_CHAR: if(!skipVariable(c))return false; break;
			case '$': if(!skipIdentifier(c))return false; break;
			case '\\': //escape character
				++(c->m_ptr);
				switch(*(c->m_ptr)){
					case '\0':
						c->setError(KVI_ERROR_UnexpectedEndOfCommand,_i18n_("EXPRESSION-BODY"));
						return false;
					default: ++(c->m_ptr); break;
				}
			break;
		}
	}
	// newer here...
	return false;
}

bool KviUserParser::skipIdentifier(KviCommand *c)
{
	__range_valid(*c->m_ptr == '$');
	++(c->m_ptr);              // skip the $

	if(isdigit(*(c->m_ptr))||(*(c->m_ptr) == '#'))c->m_ptr ++;
	else {
		if(*(c->m_ptr) == '{'){
			if(!skipCommand(c))return false;
		} else {
			// Normal identifier
			while(*(c->m_ptr) && (isalnum(*(c->m_ptr))||(*(c->m_ptr) == '.')||(*(c->m_ptr) == '_')||(*(c->m_ptr) == ':')))(c->m_ptr)++; // run up to the end (first non alphanumeric)

			if(*(c->m_ptr) == '(')
			{
				bool bRecognizedObjScopeOperator = c->recognizeObjectScopeOperator();
				c->setRecognizeObjectScopeOperator(true);
				if(!skipExpressionBody(c))return false;
				c->setRecognizeObjectScopeOperator(bRecognizedObjScopeOperator);
			}
		}
	}

	if(c->recognizeObjectScopeOperator())
	{
		// Object scope ?
		if(*(c->m_ptr) == '-')
		{
			if(*(c->m_ptr + 1) == '>')
			{
				c->m_ptr += 2;
				if((*(c->m_ptr) == '%') || (*(c->m_ptr) == '$'))return skipObjectScopeRValue(c);
				else return c->setError(KVI_ERROR_VariableOrIdentifierExpected,__tr("OBJECT OPERATOR -> (RVALUE)"),c->m_ptr);
			}
		}
	}
	return true;
}
bool KviUserParser::skipObjectScopeRValue(KviCommand *c)
{
	switch(*(c->m_ptr))
	{
		case '%':
			// Object variable
			if(!skipVariable(c))return false;
		break;
		case '$':
			// Object function or identifier call
			if(!skipIdentifier(c))return false;
		break;
		default:
			// Invalid !
			//debug("Invalid char in object scope %s, check the code man!",c->m_ptr);
			return false;
		break;
	}
	return true;
}

bool KviUserParser::skipVariable(KviCommand *c)
{
	__range_valid(*c->m_ptr == KVI_GLOBAL_VAR_CHAR);
	++(c->m_ptr);

	while(*(c->m_ptr) && (isalnum(*(c->m_ptr))||(*(c->m_ptr) == '.')||(*(c->m_ptr) == '_')))(c->m_ptr)++;

	if(*(c->m_ptr) == '[')
	{
		(c->m_ptr)++;
		if(!skipVariableKey(c))return false;
	}

	// Object scope ?
	if(*(c->m_ptr) == '-')
	{
		if(*(c->m_ptr + 1) == '>')
		{
			c->m_ptr += 2;
			if((*(c->m_ptr) == '%') || (*(c->m_ptr) == '$'))return skipObjectScopeRValue(c);
			else c->setError(KVI_ERROR_VariableOrIdentifierExpected,__tr("OBJECT OPERATOR -> (RVALUE)"),c->m_ptr);
		}
	}
	return true;
}

bool KviUserParser::skipVariableKey(KviCommand *c)
{
	c->skipSpace();
	bool inString      = false;
	for(;;){
		// First skip all the non interesting chars
		while(	*(c->m_ptr)         && (*(c->m_ptr)!=KVI_GLOBAL_VAR_CHAR) &&
				(*(c->m_ptr)!='\n') && (*(c->m_ptr)!=']')  && (*(c->m_ptr)!='$')  && 
				(*(c->m_ptr)!='"')  && (*(c->m_ptr)!='\\'))(c->m_ptr)++;
		// Interesting char
		switch(*(c->m_ptr)){
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				if(inString)c->setError(KVI_ERROR_UnexpectedEndInString,__tr("DICTIONARY KEY"));
				else c->setError(KVI_ERROR_UnexpectedEndOfCommand,__tr("DICTIONARY KEY"));
				return false;
			break;
			case '"': c->m_ptr++; inString = !inString; break;
			case ']': c->m_ptr++; if(!inString)return true; break;
			case KVI_GLOBAL_VAR_CHAR: if(!skipVariable(c))return false; break;
			case '$': if(!skipIdentifier(c))return false; break;
			case '\\': //escape character: append the last block to the processed buffer...
				c->m_ptr++;
				switch(*(c->m_ptr)){
					case '\0':
						if(inString)c->setError(KVI_ERROR_UnexpectedEndInString,__tr("DICTIONARY KEY"));
						else c->setError(KVI_ERROR_UnexpectedEndOfCommand,__tr("DICTIONARY KEY"));
						return false;
					break;
					case '\n':  //escaped newline
						c->m_ptr++; //skip it
						c->skipSpace();       //skip leading spaces
					break;
					default: c->m_ptr++; break;
				}
			break;
		}
	}
	// Newer arrive here...
	return true;
}

bool KviUserParser::skipSwitches(KviCommand *c)
{
	// This one 
	c->skipSpace();
	while(*(c->m_ptr) == '-'){
		if(!skipSingleSwitch(c))return false;
		c->skipSpace();
	}
	return true;
}

bool KviUserParser::skipSingleSwitch(KviCommand *c)
{
	__range_valid(*(c->m_ptr) == '-');
	++(c->m_ptr); // skip the - char
	// now a LETTER is required
	if( (!(*(c->m_ptr))) || (!isalpha(*(c->m_ptr))) ){
		c->setError(KVI_ERROR_InvalidOption,__tr("SWITCH PARSING AFTER '-' DELIMITER"),c->m_ptr);
		return false;
	}
	++(c->m_ptr); // skip the switch letter
	c->skipSpace();

	if(*(c->m_ptr) == '='){
		// a token follows the switch
		++(c->m_ptr);
		c->skipSpace();
		const char *aux_ptr = c->m_ptr;
		bool bInString = false;
		for(;;){
			// Now skip all the non interesting chars
			while(*aux_ptr && (*aux_ptr!='\\') && (*aux_ptr!='\n') &&
					(*aux_ptr!=' ') && (*aux_ptr!='\t') && (*aux_ptr!=';') &&
					(*aux_ptr!='$') && (*aux_ptr!='\"') && (*aux_ptr!=KVI_GLOBAL_VAR_CHAR))aux_ptr++;
			// Interesting char
			switch(*aux_ptr){
				case ' ' :
				case '\t':
				case ';' :
					if(!bInString){ // End of the token...append the last block to the buffer and return
						c->m_ptr = aux_ptr;
						return true;
					} else while((*aux_ptr == ' ')||(*aux_ptr == '\t')||
								(*aux_ptr == ';'))aux_ptr++; //In string...must skip it
				break;
				case '\n': // end of command
				case '\0': // end of command buffer...append the last block to the buffer and return
					if(bInString){
						// unsecaped newline or end of data
						return c->setError(KVI_ERROR_UnexpectedEndInString,__tr("SWITCH PARSING: PARAMETER EXTRACTION"));
					} // else it is the end of the command
					c->m_ptr = aux_ptr;
					return true;
				break;
				case '\"': // beginning or end of a string
					bInString = !bInString;
					c->m_ptr = ++aux_ptr;
				break;
				case KVI_GLOBAL_VAR_CHAR: //variable: append the last block to the buffer and process the var
					c->m_ptr = aux_ptr;
					if(!skipVariable(c))return false;
					aux_ptr = c->m_ptr;
				break;
				case '$': //system identifier: append the last block to the buffer and process the ident
					c->m_ptr = aux_ptr;
					if(!skipIdentifier(c))return false;
					aux_ptr = c->m_ptr;
				break;
				case '\\': //escape character: append the last block to the processed buffer...
					c->m_ptr = ++aux_ptr;
					switch(*aux_ptr){
						case '\0':
							if(bInString)return c->setError(KVI_ERROR_UnexpectedEndInString,__tr("SWITCH PARSING: PARAMETER EXTRACTION"));
							return true;
						break;
						case '\n':  //escaped newline
							c->m_ptr = ++aux_ptr; //skip it
							c->skipSpace();       //skip leading spaces
							aux_ptr = c->m_ptr;   //continue
						break;
						default: // Must be copied to the buffer without modification
							c->m_ptr = ++aux_ptr;
						break;
					}
				break;
			}
		}
	}

	// not '='
	return true;
}

bool KviUserParser::skipComment(KviCommand *c)
{
	__range_valid(*(c->m_ptr) == KVI_COMMENT_BEGIN_CHAR);
	while(*(c->m_ptr) && (*(c->m_ptr) != ';') && (*(c->m_ptr) != '\n'))++(c->m_ptr);
	if(*(c->m_ptr))++(c->m_ptr); //skip the terminator
	return true;
}

#include "m_kvi_uparser.moc"
