//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_settings.h"

#include "kvi_scriptmisc.h"

#include "kvi_error.h"
#include "kvi_debug.h"
#include "kvi_malloc.h"


/////////////////////////////////////////////////////////////////////////////////////////////////
///////  DNS CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: dns
	@short:
		A class for async dns lookups
	@inherits:
		[class]object[/class]<br>

	@functions:
		!fn: $resolve(&lt;hostname&gt;)
		This function will start the lookup for the provided &lt;hostname&gt;.<br>
		If the class is in the ipv6 mode ,the lookup will be attempted by using the IpV6
		protocol , othervise the standard ipv4 mode will be used.<br>
		This function returns '1' if succesfull , or '0' in case of a failure.<br>
		This function can fail if the hostname is not valid or if there is already
		another running lookup.<br>

		!fn: $setMode(&lt;mode&gt;)
		Sets the Ip protocol version type.<br>
		&lt;mode&gt; can be "ipv6" or "ipv4".<br>
		The default mode of this class is ipv4 , and that mode is always working.<br>
		If kvirc has been compiled with the IpV6 support, the ipv6 mode will be also
		working.<br>
		This function returns '1' if succesfull , '0' otherwise.<br>

		!fn: $isRunning()
		This function returns '1' if a hostname lookup is currently in progress, '0' otherwise.<br>

		!fn: $mode()
		Returns the string "ipv6" if the object is in the ipv6 mode , the string "ipv4" otherwise.<br>

		!fn: $abort()
		Aborts a running dns lookup.<br>
		Returns 1 if the object was in "lookup" state , '0' otherwise.<br>

		!fn: $query()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the string containing the hostname that was passed to the [classfnc:dns]$resolve[/classfnc]() function

		!fn: $hostname()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the hostname of the resolved host.<br>
		Please note that this may be not the same as returned by [classfnc:dns]$query[/classfnc]().<br>

		!fn: $ip()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the ip address of the resolved host.<br>

		!fn: $alias1()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the alias hostname of the resolved host (if available).<br>

		!fn: $alias2()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		It returns the second alias hostname of the resolved host (if available).<br>

		!fn: $success()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		Returns '1' if the last lookup was succesfull , '0' otherwise (and in that
		case the last error can be retrieved by using [classfnc:dns]$lastError[/classfnc].<br>

		!fn: $lastError()
		This function returns a valid value only when the OnDnsDone event has been
		triggered and no next lookup has been started yet (Eg : inside the OnDnsDone
		event handler or in the handlers of the "done" signal).<br>
		Returns the error string in case of an unsuccesfull lookup.<br>

	@events:
		!ev: OnDnsDone()
		Triggered when a host has been resolved.<br>
		<b>This event has a default event handler that emits the "done" signal.</b><br>
		If you provide a new handler for this event , and still want the signal to be emitted
		you have to emit the signal by yourself.<br>
		<example>
		[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc](done)
		</example>

	@signals:
		!sg: done()
		Emitted from the default OnDnsDone event handler.<br>

	@description:
		A class that implements asynchronous dns lookups.<br>
		It can lookup the IpV4 hostnames and the IpV6 ones if kvirc has been
		compiled with such support.<br>
		Basically , you create a dns object instance and either override the [classevent:dns]OnDnsDone[/classevent]
		event or connect the [classsignal:dns]done[/classsignal] signal:
		then call the [classfnc:dns]$resolve[/classfnc]() function.<br>
		In the event handler or signal slot you retrieve the information
		by calling the [classfnc:dns]$ip[/classfnc]() , [classfnc:dns]$hostname[/classfnc](),
		[classfnc:dns]$query[/classfnc]() , [classfnc:dns]$success[/classfnc]() ,
		[classfnc:dns]$lastError[/classfnc]() , [classfnc:dns]$alias1[/classfnc]() and
		[classfnc:dns]$alias2[/classfnc]() functions.<br>
		The object can resolve only one hostname at once , but can be used
		multiple times.<br>
		A lookup can be aborted by using the [classfnc:dns]$abort[/classfnc]() function.<br>

	@examples:
		Simple example:<br>
		A standalone resolver object: we define a subclass of dns that is useful for experiments.<br>
		<example>
		[cmd]class[/cmd](mydns,dns)
		{
		&nbsp;	[cmd:class]function[/cmd](constructor)
		&nbsp;	{
		&nbsp;		[cmd]connect[/cmd] [fnc]$this[/fnc] done [fnc]$this[/fnc] hostResolved
		&nbsp;	}
		&nbsp;	[cmd:class]function[/cmd](hostResolved)
		&nbsp;	{
		&nbsp;		[cmd]echo[/cmd] Success ? : [fnc]$this[/fnc]->[classfnc:dns]$success[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Last error : [fnc]$this[/fnc]->[classfnc:dns]$lastError[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Query : [fnc]$this[/fnc]->[classfnc:dns]$query[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Hostname : [fnc]$this[/fnc]->[classfnc:dns]$hostname[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Ip : [fnc]$this[/fnc]->[classfnc:dns]$ip[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Alias1 : [fnc]$this[/fnc]->[classfnc:dns]$alias1[/classfnc]()
		&nbsp;		[cmd]echo[/cmd] Alias2 : [fnc]$this[/fnc]->[classfnc:dns]$alias2[/classfnc]()
		&nbsp;	}
		}
		</example>
		Copy the example above to the code tester in the script dialog and execute it once.<br>
		You will have the "mydns" class defined.<br>
		Now you can create an instance of this class by typing in the console:<br>
		<example>
		%X = [fnc]$new[/fnc](mydns,[fnc]$root[/fnc],mytestdns)
		</example>
		The object id in the variable %X now points to the instance.<br>
		Now you are able to resolve hosts by typing:<br>
		<example>
		%X->[classfnc:dns]$resolve[/classfnc](&lt;hostname_here&gt;)
		</example>
		The result will be "echoed" to the console window.<br>
		Try it with valid and invalid hostnames , and also with ip addresses.<br>
		You can also test the $abort function and the return values from each call.<br>
	@seealso:
		class [class]object[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptDns::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("resolve",(scriptObjectFunction)&KviScriptDns::builtinFunction_RESOLVE);
	d->addBuiltinFunction("isRunning",(scriptObjectFunction)&KviScriptDns::builtinFunction_ISRUNNING);
	d->addBuiltinFunction("setMode",(scriptObjectFunction)&KviScriptDns::builtinFunction_SETMODE);
	d->addBuiltinFunction("abort",(scriptObjectFunction)&KviScriptDns::builtinFunction_ABORT);
	d->addBuiltinFunction("query",(scriptObjectFunction)&KviScriptDns::builtinFunction_QUERY);
	d->addBuiltinFunction("hostname",(scriptObjectFunction)&KviScriptDns::builtinFunction_HOSTNAME);
	d->addBuiltinFunction("ip",(scriptObjectFunction)&KviScriptDns::builtinFunction_IP);
	d->addBuiltinFunction("alias1",(scriptObjectFunction)&KviScriptDns::builtinFunction_ALIAS1);
	d->addBuiltinFunction("alias2",(scriptObjectFunction)&KviScriptDns::builtinFunction_ALIAS2);
	d->addBuiltinFunction("success",(scriptObjectFunction)&KviScriptDns::builtinFunction_SUCCESS);
	d->addBuiltinFunction("lastError",(scriptObjectFunction)&KviScriptDns::builtinFunction_LASTERROR);
	d->addBuiltinFunction("mode",(scriptObjectFunction)&KviScriptDns::builtinFunction_MODE);

	KviScriptEventStruct * s = new KviScriptEventStruct;
	s->szName = "OnDnsDone";
	s->szBuffer = "$this->$emit(done)";
	d->addDefaultEvent(s);
}

KviScriptDns::KviScriptDns(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptObject(cntrl,p,name,pDef)
{
	m_pDns = new KviAsyncDns();
	connect(m_pDns,SIGNAL(dnsFinished(KviDnsStruct *)),this,SLOT(dnsFinished(KviDnsStruct *)));
	m_bIpV6 = false;
	m_pDnsStruct = 0;
}

KviScriptDns::~KviScriptDns()
{
	delete m_pDns;
	if(m_pDnsStruct)delete m_pDnsStruct;
}

void KviScriptDns::dnsFinished(KviDnsStruct * dns)
{
	__range_valid(m_pDnsStruct == 0);
	m_pDnsStruct = m_pDns->forgetDnsStruct();
	KviStr params;
	triggerEvent("OnDnsDone",params);
}


int KviScriptDns::builtinFunction_RESOLVE(QList<KviStr> * params,KviStr &buffer)
{
	if(m_pDns->isRunning())
	{
		buffer.append('0');
		return KVI_ERROR_Success;
	}
	if(m_pDnsStruct)
	{
		delete m_pDnsStruct;
		m_pDnsStruct = 0;
	}
	if(!params)return KVI_ERROR_MissingParameter;
	KviStr * pStr = params->first();
	if(!pStr)return KVI_ERROR_MissingParameter;
	if(!m_pDns->resolve(pStr->ptr(),0,m_bIpV6))buffer.append('0');
	else buffer.append('1');
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_ISRUNNING(QList<KviStr> * params,KviStr &buffer)
{
	buffer.append(m_pDns->isRunning() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_MODE(QList<KviStr> * params,KviStr &buffer)
{
	buffer.append(m_bIpV6 ? "ipv6" : "ipv4");
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_QUERY(QList<KviStr> * params,KviStr &buffer)
{
	if(m_pDnsStruct)buffer.append(m_pDnsStruct->szHostToResolve);
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_HOSTNAME(QList<KviStr> * params,KviStr &buffer)
{
	if(m_pDnsStruct)buffer.append(m_pDnsStruct->szHostname);
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_IP(QList<KviStr> * params,KviStr &buffer)
{
	if(m_pDnsStruct)buffer.append(m_pDnsStruct->szIp);
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_ALIAS1(QList<KviStr> * params,KviStr &buffer)
{
	if(m_pDnsStruct){
		if(*(m_pDnsStruct->szAlias1.ptr()) != '*')buffer.append(m_pDnsStruct->szAlias1);
	}
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_ALIAS2(QList<KviStr> * params,KviStr &buffer)
{
	if(m_pDnsStruct){
		if(*(m_pDnsStruct->szAlias2.ptr()) != '*')buffer.append(m_pDnsStruct->szAlias2);
	}
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_SUCCESS(QList<KviStr> * params,KviStr &buffer)
{
	if(m_pDnsStruct)buffer.append((m_pDnsStruct->iError == KVI_ERROR_Success) ? '1' : '0');
	else buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_LASTERROR(QList<KviStr> * params,KviStr &buffer)
{
	if(m_pDnsStruct)buffer.append(kvi_getErrorString(m_pDnsStruct->iError));
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_ABORT(QList<KviStr> * params,KviStr &buffer)
{
	buffer.append(m_pDns->abort() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptDns::builtinFunction_SETMODE(QList<KviStr> * params,KviStr &buffer)
{
	if(!params)return KVI_ERROR_MissingParameter;
	KviStr * pStr = params->first();
	if(!pStr)return KVI_ERROR_MissingParameter;
	if(kvi_strEqualCI(pStr->ptr(),"ipv6"))
	{
#ifdef COMPILE_NEED_IPV6
		m_bIpV6 = true;
#else
		buffer.append('0'); //no support
		return KVI_ERROR_Success;
#endif
	} else if(kvi_strEqualCI(pStr->ptr(),"ipv4"))
	{
		m_bIpV6 = false;
	} else return KVI_ERROR_InvalidParameter;
	buffer.append('1');
	return KVI_ERROR_Success;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
///////  TIMER CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: timer
	@short:
		A timer object
	@inherits:
		[class]object[/class]<br>

	@functions:
		!fn: $start(&lt;interval&gt;)
		Starts this timer with a specified interval (timeout) in milliseconds.<br>
		If the timer was previously running , it is first stopped.<br>
		This function returns '1' if succesfull , or '0' if
		the underlying system was not able to provide a timer service:<br>
		usually you can avoid the return value of this function.<br>

		!fn: $stop()
		Stops this timer.<br>

		!fn: $singleShot(&lt;interval&gt;,[bAutodestroy]);
		Starts this timer with a specified interval (timeout) in milliseconds.<br>
		The timer will "fire" only once.<br>
		If the &lt;bAutodestroy&gt; parameter is provided , and is '1'
		this timer object will automatically destroy itself, otherwise
		it will just return in stopped state.<br>
		If the timer was previously running , it is first stopped.<br>
		This function returns '1' if succesfull , or '0' if
		the underlying system was not able to provide a timer service:<br>
		usually you can avoid the return value of this function.<br>

		!fn: $isRunning()
		Returns '1' if the timer is currently running , '0' if not.<br>

		!fn: $interval()
		Returns the interval between the timer 'shots' if the timer
		is currently running , '0' otherwise.<br>

	@events:
		!ev: OnTimeout()
		Triggered by this timer object when the interval of time has passed.<br>
		The default event handler for this event emits the [classsignal:timer]timeout[/classsignal].<br>
		If you provide a new event handler and still want the signal to be emitted
		you must emit it by yourself.<br>
		<example>
		[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:timer]timeout[/classsignal])
		</example>

	@signals:
		!sg: timeout()
		This signal is emitted by the default handler for the [classevent:timer]OnTimeout[/classevent].
		event handler.<br>
		If you provide a new event handler and still want the signal to be emitted
		you must emit it by yourself.<br>
		<example>
		[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:timer]timeout[/classsignal])
		</example>

	@description:
		A simple timer object with a millisecond precision.<br>
		It fires the [classevent:timer]OnTimeout[/classevent] event after the
		specified delay.<br> The defaul event handler for that event
		emits the [classsignal:timer]timeout[/classsignal] signal.<br>
		The timer can be continous (started by [classfnc:timer]$start[/classfnc]())
		or "single shot" (started by [classfnc:timer]$singleShot[/classfnc]()).<br>
		In the single shot mode the timer has also the ability of autodestroy itself.<br>

	@examples:
		A class for saying text line by line to a channel (or another window).<br>
		It is useful when you have bots on the channel and they will likely kick
		you if you flood the channel with a long text.<br>
		(If the bots are not your problem , the problem may be the server
		flood protection that disconnects you for 'Excess Flood').<br>
		So here we go: we derive a class from [class]timer[/class].<br>
		<example>
		&nbsp;[cmd]class[/cmd](asciitimer,[class]timer[/class])
		&nbsp;{
		&nbsp;	[cmd:class]function[/cmd](run)
		&nbsp;	{
		&nbsp;      # The second param is the window to say the text to
		&nbsp;		[fnc]$this[/fnc]->%window = $2
		&nbsp;      # The remaining params are the "text" to say
		&nbsp;		[fnc]$this[/fnc]->%text = "$3-"
		&nbsp;      # The first parameter is the timeout in msecs
		&nbsp;		[cmd]setreturn[/cmd] [fnc]$this[/fnc]->[classfnc:timer]$start[/classfnc]($1)
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]event[/cmd]([classevent:timer]OnTimeout[/classevent])
		&nbsp;	{
		&nbsp;		%line = [fnc]$strLeftToFirstCS[/fnc]([fnc]$lf[/fnc],"[fnc]$this[/fnc]->%text")
		&nbsp;		[fnc]$this[/fnc]->%text = [fnc]$strRightFromFirstCS[/fnc]([fnc]$lf[/fnc],"[fnc]$this[/fnc]->%text")
		&nbsp;		[cmd]say[/cmd] [fnc]$this[/fnc]->%window %line
		&nbsp;		[cmd]if[/cmd]("[fnc]$this[/fnc]->%text" == "")[cmd]destroy[/cmd] [fnc]$this[/fnc]
		&nbsp;	}
		&nbsp;}
		</example>
		Now you can use the following code to run it (maybe place it in an
		appropriate alias)...<br>
		<example>
		%test = [fnc]$new[/fnc](asciitimer,[fnc]$root[/fnc],test)
		%test->$run(1000,#channel,"This[fnc]$lf[/fnc]\Is[fnc]$lf[/fnc]\A[fnc]$lf[/fnc]\Test[fnc]$lf[/fnc]\String[fnc]$lf[/fnc]\With[fnc]$lf[/fnc]\Newlines")
		</example>
	@seealso:
		class [class]object[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptTimer::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("start",(scriptObjectFunction)&KviScriptTimer::builtinFunction_START);
	d->addBuiltinFunction("stop",(scriptObjectFunction)&KviScriptTimer::builtinFunction_STOP);
	d->addBuiltinFunction("isRunning",(scriptObjectFunction)&KviScriptTimer::builtinFunction_ISRUNNING);
	d->addBuiltinFunction("interval",(scriptObjectFunction)&KviScriptTimer::builtinFunction_INTERVAL);
	d->addBuiltinFunction("singleShot",(scriptObjectFunction)&KviScriptTimer::builtinFunction_SINGLESHOT);

	KviScriptEventStruct * s = new KviScriptEventStruct;
	s->szName = "OnTimeout";
	s->szBuffer = "$this->$emit(timeout)";
	d->addDefaultEvent(s);
}

KviScriptTimer::KviScriptTimer(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptObject(cntrl,p,name,pDef)
{
	m_iTimerId = -1;
	m_iTimeout = 0;
}

KviScriptTimer::~KviScriptTimer()
{
	if(m_iTimerId != -1)killTimer(m_iTimerId);
}

void KviScriptTimer::timerEvent(QTimerEvent *e)
{
	if(e->timerId() == m_iTimerId)
	{
		KviStr params;
		triggerEvent("OnTimeout",params);
		if(m_bSingleShot)
		{
			killTimer(m_iTimerId);
			m_iTimerId = -1;
			m_iTimeout = 0;
			m_bSingleShot = false;
			if(m_bAutoDestroyAfterSingleShot)dieOutOfThisEventStep();
		}
	}
}

int KviScriptTimer::builtinFunction_START(QList<KviStr> *params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			bool bOk = false;
			int i = pS->toInt(&bOk);
			if(bOk && (i >= 0))
			{
				if(m_iTimerId != -1)killTimer(m_iTimerId);
				m_iTimerId    = startTimer(i);
				m_iTimeout    = i;
				m_bSingleShot = false;
				buffer.append((m_iTimerId != -1) ? '1' : '0');
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}

	return KVI_ERROR_MissingParameter;
}

int KviScriptTimer::builtinFunction_SINGLESHOT(QList<KviStr> *params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			bool bOk = false;
			int i = pS->toInt(&bOk);
			if(bOk && (i >= 0))
			{
				if(m_iTimerId != -1)killTimer(m_iTimerId);
				m_iTimerId    = startTimer(i);
				m_iTimeout    = i;
				m_bAutoDestroyAfterSingleShot = false;
				m_bSingleShot = true;
				KviStr * pS2 = params->next();
				if(pS2)if(kvi_strEqualCI(pS2->ptr(),"1"))m_bAutoDestroyAfterSingleShot = true;
				buffer.append((m_iTimerId != -1) ? '1' : '0');
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}

	return KVI_ERROR_Success;
}

int KviScriptTimer::builtinFunction_STOP(QList<KviStr> *,KviStr &buffer)
{
	if(m_iTimerId == -1)return KVI_ERROR_Success;
	killTimer(m_iTimerId);
	m_iTimerId = -1;
	m_iTimeout = 0;
	m_bSingleShot = false;
	return KVI_ERROR_Success;
}

int KviScriptTimer::builtinFunction_ISRUNNING(QList<KviStr> *,KviStr &buffer)
{
	buffer.append((m_iTimerId != -1) ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptTimer::builtinFunction_INTERVAL(QList<KviStr> *,KviStr &buffer)
{
	KviStr timeout(KviStr::Format,"%d",m_iTimeout);
	buffer.append(timeout);
	return KVI_ERROR_Success;
}


/////////////////////////////////////////////////////////////////////////////////////////////////
///////  FILE CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: file
	@short:
		A file interface object
	@inherits:
		[class]object[/class]<br>

	@functions:

		!fn: $setFileName(&lt;filename&gt;)
		Sets the filename of this file object.<br>
		If a file was previously open , it is closed first.<br>

		!fn: $fileName()
		Returns the filename for this file object.<br>

		!fn: $exists()
		Returns '1' if the file exists , '0' otherwise.<br>
		A filename must have been set previously.<br>

		!fn: $remove()
		Removes the file.<br>
		A filename must have been set previously.<br>
		Returns '1' if succesfull, '0' otherwise.<br>

		!fn: $size()
		Returns the size of the file.<br>
		A filename must have been set previously.<br>

		!fn: $open(&lt;mode&gt;)
		Opens the file in the specified mode.<br>
		&lt;mode&gt; can be any combination of the following flags:<br>
		<b>r</b> - Read access<br>
		<b>w</b> - Write access<br>
		<b>a</b> - Append (seek to end of the file)<br>
		<b>t</b> - Truncate the file to zero size<br>
		When the 'w' flag is specified and the file does not exist , it is created.<br>
		When ONLY the 'w' flag is specified , the file is truncated to zero size.<br>
		A filename must be set before this function can be called.<br>
		Returns '1' if succesfull, '0' otherwise.<br>

		!fn: $close()
		Closes this file object.<br>

		!fn: $flush()
		Flushes the cached file buffers to disk.<br>
		(The access is buffered to increase the performance).<br>
		[classfnc:file]$close[/classfnc]() calls it automatically, so
		you should not need this function.<br>

		!fn: $read(&lt;numBytes&gt;)
		Reads at most &lt;numBytes&gt; from the file and returns the string.<br>
		This function is not able to process NULL characters: you will be loosing
		data if the file is not a text one.<br>

		!fn: $readLine(&lt;maxNumBytes&gt;)
		Reads a line of text from the file.<br>
		This function will read at most &lt;numBytes&gt;.<br>
		This function is not able to process NULL characters: you will be loosing
		data if the file is not a text one.<br>

		!fn: $readHex(&lt;numBytes&gt;)
		Reads at most &lt;numBytes&gt; from the file and returns the hexadecimal
		encoding of the string (See [fnc]$strToHex[/fnc]() and [fnc]$hexToStr[/fnc]()).<br>

		!fn: $write(&lt;string_data&gt;)
		Writes the &lt;string_data&gt; to disk.<br>
		Returns the number of bytes written or '-1' in case of an error.<br>
		This function can NOT write 0 characters to the file, and is useful
		mainly for text files.<br>

		!fn: $writeHex(&lt;hex_data&gt;)
		Writes the &lt;hex_data&gt; to string.<br>
		The &lt;hex_data&gt; is a string of hexadecimal couples of digits.<br>
		Each couple of hex digits describes a byte.<br>
		Returns the number of bytes written to disk or '-1' in case of an error.<br>
		This function is useful mainly for binary files (the ones that usually
		contain unprintable characters and NULLs.<br>
		See also [fnc]$strToHex[/fnc]() and [fnc]$hexToStr[/fnc]().<br>

		!fn: $isEof()
		Returns '1' if the end of the file was reached , '0' otherwise.<br>

		!fn: $seek(&lt;position&gt;)
		Moves the file index to the specified &lt;position&gt;.<br>
		Returns '1' if succesfull , '0' otherwise.<br>

		!fn: $pos()
		Returns the file position.<br>

	@description:
		This object class is an interface to the system files.<br>
		You can open files , read data and write it.<br>

		The constructor accepts an additional parameter : the name of the file;
		in this way you can avoid to call [classfnc:file]$setFileName[/classfnc].<br>

		Note:<br>
		It is NOT a good idea to keep a file open for long time,
		you should open a file only when needed and close it immediately after
		you have terminated your operations.<br>
		This because the file may be accessed by other programs , and
		the disk storage I/O is really slower than the memory one (so you should
		perform most operations in memory, when possible, and not write & seek thru the file).<br>
	
	@examples:
		A funky implementation of the cat command.<br>
		It reads a text file and echoes it to the current window.<br>
		You can easily convert it to an alias.<br>
		<example>
		&nbsp;%f = [fnc]$new[/fnc](file,[fnc]$root[/fnc],dummy,/afilenamewithpath)
		&nbsp;[cmd]if[/cmd](%f->[classfnc:file]$open[/classfnc](r))
		&nbsp;{
		&nbsp;	[cmd]while[/cmd](!%f->[classfnc:file]$isEof[/classfnc]())[cmd]echo[/cmd] %f->[classfnc:file]$readLine[/classfnc](1000)
		&nbsp;	# close()is not really needed...but let's be pedantic
		&nbsp;	%f->[classfnc:file]$close[/classfnc]()
		&nbsp;} [cmd:if]else[/cmd] [cmd]echo[/cmd] Can't open the file for reading.
		&nbsp;[cmd]destroy[/cmd] %f
		</example>
	@seealso:
		class [class]object[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptFile::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setFileName",(scriptObjectFunction)&KviScriptFile::builtinFunction_SETFILENAME);
	d->addBuiltinFunction("fileName",(scriptObjectFunction)&KviScriptFile::builtinFunction_FILENAME);
	d->addBuiltinFunction("open",(scriptObjectFunction)&KviScriptFile::builtinFunction_OPEN);
	d->addBuiltinFunction("close",(scriptObjectFunction)&KviScriptFile::builtinFunction_CLOSE);
	d->addBuiltinFunction("flush",(scriptObjectFunction)&KviScriptFile::builtinFunction_FLUSH);
	d->addBuiltinFunction("size",(scriptObjectFunction)&KviScriptFile::builtinFunction_SIZE);
	d->addBuiltinFunction("exists",(scriptObjectFunction)&KviScriptFile::builtinFunction_EXISTS);
	d->addBuiltinFunction("remove",(scriptObjectFunction)&KviScriptFile::builtinFunction_REMOVE);
	d->addBuiltinFunction("isEof",(scriptObjectFunction)&KviScriptFile::builtinFunction_ISEOF);
	d->addBuiltinFunction("pos",(scriptObjectFunction)&KviScriptFile::builtinFunction_POS);
	d->addBuiltinFunction("seek",(scriptObjectFunction)&KviScriptFile::builtinFunction_SEEK);
	d->addBuiltinFunction("write",(scriptObjectFunction)&KviScriptFile::builtinFunction_WRITE);
	d->addBuiltinFunction("writeHex",(scriptObjectFunction)&KviScriptFile::builtinFunction_WRITEHEX);
	d->addBuiltinFunction("read",(scriptObjectFunction)&KviScriptFile::builtinFunction_READ);
	d->addBuiltinFunction("readLine",(scriptObjectFunction)&KviScriptFile::builtinFunction_READLINE);
	d->addBuiltinFunction("readHex",(scriptObjectFunction)&KviScriptFile::builtinFunction_READHEX);

}

KviScriptFile::KviScriptFile(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptObject(cntrl,p,name,pDef)
{
	m_pFile = 0;
}

KviScriptFile::~KviScriptFile()
{
	delete m_pFile;
}

bool KviScriptFile::init(QList<KviStr> * params)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS){
			m_pFile = new QFile(pS->ptr());
			return true;
		}
	}
	m_pFile = new QFile();
	return true;
}

int KviScriptFile::builtinFunction_SETFILENAME(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS){
			delete m_pFile;
			m_pFile = new QFile(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_FILENAME(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(m_pFile->name());
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_OPEN(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS){
			int flags = 0;
			if(pS->contains('r',false))flags |= IO_ReadOnly;
			if(pS->contains('w',false))flags |= IO_WriteOnly;
			if(pS->contains('a',false))flags |= IO_Append;
			if(pS->contains('t',false))flags |= IO_Truncate;
			buffer.append(m_pFile->open(flags) ? '1' : '0');
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_SEEK(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS){
			bool bOk = false;
			int pos = pS->toInt(&bOk);
			if(bOk)
			{
				buffer.append(m_pFile->at(pos) ? '1' : '0');
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_WRITE(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS){
			int written = m_pFile->writeBlock(pS->ptr(),pS->len());
			KviStr tmp(KviStr::Format,"%d",written);
			buffer.append(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_READ(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS){
			bool bOk = false;
			unsigned int len = pS->toUInt(&bOk);
			if(bOk && (len > 0))
			{
				char * buf = (char *)kvi_malloc(len);
				m_pFile->readBlock(buf,len);
				buffer.append(buf);
				kvi_free(buf);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_READHEX(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS){
			bool bOk = false;
			unsigned int len = pS->toUInt(&bOk);
			if(bOk && (len > 0))
			{
				char * buf = (char *)kvi_malloc(len);
				int readed = m_pFile->readBlock(buf,len);
				if(readed > 0)
				{
					KviStr tmp;
					tmp.bufferToHex(buf,readed);
					buffer.append(tmp);
				}
				kvi_free(buf);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}


int KviScriptFile::builtinFunction_READLINE(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS){
			bool bOk = false;
			unsigned int len = pS->toUInt(&bOk);
			if(bOk && (len > 0))
			{
				char * buf = (char *)kvi_malloc(len);
				m_pFile->readLine(buf,len);
				buf[len - 1] = '\0'; //kill the trailing newline
				buffer.append(buf);
				kvi_free(buf);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_WRITEHEX(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS){
			char * buf = 0;
			int len = pS->hexToBuffer(&buf,false);
			if(buf)
			{
				int written = m_pFile->writeBlock(buf,len);
				KviStr tmp(KviStr::Format,"%d",written + 1);
				buffer.append(tmp);
				kvi_free(buf);
			} else buffer.append("-1");
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_CLOSE(QList<KviStr> *,KviStr &buffer)
{
	m_pFile->close();
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_FLUSH(QList<KviStr> *,KviStr &buffer)
{
	m_pFile->flush();
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_SIZE(QList<KviStr> *,KviStr &buffer)
{
	KviStr str(KviStr::Format,"%u",m_pFile->size());
	buffer.append(str);
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_POS(QList<KviStr> *,KviStr &buffer)
{
	KviStr str(KviStr::Format,"%d",m_pFile->at());
	buffer.append(str);
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_EXISTS(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(m_pFile->exists() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_REMOVE(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(m_pFile->remove() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_ISEOF(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(m_pFile->atEnd() ? '1' : '0');
	return KVI_ERROR_Success;
}

#include "m_kvi_scriptmisc.moc"
