//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_scriptlayout.h"
#include "kvi_scriptwidget.h"

#include "kvi_error.h"

/*
	@class: layout
	@short:
		Manages widget geometries
	@inherits:
		<a href="class_object.kvihelp">object</a>
	@functions:
		!fn: $margin()
		Returns the margin of this layout in pixels.<br>

		!fn: $setMargin(&lt;margin&gt;)
		Sets the margin of this layout (value in pixels).<br>

		!fn: $spacing()
		Returns the spacing value of this layout in pixels.<br>

		!fn: $setSpacing(&lt;spacing&gt;)
		Sets the spacing of this layout (value in pixels).<br>

		!fn: $add(&lt;widget&gt;,&lt;row&gt;,&lt;column&gt;)
		Adds a widget to this layout in the specified row and column.<br>
		The widget's geometry will be automatically managed from this moment.<br>
		The widget must be a children of the parent of this layout.<br>

		!fn: $addMultiCell(&lt;widget&gt;,&lt;fromRow&gt;,&lt;toRow&gt;,&lt;fromCol&gt;,&lt;toCol&gt;)
		Adds a widget tho this layout assigning it multiple adiacent rows and columns.<br>
		The widget's geometry will be automatically managed from this moment.<br>
		The widget must be a children of the parent of this layout.<br>

		!fn: $setRowStretch(&lt;row&gt;,&lt;factor&gt;)
		Sets the row stretch factor to &lt;factor&gt;.<br>
		The stretch factor is relative to the other rows: the rows with bigger
		stretch factor take more space in the layout.<br>
		If &lt;factor&gt; is zero , the row will not grow at all (unless all the rows have
		stretch factor 0).<br>
		The default stretch factor is 0.<br>

		!fn: $setColStretch(&lt;col&gt;,&lt;factor&gt;)
		Sets the column stretch factor to &lt;factor&gt;.<br>
		The stretch factor is relative to the other columns: the columns with bigger
		stretch factor take more space in the layout.<br>
		If &lt;factor&gt; is zero , the column will not grow at all (unless all the columns have
		stretch factor 0).<br>
		The default stretch factor is 0.<br>

	@description:
		A layout is an abstract object used to automatically manage widget geometries.<br>
		It is organized in cells identified by row and column coordinates.<br>
		You create a layout object as a children of a "main" widget.<br>
		Then create the "main" widget's children and add them to the layout in
		a specified cell (or set of adiacent cells).<br>
		After the "main" widget has been shown , the layout will adjust
		the children geometries to fit in the "main" widget and occupy all the available space.<br>
		The "margin" property of the layout is the space on the borders of the main widget.<br>
		The "spacing" property is the space left child between widgets.<br>

	@examples:
		A really simple (and useless) example:<br>
		A toplevel widget with a layout that manages two children.<br>
		One in the top left (0,0) corner and another in the bottom right (1,1) corner.<br>
		<example>
			# Create the toplevel widget
			%top = <a href="s_new.kvihelp">$new</a>(<a href="class_widget.kvihelp">widget</a>,<a href="s_root.kvihelp">$root</a>,toplevel)
			%top-&gt;<a href="class_widget.kvihelp#setgeometry">$setGeometry</a>(0,0,400,400)
			# Create the layout that will manage the children
			%layout = <a href=s_new.kvihelp">$new</a>(layout,%top,toplevel_layout)
			%layout-&gt;<a href="class_layout.kvihelp#setmargin">$setMargin</a>(5)
			%layout-&gt;<a href="class_layout.kvihelp#setspacing">$setSpacing</a>(2)
			# Create two children
			%child1 = <a href="s_new.kvihelp">$new</a>(<a href="class_widget.kvihelp">widget</a>,%top,managed_child_1)
			%child1-&gt;<a href="class_widget.kvihelp#setbackgroundcolor">$setBackgroundColor</a>(FFFFFF)
			%layout-&gt;<a href="class_layout.kvihelp#add">$add</a>(%child1,0,0)
			%child2 = <a href="s_new.kvihelp">$new</a>(<a href="class_widget.kvihelp">widget</a>,%top,managed_child_2)
			%child2-&gt;<a href="class_widget.kvihelp#setbackgroundcolor">$setBackgroundColor</a>(A0A0A0)
			%layout-&gt;<a href="class_layout#add">$add</a>(%child2,1,1)
			%top-&gt;<a href="class_widget.kvihelp#show">$show</a>(1)
		</example>
	@seealso:
		<a href="class_object.kvihelp">Object class</a>,
		<a href="class_widget.kvihelp">Widget class</a>,
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/


void KviScriptLayout::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setMargin",(scriptObjectFunction)&KviScriptLayout::builtinFunction_SETMARGIN);
	d->addBuiltinFunction("margin",(scriptObjectFunction)&KviScriptLayout::builtinFunction_MARGIN);
	d->addBuiltinFunction("setSpacing",(scriptObjectFunction)&KviScriptLayout::builtinFunction_SETSPACING);
	d->addBuiltinFunction("spacing",(scriptObjectFunction)&KviScriptLayout::builtinFunction_SPACING);
	d->addBuiltinFunction("add",(scriptObjectFunction)&KviScriptLayout::builtinFunction_ADD);
	d->addBuiltinFunction("addMultiCell",(scriptObjectFunction)&KviScriptLayout::builtinFunction_ADDMULTICELL);
	d->addBuiltinFunction("setRowStretch",(scriptObjectFunction)&KviScriptLayout::builtinFunction_SETROWSTRETCH);
	d->addBuiltinFunction("setColStretch",(scriptObjectFunction)&KviScriptLayout::builtinFunction_SETCOLSTRETCH);
}

KviScriptLayout::KviScriptLayout(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptObject(cntrl,p,name,pDef)
{
	m_pLayout = 0;
}

KviScriptLayout::~KviScriptLayout()
{
	if(m_pLayout)delete m_pLayout;
}

bool KviScriptLayout::init(QList<KviStr> *)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pLayout = new QGridLayout(((KviScriptWidget *)parent())->m_pWidget,0,0,0,0);
			connect(m_pLayout,SIGNAL(destroyed()),this,SLOT(layoutDestroyed()));
			return true;
		}
	}
	return false;
}

void KviScriptLayout::layoutDestroyed()
{
	m_pLayout = 0;
	dieOutOfThisEventStep();
}


int KviScriptLayout::builtinFunction_MARGIN(QList<KviStr> *,KviStr &buffer)
{
	KviStr str(KviStr::Format,"%d",m_pLayout->margin());
	buffer.append(str);
	return KVI_ERROR_Success;
}

int KviScriptLayout::builtinFunction_SETMARGIN(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			m_pLayout->setMargin(pS->toInt());
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptLayout::builtinFunction_SPACING(QList<KviStr> *,KviStr &buffer)
{
	KviStr str(KviStr::Format,"%d",m_pLayout->spacing());
	buffer.append(str);
	return KVI_ERROR_Success;
}

int KviScriptLayout::builtinFunction_SETSPACING(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			m_pLayout->setSpacing(pS->toInt());
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptLayout::builtinFunction_ADD(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			KviScriptObject * o = controller()->findObjectById(pS->ptr());
			if(o)
			{
				if(o->inherits("KviScriptWidget"))
				{
					int row=0;
					int col=0;
					pS = params->next();
					if(pS)row = pS->toInt();
					else return KVI_ERROR_MissingParameter;
					pS = params->next();
					if(pS)col = pS->toInt();
					else return KVI_ERROR_MissingParameter;
					m_pLayout->addWidget(((KviScriptWidget *)o)->m_pWidget,row,col);
				} else return KVI_ERROR_ObjectIsNotAWidget;
			} else return KVI_ERROR_ObjectNotFound;
		} else return KVI_ERROR_MissingObjectId;
	} else return KVI_ERROR_MissingParameter;
	return KVI_ERROR_Success;
}

int KviScriptLayout::builtinFunction_ADDMULTICELL(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			KviScriptObject * o = controller()->findObjectById(pS->ptr());
			if(o)
			{
				if(o->inherits("KviScriptWidget"))
				{
					int fromRow=0;
					int toRow=0;
					int fromCol=0;
					int toCol=0;
					pS = params->next();
					if(pS)fromRow = pS->toInt();
					else return KVI_ERROR_MissingParameter;
					pS = params->next();
					if(pS)toRow = pS->toInt();
					else return KVI_ERROR_MissingParameter;
					pS = params->next();
					if(pS)fromCol = pS->toInt();
					else return KVI_ERROR_MissingParameter;
					pS = params->next();
					if(pS)toCol = pS->toInt();
					else return KVI_ERROR_MissingParameter;
					m_pLayout->addMultiCellWidget(((KviScriptWidget *)o)->m_pWidget,fromRow,toRow,fromCol,toCol);
				} else return KVI_ERROR_ObjectIsNotAWidget;
			} else return KVI_ERROR_ObjectNotFound;
		} else return KVI_ERROR_MissingObjectId;
	} else return KVI_ERROR_MissingParameter;
	return KVI_ERROR_Success;
}

int KviScriptLayout::builtinFunction_SETROWSTRETCH(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(!pS)return KVI_ERROR_MissingParameter;
		int idx = pS->toInt();
		pS = params->next();
		if(!pS)return KVI_ERROR_MissingParameter;
		int factor = pS->toInt();
		m_pLayout->setRowStretch(idx,factor);
	} else return KVI_ERROR_MissingParameter;
	return KVI_ERROR_Success;
}

int KviScriptLayout::builtinFunction_SETCOLSTRETCH(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(!pS)return KVI_ERROR_MissingParameter;
		int idx = pS->toInt();
		pS = params->next();
		if(!pS)return KVI_ERROR_MissingParameter;
		int factor = pS->toInt();
		m_pLayout->setColStretch(idx,factor);
	} else return KVI_ERROR_MissingParameter;
	return KVI_ERROR_Success;
}

#include "m_kvi_scriptlayout.moc"
