//
//   File : kvi_popupmenu.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_popupmenu.cpp)
//   Last major modification : Wed Apr 21 1999 21:36:26 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CLASS_NAME_ "KviPopupMenu"
//#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_popupmenu.h"

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_uparser.h"
#include "kvi_error.h"
#include "kvi_defines.h"
#include "kvi_locale.h"
#include "kvi_fileutils.h"
#include "kvi_string.h"

#include <qpixmap.h>
#include <qiconset.h>
#include <qcursor.h>

// declared in kvi_app.cpp
extern QPixmap * g_pixViewOut[KVI_OUT_NUM_IMAGES];
extern KviPopupMenu *g_pChannelPopup;
extern KviPopupMenu *g_pConsolePopup;
extern KviPopupMenu *g_pUserlistPopup;
extern KviPopupMenu *g_pNotifylistPopup;
extern KviPopupMenu *g_pQueryPopup;
extern KviPopupMenu *g_pDccChatPopup;

KviPopupMenu::KviPopupMenu(QWidget * parent,const char * name)
:QPopupMenu(parent,name)
{
	m_pDataList   = new QList<KviPopupMenuData>;
	m_pDataList->setAutoDelete(true);
	connect(this,SIGNAL(activated(int)),this,SLOT(localItemClicked(int)));
	m_pParent     = 0;
	m_pParentWindow = 0;
	m_bInTestMode = false;
	m_bLocked     = false;
}

KviPopupMenu::~KviPopupMenu()
{
	clearAll();
	delete m_pDataList;
}

void KviPopupMenu::copyFromPopup(KviPopupMenu *popup)
{
	__range_valid(popup);
	clearAll();
	int idx = 0;
	for(KviPopupMenuData * d=popup->m_pDataList->first();d;d=popup->m_pDataList->next()){
		if(d->menu != 0){
			KviStr yourName(KviStr::Format,"%s_sub%d",name(),idx);
			KviPopupMenu * sub = new KviPopupMenu(0,yourName.ptr());
			sub->copyFromPopup(d->menu);
			//KviStr tmp = popup->text(d->id);
			insertPopupItem(d->text.ptr(),d->image.ptr(),sub);
			idx++;
		} else {
			if((d->id == 0) && kvi_strEqualCS(d->buffer.ptr()," ")){
				insertSeparatorItem();
			} else {
				//KviStr tmp = popup->text(d->id);
				insertNormalItem(d->text.ptr(),d->image.ptr(),d->buffer.ptr());
			}
		}
	}	
}

void KviPopupMenu::insertSeparatorItem()
{
	KviPopupMenuData *data = new KviPopupMenuData;
	insertSeparator();
	data->id = 0;
	data->buffer = " ";
	data->text = "";
	data->menu = 0;
	m_pDataList->append(data);
}

int KviPopupMenu::insertNormalItem(const char *name,const char *img,const char *buffer)
{
	KviPopupMenuData *data = new KviPopupMenuData;
	bool bDone = false;
	KviStr pName(name);
	if(m_pParentWindow)
		m_pParentWindow->m_pFrm->m_pUserParser->processString(name,
		m_pParentWindow,pName);
	KviStr image(img);
	if(image.hasData()){
		if(image.isNum()){
			bool bOk=false;
			int imId = image.toInt(&bOk);
			if(imId >= 0){
				imId = imId % KVI_OUT_NUM_IMAGES;
				data->id = insertItem(QIconSet(*g_pixViewOut[imId]),_CHAR_2_QSTRING(pName.ptr()));
				data->text = name;
				data->image = image;
				bDone = true;
			}
		} else {
			KviStr imagePath;
			if(g_pApp->findImage(imagePath,image.ptr())){
				QPixmap pix(imagePath.ptr());
				if(!pix.isNull()){
					data->id = insertItem(QIconSet(pix),_CHAR_2_QSTRING(pName.ptr()));
					data->text = name;
					data->image = image; //Not imagePath!..so if it is moved we will be able to find it again
					bDone = true;
				} else debug("Unable to find image file %s",image.ptr());
			}
		}
	}
	if(!bDone)data->id = insertItem(_CHAR_2_QSTRING(pName.ptr()));
	data->buffer = buffer;
	data->buffer.stripWhiteSpace();
	data->text = name;
	data->menu = 0;
	m_pDataList->append(data);
	return data->id;
}

void KviPopupMenu::insertPopupItem(const char *name,const char *img,KviPopupMenu *menu)
{
	KviPopupMenuData *data = new KviPopupMenuData;
	bool bDone = false;
	KviStr pName(name);
	if(m_pParentWindow)
		m_pParentWindow->m_pFrm->m_pUserParser->processString(name,
		m_pParentWindow,pName);
	KviStr image(img);
	if(image.hasData()){
		if(image.isNum()){
			bool bOk=false;
			int imId = image.toInt(&bOk);
			if(imId >= 0){
				imId = imId % KVI_OUT_NUM_IMAGES;
				data->id = insertItem(QIconSet(*g_pixViewOut[imId]),_CHAR_2_QSTRING(pName.ptr()),menu);
				data->text = name;
				data->image = image;
				bDone = true;
			}
		} else {
			KviStr imagePath;
			if(g_pApp->findImage(imagePath,image.ptr())){
				QPixmap pix(imagePath.ptr());
				if(!pix.isNull()){
					data->id = insertItem(QIconSet(pix),_CHAR_2_QSTRING(pName.ptr()),menu);
					data->text = name;
					data->image = image; //Not imagePath!..so if it is moved we will be able to find it again
					bDone = true;
				}
			} else debug("Unable to find image file %s",image.ptr());
		}
	}
	if(!bDone)data->id = insertItem(_CHAR_2_QSTRING(pName.ptr()),menu);
	data->text = name;
	data->menu = menu;
	m_pDataList->append(data);
	menu->m_pParent = this;
	menu->m_pParentWindow = m_pParentWindow;
}

void KviPopupMenu::localItemClicked(int id)
{
	for(KviPopupMenuData * d=m_pDataList->first();d;d=m_pDataList->next()){
		if(d->id == id && d->menu == 0){
			// Only the top menu emits itemClicked() (or testItemClicked if in test mode)
			if(m_pParent != 0)m_pParent->itemClickedRedirect(d->buffer,d->id);
			else {
				if(m_bInTestMode)emit testItemClicked(d->id);
				else {
					m_bLocked = true;
					emit itemClicked(d->buffer);
					m_bLocked = false;
				}
			}
			return;
		}
	}
	__debug_1arg("Oops... cannot find menu item %d",id);
}

void KviPopupMenu::itemClickedRedirect(const KviStr &buffer,int id)
{
	if(m_pParent != 0)m_pParent->itemClickedRedirect(buffer,id);
	else {
		if(m_bInTestMode)emit testItemClicked(id);
		else {
			m_bLocked = true;
			emit itemClicked(buffer);
			m_bLocked = false;
		}
	}
}

void KviPopupMenu::clearAll()
{
	KviPopupMenuData * data;
	while((data = m_pDataList->first())){
		if(data->menu != 0)delete data->menu;
		m_pDataList->removeFirst();
	}
	clear();
}

const KviStr KviPopupMenu::text(int id)
{
	for(KviPopupMenuData * d=m_pDataList->first();d;d=m_pDataList->next()){
		if(d->id == id)return d->text;
	}
}

void KviPopupMenu::setFont(const QFont &font)
{
	// Propagate the font to children
	for(KviPopupMenuData *d= m_pDataList->first();d;d=m_pDataList->next()){
		if(d->menu != 0)d->menu->setFont(font);
	}
	QPopupMenu::setFont(font);
}

void KviPopupMenu::setPaletteRecursive(const QPalette &pal)
{
	// Propagate the palette to children
	for(KviPopupMenuData *d= m_pDataList->first();d;d=m_pDataList->next()){
		if(d->menu != 0)d->menu->setPalette(pal);
	}
	setPalette(pal);
}

void KviPopupMenu::doPopup(const QWidget *receiver,const char *memberSlot)
{
	disconnect(this,SIGNAL(itemClicked(const KviStr &)),0,0);
	if(m_bInTestMode){
		disconnect(this,SIGNAL(testItemClicked(int)),0,0);
		m_bInTestMode = false;
	}

	connect(this,SIGNAL(itemClicked(const KviStr &)),receiver,memberSlot);
	reloadMenu(receiver);
	popup(QCursor::pos());
}

// overloaded for fixed pos
void KviPopupMenu::doPopup(const QWidget *receiver,const char *memberSlot,QPoint pos)
{
	disconnect(this,SIGNAL(itemClicked(const KviStr &)),0,0);
	if(m_bInTestMode){
		disconnect(this,SIGNAL(testItemClicked(int)),0,0);
		m_bInTestMode = false;
	}

	connect(this,SIGNAL(itemClicked(const KviStr &)),receiver,memberSlot);
	reloadMenu(receiver);
	popup(pos);
}

void KviPopupMenu::doTestPopup(const QWidget *receiver,const char *memberSlot)
{
	disconnect(this,SIGNAL(itemClicked(const KviStr &)),0,0);
	disconnect(this,SIGNAL(testItemClicked(int)),0,0);
	m_bInTestMode = true;

	connect(this,SIGNAL(testItemClicked(int)),receiver,memberSlot);
	reloadMenu(receiver);
	popup(QCursor::pos());
}

void KviPopupMenu::reloadMenu(const QWidget *parentWindow)
{
	KviStr buffer;
	KviStr error;
	m_pParentWindow = (KviWindow *)parentWindow;
	if(!m_pParentWindow)
		m_pParentWindow = (KviWindow *)g_pApp->m_pFrameList->first()->m_pConsole;
	buildDataBuffer(buffer);
	loadFromDataBuffer(buffer,error);
}

//===========================================================================//
//
// Load and save popups....
//

//#warning "Maybe write a SIMPLIER code to save and load popups for 1.0.0 ???"

bool KviPopupMenu::saveAllPopups(const char *filename)
{
	QFile f(filename);
	if(!f.open(IO_WriteOnly | IO_Truncate))return false;

	f.writeBlock(KVI_MAGIC_STRING,KVI_MAGIC_STRING_LENGTH);

	g_pChannelPopup->save(&f,"channel");
	g_pConsolePopup->save(&f,"console");
	g_pUserlistPopup->save(&f,"userlist");
	g_pNotifylistPopup->save(&f,"notifylist");
	g_pQueryPopup->save(&f,"query");
	g_pDccChatPopup->save(&f,"dccchat");

	f.close();
	return true;
}

bool KviPopupMenu::loadAllPopups(const char *filename,KviStr &error)
{
	KviStr buffer;
	if(!kvi_loadFile(filename,buffer)){
		error = _i18n_("Unable to load file");
		return false;
	}

	if(!kvi_strEqualCSN(KVI_MAGIC_STRING,buffer.ptr(),KVI_MAGIC_STRING_LENGTH)){
		error = _i18n_("Invalid magic");
		__debug(buffer.ptr());
		return false; //not a kvirc file
	}

	buffer.cutLeft(KVI_MAGIC_STRING_LENGTH);

	while(buffer.hasData()){
		buffer.stripLeftWhiteSpace();

		if(!kvi_strEqualCSN("[POPUP ",buffer.ptr(),7)){
			error = _i18n_("Invalid POPUP label");
			__debug(buffer.ptr());
			return false;
		}

		buffer.cutLeft(7);
		buffer.stripLeftWhiteSpace();

		KviPopupMenu *target = 0;

		if(kvi_strEqualNoLocaleCIN("channel",buffer.ptr(),7)){
			target = g_pChannelPopup;
			buffer.cutLeft(7);
		} else if(kvi_strEqualNoLocaleCIN("console",buffer.ptr(),7)){
			target = g_pConsolePopup;
			buffer.cutLeft(7);
		} else if(kvi_strEqualNoLocaleCIN("userlist",buffer.ptr(),8)){
			target = g_pUserlistPopup;
			buffer.cutLeft(8);
		} else if(kvi_strEqualNoLocaleCIN("notifylist",buffer.ptr(),10)){
			target = g_pNotifylistPopup;
			buffer.cutLeft(10);
		} else if(kvi_strEqualNoLocaleCIN("query",buffer.ptr(),5)){
			target = g_pQueryPopup;
			buffer.cutLeft(5);
		} else if(kvi_strEqualNoLocaleCIN("dccchat",buffer.ptr(),7)){
			target = g_pDccChatPopup;
			buffer.cutLeft(7);
		} else {
			error = __tr("Unknown popup name");
			__debug(buffer.ptr());
			return false;
		}

		buffer.stripLeftWhiteSpace();
		if(!kvi_strEqualNoLocaleCIN("bytes ",buffer.ptr(),6)){
			error = __tr("Syntax error in POPUP label");
			__debug(buffer.ptr());
			return false;
		}

		buffer.cutLeft(6);
		buffer.stripLeftWhiteSpace();

		if(!isdigit(*(buffer.ptr()))){
			error = _i18n_("Syntax error in [POPUP] label (digit expected)");
			__debug(buffer.ptr());
			return false;
		}

		KviStr numBytes;
		if(!buffer.getToken(numBytes,']')){
			error = _i18n_("Syntax error in [POPUP] label (can not extract popup length)");
			__debug(buffer.ptr());
			return false;
		}

		bool bOk = false;
		int bytes = numBytes.toInt(&bOk);
		if(!bOk){
			error = _i18n_("Unrecogized token ");
			error +=numBytes;
			__debug(buffer.ptr());
			return false;
		} //syntax error again...someone messed with the file

		buffer.cutLeft(1); //cut the \n
		if(buffer.len() < bytes){
			error = _i18n_("Popup buffer smaller than declared length (");
			error += numBytes;
			error +=")";
			__debug(buffer.ptr());
			return false;
		}

		KviStr popupBuf = buffer.left(bytes);
		buffer.cutLeft(bytes);

		buffer.stripLeftWhiteSpace();

		if(!kvi_strEqualCSN("[ENDPOPUP]\n",buffer.ptr(),11)){
			error = __tr("Something went wrong with the popup buffer lenght");
			__debug(buffer.ptr());
			return false;
		}
		buffer.cutLeft(11);

		if(target->m_bLocked){
			error = __tr("Attempted to self-modify a locked popup");
			debug("WARNING: Attempted to self-modify locked popup, skipping");
			return false;
		}

		if(!target->loadFromDataBuffer(popupBuf,error)){
			__debug(popupBuf.ptr());
			return false;
		}
	}
	return true;
}

void KviPopupMenu::save(QFile *f,const char *popupName)
{
	f->writeBlock("[POPUP ",7);
	f->writeBlock(popupName,strlen(popupName));
	f->writeBlock(" bytes ",7);

	KviStr buffer;
	buildDataBuffer(buffer);

	KviStr szNBytes(KviStr::Format,"%d",buffer.len());
	f->writeBlock(szNBytes.ptr(),szNBytes.len());
	f->writeBlock("]\n",2);
	f->writeBlock(buffer.ptr(),buffer.len());
	f->writeBlock("\n[ENDPOPUP]\n",12);
}

//
// An item with id=0 , menu=0 and buffer = " " is a separator.
// Normal menu item buffers have the initial spaces stripped at
// insertion time
//

bool KviPopupMenu::loadFromDataBuffer(KviStr &buffer,KviStr &error)
{
	clearAll();
	buffer.stripLeftWhiteSpace();
	int idx  = 0;
	while(buffer.hasData()){
		if(kvi_strEqualNoLocaleCIN("[separator]\n",buffer.ptr(),12)){
			buffer.cutLeft(12);
			insertSeparatorItem();
		} else if(kvi_strEqualNoLocaleCIN("[item]\n",buffer.ptr(),7)){
			buffer.cutLeft(7);
			KviStr image,name,bytes;
			if(!buffer.getLine(image)){
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if(!buffer.getLine(name)){
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if(!buffer.getLine(bytes)){
				error = __tr("Unexpected end of buffer");
				return false;
			}

			if(!kvi_strEqualNoLocaleCIN("image=",image.ptr(),6)){
				error = __tr("Unrecognized item image label");
				return false;
			}
			image.cutLeft(6);
			image.stripWhiteSpace();
			if(!kvi_strEqualNoLocaleCIN("name=",name.ptr(),5)){
				error = __tr("Unrecognized item name label");
				return false;
			}
			name.cutLeft(5);
			name.stripWhiteSpace();
			if(!kvi_strEqualNoLocaleCIN("bytes=",bytes.ptr(),6)){
				error = __tr("Unrecognized item length label");
				return false;
			}
			bytes.cutLeft(6);
			bytes.stripWhiteSpace();
			bool bOk = false;
			int bLen = bytes.toInt(&bOk);
			if(!bOk){
				error = __tr("Unrecognized item length");
				return false;
			}
			if((bLen < 0) || (bLen > buffer.len())){
				error = __tr("Buffer is shorter than the declared length");
				return false;
			}
			KviStr pBuf = buffer.left(bLen);
			buffer.cutLeft(bLen);
			buffer.stripLeftWhiteSpace();
			if(!kvi_strEqualNoLocaleCIN("[enditem]\n",buffer.ptr(),10)){
				error = __tr("Something went wrong with the buffer length");
				return false;
			}
			buffer.cutLeft(10);
			insertNormalItem(name.ptr(),image.ptr(),pBuf.ptr());
		} else if(kvi_strEqualNoLocaleCIN("[submenu]\n",buffer.ptr(),10)){
			buffer.cutLeft(10);
			KviStr image,szName,bytes;
			if(!buffer.getLine(image)){
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if(!buffer.getLine(szName)){
				error = __tr("Unexpected end of buffer");
				return false;
			}
			buffer.stripLeftWhiteSpace();
			if(!buffer.getLine(bytes)){
				error = __tr("Unexpected end of buffer");
				return false;
			}

			if(!kvi_strEqualNoLocaleCIN("image=",image.ptr(),6)){
				error = __tr("Unrecognized submenu image label");
				return false;
			}
			image.cutLeft(6);
			image.stripWhiteSpace();
			if(!kvi_strEqualNoLocaleCIN("name=",szName.ptr(),5)){
				error = __tr("Unrecognized submenu name label");
				return false;
			}
			szName.cutLeft(5);
			szName.stripWhiteSpace();
			if(!kvi_strEqualNoLocaleCIN("bytes=",bytes.ptr(),6)){
				error = __tr("Unrecognized submenu length label");
				return false;
			}
			bytes.cutLeft(6);
			bytes.stripWhiteSpace();
			bool bOk = false;
			int bLen = bytes.toInt(&bOk);
			if(!bOk){
				error = __tr("Unrecognized submenu length");
				return false;
			}
			if((bLen < 0) || (bLen > buffer.len())){
				error = __tr("Buffer is shorter than the declared length");
				return false;
			}
			KviStr mBuf = buffer.left(bLen);
			buffer.cutLeft(bLen);
			buffer.stripLeftWhiteSpace();
			if(!kvi_strEqualNoLocaleCIN("[endsubmenu]\n",buffer.ptr(),13)){
				error = __tr("Something went wrong with the buffer length");
				return false;
			}
			buffer.cutLeft(13);

			KviStr yourName(KviStr::Format,"%s_sub%d",name(),idx);

			KviPopupMenu *menu = new KviPopupMenu(0,yourName.ptr());
			if(!menu->loadFromDataBuffer(mBuf,error)){
				delete menu;
				return false;
			}

			insertPopupItem(szName.ptr(),image.ptr(),menu);
			idx++;
		} else {
			error = __tr("Unrecognized popup item");
			return false;
		}

		buffer.stripLeftWhiteSpace();
	}
	return true;
}

void KviPopupMenu::buildDataBuffer(KviStr &str)
{
	for(KviPopupMenuData * d=m_pDataList->first();d;d=m_pDataList->next()){
		if(d->menu == 0){
			if(d->id == 0){
				if(kvi_strEqualCSN(d->buffer.ptr()," ",1)){
					str.append("[separator]\n");
					continue;
				}
			}
			str.append("[item]\n");
			str.append("image=");
			if(d->image.hasData())str.append(d->image);
			str.append("\nname=");
			str.append(d->text);
			str.append("\nbytes=");
			KviStr tmp(KviStr::Format,"%d",d->buffer.len());
			str.append(tmp);
			str.append("\n");
			str.append(d->buffer);
			str.append("\n[enditem]\n");
		} else {
			str.append("[submenu]\n");
			str.append("image=");
			if(d->image.hasData())str.append(d->image);
			str.append("\nname=");
			str.append(d->text);
			KviStr auxBuf;
			d->menu->buildDataBuffer(auxBuf);
			str.append("\nbytes=");
			KviStr tmp(KviStr::Format,"%d",auxBuf.len());
			str.append(tmp);
			str.append("\n");
			str.append(auxBuf);
			str.append("\n[endsubmenu]\n");
		}
	}
}
/*
void KviPopupMenu::buildPDataBuffer(KviStr &str, char *name =0)
{
	for(KviPopupMenuData * d=m_pDataList->first();d;d=m_pDataList->next()){
		if(d->menu == 0){
			if(d->id == 0){
				if(kvi_strEqualCSN(d->buffer.ptr()," ",1)){
					str.append("separator\n");
					continue;
				}
			}
			str.append("item (");
			str.append(text(d->id));
			if(d->image.hasData()) {
				str.append(",");
				str.append(d->image);
			}
			str.append("): ");
			str.append("{\n");
			str.append(d->buffer);
			str.append("\n}\n");
		} else {
			str.append("submenu (");
			str.append(text(d->id));
			if(d->image.hasData()) {
				str.append(",");
				str.append(d->image);
			}
			str.append("): ");
			str.append("{\n");

			KviStr auxBuf;
			d->menu->buildPDataBuffer(auxBuf);
			str.append(auxBuf);
			str.append("\n}\n");
		}
	}
}
*/

//=============================================================================
//
// Uparser code to generate menus : auxiliary function
//
// popup(popupname) <-- uparser
// { <----this function starts here
// } <----and ends here
//

bool KviUserParser::generatePopupFromBuffer(KviPopupMenu *m,KviCommand *c,bool bAppend)
{
	if(!bAppend)m->clearAll();

	if(*(c->m_ptr) != '{'){
		c->setError(KVI_ERROR_OpenBraceExpected,"POPUP-GENERATOR",c->m_ptr);
		return false;
	}
	++(c->m_ptr);

	int idx = 0;

	for(;;){
		c->skipWhitespace();
		if(!(*(c->m_ptr))){ // Missing terminator...
			c->setError(KVI_ERROR_UnexpectedEndOfCommand,"POPUP-GENERATOR");
			return false;
		}
		if(*(c->m_ptr) == '}'){ // Finished
			++c->m_ptr;
			return true;
		}

		if(kvi_strEqualNoLocaleCIN(c->m_ptr,"item",4)){
			c->m_ptr+=4;
			c->skipSpace();
			if(*(c->m_ptr) != '('){
				c->setError(KVI_ERROR_OpenParenthesisExpected,"POPUP-ITEM-LABEL",c->m_ptr);
				return false;
			}
			++c->m_ptr;
			KviStr itemLabel;
			if(!processFncSingleParam(c,itemLabel))return false;
//			itemLabel.stripWhiteSpace();

			KviStr itemImage;
			if(!processFncFinalPart(c,itemImage))return false;
//			itemImage.stripWhiteSpace();

			c->skipSpace();

			if(*(c->m_ptr) == ':'){ //Allowed but not required
				++c->m_ptr;
				c->skipSpace();
			}

			if(!(*(c->m_ptr))){
				c->setError(KVI_ERROR_PopupItemCommandExpected,"POPUP-ITEM-LABEL");
				return false;
			}

			const char * aux = c->m_ptr;
			if(!skipCommand(c))return false;

			KviStr buffer(aux,c->m_ptr);

/*
			// test -- should make no problems...
			// well testing if last char is '}' would be ql :)
			if (*aux == '{') {
				buffer.cutLeft(1);
				buffer.cutRight(1);
			}
*/
			m->insertNormalItem(itemLabel.ptr(),itemImage.ptr(),buffer.ptr());

			continue;
		}

		if(kvi_strEqualNoLocaleCIN("separator",c->m_ptr,9)){
			c->m_ptr+=9;
			c->skipSpace();
			if((*(c->m_ptr) == ':') || (*(c->m_ptr) == ';')){ //Allowed but not required
				++c->m_ptr;
			}
			m->insertSeparatorItem();
			continue;
		}
		if(kvi_strEqualNoLocaleCIN(c->m_ptr,"submenu",7)){
			c->m_ptr+=7;
			c->skipSpace();
			if(*(c->m_ptr) != '('){
				c->setError(KVI_ERROR_OpenParenthesisExpected,"POPUP-SUBMENU-LABEL",c->m_ptr);
				return false;
			}
			++c->m_ptr;
			KviStr itemLabel;
			if(!processFncSingleParam(c,itemLabel))return false;
//			itemLabel.stripWhiteSpace();

			KviStr itemImage;
			if(!processFncFinalPart(c,itemImage))return false;
//			itemImage.stripWhiteSpace();

			c->skipSpace();

			if(*(c->m_ptr) == ':'){ //Allowed but not required
				++c->m_ptr;
				c->skipSpace();
			}

			if(!(*(c->m_ptr))){
				c->setError(KVI_ERROR_PopupItemCommandExpected,"POPUP-SUBMENU-LABEL");
				return false;
			}

			KviStr yourName(KviStr::Format,"%s_sub%d",m->name(),idx);
			KviPopupMenu * menu = new KviPopupMenu(0,yourName.ptr());
			if(!generatePopupFromBuffer(menu,c,false)){
				delete menu;
				return false;
			}
			m->insertPopupItem(itemLabel.ptr(),itemImage.ptr(),menu);
			idx++;
			continue;
		}
		c->setError(KVI_ERROR_ItemPopupOrSeparatorExpected,"POPUP-GENERATOR",c->m_ptr);
		return false;
	}
}

//#warning "Set all KVIPOPUP MENU parents to 0!!"

bool KviUserParser::skipPopupBuffer(KviCommand *c)
{
	if(*(c->m_ptr) != '{'){ c->setError(KVI_ERROR_OpenBraceExpected,"POPUP-GENERATOR",c->m_ptr); return false; }
	++(c->m_ptr);
	for(;;){
		c->skipWhitespace();
		if(!(*(c->m_ptr))){ c->setError(KVI_ERROR_UnexpectedEndOfCommand,"POPUP-GENERATOR"); return false; }
		if(*(c->m_ptr) == '}'){ ++c->m_ptr; return true; }
		if(kvi_strEqualNoLocaleCIN(c->m_ptr,"item",4)){
			c->m_ptr+=4;
			c->skipSpace();
			if(*(c->m_ptr) != '('){ c->setError(KVI_ERROR_OpenParenthesisExpected,"POPUP-ITEM-LABEL",c->m_ptr); return false; }
			++c->m_ptr;
			if(!skipExpressionBody(c))return false;
			c->skipSpace();
			if(*(c->m_ptr) == ':'){ ++c->m_ptr; c->skipSpace(); }
			if(!(*(c->m_ptr))){ c->setError(KVI_ERROR_PopupItemCommandExpected,"POPUP-ITEM-LABEL"); return false; }
			if(!skipCommand(c))return false;
			continue;
		}
		if(kvi_strEqualNoLocaleCIN("separator",c->m_ptr,9)){
			c->m_ptr+=9;
			c->skipSpace();
			if((*(c->m_ptr) == ':') || (*(c->m_ptr) == ';')){ ++c->m_ptr; }
			continue;
		}
		if(kvi_strEqualNoLocaleCIN(c->m_ptr,"submenu",7)){
			c->m_ptr+=7;
			c->skipSpace();
			if(*(c->m_ptr) != '('){ c->setError(KVI_ERROR_OpenParenthesisExpected,"POPUP-SUBMENU-LABEL",c->m_ptr); return false; }
			++c->m_ptr;
			if(!skipExpressionBody(c))return false;
			c->skipSpace();
			if(*(c->m_ptr) == ':'){ ++c->m_ptr; c->skipSpace(); }
			if(!(*(c->m_ptr))){ c->setError(KVI_ERROR_PopupItemCommandExpected,"POPUP-SUBMENU-LABEL"); return false; }
			if(!skipPopupBuffer(c))return false;
			continue;
		}
		c->setError(KVI_ERROR_ItemPopupOrSeparatorExpected,"POPUP-GENERATOR",c->m_ptr);
		return false;
	}
}

#include "m_kvi_popupmenu.moc"
