//
//   File : kvi_ircsocket.cpp (/usr/build/NEW_kvirc/kvirc/kvirc/kvi_ircsocket.cpp)
//   Last major modification : Mon Jan 11 1999 19:56:39 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//#define _KVI_DEBUG_CLASS_NAME_ "KviIrcSocket"
//#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_debug.h"
#include "kvi_ircsocket.h"
#include "kvi_frame.h"
#include "kvi_error.h"
#include "kvi_options.h"
#include "kvi_sparser.h"
#include "kvi_systray.h"
#include "kvi_settings.h"

#include "kvi_memmove.h"
#include "kvi_malloc.h"

#include <unistd.h> //for close()
#include <fcntl.h>
#include <errno.h>
#include <netdb.h> //for gethostbyname()

// 500 ms should be enough
#define KVI_DATA_FLUSH_TIMEOUT 500

#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	#include "kvi_translate.h"
	extern KviCharsetTranslator * g_pCharsetTranslator;
#endif

//=== construct and destroy =================================================//

KviIrcSocket::KviIrcSocket(KviFrame *pFrm,KviServerParser *pPrs,KviSysTrayIoLed *pSysTrayIoLed)
{
	m_pFrm           = pFrm;
	m_pSysTrayIoLed  = pSysTrayIoLed;
	m_pServerParser  = pPrs;
	m_sock           = -1;
	m_pRsn           = 0;
	m_pWsn           = 0;
	m_error          = KVI_ERROR_NoError;
	m_state          = Ready;
	m_iReadBufLen    = 0;
	m_pReadBuffer    = 0;
	m_pSendQueueHead = 0;
	m_pSendQueueTail = 0;
	m_pFlushTimer    = new QTimer();
	reset_clearProxyState();
	clearInternalCounters();
	connect(m_pFlushTimer,SIGNAL(timeout()),this,SLOT(flushSendQueue()));
}

KviIrcSocket::~KviIrcSocket()
{
	reset();
	delete m_pFlushTimer;
}

//=== data queue functions ==================================================//
//
// queue_insertMessage : appends a KviIrcSocketMsgEntry to the tail of
//     the message queue. The next_ptr for this message is set to 0.
// queue_removeMessage : removes a message from the head of the queue.
//

void KviIrcSocket::queue_insertMessage(KviIrcSocketMsgEntry *msg_ptr)
{
	__range_valid(msg_ptr);
	__range_valid(msg_ptr->data_ptr);
	__range_valid(msg_ptr->data_len);
	msg_ptr->next_ptr = 0;
	if(m_pSendQueueHead){
		m_pSendQueueTail->next_ptr = msg_ptr;
		m_pSendQueueTail = msg_ptr;
	} else {
		m_pSendQueueHead = msg_ptr;
		m_pSendQueueTail = msg_ptr;
	}
}

bool KviIrcSocket::queue_removeMessage()
{
	__range_valid(m_pSendQueueTail);
	__range_valid(m_pSendQueueHead);
	if(m_pSendQueueHead->data_ptr)kvi_free((void *)m_pSendQueueHead->data_ptr);
	KviIrcSocketMsgEntry *aux_ptr = m_pSendQueueHead;
	m_pSendQueueHead = aux_ptr->next_ptr;
	kvi_free((void *)aux_ptr);
	if(m_pSendQueueHead == 0){
		m_pSendQueueTail = 0;
		return false;
	} else return true;
}

inline void KviIrcSocket::queue_removeAllMessages()
{ if(m_pSendQueueHead)while(queue_removeMessage()); }

//=== reset functions =======================================================//
//
// reset_clearSendQueue : removes all messages from the queue and 
//      stops the flush timer.
//

inline void KviIrcSocket::reset_killReadNotifier()
{ if(m_pRsn){ delete m_pRsn; m_pRsn = 0; } }

inline void KviIrcSocket::reset_killWriteNotifier()
{ if(m_pWsn){ delete m_pWsn; m_pWsn = 0; } }

inline void KviIrcSocket::reset_killSocket()
{ if(m_sock != -1){ close(m_sock); m_sock = -1; } }

inline void KviIrcSocket::reset_destroyReadBuffer()
{
	if(m_pReadBuffer)kvi_free(m_pReadBuffer);
	m_pReadBuffer = 0;
	m_iReadBufLen = 0;
}

void KviIrcSocket::reset_clearSendQueue()
{
	if(m_pFlushTimer->isActive())m_pFlushTimer->stop();
	queue_removeAllMessages();
}

void KviIrcSocket::reset_clearProxyState()
{
	m_proxy.ircHostPort = 0;
	m_proxy.szPassword   = "";
	m_proxy.szUsername   = "";
	m_proxy.state        = KVI_PROXYSTATE_INACTIVE;
}

void KviIrcSocket::reset()
{
	reset_killReadNotifier();
	reset_killWriteNotifier();
	reset_killSocket();
	reset_destroyReadBuffer();
	reset_clearSendQueue();
	reset_clearProxyState();
	m_state = Ready;
}

void KviIrcSocket::clearInternalCounters()
{
	m_uReadBytes   = 0;
	m_uSentBytes   = 0;
	m_uReadPackets = 0;
	m_uSentPackets = 0;
}

bool KviIrcSocket::connectTo(const char *szIp,const char *szPort,KviProxyStruct *prx,bool bIpV6)
{
	if(m_state != Ready){
		m_error = KVI_ERROR_AnotherConnectionInProgress;
		return false;
	}

	//We need a clean state
	__range_invalid(m_pRsn);
	__range_invalid(m_pWsn);
	__range_valid(m_sock < 0);
	reset_clearSendQueue();
	reset_clearProxyState();
	clearInternalCounters();
	m_error = KVI_ERROR_NoError;

#ifdef COMPILE_NEED_IPV6

	if(bIpV6)
	{
		struct in6_addr hostAddress;
		unsigned short int port;
		bool bOk;
	
	
		// check the proxy stuff...
//		if(prx){
//			// have to use a proxy
//			// check the proxy IP address
//			if(!kvi_stringIpToBinaryIp_V6(prx->szIp.ptr(),&hostAddress)){
//				m_error = KVI_ERROR_InvalidProxyAddress;
//				return false;
//			}
//			// check the irc host ip
//			if(!kvi_stringIpToBinaryIp_V6(szIp,&(m_proxy.irc6HostAddress))){
//				m_error = KVI_ERROR_InvalidIpAddress;
//				return false;
//			}
//			// check the proxy port now...
//			port = prx->szPort.toUShort(&bOk);
//			if(!bOk){
//				m_pFrm->socketEvent(KVI_SOCKEVENT_UsingDefaultPort1080);
//				port = 1080;
//			}
//			// and the irc host port
//			KviStr szP(szPort);
//			m_proxy.ircHostPort = szP.toUShort(&bOk);
//			if(!bOk){
//				m_pFrm->socketEvent(KVI_SOCKEVENT_UsingDefaultPort6667);
//				m_proxy.ircHostPort = 6667;
//			}
//			// remember the rest of proxy data...
//			m_proxy.szPassword = prx->szPassword;
//			m_proxy.szUsername = prx->szUsername;
//			m_proxy.bProtoV5   = prx->bProtoV5;
//			m_proxy.state      = KVI_PROXYSTATE_CONNECTING;
//			m_proxy.bUsingIpV6 = true;
//		} else {
			// get the address
		if(!kvi_stringIpToBinaryIp_V6(szIp,&hostAddress)){
			m_error = KVI_ERROR_InvalidIpAddress;
			return false;
		}	
		// check the port
		KviStr szP(szPort);
		port = szP.toUShort(&bOk);
		if(!bOk){
			m_pFrm->socketEvent(KVI_SOCKEVENT_UsingDefaultPort6667);
			port = 6667;
		}
	
		// create the socket
		m_sock = ::socket(PF_INET6,SOCK_STREAM,0);
		if(m_sock < 0){
			m_error = KVI_ERROR_SocketCreationFailed;
			return false;
		}

		if(g_pOptions->m_bBindIrcSocketToSpecificInterface)
		{
			struct in6_addr localHostAddress;
			if(kvi_stringIpToBinaryIp_V6(g_pOptions->m_szLocalHostIp.ptr(),&localHostAddress))
			{
				struct sockaddr_in6 localHostSockAddr;
				localHostSockAddr.sin6_family = AF_INET6;
				localHostSockAddr.sin6_flowinfo = 0;
				localHostSockAddr.sin6_port   = 0;
				localHostSockAddr.sin6_addr   = localHostAddress;
				if(bind(m_sock,(struct sockaddr *)&localHostSockAddr,sizeof(localHostSockAddr)) != 0)
				{
					debug("Can't bind socket to the specified address (bind() call failed) (IPV6) (disabling)");
					g_pOptions->m_bBindIrcSocketToSpecificInterface = false;
				}
				
			} else {
				g_pOptions->m_bBindIrcSocketToSpecificInterface = false;
				debug("Can't bind socket to the specified address (disabling) (IPV6)");
			}
		}

	
		// make it non blocking
		if(fcntl(m_sock,F_SETFL,O_NONBLOCK) < 0){
			reset_killSocket();
			m_error = KVI_ERROR_AsyncSocketFailed;
			return false;
		}
	
		// fill the sockaddr structure
		struct sockaddr_in6 hostSockAddr;
		hostSockAddr.sin6_family = AF_INET6;
		hostSockAddr.sin6_flowinfo = 0;
		hostSockAddr.sin6_port   = htons(port);
		hostSockAddr.sin6_addr   = hostAddress;
	
		// go!
		m_pFrm->socketEvent(KVI_SOCKEVENT_ContactingHost);
	
		if(::connect(m_sock,(struct sockaddr*)(&hostSockAddr),sizeof(hostSockAddr))<0){
			if(errno != EINPROGRESS){
				int sockError=errno;
				if(sockError==0){
					_this_should_be_socklen_t iSize=sizeof(int);
					if(getsockopt(m_sock,SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize)<0)sockError=0;
				}
				reset_killSocket();
				if(sockError)setErrorFromSystemError(sockError);
				else m_error = KVI_ERROR_UnknownError; //Error 0 ?
				return false;
			}
		}

		// and setup the WRITE notifier...
		m_pWsn = new QSocketNotifier(m_sock,QSocketNotifier::Write);
	
		QObject::connect(m_pWsn,SIGNAL(activated(int)),this,SLOT(slot_writeNotifierFired(int)));
	
		m_pWsn->setEnabled(true);
		// set the timer
		if(g_pOptions->m_iSocketTimeout < 5)g_pOptions->m_iSocketTimeout = 5;
		QTimer::singleShot(g_pOptions->m_iSocketTimeout * 1000,this,SLOT(timeout()));
		// and wait for connect
		m_state = Connecting;

		return true;
	}
#endif

	struct in_addr hostAddress;
	unsigned short int port;
	bool bOk;


	// check the proxy stuff...
	if(prx){
		// have to use a proxy
		// check the proxy IP address
		if(!kvi_stringIpToBinaryIp(prx->szIp.ptr(),&hostAddress)){
			m_error = KVI_ERROR_InvalidProxyAddress;
			return false;
		}
		// check the irc host ip
		if(!kvi_stringIpToBinaryIp(szIp,&m_proxy.ircHostAddress)){
			m_error = KVI_ERROR_InvalidIpAddress;
			return false;
		}
		// check the proxy port now...
		port = prx->szPort.toUShort(&bOk);
		if(!bOk){
			m_pFrm->socketEvent(KVI_SOCKEVENT_UsingDefaultPort1080);
			port = 1080;
		}
		// and the irc host port
		KviStr szP(szPort);
		m_proxy.ircHostPort = szP.toUShort(&bOk);
		if(!bOk){
			m_pFrm->socketEvent(KVI_SOCKEVENT_UsingDefaultPort6667);
			m_proxy.ircHostPort = 6667;
		}
		// remember the rest of proxy data...
		m_proxy.szPassword = prx->szPassword;
		m_proxy.szUsername = prx->szUsername;
		m_proxy.bProtoV5   = prx->bProtoV5;
		m_proxy.bProtoHttp = prx->bProtoHttp;
		m_proxy.state      = KVI_PROXYSTATE_CONNECTING;
	} else {
		// get the address
		if(!kvi_stringIpToBinaryIp(szIp,&hostAddress)){
			m_error = KVI_ERROR_InvalidIpAddress;
			return false;
		}	
		// check the port
		KviStr szP(szPort);
		port = szP.toUShort(&bOk);
		if(!bOk){
			m_pFrm->socketEvent(KVI_SOCKEVENT_UsingDefaultPort6667);
			port = 6667;
		}
	}

	// create the socket
	m_sock = ::socket(PF_INET,SOCK_STREAM,0);
	if(m_sock < 0){
		m_error = KVI_ERROR_SocketCreationFailed;
		return false;
	}

	if(g_pOptions->m_bBindIrcSocketToSpecificInterface)
	{
		struct in_addr localHostAddress;
		if(kvi_stringIpToBinaryIp(g_pOptions->m_szLocalHostIp.ptr(),&localHostAddress))
		{
			struct sockaddr_in localHostSockAddr;
			localHostSockAddr.sin_family = AF_INET;
			localHostSockAddr.sin_port   = 0;
			localHostSockAddr.sin_addr   = localHostAddress;
			if(bind(m_sock,(struct sockaddr *)&localHostSockAddr,sizeof(localHostSockAddr)) != 0)
			{
				debug("Can't bind socket to the specified address (bind() call failed) (IPV4) (disabling)");
				g_pOptions->m_bBindIrcSocketToSpecificInterface = false;
			}
			
		} else {
			g_pOptions->m_bBindIrcSocketToSpecificInterface = false;
			debug("Can't bind socket to the specified address (disabling) (IPV4)");
		}
	}

	// make it non blocking
	if(fcntl(m_sock,F_SETFL,O_NONBLOCK) < 0){
		reset_killSocket();
		m_error = KVI_ERROR_AsyncSocketFailed;
		return false;
	}

	// fill the sockaddr structure
	struct sockaddr_in hostSockAddr;
	hostSockAddr.sin_family = AF_INET;
	hostSockAddr.sin_port   = htons(port);
	hostSockAddr.sin_addr   = hostAddress;

	// go!
	m_pFrm->socketEvent(KVI_SOCKEVENT_ContactingHost);

	if(::connect(m_sock,(struct sockaddr*)(&hostSockAddr),sizeof(hostSockAddr))<0){
		if(errno != EINPROGRESS){
			int sockError=errno;
			if(sockError==0){
				_this_should_be_socklen_t iSize=sizeof(int);
				if(getsockopt(m_sock,SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize)<0)sockError=0;
			}
			reset_killSocket();
			if(sockError)setErrorFromSystemError(sockError);
			else m_error = KVI_ERROR_UnknownError; //Error 0 ?
			return false;
		}
	}

	// and setup the WRITE notifier...
	m_pWsn = new QSocketNotifier(m_sock,QSocketNotifier::Write);

	QObject::connect(m_pWsn,SIGNAL(activated(int)),this,SLOT(slot_writeNotifierFired(int)));

	m_pWsn->setEnabled(true);
	// set the timer
	if(g_pOptions->m_iSocketTimeout < 5)g_pOptions->m_iSocketTimeout = 5;
	QTimer::singleShot(g_pOptions->m_iSocketTimeout * 1000,this,SLOT(timeout()));
	// and wait for connect
	m_state = Connecting;
	return true;
}

//=== slot_writeNotifierFired ===============================================//
//
// This is the write notifier m_pWsn slot.
// It is enabled after the connect() call in connectTo()
// This function checks for connect errors.
//
void KviIrcSocket::slot_writeNotifierFired(int)
{
	// Sanyty check
	__enter("writeNotifierFired");
	__range_valid(m_state == Connecting);
	int sockError;
	_this_should_be_socklen_t iSize=sizeof(int);
	if(getsockopt(m_sock,SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize)<0)sockError = -1;
	if(sockError != 0){
		//failed
		if(sockError > 0)setErrorFromSystemError(sockError);
		else m_error = KVI_ERROR_UnknownError; //Error 0 ?
		m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
//#warning "is this safe ?"
		reset();
	} else {
		//Succesfully connected...
//#warning "is this safe ?"
		reset_killWriteNotifier();
		m_state = Connected;
		// create the correct read notifier now...
		// If we're using a proxy , we connect the read notifier
		// to a special slot that will handle the proxy login
		// and then dynamically change the slot to the
		// "normal" (irc) one.
		m_pRsn = new QSocketNotifier(m_sock,QSocketNotifier::Read);
		if(m_proxy.state == KVI_PROXYSTATE_CONNECTING){
			// proxy connection
			QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(slot_readProxyData(int)));
			m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectedToProxy);
			if(m_proxy.bProtoHttp)proxy_loginHttp();
			else if(m_proxy.bProtoV5)proxy_loginV5();
			else proxy_loginV4();
		} else {
			// normal irc connection
			QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(slot_readIrcData(int)));
			m_pFrm->socketEvent(KVI_SOCKEVENT_Connected);
		}
		m_pRsn->setEnabled(true);
	}
}

void KviIrcSocket::proxy_loginV4()
{
	__enter("proxy_loginV4");
	// SOCKSV4 protocol
	//
	// 1) CONNECT
	//
	// The client connects to the SOCKS server and sends a CONNECT request when
	// it wants to establish a connection to an application server. The client
	// includes in the request packet the IP address and the port number of the
	// destination host, and userid, in the following format.
	//
	//                +----+----+----+----+----+----+----+----+----+----+....+----+
	//                | VN | CD | DSTPORT |      DSTIP        | USERID       |NULL|
	//                +----+----+----+----+----+----+----+----+----+----+....+----+
	// # of bytes:       1    1      2              4           variable       1
	//
	// VN is the SOCKS protocol version number and should be 4. CD is the
	// SOCKS command code and should be 1 for CONNECT request. NULL is a byte
	// of all zero bits.
	//
	// The SOCKS server checks to see whether such a request should be granted
	// based on any combination of source IP address, destination IP address,
	// destination port number, the userid, and information it may obtain by
	// consulting IDENT, cf. RFC 1413.  If the request is granted, the SOCKS
	// server makes a connection to the specified port of the destination host.
	// A reply packet is sent to the client when this connection is established,
	// or when the request is rejected or the operation fails. 
	//
	m_pFrm->socketEvent(KVI_SOCKEVENT_UsingProxyProtoV4);
	m_proxy.szUsername.stripWhiteSpace();
	// spaces are allowed in a password ?
	// If yes it is senseless...
	m_proxy.szPassword.stripWhiteSpace();
	// the protocol does not specify the "userid" format...
	// so build an userid from the pass and/or username...
	KviStr szUserAndPass=m_proxy.szUsername;
	if(m_proxy.szPassword.hasData()){
		if(szUserAndPass.hasData())szUserAndPass+=" ";
		szUserAndPass+=m_proxy.szPassword;
	}
	int iLen = szUserAndPass.len()+9;
	// build the request packet
	char *bufToSend = new char[iLen];
	bufToSend[0]=(unsigned char)4; //Version 4
	bufToSend[1]=(unsigned char)1; //Connect
	Q_UINT16 port=(Q_UINT16)htons(m_proxy.ircHostPort);
	kvi_memmove((void *)(bufToSend+2),(void *)&port,2);
	Q_UINT32 host=(Q_UINT32)m_proxy.ircHostAddress.s_addr;
	kvi_memmove((void *)(bufToSend+4),(void *)&host,4);
	kvi_memmove((void *)(bufToSend+8),(void *)(szUserAndPass.ptr()),szUserAndPass.len());
	// send it into hyperspace...
	m_proxy.state = KVI_PROXYSTATE_V4FINAL;
	m_pFrm->socketEvent(KVI_SOCKEVENT_ProxySentV4Request);
	sendRawData(bufToSend,iLen);
	delete[] bufToSend;
	// and wait for reply...
}

void KviIrcSocket::proxy_loginV5()
{
	__enter("proxy_loginV5");
	// SOCKSV5 protocol.
	//
	// When a TCP-based client wishes to establish a connection to an object
	// that is reachable only via a firewall (such determination is left up
	// to the implementation), it must open a TCP connection to the
	// appropriate SOCKS port on the SOCKS server system.  The SOCKS service
	// is conventionally located on TCP port 1080.  If the connection
	// request succeeds, the client enters a negotiation for the
	// authentication method to be used, authenticates with the chosen
	// method, then sends a relay request.  The SOCKS server evaluates the
	// request, and either establishes the appropriate connection or denies
	// it.
	//
	// The client connects to the server, and sends a version
	// identifier/method selection message:
	//
	//                    +----+----------+----------+
	//                    |VER | NMETHODS | METHODS  |
	//                    +----+----------+----------+
	//                    | 1  |    1     | 1 to 255 |
	//                    +----+----------+----------+
	//
	// The VER field is set to X'05' for this version of the protocol.  The
	// NMETHODS field contains the number of method identifier octets that
	// appear in the METHODS field.
	// The values currently defined for METHOD are:
	//
	//      o  X'00' NO AUTHENTICATION REQUIRED
	//      o  X'01' GSSAPI
	//      o  X'02' USERNAME/PASSWORD
	//      o  X'03' CHAP
	//      o  X'04' to X'7F' IANA ASSIGNED
	//      o  X'80' to X'FE' RESERVED FOR PRIVATE METHODS
	//      o  X'FF' NO ACCEPTABLE METHODS
	//
	m_pFrm->socketEvent(KVI_SOCKEVENT_UsingProxyProtoV5);
	m_proxy.szUsername.stripWhiteSpace();
	// spaces are allowed in a password ?
	// If yes it is senseless...
	m_proxy.szPassword.stripWhiteSpace();
	char bufToSend[4];
	bufToSend[0]=(unsigned char)5; //use version 5
	int sendLen = 3;
	if(m_proxy.szUsername.isEmpty() || m_proxy.szUsername.isEmpty()){
		// no auth needed.
		bufToSend[1]=(unsigned char)1; //select one method
		bufToSend[2]=(unsigned char)0; //select method 0: no auth
	} else {
		// we can provide a password and username if needed...
		bufToSend[1]=(unsigned char)2; //select from two methods
		bufToSend[2]=(unsigned char)0; //method 0 or
		bufToSend[3]=(unsigned char)2; //method 2 username/pass auth
		sendLen = 4;
	}
	// notify the user before sending...since we may get disconnected
	m_pFrm->socketEvent(KVI_SOCKEVENT_ProxySentV5MethodRequest);
	m_proxy.state = KVI_PROXYSTATE_SELECTMETHOD;
	sendRawData(bufToSend,sendLen);
	// and wait for response
}

void KviIrcSocket::proxy_loginHttp()
{
	__enter("proxy_loginHttp");
	//   HTTP protocol.
	//
	//   The client connects to the HTTP proxy server, usually located on TCP
	//   port 8080, and uses the CONNECT method to specify the hostname and
	//   the port number to connect to. The hostname and port number are
	//   separated by a colon, and both of them must be specified. The
	//   host:port part is followed by a space and a string specifying the
	//   HTTP version number, e.g. HTTP/1.0, and the line terminator (CR-LF
	//   pair, or a single LF).
	//
	//   After that, there is a series of zero or more of HTTP request header
	//   lines, followed by an empty line. Each of those header lines is also
	//   terminated by the CR-LF pair, or a single LF. The empty line is simply
	//   another CR-LF pair, or another LF.
	//
	//   After the empty line, if the handshake to establish the connection
	//   was successful, IRC data transfer can begin.
	//
	m_pFrm->socketEvent(KVI_SOCKEVENT_UsingProxyProtoHttp);
	m_proxy.szUsername.stripWhiteSpace();
	// spaces shouldn't be allowed in a password
	m_proxy.szPassword.stripWhiteSpace();
	char bufToSend[128];
	KviStr hostAddress;
	kvi_binaryIpToString(m_proxy.ircHostAddress,hostAddress);
	int sendLen = snprintf(bufToSend, sizeof (bufToSend),
		"CONNECT %s:%d HTTP/1.0\r\n\r\n", hostAddress.ptr(),
		m_proxy.ircHostPort);
	//if(m_proxy.szUsername.isEmpty() || m_proxy.szUsername.isEmpty()){
	// notify the user before sending...since we may get disconnected
	m_pFrm->socketEvent(KVI_SOCKEVENT_ProxySentHttpConnectRequest);
	m_proxy.state = KVI_PROXYSTATE_HTTPHANDSHAKE;
	sendRawData(bufToSend,sendLen);
	// and wait for response
}

void KviIrcSocket::proxy_authUserPassV5()
{
	__enter("proxy_authUserPassV5");
	//   Once the SOCKS V5 server has started, and the client has selected the
	//   Username/Password Authentication protocol, the Username/Password
	//   subnegotiation begins.  This begins with the client producing a
	//   Username/Password request:
	//
	//           +----+------+----------+------+----------+
	//           |VER | ULEN |  UNAME   | PLEN |  PASSWD  |
	//           +----+------+----------+------+----------+
	//           | 1  |  1   | 1 to 255 |  1   | 1 to 255 |
	//           +----+------+----------+------+----------+
	//
	//   The VER field contains the current version of the subnegotiation,
	//   which is X'01'. The ULEN field contains the length of the UNAME field
	//   that follows. The UNAME field contains the username as known to the
	//   source operating system. The PLEN field contains the length of the
	//   PASSWD field that follows. The PASSWD field contains the password
	//   association with the given UNAME.
	//
	unsigned int lPass=(unsigned int)m_proxy.szPassword.len();
	if(lPass>255)lPass=255;
	unsigned int lUser=(unsigned int)m_proxy.szUsername.len();
	if(lUser>255)lUser=255;
	int iLen=lPass+lUser+3;
	char *bufToSend=new char[iLen];
	bufToSend[0]=(unsigned char)1;                           //version x'01'
	bufToSend[1]=(unsigned char)lUser;                       //length of the username
	kvi_memmove((void *)(bufToSend+2),(void *)m_proxy.szUsername.ptr(),lUser); //username
	bufToSend[2+lUser]=(unsigned char)lPass;             //length of the password
	kvi_memmove((void *)(bufToSend+3+lUser),(void *)m_proxy.szPassword.ptr(),lPass);
	// spit out the buffer and wait
	m_pFrm->socketEvent(KVI_SOCKEVENT_ProxySentUserAndPass);
	m_proxy.state = KVI_PROXYSTATE_WAITFORUSERANDPASS;
	sendRawData(bufToSend,iLen);
	delete[] bufToSend;
	// and wait for response...
}

void KviIrcSocket::proxy_sendTargetDataV5()
{
	__enter("proxy_sendTargetDataV5");
	//   Once the method-dependent subnegotiation has completed, the client
	//   sends the request details.  If the negotiated method includes
	//   encapsulation for purposes of integrity checking and/or
	//   confidentiality, these requests MUST be encapsulated in the method-
	//   dependent encapsulation.
	//
	//   The SOCKS request is formed as follows:
	//
	//           +----+-----+------+------+----------+----------+
	//           |VER | CMD | FLAG | ATYP | DST.ADDR | DST.PORT |
	//           +----+-----+------+------+----------+----------+
	//           | 1  |  1  |  1   |  1   | Variable |    2     |
	//           +----+-----+------+------+----------+----------+
	//
	//   Where:
	//
	//      o VER    protocol version: X'05'
	//      o CMD
	//         o CONNECT X'01'
	//         o BIND X'02'
	//         o UDP ASSOCIATE X'03'
	//         o  X'04' to X'7F' IANA ASSIGNED
	//         o  X'80' to X'FF' RESERVED FOR PRIVATE METHODS
	//      o FLAG   command dependent flag (defaults to X'00')
	//      o ATYP   address type of following address
	//        o IP V4 address: X'01'
	//        o DOMAINNAME: X'03'
	//        o IP V6 address: X'04'
	//      o DST.ADDR       desired destination address
	//      o DST.PORT desired destination port in network octet
	//         order
	//
	//      The SOCKS server will typically evaluate the request based on
	//      source and destination addresses, and return one or more reply
	//      messages, as appropriate for the request type.
	//
	//   In an address field (DST.ADDR, BND.ADDR), the ATYP field specifies
	//   the type of address contained within the field:
	//
	//             o  X'01'
	//
	//   The address is a version-4 IP address, with a length of 4 octets.
	//
	//             o  X'03'
	//
	//   The address field contains a fully-qualified domain name.  The first
	//   octet of the address field contains the number of octets of name that
	//   follow, there is no terminating NUL octet.
	//
	//             o  X'04'
	//
	//   The address is a version-6 IP address, with a length of 16 octets.
	char bufToSend[10];
	bufToSend[0]=(unsigned char)5;           //Proto 5
	bufToSend[1]=(unsigned char)1;           //CONNECT
	bufToSend[2]=(unsigned char)0;           //RSV
	bufToSend[3]=(unsigned char)1;           //IPV4
	Q_UINT32 host=(Q_UINT32)m_proxy.ircHostAddress.s_addr;
	kvi_memmove((void *)(bufToSend+4),(void *)&host,4);
	Q_UINT16 port=(Q_UINT16)htons(m_proxy.ircHostPort);
	kvi_memmove((void *)(bufToSend+8),(void *)&port,2);
	// send it into hyperspace...
	m_proxy.state = KVI_PROXYSTATE_V5FINAL;
	m_pFrm->socketEvent(KVI_SOCKEVENT_ProxySentTargetData);
	sendRawData(bufToSend,10);
	// and wait for reply...
}

//=== handleInvalidSocketRead ===============================================//
//
// Checks if the socket error is a transient error
// If it is not a transient error it resets the socket
// and fires the appropriate event.
// Otherwise it does nothing.
//
void KviIrcSocket::handleInvalidSocketRead(int readedLength,int eventToFireInCaseOfError)
{
	__range_valid(readedLength <= 0);
	if(readedLength==0){
		m_error = KVI_ERROR_RemoteEndClosedConnection;
		m_pFrm->socketEvent(eventToFireInCaseOfError);
//#warning "Is this safe ?"
		reset();
	} else {
		//check for transmission errors
		if((errno != EAGAIN) && (errno != EINTR)){
			if(errno > 0)setErrorFromSystemError(errno);
			else m_error = KVI_ERROR_RemoteEndClosedConnection;
			m_pFrm->socketEvent(eventToFireInCaseOfError);
//#warning "Is this safe ?"
			reset();
		} //else transient error...wait again...
	}
}

void KviIrcSocket::slot_readProxyData(int)
{
	char buffer[256];
	int readLength;
	if(m_proxy.bProtoHttp)readLength = slot_readLine(buffer,256);
	else readLength = read(m_sock,buffer,256);
	// check for errors..
	if(readLength <= 0){
		handleInvalidSocketRead(readLength,KVI_SOCKEVENT_ConnectionFailed);
		return;
	}
	// we need at least two bytes...
	if(readLength < 2){
		// a single byte of reply means:
		// - connection through a 1 bps modem
		// - a totally blocked network
		// - remote host is not a SOCKS server
		// Anyway....it is always a meaningless reply
		// better to try again later :)
		m_error = KVI_ERROR_UnrecognizedProxyReply;
		m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
		reset();
		return;
	}
	// handle the reply
	switch(m_proxy.state){
		case KVI_PROXYSTATE_V4FINAL:
			//V4 final reply
			proxy_handleV4FinalReply((unsigned char)buffer[1]);
			break;
		case KVI_PROXYSTATE_SELECTMETHOD:
			//V5 method selection reply
			proxy_handleV5MethodReply((unsigned char)buffer[1]);
			break;
		case KVI_PROXYSTATE_WAITFORUSERANDPASS:
			//V5 user and pass reply
			proxy_handleV5AuthReply((unsigned char)buffer[1]);
			break;
		case KVI_PROXYSTATE_V5FINAL:
			//V5 final reply
			proxy_handleV5FinalReply((unsigned char)buffer[1]);
			break;
		case KVI_PROXYSTATE_HTTPHANDSHAKE:
			//HTTP CONNECT reply.
			proxy_handleHttpHandshake(buffer);
			break;
		default:
			// meaningless reply...maybe the remote host is not a SOCKS server ?
			m_error = KVI_ERROR_UnrecognizedProxyReply;
			m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
			reset();
			break;
	}
}

int KviIrcSocket::slot_readLine(char *buffer, int bufferSize)
{
	int len = 0;
	while(1){
		if (read (m_sock, &buffer[len], 1) < 1)
			return -1;
		if (buffer[len] == '\n' || len >= bufferSize - 1){
			buffer[len] = 0;
			len++;
			return len;
		}
		len++;
	}
}

void KviIrcSocket::proxy_handleV5AuthReply(unsigned char reply)
{
	//   The server verifies the supplied UNAME and PASSWD, and sends the
	//   following response:
	//
	//                        +----+--------+
	//                        |VER | STATUS |
	//                        +----+--------+
	//                        | 1  |   1    |
	//                        +----+--------+
	//
	//   A STATUS field of X'00' indicates success. If the server returns a
	//   `failure' (STATUS value other than X'00') status, it MUST close the
	//   connection.
	//
	if(reply == 0){
		m_pFrm->socketEvent(KVI_SOCKEVENT_ProxyAuthOK);
		proxy_sendTargetDataV5();
		return;
	}
	m_error = KVI_ERROR_ProxyAuthFailed;
	m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
	reset();
}

void KviIrcSocket::proxy_handleV5MethodReply(unsigned char reply)
{
	//   The server selects from one of the methods given in METHODS, and
	//   sends a METHOD selection message:
	//
	//                            +----+--------+
	//                            |VER | METHOD |
	//                            +----+--------+
	//                            | 1  |   1    |
	//                            +----+--------+
	//
	//   If the selected METHOD is X'FF', none of the methods listed by the
	//   client are acceptable, and the client MUST close the connection.
	//
	// The values currently defined for METHOD are:
	//
	//      o  X'00' NO AUTHENTICATION REQUIRED
	//      o  X'01' GSSAPI
	//      o  X'02' USERNAME/PASSWORD
	//      o  X'03' CHAP
	//      o  X'04' to X'7F' IANA ASSIGNED
	//      o  X'80' to X'FE' RESERVED FOR PRIVATE METHODS
	//      o  X'FF' NO ACCEPTABLE METHODS
	//
	if(reply == 0){
		m_pFrm->socketEvent(KVI_SOCKEVENT_ProxyMethodSelectedNoAuth);
		proxy_sendTargetDataV5();
		return;
	}
	if(reply == 2){
		m_pFrm->socketEvent(KVI_SOCKEVENT_ProxyMethodSelectedUserPass);
		proxy_authUserPassV5();
		return;
	}
	//Request rejected
	if(reply == 0xFF){
		m_error = KVI_ERROR_ProxyNoAcceptableMethods;
		m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
		reset();
	} else {
		// unrecognized...
		m_error = KVI_ERROR_UnrecognizedProxyReply;
		m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
		reset();
	}
}

void KviIrcSocket::proxy_handleV5FinalReply(unsigned char reply)
{
	//
	//   The SOCKS request information is sent by the client as soon as it has
	//   established a connection to the SOCKS server, and completed the
	//   authentication negotiations.  The server evaluates the request, and
	//   returns a reply formed as follows:
	//
	//           +----+-----+------+------+----------+----------+
	//           |VER | REP | FLAG | ATYP | BND.ADDR | BND.PORT |
	//           +----+-----+------+------+----------+----------+
	//           | 1  |  1  |  1   |  1   | Variable |    2     |
	//           +----+-----+------+------+----------+----------+
	//
	//   Where:
	//             o  VER    protocol version: X'05'
	//             o  REP    Reply field:
	//                o  X'00' succeeded
	//                o  X'01' general SOCKS server failure
	//                o  X'02' connection not allowed by ruleset
	//                o  X'03' Network unreachable
	//                o  X'04' Host unreachable
	//                o  X'05' Connection refused
	//                o  X'06' TTL expired
	//                o  X'07' Command not supported
	//                o  X'08' Address type not supported
	//                o  X'09' Invalid address
	//                o  X'0A' to X'FF' unassigned
	//             o  FLAG   command dependent flag
	//             o  ATYP   address type of following address
	//                o  IP V4 address: X'01'
	//                o  DOMAINNAME: X'03'
	//                o  IP V6 address: X'04'
	//             o  BND.ADDR       server bound address
	//             o  BND.PORT       server bound port in network octet order
	//
	if(reply==0){
		// Request granted
		m_pFrm->socketEvent(KVI_SOCKEVENT_ProxyReply00Success);
		// change the slot....
		QObject::disconnect(m_pRsn,SIGNAL(activated(int)),this,SLOT(slot_readProxyData(int)));
		QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(slot_readIrcData(int)));
		m_pFrm->socketEvent(KVI_SOCKEVENT_Connected);	
	} else {
		//Request rejected
		switch(reply){
			case 1: m_error = KVI_ERROR_ProxyReply01GeneralSOCKSFailure; break;
			case 2: m_error = KVI_ERROR_ProxyReply02ConnectionNotAllowed; break;
			case 3: m_error = KVI_ERROR_ProxyReply03NetworkUnreachable; break;
			case 4: m_error = KVI_ERROR_ProxyReply04HostUnreachable; break;
			case 5: m_error = KVI_ERROR_ProxyReply05ConnectionRefused; break;
			case 6: m_error = KVI_ERROR_ProxyReply06TTLExpired; break;
			case 7: m_error = KVI_ERROR_ProxyReply07CommandNotSupported; break;
			case 8: m_error = KVI_ERROR_ProxyReply08AddressTypeNotSupported; break;
			case 9: m_error = KVI_ERROR_ProxyReply09InvalidAddress; break;
			default: m_error = KVI_ERROR_UnrecognizedProxyReply; break;
		}
		m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
		reset();
	}
}

void KviIrcSocket::proxy_handleV4FinalReply(unsigned char reply)
{
	// If the request is granted, the SOCKS
	// server makes a connection to the specified port of the destination host.
	// A reply packet is sent to the client when this connection is established,
	// or when the request is rejected or the operation fails. 
	//
	//
	//                +----+----+----+----+----+----+----+----+
	//                | VN | CD | DSTPORT |      DSTIP        |
	//                +----+----+----+----+----+----+----+----+
	// # of bytes:       1    1      2              4
	//
	// VN is the version of the reply code and should be 0. CD is the result
	// code with one of the following values:
	//
	//        90: request granted
	//        91: request rejected or failed
	//        92: request rejected becasue SOCKS server cannot connect to
	//            identd on the client
	//        93: request rejected because the client program and identd
	//            report different user-ids
	//
	// The remaining fields are ignored.
	//
	// The SOCKS server closes its connection immediately after notifying
	// the client of a failed or rejected request. For a successful request,
	// the SOCKS server gets ready to relay traffic on both directions. This
	// enables the client to do I/O on its connection as if it were directly
	// connected to the application server.
	if(reply==90){
		// Request granted
		m_pFrm->socketEvent(KVI_SOCKEVENT_ProxyReply90RequestGranted);
		// change the slot....
		QObject::disconnect(m_pRsn,SIGNAL(activated(int)),this,SLOT(slot_readProxyData(int)));
		QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(slot_readIrcData(int)));
		m_pFrm->socketEvent(KVI_SOCKEVENT_Connected);
	} else {
		//Request rejected
		switch(reply){
			case 91: m_error = KVI_ERROR_ProxyReply91RequestFailed; break;
			case 92: m_error = KVI_ERROR_ProxyReply92IdentFailed; break;
			case 93: m_error = KVI_ERROR_ProxyReply93IdentNotMatching; break;
			default: m_error = KVI_ERROR_UnrecognizedProxyReply; break;
		}
		m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
		reset();
	}
	// Just looked out of the window...
	// Hmmmm...strange light outside...
	// Looked at the clock...6:34 !
	// I think I'll go sleep.... :)
}

void KviIrcSocket::proxy_handleHttpHandshake(char *reply)
{
	// Proxy response.
	// After the empty line in the request, the client will wait for a response
	// from the proxy. The proxy will evaluate the request, make sure that it
	// is valid, and that the user is authorized to request such a connection.
	// If everything is in order, the proxy will make a connection to the
	// destination server, and, if successful, send a "200 Connection
	// established" response to the client. Again, the response follows the
	// HTTP/1.0 protocol, so the response line starts with the protocol version
	// specifier, and the response line is followed by zero or more response
	// headers, followed by an empty line. The line separator is CR-LF pair, or
	// a single LF.
	//
	if((qstrlen(reply) > 11)&&(!memcmp(reply, "HTTP/", 5))&&(!memcmp(reply+9, "200", 3)))
	{
		m_pFrm->socketEvent(KVI_SOCKEVENT_ProxyReplyHttp200Success);
		for (;;){
			//read until blank line
			slot_readLine(reply,sizeof(reply));
			if (!reply[0] || (reply[0] == '\r' && !reply[1]))
			break;
		}
		//change the slot
		QObject::disconnect(m_pRsn,SIGNAL(activated(int)),this,SLOT(slot_readProxyData(int)));
		QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(slot_readIrcData(int)));
		m_pFrm->socketEvent(KVI_SOCKEVENT_Connected);
	} else {
		m_error = KVI_ERROR_UnrecognizedProxyReply;
		m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
		reset();
	}
}

void KviIrcSocket::slot_readIrcData(int)
{
	//read data
	char buffer[1025];
	int readLength = read(m_sock,buffer,1024);
	if(readLength <= 0){
		handleInvalidSocketRead(readLength,KVI_SOCKEVENT_Disconnected);
		return;
	}
	m_uReadBytes += readLength;
	//terminate our buffer
	(*(buffer+readLength))='\0';
	register char *p=buffer;
	char *beginOfCurData = buffer;
	int   bufLen = 0;
	char *messageBuffer = (char *)kvi_malloc(1);
	//Shut up the socket notifier
	//in case that we enter in a local loop somewhere
	//while processing data...
	m_pRsn->setEnabled(false);
	while(*p){
		if((*p == '\r' )||(*p == '\n')){
			//found a CR or LF...
			//prepare a message buffer
			bufLen = p - beginOfCurData;
			//check for previous unterminated data
			if(m_iReadBufLen > 0){
				__range_valid(m_pReadBuffer);
				messageBuffer = (char *)kvi_realloc(messageBuffer,bufLen + m_iReadBufLen + 1);
				kvi_memmove(messageBuffer,m_pReadBuffer,m_iReadBufLen);
				kvi_memmove((void *)(messageBuffer + m_iReadBufLen),beginOfCurData,bufLen);
				*(messageBuffer + bufLen + m_iReadBufLen) = '\0';
				m_iReadBufLen = 0;
				kvi_free(m_pReadBuffer);
				m_pReadBuffer = 0;
			} else {
				__range_invalid(m_pReadBuffer);
				messageBuffer = (char *)kvi_realloc(messageBuffer,bufLen + 1);
				kvi_memmove(messageBuffer,beginOfCurData,bufLen);
				*(messageBuffer + bufLen) = '\0';
			}
			m_uReadPackets++;
			m_pSysTrayIoLed->dataMessageReceived();
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
			if(g_pOptions->m_bUseCharsetTranslation)
			{
				g_pCharsetTranslator->translateToClient(messageBuffer);
			}
#endif
			m_pServerParser->parseMessage(messageBuffer);
			while(*p && ((*p=='\r')||(*p=='\n')) )p++;
			beginOfCurData = p;
		} else p++;
	}
	//now *p == '\0'
	//beginOfCurData points to '\0' if we have
	//no more stuff to parse , or points to something
	//different than '\r' or '\n'...
	if(*beginOfCurData){
		//Have remaining data...in the local buffer
		bufLen = p - beginOfCurData;
		if(m_iReadBufLen > 0){
			//and there was more stuff saved... (really slow connection)
			__range_valid(m_pReadBuffer);
			m_pReadBuffer =(char *)kvi_realloc(m_pReadBuffer,m_iReadBufLen + bufLen);
			kvi_memmove((void *)(m_pReadBuffer+m_iReadBufLen),beginOfCurData,bufLen);
			m_iReadBufLen += bufLen;
		} else {
			//
			__range_invalid(m_pReadBuffer);
			m_iReadBufLen = bufLen;
			m_pReadBuffer =(char *)kvi_malloc(m_iReadBufLen);
			kvi_memmove(m_pReadBuffer,beginOfCurData,m_iReadBufLen);
		}
		//The m_pReadBuffer contains at max 1 irc message...
		//that can not be longer than 510 bytes (the message is not CRLF terminated)
		if(m_iReadBufLen > 510)debug("WARNING : Receiving an invalid irc message from server.");
	}
	kvi_free(messageBuffer);
	//ready to receive
	m_pRsn->setEnabled(true);
}


//=== Message send stuff ====================================================//
// Max buffer that can be sent to an IRC server is 512 bytes
// including CRLF. (ircd simply 'cuts' messages to 512 bytes
// and discards the remainig part)
// Note that 510 bytes of data is a reasonably long message :)
//
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 0123456789\r\n
//
// We keep a list of data to send , and flush it as soon as we can.
//
bool KviIrcSocket::sendFmtData(const char *fmt,...)
{
	if(m_state == Connected){
		//new buffer	
		KviIrcSocketMsgEntry *ptr = (KviIrcSocketMsgEntry *)kvi_malloc(sizeof(KviIrcSocketMsgEntry));
		ptr->data_ptr = (char *)kvi_malloc(512);
		va_list(list);
		va_start(list,fmt);
		bool bTruncated;
		//sprintf the buffer up to 512 chars (adds a CRLF too)
		ptr->data_len = kvi_irc_vsnprintf(ptr->data_ptr,fmt,list,&bTruncated);
		va_end(list);
		//adjust the buffer size
		if(ptr->data_len < 512)ptr->data_ptr = (char *)kvi_realloc(ptr->data_ptr,ptr->data_len);
		if(bTruncated)m_pFrm->socketEvent(KVI_SOCKEVENT_MessageTruncated);
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
		if(g_pOptions->m_bUseCharsetTranslation)
		{
			g_pCharsetTranslator->translateToServer(ptr->data_ptr,ptr->data_len);
		}
#endif
		queue_insertMessage(ptr);
		flushSendQueue();
		return true;
	}
	m_error = KVI_ERROR_NotConnectedToServer;
	return false; //not connected
}

bool KviIrcSocket::sendRawData(const char *buffer,int buflen)
{
	if(m_state == Connected){
		//new buffer	
		KviIrcSocketMsgEntry *ptr = (KviIrcSocketMsgEntry *)kvi_malloc(sizeof(KviIrcSocketMsgEntry));
		ptr->data_ptr = (char *)kvi_malloc(buflen);
		ptr->data_len = buflen;
		kvi_memmove(ptr->data_ptr,buffer,buflen);
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
		if(g_pOptions->m_bUseCharsetTranslation)
		{
			g_pCharsetTranslator->translateToServer(ptr->data_ptr,ptr->data_len);
		}
#endif
		queue_insertMessage(ptr);
		flushSendQueue();
		return true;
	}
	m_error = KVI_ERROR_NotConnectedToServer;
	return false; //not connected
}

bool KviIrcSocket::sendData(const char *buffer,int buflen)
{
	if(m_state == Connected){
		//new buffer	
		KviIrcSocketMsgEntry *ptr = (KviIrcSocketMsgEntry *)kvi_malloc(sizeof(KviIrcSocketMsgEntry));
		if(buflen < 0)buflen = strlen(buffer);
		if(buflen > 510){
			buflen = 510;
			m_pFrm->socketEvent(KVI_SOCKEVENT_MessageTruncated);
		}
		ptr->data_ptr = (char *)kvi_malloc(buflen+2);
		ptr->data_len = buflen+2;
		kvi_memmove(ptr->data_ptr,buffer,buflen);
		*(ptr->data_ptr+buflen)='\r';
		*(ptr->data_ptr+buflen+1)='\n';
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
		if(g_pOptions->m_bUseCharsetTranslation)
		{
			g_pCharsetTranslator->translateToServer(ptr->data_ptr,ptr->data_len);
		}
#endif
		queue_insertMessage(ptr);
		flushSendQueue();
		return true;
	}
	m_error = KVI_ERROR_NotConnectedToServer;
	return false; //not connected
}

//=== flushSendQueue ========================================================//
//
// Attempts to send as much as possible to the server
// If fails (happens only on really lagged servers)
// calls itself with a QTimer shot after KVI_DATA_FLUSH_TIMEOUT ms
// to retry again...
//
void KviIrcSocket::flushSendQueue()
{
	// If we're called from the flush timer , stop it
	if(m_pFlushTimer->isActive())m_pFlushTimer->stop();
	if(!m_pSendQueueHead)return;
	// Ok...have something to send...
	if(m_state == Connected){
		while(m_pSendQueueHead){
			// Write one data buffer...
			int result = write(m_sock,m_pSendQueueHead->data_ptr,m_pSendQueueHead->data_len);
			if(result == m_pSendQueueHead->data_len){
				// Succesfull send...remove this data buffer
				m_uSentPackets++;
				m_uSentBytes += result;
				queue_removeMessage();
				m_pSysTrayIoLed->dataMessageSent();
				// And try next buffer...
				continue;
			} else {
				// Something wrong ?
				if(result >= 0){
					// Partial send...need to finish it later
					m_pSendQueueHead->data_len -= result;
					m_pSendQueueHead->data_ptr = (char *)kvi_realloc(m_pSendQueueHead->data_ptr,m_pSendQueueHead->data_len);
					m_uSentBytes += result;
					m_pFrm->socketEvent(KVI_SOCKEVENT_PartialSocketWrite);
					// Async continue...
					m_pFlushTimer->start(KVI_DATA_FLUSH_TIMEOUT);
					return;
				} else {
					// Oops...error ?
					if((errno == EAGAIN)||(errno == EINTR)){
						// Transient error...partial send as before...
						m_pFrm->socketEvent(KVI_SOCKEVENT_PartialSocketWrite);
						// Async continue...
						m_pFlushTimer->start(KVI_DATA_FLUSH_TIMEOUT);
						return;
					} else {
						// Disconnected... :(
						setErrorFromSystemError(errno);
						m_pFrm->socketEvent(KVI_SOCKEVENT_Disconnected);
						reset();
						return;
					}
				}
			}
		}
		//flushed completly ...
	} else debug("WARNING : flushSendQueue called with no irc connection");
}

void KviIrcSocket::timeout()
{
	if(m_state == Connecting){
		m_error = KVI_ERROR_ConnectionTimedOut;
		m_pFrm->socketEvent(KVI_SOCKEVENT_ConnectionFailed);
		reset();
	}
}

void KviIrcSocket::setErrorFromSystemError(int errorNum)
{
	switch(errorNum){
		case EBADF:        m_error = KVI_ERROR_BadFileDescriptor;     break;
		case EFAULT:       m_error = KVI_ERROR_OutOfAddressSpace;     break;
		case ECONNREFUSED: m_error = KVI_ERROR_ConnectionRefused;     break;
		case ENOTSOCK:     m_error = KVI_ERROR_KernelNetworkingPanic; break;
		case ETIMEDOUT:    m_error = KVI_ERROR_ConnectionTimedOut;    break;
		case ENETUNREACH:  m_error = KVI_ERROR_NetworkUnreachable;    break;
		case EPIPE:        m_error = KVI_ERROR_BrokenPipe;            break;
		// Unhandled error...pass errno to the strerror function
		default:           m_error = -errorNum;                       break;
	}
}

bool KviIrcSocket::getHostByName(const char *host,KviStr &szIp,bool bIpV6)
{
	//Blocking DNS call , to use only if all other systems fail...
#ifdef COMPILE_NEED_IPV6
	if(bIpV6)
	{
//		struct in6_addr in6Addr;
		struct addrinfo * pRet;
		struct addrinfo hints;
		hints.ai_flags = 0;
		hints.ai_family = PF_INET6;
		hints.ai_socktype = SOCK_STREAM;
		hints.ai_addrlen = 0;
		hints.ai_canonname = 0;
		hints.ai_addr = 0;
		hints.ai_next = 0;
		int retVal = getaddrinfo(host,0,&hints,&pRet);
		if(retVal != 0)
		{
			switch(retVal)
			{

				case EAI_ADDRFAMILY: m_error = KVI_ERROR_HostNotFound; break;
				case EAI_NODATA:     m_error = KVI_ERROR_ValidNameButNoIpAddress; break;
				case EAI_FAIL:       m_error = KVI_ERROR_UnrecoverableNameserverError; break;
				case EAI_AGAIN:      m_error = KVI_ERROR_DnsTemporaneousFault; break;
				//EAI_ADDRFAMILY  address family for hostname not supported
				//EAI_BADFLAGS    invalid value for ai_flags
				//EAI_FAMILY      ai_family not supported
				//EAI_MEMORY      memory allocation failure
				//EAI_NONAME      hostname nor servname provided, or not known
				//EAI_SERVICE     servname not supported for ai_socktype
				//EAI_SOCKTYPE    ai_socktype not supported				
				//EAI_SYSTEM      system error returned in errno
				default:             m_error = KVI_ERROR_DnsQueryFailed; break;
			}
		} else kvi_binaryIpToString_V6(((sockaddr_in6 *)(pRet->ai_addr))->sin6_addr,szIp);
		freeaddrinfo(pRet);
		return (m_error == KVI_ERROR_Success);
	}
#endif

	struct hostent *pHostInfo = gethostbyname( host );
	if(!pHostInfo){
		switch(h_errno){
			case HOST_NOT_FOUND: m_error = KVI_ERROR_HostNotFound;                 break;
			case NO_ADDRESS:     m_error = KVI_ERROR_ValidNameButNoIpAddress;      break;
			case NO_RECOVERY:    m_error = KVI_ERROR_UnrecoverableNameserverError; break;
			case TRY_AGAIN:      m_error = KVI_ERROR_DnsTemporaneousFault;         break;
			default:             m_error = KVI_ERROR_DnsQueryFailed;               break;
		}
		return false;
	}
	if(!kvi_binaryIpToString(*(struct in_addr *)pHostInfo->h_addr,szIp)){
		m_error = KVI_ERROR_DnsQueryFailed;
		return false;
	}
	return true;
}

unsigned long KviIrcSocket::getSockAddress(){
	if(m_state != Connected){
		m_error = KVI_ERROR_NotConnectedToServer;
		return 0;
	}
	struct sockaddr_in name;
	_this_should_be_socklen_t len = sizeof(name);
//	getsockname(m_sock, (struct sockaddr *)&name,&len);
	if(getsockname(m_sock, (struct sockaddr *)&name,&len)<0){
		m_error = KVI_ERROR_GetsocknameFailed;
		return 0;
	}
	//I assume that getsockname returns data in Network byte order...
	//The man page misses to specify that...
	//htonl ???
	//190599 : Experience teaches us...
	// getsockname returns the address in network byte order.
	return name.sin_addr.s_addr;
}

bool KviIrcSocket::getLocalHostIp(KviStr &szIp,bool bIpV6){
	if(m_state != Connected){
		m_error = KVI_ERROR_NotConnectedToServer;
		return false;
	}
#ifdef COMPILE_NEED_IPV6
	if(bIpV6)
	{
		struct sockaddr_in6 name;
		_this_should_be_socklen_t len = sizeof(name);
		if(getsockname(m_sock, (struct sockaddr *)&name,&len)<0){
			m_error = KVI_ERROR_GetsocknameFailed;
			return false;
		}
		//I assume that getsockname returns data in Network byte order...
		//The man page misses to specify that...
		if(!kvi_binaryIpToString_V6(name.sin6_addr,szIp)){
			m_error = KVI_ERROR_GetsocknameFailed;
			return false;
		}
		return true;
	}
#endif
	struct sockaddr_in name;
	_this_should_be_socklen_t len = sizeof(name);
	if(getsockname(m_sock, (struct sockaddr *)&name,&len)<0){
		m_error = KVI_ERROR_GetsocknameFailed;
		return false;
	}
	//I assume that getsockname returns data in Network byte order...
	//The man page misses to specify that...
	if(!kvi_binaryIpToString(name.sin_addr,szIp)){
		m_error = KVI_ERROR_GetsocknameFailed;
		return false;
	}
	return true;
}

#include "m_kvi_ircsocket.moc"
