//
//   File : kvi_expression.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_expression.cpp)
//   Last major modification : Sat Feb 27 1999 19:31:10 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_app.h"
#include "kvi_uparser.h"
#include "kvi_ircsocket.h"
#include "kvi_console.h"
#include "kvi_defines.h"
#include "kvi_frame.h"
#include "kvi_options.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_exprtree.h"


bool KviUserParser::evaluateExpr(KviCommand *c,long * result)
{
	// This one evaluates an expression
	KviExprTree * root = evaluateExpression(c);
	if(!root)return false;
	if(*(c->m_ptr) != ')'){
		c->setError(KVI_ERROR_ParenthesisMismatch,_i18n_("EXPRESSION-END"),c->m_ptr);
		return false;
	}
	*result = root->calculate(c);
	delete root;
	++(c->m_ptr);
	return (!c->hasError());
}

KviExprTree * KviUserParser::evaluateExpression(KviCommand *c)
{
	// This one evaluates the expression up to a (not matching) ')' or a \0
	KviExprTree * op = evaluateOperand(c);
	if(!op)return 0;
	c->skipSpace();
	if((*(c->m_ptr) == ')')||(*(c->m_ptr) == '\0'))return op;//single operand
	KviExprTree * curOper = evaluateOperator(c);
	if(!curOper){
		delete op;
		return 0;
	}
	curOper->setLeft(op);
	return evaluateExpression_RightOperand(curOper,c);
}

KviExprTree * KviUserParser::evaluateOperand(KviCommand *c)
{
	// This one tries to extract a numeric operand from the data buffer
	// Moves m_ptr to the character after the last char of the operand
	// Returns a valid KviExprTreeNode of succesful 
	// otherwise returns 0 and sets c->m_err appropriately
	c->skipSpace();

	KviExprTree * pTop = 0;
	KviExprTree * pCur = 0;
	KviExprTree * pLast = 0;

	// Check for unary ops...
	while(*(c->m_ptr) == '-' || *(c->m_ptr) == '~' || *(c->m_ptr) == '!'){
		// Unary op
		switch(*(c->m_ptr)){
			case '-':
				pCur = new KviExprTree(KviExprTree::UnaryOperator,OP_NEGATE);
			break;
			case '~':
				pCur = new KviExprTree(KviExprTree::UnaryOperator,OP_BITNOT);
			break;
			case '!':
				pCur = new KviExprTree(KviExprTree::UnaryOperator,OP_NOT);
			break;
		}
		// And we work with the child
		if(pTop == 0)pTop = pCur;
		else pLast->setLeft(pCur);
		pLast = pCur;
		++(c->m_ptr);
		c->skipSpace();
	}

	if(*(c->m_ptr) == '('){
		// subexpression
		++(c->m_ptr);
		pCur = evaluateExpression(c);
		if(!pCur){                              // failed somewhere in the call
			if(pTop)delete pTop;
			return 0;
		}
		if(*(c->m_ptr) != ')'){                         // failed here
			if(pTop)delete pTop;
			delete pCur;
			c->setError(KVI_ERROR_ParenthesisMismatch,_i18n_("SUBEXPRESSION"));
			return 0;
		}
		++(c->m_ptr);
	} else {
		// here *m_ptr may be something that is not a digit too!
		pCur = evaluateSimpleOperand(c);
		if(!pCur){                            // failed somewere
			if(pTop)delete pTop;
			return 0;
		}
	}
	if(pTop)pLast->setLeft(pCur);
	else return pCur;
	return pTop;
}

KviExprTree * KviUserParser::evaluateStringOperand(KviCommand *c)
{
	__range_valid(*(c->m_ptr) == '"');
	KviExprTree * node = new KviExprTree(KviExprTree::StringOperand,0);
	const char *aux_ptr = ++(c->m_ptr);

	for(;;){
		// Now skip all the non interesting chars
		while(*aux_ptr && (*aux_ptr != '\\') && (*aux_ptr != '\n') &&
				(*aux_ptr != '"') && (*aux_ptr != '$') && (*aux_ptr != KVI_GLOBAL_VAR_CHAR))++aux_ptr;
		// Interesting char
		switch(*aux_ptr){
			case '\n':
			case '\0': // end of command buffer...append the last block to the buffer and return
				c->setError(KVI_ERROR_UnexpectedEndInString,_i18n_("EXPRESSION-STRING-OPERAND"));
				delete node;
				return 0;
			break;
			case '"' : // append the last block to the buffer and return
				node->m_string.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr; //don't forget to point to next char
				return node;
			break;
			case KVI_GLOBAL_VAR_CHAR: //variable: append the last block to the buffer and process the var
				node->m_string.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processVariable(c,node->m_string)){
					delete node;
					return 0;
				}
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				node->m_string.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!processIdentifier(c,node->m_string)){
					delete node;
					return 0;
				}
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				node->m_string.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						c->setError(KVI_ERROR_UnexpectedEndInString,_i18n_("EXPRESSION-STRING-OPERAND"));
						delete node;
						return 0;
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					default: // Must be copied to the buffer without modification
						node->m_string.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	return 0; //Newer here
}

KviExprTree * KviUserParser::evaluateSimpleOperand(KviCommand *c)
{
	// This one extracts a number from the data buffer
	KviStr tempNum;
	const char *aux =  c->m_ptr;
	switch(*aux){
		case '%': if(!processVariable(c,tempNum))return 0; break;
		case '$': if(!processIdentifier(c,tempNum))return 0; break;
		case '"': return evaluateStringOperand(c); break;
		case '0':
			++(c->m_ptr);
			if(! (  isalpha(*(c->m_ptr))  ||   isxdigit(*(c->m_ptr))  )  )tempNum = "0";
			else {
				++(c->m_ptr);
				if(isalpha(*(c->m_ptr))||isxdigit(*(c->m_ptr)))++(c->m_ptr);
				while(isxdigit(*(c->m_ptr)))++(c->m_ptr);
				tempNum.setStr(aux,(c->m_ptr) - aux);
			}
		break;
		default:
			if(!isxdigit(*(c->m_ptr))){
				c->setError(KVI_ERROR_UnexpectedCharacter,_i18n_("EXPRESSION-OPERAND"),c->m_ptr);
				return 0;
			}
			while(isxdigit(*(c->m_ptr)))++(c->m_ptr);
			tempNum.setStr(aux,(c->m_ptr) - aux);
		break;
	}

	aux = tempNum.ptr();
	int base = 10;
	if(*aux == '0'){
		++aux;
		switch(tolower(*aux)){
			case 'x': tempNum.cutLeft(2); base = 16; break;
			case 'h': tempNum.cutLeft(2); base = 16; break;

			case 'i': tempNum.cutLeft(2); base = 2;  break; // can not use b because it is a hex digit
			case 'j': tempNum.cutLeft(2); base = 3;  break;
			case 'k': tempNum.cutLeft(2); base = 4;  break;
			case 'l': tempNum.cutLeft(2); base = 5;  break;
			case 'm': tempNum.cutLeft(2); base = 6;  break;
			case 'n': tempNum.cutLeft(2); base = 7;  break;
			case 'o': tempNum.cutLeft(2); base = 8;  break;
			case 'p': tempNum.cutLeft(2); base = 9;  break;
			case 'q': tempNum.cutLeft(2); base = 10; break;
			case 'r': tempNum.cutLeft(2); base = 11; break;
			case 's': tempNum.cutLeft(2); base = 12; break;
			case 't': tempNum.cutLeft(2); base = 13; break;
			case 'u': tempNum.cutLeft(2); base = 14; break;
			case 'v': tempNum.cutLeft(2); base = 15; break;
			default: break; // unknown ...maybe a single 0
		}
	}
	bool bOk;
	long res = tempNum.toLongExt(&bOk,base);
	if(!bOk){
		c->setError(KVI_ERROR_UnexpectedCharacter,_i18n_("EXPRESSION-OPERAND"),tempNum.ptr());
		return 0;
	}
	return new KviExprTree(KviExprTree::NumericOperand,res);
}

KviExprTree * KviUserParser::evaluateExpression_RightOperand(KviExprTree * curTopOperator,KviCommand *c)
{
	__range_valid(curTopOperator);
	// Now curTopOperator has the left subtree (one node) set
	// and it points to the TOP (=ROOT) node
	// Evaluate the rest
	KviExprTree * operand;
	KviExprTree * incompleteOperator = curTopOperator;
	KviExprTree * auxOperator;

	for(;;){
		operand = evaluateOperand(c);
		if(!operand){ delete curTopOperator; return 0; }
		c->skipSpace();
		if(*(c->m_ptr) == ')' || *(c->m_ptr) == '\0'){ incompleteOperator->setRight(operand); return curTopOperator; }

		auxOperator = evaluateOperator(c);
		if(!auxOperator){ delete curTopOperator; delete operand; return 0; }

		//now compare operators...
		if(incompleteOperator->firstBinaryOperator() >= auxOperator->firstBinaryOperator()){
			incompleteOperator->setRight(auxOperator);
			auxOperator->setLeft(operand);
		} else {
			// precedence of operators...
			incompleteOperator->setRight(operand); //right tree complete
			// go up until we find an operator with lower precedence than auxOperator (>=)
			KviExprTree * tempOperator = incompleteOperator->parentWithPrecedenceLowerThan(auxOperator->m_value);
			if(tempOperator == 0){ //reached the top
				auxOperator->setLeft(curTopOperator);
				curTopOperator = auxOperator;
			} else {
				__range_valid(tempOperator->right());
				auxOperator->setLeft(tempOperator->right());
				tempOperator->setRight(auxOperator);
			}
		}
		incompleteOperator = auxOperator;
		__range_valid(incompleteOperator->right() == 0);
	}
	return 0; //newer here
}

KviExprTree * KviUserParser::evaluateOperator(KviCommand *c)
{
	// This one extracts the first operator it finds	
	c->skipSpace();
	long op;
	switch(*(c->m_ptr)){
		// single ops
		case '^': op = OP_POWER; ++(c->m_ptr); break;
		case '*': op = OP_MUL;   ++(c->m_ptr); break;
		case '/': op = OP_DIV;   ++(c->m_ptr); break;
		case '%': op = OP_MOD;   ++(c->m_ptr); break;
		case '+': op = OP_ADD;   ++(c->m_ptr); break;
		case '-': op = OP_SUB;   ++(c->m_ptr); break;
		case '@':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_CSEQ , ++(c->m_ptr);
			else {
				c->setError(KVI_ERROR_UnknownOperator,_i18n_("EXPRESSION-OPERATOR"),c->m_ptr);
				return 0;
			}
		break;
		case '=':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_EQ , ++(c->m_ptr);
			else {
				c->setError(KVI_ERROR_UnknownOperator,_i18n_("EXPRESSION-OPERATOR"),c->m_ptr);
				return 0;
			}
		break;
		case '!':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_NE , ++(c->m_ptr);
			else {
				c->setError(KVI_ERROR_UnknownOperator,_i18n_("EXPRESSION-OPERATOR"),c->m_ptr);
				return 0;
			}
		break;
		case '>':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_GE , ++(c->m_ptr);
			else if(*(c->m_ptr) == '>')op = OP_SHR , ++(c->m_ptr);
			else op = OP_GT;
		break;
		case '<':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_LE , ++(c->m_ptr);
			else if(*(c->m_ptr) == '<')op = OP_SHL , ++(c->m_ptr);
			else op = OP_LT;
		break;
		case '&':
			++(c->m_ptr);
			if(*(c->m_ptr) == '&')op = OP_AND , ++(c->m_ptr);
			else op = OP_BITAND;
		break;
		case '|':
			++(c->m_ptr);
			if(*(c->m_ptr) == '|')op = OP_OR  , ++(c->m_ptr);
			else op = OP_BITOR;
		break;
		default:
			c->setError(KVI_ERROR_UnknownOperator,_i18n_("EXPRESSION-OPERATOR"),c->m_ptr);
			return 0;
		break;
	}

	return new KviExprTree(KviExprTree::BinaryOperator,op);
}


