//
//   File : kvi_dcc_voice.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_dcc_voice.cpp)
//   Last major modification : Sat Apr 17 1999 18:54:50 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CHECK_RANGE_
//#define _KVI_DEBUG_CLASS_NAME_ "KviDccVoice"
#include "kvi_debug.h"

#include "kvi_dcc_voice.h"
#include "kvi_dcc_core.h"
#include "kvi_dcc_manager.h"
#include "kvi_app.h"
#include "kvi_ircview.h"
#include "kvi_input.h"
#include "kvi_frame.h"
#include "kvi_options.h"
#include "kvi_locale.h"

#include "kvi_ircsocket.h"
#include "kvi_netutils.h"
#include "kvi_settings.h"

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>

#include "kvi_error.h"

#include <sys/stat.h>   // for open()
#include <sys/ioctl.h>  // for ioctl()

//Check for the *SS Api....we don't want to fail here...

#ifndef COMPILE_WITH_NO_SOUND
	#ifdef HAVE_LINUX_SOUNDCARD_H
		#include <linux/soundcard.h>
	#else
		#ifdef HAVE_SYS_SOUNDCARD_H
			#include <sys/soundcard.h>
		#else
			#ifdef HAVE_SOUNDCARD_H
				#include <soundcard.h>			
			#else
				//CAN NOT COMPILE :(
				#define COMPILE_WITH_NO_SOUND
				#warning "Cannot find the soundcard.h header; you will NOT be able to use DCC Voice"
			#endif
		#endif
	#endif
#endif

// Time to sleep if no data has to be read or sent
// In DCC Voice it can be a bit longer...
#define KVI_DCC_IDLE_TIME_USECS 400

//DO NOT CHANGE THIS
#define KVI_AUDIO_DEVICE "/dev/dsp"
#define KVI_FRAG_SIZE 0x00080008
#define KVI_FORMAT AFMT_S16_LE
#define KVI_NUM_CHANNELS 1
#define KVI_SPEED 8000


//
// Friendly suggestion for programmers :
//     If you want your life to be simple and plain : do NOT use threads.
//     If you like risk , and you are patient : DO use threads , and get fun with it :)
//

//Declared in kvi_app.cpp and managed by KviApp class
//extern QPixmap * g_pWindowIcon[KVI_WND_NUM_ICONS];
extern QPixmap * g_pixViewOut[KVI_OUT_NUM_IMAGES];
//Global event dispatcher
extern KviThreadEventDispatcher *g_pThreadEventDispatcher;

/*
	@document: doc_dcc_voice_proto.kvihelp
	@title: The KVIrc DCC Voice protocol
		The DCC Voice protocol was initially inspired by the DCC TALK
		protocol implemented in the VIrc IRC client for Windows.<br>
		Since no technical documentation was available,
		I had to reimplement everything by myself.<br>
		The protocol is not excellent technically, 
		but behaves well with slow connections.<br>
		The requesting client sets up a listening socket
		and sends a CTCP DCC to the target client:<br>
		<b>DCC VOICE &lt;codec&gt; &lt;host_address&gt; &lt;host_port&gt; &lt;sample_rate&gt;</b><br>
		&lt;codec&gt; is the name of the compression algorythm used for the data transfer.<br>
		Currently only the ADPCM codec is implemented (see below).<br>
		&lt;host_address&gt; is an unsigned integer in network byte order
		that specifies the IP address of the source host (as in a normal DCC CHAT or SEND).<br>
		&lt;host_port&gt; is the port number that the requesting client
		listens on.<br>
		&lt;sample_rate&gt; is the .... yes ... number of samples per second
		of the audio data.<br>
		Currently only 8000 Hz sample rate is implemented.<br>
		The target client receives the DCC request and connects to the
		source host on the specified port.<br>
		Once the connection is established, the audio data transfer can start at any time from
		any of the two remote ends.<br>
		The audio messages are sent in form of streams of compressed data packets.<br>
		The adpcm compression algorithm (see kvi_adpcm.cpp/h in the distribution)
		is fast and simple; it uses a 1:4 compression rate and compresses 16 bit samples.<br>
		In my implementation, I have chosen to compress 2048 byte packets of audio data (1024 16 bit samples),
		obtaining 512 byte compressed data packets.<br>
		So one of the ends starts sending the audio data packets and the other receives and places
		it in a queue (this is an optimisation for common (slow) connections: lose time but gain data integrity)
		after uncompressing it.<br>
		The receiver queues packets until a message-termination packet is received.<br>
		The termination packet is a 512 byte packet with at least 5 first bytes filled with zeros.<br>
		Yes...it is not so "technical," but it works excellently (this packet should be
		treated as a terminator and ignored, as the other 507 bytes are meaningless (We could use it as a "roger-beep" ? :) ).<br>
		Once the terminator has been received the playback can start on the other side dequeuing
		the uncompressed audio data.<br>
		The current implementation is for half-duplex audio cards but it could be easily ported to
		take advantage of full-duplex cards.<br>
*/

//============ KviDccChat ============//

KviDccVoice::KviDccVoice(KviFrame *lpFrm,const char *name)
:KviWindow(name,KVI_WND_TYPE_VOICE,lpFrm)
{
	g_pThreadEventDispatcher->registerObject(this);

	m_pSplitter   = new QSplitter(QSplitter::Horizontal,this);
	m_pView       = new KviIrcView(m_pSplitter,lpFrm,this);
	m_pTalkButton = new QPushButton(__tr("&Talk"),this);
	m_pTalkButton->setEnabled(false);
	connect(m_pTalkButton,SIGNAL(pressed()),this,SLOT(talkPressed()));
	connect(m_pTalkButton,SIGNAL(released()),this,SLOT(talkReleased()));
	m_pStatLabel  = new QLabel(__tr("No connection"),this);
	m_pStatLabel->setFrameStyle(QFrame::WinPanel|QFrame::Sunken);

	m_bThreadRunning = false;

	m_pDccVoiceStruct = new KviDccVoiceStruct;
	m_pDccVoiceStruct->dccVoiceParent = this;
	m_pDccVoiceStruct->soundDevice = g_pOptions->m_szSoundDevice.ptr();
	m_pDccVoiceStruct->soundFd = -1;
	m_pDccVoiceStruct->fullDuplex = g_pOptions->m_bFullDuplexSoundDevice;
	m_pDccVoiceStruct->state = KVI_DCC_VOICE_STATE_IDLE;
	m_pDccVoiceStruct->prevState = KVI_DCC_VOICE_STATE_IDLE;
	m_pDccVoiceStruct->sendHead = 0;
	m_pDccVoiceStruct->sendTail = 0;
	m_pDccVoiceStruct->recvHead = 0;
	m_pDccVoiceStruct->recvTail = 0;
	m_pDccVoiceStruct->lastSentSize = 0;
	m_pDccVoiceStruct->completeBuffers = 0;
	m_pDccVoiceStruct->lastReadSize = 0;
	m_pDccVoiceStruct->tempSample = 0;
	m_pDccVoiceStruct->recvQueueCount = 0;
	m_pDccVoiceStruct->use090CompatibleCompression = g_pOptions->m_b090CompatibleVoiceCompression;
	m_pDccVoiceStruct->uPortToListenOn = 0;
	kvi_threadMutexInit(&(m_pDccVoiceStruct->playMutex),0);
	m_pDccVoiceStruct->bPlaying = false;

	for(int i=0;i<KVI_READ_BUF_SIZE_IN_CHARS;i++){
		m_pDccVoiceStruct->soundReadBuffer[i]=0;
	}

	m_szLocalNick = m_pFrm->m_global.szCurrentNick;
	m_szLocalMask = m_pFrm->m_global.szCurrentMaskFromServer;
	if(m_szLocalNick.isEmpty())m_szLocalNick = __tr("me");
	if(m_szLocalMask.isEmpty())m_szLocalMask = __tr("me!me@localhost");

	
	setFocusHandler(m_pView);
}


//============ ~KviDccChat ============//

KviDccVoice::~KviDccVoice()
{
	g_pThreadEventDispatcher->unregisterObject(this);
	if(m_bThreadRunning)kvi_threadCancel(m_thread);
	if(m_pDccVoiceStruct){
//		delete m_pDccVoiceStruct->outQueue;
		if(m_pDccVoiceStruct->soundFd != -1) ::close(m_pDccVoiceStruct->soundFd);
		if(m_pDccVoiceStruct->tempSample != 0)delete m_pDccVoiceStruct->tempSample;
		kvi_threadMutexDestroy(&(m_pDccVoiceStruct->playMutex));
		while(m_pDccVoiceStruct->sendHead){
			SampleStruct *s=m_pDccVoiceStruct->sendHead;
			m_pDccVoiceStruct->sendHead=s->next;
			delete s;
		}			
		while(m_pDccVoiceStruct->recvHead){
			SampleStruct *s=m_pDccVoiceStruct->recvHead;
			m_pDccVoiceStruct->recvHead=s->next;
			m_pDccVoiceStruct->recvQueueCount--;
			delete s;
		}
		__range_valid(m_pDccVoiceStruct->recvQueueCount == 0);
		delete m_pDccVoiceStruct;
	}
}



//================ myIconPtr =================//

QPixmap * KviDccVoice::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_VOICE];
}

//=============== applyOptions ================//

void KviDccVoice::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages,false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);

	if(!g_pOptions->m_pixLabelsBack->isNull()){
		m_pStatLabel->setBackgroundPixmap(*(g_pOptions->m_pixLabelsBack));
	} else {
		m_pStatLabel->setBackgroundColor(g_pOptions->m_clrLabelsBack);
	}
	QPalette pal(m_pStatLabel->palette());
	pal.setColor(QPalette::Normal,QColorGroup::Text,g_pOptions->m_clrLabelsActiveFore);
	m_pStatLabel->setPalette(pal);

	m_pStatLabel->setFont(g_pOptions->m_fntLabels);
}

bool KviDccVoice::event(QEvent * e)
{
	if(e->type() != QEvent::User)return KviWindow::event(e);
	switch(((KviDccEvent *)e)->m_type){
		case KVI_DCC_EVENT_ERROR:
			outputNoFmt(KVI_OUT_DCCERROR,((KviDccEvent *)e)->m_dataString.ptr());
			m_pStatLabel->setText(__tr("No connection"));
			m_pTalkButton->setEnabled(false);
			if(m_bThreadRunning)m_bThreadRunning = false;
			// The thread is dead now...m_pDccVoiceStruct is safe.
			output(KVI_OUT_DCCINFO,__tr("Connection terminated (%s:%d)"),m_pDccVoiceStruct->szAddress.ptr(),m_pDccVoiceStruct->uPort);
		break;
		case KVI_DCC_EVENT_MSG:
			outputNoFmt(KVI_OUT_DCCINFO,((KviDccEvent *)e)->m_dataString.ptr());
		break;
		case KVI_DCC_EVENT_LISTENING:
			m_pFrm->m_pSocket->sendData(((KviDccEvent *)e)->m_dataString.ptr(),((KviDccEvent *)e)->m_dataString.len());
			outputNoFmt(KVI_OUT_DCCINFO,__tr("Sent DCC voice request, waiting for reply..."));
		break;
		case KVI_DCC_EVENT_ENABLETALK:
			m_pTalkButton->setEnabled(true);
		break;
		case KVI_DCC_EVENT_DISABLETALK:
			m_pTalkButton->setEnabled(false);
		break;
		case KVI_DCC_EVENT_BUFFEREDTIME:
			m_pStatLabel->setText(_CHAR_2_QSTRING(((KviDccEvent *)e)->m_dataString.ptr()));
		break;
		case KVI_DCC_EVENT_CONNECTIONESTABILISHED:
			m_pStatLabel->setText(__tr("Connected"));
		break;
		default:
			return KviWindow::event(e);
		break;
	}

	return true;
}

void KviDccVoice::talkPressed()
{
	if(!m_bThreadRunning)return;
	kvi_threadMutexLock(&(m_pDccVoiceStruct->playMutex));
	if(!m_pDccVoiceStruct->bPlaying && !m_pDccVoiceStruct->fullDuplex)m_pDccVoiceStruct->state = KVI_DCC_VOICE_STATE_TALK;
//	else debug("Is PLAYING");
	kvi_threadMutexUnlock(&(m_pDccVoiceStruct->playMutex));
}

void KviDccVoice::talkReleased()
{
	if(!m_bThreadRunning)return;
//	kvi_threadMutexLock(&(dcc->playMutex));
	if(m_pDccVoiceStruct->state == KVI_DCC_VOICE_STATE_TALK)
		m_pDccVoiceStruct->state = KVI_DCC_VOICE_STATE_IDLE;
//	kvi_threadMutexUnlock(&(dcc->playMutex));
}


//================ resizeEvent ===============//

void KviDccVoice::resizeEvent(QResizeEvent *)
{
	int btnSize = m_pTalkButton->sizeHint().height();
	m_pStatLabel->setGeometry(0,0,width() / 2,btnSize);
	m_pTalkButton->setGeometry(width() / 2,0,width() / 2,btnSize);
	m_pSplitter->setGeometry(0,btnSize,width(),height() - btnSize);
}

// ----------------------------->
//               SOUNDCARD STUFF
// ----------------------------->

inline int kvi_dccVoiceThread_openSoundCard(KviDccVoiceStruct * dcc,bool bRead)
{
#ifndef COMPILE_WITH_NO_SOUND
	if(dcc->soundFd != -1)return KVI_ERROR_SoundCardAlreadyOpen;
	if(dcc->fullDuplex){
		if(dcc->soundDevice.hasData())dcc->soundFd=open(dcc->soundDevice.ptr(),O_RDWR);
		if(dcc->soundFd<0)dcc->soundFd=open(KVI_AUDIO_DEVICE,O_RDWR);
	} else {
		if(dcc->soundDevice.hasData())dcc->soundFd=open(dcc->soundDevice.ptr(),bRead ? O_RDONLY : O_WRONLY);
		if(dcc->soundFd<0)dcc->soundFd=open(KVI_AUDIO_DEVICE,bRead ? O_RDONLY : O_WRONLY);
	}
	if(dcc->soundFd<0)return KVI_ERROR_CanNotOpenDevDsp;
	static int frag=KVI_FRAG_SIZE;
	if(ioctl(dcc->soundFd,SNDCTL_DSP_SETFRAGMENT,&frag)<0)return KVI_ERROR_CanNotInitializeSoundCard;
	static int fmt=KVI_FORMAT;
	if(ioctl(dcc->soundFd,SNDCTL_DSP_SETFMT,&fmt)<0)return KVI_ERROR_CanNotInitializeSoundCard;
	static int chans=KVI_NUM_CHANNELS;
	if(ioctl(dcc->soundFd,SNDCTL_DSP_CHANNELS,&chans)<0)return KVI_ERROR_CanNotInitializeSoundCard;
	static int speed=KVI_SPEED;
	if(ioctl(dcc->soundFd,SNDCTL_DSP_SPEED,&speed)<0)return KVI_ERROR_CanNotInitializeSoundCard;
	return KVI_ERROR_Success;
#else
	return KVI_ERROR_CompiledWithNoSound;
#endif
}

static int kvi_dccVoiceThread_openSoundCardForWriting(KviDccVoiceStruct * dcc)
{
	return kvi_dccVoiceThread_openSoundCard(dcc,false);
}
static int kvi_dccVoiceThread_openSoundCardForReading(KviDccVoiceStruct * dcc)
{
	return kvi_dccVoiceThread_openSoundCard(dcc,true);
}

int KviDccVoice::checkSoundCard(KviDccVoiceStruct * dcc)
{
	int iError = kvi_dccVoiceThread_openSoundCardForReading(dcc);
	if(dcc->soundFd != -1) ::close(dcc->soundFd);
	dcc->soundFd = -1;
	return iError;
}

static void kvi_dccVoiceThread_readSampleFromSoundcard(KviDccVoiceStruct * dcc)
{
#ifndef COMPILE_WITH_NO_SOUND
	//read a sample (2048 bytes)
	//should be blocking in this thread
	kvi_threadTestCancel();

	read(dcc->soundFd,(void *)dcc->soundReadBuffer,KVI_READ_BUF_SIZE_IN_CHARS);

	kvi_threadTestCancel();

	//compress
	if(dcc->use090CompatibleCompression){
		dcc->readAdpcm.index=0;
		dcc->readAdpcm.valprev=0;
	}

//	kvi_threadTestCancel();

	SampleStruct *s=new SampleStruct;
	//compress (1024 shorts to 512 byte samples)
	ADPCM_compress((short *)(dcc->soundReadBuffer),s->buffer,KVI_READ_BUF_SIZE_IN_SHORTS,&(dcc->readAdpcm));
	//append to the list
	s->next=0;

	if(dcc->sendHead){
		//other data in queue
		dcc->sendTail->next=s;
		dcc->sendTail=s;
	} else {
		//no data in queue
		dcc->sendHead=s;
		dcc->sendTail=s;
	}

	kvi_threadTestCancel();
#endif
}

static void kvi_dccVoiceThread_appendEofToOutQueue(KviDccVoiceStruct * dcc)
{
#ifndef COMPILE_WITH_NO_SOUND
	SampleStruct *s=new SampleStruct;
	for(int i=0;i<6;i++){
		s->buffer[i] = 0;
	}
	//append to the list
	s->next=0;

	if(dcc->sendHead){
		//other data in queue
		dcc->sendTail->next=s;
		dcc->sendTail=s;
	} else {
		//no data in queue
		dcc->sendHead=s;
		dcc->sendTail=s;
	}
#endif
}

static bool kvi_dccVoiceThread_sendSample(int sock,KviDccVoiceStruct * dcc)
{
#ifndef COMPILE_WITH_NO_SOUND
	struct timeval t;
	fd_set fset;
	t.tv_sec  = 0;
	t.tv_usec = 1; //we could wait more ?
	FD_ZERO(&fset);
	FD_SET(sock,&fset);

	kvi_threadTestCancel();

	if(select(sock+1,0,&fset,0,&t)>0){
		kvi_threadTestCancel();
		//get the first available buffer and try to send it
		SampleStruct *s=dcc->sendHead;
		//dcc->lastSentSize can be either 0 here...
		int toSend=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS - dcc->lastSentSize;
		//so toSend=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS;
		int wrote=write(sock,(void *)((s->buffer)+dcc->lastSentSize),toSend);

		kvi_threadTestCancel();

		// int wrote=write(m_sock,(void *)(s->buffer)+0,KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS);
		if(wrote<0)kvi_dccCore_errorExitThread(__tr("Error while writing to the socket: Connection lost"),dcc->dccVoiceParent);
		else {
			if(wrote==toSend){
				dcc->lastSentSize=0;
				dcc->sendHead=s->next;
				if(!dcc->sendHead)dcc->sendTail=0;
				delete s;

				kvi_threadTestCancel();

			} else dcc->lastSentSize+=wrote; //and finish next time
			return true;
		}
	} else return false;
#endif
	return true; //newer here
}



static bool kvi_dccVoiceThread_receiveData(int sock,KviDccVoiceStruct * dcc)
{
#ifndef COMPILE_WITH_NO_SOUND
	if(!kvi_dccCore_selectForRead(sock)){
		kvi_threadTestCancel();
		return false;
	} // else data to read...
	if((!dcc->lastReadSize) && (!dcc->tempSample))dcc->tempSample=new SampleStruct; //new buffer begin
	//if m_lastReadSize is 0
	int toRead=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS- dcc->lastReadSize;
	//toread=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS-0;
	kvi_threadTestCancel();
	int readed=read(sock,(void *)((dcc->tempSample->buffer)+ dcc->lastReadSize),toRead);
	kvi_threadTestCancel();
	//read(m_sock,(void *)m_tempBuff->buffer+0),KVI_COMPRESSED_SAMPLE....);
	if(readed <= 0){
		// oooops ?
		if(readed == 0)kvi_dccCore_errorExitThread(__tr("Remote end has closed the connection"),dcc->dccVoiceParent);
		else { // error ?
			if((errno == EINTR)||(errno == EAGAIN))return false;
			// Yes...error :(
			int iError;
			switch(errno){
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				// Unhandled error...pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// post the error event to the parent and exit
			dcc->tmpBuffer.sprintf(__tr("READ ERROR: %s"),kvi_getErrorString(iError));
			kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccVoiceParent);
		}
	} else {
		if(readed==toRead){
			dcc->lastReadSize=0; //next time start a new buffer
			//append the buf
			dcc->tempSample->next=0;
			if(dcc->recvHead){ //other data in queue
				dcc->recvTail->next=dcc->tempSample;
				dcc->recvTail=dcc->tempSample;
			} else { //no data in queue
				dcc->recvHead=dcc->tempSample;
				dcc->recvTail=dcc->tempSample;
			}
			kvi_threadTestCancel();
			//check if it was an EOF
			if(dcc->tempSample->buffer[0]==0){
				if(dcc->tempSample->buffer[1]==0){
					if(dcc->tempSample->buffer[2]==0){
						if(dcc->tempSample->buffer[3]==0){
							if(dcc->tempSample->buffer[4]==0){
								if(dcc->tempSample->buffer[5]==0)dcc->completeBuffers++;
							}
						}
					}
				}
			}
			dcc->recvQueueCount++;

			dcc->tmpBuffer.sprintf(__tr("Buffered: %d milliseconds"),dcc->recvQueueCount * 128);
			KviDccEvent * msg = new KviDccEvent(KVI_DCC_EVENT_BUFFEREDTIME,dcc->tmpBuffer.ptr());
			g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

			dcc->tempSample=0;
		} else dcc->lastReadSize+=readed; //continue here
	}
	kvi_threadTestCancel();
#endif
	return true;
}

static void kvi_dccVoiceThread_playbackLoop(int sock,KviDccVoiceStruct * dcc)
{
#ifndef COMPILE_WITH_NO_SOUND

	kvi_threadMutexLock(&(dcc->playMutex));
	if((dcc->state == KVI_DCC_VOICE_STATE_TALK) && !dcc->fullDuplex){
		kvi_threadMutexUnlock(&(dcc->playMutex));
		return;
	} else {
		dcc->bPlaying = true;
		kvi_threadMutexUnlock(&(dcc->playMutex));
	}

	KviDccEvent * msg = new KviDccEvent(KVI_DCC_EVENT_DISABLETALK,"");
	if(!dcc->fullDuplex)
		g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

	dcc->writeAdpcm.index=0;
	dcc->writeAdpcm.valprev=0;

	kvi_threadTestCancel();

	int iError;
	if(dcc->fullDuplex){
		if(dcc->soundFd < 0)iError = kvi_dccVoiceThread_openSoundCard(dcc,true);
		else iError = KVI_ERROR_Success;
	} else iError = kvi_dccVoiceThread_openSoundCardForWriting(dcc);
	if(iError != KVI_ERROR_Success){
		if(dcc->soundFd != -1) ::close(dcc->soundFd);
		dcc->soundFd = -1;
		dcc->tmpBuffer.sprintf(__tr("WARNING: Cannot open sound card for writing (%s)"),kvi_getErrorString(iError));
		KviDccEvent * msg = new KviDccEvent(KVI_DCC_EVENT_MSG,dcc->tmpBuffer.ptr());
		g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

		kvi_threadTestCancel();
		usleep(1000); // Give time to release the button , process the message and so on...
		kvi_threadTestCancel();
	} else {
		bool notEof=true;
		while(dcc->recvHead && notEof){
			kvi_threadTestCancel();
			SampleStruct *s=dcc->recvHead;
			//check if it is EOF for us
			//gnary hack but works (6 bytes 0 is a complete rumour);
			if(s->buffer[0]==0){
				if(s->buffer[1]==0){
					if(s->buffer[2]==0){
						if(s->buffer[3]==0){
							if(s->buffer[4]==0){
								if(s->buffer[5]==0){
									notEof=false;
									dcc->completeBuffers--;
								}
							}
						}
					}
				}
			}

			if(notEof){ //play the sample
				if(dcc->use090CompatibleCompression){
					dcc->writeAdpcm.index=0;
					dcc->writeAdpcm.valprev=0;
				}
				ADPCM_uncompress(s->buffer,(short *)(dcc->soundWriteBuffer),KVI_WRITE_BUF_SIZE_IN_SHORTS,&(dcc->writeAdpcm));
				//block TODO : check for space in the buffer
				kvi_threadTestCancel();
				write(dcc->soundFd,(void *)(dcc->soundWriteBuffer),KVI_WRITE_BUF_SIZE_IN_CHARS);
				kvi_threadTestCancel();
				//do something while playing 
				if(dcc->sendHead){
					if(!kvi_dccVoiceThread_sendSample(sock,dcc)){
						kvi_threadTestCancel();
						if(!kvi_dccVoiceThread_receiveData(sock,dcc))usleep(50);
					}
				} else {
					kvi_dccVoiceThread_receiveData(sock,dcc);
					kvi_threadTestCancel();
				}
				kvi_threadTestCancel();
			} else kvi_threadTestCancel();
			dcc->recvHead=s->next;
			if(dcc->recvHead==0)dcc->recvTail=0;
			dcc->recvQueueCount--;
			delete s;
		}
		// Close the soundcard
		kvi_threadTestCancel();
		usleep(100);
		::close(dcc->soundFd);
		dcc->soundFd = -1;
	}

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf(__tr("Buffered: %d milliseconds"),dcc->recvQueueCount * 128);
	msg = new KviDccEvent(KVI_DCC_EVENT_BUFFEREDTIME,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

	msg = new KviDccEvent(KVI_DCC_EVENT_ENABLETALK,"");
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

	kvi_threadMutexLock(&(dcc->playMutex));
	dcc->bPlaying = false;
	kvi_threadMutexUnlock(&(dcc->playMutex));
#endif
}

//=====================================================================================================
//
// Main routines for the slave DCC Chat thread 
//

static int kvi_dccVoiceThread_mainLoop(int sock,KviDccVoiceStruct * dcc)
{
#ifndef COMPILE_WITH_NO_SOUND
	// Main DCC Voice loop
	// This funciton newer returns.
	// It is terminated when kvi_threadExit() is called
	// from one of the internal calls
	// Or if the parent object calls kvi_threadCancel().
	KviDccEvent * msg = new KviDccEvent(KVI_DCC_EVENT_CONNECTIONESTABILISHED,"");
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

	msg = new KviDccEvent(KVI_DCC_EVENT_ENABLETALK,"");
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

	for(;;){ //forever
		kvi_threadTestCancel();
		if(dcc->state != KVI_DCC_VOICE_STATE_TALK){
//			debug("Not talking");
			// Sampling nothing...take it easy
			if(dcc->prevState == KVI_DCC_VOICE_STATE_TALK){
				// Just finished to sample
				kvi_dccVoiceThread_appendEofToOutQueue(dcc);
				kvi_threadTestCancel();
				__range_valid(dcc->soundFd != -1);
				::close(dcc->soundFd);
				dcc->soundFd = -1;
				dcc->prevState = KVI_DCC_VOICE_STATE_IDLE;
			}
			if(dcc->sendHead){
				//First send data
				if(!kvi_dccVoiceThread_sendSample(sock,dcc)){
					kvi_threadTestCancel();
					// check for incoming data
					if(!kvi_dccVoiceThread_receiveData(sock,dcc))usleep(300);
					kvi_threadTestCancel();
				}
			} else {
				//First check for incoming data
				if(!kvi_dccVoiceThread_receiveData(sock,dcc)){
					kvi_threadTestCancel();
					// check for incoming data
//					if(!kvi_dccVoiceThread_sendSample(sock,dcc))usleep(300);
//					kvi_threadTestCancel();
				}
			}
			//if have complete buffers , playback
			if(dcc->completeBuffers > 0){
				kvi_dccVoiceThread_playbackLoop(sock,dcc);
			} else usleep(100);
			kvi_threadTestCancel();
		} else {
//			debug("talking");
			// Sampling... check if just started
			if(dcc->prevState != KVI_DCC_VOICE_STATE_TALK){
				//just started to talk
				__range_valid(dcc->prevState == KVI_DCC_VOICE_STATE_IDLE);
				int iError;
				if(dcc->fullDuplex){
					if(dcc->soundFd < 0)iError = kvi_dccVoiceThread_openSoundCard(dcc,true);
					else iError = KVI_ERROR_Success;
				} else iError = kvi_dccVoiceThread_openSoundCardForReading(dcc);
				if(iError != KVI_ERROR_Success){
					if(dcc->soundFd != -1) ::close(dcc->soundFd);
					dcc->soundFd = -1;
					dcc->tmpBuffer.sprintf(__tr("WARNING: Cannot open sound card for reading (%s)"),kvi_getErrorString(iError));
					KviDccEvent * msg = new KviDccEvent(KVI_DCC_EVENT_MSG,dcc->tmpBuffer.ptr());
					g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

					kvi_threadTestCancel();
					usleep(5000); // Give time to release the button , process the message and so on...
					kvi_threadTestCancel();
					continue;     // And retry
				} else dcc->prevState = KVI_DCC_VOICE_STATE_TALK;
				dcc->readAdpcm.index=0;
				dcc->readAdpcm.valprev=0;
			}
			kvi_threadTestCancel();
			kvi_dccVoiceThread_readSampleFromSoundcard(dcc);
			kvi_threadTestCancel();
			if(!kvi_dccVoiceThread_sendSample(sock,dcc)){
				kvi_threadTestCancel();
				// check for incoming data
				kvi_dccVoiceThread_receiveData(sock,dcc);
				kvi_threadTestCancel();
			}
		}
		kvi_threadTestCancel();
	}
#endif
	return 0; //Newer reached
}

//=====================================================================================================
//
// Accept a DCC VOICE REQUEST from another user on IRC
//

static void * kvi_dccVoiceThread_acceptDccRequest(void * data)
{
	__debug("slaveThread");
	// Hack for Solaris (explained also in kvi_dns.cpp)
//#ifdef COMPILE_IGNORE_SIGALARM
//	kvi_threadIgnoreSigalarm();
//#endif
	kvi_threadInitialize();
	// Specific routine for accepting DCC Voice requests
	// from users on IRC.
	// An user sent us a CTCP DCC with a port and an address
	// Now we try to contact that host and initiate a DCC Voice...
	KviDccVoiceStruct * dcc = (KviDccVoiceStruct *)data; //Maybe convert it in just casts...
	// Prepare the target data
	struct in_addr     inAddress;
	struct sockaddr_in hostSockAddr;

	inAddress.s_addr        = dcc->uAddress;
	hostSockAddr.sin_family = AF_INET;
	hostSockAddr.sin_port   = htons(dcc->uPort);
	hostSockAddr.sin_addr   = inAddress;

	kvi_threadTestCancel();
	// Let's go...
	int sock = ::socket(PF_INET,SOCK_STREAM,0);
	// Don't forget to close the socket when terminating.
	kvi_threadCleanupPush(kvi_dccCore_threadCleanupCloseSocket,&sock);

	if(sock < 0)kvi_dccCore_errorExitThread(__tr("Unable to create a stream socket. DCC voice failed"),dcc->dccVoiceParent);

	kvi_threadTestCancel();

	if(fcntl(sock, F_SETFL, O_NONBLOCK) < 0)kvi_dccCore_errorExitThread(__tr("Unable to create a non-blocking stream socket. DCC voice failed"),dcc->dccVoiceParent);

	kvi_threadTestCancel();

	// Ok...now try to connect
	int iError = kvi_dccCore_connect(sock,(struct sockaddr *)(&hostSockAddr),sizeof(hostSockAddr));

	if(iError != KVI_ERROR_Success){
		dcc->tmpBuffer.sprintf(__tr("CONNECT ERROR: %s"),kvi_getErrorString(iError));
		kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccVoiceParent);
	}

	kvi_threadTestCancel();

	// Now wait for connection...

	KviStr tmp(KviStr::Format,__tr("Connecting to %s on port %u"),dcc->szAddress.ptr(),dcc->uPort);
	KviDccEvent * msg = new KviDccEvent(KVI_DCC_EVENT_MSG,tmp.ptr());
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

	kvi_threadTestCancel();

	iError = kvi_dccCore_waitForOutgoingConnection(sock);
	if(iError != KVI_ERROR_Success){
		dcc->tmpBuffer.sprintf(__tr("CONNECT ERROR: %s"),kvi_getErrorString(iError));
		kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccVoiceParent);
	}

	kvi_threadTestCancel();

	msg = new KviDccEvent(KVI_DCC_EVENT_MSG,__tr("Connection established"));
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccVoiceParent);

	kvi_threadTestCancel();

	kvi_dccVoiceThread_mainLoop(sock,dcc);

	// Code that is newer reached...
	kvi_threadCleanupPop(0); //Do not execute

	return 0;
}


void KviDccVoice::acceptDccRequest(const char *nick,const char *username,const char *host,unsigned long uAddress,unsigned short uPort)
{
	m_szRemoteNick = nick;
	m_szRemoteMask.sprintf("%s!%s@%s",nick,username,host);
	// Called from KviDccManager
	// An user sent us a CTCP DCC , and we accepted it.
	// Connect to his host
#ifdef _KVI_DEBUG_CHECK_RANGE_
	if(m_bThreadRunning)debug("WARNING: acceptDccRequest was called twice!");
#endif

	int iError = checkSoundCard(m_pDccVoiceStruct);
	if(iError != KVI_ERROR_Success){
		output(KVI_OUT_DCCERROR,__tr("Sound card check failed (%s). Cannot accept DCC voice"),
			kvi_getErrorString(iError));
		return;
	}

	m_pDccVoiceStruct->nick      = nick;
	m_pDccVoiceStruct->username  = username;
	m_pDccVoiceStruct->host      = host;
	m_pDccVoiceStruct->uAddress  = uAddress;
//	m_pDccVoiceStruct->state     = KVI_DCC_VOICE_STATE_IDLE;
	struct in_addr addr;
	addr.s_addr = uAddress;
	kvi_binaryIpToString(addr,m_pDccVoiceStruct->szAddress);
	m_pDccVoiceStruct->uPort     = uPort;

	// Create the slave thread...

	m_bThreadRunning = true;

	if(kvi_threadCreate(&m_thread,0,kvi_dccVoiceThread_acceptDccRequest,m_pDccVoiceStruct) != 0){
		m_bThreadRunning = false;
		outputNoFmt(KVI_OUT_DCCERROR,__tr("Cannot create slave thread. DCC voice failed"));
	}
}

//=====================================================================================================
//
// Request a DCC to an irc user
//

static void * kvi_dccVoiceThread_requestDccVoice(void * data)
{
	__debug("slaveThread");
	// Hack for Solaris (explained also in kvi_dns.cpp)
//#ifdef COMPILE_IGNORE_SIGALARM
//	kvi_threadIgnoreSigalarm();
//#endif
	kvi_threadInitialize();

	KviDccEvent *e;
	// with kvi_threadCleanupPush/pop we need to declare the variables at the beginning of the function...
	int newsock = -1;

	KviDccVoiceStruct * dcc = (KviDccVoiceStruct *)data; //Maybe convert it in just casts...

	struct sockaddr_in sockAddress;
	sockAddress.sin_family      = AF_INET;
	sockAddress.sin_port        = htons(dcc->uPortToListenOn); // :)
	sockAddress.sin_addr.s_addr = INADDR_ANY;

	kvi_threadTestCancel();
	// Let's go...
	int sock = ::socket(PF_INET,SOCK_STREAM,0);
	// Don't forget to close the socket when terminating.
	kvi_threadCleanupPush(kvi_dccCore_threadCleanupCloseSocket,&sock);

	if(sock < 0)kvi_dccCore_errorExitThread(__tr("Unable to create a listening socket. DCC voice failed"),dcc->dccVoiceParent);

	kvi_threadTestCancel();

	if((bind(sock,(struct sockaddr *)&sockAddress,sizeof(sockAddress))<0)||(listen(sock,100)<0) )
		kvi_dccCore_errorExitThread(__tr("Unable to set up a listening socket. DCC voice failed"),dcc->dccVoiceParent);

	kvi_threadTestCancel();

	_this_should_be_socklen_t iSize = sizeof(sockAddress);
	getsockname(sock,(struct sockaddr*)&sockAddress,&iSize);
	dcc->uPort = ntohs(sockAddress.sin_port);

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf(__tr("Listening on port %u"),dcc->uPort);
	e = new KviDccEvent(KVI_DCC_EVENT_MSG,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccVoiceParent);

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf("PRIVMSG %s :%cDCC VOICE adpcm %u %u 8000%c",dcc->nick.ptr(),0x01,ntohl(dcc->uAddress),dcc->uPort,0x01);
	e = new KviDccEvent(KVI_DCC_EVENT_LISTENING,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccVoiceParent);

	kvi_threadTestCancel();

	newsock = kvi_dccCore_waitForIncomingConnection(sock,&(dcc->uAddress),&(dcc->uPort),&(dcc->szAddress));

	kvi_threadCleanupPop(0); //Do not execute
	close(sock); //close the old socket...
	kvi_threadCleanupPush(kvi_dccCore_threadCleanupCloseSocket,&newsock);

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf(__tr("Connected to %s on port %u"),dcc->szAddress.ptr(),dcc->uPort);
	e = new KviDccEvent(KVI_DCC_EVENT_MSG,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccVoiceParent);

	kvi_threadTestCancel();

	kvi_dccVoiceThread_mainLoop(newsock,dcc);

	// Code that is newer reached...
	kvi_threadCleanupPop(0); //Do not execute

	return 0;
}

void KviDccVoice::requestDcc(const char *nick,const char *username,const char *host)
{
	m_szRemoteNick = nick;
	m_szRemoteMask.sprintf("%s!%s@%s",nick,username,host);
	// Called from KviDccManager
	// Request a DCC connection to an user
#ifdef _KVI_DEBUG_CHECK_RANGE_
	if(m_bThreadRunning)debug("WARNING: requestDcc was called twice!");
#endif

	int iError = checkSoundCard(m_pDccVoiceStruct);
	if(iError != KVI_ERROR_Success){
		output(KVI_OUT_DCCERROR,__tr("Sound card check failed (%s). Cannot request DCC voice"),
			kvi_getErrorString(iError));
		return;
	}

	m_pDccVoiceStruct->nick      = nick;
	m_pDccVoiceStruct->username  = username;
	m_pDccVoiceStruct->host      = host;
	m_pDccVoiceStruct->uAddress  = m_pFrm->m_pSocket->getSockAddress();
	m_pDccVoiceStruct->uPortToListenOn = m_pFrm->m_pDccManager->getDccSendListenPort();

//	m_pDccVoiceStruct->state     = KVI_DCC_VOICE_STATE_IDLE;
	if(g_pOptions->m_bUseUserDefinedIpForDccRequests){
		struct in_addr addr;
		if(kvi_stringIpToBinaryIp(g_pOptions->m_szDccLocalIpAddress.ptr(),&addr)){
			output(KVI_OUT_DCCINFO,__tr("Using %s as localhost address"),g_pOptions->m_szDccLocalIpAddress.ptr());
			m_pDccVoiceStruct->uAddress = addr.s_addr;
		} else {
			output(KVI_OUT_DCCERROR,__tr("IP address %s is invalid : Using standard IP address"),g_pOptions->m_szDccLocalIpAddress.ptr());
			g_pOptions->m_bUseUserDefinedIpForDccRequests = false;
		}
	}

	if(m_pDccVoiceStruct->uAddress == 0){
//#warning "Try to use the USER SUPPLIED LOCAL HOST IP ADDRESS"
		outputNoFmt(KVI_OUT_DCCERROR,__tr("Cannot resolve localhost. DCC voice failed"));
		return;
	}
	m_pDccVoiceStruct->szAddress = __tr("unknown");
	m_pDccVoiceStruct->uPort     = 0;
	// Create the slave thread...
	m_bThreadRunning = true;

	if(kvi_threadCreate(&m_thread,0,kvi_dccVoiceThread_requestDccVoice,m_pDccVoiceStruct) != 0){
		m_bThreadRunning = false;
		outputNoFmt(KVI_OUT_DCCERROR,__tr("Cannot create slave thread. DCC voice failed"));
	}

}

#include "m_kvi_dcc_voice.moc"
