//
//   File : kvi_dcc_send.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_dcc_send.cpp)
//   Last major modification : Sat Apr 17 1999 18:54:50 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CHECK_RANGE_
//#define _KVI_DEBUG_CLASS_NAME_ "KviDccSend"
#include "kvi_debug.h"

#include "kvi_dcc_send.h"
#include "kvi_dcc_manager.h"
#include "kvi_app.h"
#include "kvi_ircview.h"
#include "kvi_input.h"
#include "kvi_frame.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_defines.h"

#include "kvi_ircsocket.h"
#include "kvi_netutils.h"
#include "kvi_settings.h"

#include "kvi_dcc_core.h"
#include "kvi_console.h"
#include "kvi_channel.h" //for KviChanLabel

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include "kvi_error.h"
#include "kvi_opaquelabel.h"

#include "kvi_event.h"
#include "kvi_uparser.h"

#include <qnamespace.h>

// Time to sleep if no data has to be read or sent
#define KVI_DCC_IDLE_TIME_USECS 200

//#define KVI_DCC_SEND_MAX_BLOCK_SIZE 4096
//#warning "Make it optional ? (KVI_DCC_FAILED_WRITES_LIMIT_FOR_STALLED) in order to stop flashing in the dcc send progress"
#define KVI_DCC_FAILED_WRITES_LIMIT_FOR_STALLED 10
//
// Friendly suggestion for programmers :
//     If you want your life to be simple and plain : do NOT use threads.
//     If you like risk , and you are patient : DO use threads , and get fun with it :)
//

//Declared in kvi_app.cpp and managed by KviApp class
extern QPixmap * g_pixViewOut[KVI_OUT_NUM_IMAGES];
extern KviThreadEventDispatcher *g_pThreadEventDispatcher;
extern KviEventManager *g_pEventManager;

//============ KviDccSend ============//

KviDccSend::KviDccSend(KviFrame *lpFrm,const char *name)
:KviWindow(name,KVI_WND_TYPE_SEND,lpFrm)
{
	g_pThreadEventDispatcher->registerObject(this);

	m_pSplitter      = new QSplitter(QSplitter::Horizontal,this);
	m_pView          = new KviIrcView(m_pSplitter,lpFrm,this);
	m_pInput         = 0;
	m_pProgress      = new QProgressBar(100,this);

	m_pStatLabel     = new KviChanLabel(__tr("No connection"),this);
	m_pFileLabel     = new KviChanLabel(__tr("File: None"),this);

	m_pProgress->setFrameStyle(QFrame::Panel|QFrame::Sunken);

	m_dccSendStruct  = 0;
	m_bThreadRunning = false;

	setFocusHandler(m_pView);

	applyLabelsOptions();
	setFileProgress(0);
}

//============ ~KviDccSend ============//

KviDccSend::~KviDccSend()
{
	g_pThreadEventDispatcher->unregisterObject(this);
	// Kill the child thread if it is still here
	if(m_bThreadRunning){
		kvi_threadCancel(m_thread);
		m_bThreadRunning = false;
	}

	destroyDccSendStruct();
}

void KviDccSend::destroyDccSendStruct()
{
	if(m_dccSendStruct != 0){
		kvi_threadMutexDestroy(&(m_dccSendStruct->resumeMutex));
		if(m_dccSendStruct->file != 0){
			m_dccSendStruct->file->close();
			delete m_dccSendStruct->file;
		}
		delete m_dccSendStruct;
		m_dccSendStruct = 0;
	}
}

void KviDccSend::setFileProgress(int prog)
{
	setProgress(prog);
	m_pProgress->setProgress(prog);
}

//================ myIconPtr =================//

QPixmap * KviDccSend::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_SEND];
}

//=============== applyOptions ================//

void KviDccSend::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages,false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);
	applyLabelsOptions();
}

//=============== applyOptions ================//

void KviDccSend::applyLabelsOptions()
{
	if(!g_pOptions->m_pixLabelsBack->isNull()){
		m_pFileLabel->setBackgroundPixmapPointer(g_pOptions->m_pixLabelsBack);
		m_pStatLabel->setBackgroundPixmapPointer(g_pOptions->m_pixLabelsBack);
	} else {
		m_pFileLabel->setBackColor(g_pOptions->m_clrLabelsBack);
		m_pStatLabel->setBackColor(g_pOptions->m_clrLabelsBack);
	}
	m_pFileLabel->setTextColor(g_pOptions->m_clrLabelsActiveFore);
	m_pStatLabel->setTextColor(g_pOptions->m_clrLabelsActiveFore);
	m_pFileLabel->setFont(g_pOptions->m_fntLabels);
	m_pStatLabel->setFont(g_pOptions->m_fntLabels);
}

bool KviDccSend::event(QEvent * e)
{
	if(e->type() != QEvent::User)return KviWindow::event(e);
	switch(((KviDccEvent *)e)->m_type){
		case KVI_DCC_EVENT_ERROR:
			outputNoFmt(KVI_OUT_DCCERROR,((KviDccEvent *)e)->m_dataString.ptr());
			m_bThreadRunning = false;
			// The thread is dead now...m_dccSendStruct is safe.
			output(KVI_OUT_DCCINFO,__tr("Connection terminated (%s:%d)"),m_dccSendStruct->szAddress.ptr(),m_dccSendStruct->uPort);
			destroyDccSendStruct();
		break;
		case KVI_DCC_EVENT_MSG:
			outputNoFmt(KVI_OUT_DCCINFO,((KviDccEvent *)e)->m_dataString.ptr());
		break;
		case KVI_DCC_EVENT_DCCSENDSTATUS:
			m_pStatLabel->setText(((KviDccEvent *)e)->m_dataString.ptr());
			setFileProgress(((KviDccEvent *)e)->m_param);
		break;
		case KVI_DCC_EVENT_FINISHED:
			outputNoFmt(KVI_OUT_INTERNAL,((KviDccEvent *)e)->m_dataString.ptr());
			m_bThreadRunning = false;
			// The thread is dead now...m_pDccSendStruct is safe.
			output(KVI_OUT_DCCINFO,__tr("Connection terminated (%s:%d)"),m_dccSendStruct->szAddress.ptr(),m_dccSendStruct->uPort);
			if(g_pOptions->m_bNotifyDccSendCompletionInConsole){
				m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,__tr("DCC file transfer with %s completed [%s: %u bytes]"),
					m_dccSendStruct->nick.ptr(),m_dccSendStruct->originalFileName.ptr(),m_dccSendStruct->fileLength);
			}
			if(g_pEventManager->eventEnabled(m_dccSendStruct->bGetSession ? KviEvent_OnDccGetTransferComplete : KviEvent_OnDccSendTransferComplete)){
				KviStr tmp(KviStr::Format,"%s %s %s %s %s %u",
					m_dccSendStruct->nick.ptr(),
					m_dccSendStruct->username.ptr(),
					m_dccSendStruct->host.ptr(),
					m_dccSendStruct->szAddress.ptr(),
					m_dccSendStruct->originalFileName.ptr(),
					m_dccSendStruct->fileLength);
				m_pFrm->m_pUserParser->callEvent(m_dccSendStruct->bGetSession ? KviEvent_OnDccGetTransferComplete : KviEvent_OnDccSendTransferComplete,this,tmp);
			}
			destroyDccSendStruct();
			if(g_pOptions->m_bAutoCloseDccSendOnSuccess)close(); //safe ?
		break;
		case KVI_DCC_EVENT_LISTENING:
			m_pFrm->m_pSocket->sendData(((KviDccEvent *)e)->m_dataString.ptr(),((KviDccEvent *)e)->m_dataString.len());
			outputNoFmt(KVI_OUT_DCCINFO,__tr("Sent DCC send request, waiting for reply..."));
		break;
		default:
			return KviWindow::event(e);
		break;
	}

	return true;
}

//================ resizeEvent ===============//

void KviDccSend::resizeEvent(QResizeEvent *)
{
	int hght = m_pFileLabel->sizeHint().height();
	m_pFileLabel->setGeometry(0,0,width(),hght);
	m_pProgress->setGeometry(0,hght,width(),hght);
	m_pStatLabel->setGeometry(0,hght * 2,width(),hght);
	m_pSplitter->setGeometry(0,hght * 3,width(),height() - (hght * 3));
}


void KviDccSend::acceptDccSendRequest(KviDccSendRequestStruct * dccSend)
{
	QFile * f = new QFile(dccSend->filePath.ptr());
	if(dccSend->resumeValue != 0){
		if(!f->open(IO_WriteOnly | IO_Append)){
			delete f;
			output(KVI_OUT_DCCERROR,__tr("Cannot start a RESUME session: The file %s does not exist"),dccSend->filePath.ptr());
			return;
		}
	} else {
		if(!f->open(IO_WriteOnly | IO_Truncate)){
			delete f;
			output(KVI_OUT_DCCERROR,__tr("Cannot open file %s for writing. DCC get failed"),dccSend->filePath.ptr());
			return;
		}
	}
	m_dccSendStruct = new KviDccSendStruct;
	m_dccSendStruct->dccSendParent    = this;
	m_dccSendStruct->nick             = dccSend->nick;
	m_dccSendStruct->username         = dccSend->username;
	m_dccSendStruct->host             = dccSend->host;
	m_dccSendStruct->filePath         = dccSend->filePath;
	m_dccSendStruct->fileLength       = dccSend->fileLength;
	m_dccSendStruct->uAddress         = dccSend->uAddress;
	m_dccSendStruct->uPort            = dccSend->uPort;
	m_dccSendStruct->receivedBytes    = dccSend->resumeValue;
	m_dccSendStruct->file             = f;
	m_dccSendStruct->originalFileName = dccSend->originalFileName;
	m_dccSendStruct->bListening       = false;
	m_dccSendStruct->bGetSession      = true;
	m_dccSendStruct->bSendZeroAck     = g_pOptions->m_bDccGetSendZeroAck;
//	m_dccSendStruct->uPortToListenOn  = 0; //not meaningful

	kvi_threadMutexInit(&(m_dccSendStruct->resumeMutex),0);

	KviStr tmp(KviStr::Format,__tr("File: %s [%u bytes]"),
		m_dccSendStruct->filePath.ptr(),m_dccSendStruct->fileLength);
	m_pFileLabel->setText(tmp.ptr());

	if(m_dccSendStruct->receivedBytes != 0){
		// need to resume
		m_pFrm->m_pSocket->sendFmtData("PRIVMSG %s :%cDCC RESUME %s %u %u%c",m_dccSendStruct->nick.ptr(),0x01,
				m_dccSendStruct->originalFileName.ptr(),m_dccSendStruct->uPort,
				m_dccSendStruct->receivedBytes,0x01);
		output(KVI_OUT_DCCINFO,__tr("Sent DCC RESUME request to %s [%s: %u of %u bytes], waiting for reply..."),
				m_dccSendStruct->nick.ptr(),m_dccSendStruct->originalFileName.ptr(),
				m_dccSendStruct->receivedBytes,m_dccSendStruct->fileLength);
	} else initiateGet();

}

static void kvi_threadGetPostStatusEvent(KviDccSendStruct * dcc)
{
	// Some calculations
	// Bug : If the user starts the dcc before midnight
	// and current time is after midnight we will have
	// negative elapsedSecs here..
	int elapsedSecs = dcc->startTime.secsTo(QTime::currentTime());
	if(elapsedSecs >= 0){
		int speed = (elapsedSecs ? (dcc->realBytesCount / elapsedSecs) : 0);

		int remainingSecs = dcc->fileLength - dcc->receivedBytes; //size

		if(speed > 0){
			remainingSecs = remainingSecs / speed;
			dcc->tmpBuffer.sprintf(
				__tr("Received %u bytes in %d min %d sec (%d bytes/sec): %d min %d sec remaining"),
				dcc->realBytesCount,elapsedSecs / 60,elapsedSecs % 60,speed,remainingSecs / 60,remainingSecs % 60);
		} else {
			dcc->tmpBuffer.sprintf(
				__tr("Received %u bytes in %d min %d sec (%d bytes/sec): ? min ? sec remaining"),
				dcc->realBytesCount,elapsedSecs / 60,elapsedSecs % 60,speed);
		}
	} else dcc->tmpBuffer = __tr("Yaaawn...midnight passed. I'm too tired to display more info...");

	KviDccEvent * e = new KviDccEvent(KVI_DCC_EVENT_DCCSENDSTATUS,dcc->tmpBuffer.ptr());

	if(dcc->fileLength > 0){
		// The progress
		e->m_param = (100 * dcc->receivedBytes) / dcc->fileLength;
	} else e->m_param = 100;

	g_pThreadEventDispatcher->postEvent(e,dcc->dccSendParent);
}

static bool kvi_threadGetReadIncomingData(int sock,KviDccSendStruct * dcc)
{
	// Check if there are data available on sock,
	// read it , split into messages and post data events
	// to the parent DCC window class.
	kvi_threadTestCancel();

	if(!kvi_dccCore_selectForRead(sock)){
		kvi_threadTestCancel();
		usleep(KVI_DCC_IDLE_TIME_USECS);
		return false;
	}
	// else data to read...
	kvi_threadTestCancel();
	//read data
	char buffer[1024];
	int readLength = read(sock,buffer,1024); //read() is not atomic :)


	kvi_threadTestCancel(); //Yes..I like this function call

	if(readLength <= 0){
		// oooops ?
		if(readLength == 0){
			// plain close. no probs
			// post the close event to the parent.
			// it is a KVI_DCC_EVENT_ERROR just because I need make the parent close the connection
			kvi_dccCore_errorExitThread(__tr("Remote end has closed the connection"),dcc->dccSendParent);
		} else {
			// error ?
			if((errno == EINTR)||(errno == EAGAIN))return false;
			// Yes...error :(
			int iError;
			switch(errno){
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				// Unhandled error...pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// post the error event to the parent and exit
			dcc->tmpBuffer.sprintf(__tr("READ ERROR: %s"),kvi_getErrorString(iError));
			kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccSendParent);
		}
	}
	// successfull read.
	kvi_threadTestCancel(); //Yeah!
	dcc->receivedBytes += readLength;
	dcc->realBytesCount += readLength;
	int writtenBytes = dcc->file->writeBlock(buffer,readLength);
	kvi_threadTestCancel();
	if(writtenBytes < readLength)kvi_dccCore_errorExitThread(__tr("Unable to write to file. DCC get failed"),dcc->dccSendParent);

	kvi_threadGetPostStatusEvent(dcc);

	kvi_threadTestCancel();

	return true;
}

static void kvi_threadGetSendAck(int sock,KviDccSendStruct *dcc)
{
	unsigned int toAckBytes=htonl(dcc->receivedBytes);
	::write(sock,&toAckBytes,sizeof(unsigned int));
	if(dcc->receivedBytes >= dcc->fileLength){
		KviDccEvent *e = new KviDccEvent(KVI_DCC_EVENT_FINISHED,__tr("File transfer completed"));
		g_pThreadEventDispatcher->postEvent(e,dcc->dccSendParent);
		kvi_threadExit(0); // the cleanup call will close the socket		
	}

}

static void kvi_threadGetLoop(int sock,KviDccSendStruct * dcc)
{
	if(dcc->bSendZeroAck)kvi_threadGetSendAck(sock,dcc); //Send a zero ack...some clients need that...

	for(;;){
		kvi_threadTestCancel();
		if(kvi_threadGetReadIncomingData(sock,dcc))
		{
			kvi_threadTestCancel();
			kvi_threadGetSendAck(sock,dcc);
		}
	}
}

static void *kvi_threadInitiateGet(void * data)
{
	kvi_threadInitialize();
	// Specific routine for accepting DCC Send requests
	// from users on IRC.
	// An user sent us a CTCP DCC with a port and an address
	// Now we try to contact that host and initiate a DCC Send...
	KviDccSendStruct * dcc = (KviDccSendStruct *)data; //Maybe convert it in just casts...
	// Prepare the target data
	struct in_addr inAddress;
	inAddress.s_addr=dcc->uAddress;
	struct sockaddr_in hostSockAddr;
	hostSockAddr.sin_family = AF_INET;
	hostSockAddr.sin_port   = htons(dcc->uPort);
	hostSockAddr.sin_addr   = inAddress;

	kvi_threadTestCancel();
	// Let's go...
	int sock = ::socket(PF_INET,SOCK_STREAM,0);
	// Don't forget to close the socket when terminating.
	// Will this cleanup routine be called also from an internal function call ?
	kvi_threadCleanupPush(kvi_dccCore_threadCleanupCloseSocket,&sock);

	if(sock < 0)kvi_dccCore_errorExitThread(__tr("Unable to create a stream socket. DCC get failed"),dcc->dccSendParent);

	kvi_threadTestCancel();

	if(fcntl(sock, F_SETFL, O_NONBLOCK) < 0)kvi_dccCore_errorExitThread(__tr("Unable to create a non-blocking stream socket. DCC get failed"),dcc->dccSendParent);

	kvi_threadTestCancel();

	// Ok...now try to connect
	int iError = kvi_dccCore_connect(sock,(struct sockaddr *)(&hostSockAddr),sizeof(hostSockAddr));

	if(iError != KVI_ERROR_Success){
		dcc->tmpBuffer.sprintf(__tr("CONNECT ERROR: %s"),kvi_getErrorString(iError));
		kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccSendParent);
	}

	kvi_threadTestCancel();

	// Now wait for connection...

	KviStr tmp(KviStr::Format,__tr("Connecting to %s on port %u"),dcc->szAddress.ptr(),dcc->uPort);
	KviDccEvent * msg = new KviDccEvent(KVI_DCC_EVENT_MSG,tmp.ptr());
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccSendParent);

	kvi_threadTestCancel();

	iError = kvi_dccCore_waitForOutgoingConnection(sock);

	if(iError != KVI_ERROR_Success){
		dcc->tmpBuffer.sprintf(__tr("CONNECT ERROR: %s"),kvi_getErrorString(iError));
		kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccSendParent);
	}

	kvi_threadTestCancel();

	msg = new KviDccEvent(KVI_DCC_EVENT_MSG,__tr("Connected: Receiving file"));
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccSendParent);

	kvi_threadTestCancel();

	dcc->startTime = QTime::currentTime();
	dcc->realBytesCount = 0;

	kvi_threadGetPostStatusEvent(dcc);
	kvi_threadGetLoop(sock,dcc);

	// Code that is newer reached...
	kvi_threadCleanupPop(0); //Do not execute

	return 0;
}

void KviDccSend::initiateGet()
{
	// This is called either by a direct DCC Get request
	// or a resume request from KviDccManager
//	debug("initiateGet");
	__range_valid(m_dccSendStruct);

#ifdef _KVI_DEBUG_CHECK_RANGE_
	if(m_bThreadRunning)debug("WARNING: initiateGet was called twice!");
#endif

	struct in_addr addr;
	addr.s_addr = m_dccSendStruct->uAddress;
	kvi_binaryIpToString(addr,m_dccSendStruct->szAddress);
	// Create the slave thread...
	m_bThreadRunning = true;

	if(kvi_threadCreate(&m_thread,0,kvi_threadInitiateGet,m_dccSendStruct) != 0){
		m_bThreadRunning = false;
		outputNoFmt(KVI_OUT_DCCERROR,__tr("Cannot create slave thread. DCC get failed"));
	}
}

//=====================================================================================================
//
// Request a DCC to an irc user
//

static void kvi_threadSendNotifyStalled(KviDccSendStruct * dcc)
{
	int elapsedSecs = dcc->startTime.secsTo(QTime::currentTime());
	if(elapsedSecs >= 0){
		int speed = (elapsedSecs ? (dcc->realSentBytesCount / elapsedSecs) : 0);

		dcc->tmpBuffer.sprintf(__tr("Sent %u bytes in %d min %d sec (%d bytes/sec): Received ack for %d bytes (stalled)"),
			dcc->realSentBytesCount,elapsedSecs / 60,elapsedSecs % 60,speed,dcc->ackedBytes);
	} else dcc->tmpBuffer = __tr("Stalled: Since it's past midnight, maybe the remote side felt asleep?");

	KviDccEvent * e = new KviDccEvent(KVI_DCC_EVENT_DCCSENDSTATUS,dcc->tmpBuffer.ptr());
	if(dcc->fileLength > 0){
		e->m_param = (100 * dcc->ackedBytes) / dcc->fileLength;
	} else e->m_param = 100;
	g_pThreadEventDispatcher->postEvent(e,dcc->dccSendParent);
}

static void kvi_threadSendPostStatusEvent(KviDccSendStruct * dcc)
{
	int elapsedSecs = dcc->startTime.secsTo(QTime::currentTime());
	if(elapsedSecs >= 0){
		int speed = (elapsedSecs ? (dcc->realSentBytesCount / elapsedSecs) : 0);

		dcc->tmpBuffer.sprintf(__tr("Sent %u bytes in %d min %d sec (%d bytes/sec): Received ack for %d bytes"),
			dcc->realSentBytesCount,elapsedSecs / 60,elapsedSecs % 60,speed,dcc->ackedBytes);

	} else dcc->tmpBuffer = __tr("Yaaawn...midnight passed. We're closed. Info will be displayed again tomorrow morning");

	KviDccEvent * e = new KviDccEvent(KVI_DCC_EVENT_DCCSENDSTATUS,dcc->tmpBuffer.ptr());
	if(dcc->fileLength > 0){
		e->m_param = (100 * dcc->ackedBytes) / dcc->fileLength;
	} else e->m_param = 100;

	g_pThreadEventDispatcher->postEvent(e,dcc->dccSendParent);
}

static void kvi_threadWriteBlockOfData(int sock,KviDccSendStruct *dcc)
{
	char buffer[KVI_DCC_SEND_MAX_BLOCK_SIZE];

	if(((dcc->bFastSend) || (dcc->realSentBytesCount == 0) || (dcc->ackedBytes >= dcc->sentBytes))
			&& (dcc->sentBytes < dcc->fileLength)){

		fd_set out_fd_set;
		struct timeval tv;
		// First check for write ability
		FD_ZERO(&out_fd_set);
		FD_SET(sock,&out_fd_set);
		tv.tv_sec = 0;
		tv.tv_usec = KVI_DCC_IDLE_TIME_USECS;

		kvi_threadTestCancel();

		// maybe I should check for select() errors here too ?
		if(select(sock + 1,0,&out_fd_set,0,&tv) != 1){
			dcc->stalledCount++;
			if(dcc->stalledCount == KVI_DCC_FAILED_WRITES_LIMIT_FOR_STALLED)kvi_threadSendNotifyStalled(dcc);
			usleep(KVI_DCC_IDLE_TIME_USECS); //relax
			return;
		}
		// else can write
		dcc->stalledCount = 0;

		kvi_threadTestCancel();
		
		int toWriteBytes = dcc->fileLength - dcc->sentBytes;
		if(toWriteBytes > dcc->blockSize)toWriteBytes = dcc->blockSize;

		toWriteBytes = dcc->file->readBlock(buffer,toWriteBytes);

		kvi_threadTestCancel();

		int wroteBytes = ::write(sock,buffer,toWriteBytes);

		kvi_threadTestCancel();

		if(wroteBytes < toWriteBytes){
			// oooops
			if(wroteBytes == -1){
				//error
				if((errno != EINTR)&&(errno != EAGAIN)){
					int iError;
					switch(errno){
						case EBADF:        iError = KVI_ERROR_BadFileDescriptor;     break;
						case EFAULT:       iError = KVI_ERROR_OutOfAddressSpace;     break;
						case EINVAL:       iError = KVI_ERROR_BadFileDescriptor;     break;
						case ENOSPC:       iError = KVI_ERROR_BrokenPipe;            break;
						case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
						case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
						// Unhandled error...pass errno to the strerror function
						default:           iError = -errno;                          break;
					}
					dcc->tmpBuffer.sprintf(__tr("WRITE ERROR: %s"),kvi_getErrorString(iError));
					kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccSendParent);
				} else dcc->file->at(dcc->file->at() - toWriteBytes);
			} else {
				// seek back the file
				int goBack = toWriteBytes - wroteBytes;
				dcc->file->at(dcc->file->at() - goBack);
			}
		}

		kvi_threadTestCancel();

		if(wroteBytes > 0){
			dcc->realSentBytesCount += wroteBytes;
			dcc->sentBytes += wroteBytes;
		}

		kvi_threadSendPostStatusEvent(dcc);

		kvi_threadTestCancel();

	}
	usleep(KVI_DCC_IDLE_TIME_USECS); //relax...always..
}


static void kvi_threadReceiveAck(int sock,KviDccSendStruct *dcc)
{
	// Check if there are data available on sock,
	// read it , split into messages and post data events
	// to the parent DCC window class.

	kvi_threadTestCancel();

	if(!kvi_dccCore_selectForRead(sock)){
		kvi_threadTestCancel();
		usleep(KVI_DCC_IDLE_TIME_USECS);
		return;
	} // else data to read...

	kvi_threadTestCancel();

	//read data
	unsigned int ack;
	int readLength = read(sock,&ack,sizeof(unsigned int)); //read() is not atomic :)

	kvi_threadTestCancel(); //Yes..I like this function call

	if(readLength < (int)sizeof(unsigned int)){
		// oooops ?
		if(readLength == 0)kvi_dccCore_errorExitThread(__tr("Remote end has closed the connection"),dcc->dccSendParent);
		else if(readLength < 0){
			// error ?
			if((errno == EINTR)||(errno == EAGAIN))return;
			// Yes...error :(
			int iError;
			switch(errno){
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				// Unhandled error...pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// post the error event to the parent and exit
			dcc->tmpBuffer.sprintf(__tr("READ ERROR: %s"),kvi_getErrorString(iError));
			kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccSendParent);
		} else kvi_dccCore_errorExitThread(__tr("Unrecoverable read error: Fragmented ack from peer"),dcc->dccSendParent);
	}

	// successfull read.

	kvi_threadTestCancel(); //Yeah!

	dcc->ackedBytes =ntohl(ack);

	if(dcc->ackedBytes >= dcc->fileLength){
		KviDccEvent *e = new KviDccEvent(KVI_DCC_EVENT_FINISHED,__tr("File transfer completed"));
		g_pThreadEventDispatcher->postEvent(e,dcc->dccSendParent);
		kvi_threadExit(0); // the cleanup call will close the socket		
	}

	kvi_threadSendPostStatusEvent(dcc);

	kvi_threadTestCancel();
}

static void kvi_threadSendLoop(int sock,KviDccSendStruct * dcc)
{
	for(;;){
		kvi_threadTestCancel();
		kvi_threadWriteBlockOfData(sock,dcc);
		kvi_threadTestCancel();
		kvi_threadReceiveAck(sock,dcc);
	}
}


static void * kvi_threadRequestDccSend(void * data)
{
	__debug("slaveThread");
	// Hack for Solaris (explained also in kvi_dns.cpp)

	kvi_threadInitialize();
	KviDccEvent * e;
	// with pthread_cleanup_push/pop we need to declare the variables at the beginning of the function...
	int newsock = -1;

	KviDccSendStruct * dcc = (KviDccSendStruct *)data; //Maybe convert it in just casts...

	struct sockaddr_in sockAddress;

	sockAddress.sin_family      = AF_INET;
	sockAddress.sin_port        = htons(dcc->uPortToListenOn);
	sockAddress.sin_addr.s_addr = INADDR_ANY;

	kvi_threadTestCancel();
	// Let's go...
	int sock = ::socket(PF_INET,SOCK_STREAM,0);
	// Don't forget to close the socket when terminating.
	kvi_threadCleanupPush(kvi_dccCore_threadCleanupCloseSocket,&sock);

	if(sock < 0)kvi_dccCore_errorExitThread(__tr("Unable to create a listening socket. DCC send failed"),dcc->dccSendParent);

	kvi_threadTestCancel();

	if((bind(sock,(struct sockaddr *)&sockAddress,sizeof(sockAddress))<0)||(listen(sock,100)<0) )
		kvi_dccCore_errorExitThread(__tr("Unable to set up a listening socket. DCC send failed"),dcc->dccSendParent);

	kvi_threadTestCancel();

	_this_should_be_socklen_t iSize = sizeof(sockAddress);
	getsockname(sock,(struct sockaddr*)&sockAddress,&iSize);

//	if(!kvi_dccCore_doListen(sock,&sockAddress)){
//		kvi_dccCore_errorExitThread(__tr("Unable to setup a listening socket. DCC Send failed."),dcc->dccSendParent);
//	}

	dcc->uPort    = ntohs(sockAddress.sin_port);
//	dcc->uAddress = m_pFrm->m_pSocket->getSockAddress();

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf(__tr("Listening on port %u"),dcc->uPort);
	e = new KviDccEvent(KVI_DCC_EVENT_MSG,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccSendParent);

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf("PRIVMSG %s :%cDCC SEND %s %u %u %u%c",dcc->nick.ptr(),0x01,
		dcc->originalFileName.ptr(),ntohl(dcc->uAddress),dcc->uPort,dcc->fileLength,0x01);

	e = new KviDccEvent(KVI_DCC_EVENT_LISTENING,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccSendParent);

	kvi_threadTestCancel();

	// Here we could also receive a RESUME message
	kvi_threadMutexLock(&(dcc->resumeMutex));
	dcc->bListening = true;
	kvi_threadMutexUnlock(&(dcc->resumeMutex));

	newsock = kvi_dccCore_waitForIncomingConnection(sock,&(dcc->uAddress),&(dcc->uPort),&(dcc->szAddress));

	kvi_threadMutexLock(&(dcc->resumeMutex));
	dcc->bListening = false;
	kvi_threadMutexUnlock(&(dcc->resumeMutex));

	// If we have received a resume here , the file is already at the right position
	// and sentBytes is already updated

	kvi_threadCleanupPop(0); //Do not execute
	close(sock); //close the old socket...

	kvi_threadCleanupPush(kvi_dccCore_threadCleanupCloseSocket,&newsock);

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf(__tr("Connected to %s on port %u"),dcc->szAddress.ptr(),dcc->uPort);
	e = new KviDccEvent(KVI_DCC_EVENT_MSG,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccSendParent);

	kvi_threadTestCancel();

	dcc->startTime = QTime::currentTime();

	kvi_threadSendLoop(newsock,dcc);

	// Code that is newer reached...
	kvi_threadCleanupPop(0); //Do not execute

	return 0;
}

void KviDccSend::requestDccSend(KviDccSendRequestStruct * dccSend)
{
	QFile * f = new QFile(dccSend->filePath.ptr());
	if(!f->open(IO_ReadOnly)){
		delete f;
		output(KVI_OUT_DCCERROR,__tr("Cannot open file %s for reading. DCC send failed"),dccSend->filePath.ptr());
		return;
	}

	QFileInfo fi(*f);

	m_dccSendStruct = new KviDccSendStruct;
	m_dccSendStruct->dccSendParent      = this;
	m_dccSendStruct->nick               = dccSend->nick;
	m_dccSendStruct->username           = dccSend->username;
	m_dccSendStruct->host               = dccSend->host;
	m_dccSendStruct->filePath           = dccSend->filePath;
	m_dccSendStruct->fileLength         = fi.size();
	m_dccSendStruct->uPort              = 0;
	m_dccSendStruct->sentBytes          = 0;
	m_dccSendStruct->realSentBytesCount = 0;
	m_dccSendStruct->file               = f;
	m_dccSendStruct->originalFileName   = fi.fileName();
	m_dccSendStruct->bListening         = false;
	m_dccSendStruct->bFastSend          = g_pOptions->m_bUseFastDccSend;
	m_dccSendStruct->ackedBytes         = 0;
	m_dccSendStruct->stalledCount       = 0;
	m_dccSendStruct->blockSize          = g_pOptions->m_iDccSendBlockSize;
	m_dccSendStruct->bGetSession        = false;
//	m_dccSendStruct->bSendZeroAck       = g_pOptions->m_bDccGetSendZeroAck; //not meaningful
	m_dccSendStruct->uPortToListenOn    = m_pFrm->m_pDccManager->getDccSendListenPort();

	if(m_dccSendStruct->blockSize > KVI_DCC_SEND_MAX_BLOCK_SIZE){
		m_dccSendStruct->blockSize = KVI_DCC_SEND_MAX_BLOCK_SIZE;
		g_pOptions->m_iDccSendBlockSize = KVI_DCC_SEND_MAX_BLOCK_SIZE;
	}

	if(m_dccSendStruct->blockSize <= 0){
		m_dccSendStruct->blockSize = 512;
		g_pOptions->m_iDccSendBlockSize = 512;
	}
	kvi_threadMutexInit(&(m_dccSendStruct->resumeMutex),0);
	if(g_pOptions->m_bReplaceSpacesInDccSendFileNames)m_dccSendStruct->originalFileName.replaceAll(' ',"_"); //replace all spaces with underscores

	m_dccSendStruct->uAddress  = m_pFrm->m_pSocket->getSockAddress();

	if(g_pOptions->m_bUseUserDefinedIpForDccRequests){
		struct in_addr addr;
		if(kvi_stringIpToBinaryIp(g_pOptions->m_szDccLocalIpAddress.ptr(),&addr)){
			output(KVI_OUT_DCCINFO,__tr("Using %s as localhost address"),g_pOptions->m_szDccLocalIpAddress.ptr());
			m_dccSendStruct->uAddress = addr.s_addr;
		} else {
			output(KVI_OUT_DCCERROR,__tr("IP %s is invalid : Using standard IP address"),g_pOptions->m_szDccLocalIpAddress.ptr());
			g_pOptions->m_bUseUserDefinedIpForDccRequests = false;
		}
	}

	if(m_dccSendStruct->uAddress == 0){
		outputNoFmt(KVI_OUT_DCCERROR,__tr("Cannot resolve localhost. DCC send failed"));
		return;
	}

	m_dccSendStruct->szAddress = __tr("unknown");

	KviStr tmp(KviStr::Format,__tr("File: %s [%u bytes]"),
		m_dccSendStruct->filePath.ptr(),m_dccSendStruct->fileLength);
	m_pFileLabel->setText(tmp.ptr());

	// now start listening....
	m_bThreadRunning = true;
	if(kvi_threadCreate(&m_thread,0,kvi_threadRequestDccSend,m_dccSendStruct) != 0){
		m_bThreadRunning = false;
		outputNoFmt(KVI_OUT_DCCERROR,__tr("Cannot create slave thread. DCC send failed"));
	}
}

bool KviDccSend::youAreListening(unsigned short uPort,const char *nick)
{
	if(!m_bThreadRunning)return false;
	if(!m_dccSendStruct)return false;
	bool bOk = false;
	kvi_threadMutexLock(&(m_dccSendStruct->resumeMutex));
	if(m_dccSendStruct->bListening){ //This is protected by the mutex
		if(uPort == m_dccSendStruct->uPort){
			bOk = kvi_strEqualCI(nick,m_dccSendStruct->nick.ptr());
		}
	}
	kvi_threadMutexUnlock(&(m_dccSendStruct->resumeMutex));
	return bOk;
}

bool KviDccSend::resumeForCurrentSend(unsigned long uResumePos)
{
	if(!m_bThreadRunning)return false;
	if(!m_dccSendStruct)return false;
	bool bOk = false;
	kvi_threadMutexLock(&(m_dccSendStruct->resumeMutex));
	if(m_dccSendStruct->bListening){ //This is protected by the mutex
		if(uResumePos <= m_dccSendStruct->fileLength){
			bOk = m_dccSendStruct->file->at(uResumePos);
			if(bOk){
				output(KVI_OUT_DCCINFO,__tr("Acknowledged RESUME. Transfer will initiate from position %u"),uResumePos);
				m_dccSendStruct->sentBytes = uResumePos;
			}
		}
	}
	kvi_threadMutexUnlock(&(m_dccSendStruct->resumeMutex));
	return bOk;
}

#include "m_kvi_dcc_send.moc"
