//
//   File : kvi_thread.cpp (/usr/build/NEW_kvirc/kvirc/src/kvilib/kvi_thread.cpp)
//   Last major modification : Tue Jul 6 1999 16:04:45 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//#define _KVI_DEBUG_CLASS_NAME_ "KviThread"
//#define _KVI_DEBUG_CHECK_RANGE_

#ifndef _GNU_SOURCE
	#define _GNU_SOURCE
#endif

#include "kvi_debug.h"

#define _KVI_THREAD_CPP_
#include "kvi_thread.h"

#include <unistd.h> //for pipe()
#include <errno.h>
#include <signal.h>

#include "kvi_settings.h"

#ifdef COMPILE_IGNORE_SIGALARM

void kvi_threadIgnoreSigalarm()
{
	// Funky hack for some Solaris machines (maybe others ?)
	// For an obscure (at least to me) reason
	// when using threads ,some part of the system
	// starts kidding us by sending a SIGALRM in apparently
	// "random" circumstances. (Xlib ?) (XServer ?)
	// The default action for SIGALRM is to exit the application.
	// Could not guess more about this stuff...
	// Here goes a "blind" hack for that.
	struct sigaction ignr_act;
	ignr_act.sa_handler = SIG_IGN;
	sigemptyset(&ignr_act.sa_mask);

#ifdef SA_NOMASK
	ignr_act.sa_flags   = SA_NOMASK;
#else
	ignr_act.sa_flags   = 0;
#endif

#ifdef SA_RESTART
	ignr_act.sa_flags  |= SA_RESTART;
#endif

	if(sigaction(SIGALRM,&ignr_act,0) == -1)debug("Failed to set SIG_IGN for SIGALRM.");
}

#else

void kvi_threadIgnoreSigalarm()
{
	// Not used
}

#endif

void kvi_threadSigpipeHandler(int)
{
	debug("Thread ????: Caught SIGPIPE: ignoring.");
}

void kvi_threadCatchSigpipe()
{
	struct sigaction act;
	act.sa_handler=&kvi_threadSigpipeHandler;
	sigemptyset(&(act.sa_mask));
	sigaddset(&(act.sa_mask), SIGPIPE);
	// CC: take care of SunOS which automatically restarts interrupted system
	// calls (and thus does not have SA_RESTART)
#ifdef SA_NOMASK
	act.sa_flags   = SA_NOMASK;
#else
	act.sa_flags   = 0;
#endif

#ifdef SA_RESTART
	act.sa_flags  |= SA_RESTART;
#endif

	if(sigaction(SIGPIPE,&act,0L) == -1)debug("Failed to set the handler for SIGPIPE."); 
}

void kvi_threadEnableDeferredCancel()
{
#ifdef _DECTHREADS_
	pthread_setcancel(CANCEL_ON);
	pthread_setasynccancel(CANCEL_OFF);
#else
	pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,0);
	pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED,0);  //Yes...this is the only safe one.
#endif
}

void kvi_threadDetachSelf()
{
#ifdef _DECTHREADS_
	pthread_t thr = pthread_self();
	pthread_detach(&thr);
#else
	pthread_detach(pthread_self());
#endif
}

void kvi_threadInitialize()
{
	kvi_threadDetachSelf();
	kvi_threadEnableDeferredCancel();
	kvi_threadIgnoreSigalarm();
	kvi_threadCatchSigpipe();
}

// Whom do I shot ? :)

// pipe : 0 is for READING , 1 for WRITING


KviThreadEventDispatcher::KviThreadEventDispatcher()
:QObject()
{
	kvi_threadMutexInit(&(m_mutex),0);

	if(pipe(m_pipe) != 0){
		debug("Thread controller : Can't create the communication pipe too many file descriptors open ?");
	}
	m_pSn = new QSocketNotifier(m_pipe[0],QSocketNotifier::Read);
	QObject::connect(m_pSn,SIGNAL(activated(int)),this,SLOT(socketNotifierFired(int)));
	m_pSn->setEnabled(true);

	m_pReceiverList = new QList<QObject>;
	m_pReceiverList->setAutoDelete(false);

	m_pEventList = new QList<KviThreadEventStruct>;
	m_pEventList->setAutoDelete(true);
}

KviThreadEventDispatcher::~KviThreadEventDispatcher()
{

	kvi_threadMutexDestroy(&(m_mutex));
	close(m_pipe[0]);
	close(m_pipe[1]);
	delete m_pSn;
	delete m_pReceiverList;
#ifdef _KVI_DEBUG_CHECK_RANGE_
	if(m_pEventList->count() > 0){
		debug("Thread controller : %d undispatched events",m_pEventList->count());
	}
#endif
	KviThreadEventStruct * str;
	while((str = m_pEventList->first())){
		delete str->pEvent;
		m_pEventList->removeFirst();
	}
	delete m_pEventList;
}

void KviThreadEventDispatcher::registerObject(QObject * o)
{
//	__enter("registerObject");
	// Probably main thread.
	kvi_threadMutexLock(&(m_mutex));
	m_pReceiverList->append(o);
	kvi_threadMutexUnlock(&(m_mutex));
//	__leave("registerObject");
}

void KviThreadEventDispatcher::unregisterObject(QObject * o)
{
//	__enter("unregisterObject");
	// Probably main thread.
	kvi_threadMutexLock(&(m_mutex));
	m_pReceiverList->removeRef(o);
	kvi_threadMutexUnlock(&(m_mutex));
//	__leave("unregisterObject");
}

void KviThreadEventDispatcher::postEvent(QEvent *e,QObject *r)
{
	// CLIENT THREAD
	// Post an event to be sent
	// after the dispatcher has been awaken...
	kvi_threadMutexLock(&(m_mutex));

	KviThreadEventStruct * str = new KviThreadEventStruct;
	str->pEvent = e;
	str->pReceiver = r;

	__range_valid(m_pReceiverList->findRef(r) != -1);
	m_pEventList->append(str);
	// Wake up the dispatcher
	write(m_pipe[1],"1",1); //Here an atomic write would really help :) , or a method to suspend all other threads...

	kvi_threadMutexUnlock(&(m_mutex));
}

void KviThreadEventDispatcher::socketNotifierFired(int fd)
{
	// MAIN THREAD
	char buffer[32];
	// Wake up the dispatcher...
	kvi_threadMutexLock(&(m_mutex));

	m_pSn->setEnabled(false);
	// read everything from the pipe...I really guess that there will be no more than few bytes of data..
	if(read(m_pipe[0],buffer,32) < 0){
		debug("Thread controller : Error %d while reading from the communication pipe",errno);
	}
	// If for some reason there are more than 512 bytes...well , something is going wrong...
	// Now dispatch all events...
	KviThreadEventStruct * s;
	while((s = m_pEventList->first())){
#ifdef _KVI_DEBUG_CLASS_NAME_
	#warning "maybe remove this find ref ?"
#endif
		if(m_pReceiverList->findRef(s->pReceiver) == -1){
			debug("Thread controller : Unregistered receiver in events queue.");
			delete s->pEvent;
			m_pEventList->removeFirst();
		} else {
			QEvent * e = s->pEvent;
			QObject * o = s->pReceiver;
			m_pEventList->removeFirst();
			kvi_threadMutexUnlock(&(m_mutex)); //Unlock , if the ,main thread wants to unregister some objects in the meantime...
			o->event(e);
			kvi_threadMutexLock(&(m_mutex)); //Relock...
			delete e;
		}
	}
	m_pSn->setEnabled(true);

	kvi_threadMutexUnlock(&(m_mutex));
}

//======================================================================================================
// Useful events

//
// Absolutely UNSECURE.
// Qt should NOT use the ENUM for the event types...these should be well documented defines
// In this way it is IMPOSSIBLE to create really safe custom events.
//
KviDccEvent::KviDccEvent(int type,const char * data)
:QEvent(QEvent::User)
{
	m_dataString = data;
	m_type = type;
}

KviDccEvent::~KviDccEvent()
{
}

#include "m_kvi_thread.moc"
