/***************************************************************************
 *   Copyright (C) 2006 by Daniel Gollub                                   *
 *                            <dgollub@suse.de>                            *
 *                         Danny Kukawka                                   *
 *                            <dkukawka@suse.de>, <danny.kukawka@web.de>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

/*! 
 * \file	detaileddialog.cpp
 * \brief	In this file can be found the detailed dialog related code.
 * \author	Daniel Gollub <dgollub@suse.de>
 * \author	Danny Kukawka, <dkukawka@suse.de>, <danny.kukawka@web.de>
 * \date	2006
 */

// KDE headers:
#include <kled.h>
#include <kiconloader.h>

// QT headers:
#include <qpushbutton.h>
#include <qlayout.h>
#include <qtimer.h>
#include <qgroupbox.h>

#include "kpowersave.h"
#include "detaileddialog.h"
#include "pdaemon.h"


/*! 
 * This is the default constructor of the class detaileddialog. 
 */
detaileddialog::detaileddialog( pDaemon *pdaemon, QPixmap *_pixmap, QWidget* parent, const char* name )
    			      : detailed_Dialog( parent, name, FALSE, WDestructiveClose ) {
	myDebug ("detaileddialog::detaileddialog");
	pd = pdaemon;
	pixmap = _pixmap;
	
	int batteries = numBatteries();
	numOfCPUs = pd->getCPUNum();

	this->setCaption(i18n("KPowersave Information Dialog"));
	
	// use this as compromise with current translation process
	// TODO: remove them in the next translation round
	GeneralGroup->setTitle(i18n("Miscellaneous"));
	ProcessorGroup->setTitle(i18n("CPUs"));

	if (batteries > 1) batteries++;

	ProcessorGridLayout = new QGridLayout(ProcessorFrame, numOfCPUs, 2, 0, 5, "ProcessorGridLayout");
	
	if (batteries > 0) {
		BatteryGroup->setTitle(i18n("Battery state:").remove(":"));
		BatteryGridLayout = new QGridLayout(BatteryFrame, batteries, 2, 0, 5, "BatteryGridLayout");

		for (int i = 0;  i < batteries; i++) {
			QLabel *Label = new QLabel(BatteryFrame, "BatteryLabel");
			if ((numBatteries() > 1) && (i == 0))
				Label->setText( i18n( "Total:" ));
			else if ((numBatteries() > 1) && (i > 0))
				Label->setText( i18n( "Battery %1" ).arg(i));
			else 
				Label->setText( i18n( "Battery %1" ).arg(i + 1));
	
			BatteryGridLayout->addWidget( Label, i , 0);
	
			KProgress *PBar = new KProgress(BatteryFrame, "BatteryPBar");
			PBar->setTextEnabled(true);
	
			BatteryPBar.append( PBar );
			BatteryGridLayout->addWidget( PBar, i , 1);
		}
		BatteryFrame->adjustSize();
		connect(pd, SIGNAL(generalDataChanged()), this, SLOT(setBattery()));
		connect(pd, SIGNAL(batteryInfoEvent()), this, SLOT(setBattery()));
		setBattery();
	} else {
		BatteryGroup->hide();
	}

	pd->checkCPUSpeed();

	ProcessorPictogram->setPixmap(SmallIcon("processor", 22));

	for (int i = 0; i < numOfCPUs; i++) {
		QLabel *Label = new QLabel(ProcessorFrame, "ProcessorLabel");
		Label->setText( i18n( "Processor %1" ).arg(i + 1));
		ProcessorGridLayout->addWidget( Label, i , 0);

		KProgress *CPUPBar = new KProgress(ProcessorFrame, "ProcessorPBar");
		CPUPBar->setTextEnabled(true);

		ProcessorPBar.append( CPUPBar );
		ProcessorGridLayout->addWidget( CPUPBar, i , 1);
	}
	ProcessorFrame->adjustSize();
	
	connect(OkButton, SIGNAL(clicked()), this, SLOT(closeDetailedDlg()));
	connect(pd, SIGNAL(generalDataChanged()), this, SLOT(setAC()));
	connect(pd, SIGNAL(schemeDataChanged()), this, SLOT(setInfos()));
	connect(pd, SIGNAL(generalDataChanged()), this, SLOT(setInfos()));

	if (pd->cpufreq_policy != CPU_UNSUPP) {
		// Check if cpufreq is available
		pd->getCPUMaxSpeed();
		setProcessor();	
	} else {
		// .. if not, use cpu throttling
		if (!pd->getCPUThrottlingState() || numOfCPUs <= 1) {
			connect(pd, SIGNAL(generalDataChanged()), this, SLOT(setProcessorThrottling()));
		}
		setProcessorThrottling();
	}

	setAC();
	setInfos();
	
	QString dummy;
	dummy = i18n("HAL daemon:");
	dummy = i18n("D-BUS daemon:");
	dummy = i18n("Autosuspend activated:");
	dummy = i18n("Modules unloaded:");
	dummy = i18n("enabled");
}

/*! This is the default destructor of class detaileddialog. */
detaileddialog::~detaileddialog()
{
	myDebug ("detailedDialog::~detaileddialog");
	// no need to delete child widgets, Qt does it all for us
}

/*!
 * \b SLOT called if the dialog is closed by the user.
 * We do some cleanups here.
 */
void detaileddialog::closeDetailedDlg() {
	myDebug ("detailedDialog::closeDetailedDlg");
	this->close();
	delete(this);
}

/*!
 * \b SLOT to set up the battery progress widgets.
 */
void detaileddialog::setBattery() {
	myDebug ("detaileddialog::setBattery");

	QString minutes;
	BatteryDetailed bd;
	BatteryGeneral bg;
	int batteries = numBatteries();

	if (batteries > 1) batteries++;

	for (int i=0; i < batteries ; i++) {

		if ( (numBatteries() > 1) && (i == 0)) {
			getBatteriesInfo(&bg);
		}
		else {
			int j = i;
			if (numBatteries() > 1 ) j -= 1;
			
			getBatteryDetailedInfo(j, &bd);
			getBatteryInfo(j, &bg);

			BatteryPBar[i]->setTextEnabled(true);
			BatteryPBar[i]->reset();

			if (bd.present != PRESENT_YES) {
				// FIXME: remove the '.' befor the next translation round
				BatteryPBar[i]->setFormat(i18n("not present.").remove("."));
				BatteryPBar[i]->setProgress(0);
				BatteryPBar[i]->setEnabled(FALSE);
				continue;
			}
		}

		int hours = bg.remaining_minutes / 60;
		minutes.setNum(bg.remaining_minutes % 60);
		minutes = minutes.rightJustify(2, '0');	

		// CHARG_STATE_CHARG_DISCHARG --> display only the percentage
		if (bg.charging_state == CHARG_STATE_CHARG_DISCHARG || bg.remaining_minutes < 0 ) {
			BatteryPBar[i]->setFormat("%p%");
		} else if (bg.charging_state == CHARG_STATE_CHARGING && checkACPI() == APM ) {
			// this should fix apm, where we have no time info if charging
			BatteryPBar[i]->setFormat("%p% " + i18n("charged"));
		} else if (bg.charging_state == CHARG_STATE_CHARGING) {
			QString temp = i18n("%1:%2 h until charged").arg(hours).arg(minutes);
			BatteryPBar[i]->setFormat(temp);
		} else if (bg.charging_state == CHARG_STATE_DISCHARGING) {
			QString temp = i18n("%1:%2 h remaining").arg(hours).arg(minutes);
			BatteryPBar[i]->setFormat(temp);
		
		} else {
			//fallback 
			BatteryPBar[i]->setFormat(i18n("unknown"));
		}

		if (bg.remaining_percent < 0)
			BatteryPBar[i]->setProgress(0);
		else	
			BatteryPBar[i]->setProgress(bg.remaining_percent);
		BatteryPBar[i]->setEnabled(true);
	}

	BatteryPictogram->setPixmap(*pixmap);
}

/*!
 * \b SLOT to set up the Processor/CPU progress bar widgets.
 */
void detaileddialog::setProcessor() {
	myDebug ("detaileddialog::setProcessor");

	pd->checkCPUSpeed();

	for (int i=0; i < numOfCPUs; i++) {
		myDebug("ID: %i(%i) cur_freq: %i max_freq: %i", i, pd->cpufreq_speed.count(), pd->cpufreq_speed[i], pd->cpufreq_max_speed[i]); 

		//ProcessorPBar[i]->setTextEnabled(true);
		if (pd->cpufreq_speed[i] > 0) {
			// CPU/Core is back from offline
			if(ProcessorPBar[i]->progress() == 0)
				pd->getCPUMaxSpeed();

			if(ProcessorPBar[i]->progress() != pd->cpufreq_speed[i]) {
				// get max cpu freq and set it to the max of the progressbar
				int maxfreq = pd->cpufreq_max_speed[i];
				ProcessorPBar[i]->setTotalSteps(maxfreq);
	
				// display  1400 MHz instead of 1400%
				ProcessorPBar[i]->setFormat(i18n("%v MHz"));
				ProcessorPBar[i]->setProgress(pd->cpufreq_speed[i]);
				ProcessorPBar[i]->setEnabled(true);
			}
		} else {
			ProcessorPBar[i]->setFormat(i18n("deactivated"));
			ProcessorPBar[i]->setProgress(0);
			ProcessorPBar[i]->setEnabled(FALSE);
		}
	}
	QTimer::singleShot(333, this, SLOT(setProcessor()));
}

/*!
 * \b SLOT to set up the Processor/CPU bar widgets for cpu throttling machines.
 */
void detaileddialog::setProcessorThrottling() {
	myDebug("detaileddialog::setProcessorThrottling");

	bool throttling = pd->getCPUThrottlingState();
	pd->checkCPUSpeedThrottling();

	for (int i=0; i < numOfCPUs; i++) { 
		if (throttling)
			myDebug("Throttling CPU - freq: %i throttling state: %i %%", 
				pd->cpufreq_speed[i], pd->cpu_throttling[i]);
		else
			myDebug("CPU - freq: %i", pd->cpufreq_speed[i]);
	
		if (throttling && pd->cpufreq_speed[i] > 0 && pd->cpu_throttling[i] >= 0) {
			// get max cpu freq and set it to the max of the progressbar
			ProcessorPBar[i]->setTotalSteps(100);
			QString ProgressString = QString("%1% (%2 MHz)").arg(100 - pd->cpu_throttling[i]).arg(pd->cpufreq_speed[i]);
			ProcessorPBar[i]->setFormat(i18n(ProgressString));
			ProcessorPBar[i]->setProgress(100 - pd->cpu_throttling[i]);
			ProcessorPBar[i]->setEnabled(true);
		} else if (pd->cpufreq_speed[i] < 0) {
			ProcessorPBar[i]->setFormat(i18n("deactivated"));
			ProcessorPBar[i]->setProgress(0);
			ProcessorPBar[i]->setEnabled(FALSE);
		} else {
			ProcessorPBar[i]->setTotalSteps(pd->cpufreq_speed[i]);
			ProcessorPBar[i]->setFormat(i18n("%v MHz"));
			ProcessorPBar[i]->setProgress(pd->cpufreq_speed[i]);
			ProcessorPBar[i]->setEnabled(true);
		}
	}

	if (throttling || numOfCPUs > 1) {
		// currently there are no events we can use to get actual data
		// so we recheck data ever 2 secs to register changes in the 
		// throttling state and if a CPU/core online state change
		QTimer::singleShot(2000, this, SLOT(setProcessorThrottling()));
	}

}

/*!
 * \b SLOT to set up the AC status within the Led widget.
 */
void detaileddialog::setAC() {
	myDebug ("detaileddialog::setAC");

	if (pd->on_AC_power == 1) {
		LabelACStatus->setText( i18n("plugged in") );
		LedAC->on();
	} else {
		LedAC->off();
		LabelACStatus->setText( i18n("unplugged") );
	}
}

/*!
 * \b SLOT to set all additional informtation as e.g. CPUFrequency policy
 * or current scheme
 */
void detaileddialog::setInfos() {
	myDebug ("detaileddialog::setInfos");

	QString display;
	QString displayValue;

	if(!pd->schemes.currentScheme().isEmpty())
		display += i18n("Current Scheme: ") + "\n";
		displayValue += i18n(pd->schemes.currentScheme()) + "\n";
		if(pd->schemes.currentScheme() == pd->schemes.acScheme())
			InfoPictogram->setPixmap(SmallIcon("scheme_power", 22));
		else if(pd->schemes.currentScheme() == pd->schemes.batteryScheme())
			InfoPictogram->setPixmap(SmallIcon("scheme_powersave", 22));
		else if(pd->schemes.currentScheme() == "Acoustic")
			InfoPictogram->setPixmap(SmallIcon("scheme_acoustic", 22));
		else if(pd->schemes.currentScheme() == "Presentation")
			InfoPictogram->setPixmap(SmallIcon("scheme_presentation", 22));
		else if(pd->schemes.currentScheme() == "AdvancedPowersave")
			InfoPictogram->setPixmap(SmallIcon("scheme_advanced_powersave", 22));
		else
			InfoPictogram->setPixmap(SmallIcon("kpowersave", 22));

	if(pd->daemon_running == 1) {
		if (pd->cpufreq_policy != CPU_UNSUPP) {
			display += i18n("Current CPU Frequency Policy:") + "\n";
			switch (pd->cpufreq_policy){
				case CPU_HIGH:
					displayValue += i18n("Performance")  + "\n";
					break;
				case CPU_AUTO:
					displayValue += i18n("Dynamic") + "\n";
					break;
				case CPU_LOW:
					displayValue += i18n("Powersave") + "\n";
					break;
			}
		}

		if (numBatteries() > 0 && pd->battery_state != BAT_HAL_ERROR && pd->battery_state != BAT_NONE ) {
			int batteries = numBatteries();
			
			for (int i=0; i <= batteries ; i++) {
				BatteryDetailed bd;
				getBatteryDetailedInfo(i, &bd);
				if (bd.present != PRESENT_YES)
					batteries--;
			}

			if (batteries > 0) {
				display += i18n("Battery state:") + "\n";
				switch (pd->battery_state){
					case BAT_CRIT:
						displayValue += i18n("Critical") + "\n";
						break;
					case BAT_LOW:
						displayValue += i18n("Low") + "\n";
						break;
					case BAT_WARN:
						displayValue += i18n("Warning") + "\n";
						break;
					case BAT_NORM:
						displayValue += i18n("ok") + "\n";
						break;
					default:
						displayValue += i18n("unknown") + "\n";
						break;
				}
			}
		}
		
		if(pd->getBrightnessLevels() > 0) {
			display += i18n("Set brightness supported:") + "\n";
			displayValue += i18n("yes") + "\n";
		} else {
			display += i18n("Set brightness supported:") + "\n";
			displayValue += i18n("no") + "\n";
		}

		display += i18n("Powersave Daemon:");
		displayValue += i18n("running");
	}
	else {
		display += i18n("Powersave Daemon:");
		displayValue += i18n("not running");
	}

	if(!display.isEmpty())
		InfoLabel->setText(display);
		InfoLabelValue->setText(displayValue);
}

#include "detaileddialog.moc"
