/* @(#) buffer.c 1.18 @(#) */
/***************************************************************\
*	Copyright (c) 1999 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: BUFFER
\***************************************************************/

#define _KOALAMUD_BUFFER_C "@(#) nitehawk@localhost.1ststep.net|lib/koala/buffer.c|20000827025247|12619 @(#)"

#include "autoconf.h"

#include "version.h"
#include "koalatypes.h"
#include "buffer.h"
#include "network.h"
#include "log.h"

/* Ring state parameters */
/* head = first byte of the buffer
 * tail = one byte past the end of the buffer
 * If head == tail, the buffer is empty
 * if head == tail+1 || (head == 0 && tail == RINGSIZE), the buffer is full
 * If tail < head, we have wrapped around the end of the ring
 *
 * Maximum ring fill level is RINGSIZE-1
 */

/* Output buffer functions */
/* Queue data to be sent */
koalaerror buffer_queue(pdescriptor desc, const char *data, int len)
{
	int space = 0;
	int numtocopytoend = 0;
	int numtocopytohead = 0;
	int totaltocopy = 0;

	/* If the head is the same as the tail, then we want to put the data at
	 * the beginning of the ring for easier handling
	 */
	if (desc->buffer.outhead == desc->buffer.outtail)
	{
		desc->buffer.outhead = 0;
		desc->buffer.outtail = len < OUTRINGSIZE ? len : OUTRINGSIZE;
		strncpy(desc->buffer.outring, data, desc->buffer.outtail);
		return KESUCCESS;
	}

	/* determine how much space is available in the ring */
	if ((desc->buffer.outtail < desc->buffer.outhead))
	{
		space = desc->buffer.outhead - desc->buffer.outtail;
		totaltocopy = len < space ? len : space;
		numtocopytoend = totaltocopy;
		numtocopytohead = 0;
	}
	else
	{
		space = OUTRINGSIZE - desc->buffer.outtail + desc->buffer.outhead;
		totaltocopy = len < space ? len : space;
		if ((OUTRINGSIZE - desc->buffer.outtail) < totaltocopy)
		{
			numtocopytoend = OUTRINGSIZE - desc->buffer.outtail;
			numtocopytohead = totaltocopy - numtocopytoend;
		}
		else
		{
			numtocopytoend = totaltocopy;
			numtocopytohead = 0;
		}
	}

	if (numtocopytoend)
	{
		strncpy(desc->buffer.outring + desc->buffer.outtail, data,
				numtocopytoend);
		desc->buffer.outtail += numtocopytoend;
	}
	if (numtocopytohead)
	{
		strncpy(desc->buffer.outring, data, numtocopytohead);
		desc->buffer.outtail = numtocopytohead;
	}

	return KESUCCESS;
}

/* Write data out to descriptor */
koalaerror buffer_sendbytes(pdescriptor desc, int len)
{
	int numtosend = 0;
	int fromhead = 0;
	int fromtail = 0;
	int buffered = 0;

	/* If the buffer is empty, do a quick return */
	if (desc->buffer.outhead == desc->buffer.outtail)
		return KESUCCESS;

	/* figure out how much is in the buffer to be sent */
	if (desc->buffer.outtail > desc->buffer.outhead)
	{
		buffered = desc->buffer.outtail - desc->buffer.outhead;
		numtosend = buffered > len ? len : buffered;
		fromhead = numtosend;
		fromtail = 0;
	}
	else
	{
		buffered = OUTRINGSIZE - desc->buffer.outtail + desc->buffer.outhead;
		numtosend = buffered > len ? len : buffered;
		fromhead = OUTRINGSIZE - desc->buffer.outhead;
		if (fromhead >= numtosend)
		{
			fromhead = numtosend;
			fromtail = 0;
		}
		else
		{
			fromtail = numtosend - fromhead;
		}
	}
	
	/* Send data out to socket */
	if (fromhead)
	{
		netwrite(desc, desc->buffer.outring + desc->buffer.outhead, fromhead);
	}
	if (fromtail)
	{
		netwrite(desc, desc->buffer.outring, fromtail);
	}

	/* Update the position pointers */
	if (numtosend >= buffered) /* should never be greater then, but just in
								  case */
	{
		/* If we just sent all the data in the buffer, we can just set the
		 * position to 0
		 */
		desc->buffer.outhead = desc->buffer.outtail = 0;
	}
	else
	{
		if (fromtail) /* Buffer is currently wrapped */
		{
			desc->buffer.outhead = fromtail;
		}
		else
		{
			desc->buffer.outhead += numtosend;
		}
	}

	return KESUCCESS;
}

/* Is the out buffer empty? */
inline bool buffer_outbufempty(pdescriptor desc)
{
	return (desc->buffer.outhead == desc->buffer.outtail);
}

/* Input buffer functions */
/* Read data from socket to input buffer */
koalaerror buffer_receive(pdescriptor desc)
{
	unsigned int numread;
	int space = 0;
	int numtocopytoend = 0;
	int numtocopytohead = 0;
	int totaltocopy = 0;

	/* If the head is the same as the tail, then we want to put the data at
	 * the beginning of the ring for easier handling
	 */
	if (desc->buffer.inhead == desc->buffer.intail)
	{
		space = INRINGSIZE - 1;
		numtocopytoend = space;
		numtocopytohead = 0;
		desc->buffer.inhead = desc->buffer.intail = 0;
	}
	/* determine how much space is available in the ring */
	else if (desc->buffer.intail < desc->buffer.inhead)
	{
		space = desc->buffer.inhead - desc->buffer.intail;
		numtocopytoend = space;
		numtocopytohead = 0;
	}
	else
	{
		space = INRINGSIZE - desc->buffer.intail + desc->buffer.inhead;
		totaltocopy = space;
		if ((INRINGSIZE - desc->buffer.intail) < totaltocopy)
		{
			numtocopytoend = INRINGSIZE - desc->buffer.intail;
			numtocopytohead = totaltocopy - numtocopytoend;
		}
		else
		{
			numtocopytoend = totaltocopy;
			numtocopytohead = 0;
		}
	}
	/* Read as much data as possible from the descriptor */
	if (numtocopytoend)
	{
		numread = numtocopytoend;
		netread(desc, desc->buffer.inring + desc->buffer.intail, &numread);
		desc->buffer.intail += (numread - 1);
		if (numread < numtocopytoend)
		{
			/* We have already read everything that is available, return now
			 * to prevent blocking */
			return KESUCCESS;
		}
	}
	if (numtocopytohead)
	{
		numread = numtocopytohead;
		netread(desc, desc->buffer.inring, &numread);
		desc->buffer.intail = numread - 1;
	}

	return KESUCCESS;
}

/* Read a number of bytes from buffer */
koalaerror buffer_readbytes(pdescriptor desc, char *buf, int *len, bool exact)
{
	int buffered = 0;
	int fromhead = 0;
	int fromtail = 0;

	if (desc->buffer.inhead == desc->buffer.intail)
	{
		*len = 0;
		return KENOTENOUGH;
	}

	/* Figure out how much data is in the buffer */
	if (desc->buffer.inhead > desc->buffer.intail)
	{
		/* Buffer is wrapped */
		buffered = INRINGSIZE - desc->buffer.inhead + desc->buffer.intail;
	}
	else
	{
		/* Buffer is not wrapped */
		buffered = desc->buffer.intail - desc->buffer.inhead;
	}

	/* Do we have enough data? */
	if (exact && buffered < *len)
	{
		*len = 0;
		return KENOTENOUGH;
	}

	/* Figure out how much data to copy from each half of the buffer */
	if (*len > buffered)
	{
		*len = buffered;
	}
	
	if (*len + desc->buffer.inhead > INRINGSIZE)
	{
		/* We have to handle buffer wrapping */
		fromtail = INRINGSIZE - desc->buffer.inhead;
		fromhead = *len - fromtail;
	}
	else
	{
		/* We won't wrap the buffer during byte grabbing, only need to deal
		 * with fromtail
		 */
		fromtail = *len;
	}

	/* Copy data into the buffer */
	if (fromtail)
	{
		memcpy(buf, desc->buffer.inring + desc->buffer.inhead, *len);
		desc->buffer.inhead += *len;
	}
	if (fromhead)
	{
		memcpy(buf, desc->buffer.inring, *len);
		desc->buffer.inhead = *len - 1;
	}

	return KESUCCESS;
}

/* Read single char from buffer */
char buffer_readchar(pdescriptor desc)
{
	char c = '\0';

	if (desc->buffer.inhead == desc->buffer.intail)
	{
		return '\0';
	}

	c = desc->buffer.inring[desc->buffer.inhead];
	desc->buffer.inhead++;
	if (desc->buffer.inhead >= INRINGSIZE - 1)
	{
		desc->buffer.inhead = 0;
	}

	return c;
}

/* Read a single word delimited by whitespace from the buffer */
/* special note:  if the first non-whitespace is not alphanumeric, we end the
 * word on the first alphanumeric */
koalaerror buffer_readword(pdescriptor desc, char *word, int maxlen)
{
	/* For now search the buffer ourselves.  It may be possible to adopt
	 * strsep to do the work, but the semantics for wraping the buffer are
	 * currently unclear */

	char *position = desc->buffer.inring + desc->buffer.inhead;
	char *cpy = word; 
	int num = 0;
	bool wordstarted = FALSE;
	bool special = FALSE;

	/* If the buffer is empty */
	if (desc->buffer.inhead == desc->buffer.intail)
	{
		return KENOTENOUGH;
	}

	while (TRUE)
	{
		/* Make sure we havn't ran out of buffer space */
		if (desc->buffer.inhead == desc->buffer.intail)
		{
			if (!wordstarted)
			{
				/* We emptied the buffer without reaching the start of a word,
				 * Report that we didn't have enough buffer */
				return KENOTENOUGH;
			}
			else
			{
				/* We emptied the buffer without reaching an end char */
				/* This could mean that the rest of the word is still pending on
				 * the network or we have the entire word with no termination.
				 * In either case, we report it as a success */
				return KESUCCESS;
			}
		}

		/* If we havn't started the word yet */
		if (!wordstarted)
		{
			if (isgraph(*position))
			{
				wordstarted = TRUE;
				if (!isalnum(*position))
				{
					special = TRUE;
				}
			}
			else
			{
				position++;
				desc->buffer.inhead++;
				if (desc->buffer.inhead > INRINGSIZE)
				{
					desc->buffer.inhead = 0;
					position = desc->buffer.inring;
				}
				continue;
			}
		}

		/* Is the current character white space? */
		if (isspace(*position))
		{
			/* We must have reached the end of the word, null terminate it and
			 * return success */
			*cpy = '\0';
			/* Skip the terminating space */
			desc->buffer.inhead++;
			if (desc->buffer.inhead > INRINGSIZE)
			{
				desc->buffer.inhead = 0;
				position = desc->buffer.inring;
			}
			return KESUCCESS;
		}

		/* If this is a special word and we have an alphanumeric, end the word
		 */
		if (special && isalnum(*position))
		{
			/* We must have reached the end of the word, null terminate it and
			 * return success */
			*cpy = '\0';
			return KESUCCESS;
		}

		if ((num + 1) == maxlen)
		{
			/* We ran out of buffer space for the word.  Caller can allocate
			 * more memory and call us again while pointing to the null
			 * terminator to finish getting the word
			 */
			*cpy = '\0';
			return KENOMEM;
		}

		/* If we get to this point, we want to copy the current character and
		 * increment our position */
		*cpy++ = *position++;
		num++;
		desc->buffer.inhead++;
		if (desc->buffer.inhead > INRINGSIZE)
		{
			desc->buffer.inhead = 0;
			position = desc->buffer.inring;
		}
	}

	return KESUCCESS;
}

/* Read one line from the buffer terminated by \r or \n */
koalaerror buffer_readline(pdescriptor desc, char *line, int maxlen)
{
	/* For now search the buffer ourselves.  It may be possible to adopt
	 * strsep to do the work, but the semantics for wraping the buffer are
	 * currently unclear */

	char *position = desc->buffer.inring + desc->buffer.inhead;
	char *cpy = line; 
	int num = 0;

	/* If the buffer is empty */
	if (desc->buffer.inhead == desc->buffer.intail)
	{
		return KENOTENOUGH;
	}

	/* When copying a line, we copy everything, including leading white space
	 */
	while (TRUE)
	{
		/* Make sure we havn't ran out of buffer space */
		if (desc->buffer.inhead == desc->buffer.intail)
		{
			/* We emptied the buffer without reaching an end char */
			/* This could mean that the rest of the line is still pending on
			 * the network or we have the entire line with no termination.
			 * In either case, we report it as a success */
			return KESUCCESS;
		}

		/* Is the current character white space? */
		if (*position == '\r' || *position == '\n')
		{
			/* Purge the pairing character as well, if it exists */
			if (*position == '\r' && (*(position + 1) == '\n'))
			{
				desc->buffer.inhead++;
				if (desc->buffer.inhead >= INRINGSIZE)
				{
					desc->buffer.inhead = 0;
				}
			}
			if (*position == '\n' && (*(position + 1) == '\r'))
			{
				desc->buffer.inhead++;
				if (desc->buffer.inhead >= INRINGSIZE)
				{
					desc->buffer.inhead = 0;
				}
			}

			/* We must have reached the end of the line, null terminate it and
			 * return success */
			*cpy = '\0';
			return KESUCCESS;
		}

		if (num == maxlen)
		{
			/* We ran out of buffer space for the line.  Caller can allocate
			 * more memory and call us again while pointing to the null
			 * terminator to finish getting the line
			 * Don't null terminate this (testing)
			 */
			//*cpy = '\0';
			return KENOMEM;
		}

		/* If we get to this point, we want to copy the current character and
		 * increment our position */
		*cpy++ = *position++;
		num++;
		desc->buffer.inhead++;
		if (desc->buffer.inhead > INRINGSIZE)
		{
			desc->buffer.inhead = 0;
			position = desc->buffer.inring;
		}
	}

	return KESUCCESS;
}

/* Flush remainder of current line from the input buffer */
koalaerror buffer_flushline(pdescriptor desc)
{
	/* For now search the buffer ourselves.  It may be possible to adopt
	 * strsep to do the work, but the semantics for wraping the buffer are
	 * currently unclear */

	char *position = desc->buffer.inring + desc->buffer.inhead;

	/* If the buffer is empty */
	if (desc->buffer.inhead == desc->buffer.intail)
	{
		return KESUCCESS;
	}

	/* When copying a line, we copy everything, including leading white space
	 */
	while (TRUE)
	{
		/* Make sure we havn't ran out of buffer space */
		if (desc->buffer.inhead == desc->buffer.intail)
		{
			/* If we are out of buffer space, we are definatly at the end of
			 * the current line */
			return KESUCCESS;
		}

		/* Is the current character a cr or lf? */
		if (*position == '\r' || *position == '\n')
		{
			/* Purge the pairing character as well, if it exists */
			if (*position == '\r' && (*(position + 1) == '\n'))
			{
				desc->buffer.inhead++;
				if (desc->buffer.inhead >= INRINGSIZE)
				{
					desc->buffer.inhead = 0;
				}
			}
			if (*position == '\n' && (*(position + 1) == '\r'))
			{
				desc->buffer.inhead++;
				if (desc->buffer.inhead >= INRINGSIZE)
				{
					desc->buffer.inhead = 0;
				}
			}

			/* We reached the end of the line, return */
			return KESUCCESS;
		}

		/* Once we get here, we know that we are on a character of the line,
		 * hop past it */
		position++;
		desc->buffer.inhead++;
		if (desc->buffer.inhead > INRINGSIZE)
		{
			desc->buffer.inhead = 0;
			position = desc->buffer.inring;
		}
	}

	return KESUCCESS;
}

/* Verify that we have a full line of input buffered */
bool buffer_isalinein(pdescriptor desc)
{
	char *pos;
	int count = 0;
	int buffered;
	bool wrapped = FALSE;

	/* Verify descriptor to be safe */
	if (!desc)
	{
		logmsg(LOGERR, "Bad descriptor caught");
		return FALSE;
	}

	/* Point the start of the buffer */
	pos = desc->buffer.inring + desc->buffer.inhead;

	/* Ammount buffered */
	buffered = desc->buffer.inhead < desc->buffer.intail ?
			desc->buffer.intail - desc->buffer.inhead :
			INRINGSIZE - (desc->buffer.inhead - desc->buffer.intail);

	/* If the buffer is empty */
	if (desc->buffer.inhead == desc->buffer.intail)
	{
		return FALSE;
	}

	 /* Look for the end of a line
	 */
	while (count < buffered)
	{
		/* Is the current character a cr or lf? */
		if (*pos == '\r' || *pos == '\n')
		{
			/* We have an EOL character */
			return TRUE;
		}

		/* Once we get here, we know that we are on a character of the line,
		 * hop past it */
		pos++;
		count++;
		if (desc->buffer.inhead + count > INRINGSIZE && !wrapped)
		{
			pos = desc->buffer.inring;
			wrapped = TRUE;
		}
	}

	return FALSE;
}
