/*
vmc-fast.c

C portion of the kForth virtual machine
Copyright (c) 1998--2004 Krishna Myneni, Creative Consulting
for Research and Education

Revisions:
	9-27-1998 -- created.
	3-1-1999  -- added C_open, C_lseek, C_close, C_read, C_write
	3-2-1999  -- fixed C_open, added C_ioctl
	5-27-1999 -- added C_key, C_accept
	6-09-1999 -- added C_numberquery
	6-12-1999 -- fixed sign for C_numberquery
	7-14-1999 -- fixed C_numberquery to reject junk for base > 10
	9-12-1999 -- added C_system
	10-7-1999 -- added C_chdir
	10-9-1999 -- added C_timeanddate
	10-28-1999 -- added C_keyquery
	01-26-2001 -- added C_usec
	09-19-2001 -- modified C_accept to handle backspace key
	09-05-2002 -- added C_search, C_compare, and C_msfetch
	03-18-2004 -- added C_facosh, C_fasinh, C_fatanh, C_fcosh, C_fsinh,
                            C_ftanh
	03-21-2004 -- recoded single argument math functions using a
                        MACRO based on BK's 3/20/04 revisions to vmc.c
        09-04-2004 -- added functions save_term(), restore_term(),
                        echo_on(), and echo_off() to fix echo problems
                        with KEY?
	09-05-2004 -- added signal handler interface
*/

#include<sys/types.h>
#include<sys/time.h>
#include<sys/timeb.h>
#include<sys/stat.h>
#include<termios.h>
#include<stdio.h>
#include<signal.h>
#include<unistd.h>
#include<time.h>
#include<fcntl.h>
#include<stdlib.h>
#include<math.h>

#define OP_IVAL 'I'
#define OP_ADDR 'A'
#define WSIZE 4
#define TRUE -1
#define FALSE 0
#define E_V_NOTADDR 1
#define E_V_BADCODE 6
#define E_V_STK_UNDERFLOW   7
#define E_V_QUIT  8

#define byte unsigned char

extern int* GlobalSp;
extern byte* GlobalIp;
extern int* GlobalRp;
extern int* BottomOfStack;
extern int* BottomOfReturnStack;
extern int Base;

extern int vm(byte*);

struct timeval ForthStartTime;
struct termios tios0;
double* pf;
double f;
char temp_str[256];
char key_query_char = 0;

// signal dispatch table

void* signal_xtmap [32] =
{
    NULL,              //  1  SIGHUP, Hangup
    NULL,              //  2  SIGINT, Interrupt
    NULL,              //  3  SIGQUIT, Quit
    NULL,              //  4  SIGILL, Illegal instruction
    NULL,              //  5  SIGTRAP, Trace trap
    NULL,              //  6  SIGABRT, Abort
    NULL,              //  7  SIGBUS,  Bus error
    NULL,              //  8  SIGFPE,  Floating-point exception
    NULL,              //  9  SIGKILL, Kill (unblockable)
    NULL,              // 10  SIGUSR1, User-defined
    NULL,              // 11  SIGSEGV, Segmentation fault
    NULL,              // 12  SIGUSR2, User-defined
    NULL,              // 13  SIGPIPE, Broken pipe
    NULL,              // 14  SIGALRM, Alarm clock
    NULL,              // 15  SIGTERM, Termination
    NULL,              // 16  SIGSTKFLT, Stack fault
    NULL,              // 17  SIGCHLD, Child status changed
    NULL,              // 18  SIGCONT, Continue execution
    NULL,              // 19  SIGSTOP, Stop (unblockable)
    NULL,              // 20  SIGTSTP, Keyboard stop
    NULL,              // 21  SIGTTIN, Background read from tty
    NULL,              // 22  SIGTTOU, Background write to tty
    NULL,              // 23  SIGURG, Urgent condition on socket
    NULL,              // 24  SIGXCPU, CPU time limit exceeded
    NULL,              // 25  SIGXFSZ, File size limit exceeded
    NULL,              // 26  SIGVTARM, Virtual alarm clock
    NULL,              // 27  SIGPROF, Profiling alarm clock
    NULL,              // 28  SIGWINCH, Window size change
    NULL,              // 29  SIGPOLL,  Pollable event occured
    NULL,              // 30  SIGPWR,  Power failure restart
    NULL,              // 31  SIGUNUSED,  Not used
    NULL
};

static void forth_signal_handler (int); 

#define DOUBLE_FUNC(x)   pf = (double*)(GlobalSp+1); *pf=x(*pf);
  
int C_ftan  () { DOUBLE_FUNC(tan)  return 0; }
int C_facos () { DOUBLE_FUNC(acos) return 0; }
int C_fasin () { DOUBLE_FUNC(asin) return 0; }
int C_fatan () { DOUBLE_FUNC(atan) return 0; }
int C_fsinh () { DOUBLE_FUNC(sinh) return 0; }
int C_fcosh () { DOUBLE_FUNC(cosh) return 0; }
int C_ftanh () { DOUBLE_FUNC(tanh) return 0; }
int C_fasinh () { DOUBLE_FUNC(asinh) return 0; }
int C_facosh () { DOUBLE_FUNC(acosh) return 0; }
int C_fatanh () { DOUBLE_FUNC(atanh) return 0; }
int C_fexp  () { DOUBLE_FUNC(exp)  return 0; }
int C_fln   () { DOUBLE_FUNC(log)  return 0; }
int C_flog  () { DOUBLE_FUNC(log10) return 0; }
 
int C_fpow ()
{
	pf = (double*)(GlobalSp + 1);
	f = *pf;
	++pf;
	*pf = pow (*pf, f);
	GlobalSp += 2;
	return 0;
}				

int C_fmin ()
{
	pf = (double*)(GlobalSp + 1);
	f = *pf;
	++pf;
	if (f < *pf) *pf = f;
	GlobalSp += 2;
	return 0;
}

int C_fmax ()
{
	pf = (double*)(GlobalSp + 1);
	f = *pf;
	++pf;
	if (f > *pf) *pf = f;
	GlobalSp += 2;
	return 0;
}

int C_open ()
{
  /* stack: ( ^str flags -- fd | return the file descriptor )
     ^str is a counted string with the pathname, flags
     indicates the method of opening (read, write, etc.)  */

  int flags, mode = 0, fd;
  char* pname;

  ++GlobalSp;
  flags = *GlobalSp;
  ++GlobalSp;

      pname = *((char**)GlobalSp);
      ++pname;
      if (flags & O_CREAT) mode = S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH;
      fd = open (pname, flags, mode);
      *GlobalSp-- = fd;
      return 0;

}
      
int C_lseek ()
{
  /* stack: ( fd offset mode -- error | set file position in fd ) */

  int fd, offset, mode;
  ++GlobalSp;
  mode = *GlobalSp++;
  offset = *GlobalSp++;
  fd = *GlobalSp;
  *GlobalSp-- = lseek (fd, offset, mode);
  return 0;
}

int C_close ()
{

  /* stack: ( fd -- err | close the specified file and return error code ) */

  int fd;
  ++GlobalSp;
  fd = *GlobalSp;
  *GlobalSp-- = close(fd);
  return 0;
}

int C_read ()
{
  /* stack: ( fd buf count -- length | read count bytes into buf from fd ) */
  int fd, count;
  void* buf;

  ++GlobalSp;
  count = *GlobalSp++;
    
      buf = *((void**)GlobalSp);
      ++GlobalSp; 
      fd = *GlobalSp;
      *GlobalSp-- = read (fd, buf, count);

      return 0;

}

int C_write ()
{
  /* stack: ( fd buf count  -- length | write count bytes from buf to fd ) */
  int fd, count;
  void* buf;

  ++GlobalSp;
  count = *GlobalSp++;

      buf = *((void**)GlobalSp);
      ++GlobalSp;
      fd = *GlobalSp;
      *GlobalSp-- = write (fd, buf, count);

      return 0;

}

int C_ioctl ()
{
  /* stack: ( fd request addr -- err | device control function ) */
  int fd, request;
  char* argp;

  ++GlobalSp;
  argp = *((char**) GlobalSp);  /* don't do type checking on argp */
  ++GlobalSp; 
  request = *GlobalSp++;
  fd = *GlobalSp;
  *GlobalSp-- = ioctl(fd, request, argp);
  return 0;
}
/*----------------------------------------------------------*/

void save_term ()
{
    tcgetattr(0, &tios0);
}

void restore_term ()
{
    tcsetattr(0, TCSANOW, &tios0);
}

void echo_off ()
{
  struct termios t;
  tcgetattr(0, &t);
  t.c_lflag &= ~ECHO;
  tcsetattr(0, TCSANOW, &t);
}

void echo_on ()
{
  struct termios t;
  tcgetattr(0, &t);
  t.c_lflag |= ECHO;
  tcsetattr(0, TCSANOW, &t);
}
/*----------------------------------------------------------*/

int C_key ()
{
  /* stack: ( -- n | wait for keypress and return key code ) */

  char ch;
  int n;
  struct termios t1, t2;

  if (key_query_char)
    {
      ch = key_query_char;
      key_query_char = 0;
    }
  else
    {
      tcgetattr(0, &t1);
      t2 = t1;
      t2.c_lflag &= ~ICANON;
      t2.c_lflag &= ~ECHO;
      t2.c_cc[VMIN] = 1;
      t2.c_cc[VTIME] = 0;
      tcsetattr(0, TCSANOW, &t2);

      do {
	n = read(0, &ch, 1);
      } while (n != 1);

      tcsetattr(0, TCSANOW, &t1);
    }

  *GlobalSp-- = ch;

 
  return 0;
}
/*----------------------------------------------------------*/

int C_keyquery ()
{
  /* stack: ( a -- b | return true if a key is available ) */

  char ch = 0;
  struct termios t1, t2;

  if (key_query_char)
    {
      *GlobalSp-- = -1;
    }
  else
    {
      tcgetattr(0, &t1);
      t2 = t1;
      t2.c_lflag &= ~ICANON;
      t2.c_lflag &= ~ECHO;
      t2.c_cc[VMIN] = 0;
      t2.c_cc[VTIME] = 0;
      tcsetattr(0, TCSANOW, &t2);

      *GlobalSp-- = read(0, &ch, 1) ? -1 : 0;
      if (ch) key_query_char = ch;
      tcsetattr(0, TCSANOW, &t1);
    }


  return 0;
}      
/*----------------------------------------------------------*/

int C_accept ()
{
  /* stack: ( a n1 -- n2 | wait for n characters to be received ) */

  char ch, *cp, *cpstart, *bksp = "\010 \010";
  int n1, n2, nr;
  struct termios t1, t2;

  ++GlobalSp;
  n1 = *GlobalSp++;

  cp = *((char**)GlobalSp);
  cpstart = cp;

  tcgetattr(0, &t1);
  t2 = t1;
  t2.c_lflag &= ~ICANON;
  t2.c_lflag &= ~ECHO;
  t2.c_cc[VMIN] = 1;
  t2.c_cc[VTIME] = 0;
  tcsetattr(0, TCSANOW, &t2);


  n2 = 0;
  while (n2 < n1)
    {
      nr = read (0, cp, 1);
      if (nr == 1) 
	{
	  if (*cp == 10) 
	    break;
	  else if (*cp == 127)
	  {
	    write (0, bksp, 3);
	    --cp; --n2;
	    if (cp < cpstart) cp = cpstart;
	    if (n2 < 0) n2 = 0;
	  }
	  else
	  {
	    write (0, cp, 1);
	    ++n2; ++cp;
	  }
	}
    }
  *GlobalSp-- = n2;


  tcsetattr(0, TCSANOW, &t1);
  return 0;
}

/*----------------------------------------------------------*/

int C_numberquery ()
{
  /* stack: ( a -- d b | translate characters into number using current base ) */

  char *token, *pStr, *endp;
  int b, sign;
  unsigned u;

  ++GlobalSp;
  if (GlobalSp > BottomOfStack) return E_V_STK_UNDERFLOW; /* stack underflow */

  token = *((char**)GlobalSp);
  ++token;
  pStr = token;
  u = 0;
  sign = FALSE;
  b = FALSE;

  if ((*pStr == '-') || isdigit(*pStr) || (isalpha(*pStr) && (Base > 10)
					   && ((*pStr - 55) < Base)))
    {
      if (*pStr == '-') {sign = TRUE;}
      ++pStr;
      while (isdigit(*pStr) || (isalpha(*pStr) && (Base > 10) &&
				((*pStr - 55) < Base)))	    
	{
	  ++pStr;
	}
      if (*pStr == 0)
        {
	  u = strtoul(token, &endp, Base);
	  b = TRUE;
        }

    }

  *GlobalSp-- = u;

  *GlobalSp-- = sign;

  *GlobalSp-- = b;
  
  return 0;
}
/*----------------------------------------------------------*/

int C_system ()
{
  /* stack: ( ^str -- n | n is the return code for the command in ^str ) */

  char* cp;
  int nc, nr;

  ++GlobalSp; 
  cp = (char*) (*GlobalSp);
  nc = *cp;
  strcpy (temp_str, "exec ");
  strncpy (temp_str+5, cp+1, nc);
  temp_str[5 + nc] = 0;
  nr = system(temp_str);
  *GlobalSp-- = nr;

  return 0;
}
/*----------------------------------------------------------*/

int C_chdir ()
{
  /* stack: ( ^path -- n | set working directory to ^path; return error code ) */

  char* cp;
  int nc;

  ++GlobalSp;
  cp = (char*)(*GlobalSp);
  nc = *cp;
  strncpy (temp_str, cp+1, nc);
  temp_str[nc] = 0;
  *GlobalSp-- = chdir(temp_str);

  return 0;
}
/*-----------------------------------------------------------*/

int C_timeanddate ()
{
  /* stack: ( -- sec min hr day mo yr | fetch local time ) */

  time_t t;
  struct tm t_loc;

  time (&t);
  t_loc = *(localtime (&t));

  *GlobalSp-- = t_loc.tm_sec;
  *GlobalSp-- = t_loc.tm_min; 
  *GlobalSp-- = t_loc.tm_hour; 
  *GlobalSp-- = t_loc.tm_mday; 
  *GlobalSp-- = 1 + t_loc.tm_mon; 
  *GlobalSp-- = 1900 + t_loc.tm_year;

  return 0;
}
/*---------------------------------------------------------*/

int C_usec ()
{
  /* stack: ( u -- | delay for u microseconds ) */

  struct timeval tv1, tv2;
  unsigned int usec;

  ++GlobalSp;
  usec = *GlobalSp;

  gettimeofday (&tv1, NULL);
  tv1.tv_usec += usec;

  while (tv1.tv_usec >= 1000000)
    {
      tv1.tv_sec++;
      tv1.tv_usec -= 1000000;
    }

  do
    {
      gettimeofday (&tv2, NULL);
    } while (timercmp(&tv1, &tv2, >)) ;

  return 0;
}
/*------------------------------------------------------*/

void set_start_time ()
{
  /* this is not a word in the Forth dictionary; it is
     used by the initialization routine on startup     */

  gettimeofday (&ForthStartTime, NULL);
}

int C_msfetch ()
{
  /* stack: ( -- msec | return msec elapsed since start of Forth ) */
  
  struct timeval tv;
  gettimeofday (&tv, NULL);
  *GlobalSp-- = (tv.tv_sec - ForthStartTime.tv_sec)*1000 + 
    (tv.tv_usec - ForthStartTime.tv_usec)/1000;

  return 0;
}
/*------------------------------------------------------*/

int C_search ()
{
  /* stack: ( a1 u1 a2 u2 -- a3 u3 flag ) */

  char *str1, *str2, *cp, *cp2;
  unsigned int n, n_needle, n_haystack, n_off, n_rem;
  ++GlobalSp;
  n = *GlobalSp;
  ++GlobalSp;
  str2 = (char*)(*GlobalSp++);
  if (n > 255) n = 255;
  n_needle = n;
  n_haystack = *GlobalSp++;  // size of search buffer

  str1 = (char*)(*GlobalSp);  
  n_rem = n_haystack;
  n_off = 0;
  cp = str1;
  cp2 = NULL;

  if (n_needle > 0)
  {
      while (n_rem >= n_needle)
      {
	  cp = (char *) memchr(cp, *str2, n_rem);
	  if (cp && (n_rem >= n_needle))
	  {
	      n_rem = n_haystack - (cp - str1);
	      if (memcmp(cp, str2, n_needle) == 0)
	      {
		  cp2 = cp;
		  n_off = (int)(cp - str1);
		  break;
	      }
	      else
	      {
		  ++cp; --n_rem;
	      }
	  }
	  else
	      n_rem = 0;
      }
  }

  if (cp2 == NULL) n_off = 0;
  *GlobalSp-- = (int)(str1 + n_off); 
  *GlobalSp-- = n_haystack - n_off; 
  *GlobalSp-- = cp2 ? -1 : 0 ;

  return 0;
}
/*------------------------------------------------------*/

int C_compare ()
{
  /* stack: ( a1 u1 a2 u2 -- n ) */

  char *str1, *str2;
  int n1, n2, n, ncmp, nmin;
  ++GlobalSp;
  n2 = *GlobalSp;
  ++GlobalSp; 

  str2 = (char*)(*GlobalSp++);
  n1 = *GlobalSp++;
  str1 = (char*)(*GlobalSp);

  if ((n1 <= 0) || (n2 <= 0))
  {
      n = -1;
  }
  else
  {
      nmin = (n1 < n2) ? n1 : n2;
      ncmp = memcmp(str1, str2, nmin);

      if (ncmp == 0)
      {
	  if (n1 == n2) n = 0;
	  else if (n1 < n2) n = -1;
	  else n = 1;
      }
      else if (ncmp < 0)  n = -1;
      else n = 1;
  }
  *GlobalSp-- = n;
  return 0;
}
/*------------------------------------------------------*/

int C_setitimer ()
{
    // stack: ( timer-type avalue aoldvalue -- flag ) 
    
    int type, e;
    struct itimerval *v1, *v2;

    v2 = (struct itimerval*) *++GlobalSp; 
    v1 = (struct itimerval*) *++GlobalSp;
    type = *++GlobalSp; 

    e = setitimer (type, v1, v2);

    *GlobalSp-- = e;
    return 0;
}

int C_getitimer ()
{
    // stack: ( timer-type  avalue -- flag )
    
    int type, e;
    struct itimerval *v;

    v = (struct itimerval*) *++GlobalSp;
    type = *++GlobalSp;

    e = getitimer (type, v);

    *GlobalSp-- = e;
    return 0;
}

int C_raise ()
{
    // stack: ( signum -- ior )
    int signum = *++GlobalSp;
    *GlobalSp-- = raise(signum);
    return 0;
}

int C_forth_signal ()
{
    // Install a Forth handler for specified signal 
    // stack: ( xt n -- oldxt )

    int signum;
    void *xt, *oldxt;

    signum = *++GlobalSp;
    if ((signum > 0) && (signum < 31))
    {
	++GlobalSp;
	oldxt = signal_xtmap[signum-1];
	xt = (void *) *GlobalSp;
	switch ((int) xt)
	{
	    case (int) SIG_DFL:
		// Reset the default signal handler if xt = 0
		signal (signum, SIG_DFL);
		xt = 0;
		break;
	    case (int) SIG_IGN:
		// Ignore the signal if xt = 1
		signal (signum, SIG_IGN);
		xt = 0;
		break;
	    default:
		// All other xt s must be valid addresses to opcodes
		signal (signum, forth_signal_handler);
		break;
	}
        signal_xtmap[signum-1] = xt;
	*GlobalSp-- = (int) oldxt;
    }
    else
	return E_V_BADCODE;

    return 0;
}
/*-----------------------------------------------------*/

static void forth_signal_handler (int signum)
{
    // Take the required action for the signal
    //   by looking up and executing the appropriate
    //   Forth word which has been designated to
    //   handle this signal.

    int e;
    void* xt = signal_xtmap[signum-1];
    *GlobalSp-- = signum;
    if (xt)
    {
	e = vm((byte*) xt);
	// printf ("\nvm returns %d", e);
	// if (e == E_V_QUIT) we need to do a longjmp
    }
}


