/***************************************************************************
                          barcodecache.cpp  -  description
                             -------------------
    begin                : Sam Aug 16 2003
    copyright            : (C) 2003 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "barcodecache.h"

// KDE includes
#include <kapplication.h>
#include <kconfig.h>
#include <kstandarddirs.h>

// Qt includes
#include <qdir.h>
#include <qdom.h>
#include <qfile.h>
#include <qpixmap.h>

BarcodeCache* BarcodeCache::self = 0;

BarcodeCache* BarcodeCache::instance()
{
    if( !self )
        self = new BarcodeCache();

    return self;
}
               
BarcodeCache::BarcodeCache()
    : QObject( 0, "barcodecache" )
{
    m_xml = locateLocal( "cache", "kbarcode/index.xml" );
    m_dir = locateLocal( "cache", "kbarcode/" );

    KConfig* config = kapp->config();
    config->setGroup( "BarcodeCache" );
    m_size = config->readNumEntry( "Size", 50 );

    connect( kapp, SIGNAL( aboutToQuit() ), this, SLOT( flush() ) );

    QFile file( m_xml );
    if( file.open( IO_ReadOnly ) ) {
        QDomDocument doc( "BarcodeCache" );
        if ( doc.setContent( &file ) ) {
            QDomNode n = doc.documentElement().firstChild();
            while( !n.isNull() ) {
                QDomElement e = n.toElement(); // try to convert the node to an element.
                if( !e.isNull() )
                    if( e.tagName() == "barcode" ) {
                   
                        struct data v;
                        v.settings = readBarcode( &e );
                        v.path = e.attribute( "path", "" );
                        // keep compability with older versions of KBarcode < 1.7.3
                        QString r = e.attribute( "resolution", "300" );
                        v.resolutionx = e.attribute( "resolutionx", r ).toInt();
                        v.resolutiony = e.attribute( "resolutiony", r ).toInt();
                        
                        if( m_data.count() < size() )
                            m_data.append( v );
                    }

                n = n.nextSibling();
            }
        }
        file.close();
    }
}

BarcodeCache::~BarcodeCache()
{
}

QPixmap* BarcodeCache::read( const barcodeData & b, int resolutionx, int resolutiony, const QString & value )
{
    data v = fillStruct( b, resolutionx, resolutiony );
    v.settings.value = value;
    
    for( unsigned i = 0; i < m_data.count(); i++ )   
        if( v == m_data[i] ) {
            QPixmap* p = new QPixmap( m_data[i].path );

            if( p->isNull() ) {
                QFile::remove( m_data[0].path );
                m_data.remove( m_data.at( 0 ) );            
                return 0;
            } else
                return p;
        }

    return 0;
}

void BarcodeCache::write( const barcodeData & b, int resolutionx, int resolutiony, QPixmap* pix, const QString & value )
{
    QString path;
    do {
        path = m_dir + KApplication::randomString( 8 ) + ".png";
    } while( QFile::exists( path ) );
    
    data v = fillStruct( b, resolutionx, resolutiony );
    v.settings.value = value;
    v.path = path;
    
    pix->save( path, "PNG" );
    m_data.append( v );

    while( m_data.count() > size() && size() > 0 ) {
        QFile::remove( m_data[0].path );
        m_data.remove( m_data.at( 0 ) );
    }
}

BarcodeCache::data BarcodeCache::fillStruct( const barcodeData & b, int resolutionx, int resolutiony )
{
    data v;

    v.settings = b;
    v.resolutionx = resolutionx;
    v.resolutiony = resolutiony;

    return v;
}

void BarcodeCache::flush()
{
    KConfig* config = kapp->config();
    config->setGroup( "BarcodeCache" );
    config->writeEntry( "Size", size() );
    config->sync();
    
    QDomDocument doc( "BarcodeCache" );
    QDomElement root = doc.createElement( "Barcodes" );
    doc.appendChild( root );

    for( unsigned int i = 0; i < m_data.count(); i++ ) {
        QDomElement tag = doc.createElement( "barcode" );
        writeBarcode( &tag, &m_data[i].settings, 0, 0, true );
        tag.setAttribute( "path", m_data[i].path );
        tag.setAttribute( "resolutionx", m_data[i].resolutionx );
        tag.setAttribute( "resolutiony", m_data[i].resolutiony );

        root.appendChild( tag );
    }
    
    QString xml = doc.toString();

    QFile file( m_xml );
    if( !file.open( IO_WriteOnly ) ) {
        qDebug("Can't write barcode cache to: " + m_xml );
        return;
    }

    file.writeBlock( xml, xml.length() );

    file.close();
}

void BarcodeCache::clear()
{
    // Delete all barcodes
    for( unsigned int i = 0; i < m_data.count(); i++ )
        QFile::remove( m_data[i].path );

    // Look for files that are in the cache dir but not in our internal
    // structure. Those files can be present if kbarcode crashed.
    QDir dir( m_dir );
    dir.setNameFilter( "*.png" );
    for( unsigned int i = 0; i < dir.count(); i++ )
        QFile::remove( m_dir + dir[i] );
    
    m_data.clear();
}

