#include "../config.h"
#include "common.h"

#include <unistd.h>
#include <signal.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>

#include <inotifytools/inotifytools.h>
#include <inotifytools/inotify.h>

#define _GNU_SOURCE
#include <getopt.h>

extern char *optarg;
extern int optind, opterr, optopt;

#define EXIT_OK 0
#define EXIT_ERROR 1
#define EXIT_TIMEOUT 2

#define SORT_TOTAL_ASC  0
#define SORT_TOTAL_DESC IN_ONESHOT

// METHODS
bool parse_opts(
  int * argc,
  char *** argv,
  int * events,
  int * timeout,
  int * verbose,
  int * zero,
  int * sort,
  int * recursive
);

void print_help();

static bool done;
static int sort_event;
static int sort;

int compare( const void * first, const void * second ) {
	static unsigned int stat_one;
	static unsigned int stat_two;

	stat_one = inotifytools_get_stat_by_wd( (*(const int *)first), sort_event );
	stat_two = inotifytools_get_stat_by_wd( (*(const int *)second), sort_event );

	if ( SORT_TOTAL_DESC == sort || sort < 0 )
		return stat_two - stat_one;
	else
		return stat_one - stat_two;
}

void handle_impatient_user( int signal ) {
	static int times_called = 0;
	if ( times_called ) {
		fprintf( stderr, "No statistics collected, asked to abort before all "
		         "watches could be established.\n" );
		exit(1);
	}
	fprintf( stderr, "No statistics have been collected because I haven't "
	         "finished establishing\n"
	         "inotify watches yet.  If you are sure you want me to exit, "
	         "interrupt me again.\n" );
	++times_called;
}

void handle_signal( int signal ) {
	done = true;
}

int main(int argc, char ** argv)
{
	int events = 0;
	int timeout = 0;
	int verbose = 0;
	int zero = 0;
	int recursive = 0;
	sort = SORT_TOTAL_DESC;
	done = false;

	signal( SIGINT, handle_impatient_user );

	// Parse commandline options, aborting if something goes wrong
	if ( !parse_opts( &argc, &argv, &events, &timeout, &verbose, &zero, &sort,
	                 &recursive ) ) {
		return EXIT_ERROR;
	}

	sort_event = ( SORT_TOTAL_ASC == sort || SORT_TOTAL_DESC == sort )
	               ? 0 : abs( sort );

	if ( !inotifytools_initialize() ) {
		fprintf(stderr, "Couldn't initialize inotify.  Are you running Linux "
		                "2.6.13 or later, and was the\n"
		                "CONFIG_INOTIFY option enabled when your kernel was "
		                "compiled?  If so, \n"
		                "something mysterious has gone wrong.  Please e-mail "
		                PACKAGE_BUGREPORT "\n"
		                " and mention that you saw this message.\n");
		return EXIT_ERROR;
	}

	// Attempt to watch file
	// If events is still 0, make it all events.
	if ( !events )
		events = IN_ALL_EVENTS;

	unsigned int num_watches = 0;
	unsigned int status;
	fprintf( stderr, "Establishing watches...\n" );
	for ( int i = 0; i < argc; ++i ) {

		if ( recursive && verbose ) {
			fprintf( stderr, "Setting up watch(es) on %s\n", argv[i] );
		}

		if ( recursive ) {
			status = inotifytools_watch_recursively( argv[i], events );
		}
		else {
			status = inotifytools_watch_file( argv[i], events );
		}
		if ( !status ) {
			if ( inotifytools_error() == ENOSPC ) {
				fprintf(stderr, "Failed to watch %s; upper limit on inotify "
				                "watches reached!\n", argv[i] );
				fprintf(stderr, "Please increase the amount of inotify watches "
				        "allowed per user via `/proc/sys/fs/inotify/"
				        "max_user_watches'.\n");
			}
			else {
				fprintf(stderr, "Failed to watch %s: %s\n", argv[i],
				        strerror( inotifytools_error() ) );
			}
			return EXIT_ERROR;
		}
		if ( recursive && verbose ) {
			fprintf( stderr, "OK, %s is now being watched.\n", argv[i] );
		}
	}
	num_watches = inotifytools_get_num_watches();

	if ( verbose ) {
		fprintf( stderr, "Total of %d watches.\n",
		         num_watches );
	}
	fprintf( stderr, "Finished establishing watches, now collecting statistics.\n" );

	if ( timeout && verbose ) {
		fprintf( stderr, "Will listen for events for %d seconds.\n", timeout );
	}

	signal( SIGINT, handle_signal );
	signal( SIGHUP, handle_signal );
	signal( SIGTERM, handle_signal );
	if ( timeout ) {
		signal( SIGALRM, handle_signal );
		alarm( timeout );
	}

	inotifytools_initialize_stats();
	// Now wait till we get event
	struct inotify_event * event;

	do {
		event = inotifytools_next_event( 0 );
		if ( !event ) {
			if ( !inotifytools_error() ) {
				return EXIT_TIMEOUT;
			}
			else if ( inotifytools_error() != EINTR ) {
				fprintf(stderr, "%s\n", strerror( inotifytools_error() ) );
				return EXIT_ERROR;
			}
			else {
				continue;
			}
		}

		if ( recursive && (event->mask & IN_CREATE) ) {
			// New file - if it is a directory, watch it
			static char * new_file;

			asprintf( &new_file, "%s%s",
			          inotifytools_filename_from_wd( event->wd ), event->name );

			if ( isdir(new_file) ) {
				status = inotifytools_watch_recursively( new_file, events );
				if ( !status ) {
					fprintf( stderr, "Couldn't watch new directory %s: %s\n",
					         new_file, strerror( inotifytools_error() ) );
					fprintf( stderr, "Statistics may be incomplete or "
					         "incorrect.\n" );
				}
				else {
					num_watches = inotifytools_get_num_watches();
				}
			}
			free( new_file );
		}

	} while ( !done );

	if ( !inotifytools_get_stat_total( 0 ) ) {
		fprintf( stderr, "No events occurred.\n" );
		return EXIT_OK;
	}

	// OK, go through the watches and print stats.
	printf("total  ");
	if ( (IN_ACCESS & events) &&
	     ( zero || inotifytools_get_stat_total( IN_ACCESS ) ) )
		printf("access  ");
	if ( ( IN_MODIFY & events) &&
	     ( zero || inotifytools_get_stat_total( IN_MODIFY ) ) )
		printf("modify  ");
	if ( ( IN_ATTRIB & events) &&
	     ( zero || inotifytools_get_stat_total( IN_ATTRIB ) ) )
		printf("attrib  ");
	if ( ( IN_CLOSE_WRITE & events) &&
	     ( zero || inotifytools_get_stat_total( IN_CLOSE_WRITE ) ) )
		printf("close_write  ");
	if ( ( IN_CLOSE_NOWRITE & events) &&
	     ( zero || inotifytools_get_stat_total( IN_CLOSE_NOWRITE ) ) )
		printf("close_nowrite  ");
	if ( ( IN_OPEN & events) &&
	     ( zero || inotifytools_get_stat_total( IN_OPEN ) ) )
		printf("open  ");
	if ( ( IN_MOVED_FROM & events) &&
	     ( zero || inotifytools_get_stat_total( IN_MOVED_FROM ) ) )
		printf("moved_from  ");
	if ( ( IN_MOVED_TO & events) &&
	     ( zero || inotifytools_get_stat_total( IN_MOVED_TO ) ) )
		printf("moved_to  ");
	if ( ( IN_MOVE_SELF & events) &&
	     ( zero || inotifytools_get_stat_total( IN_MOVE_SELF ) ) )
		printf("move_self  ");
	if ( ( IN_CREATE & events) &&
	     ( zero || inotifytools_get_stat_total( IN_CREATE ) ) )
		printf("create  ");
	if ( ( IN_DELETE & events) &&
	     ( zero || inotifytools_get_stat_total( IN_DELETE ) ) )
		printf("delete  ");
	if ( ( IN_DELETE_SELF & events) &&
	     ( zero || inotifytools_get_stat_total( IN_DELETE_SELF ) ) )
		printf("delete_self  ");
	if ( ( IN_UNMOUNT & events) &&
	     ( zero || inotifytools_get_stat_total( IN_UNMOUNT ) ) )
		printf("unmount  ");

	printf("filename\n");

	// Array for sorting
	int w[num_watches];
	for ( int i = 0; i < num_watches; ++i ) {
		w[i] = i + 1;
	}
	qsort( w, num_watches, sizeof(int), compare );

	for ( int j = 0; j < num_watches; ++j ) {
		if ( !zero && !inotifytools_get_stat_by_wd( w[j], 0 ) )
			continue;
		printf("%-5d  ", inotifytools_get_stat_by_wd( w[j], 0 ) );
		if ( ( IN_ACCESS & events) &&
		     ( zero || inotifytools_get_stat_total( IN_ACCESS ) ) )
			printf("%-6d  ", inotifytools_get_stat_by_wd( w[j], IN_ACCESS ) );
		if ( ( IN_MODIFY & events) &&
		     ( zero || inotifytools_get_stat_total( IN_MODIFY ) ) )
			printf("%-6d  ", inotifytools_get_stat_by_wd( w[j], IN_MODIFY ) );
		if ( ( IN_ATTRIB & events) &&
		     ( zero || inotifytools_get_stat_total( IN_ATTRIB ) ) )
			printf("%-6d  ", inotifytools_get_stat_by_wd( w[j], IN_ATTRIB ) );
		if ( ( IN_CLOSE_WRITE & events) &&
		     ( zero || inotifytools_get_stat_total( IN_CLOSE_WRITE ) ) )
			printf("%-11d  ", inotifytools_get_stat_by_wd( w[j],
			                                               IN_CLOSE_WRITE ) );
		if ( ( IN_CLOSE_NOWRITE & events) &&
		     ( zero || inotifytools_get_stat_total( IN_CLOSE_NOWRITE ) ) )
			printf("%-13d  ", inotifytools_get_stat_by_wd( w[j],
			                                               IN_CLOSE_NOWRITE ));
		if ( ( IN_OPEN & events) &&
		     ( zero || inotifytools_get_stat_total( IN_OPEN ) ) )
			printf("%-4d  ", inotifytools_get_stat_by_wd( w[j], IN_OPEN ) );
		if ( ( IN_MOVED_FROM & events) &&
		     ( zero || inotifytools_get_stat_total( IN_MOVED_FROM ) ) )
			printf("%-10d  ", inotifytools_get_stat_by_wd( w[j], IN_MOVED_FROM ) );
		if ( ( IN_MOVED_TO & events) &&
		     ( zero || inotifytools_get_stat_total( IN_MOVED_TO ) ) )
			printf("%-8d  ", inotifytools_get_stat_by_wd( w[j], IN_MOVED_TO ) );
		if ( ( IN_MOVE_SELF & events) &&
		     ( zero || inotifytools_get_stat_total( IN_MOVE_SELF ) ) )
			printf("%-9d  ", inotifytools_get_stat_by_wd( w[j], IN_MOVE_SELF ) );
		if ( ( IN_CREATE & events) &&
		     ( zero || inotifytools_get_stat_total( IN_CREATE ) ) )
			printf("%-6d  ", inotifytools_get_stat_by_wd( w[j], IN_CREATE ) );
		if ( ( IN_DELETE & events) &&
		     ( zero || inotifytools_get_stat_total( IN_DELETE ) ) )
			printf("%-6d  ", inotifytools_get_stat_by_wd( w[j], IN_DELETE ) );
		if ( ( IN_DELETE_SELF & events) &&
		     ( zero || inotifytools_get_stat_total( IN_DELETE_SELF ) ) )
			printf("%-11d  ", inotifytools_get_stat_by_wd( w[j],
			                                               IN_DELETE_SELF ) );
		if ( ( IN_UNMOUNT & events) &&
		     ( zero || inotifytools_get_stat_total( IN_UNMOUNT ) ) )
			printf("%-7d  ", inotifytools_get_stat_by_wd( w[j], IN_UNMOUNT ) );

		printf("%s\n", inotifytools_filename_from_wd( w[j] ) );
	}

	return EXIT_OK;
}




bool parse_opts(
  int * argc,
  char *** argv,
  int * events,
  int * timeout,
  int * verbose,
  int * zero,
  int * sort,
  int * recursive
) {
	assert( argc ); assert( argv ); assert( events ); assert( timeout );
	assert( verbose ); assert( zero ); assert( sort ); assert( recursive );

	// Short options
	char * opt_string = "hra:d:zve:t:";

	// Construct array
	struct option long_opts[9];

	// --help
	long_opts[0].name = "help";
	long_opts[0].has_arg = 0;
	long_opts[0].flag = NULL;
	long_opts[0].val = (int)'h';
	// --event
	long_opts[1].name = "event";
	long_opts[1].has_arg = 1;
	long_opts[1].flag = NULL;
	long_opts[1].val = (int)'e';
	int new_event;
	// --timeout
	long_opts[2].name = "timeout";
	long_opts[2].has_arg = 1;
	long_opts[2].flag = NULL;
	long_opts[2].val = (int)'t';
	char * timeout_end = NULL;
	// --verbose
	long_opts[3].name = "verbose";
	long_opts[3].has_arg = 0;
	long_opts[3].flag = NULL;
	long_opts[3].val = (int)'v';
	// --nonzero
	long_opts[4].name = "zero";
	long_opts[4].has_arg = 0;
	long_opts[4].flag = NULL;
	long_opts[4].val = (int)'z';
	// --ascending
	long_opts[5].name = "ascending";
	long_opts[5].has_arg = 1;
	long_opts[5].flag = NULL;
	long_opts[5].val = (int)'a';
	bool sort_set = false;
	// --descending
	long_opts[6].name = "descending";
	long_opts[6].has_arg = 1;
	long_opts[6].flag = NULL;
	long_opts[6].val = (int)'d';
	// --recursive
	long_opts[7].name = "recursive";
	long_opts[7].has_arg = 0;
	long_opts[7].flag = NULL;
	long_opts[7].val = (int)'r';
	// Empty last element
	long_opts[8].name = 0;
	long_opts[8].has_arg = 0;
	long_opts[8].flag = 0;
	long_opts[8].val = 0;

	// Get first option
	char curr_opt = getopt_long(*argc, *argv, opt_string, long_opts, NULL);

	// While more options exist...
	while ( (curr_opt != '?') && (curr_opt != (char)-1) )
	{
		switch ( curr_opt )
		{
			// --help or -h
			case 'h':
				print_help();
				// Shouldn't process any further...
				return false;
				break;

			// --verbose or -v
			case 'v':
				++(*verbose);
				break;

			// --recursive or -r
			case 'r':
				++(*recursive);
				break;

			// --zero or -z
			case 'z':
				++(*zero);
				break;

			// --timeout or -t
			case 't':
				*timeout = strtoul(optarg, &timeout_end, 10);
				if ( *timeout_end != '\0' || *timeout < 0)
				{
					fprintf(stderr, "'%s' is not a valid timeout value.\n"
					        "Please specify an integer of value 0 or "
					        "greater.\n",
					        optarg);
					return false;
				}
				break;

			// --event or -e
			case 'e':
				// Get event mask from event string
				new_event = inotifytools_str_to_event(optarg);

				// If optarg was invalid, abort
				if ( new_event == -1 ) {
					fprintf(stderr, "'%s' is not a valid event!  Run with the "
					                "'--help' option to see a list of "
					                "events.\n", optarg);
					return false;
				}

				// Add the new event to the event mask
				(*events) = ( (*events) | new_event );

				break;

			// --ascending or -a
			case 'a':
				if ( sort_set ) {
					fprintf( stderr, "Please specify -a or -d once only!\n" );
					return false;
				}
				if ( 0 == strcasecmp( optarg, "total" ) ) {
					(*sort) = SORT_TOTAL_ASC;
				}
				else if ( 0 == strcasecmp( optarg, "move" ) ) {
					fprintf( stderr, "Cannot sort by `move' event; please use "
					         "`moved_from' or `moved_to'.\n" );
					return false;
				}
				else if ( 0 == strcasecmp( optarg, "close" ) ) {
					fprintf( stderr, "Cannot sort by `close' event; please use "
					         "`close_write' or `close_nowrite'.\n" );
					return false;
				}
				else {
					int event = inotifytools_str_to_event(optarg);

					// If optarg was invalid, abort
					if ( event == -1 ) {
						fprintf(stderr, "'%s' is not a valid key for "
						        "sorting!\n", optarg);
						return false;
					}

					(*sort) = event;
				}
				sort_set = true;
				break;


			// --descending or -d
			case 'd':
				if ( sort_set ) {
					fprintf( stderr, "Please specify -a or -d once only!\n" );
					return false;
				}
				if ( 0 == strcasecmp( optarg, "total" ) ) {
					(*sort) = SORT_TOTAL_DESC;
				}
				else {
					int event = inotifytools_str_to_event(optarg);

					// If optarg was invalid, abort
					if ( event == -1 ) {
						fprintf(stderr, "'%s' is not a valid key for "
						        "sorting!\n", optarg);
						return false;
					}

					(*sort) = -event;
				}
				break;

		}

		curr_opt = getopt_long(*argc, *argv, opt_string, long_opts, NULL);

	}

	(*argc) -= optind;
	*argv = &(*argv)[optind];

	// Make sure we were passed at least one file
	if ( (*argc) < 1 )
	{
		fprintf(stderr, "No files specified to watch!\n");
		return false;
	}

	if ( (*sort) != SORT_TOTAL_ASC && (*sort) != SORT_TOTAL_DESC &&
	     !(abs(*sort) & ((*events) ? (*events) : IN_ALL_EVENTS) )) {
		fprintf( stderr, "Can't sort by an event which isn't being watched "
		         "for!\n" );
		return false;
	}

	// If ? returned, invalid option
	return (curr_opt != '?');
}


void print_help()
{
	printf("inotifywatch %s\n", PACKAGE_VERSION);
	printf("Gather filesystem usage statistics using inotify.\n");
	printf("Usage: inotifywatch [ options ] file1 [ file2 ] [ ... ]\n");
	printf("Options:\n");
	printf("\t-h|--help    \tShow this help text.\n");
	printf("\t-v|--verbose \tBe verbose.\n");
	printf("\t-z|--zero\n"
	       "\t\tIn the final table of results, output rows and columns even\n"
	       "\t\tif they consist only of zeros (the default is to not output\n"
	       "\t\tthese rows and columns).\n");
	printf("\t-r|--recursive\tWatch directories recursively.\n");
	printf("\t-t|--timeout <seconds>\n"
	       "\t\tListen only for specified amount of time in seconds; if\n"
	       "\t\tomitted or 0, inotifywatch will execute until receiving an\n"
	       "\t\tinterrupt signal.\n");
	printf("\t-e|--event <event1> [ -e|--event <event2> ... ]\n"
	       "\t\tListen for specific event(s).  If omitted, all events are \n"
	       "\t\tlistened for.\n");
	printf("\t-a|--ascending <event>\n"
	       "\t\tSort ascending by a particular event, or `total'.\n");
	printf("\t-d|--descending <event>\n"
	       "\t\tSort descending by a particular event, or `total'.\n\n");
	printf("Exit status:\n");
	printf("\t%d  -  Exited normally.\n", EXIT_OK);
	printf("\t%d  -  Some error occurred.\n\n", EXIT_ERROR);
	printf("Events:\n");
	print_event_descriptions();
}

