/*
 *  PGGraphvizDocument.m
 *  graphviz
 *
 *  Created by Glen Low on Tue Dec 23 2003.
 *  Copyright (c) 2003, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "PGGraphvizDocument.h"
#import "PGGraphvizSchema.h"
#import "PGOption.h"
#import "PGAttribute.h"
#import "PGTasker.h"

@implementation PGGraphvizDocument

+ (void) initialize
	{
		[[NSUserDefaults standardUserDefaults] registerDefaults: [NSDictionary dictionaryWithObjectsAndKeys:
			@"dot", @"layout",
			@"", @"n",
			@"dot", @"T",
			nil]];
	}

- (id) init
	{
		if ((self = [super init]))
			{
				process_ = ProcessNone;
				render_ = RenderDone;
				image_ = nil;
				
				NSSet* all = [NSSet setWithObjects: @"dot", @"neato", @"twopi", @"circo", nil];
				NSSet* neatoOnly = [NSSet setWithObject: @"neato"];
				
				PGGraphvizSchema* mainSchema = [PGGraphvizSchema mainSchema];
				
				layout_ = [[PGOption alloc] initWithSchema: mainSchema prefix: @"layout"
					attribute: [PGAttribute attributeWithName: @"" usedBy: all type: @"layout"]];
				
				scale_ = [[PGOption alloc] initWithSchema: mainSchema prefix: @"s"
					attribute: [PGAttribute attributeWithName: @"" usedBy: all type: @"double"]];
				position_ = [[PGOption alloc] initWithSchema: mainSchema prefix: @"n"
					attribute: [PGAttribute attributeWithName: @"" usedBy: neatoOnly type: @"noLayout"]];
				
				verbose_ = [[PGOption alloc] initWithSchema: mainSchema prefix: @"v"
					attribute: [PGAttribute attributeWithName: @"" usedBy: all type: @""]];
				invertY_ = [[PGOption alloc] initWithSchema: mainSchema prefix: @"y"
					attribute: [PGAttribute attributeWithName: @"" usedBy: all type: @""]];
				reduceGraph_ = [[PGOption alloc] initWithSchema: mainSchema prefix: @"x"
					attribute: [PGAttribute attributeWithName: @"" usedBy: neatoOnly type: @""]];
				
				format_ = [[PGOption alloc] initWithSchema: mainSchema prefix: @"T"
					attribute: [PGAttribute attributeWithName: @"" usedBy: all type: @"format"]];
				
				graphDefaults_ = [[mainSchema optionsForComponent: @"graph"] retain];
				nodeDefaults_ = [[mainSchema optionsForComponent: @"node"] retain];
				edgeDefaults_ = [[mainSchema optionsForComponent: @"edge"] retain];
				
				options_ = [[NSMutableArray alloc] initWithObjects: verbose_, scale_, invertY_, position_, reduceGraph_, nil];
				[options_ addObjectsFromArray: graphDefaults_];
				[options_ addObjectsFromArray: nodeDefaults_];
				[options_ addObjectsFromArray: edgeDefaults_];
				
				taskers_ = [[NSMutableArray alloc] init];
				
				lastSaveToDirectory_ = nil;
				
				NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
				NSEnumerator* options = [options_ objectEnumerator];
				PGOption* nextOption;
				
				[defaultCenter
					addObserver: self
					selector: @selector (optionDidChange:)
					name: OptionDidChange
					object: layout_];
				while ((nextOption = [options nextObject]))
					[defaultCenter
						addObserver: self
						selector: @selector (optionDidChange:)
						name: OptionDidChange
						object: nextOption];
			}
		return self;
	}
	
- (void) startRender
	{
		NSMutableArray* arguments = [self arguments];
		[arguments addObject: @"-Tepdf"];
		
		PGTasker* tasker = [[PGTasker alloc]
			initWithLaunchPaths: [NSArray arrayWithObject: [[NSBundle mainBundle] pathForAuxiliaryExecutable: [[self layout] stringValue]]]
			arguments: [NSArray arrayWithObject: arguments]
			currentDirectoryPath: [self currentDirectoryPath]];
						
		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
		
		// notify me when output is ready
		[defaultCenter
			addObserver: self
			selector: @selector (rendererDidOutput:)
			name: TaskerDidOutput
			object: tasker];
		[tasker retain];	// balanced with release after removeObserver
		
		// notify me when status is ready
		[defaultCenter
			addObserver: self
			selector: @selector (rendererDidStatus:)
			name: TaskerDidStatus
			object: tasker];
		[tasker retain];	// balanced with release after removeObserver
		
		[tasker release];
		
		process_ = ProcessRendering;
		[[NSNotificationCenter defaultCenter]
			postNotificationName: DotDocumentProcessDidChange
			object: self];
		
		render_ = RenderDoing;
		[tasker startWithData: [self data]];
		
		[taskers_ addObject: tasker];
	}

- (void) startExport: (NSString*) fileName
	{
		NSMutableArray* arguments = [self arguments];
		[arguments addObject: [[self format] string]];
		[arguments addObject: [NSString stringWithFormat: @"-o%@", fileName]];
		
		PGTasker* tasker = [[PGTasker alloc]
			initWithLaunchPaths: [NSArray arrayWithObject: [[NSBundle mainBundle] pathForAuxiliaryExecutable: [[self layout] stringValue]]]
			arguments: [NSArray arrayWithObject: arguments]
			currentDirectoryPath: [self currentDirectoryPath]];
			
		// notify me when status is ready
		[[NSNotificationCenter defaultCenter]
			addObserver: self
			selector: @selector (exporterDidStatus:)
			name: TaskerDidStatus
			object: tasker];
		[tasker retain];	// balanced with release after removeObserver
		
		[tasker release];
		
		process_ = ProcessExporting;
		[[NSNotificationCenter defaultCenter]
			postNotificationName: DotDocumentProcessDidChange
			object: self];
		
		[tasker startWithData: [self data]];
		[taskers_ addObject: tasker];
	}

- (NSString*) windowNibName
	{
		// Override returning the nib file name of the document
		// If you need to use a subclass of NSWindowController or if your document supports multiple NSWindowControllers, you should remove this method and override -makeWindowControllers instead.
		return @"GraphvizDocument";
	}

- (BOOL) prepareSavePanel: (NSSavePanel*) savePanel
	{
		if (lastSaveToDirectory_)
			[savePanel setDirectory: lastSaveToDirectory_];
		[[NSNotificationCenter defaultCenter]
			postNotificationName: DotDocumentWillPrepareSavePanel object: self
			userInfo: [NSDictionary dictionaryWithObject: savePanel forKey: DotDocumentSavePanel]];
		return YES;
	}
	
- (void) printShowingPrintPanel: (BOOL) showPanels
	{
		NSImageView* printView = [[NSImageView alloc] init];
		[printView setFrameSize: [image_ size]];
		[printView setImage: image_];
		
		NSPrintInfo* printInfo = [self printInfo];
		[printInfo setVerticalPagination: NSAutoPagination];
		[printInfo setHorizontalPagination: NSAutoPagination];	
		NSPrintOperation* printOperation = [NSPrintOperation printOperationWithView: printView
			printInfo: printInfo];
		[printView release];
		[printOperation setShowPanels: showPanels];
	
		[self runModalPrintOperation: printOperation
			delegate: nil
			didRunSelector: nil
			contextInfo: nil];	
	}
	
- (void) setPrintInfo: (NSPrintInfo*) printInfo
	{
		[printInfo setVerticalPagination: NSAutoPagination];
		[printInfo setHorizontalPagination: NSAutoPagination];	
		[NSPrintInfo setSharedPrintInfo: printInfo];
	}
	
- (NSPrintInfo*) printInfo
	{
		return [NSPrintInfo sharedPrintInfo];
	}
	
- (BOOL) writeWithBackupToFile: (NSString*) fullDocumentPath ofType: (NSString*) docType saveOperation: (NSSaveOperationType) saveOperationType
	{
		if (saveOperationType == NSSaveToOperation)
			{
				[lastSaveToDirectory_ release];
				lastSaveToDirectory_ = [[fullDocumentPath stringByDeletingLastPathComponent] retain];
				[self startExport: fullDocumentPath];
				return YES;
			}
		else
			return NO;
	}
	
- (BOOL) validateMenuItem: (id <NSMenuItem>) menuItem
	{
		if ([menuItem action] == @selector (revertDocumentToSaved:))
			return NO;
		else
			return YES;
	}
	
- (Process) process
	{
		return process_;
	}
	
- (NSImage*) image
	{
		return image_;
	}
	
- (PGOption*) layout
	{
		return layout_;
	}

- (PGOption*) verbose
	{
		return verbose_;
	}
	
- (PGOption*) scale
	{
		return scale_;
	}
	
- (PGOption*) invertY
	{
		return invertY_;
	}
	
- (PGOption*) position
	{
		return position_;
	}
	
- (PGOption*) reduceGraph
	{
		return reduceGraph_;
	}

- (PGOption*) format
	{
		return format_;
	}

- (NSArray*) graphDefaults
	{
		return graphDefaults_;
	}
	
- (NSArray*) nodeDefaults
	{
		return nodeDefaults_;
	}
	
- (NSArray*) edgeDefaults
	{
		return edgeDefaults_;
	}
	
- (NSData*) data
	{
		return nil;
	}

- (NSMutableArray*) arguments
	{
		NSMutableArray* argumentArray = [NSMutableArray arrayWithCapacity: [options_ count]];

		NSEnumerator* options = [options_ objectEnumerator];
		PGOption* nextOption;
		while ((nextOption = [options nextObject]))
			if ([[nextOption usedBy] containsObject: [layout_ stringValue]])
				{
					NSString* nextArgument = [nextOption string];
					if (nextArgument)
						[argumentArray addObject: nextArgument];
				}
				
		return argumentArray;	
	}

- (NSString*) currentDirectoryPath
	{
		return nil;
	}
	
- (IBAction) stopDocument: (id) sender
	{
		NSEnumerator* eachTasker = [taskers_ objectEnumerator];
		PGTasker* nextTasker;
		while ((nextTasker = [eachTasker nextObject]))
			[nextTasker stop];
	}
	
- (IBAction) renderDocument: (id) sender
	{
		[self queueRender];
	}

- (IBAction) rememberSettings: (id) sender
	{
		[layout_ remember];
		[format_ remember];
		
		NSEnumerator* options = [options_ objectEnumerator];
		PGOption* nextOption;
		while ((nextOption = [options nextObject]))
			[nextOption remember];
	}

- (void) optionDidChange: (NSNotification*) notification
	{
		if ([[[notification object] usedBy] containsObject: [layout_ stringValue]])
			[self queueRender];
	}

- (void) queueRender
	{
		if (render_ == RenderDone)
			[self startRender];
		else
			render_ = RenderDeferred;
	}

- (void) rendererDidOutput: (NSNotification*) notification
	{
		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
		
		NSImage* newImage = [[NSImage alloc] initWithData: [[notification userInfo] objectForKey: TaskerDataItem]];
		if (newImage)
			{
				[newImage setCacheMode: NSImageCacheNever];
								
				[image_ release];
				image_ = newImage;
				
				[defaultCenter
					postNotificationName: DotDocumentImageDidChange
					object: self];
			}
		
		
		// don't notify me any more for output
		PGTasker* tasker = [notification object];
		[defaultCenter
			removeObserver: self
			name: TaskerDidOutput
			object: tasker];
		[tasker release];	// balanced with retain after addObserver

		if (render_ == RenderDeferred)
			[self startRender];
		else
			render_ = RenderDone;
	}
	
- (void) rendererDidStatus: (NSNotification*) notification
	{
		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
		PGTasker* tasker = [notification object];
		
		process_ = [tasker terminationStatus] == 0 ? ProcessRenderedOK : ProcessRenderedError;
		[defaultCenter
			postNotificationName: DotDocumentProcessDidChange
			object: self];
		
		// don't notify me any more for status
		[defaultCenter
			removeObserver: self
			name: TaskerDidStatus
			object: tasker];
		[tasker release];	// balanced with retain after addObserver

		[taskers_ removeObject: tasker];
	}

- (void) exporterDidStatus: (NSNotification*) notification
	{
		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
		PGTasker* tasker = [notification object];
		
		process_ = [tasker terminationStatus] == 0 ? ProcessExportedOK : ProcessExportedError;
		[defaultCenter
			postNotificationName: DotDocumentProcessDidChange
			object: self];
		
		// don't notify me any more for status
		[defaultCenter
			removeObserver: self
			name: TaskerDidStatus
			object: tasker];
		[tasker release];	// balanced with retain after addObserver
		
		[taskers_ removeObject: tasker];
	}

- (void) dealloc
	{
		[[NSNotificationCenter defaultCenter] removeObserver: self];

		[image_ release];
		[layout_ release];
		[verbose_ release];
		[scale_ release];
		[invertY_ release];
		[position_ release];
		[reduceGraph_ release];
		[format_ release];
		
		[graphDefaults_ release];
		[nodeDefaults_ release];
		[edgeDefaults_ release];
		[options_ release];
		
		[taskers_ release];
		
		[lastSaveToDirectory_ release];

		[super dealloc];
	}

@end

NSString* DotDocumentProcessDidChange = @"DotDocumentProcessDidChange";
NSString* DotDocumentImageDidChange = @"DotDocumentImageDidChange";

NSString* DotDocumentWillPrepareSavePanel = @"DotDocumentWillPrepareSavePanel";

NSString* DotDocumentSavePanel = @"DotDocumentSavePanel";

