/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*-----------------------------------------------------------------------------

   hdf2piv - converts hdf5 PIV data ASCII data

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

-----------------------------------------------------------------------------*/

#include <stdio.h> 
#include <stdlib.h> 
#include <assert.h> 
#include <gpiv.h>

/* #define PARFILE "scale.par"    */  /* Parameter file name */
#define PARFILE "gpivrc"     /* Parameter file name */
#define USAGE "\
Usage: hdf2piv [-e] [-h | --help] [-p | --print] [-v | --version] filename \n\
\n\
keys: \n\
-e:                    exclude data \n\
-h | --help:           this on-line help \n\
-p | --print:          print parameters to stdout \n\
-v | --version:        version number \n\
"

#define HELP  "\
hdf2piv - converts hdf5 PIV-data to ASCII data"

#define RCSID "$Id: hdf2piv.c,v 1.9 2006/01/31 14:18:04 gerber Exp $"

gboolean print_par = FALSE, exclude_data = FALSE;

void 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS]
             )
/* ----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c = '\0';
    int argc_next;

    while (--argc > 0 && (*++argv)[0] == '-') {

/*
 * argc_next is set to 1 if the next cmd line argument has to be searched for; 
 * in case that the command line argument concerns more than one char or cmd 
 * line argument needs a parameter 
 */

        argc_next = 0;
	while (argc_next == 0 && (c = *++argv[0]))
            
            switch (c) {
            case 'v':
                printf("%s\n", RCSID); 
                exit(0);
                break;
            case 'e':
                exclude_data = TRUE;
                break;
            case 'h':
                printf("%s\n", RCSID); 
                printf("%s\n",HELP);
                printf("%s\n",USAGE);
                exit(0);
                break;
            case 'p':
                print_par = TRUE;
                break;

/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

            default:
                gpiv_error(USAGE);
                break;
            }
    }

    if(argc != 1) { 
        gpiv_error("%s: %s", RCSID, USAGE);
    }

    strcpy(fname, argv[0]);
}



void 
make_fname(char *fname, 
           char *fname_in, 
           char *fname_out_img,
           char *fname_out_header,
           char *fname_out_piv,
           char *fname_out_vor, 
           char *fname_out_nstrain, 
           char *fname_out_sstrain,
           char *fname_out_par
           )
/* ----------------------------------------------------------------------------
 * define filenames
 */
{     
     gpiv_io_make_fname(fname, GPIV_EXT_GPIV, fname_in);
     if (print_par) printf("# Input file: %s\n",fname_in);

     gpiv_io_make_fname(fname, GPIV_EXT_RAW_IMAGE, fname_out_img);
     if (print_par) printf("# Output data file: %s\n",fname_out_img);
     
     gpiv_io_make_fname(fname, GPIV_EXT_HEADER, fname_out_header);
     if (print_par) printf("# Output header file: %s\n",fname_out_header);
         
     if (exclude_data == FALSE) {
         gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_out_piv);
         if (print_par) printf("# Output data file: %s\n",fname_out_piv);
         
         gpiv_io_make_fname(fname, GPIV_EXT_VOR, fname_out_vor);
         if (print_par) printf("# Input data file: %s\n",fname_out_vor);
         
         gpiv_io_make_fname(fname, GPIV_EXT_NSTR, fname_out_nstrain);
         if (print_par) printf("# Input data file: %s\n",
                                    fname_out_nstrain);
         
         gpiv_io_make_fname(fname, GPIV_EXT_SSTR, fname_out_sstrain);
         if (print_par) printf("# Input data file: %s\n",
                                    fname_out_sstrain);

         gpiv_io_make_fname(fname, GPIV_EXT_PAR, fname_out_par);
         if (print_par) printf("# Input parameter file: %s\n",
                                    fname_out_par);
     }
}



int 
main(int argc, 
     char *argv[]
     )
/* ----------------------------------------------------------------------------
 */
{
    FILE *fp_out = NULL;
    char *err_msg = NULL;
    char fname[GPIV_MAX_CHARS], 
        fname_out_img[GPIV_MAX_CHARS], 
        fname_out_header[GPIV_MAX_CHARS], 
        fname_out_piv[GPIV_MAX_CHARS], 
        fname_out_vor[GPIV_MAX_CHARS],
        fname_out_nstrain[GPIV_MAX_CHARS],
        fname_out_sstrain[GPIV_MAX_CHARS],
        fname_out_par[GPIV_MAX_CHARS],
        fname_in[GPIV_MAX_CHARS];
    char  RCSID_DATA[GPIV_MAX_CHARS], 
        c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    int nc_lines = 0, var_scale = 0;
    guint16 **img1 = NULL, **img2 = NULL;
    GpivPivData piv_data;
    GpivScalarData scdata;

    GpivImagePar image_par;
    GpivEvalPar piv_eval_par;
    GpivValidPar piv_valid_par;
    GpivPostPar piv_post_par;

    piv_data.nx = 0;
    piv_data.ny = 0;
    piv_data.point_x = NULL;
    piv_data.point_y = NULL;
    piv_data.dx = NULL;
    piv_data.dy = NULL;
    piv_data.snr = NULL;
    piv_data.peak_no = NULL;

    scdata.nx = 0;
    scdata.ny = 0;
    scdata.point_x = NULL;
    scdata.point_y = NULL;
    scdata.scalar = NULL;
    scdata.flag = NULL;

/*
 * Initializing parameters
 */
    gpiv_img_parameters_logic(&image_par, FALSE);
    gpiv_piv_parameters_logic(&piv_eval_par, FALSE);
    gpiv_valid_parameters_logic(&piv_valid_par, FALSE);
    gpiv_post_parameters_logic(&piv_post_par, FALSE);


    command_args(argc, argv, fname);
    make_fname(fname, fname_in, fname_out_img, fname_out_header, fname_out_piv,
               fname_out_vor, fname_out_nstrain, fname_out_sstrain, 
               fname_out_par);
	  
/* 
 * reading input from hdf5
 *
 * image data
 */
    if ((err_msg =
         gpiv_img_fread_hdf5_parameters(fname_in, &image_par))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

    img1 = gpiv_alloc_img(image_par);
    if (image_par.x_corr) 
        img2 = gpiv_alloc_img(image_par);
    if ((err_msg =
         gpiv_fread_hdf5_image (fname_in, img1, img2, &image_par))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

    gpiv_fwrite_image (fname_out_img, img1, img2, image_par);
    gpiv_free_img(img1, image_par);
    if (image_par.x_corr) 
        gpiv_free_img(img2, image_par);

    if ((fp_out = fopen(fname_out_header, "w")) == NULL) {
        gpiv_error("%s error: failure opening %s for output",
                   RCSID, fname_out_header);
    }

    gpiv_img_fprint_header(fp_out, image_par);
    fclose(fp_out);
/*
 * piv data
 */ 
     if (exclude_data == FALSE) {
         if ((err_msg =
              gpiv_fcount_hdf5_data(fname_in, &piv_data.ny, &piv_data.nx))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         if (print_par) gpiv_warning("piv_data: nx=%d ny=%d", 
                                     piv_data.nx, piv_data.ny);



         gpiv_alloc_pivdata(&piv_data);
         if ((err_msg =
              gpiv_fread_hdf5_piv_position(fname_in, &piv_data, RCSID_DATA, 
                                           &var_scale))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         if ((err_msg =
              gpiv_fread_hdf5_pivdata (fname_in, &piv_data, "PIV", RCSID_DATA, 
                                       &var_scale))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         gpiv_fwrite_pivdata(fname_out_piv, &piv_data, c_line, nc_lines, var_scale, 
                             RCSID_DATA);
         gpiv_free_pivdata(&piv_data);

/*
 * scalar data: vorticity
 */
         if ((err_msg =
              gpiv_fcount_hdf5_data(fname_in, &scdata.ny, &scdata.nx))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         if (print_par) gpiv_warning("scalar_data: nx=%d ny=%d", 
                                     scdata.nx, scdata.ny);
         

         gpiv_alloc_scdata(&scdata);
         if ((err_msg =
              gpiv_fread_hdf5_sc_position(fname_in, &scdata))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

          if ((err_msg =
               gpiv_fread_hdf5_scdata (fname_in, &scdata, "GPIV_VORTICITY", 
                                       RCSID_DATA))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         gpiv_fwrite_scdata(fname_out_vor, &scdata, c_line, nc_lines, var_scale, 
                            RCSID_DATA);
         gpiv_free_scdata(&scdata);
         
/*
 * scalar data: normal strain
 */
         if ((err_msg =
              gpiv_fcount_hdf5_data(fname_in, &scdata.ny, &scdata.nx))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         if (print_par) gpiv_warning("scalar_data: nx=%d ny=%d", 
                                     scdata.nx, scdata.ny);
         

         gpiv_alloc_scdata(&scdata);
         if ((err_msg =
              gpiv_fread_hdf5_sc_position(fname_in, &scdata))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         if ((err_msg =
              gpiv_fread_hdf5_scdata (fname_in, &scdata, "GPIV_VORTICITY", 
                                      RCSID_DATA))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         gpiv_fwrite_scdata(fname_out_sstrain, &scdata, c_line, nc_lines, var_scale, 
                            RCSID_DATA);
         gpiv_free_scdata(&scdata);
         
/*
 * scalar data: shear strain
 */
         if ((err_msg =
              gpiv_fcount_hdf5_data(fname_in, &scdata.ny, &scdata.nx))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         if (print_par) gpiv_warning("scalar_data: nx=%d ny=%d", 
                                     scdata.nx, scdata.ny);
         

         gpiv_alloc_scdata(&scdata);
         if ((err_msg =
              gpiv_fread_hdf5_sc_position(fname_in, &scdata))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         if ((err_msg =
              gpiv_fread_hdf5_scdata (fname_in, &scdata, "GPIV_VORTICITY", 
                                      RCSID_DATA))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         gpiv_fwrite_scdata(fname_out_nstrain, &scdata, c_line, nc_lines, var_scale, 
                            RCSID_DATA);
         gpiv_free_scdata(&scdata);

/* 
 * parameters
 */
         if ((err_msg =
              gpiv_img_fread_hdf5_parameters(fname_in, &image_par))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         if ((err_msg =
              gpiv_piv_fread_hdf5_parameters(fname_in, &piv_eval_par))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
 
         if ((err_msg =
              gpiv_valid_fread_hdf5_parameters(fname_in, &piv_valid_par))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

         if ((err_msg =
              gpiv_post_fread_hdf5_parameters(fname_in, &piv_post_par))
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);


         if ((fp_out = fopen(fname_out_par, "w")) == NULL) {
             gpiv_error("%s error: failure opening %s for output",
                        RCSID, fname_out_par);
         }
         gpiv_img_fprint_parameters(fp_out, image_par);
         gpiv_piv_fprint_parameters(fp_out, piv_eval_par);
         gpiv_valid_fprint_parameters(fp_out, piv_valid_par);
         gpiv_post_fprint_parameters(fp_out, piv_post_par);
         fclose(fp_out);
 
         if (print_par) {
             gpiv_img_print_parameters(image_par);
             gpiv_piv_print_parameters(piv_eval_par);
             gpiv_valid_print_parameters(piv_valid_par);
             gpiv_post_print_parameters(piv_post_par);
         }
     }


     exit(0);
}






