/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*************************************************************
*
*      file:      torus.c
*
*      Purpose:  Functions dealing with symmetries of surface.
*                 Has the three functions for torus.
*                 If user wants to define his/her own symmetry
*                 group, write a file like this and put function
*                 names in registry.c.
*
*/

#include "include.h"

/*
         Group elements are encoded in two-bit fields in "wrap",
         one for each dimension, with "001" for positive wrap
         and "011" for negative wrap.  You are free to encode
         any way you want, but use wrap = 0 for the identity.
*/

/* specific case of torus symmetry representation */
/* these also defined in extern.h since torus built-in */
#ifndef TWRAPBITS
#define TWRAPBITS 6
#define POSWRAP    1
#define WRAPMASK  037
#define ALLWRAPMASK 03737373737
#define NEGWRAP    WRAPMASK
#endif
#define WRAPNUM(x) ( (x)>(1<<(TWRAPBITS-2)) ? (x)-(1<<(TWRAPBITS-1)) : (x))

/*******************************************************************
*
*  function: torus_wrap
*
*  purpose:  Provide adjusted coordinates for vertices that get
*            wrapped around torus.  Serves as example for user-written
*            symmetry function.
*
*            This function uses the values of the torus periods read
*            in from the data file. Yours doesn't have the privilege
*            of defining its own datafile syntax, but you can access
*            values of parameters defined in the datafile as
*            web.params[n].value.
*/

void torus_wrap(x,y,wrap)
REAL *x;    /* original coordinates */
REAL *y;    /* wrapped coordinates  */
WRAPTYPE wrap;  /* encoded symmetry group element, TWRAPBITS bits per dimension */
{
  int i,j;
  int wrapnum;

  if ( x != y )
     memcpy((char*)y,(char*)x,SDIM*sizeof(REAL));
  for ( i = 0 ; wrap != 0 ; i++, wrap >>= TWRAPBITS )
    { wrapnum =  WRAPNUM(wrap & WRAPMASK);
      if ( wrapnum )
         for ( j = 0 ; j < SDIM ; j++ )
            y[j] += wrapnum*web.torus_period[i][j];
    }
}


/*******************************************************************
*
*  function: torus_form_pullback
*
*  purpose:  Pull back differentail forms at vertices that get
*                wrapped around torus.  Serves as example for user-written
*                symmetry function.
*
*                Torus is flat, so implementation is trivial copy.
*/

void torus_form_pullback(x,xform,yform,wrap)
REAL *x;    /* original coordinates */
REAL *xform; /* pulled back form */
REAL *yform;    /* wrapped form, input  */
WRAPTYPE wrap;  /* encoded symmetry group element, 3 bits per dimension */
{
  memcpy((char*)xform,(char*)yform,SDIM*sizeof(REAL));
}

/********************************************************************
*
*  function: torus_compose()
*
*  purpose:  do composition of two group elements
*
*/

WRAPTYPE torus_compose(wrap1,wrap2)
WRAPTYPE wrap1,wrap2;  /* the elements to compose */
{ int w = (wrap1 + wrap2) & ALLWRAPMASK;
  return w;
}


/********************************************************************
*
*  function: torus_inverse()
*
*  purpose:  return inverse of group element.
*
*/

WRAPTYPE torus_inverse(wrap)
WRAPTYPE wrap;  /* the element invert */
{
  return ((~ALLWRAPMASK)-wrap) & ALLWRAPMASK;
}
  

  
