/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/******************************************************************
*
*  File:  extern.h
*
*  Purpose:  declare global variables for Evolver program.
*/

#define DEFAULT_EDGE_COLOR BLACK
#define DEFAULT_FACET_COLOR WHITE
#define DEFAULT_TARGET_TOLERANCE (1e-4)
/* extern char *sys_errlist[];  */  /* Uncomment this if compile error */
extern int broken_pipe_flag; /* so output routines will know to quit */
extern int match_id_flag; /* to make id match datafile number, option -i */
extern char *cmdfilename; /* for saving command line read file */
extern char *current_prompt; /* prompt string being displayed */
extern int echo_flag; /* whether to echo stdin; for piped input */
extern int auto_convert_flag; /* whether to automatically convert_to_quantities */
#define DEFAULT_BRIGHTNESS 0.65
extern REAL brightness; /* midlevel gray for screen display */
extern int markedgedrawflag; /* for single-drawing edges */
extern int verbose_flag;  /* for lots of messages */
extern char loadfilename[200]; /* for LOAD command */
extern jmp_buf loadjumpbuf;  /* for LOAD command */
extern jmp_buf graphjumpbuf;  /* for errors during MS graphing */
extern unsigned long draw_thread_id; /* for graphics thread */

#define MAXOPTPARAM 100
extern int optparamcount;  /* number thereof */
struct optparam {
 int  pnum; /* which parameter this is */
 REAL grad; /* energy gradients */
 REAL velocity;  /* adjusted motion */
 REAL cg;  /* conjugate gradient history */
 REAL old_value;  /* for restoring */
 REAL oldgrad;  /* for conjugate gradient */
 int  rownum;  /* row in Hessian */
 };
extern struct optparam optparam[MAXOPTPARAM];
extern REAL **optparam_congrads;  /* constraint gradients */
#define OPTPARAM_DELTA (1.0e-4)

/* for calc_volgrads() mode */
#define NO_OPTS  0
#define DO_OPTS  1
extern int volgrads_every_flag; /* whether recalc volgrads every projection iteration */

extern int zener_drag_flag; /* whether to do zener drag */
#define ZENER_COEFF_NAME "zener_coeff"
extern int backcull_flag;  /* 3D graphics backculling */
extern int setting_backcull; 


extern REAL **vgev;  /* vector vol grads, for approx curvature */
extern REAL **vgef;  /* form vol grads, for approx curvature */

/* macro stuff */
extern int keep_macros_flag; /* to preserve macros after datafile. */
#define MACRONAMESIZE 30
struct macro { char name[MACRONAMESIZE];
               int  offset;     /* start of substitute string */
               int  subsize;    /* size of substitute string */
             };
extern struct macro  *macros;  /* dynamically allocated */
extern int macro_count;  /* number of macros defined */
extern char  *macro_subs;  /* string space for substitution strings */

extern int facet_general_flag;
extern double factorial[20];
extern int everything_quantities_flag;  /* for pure quantity version, after conversion */
extern int show_all_quantities;  /* to display default quantities also */
extern int random_seed;  /* seed for random number generators */
extern int option_q;  /* record command line option */

extern char *VERSION;
extern char *evolver_version;  /* for version checking */
extern char needed_version[30];

extern char *typenames[NUMELEMENTS];

/* silent linesplicing character */
#define MOREIN 1

/* element attr bits */
#define ALL_ATTR     (ATTR)0xFFFFFFFF
#define ALLOCATED    (ATTR)0x0001
#define NODISPLAY    (ATTR)0x0002
#define NEWELEMENT   (ATTR)0x0004
#define NEWVERTEX    (ATTR)0x0004
#define NEWEDGE      (ATTR)0x0004
#define NEWFACET     (ATTR)0x0004
#define NEWBODY      (ATTR)0x0004
#define WRAPPED      (ATTR)0x0008
#define PINNED_V     (ATTR)0x0008
#define DENSITY      (ATTR)0x0010
#define FIXEDVOL     (ATTR)0x0020
#define FIXED        (ATTR)0x0040
#define BOUNDARY     (ATTR)0x0080
#define NEGBOUNDARY  (ATTR)0x0100
#define BDRY_ENERGY  (ATTR)0x0200
#define BDRY_CONTENT (ATTR)0x1000
#define CONSTRAINT   (ATTR)0x0400
#define SURF_ENERGY  (ATTR)0x8000
#define SURF_QUANTITY (ATTR)0x1000
#define PRESSURE      (ATTR)0x0800
#define NEGCONSTRAINT (ATTR)0x4000
#define HIT_WALL      (ATTR)0x2000
#define BARE_NAKED    (ATTR)0x10000
#define Q_MIDPOINT    (ATTR)0x20000
#define TETRA_PT      (ATTR)0x40000
#define TRIPLE_PT     (ATTR)0x80000
#define HIT_ONE_SIDED (ATTR)0x100000
#define Q_MIDFACET    (ATTR)0x200000
#define Q_MIDEDGE     (ATTR)0x400000
#define AXIAL_POINT   (ATTR)0x800000
#define NO_REFINE     (ATTR)0x1000000
#define DISSOLVED     (ATTR)0x2000000
#define EDGE_DRAWN    (ATTR)0x4000000
#define ACTUALVOL     (ATTR)0x8000000
#define NO_ORIGINAL   (-1)

/* modes for move_vertices() */
#define TEST_MOVE    0

#define ACTUAL_MOVE 1
extern char *dymem; /* dynamic memory region */
extern int  dymemsize; /* size of dynamic memory region */
extern struct constraint null_constraint; /* non-constraint placeholder */
#define GETCONSTR(n) (web.constraint_addr[n]?  \
      (struct constraint*)(dymem+web.constraint_addr[n]):&null_constraint)

extern char *areaname;  /* length or area */
extern int read_command_flag;  /* whether commands at end of datafile */
extern int verb_flag;    /* set if lex looking for a verb */
extern int cond_expr_flag; /* set if parser looking for ':' in conditional expr */
extern int read_wrap_flag; /* set when reading wraps in datafile */
extern int exit_after_error; /* auto exit flag */ 
extern int exit_after_warning; /* auto exit flag */ 
extern int last_error;  /* number of last error */
extern int change_flag;  /* set during command that changes surface */
extern int assigntype;  /* type of assignment operator */
extern int uminus_flag; /* whether to interpret " -" as UMINUS */
extern char *cmdptr;    /* current command or input for parsing */
extern int logfile_flag; /* whether logging in progress */
extern char logfilename[200];
extern FILE *logfilefd;

/* structure used in check.c */
struct vvvv { element_id id;  /* which element    */
              int vord[3];    /* id's of vertices */
            };
extern int gocount;          /* number of iterations left */ 
extern int go_display_flag;     /* display each change */
extern int box_flag;  /* whether or not to show outline box */
extern int quiet_flag; /* whether to display normal output */
extern int quiet_go_flag; /* whether to display normal g output */
extern int shading_flag;  /* for facet shading by orientation */
extern int color_flag;    /* facet coloring by user */
extern int background_color;  /* graphics background */
extern int geomview_bug_flag; /* for geomview 1.6.1 picking bug */
extern int gv_binary_flag;  /* whether to do geomview in binary */
extern int gv_pipe[2]; /* for pipe for reading geomview pick commands */
extern int pickvnum,pickenum,pickfnum,pickbnum; /* geomview picks */
extern int new_vertex_id,new_edge_id,new_facet_id,new_body_id; /* just created elements */
extern int gv_vect_start; /* vector vertex start in vpicklist */
/* for piping geomview output */
#define GEOM_TO_GEOMVIEW 0
#define GEOM_NAMED_PIPE 1
#define GEOM_PIPE_COMMAND 2
extern int parallel_update_flag[NUMELEMENTS]; /* set when element info changed */
extern long global_timestamp;  /* universal clock */
extern long web_timestamp; /* something changes */
extern long graph_timestamp;  /* so graph routines know when surface changed */
extern long top_timestamp;    /* timestamp for topology changes */
extern long vedge_timestamp; /* for vertex edgelist currency */
extern long vfacet_timestamp; /* for vertex facetlist currency */
extern long bfacet_timestamp; /* for body facetlist currency */
extern long volume_timestamp; /* for volume calculation */
extern long reset_timestamp;  /* when surface loaded */
extern int ackerman_flag;  /* whether doing phase space motion */
extern int labelflag; /* whether ps doing labels */
extern int gridflag;  /* whether ps doing gridlines */
extern int ps_colorflag; /* whether ps doing color */
extern int crossingflag; /* whether ps doing crossings */
extern char ps_file_name[1000]; /* ps output file */
#define NOLABELS 0
#define LABEL_ID 1
#define LABEL_ORIG 2
extern vertex_id *vpicklist; /* for geomview picking */
extern facet_id *fpicklist;
extern REAL **to_focus;  /* used only by oglgraph.c at present */
extern REAL **from_focus;


#define PATHSIZE  60
#define NESTDEPTH 10
extern element_id junk; /* for MSC bug */
#ifndef PARALLEL_MACHINE
extern element_id xx_id;  /* for macros to prevent multiple evaluation */
extern element_id x1_id,x2_id,x3_id,x4_id,x5_id,x6_id,x7_id,x8_id,x9_id;
extern element_id xa_id,xb_id,xc_id,xd_id,xe_id,xf_id,xg_id,xh_id;
/*so nested macros don't tromp each other*/
#endif

extern struct treenode *permlist;

/* evaluation stack */
#define MAXSTACK 100
struct dstack { REAL value, deriv[2*MAXCOORD];
                REAL second[2*MAXCOORD][2*MAXCOORD]; 
              };

/* for redefining single letter commands */
extern struct expnode single_redefine[128];

/* for dynamic load library functions */
#define MAX_DLL 5
typedef void (*dll_func_type) ARGS((int , REAL*, struct dstack *));
struct dll { char *name;  /* library name */
             void *handle;     /* for dl functions */
           } ;
extern  struct dll dll_list[MAX_DLL];
#define FUNC_VALUE  1
#define FUNC_DERIV  2
#define FUNC_SECOND 3
                 
/* global variable list */
#define GLOBAL_NAME_SIZE 31
struct global
  { char name[GLOBAL_NAME_SIZE + 1];  /* 31 significant characters */
    union { REAL real;
            char *string;
            struct expnode proc;
            struct { REAL *values;
                     char *value_file;
                   } file;
            int quant;  /* quantity */
            int meth_inst; /* method values*/
            dll_func_type funcptr; /* dynamic load function */
          } value;
    REAL delta; /* for optimizing parameter differencing */
    REAL scale; /* for optimizing parameter scale matching */
    REAL *gradhess; /* for expressions in constraints etc. */
    int  flags;     /* see defines below */
    int  proc_timestamp; /* for ordering procedure defines */
  };
#define dy_globals web.dy_globals_w
#define globals ((struct global *)(dymem + dy_globals))
extern struct global *Globals; /* handy for debugging */
extern int proc_timestamp; /* for ordering procedure definitions */
extern int old_global_count; /* for error recovery */
extern int perm_flag;  /* for whether permanent assignment parsing in place */
extern int reading_comp_quant_flag;  
extern int cur_quant;  /* when reading compound quantity */
extern int compound_hess_flag; /* mode while doing compound hessian */
#define CH_GRADS 1
#define CH_HESS  2
extern int quantities_only_flag; /* for using named quantities only */
extern int gravity_quantity_num;  /* number of quantity for default gravity */
extern int gap_quantity_num;  /* number of quantity for default gap energy */
extern int default_area_quant_num; /* number of quantity for default area */
extern int rotorder_var; /* for flip_rot order */
extern int calc_quant_flag; /* set when quantity calculation underway */

/* defines for global variable flags */
#define ORDINARY_PARAM 1  /* ordinary real-valued */
#define FILE_VALUES 2    /* values to be read from file */
#define SUBROUTINE  4    /* value is a parse tree */
#define SURFACE_PARAMETER 8  /* causes recalc when variable changed */
#define PERMANENT  0x10    /* do not forget for new surface */
#define GLOB_USED  0x20    /* if allocated */
#define QUANTITY_MODULUS  0x40  /* multiplier for quantity */
#define QUANTITY_NAME  0x80  /* actual value for quantity */
#define QUANTITY_TARGET  0x100  /* target value for quantity */
#define QUANTITY_TYPE (QUANTITY_MODULUS|QUANTITY_NAME|QUANTITY_TARGET)
#define METHOD_MODULUS 0x200
#define METHOD_NAME 0x400
#define METHOD_TYPE (METHOD_MODULUS|METHOD_NAME)
#define ANY_TYPE  (FILE_VALUES|SUBROUTINE|QUANTITY_TYPE|METHOD_TYPE)
#define STRINGVAL 0x2000  /* string */
#define LEFTOVER  0x4000  /* if permanent left over from prev file */
#define OPTIMIZING_PARAMETER 0x8000  /* variable during optimization */
#define DYNAMIC_LOAD_FUNC 0x10000  /* dynamic load library function */

extern struct expnode torus_period_expr[MAXCOORD][MAXCOORD];
extern struct expnode ***view_transform_gen_expr;

/* for calc_periods() */
#define NO_ADJUST_VOLUMES 0
#define ADJUST_VOLUMES     1

/* Bits for calc_all_grads() and such */
#define CALC_FORCE     1
#define CALC_VOLGRADS 2


extern int torus_display_mode;  /* default, raw, connected, clipped */
/* modes */
#define TORUS_DEFAULT_MODE 0
#define TORUS_RAW_MODE      1
#define TORUS_CONNECTED_MODE 2
#define TORUS_CLIPPED_MODE 3

extern int read_depth,include_depth;
extern char datafilename[PATHSIZE];  /* current datafile name */
extern char filename[PATHSIZE];  /* file name in command */
extern FILE *commandfd;  /* command input file */
extern FILE *logfd;  /* command log file */
extern struct cmdfile { 
            FILE *fd;                /* for nested reads    */
            char filename[PATHSIZE]; /* command file name   */
            int line;                /* for error reporting */
            int datafile_flag;       /* whether datafile    */
         } cmdfile_stack[NESTDEPTH],datafile_stack[NESTDEPTH];
extern int datafile_flag;  /* 1 for datafile, 0 for command so expression
                                        parser knows what's up */
extern int lists_flag;  /* set when parsing space-separated lists */
#define LISTS_OFF 0
#define LISTS_SOME 1
#define LISTS_FULL 2

extern int const_expr_flag;  /* 1 for const_expr, 0 for command, so expression
                                        parser knows what's up */
extern int boundary_expr_flag; /* so parser knows when parsing boundary */
extern int reading_elements_flag; /* so parser knows attributes should not
     be accepted in expressions */
extern FILE *outfd;     /* for normal output */
extern int check_increase_flag;  /* to detect blowups */
extern int estimate_flag;    /* for toggling estimate of energy decrease */
extern int autorecalc_flag; /* for toggling autorecalc after variable assign  */
extern int autopop_flag;     /* whether to do autopopping */
extern int autochop_flag;     /* whether to do autochopping */
extern REAL autochop_size;  /* max edge length for autochop */
extern int autopop_count;  /* number of edges found */
extern int autochop_count;  /* number of edges found */
extern int parens;             /* level of parenthesis nesting */
extern int brace_depth;             /* level of brace nesting */
extern int in_quote;         /* so lexer knows when in string */
extern int in_comment;         /* so lexer knows when in string */
extern int effective_area_flag; /* use quadratic form for area around vertex */
extern int old_area_flag;     /* on for using old effective area */
extern int runge_kutta_flag; /* whether to use runge-kutta method for motion */
extern REAL total_time;      /* total scale factor */
extern REAL star_fraction;  /* weighting factor for star around vertices */
extern int area_fixed_flag;    /* for fixed area constraint */
extern REAL area_fixed_target;  /* target value for fixed area */
extern REAL area_fixed_pressure; /* Lagrange multiplier */
#define AREA_Q_ID 0x1111          /* for constraint identification */
extern int post_project_flag;     /* project to quant constr after each motion */
extern struct expnode mobility_formula;
extern int mobility_flag; /* whether mobility in effect */
extern struct expnode mobility_tensor[MAXCOORD][MAXCOORD];
extern int mobility_tensor_flag; /* whether tensor mobility in effect */
extern int check_pinning_flag;  /* for vertices changing constraints */
extern int nprocs;     /* number of parallel processors */
extern int procs_requested; /* number of processes desired */
extern REAL proc_total_area[MAXPROCS]; /* for individual processes */
extern int web_checksum; /* to see if web needs sending */
extern int dymem_checksum; /* see if dymem needs sending */
extern int comp_quant_vertex; /* during calc_quant_grad */
extern int comp_quant_vertexi; /* during calc_quant_hess */
extern int comp_quant_vertexj; /* during calc_quant_hess */
extern int comp_quant_type; /* during calc_quant_grad */
extern int comp_quant_stamp; /* coordination with eval_all and eval_sec */
extern vertex_id comp_quant_vi; /* during calc_quant_hess */
extern vertex_id comp_quant_vj; /* during calc_quant_hess */

extern REAL *f_sums;  /* facet_knot_energy, for sums to all other vertices */

extern char yytext[];
extern int yydebug;  /* flag for parser debugging */
extern int help_flag; /* avoid error message while doing help */
extern int memdebug;  /* flag for memory debugging in run_checks() */
extern int itdebug;  /* flag for iteration debugginng */
extern int tok;
extern int int_val;
extern REAL real_val;
extern int attr_etype; /* for ATTRIBUTE parse */
extern int subtype; /* for INDEXED_SUBTYPE parse */
extern int coord_num;
extern char idname[35]; /* for saving yytext */
extern char set_extra_name[100]; /* for saving name */
extern int line_no;
#define MAXHISTORY 100 
#define HISTORYSPACE 8000
extern char *history_space;
extern int  history_number;    /* number of current command */
extern int history_count;  /* number in list */
extern int  history_list[MAXHISTORY];
#define MAXCMDSIZE 2000 
extern char fulltext[MAXCMDSIZE+5]; /* for full text of commands */
extern int  fulltextsize;              /* length of command */
extern int yylval;  /* parser terminal value */
extern int aggrtype;  /* aggregate type being parsed */
extern int aggregate_depth; /* nesting depth of aggregate loops */
extern int attr_kind; /* kind of attribute being parsed */
extern char *default_name; /* for unnamed elements */
extern char last_name[50]; /* name of last element generator */
#define HISTBINS 21    /* bins in histogram (actually 1 less )*/

/* symbol table stuff */
#define SYMNAMESIZE 31
struct sym {
    char name[SYMNAMESIZE+1];
    int  type; /* element type, or see defines below */
    union { int intval;
            REAL realval;
            element_id idval;
            char *stringval;
          } value;
  };
#define SYMTYPE_INT     6
#define SYMTYPE_REAL    7
extern struct sym *elsym;  /* name of element during parsing */
extern struct sym *yysym;  /* name of identifier from lex */
extern struct sym symtable[];    /* symbol table */

/* squared curvature as part of energy */
extern int sqcurve_ignore_constr; /* set if to count fixed and constrained verts */
extern int square_curvature_flag;  /* set if to be counted */
extern int square_curvature_param;  /* which parameter for modulus */
extern int mean_curvature_param;    /* which parameter for modulus */
extern int mean_curv_int_quantity_num;    /* for everything quantities */
extern int sq_mean_curv_quantity_num; /* for everything quantities */
extern int kusner_flag;    /* set for edge square curvature */
extern int assume_oriented_flag; /* for orientation checking */
extern int boundary_curvature_flag; /* whether to include boundary vertices */
extern int conf_edge_curv_flag; /* set for conformal edge curvature squared */
extern int sqgauss_flag;  /* for squared gaussian curvature */
extern int sqgauss_param;    /* which parameter for modulus */
extern int normal_sq_mean_curvature_mi;
extern int eff_area_sq_mean_curvature_mi;
extern int sq_mean_curvature_mi;
extern int mix_sq_mean_curvature_mi;
extern int star_normal_sq_mean_curvature_mi;
extern int star_eff_area_sq_mean_curvature_mi;
extern int star_sq_mean_curvature_mi;
extern REAL target_length;  /* for string model */
extern int approx_curve_flag;  /* if approximate curvature in effect */
extern int mean_curv_int_flag;  /* for unsquared mean curvature */
extern int normal_curvature_flag; /* choice of curvature formula */
extern int div_normal_curvature_flag; /* choice of curvature formula */
/* flag bits to say if should be evaluated since modulus nonzero */
#define EVALUATE 0x0002

#define SELFSIM_NAME "self_sim_coeff"
extern int self_similar_flag;

/* for restricting motion to be along normals */
extern int normal_motion_flag;
typedef REAL pt_type[MAXCOORD];
extern pt_type *vertex_normals;  /* for storage of normals by ordinal */

/* for Dennis DeTurck unit normal motion */
extern int unit_normal_flag;
extern REAL deturck_factor;  /* weight for unit normal */

extern REAL **identmat;  /* handy identity matrix, set up in init_view */

/* homothety target value, set when homothety toggled on */
extern REAL homothety_target;

extern char *msg;      /* for constructing user messages */
extern int msgmax;     /* length allocated */
#define ERRMSGSIZE 2000
extern char errmsg[ERRMSGSIZE];  /* for error() routine */
extern int  parse_error_flag;  /* set when parser hits error */
extern int parse_errors;     /* for counting errors */
extern int  recovery_flag;      /* set while recovering from parsing error */
extern jmp_buf jumpbuf;    /* for error recovery  */
extern jmp_buf cmdbuf;    /* for command error recovery  */
extern jmp_buf m_jumpbuf[MAXPROCS];    /* for multiproc error recovery  */
#define UNRECOVERABLE 0
#define RECOVERABLE    1
#define WARNING         2
#define PARSE_ERROR    3
#define EXPRESSION_ERROR 4
#define COMMAND_ERROR 5
#define DATAFILE_ERROR 6
#define SYNTAX_ERROR 7
#define Q_ERROR 8
#define RECOVERABLE_QUIET 9

/* for queries */
extern int celement;
extern int commandverb;
extern int condition_flag;
#define ATTRIBUTE 19382
extern struct expnode *show_expr[NUMELEMENTS];  /* for element show exprs */
extern struct expnode show_command[NUMELEMENTS];  /* save for dump */
extern struct expnode show_expr_table[NUMELEMENTS];  /* actual expressions */
extern int query_intval;
extern REAL query_realval;
extern int set_query_type;
extern ATTR set_query_attr;
extern int query_coord;
/* values for set attribute queries */
#define SET_Q_ATTR  1301
#define SET_DENSITY 1302
#define SET_VOLUME  1303
#define SET_CONSTRAINT 1304
#define SET_COORD    1305
#define SET_PARAM    1306
#define SET_TAG      1307
#define UNSET_Q_ATTR 1308
#define UNSET_CONSTRAINT 1309
#define SET_COLOR  1310
#define SET_TRANS  1311
#define SET_FIXED  1312
#define DID_         4000


extern REAL volume_factorial;  /* simplex volume factor */
extern int subsimplex[1<<MAXCOORD][MAXCOORD];  /* for refining simplices */
struct simplex { int pt[MAXCOORD+1]; };
struct divedge { int endpt[2];  /* endpoints */
                 int divpt;      /* dividing point of edge */
               };



/* for expression parsing */
#define LISTMAX 200
extern struct treenode *list;    /* tree */
extern int listtop;  /* first spot reserved for root */
extern int listmax;  /* allocated spots */
/* flags for permanence of list */
#define USERCOPY 1
#define NOUSERCOPY 0

/* type of checking to do */
#define PRELIMCHECK  1
#define REGCHECK      2

/* web.representation */
#define STRING    1
#define SOAPFILM 2
#define SIMPLEX  3

/* for vol_project and sp_hessian_solve */
#define NO_SET_PRESSURE 0
#define SET_PRESSURE     1
#define PROJECT_FORCE    2

/* boundary projection types */
#define PARAMPROJ  1
#define TANGPROJ    2
#define GRADPROJ    3
#define PLAINPROJ  4

/* maximum number for constraint projection iterations */
#define MAXCONITER 10

/* for telling constr_proj to detect one-sided constraints */
#define DETECT 1
#define NO_DETECT 0

/* for telling one_sided_adjust() what to adjust */
#define ADJUST_ALL 0
#define ADJUST_VGRADS 1

#define DEFAULT_TOLERANCE  (1e-12)
extern REAL constraint_tolerance;

/* vertex averaging modes */
#define NOVOLKEEP 0
#define VOLKEEP    1
#define RAWEST     2

extern REAL  overall_size;  /* for anybody who wants to know how big */
extern int breakflag;      /* set by user interrupt */
#define BREAKFULL 1
#define BREAKLOOP 2 /* unused */
#define BREAKREPEAT 3
extern int iterate_flag;  /* so handler knows when iteration in progress */
extern int bare_edge_count;  /* edges without facets */

struct oldcoord {  
    REAL  (*coord)[MAXCOORD];  /* allocated space for old coordinates */
    REAL  energy;
    REAL optparam_values[MAXOPTPARAM];
    } ;
extern struct oldcoord saved;

/* ridge or valley indicators  */
#define RIDGE  1
#define VALLEY 2
extern int ridge_color_flag;  /* whether to differently color */
extern int edgeshow_flag;    /* whether to show edges of facets */
extern int triple_edgeshow_flag;    /* whether to show triple edges  */

/* vertex for zooming in on; default is first one read in */
extern int zoom_number;

/* for inner clipping for making zoom pictures */
extern int inner_clip_flag;
extern REAL inner_clip_rad;

/* conjugate gradient stuff */
extern int     conj_grad_flag;  /* whether conjugate gradient in effect */
extern REAL cg_oldsum;  /* total grad*grad from previous step */
extern REAL    (*cg_hvector)[MAXCOORD];  /* saved direction vector */
extern REAL cg_gamma;    /* direction adjustment factor */
extern int  ribiere_flag; /* to do Polak-Ribiere version */
#define RIBIERE_ATTR_NAME "old_force_ribiere"


/* structure for managing the facets of one body */
struct bodyface { facet_id f_id;
                  WRAPTYPE wrap;  /* wraps of base vertex */
                  int        wrapflag; /* whether wrap done */
                };

/* structure for depth sorting triangles */
struct tlink { int node; /* index in tlist */
               struct tlink *next;
             };
extern struct tsort { 
          element_id f_id;                /* facet this is for ( or edge ) */
          int color;                      /* color map index */
          int ecolor[FACET_EDGES];        /* edge colors */
          short etype[FACET_EDGES];       /* special edge type */
          int flag;                       /* set if not yet displayed */
          float x[FACET_VERTS][MAXCOORD]; /* display coordinates of vertices */
          float normal[MAXCOORD];         /* rearward normal */
          float mins[MAXCOORD];           /* minimum coordinates */
          float maxs[MAXCOORD];           /* maximum coordinates */
          float width;                    /* for edge */
          vertex_id v_id[FACET_VERTS];    /* vertices, if valid */
          unsigned int quadcode;          /* quadtree encoding */
          int sons;                       /* for topological sort */
          struct tlink *linkhead;
          int from;                       /* for loop detection */
       } *trilist;
/* flag is EDGE or FACET in lower bits, and */
#define FLIPPED_FACET 0x1000
#define WAS_BACKMOST  0x2000

/* bounding box, for PostScript */
extern REAL bbox_minx,bbox_miny,bbox_maxx,bbox_maxy;
extern int need_bounding_box;  /* flag to tell painter.c to calculate */

struct graphdata { REAL  x[MAXCOORD+1];  /* homogeneous coordinates */
                   REAL  norm[MAXCOORD]; /* unit normal */
                   int   color;          /* colormap index */
                   int   backcolor;      /* for back of facet */
                   int   ecolor;         /* edge color */
                   short etype;          /* edge type */
                   element_id id;        /* which element being graphed */
                   element_id v_id;      /* vertex */
                   int flags;
                };
/* flag bits */
/* SIMPLE_FACET and LIST_FACET are used in geomgraph.c */
#define SIMPLE_FACET  1
#define LIST_FACET     2
/* edge types */
#define  INVISIBLE_EDGE 0
#define  REGULAR_EDGE  1
#define  TRIPLE_EDGE    2
#define  SINGLE_EDGE    4
#define  BOUNDARY_EDGE 8
#define  FIXED_EDGE     0x10
#define  CONSTRAINT_EDGE 0x20
#define  BARE_EDGE      0x40
#define  SPLITTING_EDGE 0x80

typedef REAL  IColor[4];  /* r,g,b,a;  not same as OOGL ColorA */
#define IRIS_COLOR_MAX 16
extern IColor rgb_colors[IRIS_COLOR_MAX];
extern REAL facet_alpha;  /* global transparency */

/* homogeneous coordinate dimensions */
extern int HOMDIM;

/* function pointers for invoking device-specific graphics */
#ifdef NOPROTO
/* first four are for random-order plotting and are fed raw data */
extern void (*graph_start)();  /* called at start of graphing */
extern void (*graph_edge )();  /* called to graph one triangle */
extern void (*graph_facet)();  /* called to graph one triangle */
extern void (*graph_end)();     /* called at end of graphics */
/* second four are for painter algorithm output, are fed digested data */
extern void (*display_edge )();  /* hardware edge  display */
extern void (*display_facet)();  /* hardware facet display */
extern void (*init_graphics)();  /* hardware initialization */
extern void (*finish_graphics)();  /* hardware end of picture */
extern void (*close_graphics)();  /* close graphics window */
#else
extern void (*graph_start)(void); 
extern void (*graph_edge )(struct graphdata *,edge_id); 
extern void (*graph_facet)(struct graphdata *,facet_id); 
extern void (*graph_end)(void);    
extern void (*display_edge )(struct tsort *);
extern void (*display_facet)(struct tsort *);
extern void (*init_graphics)(void);
extern void (*finish_graphics)(void);
extern void (*close_graphics)(void);  /* close graphics window */
#endif

/* edge thicknesses for painter algorithm */
/* in order BARE_EDGE, FIXED_EDGE, CONSTRAINT_EDGE, BOUNDARY_EDGE,
   SINGLE_EDGE, TRIPLE_EDGE, and  other */
extern REAL *edgewidths;  /* pointer to be set by display() */
extern REAL pswidths[];  /* PostScript; values in variable.c */
extern REAL xwidths[];  /* X windows; values in variable.c */
extern REAL otherwidths[];  /* anything else; values in variable.c */

/* graphing flags */
extern  int init_flag; /* whether graphics initialized */
extern  int bdry_showflag;  /* whether to show facets on boundary */
extern  int no_wall_flag;        /* whether to suppress wall facets */
extern  int normflag;
extern  int thickenflag;
extern  int innerflag;
extern  int outerflag;
extern  int colorflag;
extern  int OOGL_flag;    /* whether MinneView initialized */
extern  int geomview_flag;    /* whether geomview initialized */
extern  int geompipe_flag;    /* for pipe only */
extern  REAL thickness;  /* for thickening */
extern  int user_thickness_flag; /* whether user has specified thickness */
extern  int view_4D_flag;  /* 0 for doing 3D xyz projection graphics, */
                                    /* 1 for outputting 4D graphics */

extern FILE * savefd;    /* file for binary dump of web */
extern FILE * data_fd;    /* file for initial data */

extern char cmapname[100];  /* colormap file name */
typedef REAL maprow[4];
extern  maprow *colormap; /* rgba colormap, values 0 to 1 */
extern int fillcolor;    /* current polygon fill color */
/* gaussian integration coefficients */
extern REAL gcombo[EDGE_CTRL][EDGE_INTERP];
extern REAL sdip[EDGE_CTRL][EDGE_INTERP];
extern REAL ssimp[EDGE_CTRL][EDGE_INTERP];
extern REAL gauss2wt[EDGE_INTERP];
extern REAL poly2partial[FACET_CTRL][2][2];
extern REAL scoeff[EDGE_CTRL][EDGE_CTRL]; /* plane area coeff */
extern REAL vcoeff[FACET_CTRL][FACET_CTRL][FACET_CTRL]; /* volume coeff*/
extern int set_by_user_gauss_1D;  /* minimums set by user */
extern int set_by_user_gauss_2D;

/* general parameters */
extern REAL **view;  /* transformation matrix */ 
extern int datafile_view_flag; /* whether datafile had view matrix */
extern int steps; 
extern int energy_init;      /* to keep track if we have current config energy */

/* for vertex popping */
struct verfacet { vertex_id v_id; facet_id f_id; };

#define MAXWULFF 100
extern REAL wulff_vector[MAXWULFF][MAXCOORD];  /* the vector components */
#ifdef NOPROTO
extern void (*get_wulff)();
#else
extern void (*get_wulff)(REAL *,REAL *);
#endif

extern int interp_bdry_param; /* flag for interp or extrap bdry param */
extern REAL **phase_data;  /* phase boundary energies */
extern char phase_file_name[PATHSIZE];  /* for dump */
extern int  phase_flag;     /* if phase boundary data in effect */
extern int phasemax;  /* number of phases */     

extern int fixed_constraint_flag;  /* set if restoring force valid */
extern REAL **leftside;  /* volume projection matrix */
extern REAL **rleftside;  /* volume projection matrix */
extern REAL *rightside;  /* right side of volume gradient constraints */
extern REAL *pressures; /* multiples of gradients to subtract from force */
extern int pressure_set_flag; /* pressures have been given values */
extern REAL *vol_deficit; /* bodyu volume deficits */
extern REAL *vol_restore; /* volume restoring gradient coefficients */
extern int no_refine;    /* keyword NOREFINE in data file header sets this
                                to disable initial triangulation of triangular
                                initial faces. */

/* interpolation structures */
#define MAXLEVEL 30
extern int reflevel;  /* refinement level */
extern REAL extrap_val[MAXLEVEL]; /* final value at level */

/* hessian minimization */
/* info for each vertex */
struct hess_verlist { 
            vertex_id v_id;  /* which vertex this is */
            int freedom;     /* degrees of freedom */
            int rownum;      /* starting row in sparse matrix  */
            REAL **proj;     /* projection matrix to constraints */
            REAL ***conhess; /* for constraint hessian */
            REAL slant;      /* cos of angle of normal from constraint */
          };
/* sparse hessian matrix entry */
struct hess_entry { 
          REAL value;
          int col;    /* which column */
          int row;    /* which row */
       };
extern struct hess_entry *entry_start;
/* sparse hessian matrix row header */
struct hess_index { int count;    /* number of entries in row */
                    union {
                      int listhead;     /* linked list start for row */
                      REAL *congrad;    /* for constraint gradient row */
                    } u;
                 };
extern struct hess_entry *hashtable;  /* the table */
extern int table_size;  /* hashtable size */
extern int hash_per_row;  /* for estimating table size */
#define HASHEMPTY (-1)
extern int hessian_by_diff_flag; /* for crude hessian */
extern int hessian_quiet_flag;  /* 1 to suppress hessian warnings */
extern int hessian_normal_flag; /* 1 for hessian constrained to normal */
extern int hessian_special_normal_flag; /*  for undocumented stuff */
extern int hessian_normal_perp_flag; /* 1 for hessian metric to normal */
extern int hessian_normal_one_flag; /* 1 for hessian constrained to 1D normal */
extern int hessian_double_normal_flag; /* 1 for hessian constrained to normal */
extern REAL hessian_slant_cutoff; /* for treating constrained vertices as
     fixed with hessian_normal */
extern int hessian_linear_metric_flag; /* linear interp dot product */
extern REAL linear_metric_mix;  /* proportion of linear interp metric */
extern REAL quadratic_metric_mix; /* proportion of quadratic interp metric */
extern int min_square_grad_flag; /* what to minimize in hessian_line_seek */
extern int hess_move_con_flag; /* whether to project to global constraints in move */
                                  /* projecting seems to be good idea */
extern REAL last_hessian_scale; /* from hessian_line_seek() */
extern REAL last_eigenvalue;    /* from eigenprobe and stuff */
extern struct hess_index *congrads; /* constraint grads on left */
extern REAL *conrhs;    /* right side of augmented matrix for constraints */
extern int A_total; /* entries for A matrix */
extern int A_rows;  /* rows for A matrix */
extern int total_entries; /* of hessian */
extern struct hess_verlist *vhead;  /* main vertex list */
extern int vcount;          /* total number of vertices */
extern struct hess_index *array;
#ifdef XXX
extern REAL *X;                 /* solution to augmented matrix  */
extern REAL *rhs;                 /* right side of augmented matrix  */
#endif
extern int hmode; /* mode of motion */
#define SINGLE_DEGREE 2
#define NORMAL_MOTION 1
#define UNRESTRICTED  0
/* type of solution; YSMP path numbers */
#define HESS_CRITICAL 3
#define HESS_DOWNHILL 7
#define FORT
#ifdef FORT
#define A_OFF 1
#else
#define A_OFF 0
#endif
/* for controlling steps */
extern int rhs_flag;     /* filling in right hand side */
extern int hess_flag;    /* filling in hessian matrix */
extern int negdiag;      /* C index of most negative entry on diagonal */
extern int bodyrowstart;  /* first row of body volume constraints */
extern int quanrowstart;  /* first row of named quantity constraints */

/* Lanczos default parameters */
#define KRYLOVDIM 100
#define NPRINT 15

/**************************************************************************/
/*    Linear system structure                                                             */
/*    for indefinite symmetric square systems, with linear constraints      */
/**************************************************************************/

/*************************************************************************
* The following data structure stores a node of the Metis separator tree
**************************************************************************/
struct SepNodeType {
  int nvtxs;     /* The number of vertices in the graph */
  int lo;        /* Low index of separator (inclusive) */
  int hi;        /* High index of separator (exclusive) */
  int isleaf;    /* Indicates if it is a leaf node */
  union { struct {
      REAL opc;      /* The opcount for this node */
      REAL subopc;   /* The opcount for the subtree of this node */
        } opc;  /* original metis fields */
        struct {
             int size;   /* size of matrix */
             int *vlist; /* list of variable numbers */
             REAL *mat;  /* triangular matrix */
        } info;  /* stuff useful in factoring */
  } u;
};
typedef struct SepNodeType SepNodeType;

extern int BK_flag; /*for enabling Bunch-Kauffman version of sparse factoring*/
extern REAL BKalpha; /* single/REAL pivot ratio */
struct BKrow { int start; /* index of first nonzero column in row */
                    REAL *entry; /* entries */
                 };
struct linsys { int flags;  /* status bits, see below */
                int N;      /* number of variables */
                /* sparse storage of original matrix in upper triangle */
                int *IA;    /* row starts (unpermuted) */
                int *JA;    /* columns of each entry, starting on diag */
                REAL *A;    /* entry values */
                int *P;     /* permutation in factors; P[0] is first variable */
                int *IP;    /* inverse permutation */
                struct SepNodeType *stree; /* metis separation tree */
                int streemax; /* max index of used stree */
                int maxsepsize; /* maximum separator size */
                int NSP;    /* size of work space, for ysmp */
                int *ISP;   /* work space for ysmp or whoever */
                /* permuted storage */
                int *pIA;    /* row starts */
                int *pJA;    /* columns of each entry, starting on diag */
                REAL *pA;    /* entry values */
                /* factorization */
                REAL lambda; /* factor A - lambda */
                int *psize;  /* types of pivots on diagonal */
                struct BKrow *rowhead; /* for rows of factors */
                /* for mindegree factorization */
                int Lsize;   /* number of entries */
                int *LIA;    /* starting indices */
                int *LJA;    /* columns for entries */
                int *LIJA;   /* starts of rows in LJA (overlapped) */
                REAL *LA;    /* values */
                /* constraint stuff */
                int concount; /* number of possible constraints */
                int CN;       /* number of actual constraints */
                int *coninx;  /* map from possible to actual constraints */
                int *coninxinv; /* inverse of coninx */
                REAL **C;     /* constraint gradients, rowwise, orth'd */
                REAL **HinvC; /* H^(-1)*C (transposed, actually) */
                REAL **CHinvCinv; /* (C^T*H^(-1)*C)^(-1) */
                /* inertia */
                int pos,neg,zero;
                /* Approximate inverse diagonal, for metric */
                REAL *apinv;
            };
/* flag bits */
#define S_HESSFILLED 1
#define S_RHSFILLED  2
#define S_FACTORED    4
#define S_PROJECTED  8
#define S_ODRV_REORDERED 0x10
#define S_JA_INCLUDED 0x20

/* psize values */
/* size of diagonal elements, 1 for 1x1, 2 for first of 2x2, 
    3 for second of 2x2, 0 for zero valued 1x1 */
#define ZEROPIVOT 0
#define ONEBYONE  1
#define FIRSTOFPAIR 2
#define SECONDOFPAIR 3

/* vector-to-form  metric */
extern struct linsys Met;

extern int ysmp_flag;  /* set if doing Yale Sparse Matrix version */
/* sparse matrix function pointers, for easy switching among algorithms */
/* values for ysmp_flag */
#define MINDEG_FACTORING 0
#define YSMP_FACTORING 1
#define METIS_FACTORING 2

/* multiply vector by original sparse matrix */
extern void (*sp_mul_func)ARGS((struct linsys *, REAL*,REAL*));

/* convert raw Hessian data to standard sparse format */
extern void (*sp_AIJ_setup_func)ARGS((struct hess_index*,int,struct linsys*));

/* set up  matrices needed for handling constraints */
extern void (*sp_constraint_setup_func)ARGS((struct hess_index *,int,
    struct linsys *));

/* set up projection to constraints using hessian metric */
extern void (*sp_hess_project_setup_func)ARGS((struct linsys *)) ;

/* factor matrix */
extern void (*sp_factor_func)ARGS((struct linsys *)) ;

/* return a vector in kernel, supposing nullity > 0 */
extern int (*sp_kernel_func)ARGS((struct linsys *,REAL *)) ;

/* solve given rhs */
extern void (*sp_solve_func)ARGS((struct linsys *,REAL *,REAL *)) ;

/* solve multiple given rhs */
extern void (*sp_solve_multi_func)ARGS((struct linsys*,REAL**,REAL**,int));

/* optional ordering of vertices */
extern void (*sp_ordering_func)ARGS((struct linsys *)) ;


/* to suppress a proliferation of warnings */
extern int pos_def_warning_flag;
extern int eigen_pos,eigen_neg,eigen_zero;  /* inertia of shifted hessian */
extern int mat_index; /* number of negatives on diagonal */
extern int mat_null; /* number of zeroes on diagonal */
extern REAL hessian_epsilon; /* cutoff for diagonal elements */
#define  HESSIAN_EPS_DEFAULT (1e-8)
extern int make_pos_def_flag;    /* force hessian to positive definiteness */
extern int hess_debug;  /* debugging flag */



/* model dependent function pointers */
#ifdef NOPROTO
extern REAL (*userfunc[])();
extern REAL (*userfunc_deriv[])();
extern REAL (*userfunc_seconds[])();
extern void (*calc_facet_energy)();
extern void (*calc_facet_forces)();
extern void (*calc_facet_volume)();
extern void (*calc_edge_energy)();
extern void (*calc_edge_forces)();
extern void (*calc_edge_area)();
extern void (*string_grad)();
extern void (*film_grad)();
#else
extern REAL (*userfunc[])(REAL*);
extern REAL (*userfunc_deriv[])(REAL*,REAL*);
extern REAL (*userfunc_seconds[])(REAL*,REAL*,REAL**);
extern void (*calc_facet_energy)(facet_id);
extern void (*calc_facet_forces)(facet_id);
extern void (*calc_facet_volume)(facet_id);
extern void (*calc_edge_energy)(edge_id);
extern void (*calc_edge_forces)(edge_id);
extern void (*calc_edge_area)(edge_id);
extern void (*string_grad)(void);
extern void (*film_grad)(void);
#endif

/* gaussian integration on [0,1] */
extern REAL *gauss1Dpt;
extern REAL *gauss1Dwt;
extern int  gauss1D_num;
extern REAL **gauss1poly;  /* interp polys at gauss pts */
extern REAL **gauss1polyd; /* and derivatives              */
                  /* values for linear or quadratic model, whichever in effect */
extern int  edge_ctrl; /* control points on edge, 2 linear, 3 quadratic */

/* 2D gauss */
typedef REAL barytype[3];
extern barytype *gauss2Dpt;
extern REAL *gauss2Dwt; 
extern REAL gauss2Dpt1[1][3]; 
extern REAL gauss2Dwt1[1];
extern REAL gauss2Dpt2[3][3]; 
extern REAL gauss2Dwt2[3];
extern REAL gauss2Dpt5[7][3]; 
extern REAL gauss2Dwt5[7];
extern REAL gauss2Dpt6[12][3]; 
extern REAL gauss2Dwt6[12];
extern REAL gauss2Dpt8[16][3]; 
extern REAL gauss2Dwt8[16];
extern REAL gauss2Dpt11[28][3]; 
extern REAL gauss2Dwt11[28];
extern REAL gauss2Dpt13[37][3]; 
extern REAL gauss2Dwt13[37];

/* general dimension gauss integration variables and arrays */
extern int ctrl_num;  /* number of control points */
extern int gauss2D_num; /* number of integration points */
extern REAL **gpoly;    /* interpolation polynomial value at integration point k
                         for polynomial of control point j */
extern REAL ***gpolypartial; /* partials of interpolation polynomials */
                         /* gausspt,dim,ctrlpt */

/* new gaussian-lagrange structure */
#define MAXGAUSSORDER 40
struct gauss_lag { int lagrange_order;  /* computed for */
                   int  gnumpts;        /* number of gauss points */
                   REAL **gausspt;      /* gaussian integration pts,
                                           in barycentric coords add to 1 */
                   REAL *gausswt;       /* weights at gauss pts */
                   int  lagpts;         /* number of lagrange pts */
                   REAL **gpoly;        /* basis polynomial of lagrange pt
                                           at gauss pt; index [g][l];
                                           mat mult lagrange to get gauss */
                   REAL ***gpolypart;   /* partials of basis polynomials;
                                           index [g][dim][l];  mult lagrange
                                           to get tangents */
                   REAL ***lpolypart;   /* partials at lagrange pts */
               };
extern struct gauss_lag gauss_lagrange[MAXCOORD][MAXGAUSSORDER];

extern REAL **metric;  /* metric values at a point */
extern REAL ***metric_partial; /* partial derivatives of metric */
extern REAL **det_array;    /* tangent vector dot products */
/*extern REAL **tang; */    /* tangents to surface at point */
extern REAL euclidean_area;    /* euclidean area for conformal metrics */
extern int klein_metric_flag;
extern int metric_convert_flag; /* whether to do form-to-vector conversion */

/* for defining pointer-pointer matrices as local variables */

/* need token-pasting, so gets weird with testing style needed */
#define bbxt 1
#define cxtc
#define bbxtcxtc  2
#if  defined(NeXT) || (__STDC__ == 1) || defined(__SUNPRO_C)
#define ddxt 1
#else
#define ddxt  bbxt/**/cxtc
#endif

#if  (ddxt == 1)

#define MAT2D(name,rows,cols)  \
  REAL *name##qXvS[rows];\
  REAL name##xJ[rows][cols];\
  REAL **name = mat2d_setup(name##qXvS,(REAL*)name##xJ,rows,cols)

#define MAT3D(name,rows,cols,levels)  \
  REAL **name##qXvS[(rows)+(rows)*(cols)];\
  REAL name##xJ[rows][cols][levels];\
  REAL *** name=mat3d_setup(name##qXvS,(REAL*)name##xJ,rows,cols,levels)

#define MAT4D(name,rows,cols,levels,tiers)  \
  REAL ***name##qXvS[(rows)+(rows)*(cols)+(rows)*(cols)*(levels)];\
  REAL name##xJ[rows][cols][levels][tiers];\
  REAL ****name=mat4d_setup(name##qXvS,(REAL*)name##xJ,rows,cols,levels,tiers)

#else

#define MAT2D(name,rows,cols)  \
  REAL *name/**/qXvS[rows];\
  REAL name/**/xJ[rows][cols];\
  REAL ** name = mat2d_setup(name/**/qXvS,(REAL*)name/**/xJ,rows,cols)

#define MAT3D(name,rows,cols,levels)  \
  REAL **name/**/qXvS[(rows)+(rows)*(cols)];\
  REAL name/**/xJ[rows][cols][levels];\
  REAL *** name = mat3d_setup(name/**/qXvS,(REAL*)name/**/xJ,rows,cols,levels)

#define MAT4D(name,rows,cols,levels,tiers)  \
  REAL ***name/**/qXvS[(rows)+(rows)*(cols)+(rows)*(cols)*(levels)];\
  REAL name/**/xJ[rows][cols][levels][tiers];\
  REAL **** name=mat4d_setup(name/**/qXvS,(REAL*)name/**/xJ,rows,cols,levels,tiers)
#endif
/* end matrix defines */


struct veredge { vertex_id v_id; edge_id e_id; }; /* used in verpopst.c */

/* symmetry group stuff */
extern char *symmetry_name;    /* to be sure datafile matches */
#ifdef NOPROTO
typedef void SYM_WRAP();      /* points to user-defined wrap function */
typedef WRAPTYPE SYM_COMP(); /* points to group composition function */
typedef WRAPTYPE SYM_INV (); /* points to group inverse function      */
typedef void SYM_FORM();      /* points to form pullback function      */
#else
typedef void SYM_WRAP(REAL*,REAL*,WRAPTYPE);
typedef WRAPTYPE SYM_COMP(WRAPTYPE,WRAPTYPE);
typedef WRAPTYPE SYM_INV (WRAPTYPE);
typedef void SYM_FORM(REAL*,REAL*,REAL*,WRAPTYPE);
#endif
extern SYM_WRAP *sym_wrap;
extern int         sym_flags;
extern SYM_FORM *sym_form_pullback;
extern SYM_INV  *sym_inverse;
extern SYM_COMP *sym_compose;
/* structure for registry.c */
struct sym_registry {
    char *name;  /* name to match in datafile SYMMETRY_GROUP phrase */
    int flags;  /* see bits below */
    SYM_WRAP *wrapper; /* point wrap function */
    SYM_COMP *compose; /* group composition function */
    SYM_INV  *inverse; /* group inverse function */
    SYM_FORM *pullback; /* form pullback function */
    };
 extern struct sym_registry sym_register[]; /* actual in registry.c */
/* flag bits */
#define HAS_FIXED_PTS 1
#define NEED_FORM_UNWRAPPING 2
#define DOUBLE_AXIAL 4
    

/* specific case of torus symmetry representation */
#define TWRAPBITS 6
#define POSWRAP    1
#define WRAPMASK  037
#define ALLWRAPMASK 03737373737
#define NEGWRAP    WRAPMASK

/* for symmetry group "rotate" */
extern int rotorder;

/* additional viewing transforms */
extern int transform_count;
extern REAL ***view_transforms;
extern int *view_transform_det; /* to see if normals need flipping */
extern int transforms_flag; /* whether to show transforms */
extern int transform_gen_count;
extern REAL ***view_transform_gens;    /* generators */
extern char transform_expr[100];  /* save it */
extern int transform_depth;  /* tree depth in transform generation */
extern int *transform_colors;
extern int *transform_gen_swap;
#define SAME_COLOR  0xf9878
#define SWAP_COLORS  0xFABC
extern int transform_colors_flag;

/* color for transparent facets */
#define CLEAR (-1)
/* color for unshown facet */
#define UNSHOWN (-2)

/* from borlandc/graphics.h */
#if      !defined(__COLORS)
#define __COLORS

enum COLORS {
     BLACK,    /* dark colors */
     BLUE,
     GREEN,
     CYAN,
     RED,
     MAGENTA,
     BROWN,
     LIGHTGRAY,
     DARKGRAY, /* light colors */
     LIGHTBLUE,
     LIGHTGREEN,
     LIGHTCYAN,
     LIGHTRED,
     LIGHTMAGENTA,
     YELLOW,
     WHITE
};
#endif


#ifdef SGI_MULTI
/* process ids */
extern int proc_ids[MAXPROCS];

#define M_ACTIVE 423
#define M_INACTIVE 0
extern int mpflag;  /* whether multiprocessing in action */
extern int m_breakflag[MAXPROCS]; /* for user interrupts */

/* locks and stuff */
extern usptr_t *lock_arena;  /* arena where locks are */
extern ulock_t locklist[_MAXLOCKS];  /* only 4096 available */
extern char lock_arena_name[]; /* for usinit() */

/* for keeping force updates from conflicting */
struct procforce { vertex_id v_id;
                   int next;  /* link */
                   REAL f[MAXCOORD];
                 };
extern struct procforce *pforce;
extern int phead[MAXPROCS][MAXPROCS]; /* [owner][calculator] */
extern struct procforce *pbase[MAXPROCS];  /* allocated to calculator */
extern int ptop[MAXPROCS];    /* used per calculator */
extern int pmax[MAXPROCS];    /* available per calculator */
#endif

/* Graphing mutual exclusion for multi-threaded version */
#ifdef WIN32
/* Yield time slice if the graphing thread is active */
extern int volatile graph_semaphore;
#define ENTER_GRAPH_MUTEX { while ( graph_semaphore )  Sleep(0); \
                                    graph_semaphore = 1; }
#define LEAVE_GRAPH_MUTEX  graph_semaphore = 0;
#else
#define ENTER_GRAPH_MUTEX 
#define LEAVE_GRAPH_MUTEX  
#endif
