/*
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * seq_dd
 *
 * Simple testing program to write a numerical sequence to a device, or
 * to verify that a device contains a desired numerical sequence. Inspired
 * by "hashdd" from Joe Thornber.
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>

#define SECTOR_SIZE		512
#define ENTRIES_PER_SECTOR	(SECTOR_SIZE / sizeof(u_int64_t))

static void usage(void)
{
	printf("Usage:\n");
	printf("\tseq_dd\n");
	printf("\t\tof=<output_file> | if=<input_file>\n");
	printf("\t\tstart=<starting_sector> [default = 0]\n");
	printf("\t\tcount=<num_sectors> [default = 8]\n");
	printf("\t\tindex=<starting_index> [default = 0]\n");
}

static int read_file(char *filename, unsigned long start,
		     unsigned long count, unsigned long index)
{
	u_int64_t *buffer;
	unsigned long i;
	int fd, rc, j, failed = 0;

	buffer = malloc(SECTOR_SIZE);
	if (!buffer) {
		fprintf(stderr, "Memory allocation error\n");
		return 1;
	}

	fd = open(filename, O_RDONLY);
	if (fd < 0) {
		fprintf(stderr, "Error opening \"%s\": %s\n",
			filename, strerror(errno));
		free(buffer);
		return 1;
	}

	rc = lseek(fd, start * SECTOR_SIZE, SEEK_SET);
	if (rc < 0) {
		fprintf(stderr, "Error seeking to starting sector %lu\n", start);
		free(buffer);
		return 1;
	}

	for (i = 0; i < count && !failed; i++, index++) {
		rc = read(fd, buffer, SECTOR_SIZE);
		if (rc != SECTOR_SIZE) {
			fprintf(stderr, "Unable to read sector %lu\n", start + i);
			break;
		}

		for (j = 0; j < ENTRIES_PER_SECTOR; j++) {
			if (buffer[j] != index) {
				failed = 1;
				break;
			}
		}
	}

	rc = (i == count) ? (failed) ? 1 : 0 : 1;

	printf("Read %lu sectors from %s starting at sector %lu\n",
	       i, filename, start);
	printf("%s %svalid\n", filename, rc ? "in" : "" );

	return rc;
}

static int write_file(char *filename, unsigned long start,
		      unsigned long count, unsigned long index)
{
	u_int64_t *buffer;
	unsigned long i;
	int fd, rc, j;

	buffer = malloc(SECTOR_SIZE);
	if (!buffer) {
		fprintf(stderr, "Memory allocation error\n");
		return 1;
	}

	fd = open(filename, O_CREAT | O_WRONLY);
	if (fd < 0) {
		fprintf(stderr, "Error opening \"%s\": %s\n",
			filename, strerror(errno));
		free(buffer);
		return 1;
	}

	rc = lseek(fd, start * SECTOR_SIZE, SEEK_SET);
	if (rc < 0) {
		fprintf(stderr, "Error seeking to starting sector %lu\n", start);
		free(buffer);
		return 1;
	}

	for (i = 0; i < count; i++, index++) {
		for (j = 0; j < ENTRIES_PER_SECTOR; j++) {
			buffer[j] = index;
		}
		rc = write(fd, buffer, SECTOR_SIZE);
		if (rc != SECTOR_SIZE) {
			fprintf(stderr, "Unable to write index %lu to sector %lu\n",
				index, start + i);
			break;
		}
	}

	printf("Wrote %lu sectors to %s starting at sector %lu\n",
	       i, filename, start);
	free(buffer);

	return (i == count) ? 0 : 1;
}

int main(int argc, char **argv)
{
	struct stat st;
	char *filename = NULL;
	char act[10];
	char action = 0;
	int i, j, rc;
	unsigned long start = 0, count = 8, index = 0;

	if (argc < 2) {
		usage();
		return 1;
	}

	for (i = 1; i < argc; i++) {
		j = strcspn(argv[i], "=");
		j++;
		if (j < 3 || j > 6) {
			usage();
			return 1;
		}
		memset(act, 0, sizeof(char)*10);
		strncpy(act, argv[i], j);
		if (!strcmp(act, "if=")) {
			filename = &(argv[i][j]);
			action = 'r';
		} else if (!strcmp(act, "of=")) {
			filename = &(argv[i][j]);
			action = 'w';
		} else if (!strcmp(act, "start=")) {
			start = strtoul(&(argv[i][j]), NULL, 10);
		} else if (!strcmp(act, "count=")) {
			count = strtoul(&(argv[i][j]), NULL, 10);
		} else if (!strcmp(act, "index=")) {
			index = strtoul(&(argv[i][j]), NULL, 10);
		} else {
			usage();
			return 1;
		}
	}

	if (!filename) {
		usage();
		return 1;
	}

	rc = stat(filename, &st);
	if (rc) {
		if (!(action == 'w' && errno == ENOENT)) {
			printf("Unable to stat %s: %s\n", filename, strerror(errno));
			return 1;
		}
	} else {
		if (!S_ISREG(st.st_mode) && !S_ISBLK(st.st_mode)) {
			printf("%s is not a regular file or a block-device.\n", filename);
			return 1;
		}
	}

	if (action == 'w') {
		rc = write_file(filename, start, count, index);
	} else if (action == 'r') {
		rc = read_file(filename, start, count, index);
	} else {
		usage();
		return 1;
	}

	return rc;
}

