/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM2 Plugin
 * File: evms2/engine/plugins/lvm2/lvm2.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvm2.h"

engine_functions_t *EngFncs;
list_anchor_t lvm2_containers;
boolean device_size_prompt;

/**
 * lvm2_setup_evms_plugin
 *
 * Register the LVM2 namespace, create the global containers list, and
 * get the config file entries for the lvm2 section.
 **/
static int lvm2_setup_evms_plugin(engine_functions_t *functions)
{
	int rc = 0;
	EngFncs	= functions;

	LOG_ENTRY();

	rc = EngFncs->register_name(LVM2_DEV_DIRECTORY);
	if (rc) {
		LOG_ERROR("Error registering directory name.\n");
		goto out;
	}

	lvm2_containers = EngFncs->allocate_list();
	if (!lvm2_containers) {
		LOG_ERROR("Error allocating container list.\n");
		rc = ENOMEM;
	}

	device_size_prompt = TRUE;
	EngFncs->get_config_bool("lvm2.device_size_prompt",
				 &device_size_prompt);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_cleanup_evms_plugin
 *
 * Discard all regions and containers. Then free the global
 * containers list and the LVM2 namespace.
 **/
static void lvm2_cleanup_evms_plugin(void)
{
	storage_container_t *container;
	list_element_t iter1, iter2;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(lvm2_containers, iter1, iter2, container) {
		my_plugin_record->functions.plugin->discard(container->objects_produced);
		my_plugin_record->container_functions->discard_container(container);
	}

	EngFncs->destroy_list(lvm2_containers);
	EngFncs->unregister_name(LVM2_DEV_DIRECTORY);

	LOG_EXIT_VOID();
}

/**
 * lvm2_can_delete
 *
 * Only data regions can be deleted.
 **/
static int lvm2_can_delete(storage_object_t *region)
{
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Checking if region %s can be deleted.\n", region->name);

	if (region->data_type != DATA_TYPE) {
		LOG_DEBUG("Cannot delete freespace region %s.\n", region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_can_expand
 *
 * Can this region be expanded? If so, build an expand_object_info_t and add
 * it to the expand_points list.
 **/
static int lvm2_can_expand(storage_object_t *region,
			   sector_count_t expand_limit,	/* a delta size */
			   list_anchor_t expand_points)	/* of type expand_object_info_t */
{
	storage_container_t *container = region->producing_container;
	container_data_t *c_data = container->private_data;
	expand_object_info_t *expand_point;
	storage_object_t *freespace;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Checking if region %s can be expanded.\n", region->name);

	rc = can_expand_region(region);
	if (rc) {
		goto out;
	}

	/* Make sure the expand_limit is at least as big as the extent-size. */
	if (expand_limit < c_data->pe_size) {
		rc = ENOSPC;
		goto out;
	}

	/* Create a new expansion info object. */
	expand_point = EngFncs->engine_alloc(sizeof(*expand_point));
	if (!expand_point) {
		rc = ENOMEM;
		goto out;
	}

	/* The region can expand into the remaining freespace. */
	freespace = get_freespace_region(container->objects_produced);
	expand_point->object = region;
	expand_point->max_expand_size = min(freespace->size, expand_limit -
					    (expand_limit % c_data->pe_size));

	EngFncs->insert_thing(expand_points, expand_point, INSERT_AFTER, NULL);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_can_shrink
 *
 * Can this region be shrunk? If yes, build a shrink_object_info_t and add
 * it to the shrink_points list.
 **/
static int lvm2_can_shrink(storage_object_t *region,
			   sector_count_t shrink_limit,	/* a delta size */
			   list_anchor_t shrink_points)	/* of type shrink_object_info_t */
{
	container_data_t *c_data = region->producing_container->private_data;
	region_data_t *r_data = region->private_data;
	shrink_object_info_t *shrink_point;
	region_mapping_t *r_map;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Checking if region %s can be shrunk.\n", region->name);

	rc = can_shrink_region(region);
	if (rc) {
		goto out;
	}

	/* Make sure the shrink_limit is at least as big as the extent-size. */
	if (shrink_limit < c_data->pe_size) {
		rc = ENOSPC;
		goto out;
	}

	/* Create a new shrink info object. */
	shrink_point = EngFncs->engine_alloc(sizeof(*shrink_point));
	if (!shrink_point) {
		rc = ENOMEM;
		goto out;
	}

	/* The region can shrink down to one extent. */
	r_map = EngFncs->first_thing(r_data->mappings, NULL);
	shrink_point->object = region;
	shrink_point->max_shrink_size = region->size -
					c_data->pe_size * r_map->stripe_count;

	/* If this shrink-point size is greater than the shrink-limit,
	 * then the region can only shrink by that limit.
	 */
	if (shrink_point->max_shrink_size > shrink_limit) {
		shrink_point->max_shrink_size = shrink_limit -
						(shrink_limit % c_data->pe_size);
	}

	EngFncs->insert_thing(shrink_points, shrink_point, INSERT_AFTER, NULL);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_discover
 *
 * Look for input objects with LVM2 metadata. Construct new containers to
 * consume these objects. Construct new regions according to the metadata
 * and export them to the engine on the output list.
 **/
static int lvm2_discover(list_anchor_t input_objects,
			 list_anchor_t output_objects,
			 boolean final_call)
{
	list_anchor_t pv_list;
	int count = 0;

	LOG_ENTRY();
	LOG_DEBUG("Beginning LVM2 Discovery\n");

	/* Allocate a list to temporarily store the PV objects. */
	pv_list = EngFncs->allocate_list();
	if (!pv_list) {
		goto out;
	}

	/* Discover all PV objects and put them on the PV list. */
	discover_pv_objects(input_objects, pv_list);

	/* Copy the remaining non-PV objects to the output list. */
	EngFncs->merge_lists(output_objects, input_objects, NULL, NULL);

	/* Construct all containers based on the PV list. */
	discover_containers(pv_list);

	check_containers(final_call);

	/* Construct all regions in the containers. */
	count = discover_regions(output_objects, final_call);

	if (final_call) {
		cleanup_containers();
	}

out:
	EngFncs->destroy_list(pv_list);
	LOG_DEBUG("LVM2 Discovery complete.\n");
	LOG_EXIT_INT(count);
	return count;
}

/**
 * lvm2_create
 *
 * Create a new region using the freespace region specified on the
 * input-objects list along with the specified options. Place the
 * new region on the output-objects list.
 **/
static int lvm2_create(list_anchor_t input_objects,
		       option_array_t *options,
		       list_anchor_t output_objects)
{
	storage_object_t *freespace, *region;
	storage_container_t *container;
	list_anchor_t objects = NULL;
	u_int64_t size, stripes, stripe_size;
	char *lv_name;
	int rc;

	LOG_ENTRY();

	/* There should only be one selected freespace region,
	 * so just get the first thing in the list.
	 */
	freespace = get_freespace_region(input_objects);
	if (!freespace) {
		LOG_ERROR("No freespace region selected.\n");
		rc = EINVAL;
		goto out;
	}
	container = freespace->producing_container;

	/* Parse and verify the user's options. */
	create_region_parse_options(container, options, &lv_name, &size,
				    &stripes, &stripe_size, &objects);

	rc = create_region_validate_options(container, lv_name, &size,
					    &stripes, &stripe_size, objects);
	if (rc) {
		LOG_ERROR("Error validating options for region create.\n");
		goto out;
	}

	/* Pre-validate extent allocation. */
	rc = prevalidate_extent_allocation(container, objects, size, stripes);
	if (rc) {
		goto out;
	}

	/* Allocate a new region. */
	region = create_new_region(container, lv_name);
	if (!region) {
		rc = ENOMEM;
		goto out;
	}

	/* Add the region to the container. */
	add_region_to_container(region, container);

	/* Allocate extents and create mappings. */
	rc = allocate_extents_for_region(region, objects, size,
					 stripes, stripe_size);
	if (rc) {
		deconstruct_region_mappings(region);
		remove_region_from_container(region);
		deallocate_region(region);
		goto out;
	}

	/* Recreate the freespace mappings. */
	delete_freespace_mappings(container);
	rc = create_freespace_mappings(container);
	if (rc) {
		/* FIXME: Any cleanup possible? */
		goto out;
	}

	EngFncs->insert_thing(output_objects, region, INSERT_AFTER, NULL);
	container->flags |= SCFLAG_DIRTY;

out:
	EngFncs->destroy_list(objects);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_discard
 *
 * Forget about these regions, but don't delete them. Just clean up the
 * private-data and remove them from their container. The Engine will call
 * to deactivate the regions during commit.
 **/
static int lvm2_discard(list_anchor_t regions)
{
	storage_object_t *region;
	list_element_t iter1, iter2;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(regions, iter1, iter2, region) {
		if (region->data_type != DATA_TYPE) {
			LOG_DEBUG("Cannot discard freespace region %s.\n",
				  region->name);
			continue;
		}

		LOG_DEBUG("Discarding region %s.\n", region->name);
		deconstruct_region_mappings(region);
		remove_region_from_container(region);
		deallocate_region(region);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm2_delete
 *
 * Delete this region and free all of its private-data. Remove the region from
 * its producing container and remove the region from any PV's parent-objects
 * list.
 **/
static int lvm2_delete(storage_object_t *region, list_anchor_t child_objects)
{
	storage_container_t *container = region->producing_container;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Deleting region %s.\n", region->name);

	rc = lvm2_can_delete(region);
	if (rc) {
		goto out;
	}

	deconstruct_region_mappings(region);
	remove_region_from_container(region);
	deallocate_region(region);

	delete_freespace_mappings(container);
	rc = create_freespace_mappings(container);
	if (rc) {
		/* FIXME: Any cleanup possible? */
		goto out;
	}

	container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_expand
 *
 * Expand this region using the specified options.
 **/
static int lvm2_expand(storage_object_t *region,
		       storage_object_t *expand_object,
		       list_anchor_t input_objects,
		       option_array_t *options)
{
	storage_container_t *container = region->producing_container;
	list_anchor_t objects = NULL;
	u_int64_t size, stripes, stripe_size, expand_delta;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Expanding region %s.\n", region->name);

	/* If region and expand_object are not the same, someone is
	 * trying to expand an object below LVM. Not allowed.
	 */
	if (region != expand_object) {
		LOG_ERROR("Cannot expand object %s below region %s.\n",
			  expand_object->name, region->name);
		rc = ENOSYS;
		goto out;
	}

	rc = can_expand_region(region);
	if (rc) {
		goto out;
	}

	/* Parse and verify the user's options. */
	expand_region_parse_options(region, options, &size,
				    &stripes, &stripe_size, &objects);

	rc = expand_region_validate_options(region, &size, &stripes,
					    &stripe_size, objects);
	if (rc) {
		LOG_ERROR("Error validating options for region expand.\n");
		goto out;
	}

	/* Check with the engine to make sure all plugins
	 * and FSIMS above LVM approve of the expand.
	 */
	expand_delta = size;
	rc = EngFncs->can_expand_by(region, &expand_delta);
	if (rc) {
		LOG_ERROR("Request to expand %s by %"PRIu64" sectors, but "
			  "engine will only allow up to %"PRIu64" sectors.\n",
			  region->name, size, expand_delta);
		goto out;
	}

	/* Pre-validate extent allocation. */
	rc = prevalidate_extent_allocation(container, objects, size, stripes);
	if (rc) {
		goto out;
	}

	/* Allocate extents and create mappings. */
	rc = allocate_extents_for_region(region, objects, size,
					 stripes, stripe_size);
	if (rc) {
		/* FIXME: Cleanup??? */
		goto out;
	}

	/* Merge consecutive mappings.*/
	rc = merge_region_mappings(region);
	if (rc) {
		/* FIXME: Cleanup??? */
		goto out;
	}

	/* Recreate the freespace mappings. */
	delete_freespace_mappings(container);
	rc = create_freespace_mappings(container);
	if (rc) {
		/* FIXME: Any cleanup possible? */
		goto out;
	}

	container->flags |= SCFLAG_DIRTY;
	if (region->flags & SOFLAG_ACTIVE) {
		region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

out:
	EngFncs->destroy_list(objects);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_shrink
 *
 * Shrink this region using the specified options.
 **/
static int lvm2_shrink(storage_object_t *region,
		       storage_object_t *shrink_object,
		       list_anchor_t input_objects,
		       option_array_t *options)
{
	storage_container_t *container = region->producing_container;
	u_int64_t size, shrink_delta;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Shrinking region %s.\n", region->name);

	/* If region and shrink_object are not the same, someone is
	 * trying to expand an object below LVM. Not allowed.
	 */
	if (region != shrink_object) {
		LOG_ERROR("Cannot shrink object %s below region %s.\n",
			  shrink_object->name, region->name);
		rc = ENOSYS;
		goto out;
	}

	rc = can_shrink_region(region);
	if (rc) {
		goto out;
	}

	/* Parse and verify the user's options. */
	shrink_region_parse_options(region, options, &size);

	rc = shrink_region_validate_options(region, &size);
	if (rc) {
		LOG_ERROR("Error validating options for region shrink.\n");
		goto out;
	}

	/* Check with the engine to make sure all plugins
	 * and FSIMS above LVM approve of the shrink.
	 */
	shrink_delta = size;
	rc = EngFncs->can_shrink_by(region, &shrink_delta);
	if (rc) {
		LOG_ERROR("Request to shrink %s by %"PRIu64" sectors, but "
			  "engine will only allow up to %"PRIu64" sectors.\n",
			  region->name, size, shrink_delta);
		goto out;
	}

	/* Remove the specified size from the region. */
	rc = deallocate_extents_from_region(region, size);
	if (rc) {
		/* FIXME: Cleanup??? */
		goto out;
	}

	/* Recreate the freespace mappings. */
	delete_freespace_mappings(container);
	rc = create_freespace_mappings(container);
	if (rc) {
		/* FIXME: Any cleanup possible? */
		goto out;
	}

	container->flags |= SCFLAG_DIRTY;
	if (region->flags & SOFLAG_ACTIVE) {
		region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_commit_changes
 *
 * All LVM2 metadata commits are done in lvm2_commit_container_changes.
 **/
static int lvm2_commit_changes(storage_object_t *region, commit_phase_t phase)
{
	LOG_ENTRY();
	region->flags &= ~(SOFLAG_DIRTY | SOFLAG_NEW);
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm2_can_activate
 *
 * All data regions can always be activated.
 **/
static int lvm2_can_activate(storage_object_t *region)
{
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Checking if region %s can be activated.\n", region->name);

	if (region->data_type != DATA_TYPE) {
		LOG_DEBUG("Cannot activate freespace region %s.\n",
			  region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_activate
 *
 * Set up DM targets and call the DM service to activate this region.
 **/
static int lvm2_activate(storage_object_t *region)
{
	dm_target_t *target_list;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Activating region %s.\n", region->name);

	rc = rename_region(region);
	if (!rc) {
		target_list = build_target_list(region);
		if (target_list) {
			rc = EngFncs->dm_activate(region, target_list);
			EngFncs->dm_deallocate_targets(target_list);
		} else {
			rc = ENOMEM;
		}
	}

	if (rc) {
		LOG_ERROR("Error activating region %s\n", region->name);
	} else {
		region->flags &= ~SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_can_deactivate
 *
 * All data regions can always be deactivated.
 **/
static int lvm2_can_deactivate(storage_object_t *region)
{
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Checking if region %s can be deactivated.\n", region->name);

	if (region->data_type != DATA_TYPE) {
		LOG_DEBUG("Cannot deactivate freespace region %s.\n",
			  region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_deactivate
 *
 * Call the DM service to deactivate this region.
 **/
static int lvm2_deactivate(storage_object_t *region)
{
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Deactivating region %s.\n", region->name);

	switch_region_name(region);
	rc = EngFncs->dm_deactivate(region);
	switch_region_name(region);

	if (!rc) {
		region->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
		remove_old_region_name(region);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_get_option_count
 *
 * Get the total number of supported options for the specified task. Return -1
 * if the task is not supported.
 **/
static int lvm2_get_option_count(task_context_t *context)
{
	int count;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_Create:
		count = LVM2_OPTION_CREATE_REGION_COUNT;
		break;

	case EVMS_Task_Expand:
		count = LVM2_OPTION_EXPAND_REGION_COUNT;
		break;

	case EVMS_Task_Shrink:
		count = LVM2_OPTION_SHRINK_REGION_COUNT;
		break;

	case EVMS_Task_Create_Container:
		count =	LVM2_OPTION_CREATE_CONTAINER_COUNT;
		break;

	case EVMS_Task_Set_Info:
		count = LVM2_OPTION_SET_REGION_INFO_COUNT;
		break;

	case EVMS_Task_Set_Container_Info:
		count = LVM2_OPTION_SET_CONTAINER_INFO_COUNT;
		break;

	case LVM2_FUNCTION_SPLIT_MAPPING:
		count = LVM2_OPTION_SPLIT_MAPPING_COUNT;
		break;

	case LVM2_FUNCTION_MERGE_MAPPINGS:
		count = 0;
		break;

	case LVM2_FUNCTION_MOVE_MAPPING:
		count = LVM2_OPTION_MOVE_MAPPING_COUNT;
		break;

	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}

/**
 * lvm2_init_task
 *
 * Initialize the task context with the appropriate information for that task.
 * This will include initializing the option-descriptor array and setting up a
 * list of acceptable objects.
 **/
static int lvm2_init_task(task_context_t *context)
{
	int rc;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_Create:
		rc = create_region_init_task(context);
		break;

	case EVMS_Task_Expand:
		if (context->object) {
			rc = expand_region_init_task(context);
		} else if (context->container) {
			rc = expand_container_init_task(context);
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Shrink:
		if (context->object) {
			rc = shrink_region_init_task(context);
		} else if (context->container) {
			rc = shrink_container_init_task(context);
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Create_Container:
		rc = create_container_init_task(context);
		break;

	case EVMS_Task_Set_Info:
		rc = set_region_info_init_task(context);
		break;

	case EVMS_Task_Set_Container_Info:
		rc = set_container_info_init_task(context);
		break;

	case LVM2_FUNCTION_SPLIT_MAPPING:
		rc = split_region_mapping_init_task(context);
		break;

	case LVM2_FUNCTION_MERGE_MAPPINGS:
		rc = merge_region_mappings_init_task(context);
		break;

	case LVM2_FUNCTION_MOVE_MAPPING:
		rc = move_region_mapping_init_task(context);
		break;

	default:
		rc = ENOSYS;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_set_objects
 *
 * Validate the objects in the selected-objects list in the task context. Make
 * any necessary modifications to the acceptable-objects list and/or the
 * option-descriptor based on these selected-objects.
 **/
static int lvm2_set_objects(task_context_t *context,
			    list_anchor_t declined_objects,	 /* of type declined_handle_t */
			    task_effect_t *effect)
{
	int rc;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_Create:
		rc = create_region_set_objects(context, effect);
		break;

	case EVMS_Task_Expand:
		if (context->object) {
			/* No selected objects when expanding regions. */
			rc = 0;
		} else if (context->container) {
			rc = expand_container_set_objects(context, effect);
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Shrink:
		if (context->object) {
			/* No selected objects when shrinking regions. */
			rc = 0;
		} else if (context->container) {
			rc = shrink_container_set_objects(context, effect);
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Create_Container:
		rc = create_container_set_objects(context, effect);
		break;

	case EVMS_Task_Set_Info:
	case EVMS_Task_Set_Container_Info:
	case LVM2_FUNCTION_SPLIT_MAPPING:
	case LVM2_FUNCTION_MERGE_MAPPINGS:
	case LVM2_FUNCTION_MOVE_MAPPING:
		/* No selected objects for set-info on regions or containers,
		 * or for splitting, merging or moving region-mappings.
		 */
		rc = 0;
		break;

	default:
		rc = ENOSYS;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_set_option
 *
 * Examine the specified value, and determine if it is valid for the task and
 * option_descriptor index. If it is acceptable, set that value in the
 * appropriate entry in the option_descriptor. The value may be adjusted if
 * necessary/allowed. If so, set the effect return value accordingly.
 **/
static int lvm2_set_option(task_context_t *context,
			   u_int32_t index,
			   value_t *value,
			   task_effect_t *effect)
{
	int rc;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_Create:
		rc = create_region_set_option(context, index, value, effect);
		break;

	case EVMS_Task_Expand:
		if (context->object) {
			rc = expand_region_set_option(context, index,
						      value, effect);
		} else if (context->container) {
			/* No options for expanding containers. */
			rc = 0;
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Shrink:
		if (context->object) {
			rc = shrink_region_set_option(context, index,
						      value, effect);
		} else if (context->container) {
			/* No options for shrinking containers. */
			rc = 0;
		} else {
			rc = EINVAL;
		}
		break;

	case EVMS_Task_Create_Container:
		rc = create_container_set_option(context, index, value, effect);
		break;

	case EVMS_Task_Set_Info:
		rc = set_region_info_set_option(context, index, value, effect);
		break;

	case EVMS_Task_Set_Container_Info:
		rc = set_container_info_set_option(context, index,
						   value, effect);
		break;

	case LVM2_FUNCTION_SPLIT_MAPPING:
		rc = split_region_mapping_set_option(context, index,
						     value, effect);
		break;

	case LVM2_FUNCTION_MERGE_MAPPINGS:
		/* No options for merging region-mappings. */
		rc = 0;
		break;

	case LVM2_FUNCTION_MOVE_MAPPING:
		rc = move_region_mapping_set_option(context, index,
						    value, effect);
		break;

	default:
		rc = ENOSYS;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_get_info
 *
 * Return LVM2-specific information about the specified region. If the
 * name field is set, only return the "extra" information pertaining
 * to that name.
 **/
static int lvm2_get_info(storage_object_t *region,
			 char *name,
			 extended_info_array_t **info)
{
	int rc;

	LOG_ENTRY();

	if (!name) {
		rc = get_region_info(region, info);

	} else if (!strncmp(name, "Mappings", strlen("Mappings"))) {
		rc = get_region_mappings_info(region, info);

	} else {
		LOG_ERROR("No support for extra region information about "
			  "\"%s\"\n", name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_set_info
 *
 * Apply the settings of the options to the given region. This API currently
 * supports renaming the region.
 **/
static int lvm2_set_info(storage_object_t *region,
			 option_array_t *options)
{
	storage_container_t *container = region->producing_container;
	char *lv_name;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Setting info for region %s.\n", region->name);

	/* No info to set for freespace regions. */
	if (region->data_type != DATA_TYPE) {
		LOG_ERROR("No information to set for freespace region %s.\n",
			  region->name);
		rc = EINVAL;
		goto out;
	}

	set_region_info_parse_options(region, options, &lv_name);
	rc = set_region_info_validate_options(region, lv_name);
	if (rc) {
		goto out;
	}

	set_new_region_name(region, lv_name);
	container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_get_plugin_info
 *
 * Return information about the LVM2 plugin. There is no "extra"
 * information about LVM2, so "info_name" should always be NULL.
 **/
static int lvm2_get_plugin_info(char *info_name,
				extended_info_array_t **info_array)
{
	extended_info_array_t *info = NULL;
	char buffer[50] = {0};
	int i = 0, rc = 0;
	
	LOG_ENTRY();

	if (info_name) {
		LOG_ERROR("No support for extra plugin information about "
			  "\"%s\"\n", info_name);
		rc = EINVAL;
		goto out;
	}

	/* Get memory for the info array. */
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) * 7);
	if (!info) {
		LOG_CRITICAL("Error allocating info array.\n");
		rc = ENOMEM;
		goto out;
	}

	/* Short Name. */
	info->info[i].name = EngFncs->engine_strdup("ShortName");
	info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->short_name);
	i++;

	/* Long Name. */
	info->info[i].name = EngFncs->engine_strdup("LongName");
	info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->long_name);
	i++;

	/* Plugin Type. */
	info->info[i].name = EngFncs->engine_strdup("Type");
	info->info[i].title = EngFncs->engine_strdup(_("Plugin Type"));
	info->info[i].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(_("Region Manager"));
	i++;

	/* Plugin Version. */
	snprintf(buffer, 50, "%d.%d.%d",
		 my_plugin_record->version.major,
		 my_plugin_record->version.minor,
		 my_plugin_record->version.patchlevel);
	info->info[i].name = EngFncs->engine_strdup("Version");
	info->info[i].title = EngFncs->engine_strdup(_("Plugin Version"));
	info->info[i].desc = EngFncs->engine_strdup(_("This is the version number of the plugin."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(buffer);
	i++;

	/* Required Engine Services Version. */
	snprintf(buffer, 50, "%d.%d.%d",
		 my_plugin_record->required_engine_api_version.major,
		 my_plugin_record->required_engine_api_version.minor,
		 my_plugin_record->required_engine_api_version.patchlevel);
	info->info[i].name = EngFncs->engine_strdup("Required_Engine_Version");
	info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
	info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine services "
						      "that this plug-in requires. It will not "
						      "run on older versions of the Engine services."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(buffer);
	i++;

	/* Required Plug-in API Version. */
	snprintf(buffer, 50, "%d.%d.%d",
		 my_plugin_record->required_plugin_api_version.plugin.major,
		 my_plugin_record->required_plugin_api_version.plugin.minor,
		 my_plugin_record->required_plugin_api_version.plugin.patchlevel);
	info->info[i].name = EngFncs->engine_strdup("Required_Plugin_Version");
	info->info[i].title = EngFncs->engine_strdup(_("Required Plug-in API Version"));
	info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine plug-in API "
						      "that this plug-in requires. It will not run "
						      "on older versions of the Engine plug-in API."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(buffer);
	i++;

	/* Required Container API Version. */
	snprintf(buffer, 50, "%d.%d.%d",
		 my_plugin_record->required_container_api_version.major,
		 my_plugin_record->required_container_api_version.minor,
		 my_plugin_record->required_container_api_version.patchlevel);
	info->info[i].name = EngFncs->engine_strdup("Required_Container_Version");
	info->info[i].title = EngFncs->engine_strdup(_("Required Container API Version"));
	info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine container "
						      "API that this plug-in requires. It will not "
						      "run on older versions of the Engine container API."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(buffer);
	i++;

	info->count = i;
	*info_array = info;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_read
 *
 * Process a read request on this region at the given starting sector for the
 * given number of sectors.
 **/
static int lvm2_read(storage_object_t *region, lsn_t lsn,
		     sector_count_t count, void *buffer)
{
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Read request on region %s, sector %"PRIu64", count %"
		  PRIu64".\n", region->name, lsn, count);

	rc = region_io(region, lsn, count, buffer, 0);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_write
 *
 * Process a write request on this region at the given starting sector for the
 * given number of sectors.
 **/
static int lvm2_write(storage_object_t *region, lsn_t lsn,
		      sector_count_t count, void *buffer)
{
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Write request on region %s, sector %"PRIu64", count %"
		  PRIu64".\n", region->name, lsn, count);

	rc = region_io(region, lsn, count, buffer, 1);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_add_sectors_to_kill_list
 *
 * Process a kill-sectors request on this region at the given starting sector
 * for the given number of sectors.
 **/
static int lvm2_add_sectors_to_kill_list(storage_object_t *region,
					 lsn_t lsn, sector_count_t count)
{
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Kill-sectors on region %s, sector %"PRIu64", count %"
		  PRIu64".\n", region->name, lsn, count);

	rc = region_io(region, lsn, count, NULL, 2);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_get_functions
 *
 * Return an array of plugin functions supported for this LVM2 region.
 **/
static int lvm2_get_functions(storage_object_t *region,
			      function_info_array_t **functions)
{
	function_info_array_t *fia;
	int rc, i = 0;

	LOG_ENTRY();

	/* No LVM2-global functions, and no functions for freespace regions. */
	if (!region || region->data_type != DATA_TYPE) {
		rc = EINVAL;
		goto out;
	}

	/* Allocate the function info array. */
	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) +
				    sizeof(function_info_t) *
				    LVM2_REGION_FUNCTION_COUNT);
	if (!fia) {
		LOG_CRITICAL("Memory error creating function-info array for "
			     "region %s\n", region->name);
		rc = ENOMEM;
		goto out;
	}

	/* Can we split at least one mapping in this region? */
	fia->info[i].function = LVM2_FUNCTION_SPLIT_MAPPING;
	fia->info[i].name = EngFncs->engine_strdup("split_mapping");
	fia->info[i].title = EngFncs->engine_strdup(_("Split Mapping"));
	fia->info[i].verb = EngFncs->engine_strdup(_("Split Mapping"));
	fia->info[i].help = EngFncs->engine_strdup(_("Split one physically contiguous logical-mapping into two mappings."));
	rc = can_split_a_region_mapping(region);
	if (rc) {
		fia->info[i].flags = EVMS_FUNCTION_FLAGS_INACTIVE;
	}
	i++;

	/* Are there any split mappings that can be merged? This should be a
	 * rare occurance, so we'll only include this one if it's possible.
	 */
	rc = can_merge_region_mappings(region);
	if (!rc) {
		fia->info[i].function = LVM2_FUNCTION_MERGE_MAPPINGS;
		fia->info[i].name = EngFncs->engine_strdup("merge_mappings");
		fia->info[i].title = EngFncs->engine_strdup(_("Merge Mappings"));
		fia->info[i].verb = EngFncs->engine_strdup(_("Merge Mappings"));
		fia->info[i].help = EngFncs->engine_strdup(_("Merge all logical mappings that are physically contiguous."));
		i++;
	}

	/* Can we move at least one of the mappings in this region? */
	fia->info[i].function = LVM2_FUNCTION_MOVE_MAPPING;
	fia->info[i].name = EngFncs->engine_strdup("move_mapping");
	fia->info[i].title = EngFncs->engine_strdup(_("Move Mapping"));
	fia->info[i].verb = EngFncs->engine_strdup(_("Move Mapping"));
	fia->info[i].help = EngFncs->engine_strdup(_("Move a logical portion of the region from its current physical location to a different physical location."));
	rc = can_move_a_region_mapping(region);
	if (rc) {
		fia->info[i].flags = EVMS_FUNCTION_FLAGS_INACTIVE;
	}
	i++;

	*functions = fia;
	fia->count = i;
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_function
 *
 * Execute an LVM2 plugin-function targetted at a specific LVM2 region.
 * Available LVM2 functions are:
 * - Split-Mapping
 *   - Splits one region-mapping into two region-mappings at a specified extent.
 * - Merge-Mappings
 *   - Merges all region-mappings that physically contiguous.
 * - Move-Mapping
 *   - Move a region-mapping from its current physical location to a new
 *     physical location.
 **/
static int lvm2_function(storage_object_t *region,
			 task_action_t action,
			 list_anchor_t objects,
			 option_array_t *options)
{
	int rc;

	LOG_ENTRY();

	/* No non-region or freespace-region functions. */
	if (!region || region->data_type != DATA_TYPE) {
		rc = EINVAL;
		goto out;
	}

	switch (action) {

	case LVM2_FUNCTION_SPLIT_MAPPING:
		rc = split_region_mapping(region, options);
		break;

	case LVM2_FUNCTION_MERGE_MAPPINGS:
		rc = merge_region_mappings(region);
		if (!rc) {
			region->producing_container->flags |= SCFLAG_DIRTY;
			region->flags |= SOFLAG_NEEDS_ACTIVATE;
		}
		break;

	case LVM2_FUNCTION_MOVE_MAPPING:
		rc = move_region_mapping(region, options);
		break;

	default:
		LOG_ERROR("Action %d is not allowed for region %s\n",
			  action, region->name);
		rc = ENOSYS;
		break;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_backup_metadata
 *
 * To backup the metadata for a region, we just need to record the region's
 * dependency on it's producing container, since all the metadata is backed
 * up relative to the container.
 **/
static int lvm2_backup_metadata(storage_object_t *region)
{
	int rc = 0;

	LOG_ENTRY();

	if (region->data_type == DATA_TYPE) {
		rc = EngFncs->save_metadata(region->name,
					    region->producing_container->name,
					    0, 0, NULL);
	}

	LOG_EXIT_INT(rc);
	return rc;
}



/**
 * Container routines.
 **/



/**
 * lvm2_can_delete_container
 *
 * A container can be deleted if it's not producing any data regions.
 **/
static int lvm2_can_delete_container(storage_container_t *container)
{
	int count, rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Checking if container %s can be deleted.\n", container->name);

	/* There is always one freespace region. */
	count = EngFncs->list_count(container->objects_produced);
	if (count > 1) {
		LOG_DEBUG("Container %s is producing %u regions. Cannot be "
			  "deleted.\n", container->name, count-1);
		rc = EBUSY;
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_can_expand_container
 *
 * Get the list of available objects and see which ones can be added to this
 * container. If any object can be added, create an expand-point to add to
 * this list. Also check if any PVs can be expanded.
 **/
static int lvm2_can_expand_container(storage_container_t *container,
				     list_anchor_t expand_points)	/* of type expand_object_info_t */
{
	expand_object_info_t *expand_point;
	storage_object_t *object;
	sector_count_t limit = 0;
	list_anchor_t objects;
	list_element_t iter;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Checking if container %s can expand.\n", container->name);

	/* Add up the sizes of the available objects that could be added to
	 * this container. If at least one object can be added, add the
	 * container to the expand_points list.
	 */
	rc = get_available_objects(container, &objects);
	if (!rc) {
		LIST_FOR_EACH(objects, iter, object) {
			rc = can_add_object(object, container);
			if (!rc) {
				limit += object->size;
			}
		}

		if (limit) {
			expand_point = EngFncs->engine_alloc(sizeof(*expand_point));
			if (expand_point) {
				LOG_DEBUG("Adding container %s to the expand-"
					  "points list.\n", container->name);
				expand_point->container = container;
				expand_point->max_expand_size = limit;
				EngFncs->insert_thing(expand_points, expand_point,
						      INSERT_AFTER, NULL);
			}
		}
		EngFncs->destroy_list(objects);
	}

	/* Ask each consumed object if they can expand. If they can,
	 * they will add themselves to the expand_points list.
	 */
	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		object->plugin->functions.plugin->can_expand(object, -1,
							     expand_points);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm2_can_shrink_container
 *
 * If any one of the consumed objects can be removed, then the container can
 * shrink. Create a shrink_object_info_t and add it to the shrink_points list.
 * Also check if any consumed objects can be shrunk.
 **/
static int lvm2_can_shrink_container(storage_container_t *container,
				     list_anchor_t shrink_points)	/* of type shrink_object_info_t */
{
	container_data_t *c_data = container->private_data;
	shrink_object_info_t *shrink_point;
	storage_object_t *object;
	sector_count_t limit = 0;
	list_element_t iter;
	pv_data_t *pv_data;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Checking if container %s can shrink.\n", container->name);

	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		rc = can_remove_object(object);
		if (!rc) {
			pv_data = object->consuming_private_data;
			limit += pv_data->pe_count * c_data->pe_size;
		}
	}

	if (limit) {
		shrink_point = EngFncs->engine_alloc(sizeof(*shrink_point));
		if (shrink_point) {
			shrink_point->container = container;
			shrink_point->max_shrink_size = limit;
			EngFncs->insert_thing(shrink_points, shrink_point,
					      INSERT_AFTER, NULL);
		}
	}

	/* Determine which (if any) consumed objects can be shrunk. If they
	 * can, ask the object's plugin if they will allow a shrink. If they
	 * can, they will add themselves to the shrink_points list.
	 */
	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		rc = can_shrink_object(object, &limit);
		if (!rc) {
			object->plugin->functions.plugin->can_shrink(object, limit,
								     shrink_points);
		}
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm2_can_expand_container_by
 *
 * When expanding a PV, LVM2 doesn't care how much the PV expands by.
 **/
static int lvm2_can_expand_container_by(storage_container_t *container,
					storage_object_t *consumed_object,
					sector_count_t *delta_size)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm2_can_shrink_container_by
 *
 * Verify that we can allow this object to be shrunk by the specified amount.
 * If we can shrink, but not by the given delta-size, reduce the delta-size
 * appropriately.
 **/
static int lvm2_can_shrink_container_by(storage_container_t *container,
					storage_object_t *consumed_object,
					sector_count_t *delta_size)
{
	u_int64_t shrink_limit;
	int rc;

	LOG_ENTRY();

	/* Make sure the consumed-object is really a PV. */
	if (consumed_object->consuming_container != container) {
		LOG_ERROR("Attempt to shrink object %s which isn't a PV in "
			  "container %s.\n", consumed_object->name,
			  container->name);
		rc = EINVAL;
		goto out;
	}

	rc = can_shrink_object(consumed_object, &shrink_limit);
	if (rc) {
		LOG_ERROR("Cannot shrink object %s.\n", consumed_object->name);
		goto out;
	}

	if (shrink_limit < *delta_size) {
		LOG_WARNING("Object %s requested to be shrunk by %"PRIu64
			    " sectors. LVM2 can only allow shrinking by %"PRIu64
			    " sectors.\n", consumed_object->name,
			    *delta_size, shrink_limit);
		*delta_size = shrink_limit;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_create_container
 *
 * Create a new container with the specified list of objects and options.
 **/
static int lvm2_create_container(list_anchor_t objects,
				 option_array_t *options,
				 storage_container_t **new_container)
{
	storage_container_t *container = NULL;
	storage_object_t *object;
	list_element_t iter;
	pv_data_t *pv_data;
	u_int64_t extent_size;
	u_int32_t pv_index = 0;
	char *vg_name;
	int rc;

	LOG_ENTRY();

	/* Parse and validate the user's options. */
	create_container_parse_options(options, &extent_size, &vg_name);

	rc = create_container_validate_options(&extent_size, vg_name, objects);
	if (rc) {
		goto out;
	}

	/* Create a new container. Get the first object from the list so we
	 * have a pointer to the underlying CSM disk-group.
	 */
	object = EngFncs->first_thing(objects, NULL);
	container = create_new_container(extent_size, vg_name,
					 object->disk_group);
	if (!container) {
		rc = ENOMEM;
		goto out;
	}

	/* Create new PVs for each object. */
	LIST_FOR_EACH(objects, iter, object) {
		rc = create_new_pv(object, container);
		if (rc) {
			goto out;
		}
	}

	/* Add each new PV to the new container. */
	LIST_FOR_EACH(objects, iter, object) {
		/* Still need to assign PV-indexes to each PV. */
		pv_data = object->consuming_private_data;
		pv_data->pv_index = pv_index++;

		add_object_to_container(object, container);

		/* Grab the disk-group from the first object that has a
		 * disk-group. The engine ensures that all objects are
		 * from the same disk-group.
		 */
		if (!container->disk_group && object->disk_group) {
			container->disk_group = object->disk_group;
		}
	}

	/* Update the freespace mappings for the new container. */
	rc = create_freespace_mappings(container);
	if (rc) {
		/* FIXME: Any cleanup possible? */
		goto out;
	}

	container->flags |= SCFLAG_DIRTY;
	*new_container = container;
	LOG_DEBUG("Created new container %s.\n", container->name);

out:
	if (rc && container) {
		/* Error cleanup. */
		LIST_FOR_EACH(objects, iter, object) {
			if (object->consuming_private_data) {
				remove_object_from_container(object, container);
				deallocate_pv_data(object);
				EngFncs->delete_all_elements(object->parent_objects);
			}
		}
		deallocate_container(container);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_expand_container
 *
 * Expand the specified container. This can take place in one of two ways.
 * 1. If expand_object is set, then one of the container's consumed objects
 *    will be expanded. Send the expand command down to the specified
 *    consumed_object. If the command succeeds, adjust the PV data for that
 *    object as necessary to expand the container.
 * 2. If expand_object is not set, then the container will expand by adding
 *    one or more objects, which are specified in the input_objects list.
 *    Add each object using the specified options.
 **/
static int lvm2_expand_container(storage_container_t *container,
				 storage_object_t *consumed_object,
				 storage_object_t *expand_object,
				 list_anchor_t input_objects,
				 option_array_t *options)
{
	storage_object_t *object;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Expanding container %s.\n", container->name);

	if (expand_object) {
		/* One of the consumed objects will be expanding. */
		rc = expand_object_in_container(container, consumed_object,
						expand_object,
						input_objects, options);
	} else {
		/* Add this list of new objects to the container. */
		LIST_FOR_EACH(input_objects, iter, object) {
			rc = add_object(object, container, options);
			if (rc) {
				LOG_ERROR("Error adding object %s to container "
					  "%s. Aborting remaining expands.\n",
					  object->name, container->name);
				break;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_shrink_container
 *
 * Shrink the specified container. This can take place in one of two ways.
 * 1. If shrink_object is set, then one of the container's consumed objects
 *    will be shrunk. Send the shrink command down to the specified
 *    consumed_object. If the command succeeds, adjust the PV data for that
 *    object as necessary to shrink the container.
 * 2. If shrink_object is not set, then the container will shrink by removing
 *    one or more objects, which are specified in the input_objects list.
 *    Remove each object using the specified options.
 **/
static int lvm2_shrink_container(storage_container_t *container,
				 storage_object_t *consumed_object,
				 storage_object_t *shrink_object,
				 list_anchor_t input_objects,
				 option_array_t *options)
{
	storage_object_t *object;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Shrinking container %s.\n", container->name);

	if (shrink_object) {
		/* One of the consumed objects will be shrinking. */
		rc = shrink_object_in_container(container, consumed_object,
						shrink_object,
						input_objects, options);
	} else {
		/* Remove this list of objects from the container. */
		LIST_FOR_EACH(input_objects, iter, object) {
			rc = remove_object(object);
			if (rc) {
				LOG_ERROR("Error removing object %s from "
					  "container %s. Aborting remaining "
					  "shrinks.\n", object->name,
					  container->name);
				break;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_discard_container
 *
 * Forget about this container. Don't delete it. Just clean up the
 * private data and deallocate the data structures. All regions
 * will be discarded before this is called.
 **/
static int lvm2_discard_container(storage_container_t *container)
{
	storage_object_t *object;
	list_element_t iter1, iter2;

	LOG_ENTRY();
	LOG_DEBUG("Discarding container %s.\n", container->name);

	/* Remove each PV from the container and free its private-data. */
	LIST_FOR_EACH_SAFE(container->objects_consumed, iter1, iter2, object) {
		remove_object_from_container(object, container);
		deallocate_pv_data(object);
		EngFncs->delete_all_elements(object->parent_objects);
	}

	/* Free the memory for the container. */
	deallocate_container(container);

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm2_delete_container
 *
 * Delete this container and its metadata. Release any consumed objects.
 **/
static int lvm2_delete_container(storage_container_t *container,
				 list_anchor_t objects_consumed)
{
	storage_object_t *object;
	list_element_t iter1, iter2;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Deleting container %s.\n", container->name);

	rc = lvm2_can_delete_container(container);
	if (rc) {
		goto out;
	}

	/* Copy this container's PVs to the output list. */
	EngFncs->concatenate_lists(objects_consumed,
				   container->objects_consumed);

	/* Erase the LVM2 metadata from each PV. */
	LIST_FOR_EACH(container->objects_consumed, iter1, object) {
		erase_metadata(object);
	}

	/* Remove each PV from the container and free its private-data. */
	LIST_FOR_EACH_SAFE(container->objects_consumed, iter1, iter2, object) {
		remove_object_from_container(object, container);
		deallocate_pv_data(object);
		EngFncs->delete_all_elements(object->parent_objects);
	}

	/* Free the memory for the container. */
	deallocate_container(container);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_commit_container_changes
 *
 * Write the LVM2 container metadata to the PV objects and/or copy any
 * mappings that are scheduled for moving. Clear the SCFLAG_DIRTY in the
 * container when done.
 **/
static int lvm2_commit_container_changes(storage_container_t *container,
					 commit_phase_t phase)
{
	container_data_t *c_data = container->private_data;
	int rc = 0;

	LOG_ENTRY();

	switch (phase) {
	case FIRST_METADATA_WRITE:
		rc = commit_container_metadata(container, FALSE);
		break;
	case MOVE:
		rc = commit_container_moves(container);
		break;
	default:
		goto out;
	}

	if (!rc && !(c_data->flags & LVM2_CONTAINER_FLAG_MOVE_PENDING)) {
		container->flags &= ~SCFLAG_DIRTY;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_get_container_info
 *
 * Return LVM2-specific information about the specified container.
 * If the name field is set, only return the "extra" information
 * pertaining to that name.
 **/
static int lvm2_get_container_info(storage_container_t *container,
				   char *name,
				   extended_info_array_t **info)
{
	int rc;

	LOG_ENTRY();

	if (!name) {
		rc = get_container_info(container, info);

	} else if (!strncmp(name, "Regions", strlen("Regions"))) {
		rc = get_container_regions_info(container, info);

	} else if (!strncmp(name, "Objects", strlen("Objects"))) {
		rc = get_container_objects_info(container, info);

	} else if (!strncmp(name, "Object", strlen("Object"))) {
		u_int32_t index = atoi(name + strlen("Object"));
		rc = get_container_object_info(container, info, index);

	} else {
		LOG_ERROR("No support for extra region information about "
			  "\"%s\"\n", name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_set_container_info
 *
 * Apply the settings of the options to the given container. This API currently
 * supports renaming the container.
 **/
static int lvm2_set_container_info(storage_container_t *container,
				   option_array_t *options)
{
	char new_container_name[EVMS_NAME_SIZE+1];
	char old_container_name[EVMS_NAME_SIZE+1];
	char *vg_name, lv_name[EVMS_NAME_SIZE+1];
	storage_object_t *region;
	list_element_t iter;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Setting info for container %s.\n", container->name);

	set_container_info_parse_options(container, options, &vg_name);
	rc = set_container_info_validate_options(container, vg_name);
	if (rc) {
		goto out;
	}

	/* Save the old container name to use when renaming the regions. */
	strncpy(old_container_name, container->name, EVMS_NAME_SIZE);

	/* Copy the new name to the container. */
	vg_name_to_container_name(vg_name, new_container_name,
				  container->disk_group);
	EngFncs->register_name(new_container_name);
	EngFncs->unregister_name(old_container_name);
	strncpy(container->name, new_container_name, EVMS_NAME_SIZE);

	/* Set the new names for each of the produced regions. */
	LIST_FOR_EACH(container->objects_produced, iter, region) {
		region_name_to_lv_name(region->name, lv_name,
				       old_container_name);
		set_new_region_name(region, lv_name);
	}

	container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm2_get_container_functions
 *
 * Return an array of plugin functions that you support for this container.
 **/
static int lvm2_get_container_functions(storage_container_t *container,
					function_info_array_t **actions)
{
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * lvm2_container_function
 *
 * Execute the plug-in function on the container.
 **/
static int lvm2_container_function(storage_container_t *container,
				   task_action_t action,
				   list_anchor_t objects,
				   option_array_t *options)
{
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * lvm2_backup_container_metadata
 *
 * Write the metadata for this container to the Engine's metadata-backup
 * database.
 **/
static int lvm2_backup_container_metadata(storage_container_t *container)
{
	int rc;

	LOG_ENTRY();

	rc = commit_container_metadata(container, TRUE);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * Table of standard plugin APIs for the LVM2 plugin.
 **/
static plugin_functions_t lvm2_functions = {
	.setup_evms_plugin		= lvm2_setup_evms_plugin,
	.cleanup_evms_plugin		= lvm2_cleanup_evms_plugin,
	.can_delete			= lvm2_can_delete,
	.can_expand			= lvm2_can_expand,
	.can_shrink			= lvm2_can_shrink,
	.discover			= lvm2_discover,
	.create				= lvm2_create,
	.discard			= lvm2_discard,
	.delete				= lvm2_delete,
	.expand				= lvm2_expand,
	.shrink				= lvm2_shrink,
	.commit_changes			= lvm2_commit_changes,
	.can_activate			= lvm2_can_activate,
	.activate			= lvm2_activate,
	.can_deactivate			= lvm2_can_deactivate,
	.deactivate			= lvm2_deactivate,
	.get_option_count		= lvm2_get_option_count,
	.init_task			= lvm2_init_task,
	.set_objects			= lvm2_set_objects,
	.set_option			= lvm2_set_option,
	.get_info			= lvm2_get_info,
	.set_info			= lvm2_set_info,
	.get_plugin_info		= lvm2_get_plugin_info,
	.read				= lvm2_read,
	.write				= lvm2_write,
	.add_sectors_to_kill_list	= lvm2_add_sectors_to_kill_list,
	.get_plugin_functions		= lvm2_get_functions,
	.plugin_function		= lvm2_function,
	.backup_metadata		= lvm2_backup_metadata,
};

/**
 * Table of container APIs for the LVM2 plugin.
 **/
static container_functions_t lvm2_container_functions = {
	.can_delete_container		= lvm2_can_delete_container,
	.can_expand_container		= lvm2_can_expand_container,
	.can_shrink_container		= lvm2_can_shrink_container,
	.can_expand_container_by	= lvm2_can_expand_container_by,
	.can_shrink_container_by	= lvm2_can_shrink_container_by,
	.create_container		= lvm2_create_container,
	.expand_container		= lvm2_expand_container,
	.shrink_container		= lvm2_shrink_container,
	.discard_container		= lvm2_discard_container,
	.delete_container		= lvm2_delete_container,
	.commit_container_changes	= lvm2_commit_container_changes,
	.get_container_info		= lvm2_get_container_info,
	.set_container_info		= lvm2_set_container_info,
	.get_plugin_functions		= lvm2_get_container_functions,
	.plugin_function		= lvm2_container_function,
	.backup_container_metadata	= lvm2_backup_container_metadata,
};

/**
 * Plugin record for the LVM2 plugin.
 **/
plugin_record_t lvm2_plugin = {
	.id = SetPluginID(EVMS_OEM_IBM,
			  EVMS_REGION_MANAGER,
			  10),
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.plugin	= {
			.major		= 13,
			.minor		= 1,
			.patchlevel	= 0
		}
	},
	.required_container_api_version = {
		.major		= 10,
		.minor		= 1,
		.patchlevel	= 0
	},
	.short_name = "LVM2",
	.long_name = "LVM2 Region Manager",
	.oem_name = "IBM",
	.functions = {
		.plugin = &lvm2_functions
	},
	.container_functions = &lvm2_container_functions
};

plugin_record_t *evms_plugin_records[] = {
	&lvm2_plugin,
	NULL
};

