/*----------------------------------------------------------------------
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: actions.c
 *---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 * Change History:
 *
 * 12/2001  John Stiles   Initial version.
 *
 *---------------------------------------------------------------------*/


/* Identify this file. */
#define ACTIONS_C    1

/*----------------------------------------------------------------------
 * Necessary include files
 *---------------------------------------------------------------------*/
#include <sys/types.h>
#include <string.h>                    /* strcat, strcpy                   */
#include <ctype.h>                     /* toupper                          */
#include <stdio.h>                     /* printf                           */
#include <stdlib.h>                    /* malloc, free                     */

#include "evmsn.h"

/*----------------------------------------------------------------------
 * Global Variables
 *---------------------------------------------------------------------*/

static char work_buffer[] = "                                                     ";


/*----------------------------------------------------------------------
 * There are No Private Constants
 * There are No Private Type Definitions
 * There are No Private Global Variables.
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * Local Functions
 *---------------------------------------------------------------------*/

int where_to_expand( object_handle_t object, object_handle_t *resize_object )
{
  int i, j, rc = 0, keyval = 0, start_index = 0, select_index = 0, item_count,
      screen_cols, screen_lines = CALC_SCREEN3_LINES;
  handle_object_info_t *object_info;
  expand_handle_array_t *resize_points;
  char *holdbuf, *col1;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT;
  *resize_object = 0;

  holdbuf = malloc( screen_cols + 1 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  rc = evms_get_expand_points( object, &resize_points );
  if ( rc == 0 ) {
    log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_expandpt, resize_points->count, info_msg_entries1 );
    // if ( resize_points->count == 1 && object == resize_points->expand_point[0].object ) {
    //   *resize_object = resize_points->expand_point[0].object;
    //     draw_status_line( "objects match, press any key to continue" );
    //     get_inputc();
    // }
    // else
    if ( resize_points->count ) {

      draw_box3( title_expandpt );

      BLANK_BUFFER( holdbuf, screen_cols )
      for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
        print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

      rc = evms_get_info( object, &object_info );
      if ( rc == 0 ) {
        sprintf( holdbuf, "%s:  %s", expand_ding, object_info->info.volume.name );
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );
        evms_free( object_info );
      }
      else {
        log_error( "%s: %s OBJECT %s %i.\n", __FUNCTION__, err_msg_get_info, err_msg_rc, rc );
      }

      drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT, screen_cols - 2 );

      if ( resize_points->count ) {

        do {
          BLANK_BUFFER( holdbuf, screen_cols - 2 )
          memcpy( holdbuf, expand_ptname_col, strlen( expand_ptname_col ));
          col1 = holdbuf + X_EXPPT_SIZE;
          memcpy( col1, expand_ptsize_col, strlen( expand_ptsize_col ));
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, COLOR_TITLE, holdbuf );

          if ( start_index + screen_lines > resize_points->count )
            item_count = resize_points->count - start_index + YPOS_BOX3TEXT + 1;
          else
            item_count = screen_lines - 1 + YPOS_BOX3TEXT + 1;

          for ( i = YPOS_BOX3TEXT + 1, j = start_index; i < item_count; i++, j++ ) {

            rc = evms_get_info( resize_points->expand_point[j].object, &object_info );
            if ( rc == 0 ) {
              BLANK_BUFFER( holdbuf, screen_cols - 2 )
              memcpy( holdbuf, object_info->info.object.name, strlen( object_info->info.object.name ));
              draw_sector_size( resize_points->expand_point[j].max_expand_size, work_buffer );
              memcpy( col1, work_buffer, strlen( work_buffer ));
              if ( j == select_index )
                print_string( XPOS_BOX3TEXT, i, COLOR_BAR, holdbuf );
              else
                print_string( XPOS_BOX3TEXT, i, COLOR_TITLE, holdbuf );

              log_debug( "%s: entry %i is: %s \n", __FUNCTION__, j, object_info->info.object.name );
              evms_free( object_info );
            }
            else {
              log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, j, err_msg_rc, rc );
            }
          }
          draw_status_line( expand_expandpt );

          keyval = key_scroller( &start_index, &select_index, resize_points->count, screen_lines );

          switch ( keyval ) {

            case KEY_ESC :
              rc = ESC_PRESSED;
              break;

            case KEY_F1  :
            case '1'     :
              draw_help_panel( 5 );
              keyval = KEY_CURSORACT;
              break;

            case KEY_ENTER :
              *resize_object = resize_points->expand_point[select_index].object;
              break;

            default:
              keyval = KEY_CURSORACT;
              break;
          }

        }  while ( keyval == KEY_CURSORACT );
      }
      // else no expand points
      else {
        rc = NO_OBJECTS_FOUND;
      }
    }

    evms_free( resize_points );
  }
  else {
    log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_get_expandpt, rc );
  }

  free( holdbuf );
  return rc;
}


int where_to_shrink( object_handle_t object, object_handle_t *resize_object )
{
  int i, j, rc = 0, keyval = 0, start_index = 0, select_index = 0, item_count,
      screen_cols, screen_lines = CALC_SCREEN3_LINES;
  handle_object_info_t *object_info;
  shrink_handle_array_t *resize_points;
  char *holdbuf, *col1;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT;
  *resize_object = 0;

  holdbuf = malloc( screen_cols + 1 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  rc = evms_get_shrink_points( object, &resize_points );
  if ( rc == 0 ) {
    log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_shrinkpt, resize_points->count, info_msg_entries1 );
    // if ( resize_points->count == 1 && object == resize_points->shrink_point[0].object ) {
    //   *resize_object = resize_points->shrink_point[0].object;
    //     draw_status_line( "objects match, press any key to continue" );
    //     get_inputc();
    // }
    // else
    if ( resize_points->count ) {

      draw_box3( title_shrinkpt );

      BLANK_BUFFER( holdbuf, screen_cols )
      for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
        print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

      rc = evms_get_info( object, &object_info );
      if ( rc == 0 ) {
        sprintf( holdbuf, "%s:  %s", shrink_ing, object_info->info.volume.name );
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );
        evms_free( object_info );
      }
      else {
        log_error( "%s: %s OBJECT %s %i.\n", __FUNCTION__, err_msg_get_info, err_msg_rc, rc );
      }

      drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT, screen_cols - 2 );

      if ( resize_points->count ) {

        do {
          BLANK_BUFFER( holdbuf, screen_cols - 2 )
          memcpy( holdbuf, shrink_ptname_col, strlen( shrink_ptname_col ));
          col1 = holdbuf + X_EXPPT_SIZE;
          memcpy( col1, shrink_ptsize_col, strlen( shrink_ptsize_col ));
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, COLOR_TITLE, holdbuf );

          if ( start_index + screen_lines > resize_points->count )
            item_count = resize_points->count - start_index + YPOS_BOX3TEXT + 1;
          else
            item_count = screen_lines - 1 + YPOS_BOX3TEXT + 1;

          for ( i = YPOS_BOX3TEXT + 1, j = start_index; i < item_count; i++, j++ ) {

            rc = evms_get_info( resize_points->shrink_point[j].object, &object_info );
            if ( rc == 0 ) {
              BLANK_BUFFER( holdbuf, screen_cols - 2 )
              memcpy( holdbuf, object_info->info.object.name, strlen( object_info->info.object.name ));
              draw_sector_size( resize_points->shrink_point[j].max_shrink_size, work_buffer );
              memcpy( col1, work_buffer, strlen( work_buffer ));
              if ( j == select_index )
                print_string( XPOS_BOX3TEXT, i, COLOR_BAR, holdbuf );
              else
                print_string( XPOS_BOX3TEXT, i, COLOR_TITLE, holdbuf );

              log_debug( "%s: entry %i is: %s \n", __FUNCTION__, j, object_info->info.object.name );
              evms_free( object_info );
            }
            else {
              log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, j, err_msg_rc, rc );
            }
          }
          draw_status_line( shrink_shrinkpt );

          keyval = key_scroller( &start_index, &select_index, resize_points->count, screen_lines );

          switch ( keyval ) {

            case KEY_ESC :
              rc = ESC_PRESSED;
              break;

            case KEY_F1  :
            case '1'     :
              draw_help_panel( 5 );
              keyval = KEY_CURSORACT;
              break;

            case KEY_ENTER :
              *resize_object = resize_points->shrink_point[select_index].object;
              break;

            default:
              keyval = KEY_CURSORACT;
              break;
          }

        }  while ( keyval == KEY_CURSORACT );
      }
      // else no shrink points
      else {
        rc = NO_OBJECTS_FOUND;
      }
    }

    evms_free( resize_points );
  }
  else {
    log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_get_shrinkpt, rc );
  }

  free( holdbuf );
  return rc;
}

/*
 * This routine takes the address of two handle arrays, allocates space
 * to copy the handles in both into one array and returns the new array.
 * It also frees the first array which it is replacing.
 */
int merge_handle_array (handle_array_t **target, handle_array_t *source)
{
    int rc=0;
    int source_size;
    int target_size;
    int total_size;
    handle_array_t *tmp;
    
    if (target && *target && source)
    {    
        source_size = sizeof (object_handle_t) * source->count;
        target_size = sizeof (object_handle_t) * (*target)->count;
        total_size  = sizeof (handle_array_t) + source_size + target_size;
        
        tmp = malloc (total_size);
        
        if (tmp) 
        {        
            tmp->count = (*target)->count + source->count;
            
            memcpy (&(tmp->handle[0]), &((*target)->handle[0]), target_size);
            memcpy (&(tmp->handle[(*target)->count]), &(source->handle[0]), source_size);
            
            evms_free (*target);
            
            *target = tmp;
        } 
        else 
        {
            rc = ENOMEM;
        }
    }
    else
    {
        rc = EINVAL;
    }
        
    return rc;
}

/*
 * This routine returns a single, merged handle array of all
 * the different feature type plugins.
 */
int get_all_feature_plugins_list (handle_array_t **plugins)
{
    int rc=0;
    
    rc = evms_get_plugin_list (EVMS_FEATURE, 0, plugins);
    
    if (rc == 0) 
    {
        handle_array_t *assoc_features;
        
        rc = evms_get_plugin_list (EVMS_ASSOCIATIVE_FEATURE, 0, &assoc_features);
        
        if (rc == 0)
        {    
            rc = merge_handle_array (plugins, assoc_features);
            
            if (rc == 0)
                evms_free (assoc_features);
            else
                evms_free (*plugins);
        }    
    }
    
    return rc;
}

int get_plugin_selection( handle_array_t *plugin_objects, plugin_handle_t *selected_plugin, char *panel_title )
{
    int i, j, rc = 0, keyval = 0, start_index = 0, select_index = 0, item_count,
      screen_cols, screen_lines = CALC_SCREEN3_LINES;
    handle_object_info_t *object_info;
    char *holdbuf, *col1;
    
    screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT;
    *selected_plugin = 0;
    
    holdbuf = malloc( screen_cols + 1 );
    if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
    }

    log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_pluglist, plugin_objects->count, info_msg_entries1 );
    
    if ( plugin_objects->count ) {
    
      draw_box3( panel_title );
    
      if ( plugin_objects->count ) {
    
        do {
          BLANK_BUFFER( holdbuf, screen_cols )
          for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
            print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );
    
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 2, COLOR_TITLE, title_sel_plugin );
          drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT, screen_cols - 2 );
    
          BLANK_BUFFER( holdbuf, screen_cols - 2 )
          memcpy( holdbuf, plug_name_col, strlen( plug_name_col ));
          col1 = holdbuf + X_PLUG_DESC;
          memcpy( col1, plug_desc_col, strlen( plug_desc_col ));
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, COLOR_TITLE, holdbuf );
    
          if ( start_index + screen_lines - 2 > plugin_objects->count )
            item_count = plugin_objects->count - start_index + YPOS_BOX3TEXT + 1;
          else
            item_count = screen_lines - 1 + YPOS_BOX3TEXT;
    
          for ( i = YPOS_BOX3TEXT + 1, j = start_index; i < item_count; i++, j++ ) {
    
            rc = evms_get_info( plugin_objects->handle[j], &object_info );
            if ( rc == 0 ) {
              BLANK_BUFFER( holdbuf, screen_cols - 2 )
              memcpy( holdbuf, object_info->info.plugin.short_name, strlen( object_info->info.plugin.short_name ));
              memcpy( col1, object_info->info.plugin.long_name, strlen( object_info->info.plugin.long_name ));
              if ( j == select_index )
                print_string( XPOS_BOX3TEXT, i, COLOR_BAR, holdbuf );
              else
                print_string( XPOS_BOX3TEXT, i, COLOR_TITLE, holdbuf );
    
              log_debug( "%s: entry %i is: %s \n", __FUNCTION__, i, object_info->info.plugin.short_name );
              evms_free( object_info );
            }
            else {
              log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, j, err_msg_rc, rc );
            }
          }
          draw_status_line( plugin_select1 );
    
          keyval = key_scroller( &start_index, &select_index, plugin_objects->count, screen_lines - 3 );
    
          switch ( keyval ) {
    
            case KEY_ESC :
              rc = ESC_PRESSED;
              break;
    
            case KEY_F1  :
            case '1'     :
              draw_help_panel( 5 );
              keyval = KEY_CURSORACT;
              break;
    
            case KEY_ENTER :
              *selected_plugin = plugin_objects->handle[select_index];
              break;
    
            default:
              keyval = KEY_CURSORACT;
              break;
          }
    
        }  while ( keyval == KEY_CURSORACT );
      }
      // else no plugins found
      else {
        rc = NO_OBJECTS_FOUND;
      }
    }
    
    evms_free( plugin_objects );

    free( holdbuf );
    return rc;
}


int select_plugin( evms_plugin_code_t type, int need_container, plugin_handle_t *selected_plugin, char *panel_title )
{
  int rc = 0;
  handle_array_t *plugin_objects = NULL;

  if ( need_container )
    rc = evms_get_plugin_list( type, SUPPORTS_CONTAINERS, &plugin_objects );
  else
  {
    /*
     * Hack. If this is a request to get feature plugins, then get all types
     * of plugins, i.e. EVMS_FEATURE and EVMS_ASSOCIATIVE_FEATURE, and merge
     * the lists into one.
     */
    if (type == EVMS_FEATURE)
        rc = get_all_feature_plugins_list( &plugin_objects );
    else
        rc = evms_get_plugin_list( type, 0, &plugin_objects );
  }    

  if ( rc == 0 ) {
    rc = get_plugin_selection( plugin_objects, selected_plugin, panel_title );
  }
  else {
    log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_get_pluglist, rc );
  }
      
  return rc;
}


int set_objects_menu( task_handle_t task_context, char *set_obj_msg )
{
  int i, j, k, rc = 0, keyval = 0, start_index = 0, select_index = 0,
      item_count, screen_cols, screen_lines = CALC_SCREEN3_LINES;
  handle_object_info_t *object_info;
  char *holdbuf, *col1, *col2;
  u_int32_t min, max;
  handle_array_t *acceptable_points, *selected_points, *candidate_points;
  declined_handle_array_t *declined_points = NULL;
  task_effect_t  effects;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT;

  holdbuf = malloc( screen_cols + 1 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  draw_box3( set_obj_msg );

  do {
    BLANK_BUFFER( holdbuf, screen_cols )
    for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
      print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

    rc = evms_get_acceptable_objects( task_context, &acceptable_points );
    if ( rc == 0 ) {
      log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_acceptable, acceptable_points->count, info_msg_entries1 );

      rc = evms_get_selected_object_limits( task_context, &min, &max );
      if ( rc != 0 ) {
        log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_get_selobjlimits, rc );
      }
      else  if ( acceptable_points->count ) {

        log_debug( "%s: %s %i , %i\n", __FUNCTION__, info_msg_get_selobjlimits, min, max );

        candidate_points = malloc( sizeof( handle_array_t ) + max * sizeof( object_handle_t ) );
        if ( !candidate_points ) {
          log_error( "%s: could not malloc candidate point list\n", __FUNCTION__ );
          evms_free( acceptable_points );
          free( holdbuf );
          return ENOMEM;
        }
        candidate_points->count = 0;
        for ( k = 0; k < max; k++ )
          candidate_points->handle[k] = 0;

        rc = evms_get_selected_objects( task_context, &selected_points );
        if ( rc == 0 ) {
          log_debug( "%s: %s %i %s.\n", __FUNCTION__, info_msg_get_selobjects, selected_points->count, info_msg_entries1 );
          for ( k = 0; k < selected_points->count; k++ )
            candidate_points->handle[k] = selected_points->handle[k];

          candidate_points->count = selected_points->count;
          evms_free( selected_points );
        }
        else {
          log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_get_selobjects, rc );
        }

        BLANK_BUFFER( holdbuf, screen_cols - 2 )
        if ( max == 1 )
          memcpy( holdbuf, setobj_one_select, strlen( setobj_one_select ));
        else {
          if ( min == max )
            sprintf( work_buffer, setobj_many_select, max );
          else {
            if ( max < acceptable_points->count )
              sprintf( work_buffer, setobj_range_select, min, max );
            else
              sprintf( work_buffer, setobj_range_select, min, acceptable_points->count );
          }
          memcpy( holdbuf, work_buffer, strlen( work_buffer ));
        }
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );
        BLANK_BUFFER( holdbuf, screen_cols - 2 )
        memcpy( holdbuf, setobj_select_col, strlen( setobj_select_col ));
        col1 = holdbuf + X_OBJSEL_SIZE;
        memcpy( col1, setobj_size_col, strlen( setobj_size_col ));
        col2 = holdbuf + X_OBJSEL_NAME;
        memcpy( col2, setobj_name_col, strlen( setobj_name_col ));
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, COLOR_TITLE, holdbuf );

        drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT, screen_cols - 2 );

        if ( start_index + screen_lines > acceptable_points->count )
          item_count = acceptable_points->count - start_index + YPOS_BOX3TEXT + 1;
        else
          item_count = screen_lines - 1 + YPOS_BOX3TEXT + 1;

        for ( i = YPOS_BOX3TEXT + 1, j = start_index; i < item_count; i++, j++ ) {

          rc = evms_get_info( acceptable_points->handle[j], &object_info );
          if ( rc == 0 ) {
            BLANK_BUFFER( holdbuf, screen_cols - 2 )
            for ( k = 0; k < candidate_points->count; k++ ) {
               if( acceptable_points->handle[j] == candidate_points->handle[k] ) {
                 holdbuf[2] = ITEM_SELECTED_CHAR;
                 break;
               }
            }
            draw_sector_size( object_info->info.object.size, work_buffer );
            memcpy( col1, work_buffer, strlen( work_buffer ));
            memcpy( col2, object_info->info.object.name, strlen( object_info->info.object.name ));
            if ( j == select_index )
              print_string( XPOS_BOX3TEXT, i, COLOR_BAR, holdbuf );
            else
              print_string( XPOS_BOX3TEXT, i, COLOR_TITLE, holdbuf );

            log_debug( "%s: entry %i is: %s \n", __FUNCTION__, j, object_info->info.object.name );
            evms_free( object_info );
          }
          else {
            log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, j, err_msg_rc, rc );
          }
        }
        if ( candidate_points->count >= min )
          draw_status_line( setobj_space_enter );
        else
          draw_status_line( setobj_space_select );

        keyval = key_scroller( &start_index, &select_index, acceptable_points->count, screen_lines );

        switch ( keyval ) {

          case KEY_ESC :
            rc = ESC_PRESSED;
            break;

          case KEY_F1  :
          case '1'     :
            draw_help_panel( 5 );
            keyval = KEY_CURSORACT;
            break;

          case ' ' :
            rc = 0;
            for ( k = 0; k < candidate_points->count; k++ ) {
               if ( rc )    // found, move others back
                 candidate_points->handle[k] = candidate_points->handle[k + 1];
               else  // check for match
               if( acceptable_points->handle[select_index] == candidate_points->handle[k] ) {
                 rc = 1;
                 candidate_points->handle[k] = candidate_points->handle[k + 1];
               }
            }
            if ( rc ) {  // found
              candidate_points->handle[candidate_points->count] = 0;
              if ( candidate_points->count > 0 )
                candidate_points->count--;
            }
            else {  // not found, add it
              if ( candidate_points->count < max ) {
                candidate_points->handle[candidate_points->count] = acceptable_points->handle[select_index];
                candidate_points->count++;
              }
            }

            if ( candidate_points->count > 0 ) {
              rc = evms_set_selected_objects( task_context, candidate_points,
                                              &declined_points, &effects );
              if ( rc == 0 ) {
                if ( declined_points ) {
                  log_debug( "%s: Count of declined objects is %i\n", __FUNCTION__, declined_points->count );
                  evms_free( declined_points );
                  rc = 1;
                }
                keyval = KEY_CURSORACT;
              }
              else {
                log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_set_selectedobj, rc );
              }
            }
            else
              keyval = KEY_CURSORACT;
            break;

          case KEY_ENTER :
            if ( candidate_points->count >= min )
              rc = 0;
            else
              keyval = KEY_CURSORACT;
            break;

          default:
            keyval = KEY_CURSORACT;
            break;
        }
        free( candidate_points );
      }
      // else no selectable points
      else {
        rc = NO_OBJECTS_FOUND;
        keyval = KEY_ENTER;
      }

      evms_free( acceptable_points );
    }
    else {
      log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_get_acceptable, rc );
      keyval = KEY_ENTER;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return rc;
}


int select_option_menu( task_handle_t task_context, u_int32_t option_index )
{
  int i, j, k, rc = 0, setopt_rc = 99, keyval = 0, start_index = 0, select_index = 0,
      item_count, screen_cols, screen_lines = CALC_SCREEN3_LINES;
  option_descriptor_t *option;
  task_effect_t  effects;
  char *holdbuf, *col1;
  value_t  cur_value;
  int *item_selected = NULL;
  value_list_t *list_values = NULL;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT;

  holdbuf = malloc( screen_cols + 1 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  draw_box3( " Option Selection List " );

  BLANK_BUFFER( holdbuf, screen_cols )
  for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
    print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

  do {

    rc = evms_get_option_descriptor( task_context, option_index, &option );
    if ( rc == 0 ) {

      BLANK_BUFFER( holdbuf, screen_cols - 2 )
      if ( EVMS_OPTION_VALUE_IS_LIST( option->flags )) {
        memcpy( holdbuf, selobj_many_select, strlen( selobj_many_select ));
        item_selected = malloc( option->constraint.list->count * sizeof( int ));
        if ( !item_selected ) {
          log_error( "%s: could not malloc item_selected array\n", __FUNCTION__ );
          evms_free( option );
          free( holdbuf );
          return ENOMEM;
        }
      }
      else
        memcpy( holdbuf, setobj_one_select, strlen( setobj_one_select ));

      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );
      BLANK_BUFFER( holdbuf, screen_cols - 2 )
      memcpy( holdbuf, setobj_select_col, strlen( setobj_select_col ));
      col1 = holdbuf + X_OBJSEL_SIZE;
      memcpy( col1, selobj_name_col, strlen( selobj_name_col ));
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, COLOR_TITLE, holdbuf );

      drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT, screen_cols - 2 );

      if ( start_index + screen_lines - 2 > option->constraint.list->count )
        item_count = option->constraint.list->count - start_index + YPOS_BOX3TEXT + 1;
      else
        item_count = screen_lines - 2 + YPOS_BOX3TEXT + 1;

      if ( EVMS_OPTION_VALUE_IS_LIST( option->flags ))
        log_debug( "%s: Value List item count == %i\n", __FUNCTION__, option->value.list->count );

      for ( i = YPOS_BOX3TEXT + 1, j = start_index; i < item_count; i++, j++ ) {
        BLANK_BUFFER( holdbuf, screen_cols - 2 )
        if ( EVMS_OPTION_VALUE_IS_LIST( option->flags )) {
          item_selected[j] = 0;
          draw_value_type( option->type, EVMS_Format_Normal, &( option->constraint.list->value[j] ), 40, work_buffer );
          log_debug( "%s: Constraint item is:  %s\n", __FUNCTION__, work_buffer );
          for ( k = 0; k < option->value.list->count; k++ ) {
            draw_value_type( option->type, EVMS_Format_Normal, &( option->value.list->value[k] ), 40, work_buffer );
            log_debug( "%s:   Value List compare item is:  %s\n", __FUNCTION__, work_buffer );
            if( !comp_value_type( option->type, &( option->constraint.list->value[j] ),
                &( option->value.list->value[k] ))) {
              holdbuf[2] = ITEM_SELECTED_CHAR;
              item_selected[j] = 1;
              break;
            }
          }
        }
        draw_value_type( option->type, EVMS_Format_Normal, &( option->constraint.list->value[j] ), 40, work_buffer );
        memcpy( col1, work_buffer, strlen( work_buffer ));

        if ( j == select_index )
          print_string( XPOS_BOX3TEXT, i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, i, COLOR_TITLE, holdbuf );
      }
      draw_status_line( setobj_space_select );

      keyval = key_scroller( &start_index, &select_index, option->constraint.list->count, screen_lines - 3 );

      switch ( keyval ) {

        case KEY_ESC :
          break;

        case KEY_F1  :
        case '1'     :
          draw_help_panel( 5 );
          keyval = KEY_CURSORACT;
          break;

        case ' ' :
          if ( EVMS_OPTION_VALUE_IS_LIST( option->flags )) {
            list_values = malloc( sizeof( value_list_t ) + option->constraint.list->count * sizeof( value_t ));
            if ( list_values ) {
              if ( item_selected[select_index] == 1 )
                item_selected[select_index] = 0;
              else
                item_selected[select_index] = 1;

              list_values->count = 0;
              for ( k = 0; k < option->constraint.list->count; k++ ) {
                if ( item_selected[k] == 1 ) {
                  copy_value_type( option->type, &( list_values->value[list_values->count] ),
                                   &( option->constraint.list->value[k] ));
                  (list_values->count)++;
                }
              }
              cur_value.list = list_values;
              setopt_rc = evms_set_option_value( task_context, option_index, &cur_value, &effects );
              for ( k = 0; k < list_values->count; k++ )
                free_value_type( option->type, &( list_values->value[k] ));

              free( list_values );
            }
            else
              log_error( "%s: could not malloc list_values array\n", __FUNCTION__ );

            keyval = KEY_CURSORACT;
          }
          else {
            copy_value_type( option->type, &cur_value, &( option->constraint.list->value[select_index] ));
            setopt_rc = evms_set_option_value( task_context, option_index, &cur_value, &effects );
            free_value_type( option->type, &cur_value );
          }
          break;

        case KEY_ENTER :
          break;

        default:
          keyval = KEY_CURSORACT;
          break;
      }

      if ( EVMS_OPTION_VALUE_IS_LIST( option->flags ))
        free( item_selected );

      evms_free( option );
    }
    else {
      log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_optdesc, option_index, err_msg_rc, rc );
      keyval = 0;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return setopt_rc;
}


int edit_option_menu( task_handle_t task_context, u_int32_t option_index )
{
//  int i, j, k, rc = 0, setopt_rc = 99, keyval = 0, start_index = 0, select_index = 0,
//      item_count, screen_cols, screen_lines = CALC_SCREEN3_LINES, option_count;
  int i, rc = 0, setopt_rc = 99, keyval = 0,
      screen_cols, screen_lines = CALC_SCREEN3_LINES;
  option_descriptor_t *option;
  task_effect_t  effects;
  char *holdbuf, *col1, *col2;
  sector_count_t  sectors;
  size_t  title_size;
  value_t  cur_value;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT;

  holdbuf = malloc( screen_cols + 1 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  draw_box3( " Edit Option " );

  BLANK_BUFFER( holdbuf, screen_cols )
  for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
    print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

  do {

    rc = evms_get_option_descriptor( task_context, option_index, &option );
    if ( rc == 0 ) {
      BLANK_BUFFER( holdbuf, screen_cols - 2 )
      if ( option->title ) {
        title_size = strlen( option->title );
        col1 = holdbuf + title_size + 5;
        memcpy( holdbuf, option->title, title_size );
        holdbuf[title_size] = ':';
      }
      else {
        memcpy( holdbuf, " No Title: ", 11 );
        col1 = holdbuf + 16;
      }

      if ( EVMS_OPTION_VALUE_IS_LIST( option->flags ) ) {
        log_error( "%s: Found list value for single option\n", __FUNCTION__ );
        return 99;
      }

      if ( option->unit == EVMS_Unit_Sectors && ( option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION ) == 0 ) {
        sectors = convert_to_sector_count( option->type, &( option->value ) );
        draw_sector_size( sectors, work_buffer );
        title_size = strlen( work_buffer );
        memcpy( col1, work_buffer, title_size );
      }
      else {
        draw_value_type( option->type, EVMS_Format_Normal, &( option->value ), 15, work_buffer );
        if ( work_buffer[0] != '\0' ) {
          title_size = strlen( work_buffer );
          memcpy( col1, work_buffer, title_size );
          if ( !draw_value_units( option->unit, work_buffer )) {
            col2 = col1 + title_size + 2;
            title_size = strlen( work_buffer );
            memcpy( col2, work_buffer, title_size );
          }
        }
      }
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_TITLE, holdbuf );

      if ( option->tip ) {
        title_size = strlen( option->tip );
        if ( title_size < screen_cols - 4 ) {
          BLANK_BUFFER( holdbuf, screen_cols - 2 )
          memcpy( holdbuf, option->tip, title_size );
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + 4, COLOR_TITLE, holdbuf );
        }
      }

      draw_status_line( editopt_space_select );
      keyval = get_inputc();

      if ( keyval == ' ' ) {
        copy_value_type( option->type, &cur_value, &( option->value ) );

        switch ( option->type ) {

          case EVMS_Type_Boolean :
            if ( cur_value.bool )
              cur_value.bool = 0;
            else
              cur_value.bool = 1;
            break;

          case EVMS_Type_Int8 :
          case EVMS_Type_Unsigned_Int8 :
          case EVMS_Type_Real32 :
          case EVMS_Type_Real64 :
          case EVMS_Type_Int :
          case EVMS_Type_Int16 :
          case EVMS_Type_Int32 :
          case EVMS_Type_Int64 :
          case EVMS_Type_Unsigned_Int :
          case EVMS_Type_Unsigned_Int16 :
          case EVMS_Type_Unsigned_Int32 :
          case EVMS_Type_Unsigned_Int64 :
            holdbuf[0] = ':';
            holdbuf[1] = ':';
            holdbuf[2] = '\0';
            print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + 3, COLOR_TITLE, holdbuf );
            BLANK_BUFFER( holdbuf, screen_cols )
            draw_status_line( editopt_number_enter );
            if ( !number_entry( XPOS_BOX3TEXT + 2, YPOS_BOX3TEXT + 3, 12, holdbuf )) {
              sscanf( holdbuf, "%Ld", &sectors );
              if ( option->unit == EVMS_Unit_Sectors && ( option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION ) == 0 )
                sectors *= 2048;   /* convert MBytes to sectors  */
              convert_from_sector_count( option->type, sectors, &cur_value );
            }
            break;

          case EVMS_Type_String :
            BLANK_BUFFER( holdbuf, screen_cols )
            rc = naming_box( _( "Enter the Name: " ),
                             screen_cols, holdbuf );
            copy_value_string( option->type, &cur_value, holdbuf );
            break;

          default:
            break;
        }
        setopt_rc = evms_set_option_value( task_context, option_index, &cur_value, &effects );

        free_value_type( option->type, &cur_value );
      }

      evms_free( option );
    }
    else {
      log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_optdesc, option_index, err_msg_rc, rc );
      free( holdbuf );
      return setopt_rc;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return setopt_rc;
}


int set_options_menu( task_handle_t task_context, char *set_opt_msg )
{
  int i, j, rc = 0, keyval = 0, start_index = 0, select_index = 0,
      item_count, screen_cols, screen_lines = CALC_SCREEN3_LINES, option_count;
  u_int32_t cur_option;
  option_descriptor_t *option;
  char *holdbuf, *col1, *col2;
  sector_count_t  sectors;
  size_t  title_size;
  int *option_settings;
  // defines for option_settings
  #define  OPT_SET_IS_COMPLETE   ( 1 << 0 )
  #define  OPT_SET_MUST_BE_SET   ( 1 << 1 )
  #define  OPT_SET_IS_INACTIVE   ( 1 << 2 )
  #define  OPT_SET_IS_LIST       ( 1 << 3 )
  #define  OPT_SET_IS_COLLECTION ( 1 << 4 )

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  rc = evms_get_option_count( task_context, &option_count );
  if ( rc == 0 ) {

    log_debug( "%s: Count of options is %i\n", __FUNCTION__, option_count );
    if ( option_count ) {

      option_settings = malloc( option_count * sizeof( int ));
      if ( !option_settings ) {
        log_error( "%s: could not malloc option_settings array\n", __FUNCTION__ );
        free( holdbuf );
        return ENOMEM;
      }
      for ( i = 0; i < option_count; i++ )
        option_settings[i] = 0;

      do {
        draw_box3( set_opt_msg );

        BLANK_BUFFER( holdbuf, screen_cols + 1 )
        for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
          print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

        BLANK_BUFFER( holdbuf, screen_cols - 2 )
        holdbuf[0] = '(';
        holdbuf[1] = SETOPT_REQUIRED_CHAR;
        memcpy( holdbuf + 2, setopt_indicate_req, strlen( setopt_indicate_req ));
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );
        drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, screen_cols - 2 );

        if ( start_index + screen_lines > option_count )
          item_count = option_count - start_index + YPOS_BOX3TEXT;
        else
          item_count = screen_lines - 1 + YPOS_BOX3TEXT;

        for ( i = YPOS_BOX3TEXT, j = start_index; i < item_count; i++, j++ ) {

          rc = evms_get_option_descriptor( task_context, j, &option );
          if ( rc == 0 ) {

            BLANK_BUFFER( holdbuf, screen_cols - 2 )
            if ( EVMS_OPTION_IS_REQUIRED( option->flags ) && EVMS_OPTION_IS_ACTIVE( option->flags )) {
              holdbuf[0] = SETOPT_REQUIRED_CHAR;
              if (( option_settings[j] & OPT_SET_IS_COMPLETE ) == 0 )
                option_settings[j] |= OPT_SET_MUST_BE_SET;
            }
            else
              option_settings[j] &= ~OPT_SET_MUST_BE_SET;

            if ( option->title ) {
              log_debug( "%s: Option %i is called %s\n", __FUNCTION__, i, option->title );
              title_size = strlen( option->title );

              if ( draw_value_units( option->unit, work_buffer ) ||
                   ( option->unit == EVMS_Unit_Sectors && ( option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION ) == 0 )) {
                sprintf( holdbuf + 2, "%s:  ", option->title );
                col1 = holdbuf + title_size + 5;
              }
              else {
                sprintf( holdbuf + 2, "%s (%s):  ", option->title, work_buffer );
                col1 = holdbuf + title_size + 8 + strlen( work_buffer );
              }
            }
            else {
              log_debug( "%s: Option %i has no title\n", __FUNCTION__, i );
              sprintf( holdbuf + 2, "  No Title:  " );

              col1 = holdbuf + 15;
            }
            col1[0] = ' ';

            if ( option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION )
              log_debug( "%s: No unit conversion specified.\n", __FUNCTION__ );

            if ( option->flags & EVMS_OPTION_FLAGS_ADVANCED )
              log_debug( "%s: Advanced Option.\n", __FUNCTION__ );

            if ( EVMS_OPTION_IS_ACTIVE( option->flags ) ) {
              option_settings[j] &= ~OPT_SET_IS_INACTIVE;
              if ( EVMS_OPTION_VALUE_IS_LIST( option->flags ) ) {
                option_settings[j] |= OPT_SET_IS_LIST;
                memcpy( col1, setopt_opt_is_list, strlen( setopt_opt_is_list ));
                log_debug( "%s: %s It contains %i items\n", __FUNCTION__, setopt_opt_is_list, option->value.list->count );
              }
              else  if ( EVMS_OPTION_HAS_VALUE( option->flags ) ) {
                if (( option_settings[j] & OPT_SET_IS_COMPLETE ) == 0 ) {
                  option_settings[j] &= ~OPT_SET_MUST_BE_SET;
                }
                if ( option->unit == EVMS_Unit_Sectors && ( option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION ) == 0 ) {
                  sectors = convert_to_sector_count( option->type, &( option->value ) );
                  draw_sector_size( sectors, work_buffer );
                  title_size = strlen( work_buffer );
                  memcpy( col1, work_buffer, title_size );
                }
                else {
                  draw_value_type( option->type, EVMS_Format_Normal, &( option->value ), 25, work_buffer );
                  if ( work_buffer[0] != '\0' ) {
                    title_size = strlen( work_buffer );
                    memcpy( col1, work_buffer, title_size );
                    if ( !draw_value_units( option->unit, work_buffer )) {
                      col2 = col1 + title_size + 2;
                      title_size = strlen( work_buffer );
                      memcpy( col2, work_buffer, title_size );
                    }
                  }
                }
              }
              if ( option->constraint_type == EVMS_Collection_List ) {
                option_settings[j] |= OPT_SET_IS_COLLECTION;
                log_debug( "%s: Constraint List of %i items\n", __FUNCTION__, option->constraint.list->count );
              }
              else if ( option->constraint_type == EVMS_Collection_Range )
                log_debug( "%s: Constraint Range\n", __FUNCTION__ );
              else
                log_debug( "%s: No constraints specified\n", __FUNCTION__ );
            }
            else {
              memcpy( col1, setopt_opt_not_active, strlen( setopt_opt_not_active ));

              if (( option_settings[j] & OPT_SET_IS_COMPLETE ) == 0 )
                option_settings[j] |= OPT_SET_IS_INACTIVE;

              log_debug( "%s: %s\n", __FUNCTION__, setopt_opt_not_active );
            }

            if ( j == select_index )
              print_string( XPOS_BOX3TEXT, i, COLOR_BAR, holdbuf );
            else
              print_string( XPOS_BOX3TEXT, i, COLOR_TITLE, holdbuf );

            option_settings[j] |= OPT_SET_IS_COMPLETE;
            evms_free( option );
          }
          else {
            log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_optdesc, j, err_msg_rc, rc );
          }
        }
        draw_status_line( setopt_space_enter );
        for ( i = 0; i < option_count; i++ ) {
          if (( option_settings[i] & OPT_SET_MUST_BE_SET ) != 0 )
            draw_status_line( setopt_space_select );
        }

        keyval = key_scroller( &start_index, &select_index, option_count, screen_lines - 2 );

        switch ( keyval ) {

          case KEY_ESC :
            rc = ESC_PRESSED;
            break;

          case KEY_F1  :
          case '1'     :
            draw_help_panel( 5 );
            keyval = KEY_CURSORACT;
            break;

          case ' ' :
            if (( option_settings[select_index] & OPT_SET_IS_INACTIVE ) == 0 ) {
              cur_option = select_index;
              if (( option_settings[select_index] & OPT_SET_IS_LIST ) != 0 )
                rc = select_option_menu( task_context, cur_option );
              else if (( option_settings[select_index] & OPT_SET_IS_COLLECTION ) != 0 )
                rc = select_option_menu( task_context, cur_option );
              else
                rc = edit_option_menu( task_context, cur_option );

              if ( rc == 0 )
                option_settings[select_index] &= ~OPT_SET_MUST_BE_SET;
            }
            keyval = KEY_CURSORACT;
            break;

          case KEY_ENTER :
            for ( i = 0; i < option_count; i++ ) {
              if (( option_settings[i] & OPT_SET_MUST_BE_SET ) != 0 )
                keyval = KEY_CURSORACT;
            }
            break;

          default:
            keyval = KEY_CURSORACT;
            break;
        }

      }  while ( keyval == KEY_CURSORACT );
    }
  }
  else {
    log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_get_acceptable, rc );
  }

  free( holdbuf );
  return rc;
}


/*----------------------------------------------------------------------
 * Public Functions
 *---------------------------------------------------------------------*/

int resize_object( object_handle_t object, task_action_t raction )
{
  int rc = 0, menu_state = 1;
  object_handle_t resize_object;
  task_handle_t resize_context = 0;
  handle_array_t *resultant_points;

  /*
   * Skip getting expansion points for a container. This is
   * basically what we get when we get acceptable objects
   * to consume.
   */
  if ( raction == EVMS_Task_Expand_Container ) {    
    resize_object = object;
    menu_state = 2;
  }
    
  while ( menu_state ) {

    switch ( menu_state ) {

      case 1 :
        menu_state = 0;  /* initialize in case of error */

        if ( raction == EVMS_Task_Expand )
          rc = where_to_expand( object, &resize_object );
        else
          rc = where_to_shrink( object, &resize_object );

        if ( rc ) {
          if ( rc == ESC_PRESSED )
            return rc;

          if ( raction == EVMS_Task_Expand )
            draw_status_line( err_msg_no_expandpt );
          else
            draw_status_line( err_msg_no_shrinkpt );

          get_inputc();
        }
        else  if ( resize_object )
                menu_state = 2;
        break;

      case 2 :
        menu_state = 0;  /* initialize in case of error */

        if ( !resize_context ) {
          if ( raction == EVMS_Task_Expand )
            log_debug( "%s: Creating Expand task using object handle %i\n", __FUNCTION__, resize_object );
          else if ( raction == EVMS_Task_Expand_Container )
            log_debug( "%s: Creating Expand Container task using object handle %i\n", __FUNCTION__, resize_object );              
          else
            log_debug( "%s: Creating Shrink task using object handle %i\n", __FUNCTION__, resize_object );

          rc = evms_create_task( resize_object, raction, &resize_context );
          if ( rc == 0 )
            menu_state = 3;
          else
            log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_create_task, rc );
        }
        break;

      case 3 :
        menu_state = 0;  /* initialize in case of error */
        if ( raction == EVMS_Task_Expand )
          rc = set_objects_menu( resize_context, _( " Expand Transmogrification Select " ));
        else if ( raction == EVMS_Task_Expand_Container )
          rc = set_objects_menu( resize_context, _( " Select Additional Storage Objects To Consume " ));            
        else
          rc = set_objects_menu( resize_context, _( " Shrink Transmogrification Select " ));

        if ( rc == 0 )
          menu_state = 4;
        else  if ( rc == ESC_PRESSED ) {
          rc = evms_destroy_task( resize_context );
          resize_context = 0;
          menu_state = 1;
        }
        break;

      case 4 :
        menu_state = 0;  /* initialize in case of error */
        if ( raction == EVMS_Task_Expand )
          rc = set_options_menu( resize_context, _( " Expand Configuration Options " ));
        else if ( raction == EVMS_Task_Expand_Container )
          rc = set_options_menu( resize_context, _( " Expand Container Configuration Options " ));            
        else
          rc = set_options_menu( resize_context, _( " Shrink Configuration Options " ));

        if ( rc == 0 ) {
          rc = evms_invoke_task( resize_context, &resultant_points );
          if ( rc == 0 ) {
            if ( resultant_points ) {
              log_debug( "%s: Resize task successful, %i resultant objects\n", __FUNCTION__, resultant_points->count );
              evms_free( resultant_points );
            }
          }
        }
        else  if ( rc == ESC_PRESSED )
          menu_state = 3;
        break;

      default:
        menu_state = 0;  /* unknown state - drop out  */
        break;
    }
  }

  if ( rc == 0 ) {
    if ( raction == EVMS_Task_Expand || raction == EVMS_Task_Expand_Container )
      draw_status_line( _( "Expand was successful, press any key to continue" ));
    else
      draw_status_line( _( "Shrink was successful, press any key to continue" ));
  }
  else {
    if ( rc == NO_OBJECTS_FOUND ) {
        draw_status_line ( err_msg_no_accept_objs );
    } else {    
        if ( raction == EVMS_Task_Expand || raction == EVMS_Task_Expand_Container )
          sprintf( work_buffer, _( "Expand failed, rc == %i" ), rc );
        else
          sprintf( work_buffer, _( "Shrink failed, rc == %i" ), rc );
        
        draw_status_line( work_buffer );        
    }
  }
  get_inputc();

  if ( resize_context )
    rc = evms_destroy_task( resize_context );

  return rc;
}


int create_new_object( evms_plugin_code_t type, int need_container, char *panel_title )
{
  int rc = 0, menu_state = 1;
  plugin_handle_t plugin_object;
  task_handle_t create_context = 0;
  handle_array_t *resultant_points;

  while ( menu_state ) {

    switch ( menu_state ) {

      case 1 :
        menu_state = 0;  /* initialize in case of error */

        rc = select_plugin( type, need_container, &plugin_object, panel_title );

        if ( rc ) {
          if ( rc == ESC_PRESSED )
            return rc;

          draw_status_line( err_msg_no_plugins );

          get_inputc();
        }
        else  if ( plugin_object )
                menu_state = 2;
        break;

      case 2 :
        menu_state = 0;  /* initialize in case of error */

        if ( !create_context ) {

          if ( need_container ) {
            log_debug( "%s: Creating Create_Container task using object handle %i\n", __FUNCTION__, plugin_object );
            rc = evms_create_task( plugin_object, EVMS_Task_Create_Container, &create_context );
          }
          else {
            log_debug( "%s: Creating Create task using object handle %i\n", __FUNCTION__, plugin_object );
            rc = evms_create_task( plugin_object, EVMS_Task_Create, &create_context );
          }

          if ( rc == 0 )
            menu_state = 3;
          else
            log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_create_task, rc );
        }
        break;

      case 3 :
        menu_state = 0;  /* initialize in case of error */
        rc = set_objects_menu( create_context, _( " Create - Select Acceptable Objects " ));

        if ( rc == 0 )
          menu_state = 4;
        else  if ( rc == ESC_PRESSED ) {
          rc = evms_destroy_task( create_context );
          create_context = 0;
          menu_state = 1;
        }
        else {
          if ( rc == NO_OBJECTS_FOUND )
            draw_status_line( err_msg_no_accept_objs );
          else {
            sprintf( work_buffer, "%s %i", err_msg_accept_objs_rc, rc );
            draw_status_line( work_buffer );
          }

          get_inputc();
        }
        break;

      case 4 :
        menu_state = 0;  /* initialize in case of error */
        rc = set_options_menu( create_context, _( " Create Configuration Options " ));

        if ( rc == 0 ) {
          rc = evms_invoke_task( create_context, &resultant_points );
          if ( rc == 0 ) {
            if ( resultant_points ) {
              log_debug( "%s: Create task successful, %i resultant objects\n", __FUNCTION__, resultant_points->count );
              evms_free( resultant_points );
            }
          }
        }
        else  if ( rc == ESC_PRESSED )
          menu_state = 3;
        break;

      default:
        menu_state = 0;  /* unknown state - drop out  */
        break;
    }
  }

  if ( rc == 0 )
    draw_status_line( _( "Create was successful, press any key to continue" ));
  else  {
    sprintf( work_buffer, _( "Create failed, rc == %i" ), rc );
    draw_status_line( work_buffer );
  }
  get_inputc();

  if ( create_context )
    rc = evms_destroy_task( create_context );

  return rc;
}


int assign_object( evms_plugin_code_t type, int need_container, char *panel_title )
{
  int rc = 0, menu_state = 1;
  plugin_handle_t plugin_object;
  task_handle_t assign_context = 0;
  handle_array_t *resultant_points;

  while ( menu_state ) {

    switch ( menu_state ) {

      case 1 :
        menu_state = 0;  /* initialize in case of error */

        rc = select_plugin( type, need_container, &plugin_object, panel_title );

        if ( rc ) {
          if ( rc == ESC_PRESSED )
            return rc;

          draw_status_line( err_msg_no_plugins );

          get_inputc();
        }
        else  if ( plugin_object )
                menu_state = 2;
        break;

      case 2 :
        menu_state = 0;  /* initialize in case of error */

        if ( !assign_context ) {

          log_debug( "%s: Creating Assign Plugin task using object handle %i\n", __FUNCTION__, plugin_object );
          rc = evms_create_task( plugin_object, EVMS_Task_Assign_Plugin, &assign_context );

          if ( rc == 0 )
            menu_state = 3;
          else
            log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_create_task, rc );
        }
        break;

      case 3 :
        menu_state = 0;  /* initialize in case of error */
        rc = set_objects_menu( assign_context, _( " Assign - Select Acceptable Objects " ));

        if ( rc == 0 )
          menu_state = 4;
        else  if ( rc == ESC_PRESSED ) {
          rc = evms_destroy_task( assign_context );
          assign_context = 0;
          menu_state = 1;
        }
        else {
          if ( rc == NO_OBJECTS_FOUND )
            draw_status_line( err_msg_no_accept_objs );
          else {
            sprintf( work_buffer, "%s %i", err_msg_accept_objs_rc, rc );
            draw_status_line( work_buffer );
          }

          get_inputc();
        }
        break;

      case 4 :
        menu_state = 0;  /* initialize in case of error */
        rc = set_options_menu( assign_context, _( " Assign Configuration Options " ));

        if ( rc == 0 ) {
          rc = evms_invoke_task( assign_context, &resultant_points );
          if ( rc == 0 ) {
            if ( resultant_points ) {
              log_debug( "%s: Assign Plugin task successful, %i resultant objects\n", __FUNCTION__, resultant_points->count );
              evms_free( resultant_points );
            }
          }
        }
        else  if ( rc == ESC_PRESSED )
          menu_state = 3;
        break;

      default:
        menu_state = 0;  /* unknown state - drop out  */
        break;
    }
  }

  if ( rc == 0 )
    draw_status_line( _( "Assign was successful, press any key to continue" ));
  else  {
    sprintf( work_buffer, _( "Assign failed, rc == %i" ), rc );
    draw_status_line( work_buffer );
  }
  get_inputc();

  if ( assign_context )
    rc = evms_destroy_task( assign_context );

  return rc;
}


int destroy_object( object_handle_t object, char *destroy_msg )
{
  int rc = 0;

  rc = select_box( destroy_msg,
                   _( "Do not destroy" ),
                   _( "Destroy" ),
                   NULL  );

  if ( rc == 2 ) {
    rc = evms_destroy( object );
    if ( rc == 0 ) {
      draw_status_line( _( "Destroy was successful, press any key to continue" ));
    }
    else {
      sprintf( work_buffer, _( "Destroy failed, rc == %i" ), rc );
      draw_status_line( work_buffer );
    }
    get_inputc();
  }

  return rc;
}


int remove_object( object_handle_t object, char *remove_msg )
{
  int rc = 0;

  rc = select_box( remove_msg,
                   _( "Do not remove" ),
                   _( "Remove" ),
                   NULL  );

  if ( rc == 2 ) {
    rc = evms_remove( object );
    if ( rc == 0 ) {
      draw_status_line( _( "Remove was successful, press any key to continue" ));
    }
    else {
      sprintf( work_buffer, _( "Remove failed, rc == %i" ), rc );
      draw_status_line( work_buffer );
    }
    get_inputc();
  }

  return rc;
}

int remove_object_from_container( object_handle_t object, char *remove_msg )
{
  int rc = 0;

  rc = select_box( remove_msg,
                   _( "Do not remove" ),
                   _( "Remove" ),
                   NULL  );

  if ( rc == 2 ) {
    rc = evms_transfer( object, 0, 0, NULL );
    if ( rc == 0 ) {
      draw_status_line( _( "Remove was successful, press any key to continue" ));
    }
    else {
      sprintf( work_buffer, _( "Remove failed, rc == %i" ), rc );
      draw_status_line( work_buffer );
    }
    get_inputc();
  }

  return rc;
}


int object_create_evms( object_handle_t object )
{
  int rc = 0, screen_cols;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX4TEXT - XPOS_BOX4RIGHT - 6;
  holdbuf = malloc( screen_cols + 1 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  rc = select_box( _( "Create the EVMS Volume?" ),
                   create_nocreate,
                   create_docreate,
                   NULL  );

  if ( rc == 2 ) {
    BLANK_BUFFER( holdbuf, screen_cols )
    rc = naming_box( _( "Enter the Volume Name: " ),
                     screen_cols, holdbuf );
    if ( rc == 0 ) {
      rc = evms_create_volume( object, holdbuf );
      if ( rc == 0 ) {
        draw_status_line( info_msg_create_ok );
      }
      else {
        sprintf( work_buffer, err_msg_create_fail, rc );
        draw_status_line( work_buffer );
      }
      get_inputc();
    }
  }

  free( holdbuf );
  return rc;
}


int object_create_compat( object_handle_t object )
{
  int rc = 0;

  rc = select_box( _( "Create the Compatibility Volume?" ),
                   create_nocreate,
                   create_docreate,
                   NULL  );

  if ( rc == 2 ) {
    rc = evms_create_compatibility_volume( object );
    if ( rc == 0 ) {
      draw_status_line( info_msg_create_ok );
    }
    else {
      sprintf( work_buffer, err_msg_create_fail, rc );
      draw_status_line( work_buffer );
    }
    get_inputc();
  }

  return rc;
}


int object_extended_info( object_handle_t object )
{
  int i, j, rc = 0, item_count, start_index = 0, keyval,
      screen_cols, cols_left, screen_lines = CALC_SCREEN3_LINES;
  extended_info_array_t *extended_info=NULL;
  char *holdbuf, *col1, *col2;
  size_t  title_size;

  draw_box3( _( " Detailed Information " ));

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT;
  holdbuf = malloc( screen_cols + 1 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    draw_status_line( err_msg_malloc_blanks );
    return ENOMEM;
  }

  // first blank the lines
  BLANK_BUFFER( holdbuf, screen_cols )
  for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
    print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

  rc = evms_get_extended_info( object, NULL, &extended_info );
  if ( rc == 0 ) {
    if ( extended_info && extended_info->count ) {
         log_debug( "%s: %s %i %s\n", __FUNCTION__, info_msg_get_extinfo, extended_info->count, info_msg_getitems );

      do {

        // blank the lines each time through
        BLANK_BUFFER( holdbuf, screen_cols )
        for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
          print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

        if ( screen_lines > extended_info->count - start_index )
          item_count = extended_info->count - start_index;
        else
          item_count = screen_lines + start_index;

        for ( i = start_index, j = YPOS_BOX3TOP + 2; i < item_count; i++, j++ ) {
          log_debug( "%s: ExtInfo item %i is called %s\n", __FUNCTION__, i, extended_info->info[i].title );
          BLANK_BUFFER( holdbuf, screen_cols )
          title_size = strlen( extended_info->info[i].title );
          cols_left = screen_cols - ( title_size + 4 );
          if ( cols_left > 50 )
            cols_left = 50;
          col1 = holdbuf + title_size + 3;
          sprintf( holdbuf, "%s:  ", extended_info->info[i].title );

          draw_value_type( extended_info->info[i].type, extended_info->info[i].format,
                           &( extended_info->info[i].value ), cols_left, work_buffer );
          title_size = strlen( work_buffer );
          cols_left -= title_size;
          if ( cols_left < 0 )
            cols_left = 0;
          memcpy( col1, work_buffer, title_size );
          col2 = col1 + title_size + 1;
          if ( !draw_value_units( extended_info->info[i].unit, work_buffer ) && cols_left ) {
            title_size = strlen( work_buffer );
            if ( title_size < cols_left )
              cols_left = title_size;
            memcpy( col2, work_buffer, cols_left );
          }

          print_string( XPOS_BOX3TEXT, j, COLOR_TITLE, holdbuf );
        }
        draw_status_line( ext_info_when_done );

        keyval = screen_scroller( &start_index, extended_info->count, screen_lines );

        switch ( keyval ) {

          case KEY_ESC :
          case KEY_ENTER :
            break;

          default:
            keyval = KEY_CURSORACT;
            break;
        }

      }  while ( keyval == KEY_CURSORACT );
    }
    else {
      draw_status_line( info_no_ext_info );
      i = get_inputc();
    }
    evms_free( extended_info );
  }
  else {
    sprintf( work_buffer, "%s %i", err_msg_get_extinfo, rc );
    log_error( "%s: %s\n", __FUNCTION__, err_msg_get_info, work_buffer );
    draw_status_line( work_buffer );
    i = get_inputc();
  }

  free( holdbuf );
  return 0;
}

int set_object_info( object_handle_t object )
{
  int rc = 0, menu_state = 1;
  task_handle_t setinfo_context = 0;
  handle_array_t *resultant_objects;
    
  while ( menu_state ) {

    switch ( menu_state ) {

      case 1 :
        menu_state = 0;  /* initialize in case of error */

        if ( !setinfo_context ) {

          log_debug( "%s: Creating Set Options task using object handle %i\n", __FUNCTION__, object );
          rc = evms_create_task( object, EVMS_Task_Set_Info, &setinfo_context );

          if ( rc == 0 )
            menu_state = 2;
          else
            log_error( "%s: %s %i.\n", __FUNCTION__, err_msg_create_task, rc );
        }
        break;

      case 2 :
        menu_state = 0;  /* initialize in case of error */
        rc = set_options_menu( setinfo_context, _( " Modify Object Properties " ));

        if ( rc == 0 ) {
          rc = evms_invoke_task( setinfo_context, &resultant_objects );
          if ( rc == 0 ) {
            if ( resultant_objects ) {
              log_debug( "%s: Set Options task successful, %i resultant objects\n", __FUNCTION__, resultant_objects->count );
              evms_free( resultant_objects );
            }
          }
        }
        else  if ( rc == ESC_PRESSED )
          menu_state = 1;
        break;

      default:
        menu_state = 0;  /* unknown state - drop out  */
        break;
    }
  }

  if ( rc != ESC_PRESSED )
  {    
    if ( rc == 0 )
      draw_status_line( _( "Modify Properties was successful, press any key to continue" ));
    else  {
      sprintf( work_buffer, _( "Modify Properties failed, rc == %i" ), rc );
      draw_status_line( work_buffer );
    }  
    get_inputc();
  }

  if ( setinfo_context )
    rc = evms_destroy_task( setinfo_context );
  
  return rc;
}
