/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: defsegmgr.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>
#include <asm/vtoc.h>

#include "390segmgr.h"


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                         PRIVATE DATA AREAS AND SUBROUTINES                           +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

static plugin_record_t my_plugin_record;
static engine_functions_t * EngFncs = NULL;

plugin_record_t                *my_plugin_record_ptr=&my_plugin_record;

static char part_names[4][8] = {{0xd3,0xd5,0xe7,0xf1,0x00},//LNX1
	{0xe5,0xd6,0xd3,0xf1,0x00},//VOL1
	{0xc3,0xd4,0xe2,0xf1,0x00},//CMS1
	"(nonl)"
};


/*
 * compute the block number from a 
 * cyl-cyl-head-head-block structure
 */
static inline int
cchhb2blk (cchhb_t *ptr, struct hd_geometry *geo) {
	return ptr->cc * geo->heads * geo->sectors +
	ptr->hh * geo->sectors +
	ptr->b;
}

/*
 * compute the block number from a 
 * cyl-cyl-head-head structure
 */
static inline int
cchh2blk (cchh_t *ptr, struct hd_geometry *geo) {
	return ptr->cc * geo->heads * geo->sectors +
	ptr->hh * geo->sectors;
}


static ibm_partition_t
get_partition_type ( char * type )
{
	int i;
	for ( i = 0; i < 3; i ++) {
		if ( ! strncmp (type,part_names[i],4) )
			break;
	}
	return i;
}



/* Function: s390_append_segment_to_object
 *
 *	Associate the specified segment and object as parent/child. Add the
 *	segment to the object's "parent" dlist, and add the object to the
 *	segment's "child" dlist. Also need to check for duplicates, so the
 *	same segment/object don't get associated more than once.
 */
static int s390_append_segment_to_object(storage_object_t       * segment,
					 storage_object_t       * object )
{
	int     rc;
	void    * handle;

	// No LOG_ENTRY or RETURN calls. Writes too many messages to the log.

	rc = ExclusiveInsertObject(object->parent_objects,
				   sizeof(storage_object_t),
				   segment,
				   REGION_TAG,
				   NULL,
				   AppendToList,
				   TRUE,
				   &handle);
	if (rc) {
		LOG_SERIOUS("Error adding segment %s as a parent to object %s\n",
			    segment->name, object->name);
		return rc;
	}

	rc = ExclusiveInsertObject(segment->child_objects,
				   sizeof(storage_object_t),
				   object,
				   object->object_type,
				   NULL,
				   AppendToList,
				   TRUE,
				   &handle);
	if (rc) {
		LOG_SERIOUS("Error adding object %s as a child to segment %s\n",
			    object->name, segment->name);
		DeleteObject(object->parent_objects, segment);
	}

	return rc;
}


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/
static int S390_SetupEVMSPlugin( engine_mode_t        mode,
				 engine_functions_t * engine_functions)
{
	int rc = 0;

	if (engine_functions) {

		EngFncs  = engine_functions;

		LOG_EXITRC();
	}

	return rc;
}


static void S390_Cleanup(void)
{
	int rc =0;

	LOG_ENTRY();

	LOG_EXIT(rc);
}




/*
 *  I will allow the object to be made into a volume (or reverted) if ...
 *
 *  (1) I actually own the object
 *  (2) Which means it is a segment and has necessarey ctl blocks
 *
 */
static int S390_can_set_volume(storage_object_t * seg, BOOLEAN flag )
{
	int rc = 0;

	LOG_ENTRY();

	LOG_EXITRC();
	return rc;
}


/*
 *  I can delete this segment if ...
 *
 *  (1) I own the segment
 *  (2) It is either a data segment or else an mbr segment and
 *      there are no data segments.
 *  (3) It isnt the first logical drive in an ebr chain
 */
static int S390_CanDestroy( storage_object_t * seg )
{
	int                rc = ENOSYS;

	LOG_ENTRY();

	LOG_EXITRC();
	return rc;
}


/*
 *  I can expand the object if ...
 *
 *  (1) it is a data segment
 *  (2) I own the segment
 *  (3) the logical disk and segment info is Ok
 *  (4) a freespace segment immediately follows it
 *
 */
static int S390_CanExpand( storage_object_t       *seg,	       // data segment
			   sector_count_t  *expand_limit,      // ?
			   dlist_t          expansion_points ) // dlist to place expand object on
{
	int rc = ENOSYS;
	LOG_ENTRY();
	LOG_EXITRC();
	return rc;

}


/*
 * I can expand the object by size sectors if:
 *
 *  (1) i own the object
 *  (2) the segment can be expanded
 *  (3) the freespace has at least a cylinder of space
 *
 * If I cannot expand because the freespace is too small
 * then I'll reduce the expand sector count to my maximum.
 *
 */
static int S390_CanExpandBy(storage_object_t * seg, sector_count_t *size)
{
	int                       rc = ENOSYS;


	LOG_ENTRY();

	LOG_EXITRC();
	return rc;
}


/*
 * I can shrink a seg if ...
 *
 *  (1) i own the object
 *  (2) it is a data segment
 *  (3) if I chop off a cylinder, the seg will still have
 *      a minimum of 1 cylinder of space
 *
 *  If not exact set new_size to closest higher value possible.
 */
static int S390_CanShrink( storage_object_t * seg,		 // object to shrink
			   sector_count_t   * shrink_limit,	 // a delta size
			   dlist_t            shrink_points )	 // of type shrink_object_info_t,
// tag = SHRINK_OBJECT_TAG
{
	int                   rc = ENOSYS;


	LOG_ENTRY();

	LOG_EXITRC();
	return rc;
}


/*
 *  I can allow the storage object to shrink by the specified amount if ...
 *
 *  (1) the shrink point is a segment and I own it
 *  (2) the segment is large enough to allow a shrink
 *  (3) i can shrink it and end the segment on a cylinder boundary
 *
 *  If the segment doesnt end on a cylinder boundary I'll return
 *  an error and report the amount that we could shrink by.
 *
 */
static int S390_CanShrinkBy( storage_object_t * seg,
			     sector_count_t   * size )
{
	int               rc = ENOSYS;

	LOG_ENTRY();

	LOG_EXITRC();
	return rc;
}
static int s390_create_segment(dlist_t output_list,
			       storage_object_t *     object,
			       u_int64_t            offset,
			       u_int64_t            size,
			       data_type_t	    type,
			       int                  index){

	int               rc = ENOSYS;
	void * waste;
	char name[EVMS_VOLUME_NAME_SIZE +1];
	storage_object_t * segment;

	LOG_ENTRY();

	sprintf(name,"%s%d",object->name,index);

	if ((rc = EngFncs->allocate_segment(name, &segment))) {
		RETURN(ENOMEM);
	}
	segment->plugin = my_plugin_rec;
	segment->size = size;
	segment->start = offset;
	segment->geometry = object->geometry;
	segment->private_data = (void *) object;
	segment->data_type = type;
	s390_append_segment_to_object(segment, object);
	InsertObject(output_list, sizeof(storage_object_t), segment, SEGMENT_TAG, NULL, AppendToList, FALSE, &waste);

	LOG_EXITRC();
	return rc;
}

static int s390_check_object(storage_object_t * object, dlist_t output_list){
	int rc=0;
	char data[512];
	ibm_partition_t partition_type;
	char type[5] = {0,};
	char name[7] = {0,};
	int blocksize = 0, // really hardsector size (in sectors)
	offset=0, size=0, psize=0, counter=0;
	unsigned int blk;
	format1_label_t f1;
	volume_label_t vlabel;
	u_int64_t io_start;
	struct hd_geometry geo;

	LOG_ENTRY();
	blocksize = object->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT; 
	io_start = LABEL_BLOCK * blocksize; // vol label is in 2nd block
	rc = EVMS_READ(object, io_start, 1, data);
	if (rc) {
		LOG_ERROR("error(%d) reading vol lable from sector(%Ld) from '%s'.\n",
			  rc, io_start, object->name);
		object->flags |= SOFLAG_CORRUPT;
	}
	if (!rc) {
		/* determine the format type
		 */

		strncpy (type, data, 4);
/*		if ((!info->FBA_layout) && (!strcmp(info->type,"ECKD"))) {
			strncpy ( name, data + 8, 6);
		} else {
			strncpy ( name, data + 4, 6);
		} */
		memcpy (&vlabel, data, sizeof(volume_label_t));

//		EBCASC(type,4);
//		EBCASC(name,6);
		partition_type = get_partition_type(type);
		LOG_DETAILS("disk: raw type(%s), type(%s), name(%s)\n",
			    type, part_names[partition_type], name);
		switch ( partition_type ) {
		case ibm_partition_cms1:
			if (*((long *)data + 13) != 0) {
				/* disk is reserved minidisk */
				long *label=(long*)data;
				blocksize = label[3] >> EVMS_VSECTOR_SIZE_SHIFT; // override hard_sector size???????
				offset = label[13];
				size = (label[7] - 1) * blocksize; 
				LOG_DEBUG("(MDSK)");
			} else {
				offset = LABEL_BLOCK + 1;
				size = object->size;
			}
			offset *= blocksize;
			/* adjust for 0 thru label block offset
			 */
			size -= offset;
			rc = s390_create_segment(output_list,
						 object,
						 offset,
						 size,
						 DATA_TYPE,
						 1);
			break;
		case ibm_partition_lnx1: 
		case ibm_partition_none:
			offset = LABEL_BLOCK + 1;
			offset *= blocksize;
			size = object->size;
			/* adjust for 0 thru label block offset
			 */
			size -= offset;
			rc = s390_create_segment(output_list,
						 object,
						 offset,
						 size,
						 DATA_TYPE,
						 1);
			break;
		case ibm_partition_vol1: 
			/* get block number and read then first format1 label */
			geo.cylinders = object->geometry.cylinders;
			geo.heads = object->geometry.heads;
			geo.sectors = object->geometry.sectors_per_track;
			geo.start = 0;

			blk = cchhb2blk(&vlabel.vtoc, &geo) + 1;  
			io_start = blk * blocksize;
			rc = EVMS_READ(object, io_start, 1, data);
			if (rc) {
				LOG_ERROR("error(%d) reading sector(%Ld) from '%s'.\n",
					  rc, io_start, object->name);
				break;
			} else {
//				ldprint_mem(data, EVMS_VSECTOR_SIZE);
			}
			memcpy (&f1, data, sizeof(format1_label_t));

			while (f1.DS1FMTID == 0xf1) {
				offset = cchh2blk(&f1.DS1EXT1.llimit, &geo);
				psize  = cchh2blk(&f1.DS1EXT1.ulimit, &geo) - 
					 offset + geo.sectors;

				counter++;
				rc = s390_create_segment(output_list,
							 object,
							 offset * blocksize,
							 psize * blocksize,
							 DATA_TYPE,
							 counter);

				blk++;
				io_start = blk * blocksize;
				rc = EVMS_READ(object, io_start, 1, data);
				if (rc) {
					LOG_ERROR("error(%d) reading sector(%Ld) from '%s'.\n",
						  rc, io_start, object->name);
					break;
				} else {
// 					ldprint_mem(data, EVMS_VSECTOR_SIZE);
				}
				memcpy (&f1, data, sizeof(format1_label_t));
			}
			if (!counter) {
				rc = s390_create_segment(output_list,
							 object,
							 io_start,
							 object->size - io_start,
							 FREE_SPACE_TYPE,
							 1);
			}
			break;
		default:
		}
	}


	RETURN(rc);
}


/*
 *  Called to run discovery code on dlist of evms objects that
 *  the engine has found so far.  We essentially need to walk the
 *  list of objects, looking for Logical Disks, and see if we
 *  recognize partitioning schemes.  If so, consume the logical
 *  disk by removing it from the dlist, and place all new segment
 *  objects on the output_object dlist. Any object we dont like in
 *  the input_object dlist must be copied to the ouput_object dlist.
 *
 */
static int S390_Discover( dlist_t input_objects, dlist_t output_objects, BOOLEAN final_call)
{
	int  rc = 0;
	int count = 0;
	storage_object_t * object;
	int size,tag,waste;

	LOG_ENTRY();
	GoToStartOfList(input_objects);
	while (!(rc = BlindExtractObject(input_objects, &size, (TAG *)&tag, NULL, (void *)&object))) {
		if (object->data_type == DATA_TYPE && object->object_type == DISK) {
			rc = s390_check_object(object, output_objects);
			if (rc) {
				// wasn't ours, put it on the output list now.
				InsertObject(output_objects, size, object, tag, NULL, AppendToList, FALSE, (void*)&waste);
			} else {
				count++;
				LOG("Found 390 segment on object %s\n",object->name );

			}
		} else {
			InsertObject(output_objects, size, object, tag, NULL, AppendToList, FALSE, (void*)&waste);
			LOG_DETAILS("Skipping object %s because not DATA_TYPE, or not DISK\n",object->name);
		}

	}
	if ((rc == DLIST_EMPTY) || (rc == DLIST_END_OF_LIST)) {
		rc = 0;
	} else {
		LOG_WARNING("Error processing input list rc = %d\n", rc);
	}

	RETURN(count);
}




/*
 * Create storage_object_t(s) from the list of objects using the given
 * options.  Return the newly allocated storage_object_t(s) in new_objects
 * list.
 */
static int S390_CreateSegment( dlist_t          input_objects,
			       option_array_t * options,
			       dlist_t          new_objects)
{
	int                 rc=ENOSYS;


	LOG_ENTRY();

	LOG_EXITRC();
	return rc;
}

/*
 *  Called to free a data segment.  If the corresponding disk partition
 *  is a logical drive then we also need to free the EBR segment
 *  as well.
 */
static int S390_DestroySegment( storage_object_t * seg, dlist_t child_objects )
{
	int          rc = ENOSYS;

	LOG_ENTRY();

	LOG_EXITRC();
	return  rc;
}

/*
 *  Called to expand a data segment.  The segment will be expanded
 *  into the freespace segment that follows the data segment.
 */
static int S390_Expand( storage_object_t *seg, storage_object_t *expand_object, dlist_t  objects, option_array_t *options )
{
	int               rc = ENOSYS;

	LOG_ENTRY();

	LOG_EXITRC();
	return rc;
}



/*
 *  Called to shrink a data segment to new_size or next smaller increment.
 *  Then, update appropriate fields in the segment and make
 *  changes to freespace segments as needed.
 *
 */
static int S390_Shrink( storage_object_t * seg,
			storage_object_t * shrink_object,
			dlist_t            objects,
			option_array_t   * options )
{
	int               rc = ENOSYS;

	LOG_ENTRY();

	LOG_EXITRC();

	return rc;
}


static int S390_AddSectorsToKillList( storage_object_t *seg, lsn_t lsn, sector_count_t count)
{
	int                         rc = EINVAL;
	storage_object_t * object = (storage_object_t*)seg->private_data;

	LOG_ENTRY();

	if (lsn+count > seg->size) {
		LOG_ERROR("KillSectors beyond end of segment, lsn=%lld count=%lld segsize=%lld\n",
			  lsn,count,seg->size);
		RETURN(EINVAL);
	}
	rc = KILLSECTORS(object, lsn + seg->start, count);

	LOG_EXITRC();
	return rc;
}


static int S390_CommitChanges( storage_object_t *obj, uint phase )
{
	int          rc = EINVAL;

	LOG_ENTRY();
	LOG_DEBUG("object= %s\n", obj->name );

	LOG_EXITRC();
	return rc;
}



static int S390_Read( storage_object_t        *seg,
		      lsn_t           offset,
		      sector_count_t  count,
		      void           *buffer )
{
	int    rc;
	storage_object_t * object = (storage_object_t*)seg->private_data;

	LOG_ENTRY();

	if (offset+count > seg->size) {
		LOG_ERROR("Read beyond end of segment, offset=%lld count=%lld segsize=%lld\n",
			  offset,count,seg->size);
		RETURN(EINVAL);
	}
	rc = EVMS_READ(object, offset + seg->start, count, buffer);

	LOG_EXITRC();
	return rc;
}

static int S390_Write( storage_object_t        *seg,
		       lsn_t           offset,
		       sector_count_t  count,
		       void           *buffer )
{
	int     rc;
	storage_object_t * object = (storage_object_t*)seg->private_data;

	LOG_ENTRY();
	if (offset+count > seg->size) {
		LOG_ERROR("Write beyond end of segment, offset=%lld count=%lld segsize=%lld\n",
			  offset,count,seg->size);
		RETURN(EINVAL);
	}

	rc = EVMS_WRITE(object, offset + seg->start, count, buffer);

	LOG_EXITRC();
	return rc;
}


/*
 * This call notifies you that your object is being made into (or part of)
 * a volume or that your object is no longer part of a volume.  The "flag"
 * parameter indicates whether the volume is being created (TRUE) or
 * removed (FALSE).
 */
static void S390_set_volume(storage_object_t * object, BOOLEAN flag)
{
	return;
}


/*
 *  Initialize a new task context by allocating the option descriptor
 *  array and by getting acceptable objects, from the engine, for the
 *  task context, i.e. create, shrink, expand, assign ...
 */
int S390_InitTask(task_context_t * context)
{
	int rc = EINVAL;

	LOG_ENTRY();

	if (context) {

		switch (context->action) {
		
		case EVMS_Task_Create:

			context->min_selected_objects = 1;
			context->max_selected_objects = 1;


			break;

		case EVMS_Task_Assign_Plugin:

			context->min_selected_objects = 1;
			context->max_selected_objects = 1;

			break;


		case EVMS_Task_Expand:

			context->min_selected_objects = 1;
			context->max_selected_objects = 1;

			break;

		case EVMS_Task_Shrink:

			context->min_selected_objects = 1;
			context->max_selected_objects = 1;

			break;

		default:
			LOG_ERROR("context->action is unknown or unsupported\n");
			break;
		}
	}

	LOG_EXITRC();
	return rc;
}



/*
 *  Returns count of options for specified task
 */
int S390_GetOptionCount(task_context_t * task)
{
	int count=0;

	LOG_ENTRY();

	switch (task->action) {
	
	case EVMS_Task_Create:
//        count = S390_CREATE_OPTION_COUNT;
		break;

	case EVMS_Task_Assign_Plugin:
		//        count = S390_ASSIGN_OPTION_COUNT;
		break;

	case EVMS_Task_Expand:
//        count = S390_EXPAND_OPTION_COUNT;
		break;

	case EVMS_Task_Shrink:
//        count = S390_SHRINK_OPTION_COUNT;
		break;

	default:
		count = 0;
		break;
	}

	return count;
}



/*
 *
 */
int S390_SetOption( task_context_t * context,
		    u_int32_t        index,
		    value_t        * value,
		    task_effect_t  * effect )
{
	int rc=EINVAL;

	LOG_ENTRY();

	// a measure of protection ...
	if (context ) {

		switch (context->action) {
		
		case EVMS_Task_Create:
//            rc = set_create_option( context, index, value, effect );
			break;

		case EVMS_Task_Assign_Plugin:
//            rc = set_assign_option( context, index, value, effect );
			break;

		case EVMS_Task_Expand:
//            rc = set_expand_option( context, index, value, effect );
			break;

		case EVMS_Task_Shrink:
//            rc = set_shrink_option( context, index, value, effect );
			break;

		default:
			LOG_ERROR("context->action is unknown or unsupported\n");
			break;
		}
	}

	LOG_EXITRC();

	return rc;
}





/*
 * Returns segment specific information
 */
int S390_GetInfo( storage_object_t  * object, char * name, extended_info_array_t * * info)
{
	int rc = EINVAL;
	extended_info_array_t   *Info;

	LOG_ENTRY();

	// a measure of protection ...
	if (info == NULL) {
		LOG_EXITRC();
		return rc;
	}

	rc    = 0;  // init to failed calloc
	*info = NULL;	  // init to no info returned

	Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( 12 * sizeof(extended_info_t) ) );
	if (Info) {
		*info = Info;
		Info->count = 3;

		SET_STRING_FIELD( Info->info[0].name, "Name" );
		SET_STRING_FIELD( Info->info[0].title, "Name" );
		SET_STRING_FIELD( Info->info[0].desc, "This is the partition name. It must be unique on the system.");
		Info->info[0].type               = EVMS_Type_String;
		Info->info[0].unit               = EVMS_Unit_None;
		SET_STRING_FIELD( Info->info[0].value.s, object->name );
		Info->info[0].collection_type    = EVMS_Collection_None;
		memset( &Info->info[0].group, 0, sizeof(group_info_t));

		SET_STRING_FIELD( Info->info[1].name, "Size" );
		SET_STRING_FIELD( Info->info[1].title, "Size" );
		SET_STRING_FIELD( Info->info[1].desc, "This is the size of the partition in sectors.");
		Info->info[1].type               = EVMS_Type_Unsigned_Int64;
		Info->info[1].unit               = EVMS_Unit_None;
		Info->info[1].value.ui64         = object->size;
		Info->info[1].format             = EVMS_Format_Normal;
		Info->info[1].collection_type    = EVMS_Collection_None;
		memset( &Info->info[1].group, 0, sizeof(group_info_t));

		SET_STRING_FIELD( Info->info[2].name, "Start" );
		SET_STRING_FIELD( Info->info[2].title, "Start LBA" );
		SET_STRING_FIELD( Info->info[2].desc, "This is the sector offset of the partition on the disk, i.e. the logical block address of the first sector of the partition.");
		Info->info[2].type               = EVMS_Type_Unsigned_Int64;
		Info->info[2].unit               = EVMS_Unit_None;
		Info->info[2].value.ui64         = object->start;
		Info->info[2].format             = EVMS_Format_Normal;
		Info->info[2].collection_type    = EVMS_Collection_None;
		memset( &Info->info[2].group, 0, sizeof(group_info_t));



	} else {
		rc = ENOMEM;
	}
	LOG_EXITRC();
	return rc;
}


/*
 *
 */
int S390_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
	int rc = EINVAL;
	extended_info_array_t   *Info;
	char                     version_string[64];
	char                     required_version_string[64];


	LOG_ENTRY();

	// a measure of protection ...
	if (info == NULL) {
		LOG_EXITRC();
		return rc;
	}

	rc    = ENOMEM;	 // init to failed calloc
	*info = NULL;	  // init to no info returned

	Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (6*sizeof(extended_info_t))  );
	if (Info) {

		Info->count = 5;

		sprintf(version_string, "%d.%d.%d",
			MAJOR_VERSION,
			MINOR_VERSION,
			PATCH_LEVEL );

		sprintf(required_version_string, "%d.%d.%d",
			my_plugin_record_ptr->required_api_version.major,
			my_plugin_record_ptr->required_api_version.minor,
			my_plugin_record_ptr->required_api_version.patchlevel );

		SET_STRING_FIELD( Info->info[0].name, "ShortName" );
		SET_STRING_FIELD( Info->info[0].title, "Short Name" );
		SET_STRING_FIELD( Info->info[0].desc, "A short name given to this plugin.");
		Info->info[0].type               = EVMS_Type_String;
		Info->info[0].unit               = EVMS_Unit_None;
		SET_STRING_FIELD( Info->info[0].value.s, my_plugin_record_ptr->short_name );
		Info->info[0].collection_type    = EVMS_Collection_None;
		memset( &Info->info[0].group, 0, sizeof(group_info_t));

		SET_STRING_FIELD( Info->info[1].name, "LongName" );
		SET_STRING_FIELD( Info->info[1].title, "Long Name" );
		SET_STRING_FIELD( Info->info[1].desc, "A longer and more descriptive name for this plugin.");
		Info->info[1].type               = EVMS_Type_String;
		Info->info[1].unit               = EVMS_Unit_None;
		SET_STRING_FIELD( Info->info[1].value.s, my_plugin_record_ptr->long_name );
		Info->info[1].collection_type    = EVMS_Collection_None;
		memset( &Info->info[1].group, 0, sizeof(group_info_t));

		SET_STRING_FIELD( Info->info[2].name, "Type" );
		SET_STRING_FIELD( Info->info[2].title, "Plugin Type" );
		SET_STRING_FIELD( Info->info[2].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		Info->info[2].type               = EVMS_Type_String;
		Info->info[2].unit               = EVMS_Unit_None;
		SET_STRING_FIELD( Info->info[2].value.s, "Partition Manager" );
		Info->info[2].collection_type    = EVMS_Collection_None;
		memset( &Info->info[2].group, 0, sizeof(group_info_t));

		SET_STRING_FIELD( Info->info[3].name, "Version" );
		SET_STRING_FIELD( Info->info[3].title, "Plugin Version" );
		SET_STRING_FIELD( Info->info[3].desc, "This is the version number of the plugin.");
		Info->info[3].type               = EVMS_Type_String;
		Info->info[3].unit               = EVMS_Unit_None;
		SET_STRING_FIELD( Info->info[3].value.s, version_string );
		Info->info[3].collection_type    = EVMS_Collection_None;
		memset( &Info->info[3].group, 0, sizeof(group_info_t));

		SET_STRING_FIELD( Info->info[4].name, "Required Version" );
		SET_STRING_FIELD( Info->info[4].title, "Required Engine Version" );
		SET_STRING_FIELD( Info->info[4].desc, "This is the version of the engine that the plugin requires. It will not run on older versions of the Engine.");
		Info->info[4].type               = EVMS_Type_String;
		Info->info[4].unit               = EVMS_Unit_None;
		SET_STRING_FIELD( Info->info[4].value.s, required_version_string );
		Info->info[4].collection_type    = EVMS_Collection_None;
		memset( &Info->info[4].group, 0, sizeof(group_info_t));

		*info = Info;

		rc = 0;
	}


	LOG_EXITRC();
	return rc;
}


/*
 * Validate the objects in the selected_objects dlist in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects dlist.
 * Modify the accepatble_objects dlist in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int S390_SetObjects( task_context_t * context,
		     dlist_t          declined_objects,
		     task_effect_t  * effect )
{

	int rc = EINVAL;

	LOG_ENTRY();

	if (context) {

		switch (context->action) {
		
		case EVMS_Task_Create:

//            rc = set_create_object( context, declined_objects, effect );
			break;

		case  EVMS_Task_Assign_Plugin:

//            rc = set_assign_object( context, declined_objects, effect );
			break;

		case EVMS_Task_Expand:

//            rc = set_expand_object( context, declined_objects, effect );
			break;

		case EVMS_Task_Shrink:

//            rc = set_shrink_object( context, declined_objects, effect );
			break;

		default:

			LOG_ERROR("context->action is unknown or unsupported\n");
			break;
		}
	}

	LOG_EXITRC();
	return rc;
}



/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                              PLUGIN FUNCTION TABLE                                   +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static struct plugin_functions_s sft={

	// the following routines are found above
	setup_evms_plugin:                   S390_SetupEVMSPlugin,
	cleanup_evms_plugin:                 S390_Cleanup,
	can_set_volume:                      S390_can_set_volume,
	can_delete:                          S390_CanDestroy,
	can_expand:                          S390_CanExpand,
	can_expand_by:                       S390_CanExpandBy,
	can_shrink:                          S390_CanShrink,
	can_shrink_by:                       S390_CanShrinkBy,
	discover:                            S390_Discover,
	create:                              S390_CreateSegment,
	delete:                              S390_DestroySegment,
	expand:                              S390_Expand,
	shrink:                              S390_Shrink,
	add_sectors_to_kill_list:            S390_AddSectorsToKillList,
	commit_changes:                      S390_CommitChanges,
	read:                                S390_Read,
	write:                               S390_Write,
	set_volume:                          S390_set_volume,
	get_option_count:                    S390_GetOptionCount,
	init_task:                           S390_InitTask,
	set_option:                          S390_SetOption,
	set_objects:                         S390_SetObjects,
	get_info:                            S390_GetInfo,
	get_plugin_info:                     S390_GetPluginInfo
};

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                       BUILD AND EXPORT AN EVMS PLUGIN RECORD                         +
+                                                                                      +
+--------------------------------------------------------------------------------------*/

static plugin_record_t plugin_record = {

	id:                               SetPluginID(EVMS_OEM_IBM, EVMS_SEGMENT_MANAGER, 2 ),

	version:                          {MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL},

	required_api_version:             {ENGINE_PLUGIN_API_MAJOR_VERION, ENGINE_PLUGIN_API_MINOR_VERION, ENGINE_PLUGIN_API_PATCH_LEVEL},

	short_name:                       "S390SegMgr",
	long_name:                        "S390 Segment Manager",
	oem_name:                         "IBM",

	functions:                        {plugin: &sft},

	container_functions:              NULL

};

// Vector of plugin record ptrs that we export for the EVMS Engine.
static plugin_record_t                *my_plugin_rec = &plugin_record;
plugin_record_t * evms_plugin_records[] = {&plugin_record, NULL};
