# -*- coding: utf-8 -*-

# Copyright (c) 2006 - 2008 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the log viewer widget and the log widget.
"""

from PyQt4.QtCore import *
from PyQt4.QtGui import *

from E4Gui.E4TabWidget import E4TabWidget

import UI.PixmapCache

class LogWidget(QTextEdit):
    """
    Class providing a specialized text edit for displaying logging information.
    """
    def __init__(self, parent = None):
        """
        Constructor
        
        @param parent reference to the parent widget (QWidget)
        """
        QTextEdit.__init__(self, parent)
        self.setAcceptRichText(False)
        self.setLineWrapMode(QTextEdit.NoWrap)
        self.setReadOnly(True)
        
        # create the context menu
        self.__menu = QMenu(self)
        self.__menu.addAction(self.trUtf8('Clear'), self.clear)
        self.__menu.addAction(self.trUtf8('Copy'), self.copy)
        self.__menu.addSeparator()
        self.__menu.addAction(self.trUtf8('Select All'), self.selectAll)
        
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.connect(self, SIGNAL("customContextMenuRequested(const QPoint &)"),
            self.__handleShowContextMenu)
        
    def __handleShowContextMenu(self, coord):
        """
        Private slot to show the context menu.
        
        @param coord the position of the mouse pointer (QPoint)
        """
        coord = self.mapToGlobal(coord)
        self.__menu.popup(coord)
        
    def appendText(self, txt):
        """
        Public method to append text to the end.
        
        @param txt text to insert (QString)
        """
        tc = self.textCursor()
        tc.movePosition(QTextCursor.End)
        self.setTextCursor(tc)
        self.insertPlainText(txt)
        self.ensureCursorVisible()

class LogViewer(E4TabWidget):
    """
    Class to provide a widget for displaying logging information.
    """
    def __init__(self, parent = None):
        """
        Constructor
        
        @param parent reference to the parent widget (QWidget)
        """
        E4TabWidget.__init__(self, parent)
        self.setTabPosition(QTabWidget.South)
        
        self.setWindowIcon(UI.PixmapCache.getIcon("eric.png"))
        
        self.__stdout = LogWidget()
        self.__stdoutIndex = self.addTab(self.__stdout, self.trUtf8("stdout"))
        
        self.__stderr = LogWidget()
        self.__stderrIndex = self.addTab(self.__stderr, self.trUtf8("stderr"))
        
        self.__menu = QMenu(self)
        self.__menu.addAction(self.trUtf8('Clear'), self.__handleClear)
        self.__menu.addAction(self.trUtf8('Copy'), self.__handleCopy)
        self.__menu.addSeparator()
        self.__menu.addAction(self.trUtf8('Select All'), self.__handleSelectAll)
        
        self.setTabContextMenuPolicy(Qt.CustomContextMenu)
        self.connect(self,SIGNAL('customTabContextMenuRequested(const QPoint &, int)'),
                     self.__handleShowContextMenu)
        
    def __handleShowContextMenu(self, coord, index):
        """
        Private slot to show the tab context menu.
        
        @param coord the position of the mouse pointer (QPoint)
        @param index index of the tab the menu is requested for (integer)
        """
        self.__menuIndex = index
        coord = self.mapToGlobal(coord)
        self.__menu.popup(coord)
        
    def __handleClear(self):
        """
        Private slot to handle the clear tab menu entry.
        """
        self.widget(self.__menuIndex).clear()
        
    def __handleCopy(self):
        """
        Private slot to handle the copy tab menu entry.
        """
        self.widget(self.__menuIndex).copy()
        
    def __handleSelectAll(self):
        """
        Private slot to handle the select all tab menu entry.
        """
        self.widget(self.__menuIndex).selectAll()
        
    def showLogTab(self, tabname):
        """
        Public method to show a particular Log-Viewer tab.
        
        @param tabname string naming the tab to be shown ("stdout", "stderr")
        """
        if tabname == "stdout":
            self.setCurrentIndex(self.__stdoutIndex)
        elif tabname == "stderr":
            self.setCurrentIndex(self.__stderrIndex)
        else:
            raise RuntimeError("wrong tabname given")
        
    def appendToStdout(self, txt):
        """
        Public slot to appand text to the "stdout" tab.
        
        @param txt text to be appended (string or QString)
        """
        self.__stdout.appendText(txt)
        QApplication.processEvents()
        
    def appendToStderr(self, txt):
        """
        Public slot to appand text to the "stderr" tab.
        
        @param txt text to be appended (string or QString)
        """
        self.__stderr.appendText(txt)
        QApplication.processEvents()
