# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a configuration dialog for the bookmarked files menu.
"""

from qt import *

from KdeQt import KQFileDialog, KQMessageBox

from BookmarkedFilesForm import BookmarkedFilesForm
import Utilities

class BookmarkedFilesDialog(BookmarkedFilesForm):
    """
    Class implementing a configuration dialog for the bookmarked files menu.
    """
    def __init__(self, bookmarks, parent=None):
        """
        Constructor
        
        @param bookmarks list of bookmarked files (QStringList)
        @param parent parent widget (QWidget)
        """
        BookmarkedFilesForm.__init__(self, parent, None, 1)
        
        self.bookmarks = QStringList(bookmarks)
        for bookmark in self.bookmarks:
            self.filesListBox.insertItem(bookmark)
            
        if len(self.bookmarks):
            self.filesListBox.setCurrentItem(0)
        
    def handleAdd(self):
        """
        Private slot to add a new entry.
        """
        bookmark = self.fileEdit.text()
        
        if bookmark.isEmpty():
            KQMessageBox.critical(self,
                self.trUtf8("Add bookmarked file"),
                self.trUtf8("You have to set a file to be bookmarked first."),
                self.trUtf8("OK"))
            return
            
        bookmark = QDir.convertSeparators(bookmark)
        self.filesListBox.insertItem(bookmark)
        self.bookmarks.append(bookmark)
        
    def handleChange(self):
        """
        Private slot to change an entry.
        """
        itm = self.filesListBox.currentItem()
        if itm == -1:
            return
            
        bookmark = self.fileEdit.text()
        
        if bookmark.isEmpty():
            KQMessageBox.critical(self,
                self.trUtf8("Change bookmarked file"),
                self.trUtf8("You have to set a bookmarked file to change it."),
                self.trUtf8("OK"))
            return
            
        bookmark = QDir.convertSeparators(bookmark)
        self.bookmarks[itm] = bookmark
        self.filesListBox.changeItem(bookmark, itm)
        
    def handleDelete(self):
        """
        Private slot to delete the selected entry.
        """
        itm = self.filesListBox.currentItem()
        if itm == -1:
            return
            
        del self.bookmarks[itm]
        self.filesListBox.removeItem(itm)
        
    def handleDown(self):
        """
        Private slot to move an entry down in the list.
        """
        curr = self.filesListBox.currentItem()
        self.swap(curr, curr+1)
        self.filesListBox.clear()
        for bookmark in self.bookmarks:
            self.filesListBox.insertItem(bookmark)
        self.filesListBox.setCurrentItem(curr+1)
        if curr+1 == len(self.bookmarks):
            self.downButton.setEnabled(0)
        self.upButton.setEnabled(1)
        
    def handleUp(self):
        """
        Private slot to move an entry up in the list.
        """
        curr = self.filesListBox.currentItem()
        self.swap(curr-1, curr)
        self.filesListBox.clear()
        for bookmark in self.bookmarks:
            self.filesListBox.insertItem(bookmark)
        self.filesListBox.setCurrentItem(curr-1)
        if curr-1 == 0:
            self.upButton.setEnabled(0)
        self.downButton.setEnabled(1)
        
    def handleFileSelect(self):
        """
        Private slot to handle the file selection via a file selection dialog.
        """
        bookmark = KQFileDialog.getOpenFileName()
        if not bookmark.isNull() and not bookmark.isEmpty():
            bookmark = QDir.convertSeparators(bookmark)
            self.fileEdit.setText(bookmark)
        
    def handleEntrySelected(self, index):
        """
        Private slot to set the lineedit depending on the selected entry.
        
        @param index the index of the selected entry (integer)
        """
        if index >= 0:
            bookmark = self.bookmarks[index]
            self.fileEdit.setText(bookmark)
            
            self.deleteButton.setEnabled(1)
            self.changeButton.setEnabled(1)
            
            if index != 0:
                self.upButton.setEnabled(1)
            else:
                self.upButton.setEnabled(0)
                
            if index+1 != len(self.bookmarks):
                self.downButton.setEnabled(1)
            else:
                self.downButton.setEnabled(0)
        else:
            self.fileEdit.clear()
            self.downButton.setEnabled(0)
            self.upButton.setEnabled(0)
            self.deleteButton.setEnabled(0)
            self.changeButton.setEnabled(0)
        
    def getBookmarkedFiles(self):
        """
        Public method to retrieve the tools list. 
        
        @return a list of tuples containing the menu text, the executable, 
            the executables arguments and a redirection flag
        """
        return self.bookmarks
        
    def swap(self, itm1, itm2):
        """
        Private method used two swap two list entries given by their index.
        
        @param itm1 index of first entry (int)
        @param itm2 index of second entry (int)
        """
        tmp = self.bookmarks[itm1]
        self.bookmarks[itm1] = self.bookmarks[itm2]
        self.bookmarks[itm2] = tmp
