/* BTP Climber - Dumps out a BTP tree
 * Copyright (C) 1999, 2000  The Regents of the University of Michigan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>

#include "btp.h"
#include "btp_node.h"
#include "b_conn.h"

#include "btp_debug.h"
#include "util.h"


#define BTPCLIMBER_TIMEOUT 30000

typedef struct _BCNode
{
  gchar* hostname;
  gint port;

  struct _BCNode* parent;
  GSList* children;
  gboolean done;

} BCNode;


static void     btpclimber_packet_func (BtpTree* tree, BtpNode* node, 
					BPacket* packet, gpointer user_data);
static void 	btpclimber_dump (void);
static void	btpclimber_dump_r (BCNode* bcnode);
static void	btpclimber_dump_pending_r(BCNode* bcnode);
static gboolean btpclimber_timeout (gpointer data);

static void     usage (char* prog, int exitval);

static gboolean edge_mode = FALSE;

static gint requests_pending = 0;
static gint total_nodes = 0;
static guint timeout;

static GSList* nodes;
static BCNode* root_node;

static BPeer* bpeer;
static Btp*   btp;


int 
main(int argc, char* argv[])
{
  extern char* optarg;
  extern int   optind;
  char c;

  GInetAddr* iface;

  GURL* url;

  GMainLoop* main_loop;
  BPacket* packet;

  /* Initialize RNG */
  srand(time(NULL));

  /* Parse args */
  while ((c = getopt(argc, argv, "e")) != -1) 
    {
      switch (c)
	{
	case 'e':
	  edge_mode = TRUE;
	  break;

	case 'h':
	case '?':
	  usage (argv[0], EXIT_SUCCESS);
	  break;
	}
    }
  if (argc == optind)
    usage (argv[0], EXIT_FAILURE);

  /* Parse the URL */
  url = gnet_url_new (argv[optind]);
  g_assert (url);

  /* Initialize btp (this isn't correct, but it works...) */
  iface = gnet_inetaddr_gethostaddr();
  g_assert (iface);
  gnet_inetaddr_set_port (iface, BTP_PORT);

  bpeer = b_peer_new (gnet_inetaddr_gethostname(), iface, FALSE);
  g_assert (bpeer);

  /* Connect to the root */
  btp = btp_join_passive (bpeer, url);
  if (!btp)
    g_error ("Could not connect to tree\n");

  /* Set up a packet callback (this writes over the btp packet func) */
  btp->tree->packet_func = btpclimber_packet_func;

  /* Send a child request */
  g_assert (btp->tree->root);

  packet = b_packet_new_info_request_neighbors (btp->tree->root);
  b_conn_send_packet (btp->tree->root->conn, packet);

  ++requests_pending;
  ++total_nodes;
  timeout = g_timeout_add (BTPCLIMBER_TIMEOUT, btpclimber_timeout , NULL);

  /* Start the main loop */
  main_loop = g_main_new(FALSE);
  g_main_run(main_loop);

  return 0;
}



static void
btpclimber_packet_func (BtpTree* tree, BtpNode* node, 
			BPacket* packet, gpointer user_data)
{
  BCNode* bcnode = NULL;
  BAddress* parent;
  GSList* children;
  GSList* i;

  g_assert (tree != NULL);
  g_assert (node != NULL);
  g_assert (packet != NULL);

  /* Ignore packets that aren't children info replies */
  if (! (packet->type == B_PACKET_TYPE_INFO_REPLY &&
	 packet->subtype == B_PACKET_INFO_SUBTYPE_NEIGHBORS))
    return;

  /* If the root node is NULL, assume this is the root node.  We do
     this so we get the canonical address of the root node, which
     makes things easier later.  It would be possible for a non-root
     node to attach to us (maliciously?), but that would just mean
     we'd get bad data. */
  if (root_node == NULL)
    {
      root_node = g_new0 (BCNode, 1);
      root_node->hostname = g_strdup (node->hostname);
      root_node->port = node->port;
      nodes = g_slist_prepend (nodes, root_node);

      bcnode = root_node;
    }

  else
    {
      /* Otherwise, find the node */
      for (i = nodes; i != NULL; i = i->next)
	{
	  BCNode* bcn = (BCNode*) i->data;

	  if (strcmp(bcn->hostname, node->hostname) == 0 && 
	      (bcn->port == node->port))
	    {
	      bcnode = bcn;
	      break;
	    }
	}
    }

  /* Make sure the node isn't NULL. */
  if (bcnode == NULL)
    {
      if (edge_mode) g_print ("# ");
      g_print ("Weird node: %s, %d\n", node->hostname, node->port);
      return;
    }

  /*    g_print ("Received response from %s, %d\n", node->hostname, node->port); */

  /* Mark it */
  --requests_pending;
  bcnode->done = TRUE;

  /* Read the neighbors addresses */
  if (b_packet_parse_info_reply_neighbors (packet, &parent, &children))
    {
      g_warning ("Received bad neighbors reply\n");
      return;
    }

  /* TODO: Check this parent versus previously found parent */
  b_address_delete (parent);  

  /* Save siblings */
  for (i = children; i != NULL; i = i->next)
    {
      BAddress* addr = (BAddress*) i->data;
      BtpNode* child;
      BCNode* bcchild;
      BPacket* packet;

      g_assert (addr != NULL);
      g_assert (addr->hostname != NULL);

      /* Create BtpNode */
      child = btp_tree_find_node (tree, addr->hostname, addr->port);
      if (child == NULL)
	child = btp_node_new (tree, addr->hostname, addr->port, NULL);
      g_assert (child);

      b_address_delete (addr);

      /* Send info request */
      packet = b_packet_new_info_request_neighbors (child);
      b_conn_send_packet (child->conn, packet);

      /* Create BCNode */
      bcchild = g_new0(BCNode, 1);
      bcchild->hostname = g_strdup(child->hostname);
      bcchild->port = child->port;
      bcchild->parent = bcnode;
      bcnode->children = g_slist_prepend (bcnode->children, bcchild);
      nodes = g_slist_prepend (nodes, bcchild);

      if (edge_mode)
	g_print ("%s:%d\t%s:%d\n", 
		 bcnode->hostname, bcnode->port,
		 bcchild->hostname, bcchild->port);

      ++requests_pending;
      ++total_nodes;
    }

  g_slist_free (children);

  /* Quit if we're done */
  if (requests_pending == 0)
    btpclimber_dump();

  /* Reset the timeout */
  g_source_remove(timeout);
  timeout = g_timeout_add(BTPCLIMBER_TIMEOUT, btpclimber_timeout, NULL);

  return;
}


/* ******************** */

static gint indent = 0;

static void
btpclimber_dump(void)
{
  if (root_node)
    {
      if (!edge_mode)
	btpclimber_dump_r(root_node);
    }
  else
    {
      if (edge_mode) g_print ("# ");
      g_print ("No response from root node\n");
    }

  g_print ("\n");
  if (edge_mode) g_print ("# ");
  g_print ("Total nodes: %d\n", total_nodes);
  if (edge_mode) g_print ("# ");
  g_print ("Requests still pending: %d\n", requests_pending);
  btpclimber_dump_pending_r (root_node);

  btp_leave (btp);
  b_peer_delete (bpeer);

  exit (EXIT_SUCCESS);
}


static void
btpclimber_dump_r(BCNode* bcnode)
{
  gint j;
  GSList* i;

  for (j = 0; j < indent; ++j)
    g_print (" ");

  g_print ("%s:%d\n", bcnode->hostname, bcnode->port);
  
  indent += 2;

  for (i = bcnode->children; i != NULL; i = i->next)
    btpclimber_dump_r((BCNode*) i->data);

  indent -= 2;
}


static void
btpclimber_dump_pending_r (BCNode* bcnode)
{
  GSList* i;
  if (!bcnode->done)
    {
      if (edge_mode) g_print ("#");
      g_print ("\t%s:%d\n", bcnode->hostname, bcnode->port);
    }

  for (i = bcnode->children; i != NULL; i = i->next)
    btpclimber_dump_pending_r((BCNode*) i->data);
}


static gboolean
btpclimber_timeout(gpointer data)
{
  if (edge_mode) g_print ("# ");
  g_print ("Timeout\n");
  btpclimber_dump();
  return FALSE;
}


static void
usage (char* prog, int exitval)
{
  fprintf (stderr, "Usage: %s [OPTION]... <URL>\n", prog);
  fprintf (stderr, "BTP tree climber\n");
  fprintf (stderr, "\t-e         print edges\n");
  exit (exitval);
}
