#include <Elementary.h>
#include "elm_priv.h"

#define ELM_TRANSIT_CHECK_OR_RETURN(transit, ...) \
   do { \
      if (!transit) { \
         CRITICAL("Elm_Transit " # transit " is NULL!"); \
         return __VA_ARGS__; \
      } \
      if (!EINA_MAGIC_CHECK(transit, ELM_TRANSIT_MAGIC)) { \
         EINA_MAGIC_FAIL(transit, ELM_TRANSIT_MAGIC); \
         return __VA_ARGS__; \
      } \
      if (transit->deleted){ \
         ERR("Elm_Transit " # transit " has already been deleted!"); \
         return __VA_ARGS__; \
      } \
   } while (0)

/**
 *
 * @defgroup Transit Transit
 * @ingroup Elementary
 *
 * Transit (see Warning below) is designed to set the various effects for the
 * Evas_Object such like translation, rotation, etc. For using Effects, Create
 * transit and insert effects which are interesting.
 * Once effects are inserted into transit, transit will manage those effects.
 * (ex deleting).
 *
 * Example:
 * @code
 * Elm_Transit *trans = elm_transit_add(5.0);
 * elm_transit_object_add(trans, obj);
 * void *effect_context = elm_transit_effect_translation_context_new(0.0, 0.0,
 *                                                               280.0, 280.0);
 * elm_transit_effect_add(transit,
 *                        elm_transit_effect_translation_op, effect_context,
 *                        elm_transit_effect_translation_context_free);
 * elm_transit_auto_reverse_set(transit, EINA_TRUE);
 * elm_transit_tween_mode_set(transit, ELM_TRANSIT_TWEEN_MODE_DECELERATE);
 * elm_transit_repeat_times_set(transit, -1);
 * @endcode
 *
 * @warning We strongly recomend to use elm_transit just when edje can not do
 * the trick. Edje has more advantage than Elm_Transit, it has more flexibility and
 * animations can be manipulated inside the theme.
 */

static const char _transit_key[] = "_elm_transit";

struct _Elm_Transit
{
#define ELM_TRANSIT_MAGIC 0xd27f190a
   EINA_MAGIC;

   Ecore_Animator *animator;
   Eina_List *effect_list;
   Eina_List *objs;
   Elm_Transit_Tween_Mode tween_mode;
   Eina_Bool auto_reverse:1;
   Eina_Bool block:1;
   Eina_Bool deleted:1;
   unsigned int effects_pending_del;
   int walking;
   struct
   {
      void (*func) (void *data, Elm_Transit *transit);
      void *arg;
   } del_data;
   struct
   {
      double duration;
      double begin;
      double current;
   } time;
   struct
   {
      int count;
      int current;
      Eina_Bool reverse;
   } repeat;
};

struct _Elm_Effect
{
   void (*animation_op) (void *data, Elm_Transit *transit, double progress);
   void (*user_data_free) (void *data, Elm_Transit *transit);
   void *user_data;
   Eina_Bool deleted:1;
};

struct _Elm_Obj_Data
{
   Eina_Bool state;
   Elm_Transit *transit;
};

typedef struct _Elm_Effect Elm_Effect;
typedef struct _Elm_Obj_Data Elm_Obj_Data;

static void
_elm_transit_object_remove_cb(void *data, Evas *e __UNUSED__, Evas_Object *obj, void *event_info __UNUSED__)
{
   Elm_Transit *transit = data;

   Elm_Obj_Data *obj_data = evas_object_data_del(obj, _transit_key);
   evas_object_pass_events_set(obj, obj_data->state);
   free(obj_data);

   transit->objs = eina_list_remove(transit->objs, obj);

   if (!transit->objs)
     elm_transit_del(transit);
}

static void
_elm_transit_object_remove(Elm_Transit *transit, Evas_Object *obj)
{
   Elm_Obj_Data *obj_data = evas_object_data_del(obj, _transit_key);
   evas_object_pass_events_set(obj, obj_data->state);
   free(obj_data);

   transit->objs = eina_list_remove(transit->objs, obj);

   evas_object_event_callback_del(obj, EVAS_CALLBACK_DEL,
                                  _elm_transit_object_remove_cb);
}

static void
_elm_transit_effect_del(Elm_Transit *transit, Elm_Effect *effect, Eina_List *elist)
{
   if (effect->user_data_free)
     effect->user_data_free(effect->user_data, transit);

   transit->effect_list = eina_list_remove_list(transit->effect_list, elist);
   free(effect);
}

static void
_remove_dead_effects(Elm_Transit *transit)
{
   Eina_List *elist, *elist_next;

   Elm_Effect *effect;

   EINA_LIST_FOREACH_SAFE(transit->effect_list, elist, elist_next, effect)
     {
        if (effect->deleted)
          {
             _elm_transit_effect_del(transit, effect, elist);
             transit->effects_pending_del--;
             if (!transit->effects_pending_del) return;
          }
     }
}

static void
_elm_transit_del(Elm_Transit *transit)
{
   Eina_List *elist, *elist_next;
   Elm_Effect *effect;

   if (transit->del_data.func)
     transit->del_data.func(transit->del_data.arg, transit);

   ecore_animator_del(transit->animator);

   EINA_LIST_FOREACH_SAFE(transit->effect_list, elist, elist_next, effect)
     _elm_transit_effect_del(transit, effect, elist);

   while (transit->objs)
     _elm_transit_object_remove(transit, eina_list_data_get(transit->objs));

   EINA_MAGIC_SET(transit, EINA_MAGIC_NONE);
   free(transit);
}

static void
_transit_animate_op(Elm_Transit *transit, double progress)
{
   Eina_List *elist;

   Elm_Effect *effect;

   transit->walking++;
   EINA_LIST_FOREACH(transit->effect_list, elist, effect)
     {
        if (transit->deleted) break;

        if (!effect->deleted)
          effect->animation_op(effect->user_data, transit, progress);
     }
   transit->walking--;

   if (transit->walking) return;

   if (transit->deleted) _elm_transit_del(transit);
   else if (transit->effects_pending_del) _remove_dead_effects(transit);
}

static Eina_Bool
_animator_animate_cb(void *data)
{
   Elm_Transit *transit = data;

   transit->time.current = ecore_loop_time_get();
   double elapsed_time = transit->time.current - transit->time.begin;

   if (elapsed_time > transit->time.duration)
     elapsed_time = transit->time.duration;

   double progress = elapsed_time / transit->time.duration;
   switch (transit->tween_mode)
     {
      case ELM_TRANSIT_TWEEN_MODE_ACCELERATE:
         progress = 1.0 - sin((ELM_PI / 2.0) + (progress * ELM_PI / 2.0));
         break;
      case ELM_TRANSIT_TWEEN_MODE_DECELERATE:
         progress = sin(progress * ELM_PI / 2.0);
         break;
      case ELM_TRANSIT_TWEEN_MODE_SINUSOIDAL:
         progress = (1.0 - cos(progress * ELM_PI)) / 2.0;
         break;
      default:
         break;
    }

   /* Reverse? */
   if (transit->repeat.reverse)
     progress = 1 - progress;

   if (transit->time.duration > 0)
     _transit_animate_op(transit, progress);

   /* Not end. Keep going. */
   if (elapsed_time < transit->time.duration)
     return ECORE_CALLBACK_RENEW;

   /* Repeat and reverse and time done! */
   if ((transit->repeat.current == transit->repeat.count)
      && (!transit->auto_reverse || transit->repeat.reverse))
     {
        elm_transit_del(transit);
	return ECORE_CALLBACK_CANCEL;
     }

   /* Repeat Case */
   if (!transit->auto_reverse || transit->repeat.reverse)
     {
        transit->repeat.current++;
        transit->repeat.reverse = EINA_FALSE;
     }
   else transit->repeat.reverse = EINA_TRUE;

   transit->time.begin = ecore_loop_time_get();

   return ECORE_CALLBACK_RENEW;
}

/**
 * Add new transit.
 *
 * @note Is not necessary to delete the transit object, it will be deleted at
 * the end of its operation.
 * @note The transit will start playing when the program enter in the main loop, is not
 * necessary to give a start to the transit.
 *
 * @param duration The duration of the transit in seconds. When transit starts
 * to run, it will last a @p duration time.
 * @return The transit object.
 *
 * @ingroup Transit
 */
EAPI Elm_Transit *
elm_transit_add(double duration)
{
   Elm_Transit *transit = ELM_NEW(Elm_Transit);

   if (!transit) return NULL;

   EINA_MAGIC_SET(transit, ELM_TRANSIT_MAGIC);

   elm_transit_tween_mode_set(transit, ELM_TRANSIT_TWEEN_MODE_LINEAR);

   transit->time.duration = duration;

   transit->time.begin = ecore_loop_time_get();
   transit->animator = ecore_animator_add(_animator_animate_cb, transit);

   return transit;
}

/**
 * Stops the animation and delete the @p transit object.
 *
 * Call this function if you wants to stop the animation before the duration
 * time. Make sure the @p transit object is still alive with
 * elm_transit_del_cb_set() function.
 * All added effects will be deleted, calling its repective data_free_cb
 * functions. The function setted by elm_tansit_del_cb_set() will be called.
 *
 * @see elm_transit_del_cb_set()
 *
 * @param transit The transit object to be deleted.
 *
 * @ingroup Transit
 * @warning Just call this function if you are sure the transit is alive.
 */
EAPI void
elm_transit_del(Elm_Transit *transit)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   if (transit->walking) transit->deleted = EINA_TRUE;
   else _elm_transit_del(transit);
}

/**
 * Add a new effect to the transit.
 *
 * @note The cb function and the data are the key to the effect. If you try to
 * add an already added effect, nothing is done.
 * @note After the first addition of an effect in @p transit, if its
 * effect list become empty again, the @p transit will be killed by
 * elm_transit_del(transit) function.
 *
 * Exemple:
 * @code
 * Elm_Transit *transit = elm_transit_add(5.0);
 * elm_transit_effect_add(transit,
 *                        elm_transit_effect_blend_op,
 *                        elm_transit_effect_blend_context_new(),
 *                        elm_transit_effect_blend_context_free);
 * @endcode
 *
 * @param transit The transit object.
 * @param cb The operation function. It is called when the animation begins,
 * it is the function that actually performs the animation. It is called with
 * the @p data, @p transit and the time progression of the animation (a double
 * value between 0.0 and 1.0).
 * @param data The context data of the effect.
 * @param data_free_cb The function to free the context data, it will be called
 * at the end of the effect, it must finalize the animation and free the
 * @p data.
 *
 * @ingroup Transit
 * @warning The transit free the context data at the and of the transition with
 * the data_free_cb function, do not use the context data in another transit.
 */
EAPI void
elm_transit_effect_add(Elm_Transit *transit, void (*cb)(void *data, Elm_Transit *transit, double progress), void *data, void (*data_free_cb)(void *data, Elm_Transit *transit))
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   EINA_SAFETY_ON_NULL_RETURN(cb);

   Elm_Effect *effect;

   Eina_List *elist;

   EINA_LIST_FOREACH(transit->effect_list, elist, effect)
     if ((effect->animation_op == cb) && (effect->user_data == data)) return;

   effect = ELM_NEW(Elm_Effect);
   if (!effect) return;

   effect->user_data_free = data_free_cb;
   effect->animation_op = cb;
   effect->user_data = data;

   transit->effect_list = eina_list_append(transit->effect_list, effect);
}

/**
 * Delete an added effect.
 *
 * This function will remove the effect from the @p transit, calling the
 * data_free_cb to free the @p data.
 *
 * @see elm_transit_effect_add()
 *
 * @note If the effect is not found, nothing is done.
 * @note If the effect list become empty, this function will call
 * elm_transit_del(transit), that is, it will kill the @p transit.
 *
 * @param transit The transit object.
 * @param cb The operation function.
 * @param data The context data of the effect.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_del(Elm_Transit *transit, void (*cb)(void *data, Elm_Transit *transit, double progress), void *data)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   EINA_SAFETY_ON_NULL_RETURN(cb);

   Eina_List *elist, *elist_next;
   Elm_Effect *effect;

   EINA_LIST_FOREACH_SAFE(transit->effect_list, elist, elist_next, effect)
     {
        if ((effect->animation_op == cb) && (effect->user_data == data))
          {
             if (transit->walking)
               {
                  effect->deleted = EINA_TRUE;
                  transit->effects_pending_del++;
               }
             else
               {
                  _elm_transit_effect_del(transit, effect, elist);
                  if (!transit->effect_list) elm_transit_del(transit);
               }
             return;
          }
     }
}

/**
 * Add new object to apply the effects.
 *
 * @note After the first addition of an object in @p transit, if its
 * object list become empty again, the @p transit will be killed by
 * elm_transit_del(transit) function.
 * @note If the @p obj belongs to another transit, the @p obj will be
 * removed from it and it will only belong to the @p transit. If the old
 * transit stays without objects, it will die.
 * @note When you add an object into the @p transit, its state from
 * evas_object_pass_events_get(obj) is saved, and it is applied when the
 * transit ends, if you change this state whith evas_object_pass_events_set()
 * after add the object, this state will change again when @p transit stops to
 * run.
 *
 * @param transit The transit object.
 * @param obj Object to be animated.
 *
 * @ingroup Transit
 * @warning See the documentation of the effect if is safe add or remove
 * an object after @p transit begins to run.
 */
EAPI void
elm_transit_object_add(Elm_Transit *transit, Evas_Object *obj)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   EINA_SAFETY_ON_NULL_RETURN(obj);

   Elm_Obj_Data *obj_data = evas_object_data_get(obj, _transit_key);

   if (obj_data)
     {
        if (obj_data->transit == transit) return;
        elm_transit_object_remove(obj_data->transit, obj);
     }

   obj_data = ELM_NEW(Elm_Obj_Data);
   obj_data->state = evas_object_pass_events_get(obj);
   obj_data->transit = transit;
   evas_object_data_set(obj, _transit_key, obj_data);

   transit->objs = eina_list_append(transit->objs, obj);


   if (transit->block)
     evas_object_pass_events_set(obj, EINA_TRUE);

   evas_object_event_callback_add(obj, EVAS_CALLBACK_DEL,
                                  _elm_transit_object_remove_cb, transit);
}

/**
 * Removes an added object from the transit.
 *
 * @note If the @p obj is not in the @p transit, nothing is done.
 * @note If the list become empty, this function will call
 * elm_transit_del(transit), that is, it will kill the @p transit.
 *
 * @param transit The transit object.
 * @param obj Object to be removed from @p transit.
 *
 * @ingroup Transit
 * @warning See the documentation of the effect if is safe add or remove
 * an object after @p transit begins to run.
 */
EAPI void
elm_transit_object_remove(Elm_Transit *transit, Evas_Object *obj)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   EINA_SAFETY_ON_NULL_RETURN(obj);

   Elm_Obj_Data *obj_data = evas_object_data_get(obj, _transit_key);

   if ((!obj_data) || (obj_data->transit != transit)) return;

   _elm_transit_object_remove(transit, obj);

   if (!transit->objs)
     elm_transit_del(transit);
}

/**
 * Get the objects of the transit.
 *
 * @param transit The transit object.
 * @return a Eina_List with the objects from the transit.
 *
 * @ingroup Transit
 */
EAPI const Eina_List *
elm_transit_objects_get(const Elm_Transit *transit)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit, NULL);

   return transit->objs;
}

/**
 * Set the event blocked when transit is operating.
 *
 * If @p disabled is EINA_TRUE, the objects of the transit will not receives
 * events from mouse and keyboard during the animation.
 * @note When you add an object with elm_transit_object_add(), its state from
 * evas_object_pass_events_get(obj) is saved, and it is applied when the
 * transit ends, if you change this state with evas_object_pass_events_set()
 * after add the object, this state will change again when @p transit stops to
 * run.
 *
 * @param transit The transit object.
 * @param disabled Disable or enable.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_event_block_set(Elm_Transit *transit, Eina_Bool disabled)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   if (transit->block == disabled) return;

   Evas_Object *obj;
   Eina_List *elist;

   transit->block = disabled;

   if (!transit->block)
     {
        EINA_LIST_FOREACH(transit->objs, elist, obj)
          {
             Elm_Obj_Data *obj_data = evas_object_data_get(obj, _transit_key);
             evas_object_pass_events_set(obj, obj_data->state);
          }
     }
   else
     {
        EINA_LIST_FOREACH(transit->objs, elist, obj)
          evas_object_pass_events_set(obj, EINA_TRUE);
     }
}

/**
 * Get the value of event blocked status.
 *
 * @see elm_transit_event_block_set()
 *
 * @param transit The Transit object
 * @return EINA_TRUE, when event block is enabled. If @p transit is NULL
 * EINA_FALSE is returned
 *
 * @ingroup Transit
 */
EAPI Eina_Bool
elm_transit_event_block_get(const Elm_Transit *transit)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit, EINA_FALSE);

   return transit->block;
}

/**
 * Set the user-callback function when the transit is deleted.
 *
 * @note Using this function twice will overwrite the first function setted.
 * @note the @p transit object will be deleted after call @p cb function.
 *
 * @param transit The transit object.
 * @param cb Callback function pointer. This function will be called before
 * the deletion of the transit.
 * @param data Callback funtion user data. It is the @p op parameter.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_del_cb_set(Elm_Transit *transit, void (*cb) (void *data, Elm_Transit *transit), void *data)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   transit->del_data.func = cb;
   transit->del_data.arg = data;
}

/**
 * Set reverse effect automatically.
 *
 * If auto reverse is setted, after running the effects with the progress
 * parameter from 0 to 1, it will call the effecs again with the progress
 * from 1 to 0. The transit will last for a time iqual to (2 * duration * repeat),
 * where the duration was setted with the function elm_transit_add and
 * the repeat with the function elm_transit_repeat_times_set().
 *
 * @param transit The transit object.
 * @param reverse EINA_TRUE means the auto_reverse is on.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_auto_reverse_set(Elm_Transit *transit, Eina_Bool reverse)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   transit->auto_reverse = reverse;
}

/**
 * Get if the auto reverse is on.
 *
 * @see elm_transit_auto_reverse_set()
 *
 * @param transit The transit object.
 * @return EINA_TRUE means auto reverse is on. If @p transit is NULL
 * EINA_FALSE is returned
 *
 * @ingroup Transit
 */
EAPI Eina_Bool
elm_transit_auto_reverse_get(Elm_Transit *transit)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit, EINA_FALSE);

   return transit->auto_reverse;
}

/**
 * Set the transit repeat count. Effect will be repeated by repeat count.
 *
 * This function sets the number of repetition the transit will run after
 * the first one, that is, if @p repeat is 1, the transit will run 2 times.
 * If the @p repeat is a negative number, it will repeat infinite times.
 *
 * @note If this function is called during the transit execution, the transit
 * will run @p repeat times, ignoring the times it already performed.
 *
 * @param transit The transit object
 * @param repeat Repeat count
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_repeat_times_set(Elm_Transit *transit, int repeat)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   transit->repeat.count = repeat;
   transit->repeat.current = 0;
}

/**
 * Get the transit repeat count.
 *
 * @see elm_transit_repeat_times_set()
 *
 * @param transit The Transit object.
 * @return The repeat count. If @p transit is NULL
 * 0 is returned
 *
 * @ingroup Transit
 */
EAPI int
elm_transit_repeat_times_get(Elm_Transit *transit)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit, 0);

   return transit->repeat.count;
}

/**
 * Set the transit animation acceleration type.
 *
 * This function sets the tween mode of the transit that can be:
 * ELM_TRANSIT_TWEEN_MODE_LINEAR - The default mode.
 * ELM_TRANSIT_TWEEN_MODE_SINUSOIDAL - Starts in accelerate mode and ends decelerating.
 * ELM_TRANSIT_TWEEN_MODE_DECELERATE - The animation will be slowed over time.
 * ELM_TRANSIT_TWEEN_MODE_ACCELERATE - The animation will accelerate over time.
 *
 * @param transit The transit object.
 * @param tween_mode The tween type.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_tween_mode_set(Elm_Transit *transit, Elm_Transit_Tween_Mode tween_mode)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit);

   transit->tween_mode = tween_mode;
}

/**
 * Get the transit animation acceleration type.
 *
 * @note @p transit can not be NULL
 *
 * @param transit The transit object.
 * @return The tween type. If @p transit is NULL
 * ELM_TRANSIT_TWEEN_MODE_LINEAR is returned.
 *
 * @ingroup Transit
 */
EAPI Elm_Transit_Tween_Mode
elm_transit_tween_mode_get(Elm_Transit *transit)
{
   ELM_TRANSIT_CHECK_OR_RETURN(transit, ELM_TRANSIT_TWEEN_MODE_LINEAR);

   return transit->tween_mode;
}

///////////////////////////////////////////////////////////////////////////////
//Resizing FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Resizing Elm_Fx_Resizing;

struct _Elm_Fx_Resizing
{
   struct _size
     {
        Evas_Coord w, h;
     } from, to;
};

/**
 * The Free function to Resizing Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Risizing effect, which consists of functions:
 * - elm_transit_effect_resizing_context_new()
 * - elm_transit_effect_resizing_op()
 * - elm_transit_effect_resizing_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Resizing context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_resizing_context_free(void *data, Elm_Transit *transit __UNUSED__)
{
   free(data);
}

/**
 * Operation function to the Resizing Effect.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note This effect will be applied to the objects that are in the transit,
 * If you change the set of objects in the transit with  elm_transit_object_add()
 * or elm_transit_object_remove(), the set of objects affected by this effect
 * will be changed too.
 *
 * This function belongs to the Risizing effect, which consists of functions:
 * - elm_transit_effect_resizing_context_new()
 * - elm_transit_effect_resizing_op()
 * - elm_transit_effect_resizing_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Resizing context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_resizing_op(void *data, Elm_Transit *transit, double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Evas_Coord w, h;
   Evas_Object *obj;
   Eina_List *elist;

   Elm_Fx_Resizing *resizing = data;

   w = resizing->from.w + (resizing->to.w * progress);
   h = resizing->from.h + (resizing->to.h * progress);

   EINA_LIST_FOREACH(transit->objs, elist, obj)
     evas_object_resize(obj, w, h);
}

/**
 * Get a new context data of Resizing Effect.
 *
 * This function belongs to the Risizing effect, which consists of functions:
 * - elm_transit_effect_resizing_context_new()
 * - elm_transit_effect_resizing_op()
 * - elm_transit_effect_resizing_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param from_w Object width size when effect begins.
 * @param from_h Object height size when effect begins.
 * @param to_w Object width size when effect ends.
 * @param to_h Object height size when effect ends.
 * @return Resizing effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_resizing_context_new(Evas_Coord from_w, Evas_Coord from_h, Evas_Coord to_w, Evas_Coord to_h)
{
   Elm_Fx_Resizing *resizing;

   resizing = ELM_NEW(Elm_Fx_Resizing);
   if (!resizing) return NULL;

   resizing->from.w = from_w;
   resizing->from.h = from_h;
   resizing->to.w = to_w - from_w;
   resizing->to.h = to_h - from_h;

   return resizing;
}

///////////////////////////////////////////////////////////////////////////////
//Translation FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Translation Elm_Fx_Translation;
typedef struct _Elm_Fx_Translation_Node Elm_Fx_Translation_Node;

struct _Elm_Fx_Translation_Node
{
   Evas_Object *obj;
   Evas_Coord x, y;
};

struct _Elm_Fx_Translation
{
   struct _position_variation
     {
        Evas_Coord dx, dy;
     } from, to;
   Eina_List *nodes;
};

static void
_translation_object_del_cb(void *data, Evas *e __UNUSED__, Evas_Object *obj, void *event_info __UNUSED__)
{
   Elm_Fx_Translation *translation = data;
   Eina_List *elist;
   Elm_Fx_Translation_Node *translation_node;

   EINA_LIST_FOREACH(translation->nodes, elist, translation_node)
     {
        if (translation_node->obj != obj)
          continue;

        translation->nodes = eina_list_remove_list(translation->nodes, elist);
        free(translation_node);
        break;
     }
}

static Eina_List *
_translation_nodes_build(Elm_Transit *transit, Elm_Fx_Translation *translation)
{
   Elm_Fx_Translation_Node *translation_node;
   const Eina_List *elist;
   Evas_Object *obj;
   Eina_List *data_list = NULL;
   const Eina_List *objs = elm_transit_objects_get(transit);

   EINA_LIST_FOREACH(objs, elist, obj)
     {
        translation_node = ELM_NEW(Elm_Fx_Translation_Node);
        if (!translation_node)
          {
             eina_list_free(data_list);
             return NULL;
          }
        translation_node->obj = obj;
        evas_object_geometry_get(obj, &(translation_node->x),
                                 &(translation_node->y), NULL, NULL);
        data_list = eina_list_append(data_list, translation_node);
        evas_object_event_callback_add(obj, EVAS_CALLBACK_DEL,
                                 _translation_object_del_cb, translation);
     }

   return data_list;
}

/**
 * The Free function to Translation Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Translation effect, which consists of functions:
 * - elm_transit_effect_translation_context_new()
 * - elm_transit_effect_translation_op()
 * - elm_transit_effect_translation_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Translation context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_translation_context_free(void *data, Elm_Transit *transit __UNUSED__)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   Elm_Fx_Translation *translation = data;
   Eina_List *elist, *elist_next;
   Elm_Fx_Translation_Node *translation_node;

   EINA_LIST_FOREACH_SAFE(translation->nodes,
                          elist, elist_next, translation_node)
     {
        evas_object_event_callback_del(translation_node->obj,
                             EVAS_CALLBACK_DEL, _translation_object_del_cb);
        translation->nodes = eina_list_remove_list(translation->nodes, elist);
        free(translation_node);
     }
   free(translation);
}

/**
 * Operation function to the Translation Effect.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note When this function begins to be called, it gets the current objects in
 * the transit, that is, elm_transit_object_remove() and elm_transit_object_add()
 * will not cause any changes in the set of objects that this effect is being
 * applied if these functions are called after the @p transit starts to run.
 *
 * This function belongs to the Translation effect, which consists of functions:
 * - elm_transit_effect_translation_context_new()
 * - elm_transit_effect_translation_op()
 * - elm_transit_effect_translation_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Translation context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 * @warning Is higher recommended just create a transit with this effect when
 * the window that the objects of the transit belongs has already been created.
 * This is because this effect needs the geometry information about the objects,
 * and if the window was not created yet, it can get a wrong information.
 */
EAPI void
elm_transit_effect_translation_op(void *data, Elm_Transit *transit, double progress __UNUSED__)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Evas_Coord x, y;
   Elm_Fx_Translation *translation = data;
   Elm_Fx_Translation_Node *translation_node;
   Eina_List *elist;

   if (!translation->nodes)
     translation->nodes = _translation_nodes_build(transit, translation);

   EINA_LIST_FOREACH(translation->nodes, elist, translation_node)
     {
        x = translation_node->x + translation->from.dx \
            + (translation->to.dx * progress);
        y = translation_node->y + translation->from.dy \
            + (translation->to.dy * progress);
        evas_object_move(translation_node->obj, x, y);
     }
}

/**
 * Get a new context data of Translation Effect
 *
 * This function belongs to the Translation effect, which consists of functions:
 * - elm_transit_effect_translation_context_new()
 * - elm_transit_effect_translation_op()
 * - elm_transit_effect_translation_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param from_dx X Position variation when effect begins.
 * @param from_dy Y Position variation when effect begins.
 * @param to_dx X Position variation when effect ends.
 * @param to_dy Y Position variation when effect ends.
 * @return Translation effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_translation_context_new(Evas_Coord from_dx, Evas_Coord from_dy, Evas_Coord to_dx, Evas_Coord to_dy)
{
   Elm_Fx_Translation *translation;

   translation = ELM_NEW(Elm_Fx_Translation);

   if (!translation) return NULL;

   translation->from.dx = from_dx;
   translation->from.dy = from_dy;
   translation->to.dx = to_dx - from_dx;
   translation->to.dy = to_dy - from_dy;

   return translation;
}

///////////////////////////////////////////////////////////////////////////////
//Zoom FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Zoom Elm_Fx_Zoom;

struct _Elm_Fx_Zoom
{
   float from, to;
};

/**
 * The Free function to Zoom Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Zoom effect, which consists of functions:
 * - elm_transit_effect_zoom_context_new()
 * - elm_transit_effect_zoom_op()
 * - elm_transit_effect_zoom_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Zoom context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_zoom_context_free(void *data, Elm_Transit *transit __UNUSED__)
{
   free(data);
}

/**
 * Operation function to the Zoom Effect
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note This effect will be applied to the objects that are in the transit,
 * If you change the set of objects in the transit with  elm_transit_object_add()
 * or elm_transit_object_remove(), the set of objects affected by this effect
 * will be changed too.
 *
 * This function belongs to the Zoom effect, which consists of functions:
 * - elm_transit_effect_zoom_context_new()
 * - elm_transit_effect_zoom_op()
 * - elm_transit_effect_zoom_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Zoom context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 * @warning Is higher recommended just create a transit with this effect when
 * the window that the objects of the transit belongs has already been created.
 * This is because this effect needs the geometry information about the objects,
 * and if the window was not created yet, it can get a wrong information.
 * @warning Is not recommended remove or add an object after the transit begins
 * to run, because the order of the objects will be affected.
 */
EAPI void
elm_transit_effect_zoom_op(void *data, Elm_Transit *transit , double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Evas_Object *obj;
   Eina_List *elist;

   Elm_Fx_Zoom *zoom = data;

   Evas_Map *map;

   Evas_Coord x, y, w, h;

   map = evas_map_new(4);
   if (!map) return;

   EINA_LIST_FOREACH(transit->objs, elist, obj)
     {
        evas_object_geometry_get(obj, &x, &y, &w, &h);
        evas_map_smooth_set(map, EINA_TRUE);
        evas_map_util_points_populate_from_object_full(map, obj, zoom->from + \
                                                       (progress * zoom->to));
        evas_map_util_3d_perspective(map, x + (w / 2), y + (h / 2), 0, 10000);
        evas_object_map_set(obj, map);
        evas_object_map_enable_set(obj, EINA_TRUE);
     }
   evas_map_free(map);
}

/**
 * Get a new context data of Zoom Effect
 *
 * @see elm_transit_effect_add()
 *
 * This function belongs to the Zoom effect, which consists of functions:
 * - elm_transit_effect_zoom_context_new()
 * - elm_transit_effect_zoom_op()
 * - elm_transit_effect_zoom_context_free()
 *
 * @param from_rate Scale rate when effect begins (1 is current rate).
 * @param to_rate Scale rate when effect ends.
 * @return Zoom effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_zoom_context_new(float from_rate, float to_rate)
{
   Elm_Fx_Zoom *zoom;

   zoom = ELM_NEW(Elm_Fx_Zoom);
   if (!zoom) return NULL;

   zoom->from = (10000 - (from_rate * 10000)) * (1 / from_rate);
   zoom->to = ((10000 - (to_rate * 10000)) * (1 / to_rate)) - zoom->from;

   return zoom;

}

///////////////////////////////////////////////////////////////////////////////
//Flip FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Flip Elm_Fx_Flip;

struct _Elm_Fx_Flip
{
   Elm_Fx_Flip_Axis axis;
   Eina_Bool cw:1;
};

/**
 * The Free function to Flip Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Flip effect, which consists of functions:
 * - elm_transit_effect_flip_context_new()
 * - elm_transit_effect_flip_op()
 * - elm_transit_effect_flip_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Flip context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_flip_context_free(void *data, Elm_Transit *transit)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Evas_Object *front, *back;
   int i;

   int count = eina_list_count(transit->objs);
   for(i = 0; i < count-1; i+=2)
     {
        front = eina_list_nth(transit->objs, i);
        back = eina_list_nth(transit->objs, i+1);
        evas_object_map_enable_set(front, EINA_FALSE);
        evas_object_map_enable_set(back, EINA_FALSE);
     }

   free(data);
}

/**
 * Operation function to the Flip Effect.
 *
 * This effect is applied to each pair of objects in the order they are listed
 * in the transit list of objects. The first object in the pair will be the
 * "front" object and the second will be the "back" object.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note This effect will be applied to the objects that are in the transit,
 * If you change the set of objects in the transit with  elm_transit_object_add()
 * or elm_transit_object_remove(), the set of objects affected by this effect
 * will be changed too.
 *
 * This function belongs to the Flip effect, which consists of functions:
 * - elm_transit_effect_flip_context_new()
 * - elm_transit_effect_flip_op()
 * - elm_transit_effect_flip_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Flip context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 * @warning Is higher recommended just create a transit with this effect when
 * the window that the objects of the transit belongs has already been created.
 * This is because this effect needs the geometry information about the objects,
 * and if the window was not created yet, it can get a wrong information.
 * @warning Is not recommended remove or add an object after the transit begins
 * to run, because the order of the objects will be affected.
 */
EAPI void
elm_transit_effect_flip_op(void *data, Elm_Transit *transit, double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Evas_Object *obj, *front, *back;
   int count;
   int i;

   Elm_Fx_Flip *flip = data;

   Evas_Map *map;

   float degree;

   Evas_Coord x, y, w, h;

   map = evas_map_new(4);
   if (!map) return;

   if (flip->cw)
     degree = (float)(progress * 180);
   else
     degree = (float)(progress * -180);

   count = eina_list_count(transit->objs);
   for(i = 0; i < count-1; i+=2)
     {
        front = eina_list_nth(transit->objs, i);
        back = eina_list_nth(transit->objs, i+1);

        if ((degree < 90) && (degree > -90))
          {
             obj = front;
             evas_object_hide(back);
             evas_object_show(front);
          }
        else
          {
             obj = back;
             evas_object_hide(front);
             evas_object_show(back);
          }

        evas_map_smooth_set(map, EINA_TRUE);
        evas_map_util_points_populate_from_object_full(map, obj, 0);
        evas_object_geometry_get(obj, &x, &y, &w, &h);
        Evas_Coord half_w = (w / 2);

        Evas_Coord half_h = (h / 2);

        if (flip->axis == ELM_TRANSIT_EFFECT_FLIP_AXIS_Y)
          {
             if ((degree >= 90) || (degree <= -90))
               {
                  evas_map_point_image_uv_set(map, 0, w, 0);
                  evas_map_point_image_uv_set(map, 1, 0, 0);
                  evas_map_point_image_uv_set(map, 2, 0, h);
                  evas_map_point_image_uv_set(map, 3, w, h);
               }
             evas_map_util_3d_rotate(map, 0, degree,
                                     0, x + half_w, y + half_h, 0);
          }
        else
          {
             if ((degree >= 90) || (degree <= -90))
               {
                  evas_map_point_image_uv_set(map, 0, 0, h);
                  evas_map_point_image_uv_set(map, 1, w, h);
                  evas_map_point_image_uv_set(map, 2, w, 0);
                  evas_map_point_image_uv_set(map, 3, 0, 0);
               }
             evas_map_util_3d_rotate(map, degree,
                                     0, 0, x + half_w, y + half_h, 0);
          }
        evas_map_util_3d_perspective(map, x + half_w, y + half_h, 0, 10000);
        evas_object_map_enable_set(front, EINA_TRUE);
        evas_object_map_enable_set(back, EINA_TRUE);
        evas_object_map_set(obj, map);
     }
   evas_map_free(map);
}

/**
 * Get a new context data of Flip Effect
 *
 * This function belongs to the Flip effect, which consists of functions:
 * - elm_transit_effect_flip_context_new()
 * - elm_transit_effect_flip_op()
 * - elm_transit_effect_flip_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param axis Flipping Axis(X or Y).
 * @param cw Flipping Direction. EINA_TRUE is clock-wise.
 * @return Flip effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_flip_context_new(Elm_Fx_Flip_Axis axis, Eina_Bool cw)
{
   Elm_Fx_Flip *flip;

   flip = ELM_NEW(Elm_Fx_Flip);
   if (!flip) return NULL;

   flip->cw = cw;
   flip->axis = axis;

   return flip;
}

///////////////////////////////////////////////////////////////////////////////
//ResizableFlip FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Resizable_Flip Elm_Fx_ResizableFlip;
typedef struct _Elm_Fx_Resizable_Flip_Node Elm_Fx_ResizableFlip_Node;

struct _Elm_Fx_Resizable_Flip_Node
{
   Evas_Object *front;
   Evas_Object *back;
   struct _vector2d
     {
        float x, y;
     } from_pos, from_size, to_pos, to_size;
};

struct _Elm_Fx_Resizable_Flip
{
   Eina_List *nodes;
   Eina_Bool cw:1;
   Elm_Fx_Flip_Axis axis;
};

static void
_resizable_flip_object_del_cb(void *data, Evas *e __UNUSED__, Evas_Object *obj, void *event_info __UNUSED__)
{
   Elm_Fx_ResizableFlip *resizable_flip = data;
   Eina_List *elist;
   Elm_Fx_ResizableFlip_Node *resizable_flip_node;

   EINA_LIST_FOREACH(resizable_flip->nodes, elist, resizable_flip_node)
     {
        if (resizable_flip_node->front == obj)
          evas_object_event_callback_del(resizable_flip_node->back,
                             EVAS_CALLBACK_DEL, _resizable_flip_object_del_cb);
        else if (resizable_flip_node->back == obj)
          evas_object_event_callback_del(resizable_flip_node->front,
                             EVAS_CALLBACK_DEL, _resizable_flip_object_del_cb);
        else continue;

        resizable_flip->nodes = eina_list_remove_list(resizable_flip->nodes,
                                                      elist);
        free(resizable_flip_node);
        break;
     }
}

static Eina_List *
_resizable_flip_nodes_build(Elm_Transit *transit, Elm_Fx_ResizableFlip *resizable_flip)
{
   Elm_Fx_ResizableFlip_Node *resizable_flip_node;
   Eina_List *data_list = NULL;

   Evas_Coord front_x, front_y, front_w, front_h;
   Evas_Coord back_x, back_y, back_w, back_h;

   int i, count;

   count = eina_list_count(transit->objs);
   for (i = 0; i < count-1; i+=2)
      {
         resizable_flip_node = ELM_NEW(Elm_Fx_ResizableFlip_Node);
         if (!resizable_flip_node)
           {
              eina_list_free(data_list);
              return NULL;
           }

         resizable_flip_node->front = eina_list_nth(transit->objs, i);
         resizable_flip_node->back = eina_list_nth(transit->objs, i+1);

         evas_object_geometry_get(resizable_flip_node->front,
                                  &front_x, &front_y, &front_w, &front_h);
         evas_object_geometry_get(resizable_flip_node->back,
                                  &back_x, &back_y, &back_w, &back_h);

         resizable_flip_node->from_pos.x = front_x;
         resizable_flip_node->from_pos.y = front_y;
         resizable_flip_node->to_pos.x = back_x - front_x;
         resizable_flip_node->to_pos.y = back_y - front_y;

         resizable_flip_node->from_size.x = front_w;
         resizable_flip_node->from_size.y = front_h;
         resizable_flip_node->to_size.x = back_w - front_w;
         resizable_flip_node->to_size.y = back_h - front_h;

         data_list = eina_list_append(data_list, resizable_flip_node);

         evas_object_event_callback_add(resizable_flip_node->back,
             EVAS_CALLBACK_DEL, _resizable_flip_object_del_cb, resizable_flip);
         evas_object_event_callback_add(resizable_flip_node->front,
             EVAS_CALLBACK_DEL, _resizable_flip_object_del_cb, resizable_flip);
      }

   return data_list;
}

static void
_set_image_uv_by_axis_y(Evas_Map *map, Elm_Fx_ResizableFlip_Node *flip, float degree)
{
   if ((degree >= 90) || (degree <= -90))
     {
        evas_map_point_image_uv_set(map, 0,
                                    (flip->from_size.x * 2) + flip->to_size.x,
                                    0);
        evas_map_point_image_uv_set(map, 1, 0, 0);
        evas_map_point_image_uv_set(map, 2, 0,
                                    (flip->from_size.y * 2) + flip->to_size.y);
        evas_map_point_image_uv_set(map, 3,
                                    (flip->from_size.x * 2) + flip->to_size.x,
                                    (flip->from_size.y * 2) + flip->to_size.y);
     }
   else
     {
        evas_map_point_image_uv_set(map, 0, 0, 0);
        evas_map_point_image_uv_set(map, 1, flip->from_size.x, 0);
        evas_map_point_image_uv_set(map, 2, flip->from_size.x,
                                    flip->from_size.y);
        evas_map_point_image_uv_set(map, 3, 0, flip->to_size.y);
     }
}

static void
_set_image_uv_by_axis_x(Evas_Map *map, Elm_Fx_ResizableFlip_Node *flip, float degree)
{
   if ((degree >= 90) || (degree <= -90))
     {
        evas_map_point_image_uv_set(map, 0, 0,
                                    (flip->from_size.y * 2) + flip->to_size.y);
        evas_map_point_image_uv_set(map, 1,
                                    (flip->from_size.x * 2) + flip->to_size.x,
                                    (flip->from_size.y * 2) + flip->to_size.y);
        evas_map_point_image_uv_set(map, 2,
                                    (flip->from_size.x * 2) + flip->to_size.x,
                                    0);
        evas_map_point_image_uv_set(map, 3, 0, 0);
     }
   else
     {
        evas_map_point_image_uv_set(map, 0, 0, 0);
        evas_map_point_image_uv_set(map, 1, flip->from_size.x, 0);
        evas_map_point_image_uv_set(map, 2, flip->from_size.x,
                                    flip->from_size.y);
        evas_map_point_image_uv_set(map, 3, 0, flip->to_size.y);
     }
}

/**
 * The Free function to Resizable Flip Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Resizable Flip effect, which consists of
 * functions:
 * - elm_transit_effect_resizable_flip_context_new()
 * - elm_transit_effect_resizable_flip_op()
 * - elm_transit_effect_resizing_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Resizable Flip context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 * @warning Is higher recommended just create a transit with this effect when
 * the window that the objects of the transit belongs has already been created.
 * This is because this effect needs the geometry information about the objects,
 * and if the window was not created yet, it can get a wrong information.
 * @warning Is not recommended remove or add an object after the transit begins
 * to run, because the order of the objects will be affected.
 */
EAPI void
elm_transit_effect_resizable_flip_context_free(void *data, Elm_Transit *transit __UNUSED__)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   Elm_Fx_ResizableFlip *resizable_flip = data;
   Eina_List *elist, *elist_next;
   Elm_Fx_ResizableFlip_Node *resizable_flip_node;

   EINA_LIST_FOREACH_SAFE(resizable_flip->nodes,
                          elist, elist_next, resizable_flip_node)
     {
        evas_object_map_enable_set(resizable_flip_node->front, EINA_FALSE);
        evas_object_map_enable_set(resizable_flip_node->back, EINA_FALSE);

        resizable_flip->nodes = eina_list_remove_list(resizable_flip->nodes,
                                                      elist);

        evas_object_event_callback_del(resizable_flip_node->back,
                             EVAS_CALLBACK_DEL, _resizable_flip_object_del_cb);
        evas_object_event_callback_del(resizable_flip_node->front,
                             EVAS_CALLBACK_DEL, _resizable_flip_object_del_cb);
        free(resizable_flip_node);
     }
   free(resizable_flip);
}

/**
 * Operation function to the Resizable Flip Effect
 *
 * This effect is applied to each pair of objects in the order they are listed
 * in the transit list of objects. The first object in the pair will be the
 * "front" object and the second will be the "back" object.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note When this function begins to be called, it gets the current objects in
 * the transit, that is, elm_transit_object_remove() and elm_transit_object_add()
 * will not cause any changes in the set of objects that this effect is being
 * applied if these functions are called after the @p transit starts to run.
 *
 * This function belongs to the Resizable Flip effect, which consists of
 * functions:
 * - elm_transit_effect_resizable_flip_context_new()
 * - elm_transit_effect_resizable_flip_op()
 * - elm_transit_effect_resizing_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Resizable Flip context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 * @warning Is higher recommended just create a transit with this effect when
 * the window that the objects of the transit belongs has already been created.
 * This is because this effect needs the geometry information about the objects,
 * and if the window was not created yet, it can get a wrong information.
 */
EAPI void
elm_transit_effect_resizable_flip_op(void *data, Elm_Transit *transit __UNUSED__, double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   Evas_Map *map;
   Evas_Object *obj;
   float x, y, w, h;
   float degree;
   Evas_Coord half_w, half_h;

   Elm_Fx_ResizableFlip *resizable_flip = data;
   Elm_Fx_ResizableFlip_Node *resizable_flip_node;
   Eina_List *elist;

   map = evas_map_new(4);
   if (!map) return;

   if (resizable_flip->cw)
      degree = (float)(progress * 180);
   else
      degree = (float)(progress * -180);

   if (!resizable_flip->nodes)
     resizable_flip->nodes = _resizable_flip_nodes_build(transit,
                                                         resizable_flip);

   EINA_LIST_FOREACH(resizable_flip->nodes, elist, resizable_flip_node)
     {
        if ((degree < 90) && (degree > -90))
          {
             obj = resizable_flip_node->front;
             evas_object_hide(resizable_flip_node->back);
             evas_object_show(resizable_flip_node->front);
          }
        else
          {
             obj = resizable_flip_node->back;
             evas_object_hide(resizable_flip_node->front);
             evas_object_show(resizable_flip_node->back);
          }

        evas_map_smooth_set(map, EINA_TRUE);

        x = resizable_flip_node->from_pos.x + \
            (resizable_flip_node->to_pos.x * progress);
        y = resizable_flip_node->from_pos.y + \
            (resizable_flip_node->to_pos.y * progress);
        w = resizable_flip_node->from_size.x + \
            (resizable_flip_node->to_size.x * progress);
        h = resizable_flip_node->from_size.y + \
            (resizable_flip_node->to_size.y * progress);
        evas_map_point_coord_set(map, 0, x, y, 0);
        evas_map_point_coord_set(map, 1, x + w, y, 0);
        evas_map_point_coord_set(map, 2, x + w, y + h, 0);
        evas_map_point_coord_set(map, 3, x, y + h, 0);

        half_w = (Evas_Coord) (w / 2);
        half_h = (Evas_Coord) (h / 2);

        if (resizable_flip->axis == ELM_TRANSIT_EFFECT_FLIP_AXIS_Y)
          {
             _set_image_uv_by_axis_y(map, resizable_flip_node, degree);
             evas_map_util_3d_rotate(map, 0, degree,
                                     0, x + half_w, y + half_h, 0);
          }
        else
          {
             _set_image_uv_by_axis_x(map, resizable_flip_node, degree);
             evas_map_util_3d_rotate(map, degree, 0,
                                     0, x + half_w, y + half_h, 0);
          }

        evas_map_util_3d_perspective(map, x + half_w, y + half_h, 0, 10000);
        evas_object_map_enable_set(resizable_flip_node->front, EINA_TRUE);
        evas_object_map_enable_set(resizable_flip_node->back, EINA_TRUE);
        evas_object_map_set(obj, map);
     }
   evas_map_free(map);
}

/**
 * Get a new context data of Resizable Flip Effect.
 *
 * This function belongs to the Resizable Flip effect, which consists of
 * functions:
 * - elm_transit_effect_resizable_flip_context_new()
 * - elm_transit_effect_resizable_flip_op()
 * - elm_transit_effect_resizing_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param axis Flipping Axis.(X or Y).
 * @param cw Flipping Direction. EINA_TRUE is clock-wise.
 * @return Resizable Flip effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_resizable_flip_context_new(Elm_Fx_Flip_Axis axis, Eina_Bool cw)
{
   Elm_Fx_ResizableFlip *resizable_flip;

   resizable_flip = ELM_NEW(Elm_Fx_ResizableFlip);
   if (!resizable_flip) return NULL;

   resizable_flip->cw = cw;
   resizable_flip->axis = axis;

   return resizable_flip;
}

///////////////////////////////////////////////////////////////////////////////
//Wipe FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Wipe Elm_Fx_Wipe;

struct _Elm_Fx_Wipe
{
   Elm_Fx_Wipe_Type type;
   Elm_Fx_Wipe_Dir dir;
};

static void
_elm_fx_wipe_hide(Evas_Map * map, Elm_Fx_Wipe_Dir dir, float x, float y, float w, float h, float progress)
{
   float w2, h2;

   switch (dir)
     {
      case ELM_TRANSIT_EFFECT_WIPE_DIR_LEFT:
         w2 = w - (w * progress);
         h2 = (y + h);
         evas_map_point_image_uv_set(map, 0, 0, 0);
         evas_map_point_image_uv_set(map, 1, w2, 0);
         evas_map_point_image_uv_set(map, 2, w2, h);
         evas_map_point_image_uv_set(map, 3, 0, h);
         evas_map_point_coord_set(map, 0, x, y, 0);
         evas_map_point_coord_set(map, 1, x + w2, y, 0);
         evas_map_point_coord_set(map, 2, x + w2, h2, 0);
         evas_map_point_coord_set(map, 3, x, h2, 0);
         break;
      case ELM_TRANSIT_EFFECT_WIPE_DIR_RIGHT:
         w2 = (w * progress);
         h2 = (y + h);
         evas_map_point_image_uv_set(map, 0, w2, 0);
         evas_map_point_image_uv_set(map, 1, w, 0);
         evas_map_point_image_uv_set(map, 2, w, h);
         evas_map_point_image_uv_set(map, 3, w2, h);
         evas_map_point_coord_set(map, 0, x + w2, y, 0);
         evas_map_point_coord_set(map, 1, x + w, y, 0);
         evas_map_point_coord_set(map, 2, x + w, h2, 0);
         evas_map_point_coord_set(map, 3, x + w2, h2, 0);
         break;
      case ELM_TRANSIT_EFFECT_WIPE_DIR_UP:
         w2 = (x + w);
         h2 = h - (h * progress);
         evas_map_point_image_uv_set(map, 0, 0, 0);
         evas_map_point_image_uv_set(map, 1, w, 0);
         evas_map_point_image_uv_set(map, 2, w, h2);
         evas_map_point_image_uv_set(map, 3, 0, h2);
         evas_map_point_coord_set(map, 0, x, y, 0);
         evas_map_point_coord_set(map, 1, w2, y, 0);
         evas_map_point_coord_set(map, 2, w2, h2, 0);
         evas_map_point_coord_set(map, 3, x, h2, 0);
         break;
      case ELM_TRANSIT_EFFECT_WIPE_DIR_DOWN:
         w2 = (x + w);
         h2 = (h * progress);
         evas_map_point_image_uv_set(map, 0, 0, h2);
         evas_map_point_image_uv_set(map, 1, w, h2);
         evas_map_point_image_uv_set(map, 2, w, h);
         evas_map_point_image_uv_set(map, 3, 0, h);
         evas_map_point_coord_set(map, 0, x, y + h2, 0);
         evas_map_point_coord_set(map, 1, w2, y + h2, 0);
         evas_map_point_coord_set(map, 2, w2, y + h, 0);
         evas_map_point_coord_set(map, 3, x, y + h, 0);
         break;
      default:
         break;
     }

   evas_map_util_3d_perspective(map, x + (w / 2), y + (h / 2), 0, 10000);
}

static void
_elm_fx_wipe_show(Evas_Map *map, Elm_Fx_Wipe_Dir dir, float x, float y, float w, float h, float progress)
{
   float w2, h2;

   switch (dir)
     {
      case ELM_TRANSIT_EFFECT_WIPE_DIR_LEFT:
         w2 = (w - (w * progress));
         h2 = (y + h);
         evas_map_point_image_uv_set(map, 0, w2, 0);
         evas_map_point_image_uv_set(map, 1, w, 0);
         evas_map_point_image_uv_set(map, 2, w, h);
         evas_map_point_image_uv_set(map, 3, w2, h);
         evas_map_point_coord_set(map, 0, x + w2, y, 0);
         evas_map_point_coord_set(map, 1, w, y, 0);
         evas_map_point_coord_set(map, 2, w, h2, 0);
         evas_map_point_coord_set(map, 3, x + w2, h2, 0);
         break;
      case ELM_TRANSIT_EFFECT_WIPE_DIR_RIGHT:
         w2 = (w * progress);
         h2 = (y + h);
         evas_map_point_image_uv_set(map, 0, 0, 0);
         evas_map_point_image_uv_set(map, 1, w2, 0);
         evas_map_point_image_uv_set(map, 2, w2, h);
         evas_map_point_image_uv_set(map, 3, 0, h);
         evas_map_point_coord_set(map, 0, x, y, 0);
         evas_map_point_coord_set(map, 1, x + w2, y, 0);
         evas_map_point_coord_set(map, 2, x + w2, h2, 0);
         evas_map_point_coord_set(map, 3, x, h2, 0);
         break;
      case ELM_TRANSIT_EFFECT_WIPE_DIR_UP:
         w2 = (x + w);
         h2 = (h - (h * progress));
         evas_map_point_image_uv_set(map, 0, 0, h2);
         evas_map_point_image_uv_set(map, 1, w, h2);
         evas_map_point_image_uv_set(map, 2, w, h);
         evas_map_point_image_uv_set(map, 3, 0, h);
         evas_map_point_coord_set(map, 0, x, y + h2, 0);
         evas_map_point_coord_set(map, 1, w2, y + h2, 0);
         evas_map_point_coord_set(map, 2, w2, y + h, 0);
         evas_map_point_coord_set(map, 3, x, y + h, 0);
         break;
      case ELM_TRANSIT_EFFECT_WIPE_DIR_DOWN:
         w2 = (x + w);
         h2 = (h * progress);
         evas_map_point_image_uv_set(map, 0, 0, 0);
         evas_map_point_image_uv_set(map, 1, w, 0);
         evas_map_point_image_uv_set(map, 2, w, h2);
         evas_map_point_image_uv_set(map, 3, 0, h2);
         evas_map_point_coord_set(map, 0, x, y, 0);
         evas_map_point_coord_set(map, 1, w2, y, 0);
         evas_map_point_coord_set(map, 2, w2, y + h2, 0);
         evas_map_point_coord_set(map, 3, x, y + h2, 0);
         break;
      default:
         break;
     }

   evas_map_util_3d_perspective(map, x + (w / 2), y + (h / 2), 0, 10000);
}

/**
 * The Free function to Wipe Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note This effect will be applied to the objects that are in the transit,
 * If you change the set of objects in the transit with  elm_transit_object_add()
 * or elm_transit_object_remove(), the set of objects affected by this effect
 * will be changed too.
 *
 * This function belongs to the Wipe effect, which consists of
 * functions:
 * - elm_transit_effect_wipe_context_new()
 * - elm_transit_effect_wipe_op()
 * - elm_transit_effect_wipe_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Wipe context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_wipe_context_free(void *data, Elm_Transit *transit)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Eina_List *elist;
   Evas_Object *obj;
   Elm_Fx_Wipe *wipe = data;
   Eina_Bool reverse = elm_transit_auto_reverse_get(transit);

   EINA_LIST_FOREACH(transit->objs, elist, obj)
     {
        if ((wipe->type == ELM_TRANSIT_EFFECT_WIPE_TYPE_SHOW && !reverse)
             || (wipe->type == ELM_TRANSIT_EFFECT_WIPE_TYPE_HIDE && reverse))
            evas_object_show(obj);
        else evas_object_hide(obj);
        evas_object_map_enable_set(obj, EINA_FALSE);
     }

   free(wipe);
}

/**
 * Operation function to the Wipe Effect.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Wipe effect, which consists of
 * functions:
 * - elm_transit_effect_wipe_context_new()
 * - elm_transit_effect_wipe_op()
 * - elm_transit_effect_wipe_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Wipe context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 * @warning Is higher recommended just create a transit with this effect when
 * the window that the objects of the transit belongs has already been created.
 * This is because this effect needs the geometry information about the objects,
 * and if the window was not created yet, it can get a wrong information.
 * @warning Is not recommended remove or add an object after the transit begins
 * to run, because the order of the objects will be affected.
 */
EAPI void
elm_transit_effect_wipe_op(void *data, Elm_Transit *transit, double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Elm_Fx_Wipe *wipe = data;

   Evas_Map *map;

   Evas_Coord _x, _y, _w, _h;

   Eina_List *elist;
   Evas_Object *obj;

   map = evas_map_new(4);
   if (!map) return;

   evas_map_smooth_set(map, EINA_TRUE);

   EINA_LIST_FOREACH(transit->objs, elist, obj)
     {
        evas_object_geometry_get(obj, &_x, &_y, &_w, &_h);

        if (wipe->type == ELM_TRANSIT_EFFECT_WIPE_TYPE_SHOW)
           _elm_fx_wipe_show(map, wipe->dir, _x, _y, _w, _h, (float)progress);

        else
           _elm_fx_wipe_hide(map, wipe->dir, _x, _y, _w, _h, (float)progress);

        evas_object_map_enable_set(obj, EINA_TRUE);
        evas_object_map_set(obj, map);
     }
   evas_map_free(map);
}

/**
 * Get a new context data of Wipe Flip Effect.
 *
 * This function belongs to the Wipe effect, which consists of
 * functions:
 * - elm_transit_effect_wipe_context_new()
 * - elm_transit_effect_wipe_op()
 * - elm_transit_effect_wipe_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param type Wipe type. Hide or show.
 * @param dir Wipe Direction.
 * @return Wipe effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_wipe_context_new(Elm_Fx_Wipe_Type type, Elm_Fx_Wipe_Dir dir)
{
   Elm_Fx_Wipe *wipe;

   wipe = ELM_NEW(Elm_Fx_Wipe);
   if (!wipe) return NULL;

   wipe->type = type;
   wipe->dir = dir;

   return wipe;
}

///////////////////////////////////////////////////////////////////////////////
//Color FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Color Elm_Fx_Color;

struct _Elm_Fx_Color
{
   struct _unsigned_color
     {
        unsigned int r, g, b, a;
     } from;
   struct _signed_color
     {
        int r, g, b, a;
     } to;
};

/**
 * The Free function to Color Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Color effect, which consists of
 * functions:
 * - elm_transit_effect_color_context_new()
 * - elm_transit_effect_color_op()
 * - elm_transit_effect_color_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Color context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_color_context_free(void *data, Elm_Transit *transit __UNUSED__)
{
   free(data);
}

/**
 * Operation function to the Color Effect.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note This effect will be applied to the objects that are in the transit,
 * If you change the set of objects in the transit with  elm_transit_object_add()
 * or elm_transit_object_remove(), the set of objects affected by this effect
 * will be changed too.
 *
 * This function belongs to the Color effect, which consists of
 * functions:
 * - elm_transit_effect_color_context_new()
 * - elm_transit_effect_color_op()
 * - elm_transit_effect_color_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Color context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_color_op(void *data, Elm_Transit *transit, double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Elm_Fx_Color *color = data;
   Evas_Object *obj;
   Eina_List *elist;

   unsigned int r, g, b, a;

   r = (color->from.r + (int)((float)color->to.r * progress));
   g = (color->from.g + (int)((float)color->to.g * progress));
   b = (color->from.b + (int)((float)color->to.b * progress));
   a = (color->from.a + (int)((float)color->to.a * progress));

   EINA_LIST_FOREACH(transit->objs, elist, obj)
     evas_object_color_set(obj, r, g, b, a);
}

/**
 * Get a new context data of Color Effect.
 *
 * @see elm_transit_effect_add()
 *
 * This function belongs to the Color effect, which consists of
 * functions:
 * - elm_transit_effect_color_context_new()
 * - elm_transit_effect_color_op()
 * - elm_transit_effect_color_context_free()
 *
 * @param  from_r        RGB R when effect begins.
 * @param  from_g        RGB G when effect begins.
 * @param  from_b        RGB B when effect begins.
 * @param  from_a        RGB A when effect begins.
 * @param  to_r          RGB R when effect ends.
 * @param  to_g          RGB G when effect ends.
 * @param  to_b          RGB B when effect ends.
 * @param  to_a          RGB A when effect ends.
 * @return               Color effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_color_context_new(unsigned int from_r, unsigned int from_g, unsigned int from_b, unsigned int from_a, unsigned int to_r, unsigned int to_g, unsigned int to_b, unsigned int to_a)
{
   Elm_Fx_Color *color;

   color = ELM_NEW(Elm_Fx_Color);
   if (!color) return NULL;

   color->from.r = from_r;
   color->from.g = from_g;
   color->from.b = from_b;
   color->from.a = from_a;
   color->to.r = to_r - from_r;
   color->to.g = to_g - from_g;
   color->to.b = to_b - from_b;
   color->to.a = to_a - from_a;

   return color;
}

///////////////////////////////////////////////////////////////////////////////
//Fade FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Fade Elm_Fx_Fade;
typedef struct _Elm_Fx_Fade_Node Elm_Fx_Fade_Node;

struct _Elm_Fx_Fade_Node
{
   Evas_Object *before;
   Evas_Object *after;
   struct _signed_color before_color, after_color;
   int before_alpha;
   int after_alpha;
   Eina_Bool inversed:1;
};

struct _Elm_Fx_Fade
{
   Eina_List *nodes;
};

static void
_fade_object_del_cb(void *data, Evas *e __UNUSED__, Evas_Object *obj, void *event_info __UNUSED__)
{
   Elm_Fx_Fade *fade = data;
   Eina_List *elist;
   Elm_Fx_Fade_Node *fade_node;

   EINA_LIST_FOREACH(fade->nodes, elist, fade_node)
     {
        if (fade_node->before == obj)
          evas_object_event_callback_del(fade_node->after,
                             EVAS_CALLBACK_DEL, _fade_object_del_cb);
        else if (fade_node->after == obj)
          evas_object_event_callback_del(fade_node->before,
                             EVAS_CALLBACK_DEL, _fade_object_del_cb);
        else continue;

        fade->nodes = eina_list_remove_list(fade->nodes, elist);
        free(fade_node);
        break;
     }
}

static Eina_List *
_fade_nodes_build(Elm_Transit *transit, Elm_Fx_Fade *fade_data)
{
   Elm_Fx_Fade_Node *fade;
   Eina_List *data_list = NULL;

   int i; int count;

   count = eina_list_count(transit->objs);
   for (i = 0; i < count-1; i+=2)
      {
         fade = ELM_NEW(Elm_Fx_Fade_Node);
         if (!fade)
           {
              eina_list_free(data_list);
              return NULL;
           }

         fade->before = eina_list_nth(transit->objs, i);
         fade->after = eina_list_nth(transit->objs, i+1);

         evas_object_color_get(fade->before,
                               &fade->before_color.r, &fade->before_color.g,
                               &fade->before_color.b, &fade->before_color.a);
         evas_object_color_get(fade->after,
                               &fade->after_color.r, &fade->after_color.g,
                               &fade->after_color.b, &fade->after_color.a);

         fade->before_alpha = (255 - fade->before_color.a);
         fade->after_alpha = (255 - fade->after_color.a);

         data_list = eina_list_append(data_list, fade);

         evas_object_event_callback_add(fade->before,
             EVAS_CALLBACK_DEL, _fade_object_del_cb, fade_data);
         evas_object_event_callback_add(fade->after,
             EVAS_CALLBACK_DEL, _fade_object_del_cb, fade_data);
      }

   return data_list;
}

/**
 * The Free function to Fade Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Fade effect, which consists of
 * functions:
 * - elm_transit_effect_fade_context_new()
 * - elm_transit_effect_fade_op()
 * - elm_transit_effect_fade_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Fade context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_fade_context_free(void *data, Elm_Transit *transit __UNUSED__)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   Elm_Fx_Fade *fade = data;
   Elm_Fx_Fade_Node *fade_node;
   Eina_List *elist, *elist_next;

   EINA_LIST_FOREACH_SAFE(fade->nodes, elist, elist_next, fade_node)
     {
        evas_object_color_set(fade_node->before, fade_node->before_color.r,
                              fade_node->before_color.g,
                              fade_node->before_color.b,
                              fade_node->before_color.a);
        evas_object_color_set(fade_node->after, fade_node->after_color.r,
                              fade_node->after_color.g,
                              fade_node->after_color.b,
                              fade_node->after_color.a);

        fade->nodes = eina_list_remove_list(fade->nodes, elist);
        evas_object_event_callback_del(fade_node->before,
                             EVAS_CALLBACK_DEL, _fade_object_del_cb);
        evas_object_event_callback_del(fade_node->after,
                             EVAS_CALLBACK_DEL, _fade_object_del_cb);
        free(fade_node);
     }

   free(fade);
}

/**
 * Operation function to the Fade Effect
 *
 * This effect is applied to each pair of objects in the order they are listed
 * in the transit list of objects. The first object in the pair will be the
 * "before" object and the second will be the "after" object.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note When this function begins to be called, it gets the current objects in
 * the transit, that is, elm_transit_object_remove() and elm_transit_object_add()
 * will not cause any changes in the set of objects that this effect is being
 * applied if these functions are called after the @p transit starts to run.
 *
 * This function belongs to the Fade effect, which consists of
 * functions:
 * - elm_transit_effect_fade_context_new()
 * - elm_transit_effect_fade_op()
 * - elm_transit_effect_fade_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Fade context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 * @warning Is higher recommended just create a transit with this effect when
 * the window that the objects of the transit belongs has already been created.
 * This is because this effect needs the color information about the objects,
 * and if the window was not created yet, it can get a wrong information.
 * @warning Is not recommended remove or add an object after the transit begins
 * to run, because the order of the objects will be affected.
 */
EAPI void
elm_transit_effect_fade_op(void *data, Elm_Transit *transit __UNUSED__, double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   Elm_Fx_Fade *fade = data;
   Eina_List *elist;
   Elm_Fx_Fade_Node *fade_node;

   float _progress;

   if (!fade->nodes)
     fade->nodes = _fade_nodes_build(transit, fade);

   EINA_LIST_FOREACH(fade->nodes, elist, fade_node)
     {
        if (progress < 0.5)
          {
             if (!fade_node->inversed)
               {
                  evas_object_hide(fade_node->after);
                  evas_object_show(fade_node->before);
                  fade_node->inversed = EINA_TRUE;
               }

             _progress = (1 - (progress * 2));

             evas_object_color_set(fade_node->before,
                                   fade_node->before_color.r * _progress,
                                   fade_node->before_color.g * _progress,
                                   fade_node->before_color.b * _progress,
                                   fade_node->before_color.a + \
                                   fade_node->before_alpha * (1 - _progress));
          }
        else
          {
             if (fade_node->inversed)
               {
                  evas_object_hide(fade_node->before);
                  evas_object_show(fade_node->after);
                  fade_node->inversed = EINA_FALSE;
               }

             _progress = ((progress - 0.5) * 2);

             evas_object_color_set(fade_node->after,
                                   fade_node->after_color.r * _progress,
                                   fade_node->after_color.g * _progress,
                                   fade_node->after_color.b * _progress,
                                   fade_node->after_color.a + \
                                   fade_node->after_alpha * (1 - _progress));
          }
     }
}

/**
 * Get a new context data of Fade Effect.
 *
 * This function belongs to the Fade effect, which consists of
 * functions:
 * - elm_transit_effect_fade_context_new()
 * - elm_transit_effect_fade_op()
 * - elm_transit_effect_fade_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @return Fade effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_fade_context_new(void)
{
   Elm_Fx_Fade *fade;
   fade = ELM_NEW(Elm_Fx_Fade);
   if (!fade) return NULL;
   return fade;
}

///////////////////////////////////////////////////////////////////////////////
//Blend FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Blend Elm_Fx_Blend;
typedef struct _Elm_Fx_Blend_Node Elm_Fx_Blend_Node;

struct _Elm_Fx_Blend_Node
{
   Evas_Object *before;
   Evas_Object *after;
   struct _signed_color from, to;
};

struct _Elm_Fx_Blend
{
   Eina_List *nodes;
};

static void
_blend_object_del_cb(void *data, Evas *e __UNUSED__, Evas_Object *obj, void *event_info __UNUSED__)
{
   Elm_Fx_Blend *blend = data;
   Eina_List *elist;
   Elm_Fx_Blend_Node *blend_node;

   EINA_LIST_FOREACH(blend->nodes, elist, blend_node)
     {
        if (blend_node->after == obj)
          evas_object_event_callback_del(blend_node->before,
                             EVAS_CALLBACK_DEL, _blend_object_del_cb);
        else if (blend_node->before == obj)
          evas_object_event_callback_del(blend_node->after,
                             EVAS_CALLBACK_DEL, _blend_object_del_cb);
        else continue;

        blend->nodes = eina_list_remove_list(blend->nodes, elist);
        free(blend_node);
        break;
     }
}

static Eina_List *
_blend_nodes_build(Elm_Transit *transit, Elm_Fx_Blend *blend)
{
   Elm_Fx_Blend_Node *blend_node;
   Eina_List *data_list = NULL;

   int i, count;

   count = eina_list_count(transit->objs);
   for (i = 0; i < count-1; i+=2)
     {
         blend_node = ELM_NEW(Elm_Fx_Blend_Node);
         if (!blend_node)
           {
              eina_list_free(data_list);
              return NULL;
           }

         blend_node->before = eina_list_nth(transit->objs, i);
         blend_node->after = eina_list_nth(transit->objs, i+1);
         evas_object_show(blend_node->before);
         evas_object_show(blend_node->after);

         evas_object_color_get(blend_node->before, &blend_node->from.r,
                               &blend_node->from.g, &blend_node->from.b,
                               &blend_node->from.a);
         evas_object_color_get(blend_node->after, &blend_node->to.r,
                               &blend_node->to.g, &blend_node->to.b,
                               &blend_node->to.a);

         data_list = eina_list_append(data_list, blend_node);

         evas_object_event_callback_add(blend_node->before,
             EVAS_CALLBACK_DEL, _blend_object_del_cb, blend);
         evas_object_event_callback_add(blend_node->after,
             EVAS_CALLBACK_DEL, _blend_object_del_cb, blend);
     }
   return data_list;
}

/**
 * The Free function to Blend Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Blend effect, which consists of
 * functions:
 * - elm_transit_effect_blend_context_new()
 * - elm_transit_effect_blend_op()
 * - elm_transit_effect_blend_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Blend context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_blend_context_free(void *data, Elm_Transit *transit __UNUSED__)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   Elm_Fx_Blend *blend = data;
   Elm_Fx_Blend_Node *blend_node;
   Eina_List *elist, *elist_next;

   EINA_LIST_FOREACH_SAFE(blend->nodes, elist, elist_next, blend_node)
     {
        evas_object_color_set(blend_node->before,
                              blend_node->from.r, blend_node->from.g,
                              blend_node->from.b, blend_node->from.a);
        evas_object_color_set(blend_node->after, blend_node->to.r,
                              blend_node->to.g, blend_node->to.b,
                              blend_node->to.a);

        if (elm_transit_auto_reverse_get(transit))
          evas_object_hide(blend_node->after);
        else
          evas_object_hide(blend_node->before);

        blend->nodes = eina_list_remove_list(blend->nodes, elist);

        evas_object_event_callback_del(blend_node->before,
                             EVAS_CALLBACK_DEL, _blend_object_del_cb);
        evas_object_event_callback_del(blend_node->after,
                             EVAS_CALLBACK_DEL, _blend_object_del_cb);
        free(blend_node);
     }
   free(data);
}

/**
 * Operation function to the Blend Effect.
 *
 * This effect is applied to each pair of objects in the order they are listed
 * in the transit list of objects. The first object in the pair will be the
 * "before" object and the second will be the "after" object.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note When this function begins to be called, it gets the current objects in
 * the transit, that is, elm_transit_object_remove() and elm_transit_object_add()
 * will not cause any changes in the set of objects that this effect is being
 * applied if these functions are called after the @p transit starts to run.
 *
 * This function belongs to the Blend effect, which consists of
 * functions:
 * - elm_transit_effect_blend_context_new()
 * - elm_transit_effect_blend_op()
 * - elm_transit_effect_blend_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Blend context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 * @warning Is higher recommended just create a transit with this effect when
 * the window that the objects of the transit belongs has already been created.
 * This is because this effect needs the color information about the objects,
 * and if the window was not created yet, it can get a wrong information.
 * @warning Is not recommended remove or add an object after the transit begins
 * to run, because the order of the objects will be affected.
 */
EAPI void
elm_transit_effect_blend_op(void *data, Elm_Transit *transit, double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Elm_Fx_Blend *blend = data;
   Elm_Fx_Blend_Node *blend_node;
   Eina_List *elist;

   if (!blend->nodes)
     blend->nodes = _blend_nodes_build(transit, blend);

   EINA_LIST_FOREACH(blend->nodes, elist, blend_node)
     {
        evas_object_color_set(blend_node->before,
                              (int)(blend_node->from.r * (1 - progress)),
                              (int)(blend_node->from.g * (1 - progress)),
                              (int)(blend_node->from.b * (1 - progress)),
                              (int)(blend_node->from.a * (1 - progress)));
        evas_object_color_set(blend_node->after,
                              (int)(blend_node->to.r * progress),
                              (int)(blend_node->to.g * progress),
                              (int)(blend_node->to.b * progress),
                              (int)(blend_node->to.a * progress));
     }
}

/**
 * Get a new context data of Blend Effect.
 *
 * @see elm_transit_effect_add()
 *
 * This function belongs to the Blend effect, which consists of
 * functions:
 * - elm_transit_effect_blend_context_new()
 * - elm_transit_effect_blend_op()
 * - elm_transit_effect_blend_context_free()
 *
 * @return Blend effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_blend_context_new(void)
{
   Elm_Fx_Blend *blend;

   blend = ELM_NEW(Elm_Fx_Blend);
   if (!blend) return NULL;

   return blend;
}

///////////////////////////////////////////////////////////////////////////////
//Rotation FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Rotation Elm_Fx_Rotation;

struct _Elm_Fx_Rotation
{
   Eina_Bool cw;
   float from, to;
};

/**
 * The Free function to Rotation Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Rotation effect, which consists of
 * functions:
 * - elm_transit_effect_rotation_context_new()
 * - elm_transit_effect_rotation_op()
 * - elm_transit_effect_rotation_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Rotation context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_rotation_context_free(void *data, Elm_Transit *transit __UNUSED__)
{
   free(data);
}

/**
 * Operation function to the Rotation Effect
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note This effect will be applied to the objects that are in the transit,
 * If you change the set of objects in the transit with  elm_transit_object_add()
 * or elm_transit_object_remove(), the set of objects affected by this effect
 * will be changed too.
 *
 * This function belongs to the Rotation effect, which consists of
 * functions:
 * - elm_transit_effect_rotation_context_new()
 * - elm_transit_effect_rotation_op()
 * - elm_transit_effect_rotation_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Rotation context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 * @warning Is higher recommended just create a transit with this effect when
 * the window that the objects of the transit belongs has already been created.
 * This is because this effect needs the geometry information about the objects,
 * and if the window was not created yet, it can get a wrong information.
 * @warning Is not recommended remove or add an object after the transit begins
 * to run, because the order of the objects will be affected.
 */
EAPI void
elm_transit_effect_rotation_op(void *data, Elm_Transit *transit, double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Elm_Fx_Rotation *rotation = data;

   Evas_Map *map;

   Evas_Coord x, y, w, h;

   float degree;

   float half_w, half_h;

   Eina_List *elist;
   Evas_Object *obj;

   map = evas_map_new(4);
   if (!map) return;

   evas_map_smooth_set(map, EINA_TRUE);

   EINA_LIST_FOREACH(transit->objs, elist, obj)
     {
        evas_map_util_points_populate_from_object_full(map, obj, 0);
        degree = rotation->from + (float)(progress * rotation->to);

        if (!rotation->cw)
           degree *= -1;

        evas_object_geometry_get(obj, &x, &y, &w, &h);

        half_w = (float)w *0.5;

        half_h = (float)h *0.5;

        evas_map_util_3d_rotate(map, 0, 0, degree, x + half_w, y + half_h, 0);
        evas_map_util_3d_perspective(map, x + half_w, y + half_h, 0, 10000);
        evas_object_map_enable_set(obj, EINA_TRUE);
        evas_object_map_set(obj, map);
     }
   evas_map_free(map);
}

/**
 * Get a new context data of Rotation Effect.
 *
 * This function belongs to the Rotation effect, which consists of
 * functions:
 * - elm_transit_effect_rotation_context_new()
 * - elm_transit_effect_rotation_op()
 * - elm_transit_effect_rotation_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param from_degree Degree when effect begins.
 * @param to_degree Degree when effect is ends.
 * @param cw Rotation direction. EINA_TRUE is clock wise.
 * @return Rotation effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_rotation_context_new(float from_degree, float to_degree, Eina_Bool cw)
{
   Elm_Fx_Rotation *rotation;

   rotation = ELM_NEW(Elm_Fx_Rotation);
   if (!rotation) return NULL;

   rotation->from = from_degree;
   rotation->to = to_degree - from_degree;
   rotation->cw = cw;

   return rotation;
}

///////////////////////////////////////////////////////////////////////////////
// ImageAnimation FX
///////////////////////////////////////////////////////////////////////////////
typedef struct _Elm_Fx_Image_Animation Elm_Fx_Image_Animation;

struct _Elm_Fx_Image_Animation
{
   Eina_List *images;
};

/**
 * The Free function to Image Animation Effect context data.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 *
 * This function belongs to the Image Animation effect, which consists of
 * functions:
 * - elm_transit_effect_image_animation_context_new()
 * - elm_transit_effect_image_animation_op()
 * - elm_transit_effect_image_animation_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Imagem Animation context data.
 * @param transit Transit object.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_image_animation_context_free(void *data, Elm_Transit *transit __UNUSED__)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   Elm_Fx_Image_Animation *image_animation = data;

   const char *image;
   Eina_List *elist, *elist_next;

   EINA_LIST_FOREACH_SAFE(image_animation->images, elist, elist_next, image)
     {
        image_animation->images = \
                       eina_list_remove_list(image_animation->images, elist);
        eina_stringshare_del(image);
     }

   free(data);
}

/**
 * Operation function to the Imagem Animation Effect.
 *
 * This effect changes the image from an icon object in the @p transit.
 *
 * @note You not need to call this function, just pass as parameter to
 * elm_transit_effect_add() function.
 * @note This effect will be applied to the objects that are in the transit,
 * If you change the set of objects in the transit with  elm_transit_object_add()
 * or elm_transit_object_remove(), the set of objects affected by this effect
 * will be changed too.
 *
 * This function belongs to the Image Animation effect, which consists of
 * functions:
 * - elm_transit_effect_image_animation_context_new()
 * - elm_transit_effect_image_animation_op()
 * - elm_transit_effect_image_animation_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param data The Imagem Animation context data.
 * @param transit Transit object.
 * @param progress The time progression, it is a double value between 0.0 and 1.0.
 *
 * @ingroup Transit
 */
EAPI void
elm_transit_effect_image_animation_op(void *data, Elm_Transit *transit, double progress)
{
   EINA_SAFETY_ON_NULL_RETURN(data);

   EINA_SAFETY_ON_NULL_RETURN(transit);

   Eina_List *elist;
   Evas_Object *obj;
   const char *type;

   type = eina_stringshare_add("icon");

   Elm_Fx_Image_Animation *image_animation = (Elm_Fx_Image_Animation *) data;

   unsigned int count = 0;
   int len = eina_list_count(image_animation->images);

   if (!len)
      count = floor(progress * len);
   else
      count = floor(progress * (len - 1));

   EINA_LIST_FOREACH(transit->objs, elist, obj)
     {
        if (elm_widget_type_check(obj, type))
          elm_icon_file_set(obj,
                          eina_list_nth(image_animation->images, count), NULL);
     }

   eina_stringshare_del(type);
}

/**
 * Get a new context data of Imagem Animation Effect.
 *
 * The @p images parameter is a list images paths. This list and
 * its contents will be deleted at the end of the effect by
 * elm_transit_effect_image_animation_context_free() function.
 *
 * Example:
 * @code
 * char buf[PATH_MAX];
 * Eina_List *images = NULL;
 * Elm_Transit *transi = elm_transit_add(4.0);
 *
 * snprintf(buf, sizeof(buf), "%s/images/icon_11.png", PACKAGE_DATA_DIR);
 * images = eina_list_append(images, eina_stringshare_add(buf));
 *
 * snprintf(buf, sizeof(buf), "%s/images/logo_small.png", PACKAGE_DATA_DIR);
 * images = eina_list_append(images, eina_stringshare_add(buf));
 *
 * elm_transit_effect_add(transit,
 *                      elm_transit_effect_image_animation_op,
 *                      elm_transit_effect_image_animation_context_new(images),
 *                      elm_transit_effect_image_animation_context_free);
 * @endcode
 *
 * This function belongs to the Image Animation effect, which consists of
 * functions:
 * - elm_transit_effect_image_animation_context_new()
 * - elm_transit_effect_image_animation_op()
 * - elm_transit_effect_image_animation_context_free()
 *
 * @see elm_transit_effect_add()
 *
 * @param images Eina_List of images file paths. This list and
 * its contents will be deleted at the end of the effect by
 * elm_transit_effect_image_animation_context_free() function.
 * @return Image Animation effect context data.
 *
 * @ingroup Transit
 */
EAPI void *
elm_transit_effect_image_animation_context_new(Eina_List *images)
{
   Elm_Fx_Image_Animation *image_animation;

   image_animation = ELM_NEW(Elm_Fx_Image_Animation);

   if (!image_animation) return NULL;

   image_animation->images = images;

   return image_animation;
}
