#ifndef _ECS_FAMILLE_CHAINE_H_
#define _ECS_FAMILLE_CHAINE_H_

/*============================================================================
 *  Prototypes des fonctions de base
 *   associées à une liste chaînée de structures `ecs_famille_t' décrivant
 *   une famille
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <bft_file.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_descr.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_famille.h"


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui crée une liste chaînée de familles à partir :
 *   - des définitions de chaque famille en fonction des numéros de descripteur
 *   - de la liste chaînée des descripteurs
 *  La fonction renvoie la tête de la liste chaînée
 *----------------------------------------------------------------------------*/

ecs_famille_t * ecs_famille_chaine__cree
(
       ecs_int_t   * *const def_fam_descr,  /* Définition des familles        */
 const ecs_int_t     *const nbr_descr_fam,  /* Nombre de descripteurs par fam */
 const ecs_int_t            num_fam_deb,    /* Premier numéro de famille      */
 const ecs_int_t            nbr_fam,        /* Nombre de familles             */
       ecs_descr_t   *const descr_tete      /* Liste chaînée des descripteurs */
);


/*----------------------------------------------------------------------------
 *  Fonction libérant la portion d'une liste chaînée de familles
 *   à partir d'un noeud dont le pointeur est donné en argument.
 *  Le noeud est à NULL au retour de la fonction
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__detruit
(
 ecs_famille_t * * this_fam_noeud
);


/*----------------------------------------------------------------------------
 *  Fonction imprimant à partir d'un noeud `ecs_famille_t' donné
 *   une liste chaînée de champs
 *   sur le flux décrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__imprime
(
 const ecs_famille_t  *const this_fam_noeud,
       ecs_int_t             imp_col,
       bft_file_t     *const fic_imp
);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets
 *   d'une chaîne de structures `ecs_famille_t'
 *----------------------------------------------------------------------------*/

float ecs_famille_chaine__ret_taille
(
 const ecs_famille_t *const this_fam_noeud
);


/*----------------------------------------------------------------------------
 *  Fonction qui ajoute à la fin d'une liste chaînée de familles
 *   réceptrice dont la tête est donnée,
 *   une liste chaînée de familles à concaténer dont la tête est donnée
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__ajoute
(
 ecs_famille_t *      * this_fam_tete,
 ecs_famille_t *const   fam_concat_tete
);


/*----------------------------------------------------------------------------
 *  Fonction qui affiche la définition de la famille de numéro donné
 *   à partir de la liste chaînée des familles dont la tête est donnée
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__affiche
(
 const ecs_int_t            num_fam,
       ecs_famille_t *const fam_tete
);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le nombre de familles
 *   de la liste chaînée des familles dont la tête est donnée
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_famille_chaine__ret_nbr
(
 const ecs_famille_t *const fam_tete
);


/*----------------------------------------------------------------------------
 *  Fonction qui copie une liste chaînée de familles
 *   dont la tête est donnée
 *----------------------------------------------------------------------------*/

ecs_famille_t * ecs_famille_chaine__copie
(
 ecs_famille_t * famille_tete
);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie pour chaque numéro de famille
 *   le nombre et la liste des identificateurs de type couleur
 *   des descripteurs de la famille
 *----------------------------------------------------------------------------*/

ecs_tab_int_t * ecs_famille_chaine__ret_ide
(
 ecs_famille_t   *const fam_tete
);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie pour chaque numéro de famille
 *   le nombre et une liste de pointeurs sur les noms des identificateurs
 *   de type groupe des descripteurs de la famille
 *----------------------------------------------------------------------------*/

ecs_tab_char_t * ecs_famille_chaine__ret_nom
(
 ecs_famille_t   *const fam_tete
);


/*----------------------------------------------------------------------------
 *  Fonction qui affiche par attribut (couleur ou groupe),
 *   les numéros des familles auxquelles l'attribut appartient
 *   à partir d'une liste chaînée de familles dont la tête est donnée
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__aff_fam_att
(
 ecs_famille_t *const fam_tete
);


/*----------------------------------------------------------------------------
 *  Fonction qui construit les 2 listes chaînées de descripteurs de type
 *   "couleur" et "groupe"
 *   pour chaque numéro de famille contenu dans le tableau donné
 *   et à partir de la liste chaînée des familles
 *
 *  Cette fonction détermine aussi les 2 tableaux donnant pour chaque famille
 *   respectivement la liste des numéros de couleurs associes à la famille
 *               et la liste des numéros de groupes  associes à la famille
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__cree_descr
(
 ecs_famille_t   *const famille,
 ecs_tab_int_t          tab_fam,
 ecs_descr_t   * *const descr_tete_couleur,
 ecs_descr_t   * *const descr_tete_groupe,
 ecs_tab_int_t   *const tab_couleur_fam,
 ecs_tab_int_t   *const tab_groupe_fam,
 int             *const nbr_max_att_fam
);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un tableau indiquant pour chaque numéro de
 *   famille si cette famille contient l'un des attributs (couleur ou groupe)
 *   fournis en argument.
 *
 *  La libération du tableau est à la charge du code appelant
 *----------------------------------------------------------------------------*/

ecs_tab_bool_t ecs_famille_chaine__indic_fam_att
(
 const ecs_famille_t   *const fam_tete,
 const ecs_tab_int_t          liste_couleur,
 const ecs_tab_char_t         liste_groupe
);


/*----------------------------------------------------------------------------
 *  Fonction qui construit un tableau de description d'attributs
 *   décrits par une liste chaînée de familles, avec pour chacun la
 *   liste des éléments portant cet attribut (à partir du numéro de
 *   famille associé à chaque élément et donné par tab_fam_ent)
 *
 *  Les arguments prefixe_coul et prefixe_grp permettent de déterminer les
 *   chaînes de caractères précédent éventuellement les numéros de couleur ou
 *   noms de groupe dans le tableau résultant tab_nom_desc.
 *  L'argument "grouper_ident" indique si l'on doit combiner les attributs
 *   appartenant exactement aux mêmes familles.
 *
 *  La libération des tableaux tab_nom_desc et tab_lst_desc est à la
 *   charge de la fonction utilisatrice.
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__att_fam_elt
(
 const ecs_famille_t     * fam_tete,
 const ecs_tab_int_t       tab_fam_elt,
 const char              * prefixe_coul,
 const char              * prefixe_grp,
       bool                grouper_ident,
       ecs_tab_char_t    * tab_nom_descr,
       ecs_tab_int_t   * * tab_lst_descr
);


#endif /* _ECS_FAMILLE_CHAINE_H_ */
