#ifndef _ECS_CHAMP_CHAINE_H_
#define _ECS_CHAMP_CHAINE_H_

/*============================================================================
 *  Prototypes des fonctions de base
 *   associées à une liste chaînée de structures `ecs_champ_t' décrivant
 *   un champ
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <bft_file.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_param_perio_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ.h"


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction libérant la portion d'une liste chaînée de structures `ecs_champ_t'
 *   à partir d'un noeud dont le pointeur est donné en argument.
 *  Le noeud est à NULL au retour de la fonction
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__detruit
(
 ecs_champ_t * * this_champ_noeud
) ;


/*----------------------------------------------------------------------------
 *  Fonction imprimant à partir d'un noeud `ecs_champ_t' donné
 *   une liste chaînée de champs
 *   sur le flux décrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__imprime
(
 const ecs_champ_t *const this_champ_noeud ,
       ecs_int_t          imp_col    ,
 const ecs_int_t          nbr_imp    ,
       bft_file_t  *const fic_imp
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets
 *   d'une chaîne de structures `ecs_champ_t' dont la tête est donnée
 *----------------------------------------------------------------------------*/

float ecs_champ_chaine__ret_taille
(
 const ecs_champ_t *const this_champ_noeud
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui retourne dans une liste chaînée de champs
  *   dont un noeud est donné la référence au champ suivant.
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__ret_suivant
(
 ecs_champ_t *const this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui recherche dans une liste chaînée de champs
 *   dont la tête de la liste est donnée,
 *   le champ correspondant au nom fourni en argument
 *
 *  La fonction retourne :
 *   -    la référence du champ trouvé,
 *   - ou NULL si aucun champ n'a été trouvé
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__trouve_nom
(
       ecs_champ_t *const this_champ_tete , /* --> Tête de liste des champs   */
                                            /*     ou est recherche le champ  */
 const char        *const nom_champ         /* --> Nom à rechercher           */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui recherche dans une liste chaînée de champs
 *   dont la tête de la liste est donnée,
 *   le premier champ correspondant au statut fourni en argument
 *
 *  La fonction retourne :
 *   -    la référence du champ trouvé,
 *   - ou NULL si aucun champ n'a été trouvé
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__trouve_statut
(
 ecs_champ_t         *const this_champ_tete , /* --> Tête de liste des champs */
                                              /*     où on recherche le champ */
 ECS_CHAMP_STATUT_E         statut            /* --> Nom à rechercher         */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui supprime un noeud (c-a-d la référence au champ donné)
 *   dans une liste chaînée de champs dont la tête est donnée
 *
 *  La tête de la liste n'est pas modifiée
 *   sauf si le noeud à supprimer est aussi la tête de la liste !
 *
 *  Attention! Le champ dont la référence est supprimée de la liste,
 *              n'est pas détruit et
 *              son lien sur un champ suivant est ré-initialisé à `NULL'
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__supprime
(
 ecs_champ_t * *const this_champ_noeud ,  /* --> Adresse du champ à supprimer */
 ecs_champ_t * *const champ_tete          /* --> Tête de la liste contenant   */
                                          /*      le noeud à supprimer        */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui ajoute à la fin d'une liste chaînée de champs réceptrice
  *   une liste chaînée de champs à concaténer.
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__ajoute
(
 ecs_champ_t * *const this_champ_tete,
 ecs_champ_t   *const champ_concat_tete
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui concatène les champs identiques
 *   de deux listes chaînées de champs auxiliaires
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__concatene
(
 ecs_champ_t * * this_champ_tete,      /* <-> Tête de la chaîne réceptrice    */
 ecs_champ_t * * concat_champ_tete,    /* <-> Tête de la chaîne contenant     */
                                       /*         les champs à concaténer     */
 size_t          nbr_elt_init_recep,   /* --> Nbr d'elts de l'entité récept.  */
 size_t          nbr_elt_ent_concat    /* --> Nbr d'elts ent. à concaténer.   */
) ;


/*----------------------------------------------------------------------------
 *  Fonction appliquant un vecteur de transformation
 *   à une liste chaînée de champs donnée
 *----------------------------------------------------------------------------*/

typedef void (* ECS_PTR_FCT_TRANSF_CHAMP)
(
       ecs_champ_t   *const this_champ ,
       size_t               nbr_elt_new,
 const ecs_tab_int_t        vect_transf
) ;


void ecs_champ_chaine__transforme
(
       ecs_champ_t         * this_champ_chaine_tete,
       size_t                nbr_elt_new,
 const ecs_tab_int_t         vect_transf,
 ECS_PTR_FCT_TRANSF_CHAMP    fct_transformation
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit les champs de la chaîne de champs des sous-éléments
 *                à partir des champs de la chaîne de champs des éléments
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__herite
(
 ecs_champ_t *champ_elt_tete,
 ecs_champ_t *champ_elt_def_sselt,
 ecs_champ_t *champ_def_sselt
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renumérote les labels et les références des champs
 *  attributs de type ECS_CHAMP_STATUT_REF_ELT
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_chaine__renum
(
 ecs_champ_t    *champ_elt_tete,
 ecs_champ_t    *champ_elt_old_new,
 size_t          nbr_elt_new
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le nombre de périodicités.
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_champ_chaine__nbr_perio
(
 ecs_champ_t  * champ_att_tete
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui met à jour le tableau renvoyé par
 * ecs_champ_def__typ_fac_cel() en fonction des connectivités ajoutées par
 * les périodicités éventuelles.
 *----------------------------------------------------------------------------*/

void ecs_champ_chaine__typ_fac_perio
(
 ecs_champ_t    * champ_att_tete,
 ecs_tab_int_t  * typ_fac
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le nombre de faces périodiques associées à chaque
 * périodicité.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_champ_chaine__nbr_fac_perio
(
 ecs_champ_t  * champ_att_tete
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit la des faces périodiques associées à la
 *  périodicité num_per de la manière suivante :
 *  =>  [i,j; ...] avec i->j
 *  On ne compte pas les faces se voyant elles-mêmes.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_champ_chaine__liste_fac_perio
(
 ecs_champ_t  * champ_att_tete,
 ecs_int_t      num_per
) ;


#endif /* _ECS_CHAMP_CHAINE_H_ */
