/**********************************************************************
 * Copyright (c) 2012 Ericsson
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 *   Bernd Hufmann - Initial API and implementation
 **********************************************************************/
package org.eclipse.linuxtools.internal.lttng2.ui.views.control.service;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.IBaseEventInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.IChannelInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.IDomainInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.IEventInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.IProbeEventInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.ISessionInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.IUstProviderInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.LogLevelType;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.TraceEventType;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.TraceLogLevel;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.impl.BaseEventInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.impl.ChannelInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.impl.DomainInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.impl.EventInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.impl.ProbeEventInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.impl.SessionInfo;
import org.eclipse.linuxtools.internal.lttng2.core.control.model.impl.UstProviderInfo;
import org.eclipse.linuxtools.internal.lttng2.ui.views.control.logging.ControlCommandLogger;
import org.eclipse.linuxtools.internal.lttng2.ui.views.control.messages.Messages;
import org.eclipse.linuxtools.internal.lttng2.ui.views.control.preferences.ControlPreferences;
import org.eclipse.linuxtools.internal.lttng2.ui.views.control.remote.ICommandResult;
import org.eclipse.linuxtools.internal.lttng2.ui.views.control.remote.ICommandShell;

/**
 * <p>
 * Service for sending LTTng trace control commands to remote host.
 * </p>
 * 
 * @author Bernd Hufmann
 */
public class LTTngControlService implements ILttngControlService {
 
    
    // ------------------------------------------------------------------------
    // Attributes
    // ------------------------------------------------------------------------
    /**
     * The command shell implementation
     */
    protected ICommandShell fCommandShell = null;
    
    /**
     * The version string.
     */
    protected String fVersion = "Unknown"; //$NON-NLS-1$

    // ------------------------------------------------------------------------
    // Constructors
    // ------------------------------------------------------------------------

    /**
     * Constructor
     * 
     * @param shell
     *            - the command shell implementation to use
     */
    public LTTngControlService(ICommandShell shell) {
        fCommandShell = shell;
    }

    // ------------------------------------------------------------------------
    // Accessors
    // ------------------------------------------------------------------------
    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#getVersion()
     */
    @Override
    public String getVersion() {
        return fVersion;
    }
    
    /**
     * Sets the version of the LTTng 2.0 control service. 
     * @param version - a version to set
     */
    public void setVersion(String version) {
        fVersion = version;
    }
    // ------------------------------------------------------------------------
    // Operations
    // ------------------------------------------------------------------------
    
    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService
     * #getSessionNames(org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public String[] getSessionNames(IProgressMonitor monitor) throws ExecutionException {
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_LIST);

        ICommandResult result = executeCommand(command.toString(), monitor);

        // Output:
        // Available tracing sessions:
        // 1) mysession1 (/home/user/lttng-traces/mysession1-20120123-083928)
        // [inactive]
        // 2) mysession (/home/user/lttng-traces/mysession-20120123-083318)
        // [inactive]
        //
        // Use lttng list <session_name> for more details

        ArrayList<String> retArray = new ArrayList<String>();
        int index = 0;
        while (index < result.getOutput().length) {
            String line = result.getOutput()[index];
            Matcher matcher = LTTngControlServiceConstants.SESSION_PATTERN.matcher(line);
            if (matcher.matches()) {
                retArray.add(matcher.group(2).trim());
            }
            index++;
        }
        return retArray.toArray(new String[retArray.size()]);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService
     * #getSession(java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public ISessionInfo getSession(String sessionName, IProgressMonitor monitor) throws ExecutionException {
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_LIST, sessionName);
        ICommandResult result = executeCommand(command.toString(), monitor);

        int index = 0;

        // Output:
        // Tracing session mysession2: [inactive]
        // Trace path: /home/eedbhu/lttng-traces/mysession2-20120123-110330
        ISessionInfo sessionInfo = new SessionInfo(sessionName);

        while (index < result.getOutput().length) {
            // Tracing session mysession2: [inactive]
            // Trace path: /home/eedbhu/lttng-traces/mysession2-20120123-110330
            //
            // === Domain: Kernel ===
            //
            String line = result.getOutput()[index];
            Matcher matcher = LTTngControlServiceConstants.TRACE_SESSION_PATTERN.matcher(line);
            if (matcher.matches()) {
                sessionInfo.setSessionState(matcher.group(2));
                index++;
                continue;
            }

            matcher = LTTngControlServiceConstants.TRACE_SESSION_PATH_PATTERN.matcher(line);
            if (matcher.matches()) {
                sessionInfo.setSessionPath(matcher.group(1).trim());
                index++;
                continue;
            }

            matcher = LTTngControlServiceConstants.DOMAIN_KERNEL_PATTERN.matcher(line);
            if (matcher.matches()) {
                // Create Domain
                IDomainInfo domainInfo = new DomainInfo(Messages.TraceControl_KernelDomainDisplayName);
                sessionInfo.addDomain(domainInfo);

                // in domain kernel
                ArrayList<IChannelInfo> channels = new ArrayList<IChannelInfo>();
                index = parseDomain(result.getOutput(), index, channels);

                // set channels
                domainInfo.setChannels(channels);
                
                // set kernel flag
                domainInfo.setIsKernel(true);
                continue;
            }

            matcher = LTTngControlServiceConstants.DOMAIN_UST_GLOBAL_PATTERN.matcher(line);
            if (matcher.matches()) {
                IDomainInfo domainInfo = new DomainInfo(Messages.TraceControl_UstGlobalDomainDisplayName);
                sessionInfo.addDomain(domainInfo);

                // in domain UST
                ArrayList<IChannelInfo> channels = new ArrayList<IChannelInfo>();
                index = parseDomain(result.getOutput(), index, channels);

                // set channels
                domainInfo.setChannels(channels);
                
                // set kernel flag
                domainInfo.setIsKernel(false);
                continue;
            }
            index++;
        }
        return sessionInfo;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService
     * #getKernelProvider(org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public List<IBaseEventInfo> getKernelProvider(IProgressMonitor monitor) throws ExecutionException {
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_LIST_KERNEL);
        ICommandResult result = executeCommand(command.toString(), monitor);

        // Kernel events:
        // -------------
        // sched_kthread_stop (type: tracepoint)
        List<IBaseEventInfo> events = new ArrayList<IBaseEventInfo>();
        getProviderEventInfo(result.getOutput(), 0, events);
        return events;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService
     * #getUstProvider()
     */
    @Override
    public List<IUstProviderInfo> getUstProvider() throws ExecutionException {
        return getUstProvider(new NullProgressMonitor());
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService
     * #getUstProvider(org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public List<IUstProviderInfo> getUstProvider(IProgressMonitor monitor) throws ExecutionException {
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_LIST_UST);

        ICommandResult result = executeCommand(command.toString(), monitor);

        // UST events:
        // -------------
        //
        // PID: 3635 - Name:
        // /home/user/git/lttng-ust/tests/hello.cxx/.libs/lt-hello
        // ust_tests_hello:tptest_sighandler (loglevel: TRACE_EMERG0) (type:
        // tracepoint)
        // ust_tests_hello:tptest (loglevel: TRACE_EMERG0) (type: tracepoint)
        //
        // PID: 6459 - Name:
        // /home/user/git/lttng-ust/tests/hello.cxx/.libs/lt-hello
        // ust_tests_hello:tptest_sighandler (loglevel: TRACE_EMERG0) (type:
        // tracepoint)
        // ust_tests_hello:tptest (loglevel: TRACE_EMERG0) (type: tracepoint)

        List<IUstProviderInfo> allProviders = new ArrayList<IUstProviderInfo>();
        IUstProviderInfo provider = null;

        int index = 0;
        while (index < result.getOutput().length) {
            String line = result.getOutput()[index];
            Matcher matcher = LTTngControlServiceConstants.UST_PROVIDER_PATTERN.matcher(line);
            if (matcher.matches()) {

                provider = new UstProviderInfo(matcher.group(2).trim());
                provider.setPid(Integer.valueOf(matcher.group(1).trim()));
                List<IBaseEventInfo> events = new ArrayList<IBaseEventInfo>();
                index = getProviderEventInfo(result.getOutput(), ++index, events);
                provider.setEvents(events);
                allProviders.add(provider);

            } else {
                index++;
            }

        }
        return allProviders;
    }

    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#createSession(java.lang.String, java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public ISessionInfo createSession(String sessionName, String sessionPath, IProgressMonitor monitor) throws ExecutionException {

        String newName = formatParameter(sessionName);
        String newPath = formatParameter(sessionPath);

        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_CREATE_SESSION, newName);

        if (newPath != null && !"".equals(newPath)) { //$NON-NLS-1$
            command.append(LTTngControlServiceConstants.OPTION_OUTPUT_PATH);
            command.append(newPath);
        }

        ICommandResult result = executeCommand(command.toString(), monitor);

        //Session myssession2 created.
        //Traces will be written in /home/user/lttng-traces/myssession2-20120209-095418
        String[] output = result.getOutput();
        
        // Get and verify session name
        Matcher matcher = LTTngControlServiceConstants.CREATE_SESSION_NAME_PATTERN.matcher(output[0]);
        String name = null;

        if (matcher.matches()) {
            name = String.valueOf(matcher.group(1).trim());
        } else {
            // Output format not expected
            throw new ExecutionException(Messages.TraceControl_CommandError + " " + command + "\n" + //$NON-NLS-1$ //$NON-NLS-2$ 
                    Messages.TraceControl_UnexpectedCommandOutputFormat + ":\n" + //$NON-NLS-1$ 
                    formatOutput(result)); 
        }

        if ((name == null) || (!name.equals(sessionName))) {
            // Unexpected name returned
            throw new ExecutionException(Messages.TraceControl_CommandError + " " + command + "\n" + //$NON-NLS-1$ //$NON-NLS-2$ 
                    Messages.TraceControl_UnexpectedNameError + ": " + name); //$NON-NLS-1$ 
        }
        
        // Get and verify session path
        matcher = LTTngControlServiceConstants.CREATE_SESSION_PATH_PATTERN.matcher(output[1]);
        String path = null;
        
        if (matcher.matches()) {
            path = String.valueOf(matcher.group(1).trim());
        } else {
            // Output format not expected
            throw new ExecutionException(Messages.TraceControl_CommandError + " " + command + "\n" + //$NON-NLS-1$ //$NON-NLS-2$ 
                    Messages.TraceControl_UnexpectedCommandOutputFormat + ":\n" + //$NON-NLS-1$ 
                    formatOutput(result)); 
        }

        if ((path == null) || ((sessionPath != null) && (!path.contains(sessionPath)))) {
            // Unexpected path
            throw new ExecutionException(Messages.TraceControl_CommandError + " " + command + "\n" + //$NON-NLS-1$ //$NON-NLS-2$ 
                    Messages.TraceControl_UnexpectedPathError + ": " + name); //$NON-NLS-1$
        }
        
        SessionInfo sessionInfo = new SessionInfo(name);
        sessionInfo.setSessionPath(path);

        return sessionInfo;
    }
    
    @Override
    public void destroySession(String sessionName, IProgressMonitor monitor) throws ExecutionException {
        String newName = formatParameter(sessionName);

        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_DESTROY_SESSION, newName);

        ICommandResult result = executeCommand(command.toString(), monitor, false);
        String[] output = result.getOutput();
        
        if (isError(result) && ((output == null) || (!LTTngControlServiceConstants.SESSION_NOT_FOUND_ERROR_PATTERN.matcher(output[0]).matches()))) {
                throw new ExecutionException(Messages.TraceControl_CommandError + " " + command.toString() + "\n" + formatOutput(result)); //$NON-NLS-1$ //$NON-NLS-2$
       }
        //Session <sessionName> destroyed
    }
    
    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#startSession(java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void startSession(String sessionName, IProgressMonitor monitor) throws ExecutionException {

        String newSessionName = formatParameter(sessionName);

        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_START_SESSION, newSessionName);

        executeCommand(command.toString(), monitor);

        //Session <sessionName> started
    }

    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#stopSession(java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void stopSession(String sessionName, IProgressMonitor monitor) throws ExecutionException {
        String newSessionName = formatParameter(sessionName);
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_STOP_SESSION, newSessionName);

        executeCommand(command.toString(), monitor);

        //Session <sessionName> stopped
        
    }
    
    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#enableChannel(java.lang.String, java.util.List, boolean, org.eclipse.linuxtools.internal.lttng2.ui.views.control.model.IChannelInfo, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void enableChannels(String sessionName, List<String> channelNames, boolean isKernel, IChannelInfo info, IProgressMonitor monitor) throws ExecutionException {

        // no channels to enable
        if (channelNames.isEmpty()) {
            return;
        }

        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_ENABLE_CHANNEL);

        for (Iterator<String> iterator = channelNames.iterator(); iterator.hasNext();) {
            String channel = (String) iterator.next();
            command.append(channel);
            if (iterator.hasNext()) {
                command.append(',');
            }
        }

        if (isKernel) {
            command.append(LTTngControlServiceConstants.OPTION_KERNEL);
        } else {
            command.append(LTTngControlServiceConstants.OPTION_UST);
        }

        String newSessionName = formatParameter(sessionName);
        command.append(LTTngControlServiceConstants.OPTION_SESSION);
        command.append(newSessionName);

        if (info != null) {
//            --discard            Discard event when buffers are full (default)

//            --overwrite          Flight recorder mode
            if (info.isOverwriteMode()) {
                command.append(LTTngControlServiceConstants.OPTION_OVERWRITE);
            }
//            --subbuf-size SIZE   Subbuffer size in bytes
//                                     (default: 4096, kernel default: 262144)
            command.append(LTTngControlServiceConstants.OPTION_SUB_BUFFER_SIZE);
            command.append(String.valueOf(info.getSubBufferSize()));

//            --num-subbuf NUM     Number of subbufers
//                                     (default: 8, kernel default: 4)
            command.append(LTTngControlServiceConstants.OPTION_NUM_SUB_BUFFERS);
            command.append(String.valueOf(info.getNumberOfSubBuffers()));
            
//            --switch-timer USEC  Switch timer interval in usec (default: 0)
            command.append(LTTngControlServiceConstants.OPTION_SWITCH_TIMER);
            command.append(String.valueOf(info.getSwitchTimer()));

//            --read-timer USEC    Read timer interval in usec (default: 200)
            command.append(LTTngControlServiceConstants.OPTION_READ_TIMER);
            command.append(String.valueOf(info.getReadTimer()));
        } 

        executeCommand(command.toString(), monitor);
        
    }

    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#disableChannel(java.lang.String, java.util.List, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void disableChannels(String sessionName, List<String> channelNames, boolean isKernel, IProgressMonitor monitor) throws ExecutionException {
        
        // no channels to enable
        if (channelNames.isEmpty()) {
            return;
        }

        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_DISABLE_CHANNEL);

        for (Iterator<String> iterator = channelNames.iterator(); iterator.hasNext();) {
            String channel = (String) iterator.next();
            command.append(channel);
            if (iterator.hasNext()) {
                command.append(',');
            }
        }

        if (isKernel) {
            command.append(LTTngControlServiceConstants.OPTION_KERNEL);
        } else {
            command.append(LTTngControlServiceConstants.OPTION_UST);
        }

        String newSessionName = formatParameter(sessionName);
        command.append(LTTngControlServiceConstants.OPTION_SESSION);
        command.append(newSessionName);

        executeCommand(command.toString(), monitor);
    }
    
    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#enableEvent(java.lang.String, java.lang.String, java.util.List, boolean, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void enableEvents(String sessionName, String channelName, List<String> eventNames, boolean isKernel, IProgressMonitor monitor) throws ExecutionException {

        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_ENABLE_EVENT);

        if (eventNames == null || eventNames.isEmpty()) {
            command.append(LTTngControlServiceConstants.OPTION_ALL);
        } else {

            for (Iterator<String> iterator = eventNames.iterator(); iterator.hasNext();) {
                String event = (String) iterator.next();
                command.append(event);
                if (iterator.hasNext()) {
                    command.append(',');
                }
            }
        }

        if (isKernel) {
            command.append(LTTngControlServiceConstants.OPTION_KERNEL);
        } else {
            command.append(LTTngControlServiceConstants.OPTION_UST);
        }

        String newSessionName = formatParameter(sessionName);

        command.append(LTTngControlServiceConstants.OPTION_SESSION);
        command.append(newSessionName);

        if (channelName != null) {
            command.append(LTTngControlServiceConstants.OPTION_CHANNEL);
            command.append(channelName);
        }
        
        command.append(LTTngControlServiceConstants.OPTION_TRACEPOINT);
        
        executeCommand(command.toString(), monitor);
        
    }

    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#enableSyscalls(java.lang.String, java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void enableSyscalls(String sessionName, String channelName, IProgressMonitor monitor) throws ExecutionException {

        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_ENABLE_EVENT);

        command.append(LTTngControlServiceConstants.OPTION_ALL);
        command.append(LTTngControlServiceConstants.OPTION_KERNEL);

        String newSessionName = formatParameter(sessionName);

        command.append(LTTngControlServiceConstants.OPTION_SESSION);
        command.append(newSessionName);

        if (channelName != null) {
            command.append(LTTngControlServiceConstants.OPTION_CHANNEL);
            command.append(channelName);
        }
        
        command.append(LTTngControlServiceConstants.OPTION_SYSCALL);
        
        executeCommand(command.toString(), monitor);
    }
    
    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#enableProbe(java.lang.String, java.lang.String, java.lang.String, java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void enableProbe(String sessionName, String channelName, String eventName, boolean isFunction, String probe, IProgressMonitor monitor) throws ExecutionException {
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_ENABLE_EVENT);

        command.append(eventName);
        command.append(LTTngControlServiceConstants.OPTION_KERNEL);

        String newSessionName = formatParameter(sessionName);
        command.append(LTTngControlServiceConstants.OPTION_SESSION);
        command.append(newSessionName);

        if (channelName != null) {
            command.append(LTTngControlServiceConstants.OPTION_CHANNEL);
            command.append(channelName);
        }
        if (isFunction) {
            command.append(LTTngControlServiceConstants.OPTION_FUNCTION_PROBE);
        } else {
            command.append(LTTngControlServiceConstants.OPTION_PROBE);
        }
        
        command.append(probe);
        
        executeCommand(command.toString(), monitor);
    }

    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#enableLogLevel(java.lang.String, java.lang.String, java.lang.String, org.eclipse.linuxtools.internal.lttng2.ui.views.control.model.LogLevelType, org.eclipse.linuxtools.internal.lttng2.ui.views.control.model.TraceLogLevel, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void enableLogLevel(String sessionName, String channelName, String eventName, LogLevelType logLevelType, TraceLogLevel level, IProgressMonitor monitor) throws ExecutionException {
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_ENABLE_EVENT);

        command.append(eventName);
        command.append(LTTngControlServiceConstants.OPTION_UST);

        String newSessionName = formatParameter(sessionName);
        command.append(LTTngControlServiceConstants.OPTION_SESSION);
        command.append(newSessionName);

        if (channelName != null) {
            command.append(LTTngControlServiceConstants.OPTION_CHANNEL);
            command.append(channelName);
        }
        
        if (logLevelType == LogLevelType.LOGLEVEL) {
            command.append(LTTngControlServiceConstants.OPTION_LOGLEVEL);
        } else if (logLevelType == LogLevelType.LOGLEVEL_ONLY) {
            command.append(LTTngControlServiceConstants.OPTION_LOGLEVEL_ONLY);
            
        } else {
            return;
        }
        command.append(level.getInName());
        
        executeCommand(command.toString(), monitor);
        
    }

    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#disableEvent(java.lang.String, java.lang.String, java.util.List, boolean, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void disableEvent(String sessionName, String channelName, List<String> eventNames, boolean isKernel, IProgressMonitor monitor) throws ExecutionException {
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_DISABLE_EVENT);

        if (eventNames == null) {
            command.append(LTTngControlServiceConstants.OPTION_ALL);
        } else {
            // no events to enable
            if (eventNames.isEmpty()) {
                return;
            }

            for (Iterator<String> iterator = eventNames.iterator(); iterator.hasNext();) {
                String event = (String) iterator.next();
                command.append(event);
                if (iterator.hasNext()) {
                    command.append(',');
                }
            }
        }

        if (isKernel) {
            command.append(LTTngControlServiceConstants.OPTION_KERNEL);
        } else {
            command.append(LTTngControlServiceConstants.OPTION_UST);
        }

        String newSessionName = formatParameter(sessionName);
        command.append(LTTngControlServiceConstants.OPTION_SESSION);
        command.append(newSessionName);

        if (channelName != null) {
            command.append(LTTngControlServiceConstants.OPTION_CHANNEL);
            command.append(channelName);
        }

        executeCommand(command.toString(), monitor);
    }

    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#getContexts(org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public List<String> getContextList(IProgressMonitor monitor) throws ExecutionException {

        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_ADD_CONTEXT, LTTngControlServiceConstants.OPTION_HELP);

        ICommandResult result = executeCommand(command.toString(), monitor);

        String[] output = result.getOutput(); 
        
        List<String> contexts = new ArrayList<String>(0);
        
        int index = 0;
        boolean inList = false;
        while (index < output.length) {
            String line = result.getOutput()[index];
            
            Matcher startMatcher = LTTngControlServiceConstants.ADD_CONTEXT_HELP_CONTEXTS_INTRO.matcher(line);
            Matcher endMatcher = LTTngControlServiceConstants.ADD_CONTEXT_HELP_CONTEXTS_END_LINE.matcher(line);

            if (startMatcher.matches()) {
                inList = true;
            } else if (endMatcher.matches()) {
                break;
            } else if (inList == true) {
                String[] tmp = line.split(","); //$NON-NLS-1$
                for (int i = 0; i < tmp.length; i++) {
                    contexts.add(tmp[i].trim());
                }
            }
            index++;
        }
        return contexts;
    }

    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#addContexts(java.lang.String, java.lang.String, java.lang.String, boolean, java.util.List, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void addContexts(String sessionName, String channelName, String eventName, boolean isKernel, List<String> contextNames, IProgressMonitor monitor) throws ExecutionException {
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_ADD_CONTEXT);

        String newSessionName = formatParameter(sessionName);
        command.append(LTTngControlServiceConstants.OPTION_SESSION);
        command.append(newSessionName);
        
        if (channelName != null) {
            command.append(LTTngControlServiceConstants.OPTION_CHANNEL);
            command.append(channelName);
        }

        if (eventName != null) {
            command.append(LTTngControlServiceConstants.OPTION_EVENT);
            command.append(eventName);
        }

        if (isKernel) {
            command.append(LTTngControlServiceConstants.OPTION_KERNEL);
        } else {
            command.append(LTTngControlServiceConstants.OPTION_UST);
        }
        
        for (Iterator<String> iterator = contextNames.iterator(); iterator.hasNext();) {
            String context = (String) iterator.next();
            command.append(LTTngControlServiceConstants.OPTION_CONTEXT_TYPE);
            command.append(context);
        }

        executeCommand(command.toString(), monitor);
        
    }

    /*
     * (non-Javadoc)
     * @see org.eclipse.linuxtools.internal.lttng2.ui.views.control.service.ILttngControlService#calibrate(boolean, org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void calibrate(boolean isKernel, IProgressMonitor monitor) throws ExecutionException {
//        String newSessionName = formatParameter(sessionName);
        StringBuffer command = createCommand(LTTngControlServiceConstants.COMMAND_CALIBRATE);
//
//        command.append(OPTION_SESSION);
//        command.append(newSessionName);

        if (isKernel) {
            command.append(LTTngControlServiceConstants.OPTION_KERNEL);
        } else {
            command.append(LTTngControlServiceConstants.OPTION_UST);
        }

        command.append(LTTngControlServiceConstants.OPTION_FUNCTION_PROBE);

        executeCommand(command.toString(), monitor);
    }
    
    // ------------------------------------------------------------------------
    // Helper methods
    // ------------------------------------------------------------------------
    /**
     * Checks if command result is an error result.
     * 
     * @param result
     *            - the command result to check
     * @return true if error else false
     */
    protected boolean isError(ICommandResult result) {
        if ((result.getResult()) != 0 || (result.getOutput().length < 1 || LTTngControlServiceConstants.ERROR_PATTERN.matcher(result.getOutput()[0]).matches())) {
            return true;
        }
        return false;
    }

    /**
     * Formats the output string as single string.
     * 
     * @param output
     *            - output array
     * @return - the formatted output
     */
    protected String formatOutput(ICommandResult result) {
        if ((result == null) || result.getOutput() == null || result.getOutput().length == 0) {
            return ""; //$NON-NLS-1$
        }
        String[] output = result.getOutput();
        StringBuffer ret = new StringBuffer();
        ret.append("Return Value: "); //$NON-NLS-1$
        ret.append(result.getResult());
        ret.append("\n"); //$NON-NLS-1$
        for (int i = 0; i < output.length; i++) {
            ret.append(output[i] + "\n"); //$NON-NLS-1$
        }
        return ret.toString();
    }

    /**
     * Parses the domain information.
     * 
     * @param output
     *            - a command output array
     * @param currentIndex
     *            - current index in command output array
     * @param channels
     *            - list for returning channel information
     * @return the new current index in command output array
     */
    protected int parseDomain(String[] output, int currentIndex, List<IChannelInfo> channels) {
        int index = currentIndex;

        // Channels:
        // -------------
        // - channnel1: [enabled]
        //
        // Attributes:
        // overwrite mode: 0
        // subbufers size: 262144
        // number of subbufers: 4
        // switch timer interval: 0
        // read timer interval: 200
        // output: splice()

        while (index < output.length) {
            String line = output[index];

            Matcher outerMatcher = LTTngControlServiceConstants.CHANNELS_SECTION_PATTERN.matcher(line);
            if (outerMatcher.matches()) {
                IChannelInfo channelInfo = null;
                while (index < output.length) {
                    String subLine = output[index];

                    Matcher innerMatcher = LTTngControlServiceConstants.CHANNEL_PATTERN.matcher(subLine);
                    if (innerMatcher.matches()) {
                        channelInfo = new ChannelInfo(""); //$NON-NLS-1$
                        // get channel name
                        channelInfo.setName(innerMatcher.group(1));

                        // get channel enablement
                        channelInfo.setState(innerMatcher.group(2));

                        // add channel
                        channels.add(channelInfo);

                    } else if (LTTngControlServiceConstants.OVERWRITE_MODE_ATTRIBUTE.matcher(subLine).matches()) {
                        String value = getAttributeValue(subLine);
                        channelInfo.setOverwriteMode(!LTTngControlServiceConstants.OVERWRITE_MODE_ATTRIBUTE_FALSE.equals(value));
                    } else if (LTTngControlServiceConstants.SUBBUFFER_SIZE_ATTRIBUTE.matcher(subLine).matches()) {
                        channelInfo.setSubBufferSize(Long.valueOf(getAttributeValue(subLine)));

                    } else if (LTTngControlServiceConstants.NUM_SUBBUFFERS_ATTRIBUTE.matcher(subLine).matches()) {
                        channelInfo.setNumberOfSubBuffers(Integer.valueOf(getAttributeValue(subLine)));

                    } else if (LTTngControlServiceConstants.SWITCH_TIMER_ATTRIBUTE.matcher(subLine).matches()) {
                        channelInfo.setSwitchTimer(Long.valueOf(getAttributeValue(subLine)));

                    } else if (LTTngControlServiceConstants.READ_TIMER_ATTRIBUTE.matcher(subLine).matches()) {
                        channelInfo.setReadTimer(Long.valueOf(getAttributeValue(subLine)));

                    } else if (LTTngControlServiceConstants.OUTPUT_ATTRIBUTE.matcher(subLine).matches()) {
                        channelInfo.setOutputType(getAttributeValue(subLine));

                    } else if (LTTngControlServiceConstants.EVENT_SECTION_PATTERN.matcher(subLine).matches()) {
                        List<IEventInfo> events = new ArrayList<IEventInfo>();
                        index = parseEvents(output, index, events);
                        channelInfo.setEvents(events);
                        // we want to stay at the current index to be able to
                        // exit the domain
                        continue;
                    } else if (LTTngControlServiceConstants.DOMAIN_KERNEL_PATTERN.matcher(subLine).matches()) {
                        return index;

                    } else if (LTTngControlServiceConstants.DOMAIN_UST_GLOBAL_PATTERN.matcher(subLine).matches()) {
                        return index;
                    }
                    index++;
                }
            }
            index++;
        }
        return index;
    }

    /**
     * Parses the event information within a domain.
     * 
     * @param output
     *            - a command output array
     * @param currentIndex
     *            - current index in command output array
     * @param events
     *            - list for returning event information
     * @return the new current index in command output array
     */
    protected int parseEvents(String[] output, int currentIndex, List<IEventInfo> events) {
        int index = currentIndex;

        while (index < output.length) {
            String line = output[index];
            if (LTTngControlServiceConstants.CHANNEL_PATTERN.matcher(line).matches()) {
                // end of channel
                return index;
            } else if (LTTngControlServiceConstants.DOMAIN_KERNEL_PATTERN.matcher(line).matches()) {
                // end of domain
                return index;
            } else if (LTTngControlServiceConstants.DOMAIN_UST_GLOBAL_PATTERN.matcher(line).matches()) {
                // end of domain
                return index;
            } 

            Matcher matcher = LTTngControlServiceConstants.EVENT_PATTERN.matcher(line);
            Matcher matcher2 = LTTngControlServiceConstants.WILDCARD_EVENT_PATTERN.matcher(line);

            if (matcher.matches()) {
                IEventInfo eventInfo = new EventInfo(matcher.group(1).trim());
                eventInfo.setLogLevel(matcher.group(2).trim());
                eventInfo.setEventType(matcher.group(3).trim());
                eventInfo.setState(matcher.group(4));
                events.add(eventInfo);
                index++;
            } else if (matcher2.matches()) {
                IEventInfo eventInfo = new EventInfo(matcher2.group(1).trim());
                eventInfo.setLogLevel(TraceLogLevel.LEVEL_UNKNOWN);
                eventInfo.setEventType(matcher2.group(2).trim());
                eventInfo.setState(matcher2.group(3));
                
                if (eventInfo.getEventType() == TraceEventType.PROBE) {
                    IProbeEventInfo probeEvent = new ProbeEventInfo(eventInfo.getName());
                    probeEvent.setLogLevel(eventInfo.getLogLevel());
                    probeEvent.setEventType(eventInfo.getEventType());
                    probeEvent.setState(eventInfo.getState());

                    // Overwrite eventinfo
                    eventInfo = probeEvent;

                    // myevent2 (type: probe) [enabled]
                    // addr: 0xc0101340
                    // myevent0 (type: probe) [enabled]
                    // offset: 0x0
                    // symbol: init_post
                    index++;
                    while (index < output.length) {
                        String probeLine = output[index];
                        // parse probe
                        Matcher addrMatcher = LTTngControlServiceConstants.PROBE_ADDRESS_PATTERN.matcher(probeLine);
                        Matcher offsetMatcher = LTTngControlServiceConstants.PROBE_OFFSET_PATTERN.matcher(probeLine);
                        Matcher symbolMatcher = LTTngControlServiceConstants.PROBE_SYMBOL_PATTERN.matcher(probeLine);
                        if (addrMatcher.matches()) {
                            String addr = addrMatcher.group(2).trim();
                            probeEvent.setAddress(addr);
                        } else if (offsetMatcher.matches()) {
                            String offset = offsetMatcher.group(2).trim();
                            probeEvent.setOffset(offset);
                        } else if (symbolMatcher.matches()) {
                            String symbol = symbolMatcher.group(2).trim();
                            probeEvent.setSymbol(symbol);
                        } else if ((LTTngControlServiceConstants.EVENT_PATTERN.matcher(probeLine).matches()) || (LTTngControlServiceConstants.WILDCARD_EVENT_PATTERN.matcher(probeLine).matches())) {
                            break;
                        } else if (LTTngControlServiceConstants.CHANNEL_PATTERN.matcher(probeLine).matches()) {
                            break;
                        } else if (LTTngControlServiceConstants.DOMAIN_KERNEL_PATTERN.matcher(probeLine).matches()) {
                            // end of domain
                            break;
                        } else if (LTTngControlServiceConstants.DOMAIN_UST_GLOBAL_PATTERN.matcher(probeLine).matches()) {
                            // end of domain
                            break;
                        }
                        index++;
                    }
                    events.add(eventInfo);
                } else {
                    events.add(eventInfo);
                    index++;
                    continue;
                }
            } else {
                index++;
            }
//            else if (line.matches(EVENT_NONE_PATTERN)) {
                // do nothing
//            } else 

        }

        return index;
    }

    /**
     * Parses a line with attributes: <attribute Name>: <attribute value>
     * 
     * @param line
     *            - attribute line to parse
     * @return the attribute value as string
     */
    protected String getAttributeValue(String line) {
        String[] temp = line.split("\\: "); //$NON-NLS-1$
        return temp[1];
    }

    /**
     * Parses the event information within a provider.
     * 
     * @param output
     *            - a command output array
     * @param currentIndex
     *            - current index in command output array
     * @param events
     *            - list for returning event information
     * @return the new current index in command output array
     */
    protected int getProviderEventInfo(String[] output, int currentIndex, List<IBaseEventInfo> events) {
        int index = currentIndex;
        while (index < output.length) {
            String line = output[index];
            Matcher matcher = LTTngControlServiceConstants.PROVIDER_EVENT_PATTERN.matcher(line);
            if (matcher.matches()) {
                // sched_kthread_stop (loglevel: TRACE_EMERG0) (type:
                // tracepoint)
                IBaseEventInfo eventInfo = new BaseEventInfo(matcher.group(1).trim());
                eventInfo.setLogLevel(matcher.group(2).trim());
                eventInfo.setEventType(matcher.group(3).trim());
                events.add(eventInfo);
            } else if (LTTngControlServiceConstants.UST_PROVIDER_PATTERN.matcher(line).matches()) {
                return index;
            }
            index++;
        }
        return index;
    }

    /**
     * Formats a command parameter for the command execution i.e. adds quotes 
     * at the beginning and end if necessary.
     * @param parameter - parameter to format
     * @return formated parameter
     */
    protected String formatParameter(String parameter) {
        if (parameter != null) {
            StringBuffer newString = new StringBuffer();
            newString.append(parameter);

            if (parameter.contains(" ")) { //$NON-NLS-1$
                newString.insert(0, "\""); //$NON-NLS-1$
                newString.append("\""); //$NON-NLS-1$
            }
            return newString.toString();
        }
        return null;
    }

    /**
     * @param strings array of string that makes up a command line
     * @return string buffer with created command line
     */
    protected StringBuffer createCommand(String... strings) {
        StringBuffer command = new StringBuffer();
        command.append(LTTngControlServiceConstants.CONTROL_COMMAND);
        command.append(getTracingGroupOption());
        command.append(getVerboseOption());
        for (String string : strings) {
            command.append(string);
        }
        return command;
    }

    /**
     * @return the tracing group option if configured in the preferences
     */
    protected String getTracingGroupOption() {
        if (!ControlPreferences.getInstance().isDefaultTracingGroup() && !ControlPreferences.getInstance().getTracingGroup().equals("")) { //$NON-NLS-1$
            return LTTngControlServiceConstants.OPTION_TRACING_GROUP + ControlPreferences.getInstance().getTracingGroup();
        }
        return ""; //$NON-NLS-1$
    }

    /**
     * @return the verbose option as configured in the preferences
     */
    protected String getVerboseOption() {
        if (ControlPreferences.getInstance().isLoggingEnabled()) {
            String level = ControlPreferences.getInstance().getVerboseLevel();
            if (ControlPreferences.TRACE_CONTROL_VERBOSE_LEVEL_VERBOSE.equals(level)) {
                return LTTngControlServiceConstants.OPTION_VERBOSE;
            }
            if (ControlPreferences.TRACE_CONTROL_VERBOSE_LEVEL_V_VERBOSE.equals(level)) {
                return LTTngControlServiceConstants.OPTION_VERY_VERBOSE;
            } 
            if (ControlPreferences.TRACE_CONTROL_VERBOSE_LEVEL_V_V_VERBOSE.equals(level)) {
                return LTTngControlServiceConstants.OPTION_VERY_VERY_VERBOSE;
            }
        }
        return ""; //$NON-NLS-1$
    }

    /**
     * Method that logs the command and command result if logging is enabled as well as forwards
     * the command execution to the shell.
     * @param command - the command to execute
     * @param monitor - a progress monitor
     * @return the command result
     * @throws ExecutionException
     */
    protected ICommandResult executeCommand(String command, IProgressMonitor monitor) throws ExecutionException {
        return executeCommand(command, monitor, true);
    }
    
    /**
     * Method that logs the command and command result if logging is enabled as well as forwards
     * the command execution to the shell.
     * @param command - the command to execute
     * @param monitor - a progress monitor
     * @param - checkForError - true to verify command result, else false
     * @return the command result
     * @throws ExecutionException in case of error result
     */
    protected ICommandResult executeCommand(String command, IProgressMonitor monitor, boolean checkForError) throws ExecutionException {
        if (ControlPreferences.getInstance().isLoggingEnabled()) {
            ControlCommandLogger.log(command);
        }

        ICommandResult result = fCommandShell.executeCommand(command.toString(), monitor);
        
        if (ControlPreferences.getInstance().isLoggingEnabled()) {
            ControlCommandLogger.log(formatOutput(result));
        }

        if (isError(result)) {
            throw new ExecutionException(Messages.TraceControl_CommandError + " " + command.toString() + "\n" + formatOutput(result)); //$NON-NLS-1$ //$NON-NLS-2$
        }
        
        return result;
    }
}
