/*********************************************************************************************************
DVR, Digital Video Recorder - a tool to record movies (audio/video), using realtime compression

It uses libavifile (see http://divx.euro.ru) and some code from kwintv (see wenk@mathematik.uni-kl.de)

This program is free software; you can redistribute it and/or modify it under the terms of the GNU General
Public License as published by the Free Software Foundation, etc.
This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY, etc.

copyright(C) february 2001, Pierre Hbert (pierre.hebert@netcourrier.com)
*********************************************************************************************************/

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>

#include <stdexcept>
#include <list>
#include <string>
#include <vector>
#include <iostream>

#include <avifile/avifile.h>
#include <avifile/avm_creators.h>
#include <avifile/image.h>
#include <avifile/avm_fourcc.h>

#include "dvr.h"
#include "framepool.h"
#include "v4l.h"
#include "dsp.h"

using namespace std;

DVR::DVR():
	good(false),
	initialized(false),
	error_message("not initialized"),
	fp(NULL),
	avi_file(NULL),
	v4l(NULL),
	status(DVR_READY),
	video_width(384),
	video_height(288),
	video_device(""),
	sound_device(""),
	file_name(""),
  allow_external_frame_reading(0),
  external_fp(NULL)
{
	param_list.addParameter("max_recording_time",	PARAM_TYPE_INT,		(void*)&max_recording_time);
	param_list.addParameter("video_width",				PARAM_TYPE_INT,		(void*)&video_width);
	param_list.addParameter("video_height",				PARAM_TYPE_INT,		(void*)&video_height);
	//param_list.addParameter("video_pixel_format",	PARAM_TYPE_INT,		(void*)&video_pixel_format);	
	param_list.addParameter("video_top_margin",		PARAM_TYPE_INT,		(void*)&video_top_margin);
	param_list.addParameter("video_bottom_margin",PARAM_TYPE_INT,		(void*)&video_bottom_margin);
	param_list.addParameter("video_codec_name",	 	PARAM_TYPE_STRING,(void*)&video_codec_name);	
	param_list.addParameter("capture_frame_rate",	PARAM_TYPE_DOUBLE,(void*)&capture_frame_rate);	
	param_list.addParameter("video_frame_rate",		PARAM_TYPE_DOUBLE,(void*)&video_frame_rate);	
	param_list.addParameter("video_device",				PARAM_TYPE_STRING,(void*)&video_device);	
	param_list.addParameter("video_channel",			PARAM_TYPE_INT,		(void*)&video_channel);
	param_list.addParameter("video_norm",					PARAM_TYPE_INT,		(void*)&video_norm);	
	param_list.addParameter("sound_recording_enabled", PARAM_TYPE_INT, (void*)&sound_recording_enabled);	
	param_list.addParameter("sound_sample_size",	PARAM_TYPE_INT,		(void*)&sound_sample_size);	
	param_list.addParameter("sound_frequency",		PARAM_TYPE_INT,		(void*)&sound_frequency);	
	param_list.addParameter("sound_channels",			PARAM_TYPE_INT,		(void*)&sound_channels);	
	param_list.addParameter("sound_format",				PARAM_TYPE_INT,		(void*)&sound_format);	
	param_list.addParameter("sound_byterate",			PARAM_TYPE_INT,		(void*)&sound_byterate);	
	param_list.addParameter("sound_device",				PARAM_TYPE_STRING,(void*)&sound_device);		
	param_list.addParameter("file_segment_size",	PARAM_TYPE_INT,		(void*)&file_segment_size);		
	param_list.addParameter("file_name",					PARAM_TYPE_STRING,(void*)&file_name);		

	param_list.addParameter("allow_external_frame_reading",	PARAM_TYPE_INT,(void*)&allow_external_frame_reading);		
		
  // default values
  max_recording_time=10;
  video_width=384;
  video_height=288;
  //video_pixel_format=24;
  video_top_margin=0;
  video_bottom_margin=0;
  video_codec_name="FFMPEG DivX5";
  capture_frame_rate=0.0;
  video_frame_rate=25.0;
  video_device="/dev/video0";
  video_channel=0;
  video_norm=3;
  sound_recording_enabled=1;
  sound_sample_size=16;
  sound_frequency=44100;
  sound_channels=1;
  sound_format=85;
  sound_byterate=8000;
  sound_device="/dev/dsp";
  file_segment_size=0;
  file_name="movie.avi";
  
	// semaphores initialisation
	sem_avi_file=semget(IPC_PRIVATE, 1, IPC_CREAT | 0666);
	semctl(sem_avi_file, 0, SETVAL, 1L);
	sem_are_you_ready=semget(IPC_PRIVATE, 1, IPC_CREAT | 0666);
	
	v4l=new V4L();

	status=DVR_READY;
}

DVR::~DVR() {
  if(external_fp) {
    delete external_fp;
  }

  if(fp) {
    delete fp;
  }

  if(v4l) {
    delete v4l;
  }

  semctl(sem_avi_file, 0, IPC_RMID, 0);
  semctl(sem_are_you_ready, 0, IPC_RMID, 0);
}

int DVR::parameterAsInt(const string &param_name) {
	return param_list.parameterAsInt(param_name);
}

double DVR::parameterAsDouble(const string &param_name) {
	return param_list.parameterAsDouble(param_name);
}

string DVR::parameterAsString(const string &param_name) {
	return param_list.parameterAsString(param_name);
}

void DVR::setParameter(const string &param_name, int value) {
	param_list.setParameter(param_name, value);
}

void DVR::setParameter(const string &param_name, double value) {
	param_list.setParameter(param_name, value);
}

void DVR::setParameter(const string &param_name, string value) {
	param_list.setParameter(param_name, value);
}

int DVR::nbFramesInEncQueue() {
	if(fp) {
		return fp->size();
	} else {
		return 0;
	}
}
	
	
bool DVR::initialize() {
	// v4l device initialisation as suggested by minji liang
	//string s=string("v4l-conf -c ")+video_device;
	//system(s.c_str());

  if(initialized) {
		char tmp[300];
    sprintf(tmp, "DVR already initialized (%s:%d)", __FILE__, __LINE__);
		error_message=string(tmp);
		return false;
  }
  
	// video initialization
  v4l->open(video_device.c_str());

	if(!v4l->isGood()) {
		char tmp[300];
    sprintf(tmp, "unable to initialize the video for linux interface (%s:%d)", __FILE__, __LINE__);
		error_message=string(tmp);
		return false;
	}

	initialized=true;
	return true;
}

bool DVR::startRecording() {
  good=false;

  // parameter control
  int min_w, min_h, max_w, max_h;
  v4l->getMinSize(min_w, min_h);
  v4l->getMaxSize(max_w, max_h);

  if(video_width<min_w || video_width>max_w) {
		char tmp[300];
    sprintf(tmp, "video width out of bounds, should be between %d and %d", min_w, max_w);
		error_message=string(tmp);
		return false;
	}
  
  if(video_height<min_h || video_height>max_h) {
		char tmp[300];
    sprintf(tmp, "video height out of bounds, should be between %d and %d", min_h, max_h);
		error_message=string(tmp);
		return false;
	}

  if(video_top_margin<0 || video_top_margin>=video_height) {
		char tmp[300];
    sprintf(tmp, "video top margin out of bounds, should be between 0 and %d", video_height-1);
		error_message=string(tmp);
		return false;
  }

  if(video_bottom_margin<0 || video_bottom_margin>=video_height) {
		char tmp[300];
    sprintf(tmp, "video bottom margin out of bounds, should be between 0 and %d", video_height-1);
		error_message=string(tmp);
		return false;
  }

  if((video_top_margin+video_bottom_margin)>=video_height) {
		char tmp[300];
    sprintf(tmp, "video bottom and top margin too high");
		error_message=string(tmp);
		return false;
  }

  if(video_frame_rate<0.0000001) {
		char tmp[300];
    sprintf(tmp, "bad video frame rate, should be greater than 0");
		error_message=string(tmp);
		return false;
  }

  if(capture_frame_rate<0.0) {
		char tmp[300];
    sprintf(tmp, "bad capture frame rate, should be greater or equal to 0");
		error_message=string(tmp);
		return false;
  }

  if(video_channel<0 || video_channel>=signed(v4l->videoChannels().size())) {
		char tmp[300];
    sprintf(tmp, "invalid channel number, should be less than %d", v4l->videoChannels().size());
		error_message=string(tmp);
		return false;
  }
  
  if(video_norm<0 || video_norm>3) {
		char tmp[300];
    sprintf(tmp, "invalid norm, should be between 0 and 3");
		error_message=string(tmp);
		return false;
  }

  if(sound_recording_enabled!=0 && sound_recording_enabled!=1) {
		char tmp[300];
    sprintf(tmp, "invalid sound_recording_enabled value, should be 0 or 1");
		error_message=string(tmp);
		return false;
  }

  if(sound_sample_size!=8 && sound_sample_size!=16) {
		char tmp[300];
    sprintf(tmp, "invalid sound sample size, should be 8 or 16");
		error_message=string(tmp);
		return false;
  }

  if(sound_frequency<1) {
		char tmp[300];
    sprintf(tmp, "invalid sound frequency, should be greater than 0");
		error_message=string(tmp);
		return false;
  }

  if(sound_channels!=1 && sound_channels!=2) {
		char tmp[300];
    sprintf(tmp, "invalid number of sound channels, should be 1 or 2");
		error_message=string(tmp);
		return false;
  }

  if(sound_format!=WAVE_FORMAT_PCM && sound_format!=WAVE_FORMAT_MPEGLAYER3) {
		char tmp[300];
    sprintf(tmp, "invalid sound format, should be %d (PCM) or %d (mp3)", WAVE_FORMAT_PCM, WAVE_FORMAT_MPEGLAYER3);
		error_message=string(tmp);
		return false;
  }

  if(sound_byterate<1) {
		char tmp[300];
    sprintf(tmp, "invalid sound byterate, should be greater than 0");
		error_message=string(tmp);
		return false;
  }

  if(file_segment_size<0) {
		char tmp[300];
    sprintf(tmp, "negative value for file segment size is not allowed");
		error_message=string(tmp);
		return false;
  }

  if(max_recording_time<0) {
		char tmp[300];
    sprintf(tmp, "invalid value for max recording time, should be greater than 0");
		error_message=string(tmp);
		return false;
  }
  
  v4l->setPixelFormat(V4L::PX_RGB24);

  int depth=24;
  switch(v4l->pixelFormat()) {
    case V4L::PX_RGB565:  video_pixel_format=24; depth=16; break;
    case V4L::PX_RGB24:   video_pixel_format=24; depth=24; break;
    case V4L::PX_YUV420P: video_pixel_format=fccI420; depth=24; break;
  };
  
	v4l->setCaptureSize(video_width, video_height);
  v4l->videoChannels()[video_channel].norm=video_norm;
	v4l->setCurrentVideoChannel(video_channel);

  if(v4l->hasAudio() && sound_recording_enabled) {
	  v4l->enableAudio(true);
  }
	/*v4l->setAudioVolume(65535);//?
	v4l->setAudioBalance(100);
	v4l->setAudioMode(VIDEO_SOUND_MONO);*/

	if(!initialized) {
		char tmp[300];
    sprintf(tmp, "trying to start to record while not initialized (%s:%d)", __FILE__, __LINE__);
		error_message=string(tmp);
		return false;
	}

	// creation of avi file
	try {
		if(file_segment_size==0) {
			avi_file=CreateIAviWriteFile(file_name.c_str());
		} else {
			avi_file=CreateSegmentedFile(file_name.c_str(), file_segment_size*1048576LL);
		}
	//} catch(FatalError &error) {
	} catch(const char *s) {
		//error.PrintAll();
		char tmp[300];
    sprintf(tmp, "avifile error upon CreateIAviWriteFile or CreateSegmentedFile (%s:%d)", __FILE__, __LINE__);
		error_message=string(tmp);
		good=false;
		return false;
	}

	good=true;
		
	nb_frames_elapsed=0;
	nb_frames_stored=0;
	nb_frames_lost_in_capture=0;
	nb_frames_lost_in_encoding=0;
	
	fp=new FramePool(video_width, video_height, depth, 100);
	fp->setTopMargin(video_top_margin);
	fp->setBottomMargin(video_bottom_margin);	

  if(allow_external_frame_reading) {
	  external_fp=new FramePool(video_width, video_height, depth, 10);
    external_fp->setTopMargin(video_top_margin);
    external_fp->setBottomMargin(video_bottom_margin);	
  }
  
	// thread management
	int num_threads;	//number of threads to synchronize
	// fusion of capture and storage num_threads=3;
	num_threads=2;
	if(sound_recording_enabled)	num_threads++;
	semctl(sem_are_you_ready, 0, SETVAL, num_threads);

	// on the verge to begin
	status=DVR_RECORDING;	

	// save the begin date
	gettimeofday(&start_recording_time, NULL);

	// launch all the threads and we're done
	pthread_create(&t_video_capture_storage, NULL, DVR::video_capture_storage, this);
	pthread_create(&t_video_encoding, NULL, DVR::video_encoding, this);	
	if(sound_recording_enabled) {
		pthread_create(&t_audio_encoding, NULL, DVR::audio_encoding, this);		
	}

	return true;
}

bool DVR::stopRecording() {
	if(!initialized) {
		char tmp[30];sprintf(tmp, "%d", __LINE__);
		error_message=string("trying to stop to record while not initialized (")+string(__FILE__)+":"+string(tmp)+")";
		return false;
	}
	
	if(status==DVR_RECORDING) {
		status=DVR_STOP_ASKED;
	}
	
	if(status==DVR_READY) {
		char tmp[30];sprintf(tmp, "%d", __LINE__);
		error_message=string("trying to stop to record while not recording (")+string(__FILE__)+":"+string(tmp)+")";
		return false;
	}

	pthread_join(t_video_capture_storage, NULL);
	pthread_join(t_video_encoding, NULL);	
	if(sound_recording_enabled) {
		pthread_join(t_audio_encoding, NULL);
	}

  if(v4l->hasAudio() && sound_recording_enabled) {
	  v4l->enableAudio(false);
  }

	status=DVR_READY;		

	if(external_fp) { delete external_fp; external_fp=NULL; }
	if(fp) { delete fp; fp=NULL; }
	if(avi_file) { delete avi_file; avi_file=NULL; }

	return true;
}

//////////////////////////////////////////////// static methods and functions ////////////////////////////////////////////////

// return current date in microseconds
double DVR::now() {
	struct timeval _now; gettimeofday(&_now, NULL);
	return double(_now.tv_sec)*1000000.0+double(_now.tv_usec);
}
/*
// increment date with <increment> microseconds
void DVR::inc_date(struct timeval &d,  increment) {
	d.tv_usec+=increment;
	while(d.tv_usec>1000000) { d.tv_usec-=1000000; d.tv_sec++; }
}
*/
// semaphore of type "rendez-vous"
void DVR::wait_for_all_threads_ready(void *o) {
	DVR *me=(DVR*)o;
	struct sembuf op;
	op.sem_num=0;op.sem_flg=0;
	
	op.sem_op=-1;semop(me->sem_are_you_ready, &op, 1);
	op.sem_op=0;semop(me->sem_are_you_ready, &op, 1);
}


void *DVR::video_capture_storage(void *o) {
	DVR *me=(DVR*)o;
	
	int nb_frames_added=0;
	unsigned char *new_frame;
	int nb_max_frames;
	double next_theoric_date, n;
	double frame_time;

  if(me->capture_frame_rate<0.0000001) {
	  frame_time=1000000.0/me->video_frame_rate;
  } else {
	  frame_time=1000000.0/me->capture_frame_rate;
  }

  nb_max_frames=int(me->video_frame_rate*double(me->max_recording_time));
	if(nb_max_frames==0) nb_max_frames=-1;

	wait_for_all_threads_ready(o);

	if(!me->good) {
		me->status=DVR_RECORD_DONE;
		return NULL;
	}


	next_theoric_date=now()+frame_time;

  bool end=false;
  do {
		new_frame=me->v4l->captureFrame();
		n=now();

    
    // first frame of the loop
    bool first_frame=true;
    
    // loop and add this frame till we are synchronized
    do {
      // for monitoring : printf("%lf\t\t%f\n", n/10000, next_theoric_date/10000);

      // if this not the first frame in this loop it means 
      // it has been lost in capture
      if(!first_frame) {
        me->nb_frames_lost_in_capture++;
      }
      
      me->nb_frames_elapsed++;		

      // add the frame if there is a slot left
      if(!me->fp->full()) {
        me->fp->addFrame(new_frame);
        nb_frames_added++;
        if(nb_frames_added>=nb_max_frames && nb_max_frames!=-1) {
          end=true;
        }
      } else {
        me->nb_frames_lost_in_encoding++;
        cerr<<"encoding queue overflow : audio/video sync lost !!!"<<endl;
      }
      // eventually add frame for external reader
      if(me->allow_external_frame_reading) {
        if(!me->external_fp->full()) {
          me->external_fp->addFrame(new_frame);
        }
      }

      // first frame of the loop
      first_frame=false;

      // compute the theorical date of the next frame
      next_theoric_date+=frame_time;
    } while(n>next_theoric_date && !end);

    // wait some time if needed (low frame rate)
    if((next_theoric_date-n)>0) {
      usleep(long(next_theoric_date-n));
    }
    
		if(nb_frames_added>=nb_max_frames && nb_max_frames!=-1) { 
      //me->status=DVR_RECORD_DONE; break; 
      end=true;
    }
    
		if(me->status==DVR_STOP_ASKED) break;
  } while(!end);

  me->status=DVR_RECORD_DONE;
  
  // add a null frame to signal the end of the stream
  me->fp->addFrame(NULL);
  if(me->allow_external_frame_reading) {
    me->external_fp->addFrame(NULL);
  }

	return NULL;
}

void *DVR::video_encoding(void *o) {
	DVR *me=(DVR*)o;
	IAviVideoWriteStream *stream=NULL;
	fourcc_t video_codec;
	int nb_max_frames;
  if(me->capture_frame_rate<0.00000001) {
	  nb_max_frames=int(me->video_frame_rate*double(me->max_recording_time));
  } else {
	  nb_max_frames=int(me->capture_frame_rate*double(me->max_recording_time));
  }
	
	if(nb_max_frames==0) nb_max_frames=-1;
	
	struct sembuf op;
	op.sem_num=0;op.sem_flg=0;

	BitmapInfo bh(me->video_width, me->video_height-me->video_top_margin-me->video_bottom_margin, me->video_pixel_format);

	// looking for the fourcc of the codec
	video_codec=0;
	CodecInfo::match(0xffffffff, CodecInfo::Video, NULL);
	
	try {
	avm::vector<CodecInfo>::iterator it_ci;
  	for(it_ci=video_codecs.begin(); it_ci!=video_codecs.end(); it_ci++) {
  		if(!(it_ci->direction & CodecInfo::Encode)) continue;
  		if(it_ci->GetName()==me->video_codec_name) {
  			video_codec=it_ci->fourcc;
  			break;
  		}
  	}
  	if(video_codec==0) {
		cerr<<"video codec '"<<me->video_codec_name<<"' not found"<<endl;
		cerr<<"available video codecs are : "<<endl;
		string s;
		for(it_ci=video_codecs.begin(); it_ci!=video_codecs.end(); it_ci++) {
      			if(!(it_ci->direction & CodecInfo::Encode)) continue;
        		if(s=="") s=string("'")+it_ci->GetName()+string("'");
          		else s+=string(", '")+it_ci->GetName()+string("'");
		}
		cerr<<s<<endl;
		me->error_message=string("codec '")+me->video_codec_name+string("' not found");
		me->good=false;
  	}

	//} catch(FatalError &error) {
	} catch(const char *s) {
		cerr<<"error on finding the video codec"<<" - "<<s<<endl;
		me->good=false;
	}

	// creating a video stream
	op.sem_op=-1;semop(me->sem_avi_file, &op, 1);
	if(me->good) {
	try {
		stream=me->avi_file->AddVideoStream(video_codec, &bh, int(1000000.0/me->video_frame_rate));
		stream->Start();
	//} catch(FatalError &error) {
	} catch(const char *s) {
		//error.PrintAll();
		char tmp[30];sprintf(tmp, "%d", __LINE__);
		me->error_message=string("avifile error upon AddVideoStream (")+string(__FILE__)+":"+string(tmp)+" - "+string(s)+")";		
		me->good=false;
	}
	}
	op.sem_op=1;semop(me->sem_avi_file, &op, 1);	

	wait_for_all_threads_ready(o);

	if(!me->good) {
		//if(stream) delete stream;
		return NULL;
	}
		
	int f=0;
  unsigned char *frame;
	for(;;) {
    frame=(unsigned char *)me->fp->getFrame();
    if(!frame) {
      break;
    }
		CImage *im=new CImage(&bh, frame, false);
    if(me->v4l->pixelFormat()==V4L::PX_RGB565) {
      CImage *im2=new CImage(im, 24);
      im->Release();
      im=im2;
    }
		op.sem_op=-1;semop(me->sem_avi_file, &op, 1);
		stream->AddFrame(im);
		op.sem_op=1;semop(me->sem_avi_file, &op, 1);
		im->Release();
		//delete im;
		me->fp->removeFrame();
		me->nb_frames_stored++;
		f++;
	}
	return 0;
}

void *DVR::audio_encoding(void *o) {
	DVR *me=(DVR*)o;
	IAviAudioWriteStream *stream=NULL;

	struct sembuf op;
	op.sem_num=0;op.sem_flg=0;

	WAVEFORMATEX wfm;
	wfm.wFormatTag=WAVE_FORMAT_PCM;
	wfm.nChannels=me->sound_channels;
	wfm.nSamplesPerSec=me->sound_frequency; //*me->sound_channels;
	wfm.nAvgBytesPerSec=me->sound_sample_size*me->sound_frequency*me->sound_channels/8;
	wfm.nBlockAlign=(me->sound_sample_size*me->sound_channels)/8;
	wfm.wBitsPerSample=me->sound_sample_size;
	wfm.cbSize=0;

	op.sem_op=-1;semop(me->sem_avi_file, &op, 1);
	try {
		stream=me->avi_file->AddAudioStream(me->sound_format, &wfm, me->sound_channels*me->sound_byterate);
		stream->Start();	
	//} catch(FatalError &error) {
	} catch(const char *s) {
		//error.PrintAll();
		char tmp[30];sprintf(tmp, "%d", __LINE__);
		me->error_message=string("avifile error upon AddAudioStream (")+string(__FILE__)+":"+string(tmp)+" - "+string(s)+")";
		me->good=false;
	}
	op.sem_op=1;semop(me->sem_avi_file, &op, 1);	

	dsp audio_source(me->sound_device.c_str());
	audio_source.open(me->sound_sample_size, me->sound_channels, me->sound_frequency);
	
	char *tmp=new char[audio_source.getBufSize()];
	
	int current_buf_size;
	double bytes_to_store;
	if(me->max_recording_time!=0) {
		bytes_to_store=double(wfm.nAvgBytesPerSec)*double(me->max_recording_time);
	} else {
		bytes_to_store=-1;
	}

	wait_for_all_threads_ready(o);

	if(!me->good) {
		//if(stream) delete stream;
		delete []tmp;
		return NULL;
	}
	
	for(;;) {
		memcpy(tmp, audio_source.readBuf(), audio_source.getBufSize());		
		
		current_buf_size=audio_source.getBufSize();
		if(bytes_to_store!=-1) {
			if(bytes_to_store<double(current_buf_size)) {
				current_buf_size=int(bytes_to_store);
			}
			bytes_to_store-=double(current_buf_size);
		}
		
		op.sem_op=-1;semop(me->sem_avi_file, &op, 1);
		stream->AddData(tmp, current_buf_size);
		op.sem_op=1;semop(me->sem_avi_file, &op, 1);		
		
		if(me->status==DVR_STOP_ASKED) break;
		if(bytes_to_store==0) break;
	}

	delete []tmp;
	
	return NULL;
}
